
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Alert,
  Image,
  Modal,
  Animated,
  Share,
  Linking,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import SignatureCanvas from 'react-native-signature-canvas';
import DateTimePicker from '@react-native-community/datetimepicker';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { Buffer } from 'buffer';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE101_024Step3 = ({ setActiveStep = () => {} }) => {
  const navigation = useNavigation();
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [inspectionResults, setInspectionResults] = useState(null);
  const [visitors, setVisitors] = useState([
    {
      id: 1,
      name: '',
      company: '',
      date: new Date().toLocaleDateString(),
      signatureStatus: false,
      signatureData: null,
    }
  ]);
  const [representativeDetails, setRepresentativeDetails] = useState({
    name: '',
    designation: '',
  });
  const [showSignatureModal, setShowSignatureModal] = useState(false);
  const [currentVisitorIndex, setCurrentVisitorIndex] = useState(0);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [currentDateIndex, setCurrentDateIndex] = useState(0);
  const signatureRef = useRef(null);
  const capturedSignature = useRef(null);
  
  // Firebase states
  const [isLoading, setIsLoading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [reportId, setReportId] = useState(null);
  const isMounted = useRef(true);

  useEffect(() => {
    loadData();
    isMounted.current = true;
    return () => {
      isMounted.current = false;
    };
  }, []);

  // Firebase helper functions
  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseInspectionResults');
      await AsyncStorage.removeItem('hseFinalReport');
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const loadData = async () => {
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      const savedResults = await AsyncStorage.getItem('hseInspectionResults');
      
      if (savedHse) {
        const hseData = JSON.parse(savedHse);
        setSelectedChecklist(hseData.selectedChecklist);
      }
      
      if (savedResults) {
        const results = JSON.parse(savedResults);
        setInspectionResults(results);
      }
    } catch (error) {
      console.error('Error loading data:', error);
      Alert.alert('Error', 'Failed to load inspection data.');
    }
  };

  const handleVisitorDetailChange = (visitorIndex, field, value) => {
    setVisitors(prev => prev.map((visitor, index) => 
      index === visitorIndex 
        ? { ...visitor, [field]: value }
        : visitor
    ));
  };

  const handleDatePress = (visitorIndex) => {
    setCurrentDateIndex(visitorIndex);
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      const formattedDate = selectedDate.toLocaleDateString();
      handleVisitorDetailChange(currentDateIndex, 'date', formattedDate);
    }
  };

  const handleRepresentativeDetailChange = (field, value) => {
    setRepresentativeDetails(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSignaturePress = (visitorIndex) => {
    setCurrentVisitorIndex(visitorIndex);
    setShowSignatureModal(true);
    capturedSignature.current = null;
  };

  const handleSignatureComplete = (signature) => {
    if (signature) {
      setVisitors(prev => prev.map((visitor, index) => 
        index === currentVisitorIndex 
          ? { ...visitor, signatureData: signature, signatureStatus: true }
          : visitor
      ));
    }
    setShowSignatureModal(false);
  };

  const handleSignatureClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
      capturedSignature.current = null;
    }
  };

  const handleSignatureConfirm = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature();
    }
  };

  const handleSignatureData = (signature) => {
    console.log('Signature captured:', signature ? signature.substring(0, 50) + '...' : signature);
    capturedSignature.current = signature;
    handleSignatureComplete(signature);
  };

  const addVisitor = () => {
    if (visitors.length >= 5) {
      Alert.alert('Maximum Visitors', 'You can only add up to 5 visitors.');
      return;
    }
    
    const newVisitor = {
      id: visitors.length + 1,
      name: '',
      company: '',
      date: new Date().toLocaleDateString(),
      signatureStatus: false,
      signatureData: null,
    };
    
    setVisitors(prev => [...prev, newVisitor]);
  };

  const removeVisitor = (visitorIndex) => {
    if (visitors.length <= 1) {
      Alert.alert('Minimum Visitors', 'At least one visitor is required.');
      return;
    }
    
    setVisitors(prev => prev.filter((_, index) => index !== visitorIndex));
  };

  const validateFields = () => {
    // Check if at least one visitor has all required fields
    const hasValidVisitor = visitors.some(visitor => 
      visitor.name.trim() && 
      visitor.company.trim() && 
      visitor.signatureStatus
    );

    if (!hasValidVisitor) {
      Alert.alert('Error', 'At least one visitor must have name, company, and signature.');
      return false;
    }

    if (!representativeDetails.name.trim()) {
      Alert.alert('Error', 'Titan Drilling representative name is required.');
      return false;
    }
    if (!representativeDetails.designation.trim()) {
      Alert.alert('Error', 'Representative designation is required.');
      return false;
    }
    return true;
  };

  const handleSubmit = async () => {
    if (!validateFields()) return;

    setIsLoading(true);
    setUploadProgress(0);

    try {
      const userId = await getAdminUserId();
      const savedHse = await AsyncStorage.getItem('hseStep1');
      const savedResults = await AsyncStorage.getItem('hseInspectionResults');
      
      if (!savedHse || !savedResults) {
        throw new Error('Previous step data is missing');
      }

      const hseData = JSON.parse(savedHse);
      const inspectionResults = JSON.parse(savedResults);

      const newReportId = Date.now().toString();
      setReportId(newReportId);

      const reportData = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        checklist: selectedChecklist,
        inspectionResults,
        visitors,
        representativeDetails,
        timestamp: new Date().toISOString(),
      };

      // Save to Firebase using the same path as Hsemodule43step3
      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${newReportId}`);
      await dbRef.set(reportData);
      setUploadProgress(50);

      // Generate PDF
      const pdfPath = await generatePDFWithHtml(reportData);
      const storagePath = `GlobalHSEInspections/admins/${userId}/reports/HSE_Induction_Report_${newReportId}.pdf`;
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath);

      // Update report with PDF URL
      await dbRef.update({
        pdfDownloadUrl,
        status: 'completed',
      });

      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));
      setUploadProgress(100);

      await flushStepData(newReportId);

      if (isMounted.current) {
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Induction report saved successfully.',
          position: 'top',
        });
      }
    } catch (error) {
      console.error('Error in handleSubmit:', error);
      if (isMounted.current) {
        setIsLoading(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to save report.',
          position: 'top',
        });
        Alert.alert('Save Error', `Failed to save report.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false });
      }
    } finally {
      if (isMounted.current) {
        setIsLoading(false);
      }
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      // Use the same database path as Hsemodule43step3
      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      navigation.navigate('HSEViewPDF', { 
        pdfUrl: pdfUrl, 
        onGoBack: () => { 
          setShowSuccessModal(false); 
        } 
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const checklist = report.checklist || {};
      const inspectionResults = report.inspectionResults || {};
      const visitors = report.visitors || [];
      const representativeDetails = report.representativeDetails || {};
      const isHSE101 = checklist.document_id === 'HSE-101';
      
      // Fetch request number from Firebase
      let requestNumber = 'N/A';
      try {
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (userEmail) {
          const userSnapshot = await database()
            .ref('/Globalusers/admins')
            .orderByChild('email')
            .equalTo(userEmail)
            .once('value');
          
          const userData = userSnapshot.val();
          if (userData) {
            const userId = Object.keys(userData)[0];
            const reportSnapshot = await database()
              .ref(`/GlobalHSEInspections/admins/${userId}/${report.id}`)
              .once('value');
            
            const reportData = reportSnapshot.val();
            if (reportData && reportData.step1 && reportData.step1.requestNumber) {
              requestNumber = reportData.step1.requestNumber;
            }
          }
        }
      } catch (error) {
        console.error('Error fetching request number:', error);
      }

      // Generate visitor table rows with signature images
      let visitorRows = '';
      visitors.forEach((visitor, index) => {
        const signatureImage = visitor.signatureStatus && visitor.signatureData 
          ? `<img src="${visitor.signatureData}" alt="Signature" style="width: 80px; height: 40px; object-fit: contain; border: 1px solid #ccc;"/>`
          : '';
        
        visitorRows += `
          <tr>
            <td>${visitor.name || ''}</td>
            <td style="text-align: center; vertical-align: middle;">
              ${signatureImage}
            </td>
            <td>${visitor.company || ''}</td>
            <td>${visitor.date || ''}</td>
          </tr>
        `;
      });

      // Generate induction categories table
      let inductionCategoriesRows = '';
      if (inspectionResults.inductionCategories) {
        const categories = inspectionResults.inductionCategories;
        Object.entries(categories).forEach(([category, isChecked]) => {
          if (category !== 'notes') {
            inductionCategoriesRows += `
              <tr>
                <td>${category}</td>
                <td style="text-align: center;">${isChecked ? '✓' : ''}</td>
              </tr>
            `;
          }
        });
      }

      // Generate content based on checklist type
      let contentSections = '';
      
      if (isHSE101) {
        // HSE-101 format with numbered sections
        const checklistData = inspectionResults.checklistData || {};
        
        // Section 1: General PPE Requirement
        if (checklistData.personal_protective_equipment) {
          contentSections += `
            <div class="content-section">
              <h3>1. All visitors to the TITAN DRILLING yard must be wearing adequate PPE. SAFETY BOOTS are mandatory throughout the workshop area.</h3>
              <p>${checklistData.personal_protective_equipment.notes}</p>
            </div>
          `;
        }

        // Section 2: Area-specific PPE
        if (checklistData.personal_protective_equipment?.additional) {
          contentSections += `
            <div class="content-section">
              <h3>2. Depending on the areas to be visited, the following PPE shall be worn:</h3>
              <ul>
                ${checklistData.personal_protective_equipment.additional.map(item => `<li>${item}</li>`).join('')}
              </ul>
            </div>
          `;
        }

        // Section 3: Safety Features
        if (checklistData.safety_features) {
          contentSections += `
            <div class="content-section">
              <h3>3. There are several safety features at the yard that all visitors must be informed of:</h3>
              <ul>
                ${checklistData.safety_features.map(feature => `<li>${feature.description}</li>`).join('')}
              </ul>
            </div>
          `;
        }

        // Section 4: NO GO AREAS
        if (checklistData.no_go_areas) {
          contentSections += `
            <div class="content-section">
              <h3>4. There are several NO GO AREAS for visitors (Applicable to drill site setup)</h3>
              <ul>
                ${checklistData.no_go_areas.restrictions.map(restriction => `<li>${restriction}</li>`).join('')}
              </ul>
            </div>
          `;
        }

        // Section 5: Workshop Area
        if (checklistData.workshop_area_rules) {
          contentSections += `
            <div class="content-section">
              <h3>5. Workshop Area</h3>
              <ul>
                ${checklistData.workshop_area_rules.restrictions.map(rule => `<li>${rule}</li>`).join('')}
              </ul>
            </div>
          `;
        }
      } else {
        // HSE-024 format (original structure)
        contentSections = `
          <div class="content-section">
            <h3>1. All visitors to any of TITAN DRILLING drill sites must be wearing adequate PPE</h3>
            <ul>
              <li>a) Hard Hat</li>
              <li>b) Safety Glasses</li>
              <li>c) Hearing Protection</li>
              <li>d) Safety Boots</li>
              <li>e) Reflective Clothing</li>
              <li>f) Dust Masks (RC Drilling)</li>
            </ul>
          </div>

          <div class="content-section">
            <h3>2. There are several safety features on all sites that all visitors must be informed of</h3>
            <ul>
              <li>The location of First Aid Kits;</li>
              <li>The location of all Fire Extinguishers;</li>
              <li>The location of TITAN DRILLING Emergency procedures/Assembly point;</li>
              <li>A copy of MSDS' of all products used on site.</li>
            </ul>
          </div>

          <div class="content-section">
            <h3>3. There are several NO GO AREAS for visitors (Applicable to drill sites)</h3>
            <ul>
              <li>Anywhere on the mast assembly or operating area;</li>
              <li>Anywhere on the main frame (deck area / back) of the rig itself;</li>
              <li>Within 3 Meters around the actual rod string whilst drilling operations are underway;</li>
              <li>Never cross the high-pressure air hose connected from compressor to Drill rig (RC);</li>
              <li>Do not operate or assist on the rig - you are not trained to do so. Our crews are not permitted to allow it.</li>
            </ul>
          </div>
        `;
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>${checklist.document_name || 'HSE Induction Report'}</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 20px;
              padding: 0;
              font-size: 12px;
            }
            .header-row {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 10px;
            }
            .logo-left {
              display: flex;
              align-items: flex-start;
            }
            .logo-left img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .doc-info-box {
              width: 340px;
            }
            .doc-info-box table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 12px;
            }
            .doc-info-box td {
              border: 1px solid #222;
              padding: 4px 8px;
              text-align: left;
            }
            .title-section {
              text-align: center;
              margin: 10px 0 0 0;
            }
            .title-section h2 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              text-decoration: underline;
            }
            .content-section {
              margin: 15px 0;
            }
            .content-section h3 {
              font-size: 14px;
              font-weight: bold;
              margin: 10px 0 5px 0;
            }
            .content-section ul {
              margin: 5px 0;
              padding-left: 20px;
            }
            .content-section li {
              margin: 2px 0;
            }
            .induction-table {
              width: 100%;
              border-collapse: collapse;
              margin: 10px 0;
            }
            .induction-table th,
            .induction-table td {
              border: 1px solid #000;
              padding: 5px;
              text-align: left;
            }
            .induction-table th {
              background-color: #f2f2f2;
              font-weight: bold;
            }
            .visitor-table {
              width: 100%;
              border-collapse: collapse;
              margin: 10px 0;
            }
            .visitor-table th,
            .visitor-table td {
              border: 1px solid #000;
              padding: 5px;
              text-align: left;
              vertical-align: top;
            }
            .visitor-table th {
              background-color: #f2f2f2;
              font-weight: bold;
            }
            .signature-cell {
              text-align: center;
              vertical-align: middle;
              min-height: 50px;
            }
            .safety-statement {
              font-weight: bold;
              font-style: italic;
              text-decoration: underline;
              text-align: center;
              margin: 15px 0;
              padding: 10px;
              background-color: #f9f9f9;
            }
            .induction-scope {
              font-weight: bold;
              text-align: center;
              margin: 15px 0;
            }
          </style>
        </head>
        <body>
          <div class="header-row">
            <div class="logo-left">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
            </div>
            <div class="doc-info-box">
              <table>
                <tr><td>Document Name</td><td>${checklist.document_name || 'HSE Induction Report'}</td></tr>
                <tr><td>Document ID</td><td>${checklist.document_id || ''}</td></tr>
                <tr><td>Version Date</td><td>${checklist.version_date || ''}</td></tr>
                <tr><td>Version</td><td>${checklist.version || ''}</td></tr>
                <tr><td>Approved</td><td>${checklist.approved_by || 'HSM'}</td></tr>
                <tr><td>Request #</td><td>${requestNumber}</td></tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h2>${isHSE101 ? 'TITAN DRILLING' : 'DRILL SITE VISITORS INDUCTION'}</h2>
          </div>

          ${contentSections}

          <div class="content-section">
            <h3>Please check each category as inducted. [✔]</h3>
            <table class="induction-table">
              <tr>
                <th>CATEGORY</th>
                <th>CHECKED</th>
              </tr>
              ${inductionCategoriesRows}
            </table>
          </div>

          <div class="safety-statement">
            TITAN DRILLING STRIVES TO MAINTAIN A SAFE AND ACCIDENT FREE WORK PLACE, YOUR CO-OPERATION WILL BE APPRECIATED
          </div>

          <div class="content-section">
            <p>This is to signify that I have been inducted as a visitor to the TITAN DRILLING area of operation and understand the terms and conditions under which I am able to visit the site.</p>
            
            <h3>VISITOR(S):</h3>
            <table class="visitor-table">
              <tr>
                <th>NAME</th>
                <th>SIGNATURE</th>
                <th>COMPANY</th>
                <th>DATE</th>
              </tr>
              ${visitorRows}
            </table>
          </div>

          <div class="content-section">
            <h3>TITAN DRILLING REPRESENTATIVE</h3>
            <p><strong>NAME:</strong> ${representativeDetails.name || ''}</p>
            <p><strong>DESIGNATION:</strong> ${representativeDetails.designation || ''}</p>
          </div>

          <div class="induction-scope">
            ${isHSE101 ? 'THIS INDUCTION IS FOR VISITORS TO KIMITETO WHO HAVE NOT UNDERGONE A FULL TITAN DRILLING INDUCTION' : 'THIS INDUCTION IS FOR VISITORS TO THE SITES WHO HAVE NOT UNDERGONE A FULL TITAN DRILLING INDUCTION'}
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `HSE_Induction_Report_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };



  const renderVisitorSection = (visitor, index) => (
    <View key={visitor.id} style={styles.visitorSection}>
      <View style={styles.visitorHeader}>
        <Text style={styles.visitorTitle}>Visitor {index + 1}</Text>
        {visitors.length > 1 && (
          <TouchableOpacity
            style={styles.removeButton}
            onPress={() => removeVisitor(index)}
          >
            <MaterialIcons name="delete" size={20} color="#FF0000" />
          </TouchableOpacity>
        )}
      </View>
      
      <View style={styles.inputGroup}>
        <Text style={styles.inputLabel}>Visitor Name *</Text>
        <TextInput
          style={styles.textInput}
          value={visitor.name}
          onChangeText={(text) => handleVisitorDetailChange(index, 'name', text)}
          placeholder="Enter visitor name"
        />
      </View>
      
      <View style={styles.inputGroup}>
        <Text style={styles.inputLabel}>Company *</Text>
        <TextInput
          style={styles.textInput}
          value={visitor.company}
          onChangeText={(text) => handleVisitorDetailChange(index, 'company', text)}
          placeholder="Enter company name"
        />
      </View>
      
      <View style={styles.inputGroup}>
        <Text style={styles.inputLabel}>Date</Text>
        <TouchableOpacity 
          style={styles.dateInput}
          onPress={() => handleDatePress(index)}
        >
          <Text style={styles.dateText}>{visitor.date}</Text>
          <MaterialIcons name="calendar-today" size={20} color="#015185" />
        </TouchableOpacity>
      </View>
      
      <View style={styles.inputGroup}>
        <Text style={styles.inputLabel}>Visitor Signature *</Text>
        {visitor.signatureStatus && visitor.signatureData ? (
          <View style={styles.signatureContainer}>
            <Text style={styles.signatureStatus}>Signature Completed</Text>
            <Image
              source={{ uri: visitor.signatureData }}
              style={styles.signatureImage}
              resizeMode="contain"
            />
            <TouchableOpacity
              style={styles.resignButton}
              onPress={() => handleSignaturePress(index)}
            >
              <Text style={styles.resignButtonText}>Re-sign</Text>
            </TouchableOpacity>
          </View>
        ) : (
          <TouchableOpacity 
            style={styles.signatureBox} 
            onPress={() => handleSignaturePress(index)}
          >
            <MaterialIcons name="edit" size={40} color="#015185" />
            <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
          </TouchableOpacity>
        )}
      </View>
    </View>
  );

  const canvasStyle = `
    .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
    .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
    canvas { height: 100% !important; width: 100% !important; }
  `;

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
       




        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Visitor Details</Text>
          {visitors.map((visitor, index) => renderVisitorSection(visitor, index))}
          
          {/* Add Visitor Button at the bottom */}
          {visitors.length < 5 && (
            <TouchableOpacity style={styles.addVisitorButton} onPress={addVisitor}>
              <MaterialIcons name="add" size={20} color="#fff" />
              <Text style={styles.addVisitorButtonText}>Add Another Visitor</Text>
            </TouchableOpacity>
          )}
        </View>

        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Titan Drilling Representative</Text>
          <View style={styles.inputGroup}>
            <Text style={styles.inputLabel}>Representative Name *</Text>
            <TextInput
              style={styles.textInput}
              value={representativeDetails.name}
              onChangeText={(text) => handleRepresentativeDetailChange('name', text)}
              placeholder="Enter representative name"
              placeholderTextColor="#888"
            />
          </View>
          <View style={styles.inputGroup}>
            <Text style={styles.inputLabel}>Designation *</Text>
            <TextInput
              style={styles.textInput}
              value={representativeDetails.designation}
              onChangeText={(text) => handleRepresentativeDetailChange('designation', text)}
              placeholder="Enter designation"
              placeholderTextColor="#888"
            />
          </View>
        </View>

        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Induction Statement</Text>
          <Text style={styles.statementText}>
            This is to signify that I have been inducted as a visitor to the TITAN DRILLING area of operation and understand the terms and conditions under which I am able to visit the site.
          </Text>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(2)}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, isLoading && styles.disabledButton]}
          onPress={handleSubmit}
          disabled={isLoading}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* SyncModal for Loading */}
      <SyncModal visible={isLoading} message="Generating Report" />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              HSE Induction Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={new Date()}
          mode="date"
          display="default"
          onChange={handleDateChange}
        />
      )}

      {/* Signature Modal with SignatureCanvas */}
      <Modal
        visible={showSignatureModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowSignatureModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Visitor {currentVisitorIndex + 1} Signature</Text>
            <Text style={styles.modalSubtitle}>Please sign below to confirm induction</Text>
            
            <View style={styles.signatureCanvasContainer}>
              <SignatureCanvas
                ref={signatureRef}
                onOK={handleSignatureData}
                onEmpty={() => {
                  console.log('Signature canvas is empty');
                  capturedSignature.current = null;
                }}
                onBegin={() => console.log('User started drawing')}
                onEnd={() => console.log('User finished drawing')}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={canvasStyle}
                penColor="#000000"
                backgroundColor="#ffffff"
                minStrokeWidth={2}
                maxStrokeWidth={4}
                onError={(error) => {
                  console.error('SignatureCanvas WebView error:', error);
                  Alert.alert('Error', 'Failed to load signature canvas.');
                }}
              />
            </View>
            
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={styles.modalCancelButton}
                onPress={() => setShowSignatureModal(false)}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalClearButton}
                onPress={handleSignatureClear}
              >
                <Text style={styles.modalClearButtonText}>Clear</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalConfirmButton}
                onPress={handleSignatureConfirm}
              >
                <Text style={styles.modalConfirmButtonText}>Confirm</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingContainer: {
    alignItems: 'center',
    paddingVertical: 15,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '600',
    color: '#015185',
    marginTop: 5,
  },
  subheadingText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 5,
  },
  checklistInfo: {
    marginBottom: width * 0.04,
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  checklistId: {
    fontSize: 14,
    color: '#666',
  },

  section: {
    marginBottom: width * 0.04,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  visitorSection: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  visitorHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  visitorTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
  },
  removeButton: {
    padding: 5,
  },
  inputGroup: {
    marginBottom: width * 0.04,
  },
  inputLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  dateInput: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
  },
  dateText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
  },
  signatureImage: {
    width: width * 0.5,
    height: height * 0.12,
    borderRadius: 10,
    backgroundColor: '#f5f5f5',
  },
  resignButton: {
    backgroundColor: '#015185',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
  },
  resignButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  signatureBox: {
    height: height * 0.15,
    backgroundColor: '#f5f5f5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#ddd',
    borderStyle: 'dashed',
  },
  signaturePlaceholder: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  addVisitorButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    marginTop: 10,
  },
  addVisitorButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginLeft: 8,
  },
  statementText: {
    fontSize: 14,
    color: '#333',
    lineHeight: 20,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '95%',
    maxHeight: '90%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 5,
  },
  modalSubtitle: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginBottom: 20,
  },
  signatureCanvasContainer: {
    alignItems: 'center',
    marginBottom: 20,
    height: height * 0.4,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 10,
    overflow: 'hidden',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    marginRight: 5,
  },
  modalCancelButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalClearButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#FF6B6B',
    marginHorizontal: 5,
  },
  modalClearButtonText: {
    color: '#FF6B6B',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalConfirmButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    backgroundColor: '#015185',
    marginLeft: 5,
  },
  modalConfirmButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '60%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
});

export default HSE101_024Step3;