
import React, { useState, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  Image,
  TextInput,
  Platform,
  ToastAndroid,
  Dimensions,
  Modal
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const HSE41_42_step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [selectedFiles, setSelectedFiles] = useState({});
  const [numericInputs, setNumericInputs] = useState({});
  const [notes, setNotes] = useState({});
  const [hseId, setHseId] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [showNoteModal, setShowNoteModal] = useState(false);
  const [currentNoteItem, setCurrentNoteItem] = useState(null);
  const [noteText, setNoteText] = useState('');

  useEffect(() => {
    loadHseData();
    
    // No cleanup needed - data should persist between steps
  }, []);

  // Reload notes and selections when screen is focused
  useFocusEffect(
    useCallback(() => {
      if (hseId) {
        loadNotes();
        loadSelections();
      }
    }, [hseId])
  );

  const loadHseData = async () => {
    try {
      console.log('Loading HSE data...');
      setIsLoading(true);
      const savedData = await AsyncStorage.getItem('hseStep1');
      console.log('Saved data:', savedData);
      
      if (savedData) {
        const data = JSON.parse(savedData);
        console.log('Parsed data:', data);
        
        setSelectedChecklist(data.selectedChecklist);
        setHseId(data.id);
        await loadChecklistData(data.selectedChecklist, data.id);
      } else {
        console.log('No saved data found');
        setSections([]);
      }
    } catch (error) {
      console.error('Error loading HSE data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load HSE data.', position: 'top' });
      setSections([]);
    } finally {
      setIsLoading(false);
    }
  };

  const loadNotes = async () => {
    try {
      if (!hseId) return;
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`hseNotes_${hseId}`) || '{}');
      setNotes(savedNotes);
    } catch (error) {
      console.error('Error loading notes:', error);
    }
  };

  const loadSelections = async () => {
    try {
      if (!hseId) return;
      const savedResults = JSON.parse(
        await AsyncStorage.getItem(`hseResults_${hseId}`) || '{}'
      );
      setSelections(savedResults.selections || {});
      setSelectedFiles(savedResults.selectedFiles || {});
      setNumericInputs(savedResults.numericInputs || {});
    } catch (error) {
      console.error('Error loading selections:', error);
    }
  };

  const loadChecklistData = async (checklist, currentHseId) => {
    try {
      console.log('Loading checklist data:', checklist);
      if (!checklist) {
        console.log('No checklist found');
        setSections([]);
        return;
      }
      
      if (!checklist.checklist) {
        console.log('No checklist.checklist found');
        setSections([]);
        return;
      }

      // Process checklist items
      const sectionEntries = Object.entries(checklist.checklist);
      console.log('Section entries:', sectionEntries);
      
      const dynamicSections = sectionEntries
        .map(([sectionKey, sectionValue], idx) => {
          console.log(`Processing section ${sectionKey}:`, sectionValue);
          if (Array.isArray(sectionValue) && sectionValue.length > 0) {
            const subItems = sectionValue.map((item, i) => {
              if (typeof item === 'object' && item !== null) {
                // Handle object items with options
                const itemKey = Object.keys(item)[0];
                const options = item[itemKey];
                return {
                  key: `${checklist.document_id}_${sectionKey}_${i}`,
                  itemName: itemKey,
                  category: sectionKey,
                  inputType: 'Yes/No',
                  sortOrder: i,
                  isInspectionRequired: false,
                  description: '',
                  instruction: '',
                  options: options,
                };
              }
              return null;
            }).filter(item => item !== null);

            if (subItems.length > 0) {
              return {
                key: `${checklist.document_id}_${sectionKey}`,
                name: sectionKey,
                subItems,
              };
            }
          }
          return null;
        })
        .filter((section) => section && Array.isArray(section.subItems) && section.subItems.length > 0);

      console.log('Dynamic sections:', dynamicSections);
      setSections(dynamicSections);

      // Initialize expanded sections
      const initialExpanded = {};
      dynamicSections.forEach((section) => {
        initialExpanded[section.key] = false;
      });
      setExpandedSections(initialExpanded);
      
      // If no sections found, show error
      if (dynamicSections.length === 0) {
        console.log('No sections found in checklist data');
        Toast.show({
          type: 'error',
          text1: 'No Data',
          text2: 'No checklist sections found. Please check your data.',
          position: 'top',
        });
      }

      // Initialize selections, files, and numeric inputs
      const savedResults = JSON.parse(
        await AsyncStorage.getItem(`hseResults_${currentHseId}`) || '{}'
      );
      const initialSelections = {};
      const initialFiles = {};
      const initialNumericInputs = {};
      dynamicSections
        .flatMap((section) => section.subItems)
        .forEach((item) => {
          initialSelections[item.key] = savedResults.selections?.[item.key] || null;
          initialFiles[item.key] = savedResults.selectedFiles?.[item.key] || null;
          initialNumericInputs[item.key] = savedResults.numericInputs?.[item.key] || '';
        });

      setSelections(initialSelections);
      setSelectedFiles(initialFiles);
      setNumericInputs(initialNumericInputs);
    } catch (error) {
      console.error('Error loading checklist data:', error);
      setSections([]);
    }
  };

  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Yes/No':
        return ['Yes', 'No'];
      case 'Okay/Faulty/N/A':
        return ['Yes', 'No'];
      case 'Ok/Reject/N/A':
        return ['Yes', 'No'];
      case 'Ok/Not Ok':
        return ['Yes', 'No'];
      case 'Numeric':
        return [];
      default:
        if (typeof inputType === 'string' && inputType.includes('/')) {
          const options = inputType.split('/').map((opt) => opt.trim()).filter((opt) => opt);
          if (options.length > 0) {
            return options;
          }
        }
        return ['Yes', 'No'];
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = {
        ...prev,
        [itemKey]: option,
      };
      saveHseResults(updatedSelections, selectedFiles, numericInputs);
      return updatedSelections;
    });
    if (Platform.OS === 'android') {
      ToastAndroid.show(`Selected: ${option}`, ToastAndroid.SHORT);
    }
  };

  const handleNumericInput = async (itemKey, value) => {
    setNumericInputs((prev) => {
      const updatedNumericInputs = {
        ...prev,
        [itemKey]: value.replace(/[^0-9]/g, ''),
      };
      setSelections((prevSelections) => {
        const updatedSelections = {
          ...prevSelections,
          [itemKey]: value ? value : null,
        };
        saveHseResults(updatedSelections, selectedFiles, updatedNumericInputs);
        return updatedSelections;
      });
      return updatedNumericInputs;
    });
  };

  const handleFilePick = (itemKey) => {
    const selection = selections[itemKey];
    if (!selection) {
      if (Platform.OS === 'android') {
        ToastAndroid.show('Please make a selection first', ToastAndroid.SHORT);
      } else {
        Alert.alert('Selection Required', 'Please make a selection first');
      }
      return;
    }

    Alert.alert(
      'Select Image',
      'Choose an option',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Take Picture', onPress: () => pickImage(itemKey, 'camera') },
        { text: 'Choose from Gallery', onPress: () => pickImage(itemKey, 'gallery') },
      ],
      { cancelable: true }
    );
  };

  const handleNotePress = async (itemKey) => {
    const selection = selections[itemKey];
    if (!selection) {
      if (Platform.OS === 'android') {
        ToastAndroid.show('Please make a selection first', ToastAndroid.SHORT);
      } else {
        Alert.alert('Selection Required', 'Please make a selection first');
      }
      return;
    }

    setCurrentNoteItem(itemKey);
    setNoteText(notes[itemKey] || '');
    setShowNoteModal(true);
  };

  const saveNote = async () => {
    if (currentNoteItem) {
      const updatedNotes = { ...notes, [currentNoteItem]: noteText };
      setNotes(updatedNotes);
      await AsyncStorage.setItem(`hseNotes_${hseId}`, JSON.stringify(updatedNotes));
      setShowNoteModal(false);
      setCurrentNoteItem(null);
      setNoteText('');
      Toast.show({
        type: 'success',
        text1: 'Note Saved',
        text2: 'Note has been saved successfully.',
        position: 'top',
      });
    }
  };

  const clearAllHseData = async () => {
    try {
      // Clear all stored data from all steps
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      
      // Also clear any HSE results and notes if they exist
      if (hseId) {
        await AsyncStorage.removeItem(`hseResults_${hseId}`);
        await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      }
      
      Toast.show({
        type: 'success',
        text1: 'Data Cleared',
        text2: 'All inspection data has been cleared successfully.',
        position: 'top',
      });
    } catch (error) {
      console.error('Error clearing HSE data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to clear some data. Please try again.',
        position: 'top',
      });
    }
  };

  const handleBackPress = () => {
    // Simply go back to step 1, don't clear data
    setActiveStep(1);
  };

  const pickImage = async (itemKey, source) => {
    const options = {
      mediaType: 'photo',
      maxWidth: 500,
      maxHeight: 500,
      quality: 0.8,
    };

    const callback = async (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        console.error('Image Picker Error: ', response.errorMessage);
        Alert.alert('Error', response.errorMessage || 'Failed to pick image');
      } else {
        const uri = response.assets[0].uri;
        setSelectedFiles((prev) => {
          const updatedFiles = {
            ...prev,
            [itemKey]: uri,
          };
          saveHseResults(selections, updatedFiles, numericInputs);
          return updatedFiles;
        });
      }
    };

    if (source === 'camera') {
      launchCamera(options, callback);
    } else {
      launchImageLibrary(options, callback);
    }
  };

  const calculateUnsafeItems = () => {
    const allItems = sections.flatMap((section) => section.subItems);
    return allItems.reduce((count, item) => {
      const selection = selections[item.key];
      if (selection && selection === 'No' && item.inputType !== 'Numeric') {
        return count + 1;
      }
      return count;
    }, 0);
  };

  const validateSelections = () => {
    const allItems = sections.flatMap((section) => section.subItems);
    if (allItems.length === 0) {
      Alert.alert('No Items', 'The selected checklist has no valid items.');
      return false;
    }

    const hasAtLeastOneSelection = allItems.some((item) => selections[item.key] !== null);
    if (!hasAtLeastOneSelection) {
      Alert.alert('Incomplete Assessment', 'Please make at least one selection.');
      return false;
    }

    const requiredItems = allItems.filter((item) => item.isInspectionRequired);
    const allRequiredSelected = requiredItems.every((item) => selections[item.key] !== null);
    if (!allRequiredSelected) {
      Alert.alert('Incomplete Assessment', 'Please complete all required items.');
      return false;
    }

    return true;
  };

  const saveHseResults = async (selectionsToSave, filesToSave, numericInputsToSave) => {
    try {
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`hseNotes_${hseId}`) || '{}');
      const hseResults = {
        selections: selectionsToSave,
        selectedFiles: filesToSave,
        numericInputs: numericInputsToSave,
        notes: savedNotes,
        sections,
        checklistTitle: selectedChecklist?.document_name,
        unsafeItems: calculateUnsafeItems(),
        timestamp: new Date().toISOString(),
      };
      await AsyncStorage.setItem(
        `hseResults_${hseId}`,
        JSON.stringify(hseResults)
      );
    } catch (error) {
      console.error('Error saving HSE results:', error);
    }
  };

  const handleNext = async () => {
    if (validateSelections()) {
      try {
        await saveHseResults(selections, selectedFiles, numericInputs);
        const assessmentDataToSave = {
          id: Date.now().toString(),
          checklistId: selectedChecklist?.document_id,
          checklistName: selectedChecklist?.document_name,
          assessmentData: selections,
          sections: sections,
          notes: notes,
          totalItems: sections.flatMap(section => section.subItems).length,
          unsafeItems: calculateUnsafeItems(),
          completedAt: new Date().toISOString(),
        };

        await AsyncStorage.setItem('hseStep2', JSON.stringify(assessmentDataToSave));
        Toast.show({
          type: 'success',
          text1: 'Assessment Saved',
          text2: 'Assessment data saved successfully.',
          position: 'top',
        });
        
        // Add a small delay to ensure data is saved before moving to step 3
        setTimeout(() => {
          try {
            console.log('Moving to step 3 from HSE41_42 step 2');
            setActiveStep(3);
          } catch (stepError) {
            console.error('Error moving to step 3:', stepError);
            Toast.show({
              type: 'error',
              text1: 'Navigation Error',
              text2: 'Failed to move to step 3. Please try again.',
              position: 'top',
            });
          }
        }, 1000);
      } catch (error) {
        console.error('Error saving HSE results:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to save assessment data.',
          position: 'top',
        });
      }
    }
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({
      ...prev,
      [sectionKey]: !prev[sectionKey],
    }));
  };

  const renderItem = (item, index) => {
    const selectedOption = selections[item.key];
    const selectedFileUri = selectedFiles[item.key];
    const numericValue = numericInputs[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {item.itemName}
            {item.isInspectionRequired && <Text style={styles.requiredText}> (Required)</Text>}
          </Text>
          {item.description && (
            <Text style={styles.detailText}>Description: {item.description}</Text>
          )}
          {item.instruction && (
            <Text style={styles.detailText}>Instruction: {item.instruction}</Text>
          )}
          <View style={styles.itemRow}>
            {item.inputType === 'Numeric' ? (
              <TextInput
                style={styles.numericInput}
                value={numericValue}
                onChangeText={(text) => handleNumericInput(item.key, text)}
                placeholder="Enter number"
                keyboardType="numeric"
              />
            ) : (
              <View style={styles.optionsContainer}>
                {options.map((option) => {
                  const isSelected = selectedOption === option;
                  return (
                    <TouchableOpacity
                      key={option}
                      style={[
                        styles.optionButton,
                        isSelected && styles.selectedOptionButton,
                        isSelected && option === 'Yes' && styles.yesButton,
                        isSelected && option === 'No' && styles.noButton,
                        isSelected && !['Yes', 'No'].includes(option) && styles.customOptionButton,
                      ]}
                      onPress={() => handleSelectOption(item.key, option)}
                    >
                      <Text
                        style={[
                          styles.optionText,
                          isSelected && styles.optionTextSelected,
                        ]}
                      >
                        {option}
                      </Text>
                    </TouchableOpacity>
                  );
                })}
              </View>
            )}
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => handleNotePress(item.key)}
              >
                <MaterialIcons name="note-add" size={24} color="#333" />
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => handleFilePick(item.key)}
              >
                <MaterialIcons name="camera-alt" size={24} color="#333" />
              </TouchableOpacity>
            </View>
          </View>
          {itemNote && (
            <Text style={styles.noteText}>
              Note: {itemNote}
            </Text>
          )}
        </View>
        {selectedFileUri && (
          <Image
            source={{ uri: selectedFileUri }}
            style={styles.selectedImage}
            onError={(e) => console.error('Image error:', e.nativeEvent.error)}
          />
        )}
        <View style={styles.itemBorder} />
      </View>
    );
  };

    return (
    <View style={styles.container}>
      <Text style={styles.checklistTitle}>
        {selectedChecklist?.document_name || 'HSE Assessment'}
      </Text>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>
            Tap on section to expand
          </Text>
        </View>

        {isLoading ? (
          <View style={styles.loadingContainer}>
            <Text style={styles.loadingText}>Loading checklist data...</Text>
          </View>
        ) : sections.length > 0 ? (
          sections.map((section) => (
            <View key={section.key}>
              <TouchableOpacity
                style={styles.sectionHeader}
                onPress={() => toggleSection(section.key)}
              >
                <View style={styles.sectionHeaderContent}>
                  <AntDesign
                    name={expandedSections[section.key] ? 'caretup' : 'caretdown'}
                    size={20}
                    color={expandedSections[section.key] ? 'red' : 'black'}
                  />
                  <Text style={styles.sectionHeaderText}>{section.name}</Text>
                </View>
              </TouchableOpacity>
              {expandedSections[section.key] && (
                <View style={styles.sectionContent}>
                  {section.subItems.length > 0 ? (
                    section.subItems.map((item, index) => renderItem(item, index))
                  ) : (
                    <Text style={styles.emptySectionText}>No assessment items</Text>
                  )}
                </View>
              )}
            </View>
          ))
        ) : (
          <View style={styles.emptyContainer}>
            <Text style={styles.emptySectionText}>No sections available</Text>
            <Text style={styles.emptySubText}>Please check your checklist data</Text>
          </View>
        )}
      </ScrollView>

      <Modal
        visible={showNoteModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowNoteModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note</Text>
            <TextInput
              style={styles.noteTextInput}
              value={noteText}
              onChangeText={setNoteText}
              placeholder="Enter your note here..."
              multiline={true}
              numberOfLines={6}
              textAlignVertical="top"
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity style={styles.modalButton} onPress={() => setShowNoteModal(false)}>
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.saveButton]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.saveButtonText]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  requiredText: {
    fontSize: 14,
    color: '#e74c3c',
    fontWeight: '600',
  },
  detailText: {
    fontSize: 14,
    color: '#666',
    marginBottom: 4,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  yesButton: {
    backgroundColor: '#015185',
  },
  noButton: {
    backgroundColor: '#FF4B2B',
  },
  naButton: {
    backgroundColor: '#2193b0',
  },
  customOptionButton: {
    backgroundColor: '#6c757d',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  numericInput: {
    flex: 1,
    height: 40,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    fontSize: 14,
    marginRight: 10,
  },
  iconsContainer: {
    flexDirection: 'row',
  },
  iconButton: {
    padding: 10,
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    marginTop: 10,
  },
  selectedImage: {
    width: 40,
    height: 40,
    marginTop: 5,
    alignSelf: 'center',
  },
  noteText: {
    fontSize: 14,
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  checklistTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginTop: 18,
    marginBottom: 4,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  loadingText: {
    fontSize: 18,
    color: '#015185',
    fontWeight: '600',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  emptySubText: {
    fontSize: 14,
    color: '#888',
    marginTop: 10,
    textAlign: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '90%',
    maxWidth: 400,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  noteTextInput: {
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    minHeight: 120,
    textAlignVertical: 'top',
    marginBottom: 20,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginHorizontal: 5,
    backgroundColor: '#f0f0f0',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    textAlign: 'center',
  },
  saveButtonText: {
    color: '#fff',
  },
});

export default HSE41_42_step2;