import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Alert,
  ActivityIndicator,
  Modal,
  Image,
  KeyboardAvoidingView,
  Platform,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import SignatureCanvas from 'react-native-signature-canvas';
import DateTimePicker from '@react-native-community/datetimepicker';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import HSE_132Json from './HSE_132_Json.js';

const { width, height } = Dimensions.get('window');

const HSE_132_step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [step1Data, setStep1Data] = useState(null);
  const [step2Data, setStep2Data] = useState(null);
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [isGeneratingPDF, setIsGeneratingPDF] = useState(false);
  const [reportId, setReportId] = useState(null);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  
  // Date and time picker states
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [showTimePicker, setShowTimePicker] = useState(false);
  const [currentDateField, setCurrentDateField] = useState('');
  const [currentTimeField, setCurrentTimeField] = useState('');
  
  // Signature modal states
  const [signatureModalVisible, setSignatureModalVisible] = useState(false);
  const [currentSignatureField, setCurrentSignatureField] = useState('');
  const [signatures, setSignatures] = useState({
    supervisor: {
      signatureData: null,
      signatureStatus: false
    },
    hseOfficer: {
      signatureData: null,
      signatureStatus: false
    }
  });

  const signatureRef = useRef(null);
  const capturedSignature = useRef(null);

  useEffect(() => {
    loadSavedData();
    checkNetworkStatus();
  }, []);

  // Auto-fill data after step2Data is loaded
  useEffect(() => {
    if (step2Data) {
      autoFillCurrentDateTime();
    }
  }, [step2Data]);





  // Auto-fill current date, time, and rig name from equipment
  const autoFillCurrentDateTime = () => {
    if (!step2Data || !step1Data) return;
    
    const now = new Date();
    const currentDate = now.toLocaleDateString();
    const currentTime = now.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    
    let newData = { ...step2Data };
    let hasChanges = false;
    
    // Auto-fill date if not exists
    if (!newData.dateOfInspection) {
      newData.dateOfInspection = currentDate;
      hasChanges = true;
    }
    
    // Auto-fill time if not exists
    if (!newData.time) {
      newData.time = currentTime;
      hasChanges = true;
    }
    
    // Auto-fill rig name from equipment if not exists
    if (!newData.rigName && step1Data?.selectedEquipment?.equipmentName) {
      newData.rigName = step1Data.selectedEquipment.equipmentName;
      hasChanges = true;
    }
    
    // Auto-fill location from equipment if not exists
    if (!newData.location && step1Data?.selectedEquipment?.country) {
      newData.location = step1Data.selectedEquipment.country;
      hasChanges = true;
    }
    
    // Save changes if any
    if (hasChanges) {
      setStep2Data(newData);
      AsyncStorage.setItem('hseStep2', JSON.stringify(newData));
    }
  };

  const checkNetworkStatus = async () => {
    const state = await NetInfo.fetch();
    setIsOnline(state.isConnected);
  };

  const loadSavedData = async () => {
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      
      if (savedHse) {
        const hseData = JSON.parse(savedHse);
        setStep1Data(hseData);
        setSelectedChecklist(hseData.selectedChecklist);
      }
      
      if (savedStep2) {
        const step2Data = JSON.parse(savedStep2);
        setStep2Data(step2Data);
      } else {
        // Initialize empty step2Data if none exists
        setStep2Data({});
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const handleInputChange = (key, value) => {
    const newData = { ...step2Data, [key]: value };
    setStep2Data(newData);
    // Save to AsyncStorage
    AsyncStorage.setItem('hseStep2', JSON.stringify(newData));
  };

  // Signature handling functions
  const openSignatureModal = (field) => {
    setCurrentSignatureField(field);
    setSignatureModalVisible(true);
    capturedSignature.current = null;
  };

  const closeSignatureModal = () => {
    setSignatureModalVisible(false);
    setCurrentSignatureField('');
  };

  const handleSignatureComplete = (signature) => {
    if (signature) {
      setSignatures(prev => ({
        ...prev,
        [currentSignatureField]: {
          ...prev[currentSignatureField],
          signatureData: signature,
          signatureStatus: true
        }
      }));
    }
    setSignatureModalVisible(false);
  };

  const handleSignatureClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
      capturedSignature.current = null;
    }
  };

  const handleSignatureConfirm = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature();
    }
  };

  const handleSignatureData = (signature) => {
    capturedSignature.current = signature;
    handleSignatureComplete(signature);
  };

  // Date and time picker functions
  const handleDatePress = (fieldKey) => {
    setCurrentDateField(fieldKey);
    setShowDatePicker(true);
  };

  const handleTimePress = (fieldKey) => {
    setCurrentTimeField(fieldKey);
    setShowTimePicker(true);
  };

  const handleDateChange = (event, date) => {
    if (Platform.OS === 'android') {
      setShowDatePicker(false);
    }
    
    if (date) {
      const newData = { ...step2Data, [currentDateField]: date.toLocaleDateString() };
      setStep2Data(newData);
      AsyncStorage.setItem('hseStep2', JSON.stringify(newData));
    }
  };

  const handleTimeChange = (event, time) => {
    if (Platform.OS === 'android') {
      setShowTimePicker(false);
    }
    
    if (time) {
      const newData = { ...step2Data, [currentTimeField]: time.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }) };
      setStep2Data(newData);
      AsyncStorage.setItem('hseStep2', JSON.stringify(newData));
    }
  };

  const handleDateConfirm = () => {
    if (Platform.OS === 'ios') {
      setShowDatePicker(false);
    }
  };

  const handleTimeConfirm = () => {
    if (Platform.OS === 'ios') {
      setShowTimePicker(false);
    }
  };

  const handleDateCancel = () => {
    setShowDatePicker(false);
  };

  const handleTimeCancel = () => {
    setShowTimePicker(false);
  };

  const formatDate = (dateString) => {
    if (!dateString) return 'Select Date';
    return dateString;
  };

  const formatTime = (timeString) => {
    if (!timeString) return 'Select Time';
    return timeString;
  };

  const clearAsyncStorage = async () => {
    try {
      await AsyncStorage.multiRemove(['hseStep1', 'hseStep2']);
      console.log('AsyncStorage cleared successfully');
    } catch (error) {
      console.error('Error clearing AsyncStorage:', error);
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Navigate to HSEViewPDF screen instead of opening in browser
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
        }
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  const getCurrentReportNumber = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      return (currentNumber + 1).toString().padStart(3, '0');
    } catch (error) {
      console.error('Error getting current report number:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      await counterRef.set(currentNumber + 1);
    } catch (error) {
      console.error('Error incrementing report counter:', error);
    }
  };

  // Generate dynamic checklist rows based on step 2 data
  const generateChecklistRows = () => {
    const checklistItems = [
      { key: 'oilSpillsCleaned', label: 'Oils spills cleaned' },
      { key: 'freeFromChemicalSpills', label: 'Free from chemical spills' },
      { key: 'sumpClosed', label: 'Sump closed' },
      { key: 'freeFromOtherWaste', label: 'Free from other waste' },
      { key: 'surroundingArea', label: 'Surrounding area' }
    ];

    let rows = '';
    checklistItems.forEach(item => {
      const value = step2Data?.[item.key] || '';
      const comment = step2Data?.[`${item.key}_comments`] || '';
      
      // Create table row with proper structure
      rows += `
        <tr>
          <td class="checklist-item">${item.label}</td>
          <td class="status-column">${value === 'Yes' ? '✓' : ''}</td>
          <td class="status-column">${value === 'No' ? '✓' : ''}</td>
          <td class="comments-column">${comment}</td>
        </tr>
      `;
    });
    
    // Add empty rows for additional items (as shown in the image)
    for (let i = 0; i < 3; i++) {
      rows += `
        <tr>
          <td class="checklist-item">_________________</td>
          <td class="status-column"></td>
          <td class="status-column"></td>
          <td class="comments-column">_________________</td>
        </tr>
      `;
    }
    
    return rows;
  };

  const generatePDF = async () => {
    try {
      const htmlContent = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Environmental Rehabilitation Checklist</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 20px;
              box-sizing: border-box;
              background-color: #fff;
            }
            .container {
              width: 100%;
              max-width: 800px;
              margin: 0 auto;
              border: 2px solid #000;
              box-sizing: border-box;
              background-color: #fff;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              padding: 20px;
              border-bottom: 2px solid #000;
              background-color: #f8f9fa;
            }
            .logo-section {
              display: flex;
              align-items: center;
              gap: 15px;
            }
            .logo {
              width: 80px;
              height: 80px;
            }
            .company-name {
              font-size: 24px;
              font-weight: bold;
              color: #015185;
              text-transform: uppercase;
            }
            .document-info {
              border: 2px solid #000;
              background-color: #fff;
            }
            .document-info table {
              border-collapse: collapse;
              width: 100%;
            }
            .document-info td {
              padding: 8px 12px;
              border: 1px solid #000;
              font-size: 12px;
              font-weight: bold;
              text-align: left;
            }
            .main-title {
              text-align: center;
              font-size: 20px;
              font-weight: bold;
              text-transform: uppercase;
              margin: 20px 0;
              padding: 15px;
              border-bottom: 2px solid #000;
              background-color: #f0f0f0;
            }
            .inspection-details {
              display: grid;
              grid-template-columns: 1fr 1fr;
              padding: 15px;
              border-bottom: 2px solid #000;
              gap: 0;
            }
            .inspection-details > div {
              padding: 8px 12px;
              border: 1px solid #000;
              font-weight: bold;
              font-size: 14px;
            }
            .inspection-details > div:nth-child(odd) {
              background-color: #f0f0f0;
            }
            .checklist-section {
              padding: 15px;
            }
            .checklist-title {
              font-size: 18px;
              font-weight: bold;
              text-align: center;
              margin-bottom: 15px;
              padding: 10px;
              background-color: #f0f0f0;
              border: 1px solid #000;
            }
            .checklist-table {
              width: 100%;
              border-collapse: collapse;
              border: 2px solid #000;
            }
            .checklist-table th {
              background-color: #f0f0f0;
              border: 1px solid #000;
              padding: 10px;
              font-weight: bold;
              text-align: center;
              font-size: 14px;
            }
            .checklist-table td {
              border: 1px solid #000;
              padding: 8px;
              font-size: 14px;
            }
            .checklist-item {
              text-align: left;
              padding-left: 15px;
            }
            .status-column {
              text-align: center;
              font-weight: bold;
            }
            .comments-column {
              text-align: left;
              padding-left: 15px;
            }
            .signature-section {
              padding: 15px;
              border-top: 2px solid #000;
            }
            .signature-title {
              font-size: 18px;
              font-weight: bold;
              text-align: center;
              margin-bottom: 15px;
              padding: 10px;
              background-color: #f0f0f0;
              border: 1px solid #000;
            }
            .signature-grid {
              display: grid;
              grid-template-columns: 1fr 1fr 1fr;
              gap: 0;
              margin-bottom: 20px;
            }
            .signature-grid > div {
              padding: 8px 12px;
              border: 1px solid #000;
              font-weight: bold;
              font-size: 14px;
              text-align: center;
            }
            .signature-grid > div:nth-child(odd) {
              background-color: #f0f0f0;
            }
            .signature-line {
              border-bottom: 1px dotted #000;
              min-height: 30px;
              margin: 10px 0;
            }
            .footer {
              text-align: center;
              padding: 15px;
              border-top: 1px solid #000;
              background-color: #f8f9fa;
              font-size: 12px;
              color: #666;
            }
          </style>
        </head>
        <body>
          <div class="container">
            <!-- Header with Logo and Company Name -->
            <div class="header">
              <div class="logo-section">
                <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" class="logo">
                <div class="company-name">TITAN DRILLING</div>
              </div>
              
              <!-- Document Information Table -->
              <div class="document-info">
                <table>
                  <tr><td>Document Name: HSE-FOR-8.1.1 REHABILITATION CHECKLIST</td></tr>
                  <tr><td>Document ID: HSE-132</td></tr>
                  <tr><td>Version Date: 05/05/2024</td></tr>
                  <tr><td>Version: 1</td></tr>
                  <tr><td>Approved: HSM</td></tr>
                  <tr><td>Request #: ${step1Data?.requestNumber || 'N/A'}</td></tr>
                </table>
              </div>
            </div>
            
            <!-- Main Title -->
            <div class="main-title">ENVIRONMENTAL REHABILITATION CHECKLIST</div>
            
            <!-- General Inspection Details Grid -->
            <div class="inspection-details">
              <div>RIG NAME</div>
              <div>${step2Data?.rigName || '_________________'}</div>
              <div>LOCATION</div>
              <div>${step2Data?.location || '_________________'}</div>
              <div>HOLE ID</div>
              <div>${step2Data?.holeId || '_________________'}</div>
              <div>DATE OF INSPECTION</div>
              <div>${step2Data?.dateOfInspection || '_________________'}</div>
              <div>TIME</div>
              <div>${step2Data?.time || '_________________'}</div>
            </div>
            
            <!-- Environmental Rehabilitation Checklist Table -->
            <div class="checklist-section">
              <div class="checklist-title">Environmental Rehabilitation Checklist</div>
              <table class="checklist-table">
                <thead>
                  <tr>
                    <th style="width: 50%;">Checklist Items</th>
                    <th style="width: 15%;">Yes</th>
                    <th style="width: 15%;">No</th>
                    <th style="width: 20%;">Comments</th>
                  </tr>
                </thead>
                <tbody>
                  ${generateChecklistRows()}
                </tbody>
              </table>
            </div>
            
            <!-- Signature Sections -->
            <div class="signature-section">
              <div class="signature-title">Personnel Information</div>
              
              <!-- Supervisor Section -->
              <div class="signature-grid">
                <div>SUPERVISOR NAME</div>
                <div>DATE</div>
                <div>SIGNATURE</div>
              </div>
              <div class="signature-grid">
                <div>${step2Data?.supervisorName || '_________________'}</div>
                <div>${step2Data?.supervisorDate || '_________________'}</div>
                <div>
                  ${signatures.supervisor.signatureStatus && signatures.supervisor.signatureData 
                    ? `<img src="${signatures.supervisor.signatureData}" alt="Supervisor Signature" style="width: 120px; height: 50px; object-fit: contain; border: 1px solid #ccc;"/>`
                    : '_________________'}
                </div>
              </div>
              
              <!-- HSE Officer Section -->
              <div class="signature-grid">
                <div>HSE OFFICER</div>
                <div>DATE</div>
                <div>SIGNATURE</div>
              </div>
              <div class="signature-grid">
                <div>${step2Data?.hseOfficerName || '_________________'}</div>
                <div>${step2Data?.hseOfficerDate || '_________________'}</div>
                <div>
                  ${signatures.hseOfficer.signatureStatus && signatures.hseOfficer.signatureData 
                    ? `<img src="${signatures.hseOfficer.signatureData}" alt="HSE Officer Signature" style="width: 120px; height: 50px; object-fit: contain; border: 1px solid #ccc;"/>`
                    : '_________________'}
                </div>
              </div>
            </div>
            
            <!-- Footer -->
            <div class="footer">
              <p>Generated on: ${new Date().toLocaleString()}</p>
              <p>Request Number: ${step1Data?.requestNumber || 'N/A'}</p>
              <p>Document Version: 1.0 | Approved by: HSM</p>
            </div>
          </div>
        </body>
        </html>
      `;

      const options = {
        html: htmlContent,
        fileName: `HSE_132_${step1Data?.requestNumber || Date.now()}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw new Error('Failed to generate PDF');
    }
  };

  const uploadPDFToFirebase = async (filePath) => {
    try {
      const fileName = `HSE_132_${Date.now()}.pdf`;
      const reference = storage().ref(`GlobalHSEInspections/admins/reports/${fileName}`);
      
      await reference.putFile(filePath);
      const downloadURL = await reference.getDownloadURL();
      
      return downloadURL;
    } catch (error) {
      console.error('Error uploading PDF to Firebase:', error);
      throw new Error('Failed to upload PDF to Firebase');
    }
  };

  const saveToFirebase = async (pdfURL) => {
    try {
      const userId = await getAdminUserId();
      const newReportId = step1Data?.requestNumber || `HSE-132-${Date.now()}`;
      
      console.log('Saving to Firebase with report ID:', newReportId);
      
      // Prepare step 3 data with all form inputs and signatures
      const step3DataForDb = {
        rigName: step2Data?.rigName || '',
        location: step2Data?.location || '',
        holeId: step2Data?.holeId || '',
        dateOfInspection: step2Data?.dateOfInspection || '',
        time: step2Data?.time || '',
        supervisorName: step2Data?.supervisorName || '',
        supervisorDate: step2Data?.supervisorDate || '',
        hseOfficerName: step2Data?.hseOfficerName || '',
        hseOfficerDate: step2Data?.hseOfficerDate || '',
        signatures: signatures,
        pdfURL: pdfURL,
        submittedAt: new Date().toISOString(),
        status: 'completed'
      };
      
      const reportData = {
        id: newReportId,  // Store the request number as 'id'
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        checklistType: 'HSE-132',
        checklistName: 'Environmental Rehabilitation Checklist',
        pdfDownloadUrl: pdfURL,
        step1Data: step1Data,
        step2Data: step2Data, // This contains the checklist data from step 2
        step3Data: step3DataForDb, // This contains the form data from step 3
        country: step1Data?.country,
        project: step1Data?.project,
        equipment: step1Data?.selectedEquipment,
        coordinates: step1Data?.coordinates,
        gpsAddress: step1Data?.gpsAddress,
      };

      // Use the same pattern as Hsemodule43step3.js
      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${newReportId}`);
      await dbRef.set(reportData);
      
      console.log('Report saved successfully to Firebase');
      console.log('Step 2 Data saved:', step2Data);
      console.log('Step 3 Data saved:', step3DataForDb);

      return newReportId;
    } catch (error) {
      console.error('Error saving to Firebase:', error);
      throw error;
    }
  };

  const handleSubmit = async () => {
    if (!step1Data || !step2Data) {
      Alert.alert('Error', 'Please complete steps 1 and 2 first.');
      return;
    }

    console.log('Validating form with step2Data:', step2Data);
    
    // Validate required fields
    const requiredFields = [
      'rigName',
      'location', 
      'holeId',
      'dateOfInspection',
      'time',
      'oilSpillsCleaned',
      'freeFromChemicalSpills',
      'sumpClosed',
      'freeFromOtherWaste',
      'surroundingArea',
      'supervisorName',
      'supervisorDate',
      'hseOfficerName',
      'hseOfficerDate'
    ];

    const missingFields = requiredFields.filter(field => !step2Data[field]);
    console.log('Missing fields:', missingFields);
    console.log('Available fields in step2Data:', Object.keys(step2Data));
    
    if (missingFields.length > 0) {
      const fieldLabels = {
        rigName: 'Rig Name',
        location: 'Location',
        holeId: 'Hole ID',
        dateOfInspection: 'Date of Inspection',
        time: 'Time',
        oilSpillsCleaned: 'Oil Spills Cleaned',
        freeFromChemicalSpills: 'Free from Chemical Spills',
        sumpClosed: 'Sump Closed',
        freeFromOtherWaste: 'Free from Other Waste',
        surroundingArea: 'Surrounding Area',
        supervisorName: 'Supervisor Name',
        supervisorDate: 'Supervisor Date',
        hseOfficerName: 'HSE Officer Name',
        hseOfficerDate: 'HSE Officer Date'
      };

      const missingLabels = missingFields.map(field => fieldLabels[field]).join(', ');
      Alert.alert('Required Fields Missing', `Please fill in: ${missingLabels}`);
      return;
    }

    // Validate signatures
    if (!signatures.supervisor.signatureStatus || !signatures.hseOfficer.signatureStatus) {
      Alert.alert('Missing Signatures', 'Please complete both signature fields before submitting.');
      return;
    }

    setIsSubmitting(true);
    setIsGeneratingPDF(true);

    try {
      // Generate PDF
      const pdfPath = await generatePDF();
      
      // Upload PDF to Firebase Storage
      const pdfURL = await uploadPDFToFirebase(pdfPath);
      
      // Save data to Firebase Database
      const newReportId = await saveToFirebase(pdfURL);
      
      // Set reportId for PDF viewing
      setReportId(newReportId);
      
      // Increment report counter
      await incrementReportCounter();
      
      // Clear saved data from AsyncStorage
      await clearAsyncStorage();
      
      setIsGeneratingPDF(false);
      
      Toast.show({
        type: 'success',
        text1: 'Success!',
        text2: 'Environmental Rehabilitation Checklist submitted successfully.',
        position: 'top',
      });
      
      // Show success modal instead of navigating immediately
      setShowSuccessModal(true);
      
    } catch (error) {
      console.error('Error in handleSubmit:', error);
      setIsGeneratingPDF(false);
      
      if (!isOnline) {
        // Save to offline queue
        const offlineData = {
          type: 'HSE_132',
          step1Data,
          step2Data,
          timestamp: Date.now(),
        };
        
        const offlineQueue = JSON.parse(await AsyncStorage.getItem('offlineQueue') || '[]');
        offlineQueue.push(offlineData);
        await AsyncStorage.setItem('offlineQueue', JSON.stringify(offlineQueue));
        
        // Clear HSE step data from AsyncStorage
        await clearAsyncStorage();
        
        Toast.show({
          type: 'info',
          text1: 'Offline Mode',
          text2: 'Data saved locally. Will sync when online.',
          position: 'top',
        });
        
        navigation.navigate('HSEInspectionHome');
      } else {
        Alert.alert('Error', error.message || 'Failed to submit checklist. Please try again.');
      }
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleBack = async () => {
    try {
      // Clear AsyncStorage when going back
      await clearAsyncStorage();
      console.log('AsyncStorage cleared when going back');
    } catch (error) {
      console.error('Error clearing AsyncStorage on back:', error);
    }
    setActiveStep(2);
  };

  if (!step1Data || !step2Data) {
    return (
      <View style={styles.container}>
        <Text style={styles.errorText}>
          {!step1Data ? 'Step 1 data not found. Please complete step 1 first.' : 
           !step2Data ? 'Step 2 data not found. Please complete step 2 first.' : 
           'Please complete steps 1 and 2 first.'}
        </Text>
        <Text style={styles.errorSubText}>
          Step 1: {step1Data ? '✓ Loaded' : '✗ Missing'}{'\n'}
          Step 2: {step2Data ? '✓ Loaded' : '✗ Missing'}
        </Text>
        <TouchableOpacity style={styles.backButton} onPress={handleBack}>
          <Text style={styles.backButtonText}>Go Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.refreshButton} onPress={loadSavedData}>
          <Text style={styles.refreshButtonText}>Refresh Data</Text>
        </TouchableOpacity>
      </View>
    );
  }

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        
        {/* Step 3 Header */}
        <View style={styles.stepHeader}>
          <Text style={styles.stepTitle}>Step 3: Complete Form & Signatures</Text>
          <Text style={styles.stepDescription}>Fill in the remaining details and add your signatures to complete the Environmental Rehabilitation Checklist.</Text>
        </View>



        {/* General Inspection Details Input */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>General Inspection Details</Text>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Rig Name *</Text>
            <TextInput
              style={styles.input}
              placeholder="Enter rig name"
              value={step2Data?.rigName || ''}
              onChangeText={(text) => handleInputChange('rigName', text)}
            />
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Location *</Text>
            <TextInput
              style={styles.input}
              placeholder="Enter location"
              value={step2Data?.location || ''}
              onChangeText={(text) => handleInputChange('location', text)}
            />
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Hole ID *</Text>
            <TextInput
              style={styles.input}
              placeholder="Enter hole ID"
              value={step2Data?.holeId || ''}
              onChangeText={(text) => handleInputChange('holeId', text)}
            />
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Date of Inspection *</Text>
            <TouchableOpacity
              style={styles.dateTimeInput}
              onPress={() => handleDatePress('dateOfInspection')}
            >
              <Text style={[
                styles.dateTimeInputText,
                !step2Data?.dateOfInspection && styles.dateTimePlaceholderText
              ]}>
                {formatDate(step2Data?.dateOfInspection)}
              </Text>
              <MaterialIcons name="calendar-today" size={20} color="#015185" />
            </TouchableOpacity>
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Time *</Text>
            <TouchableOpacity
              style={styles.dateTimeInput}
              onPress={() => handleTimePress('time')}
            >
              <Text style={[
                styles.dateTimeInputText,
                !step2Data?.time && styles.dateTimePlaceholderText
              ]}>
                {formatTime(step2Data?.time)}
              </Text>
              <MaterialIcons name="access-time" size={20} color="#015185" />
            </TouchableOpacity>
          </View>
        </View>

        {/* Environmental Rehabilitation Checklist Review */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Environmental Rehabilitation Checklist</Text>
          <View style={styles.checklistReview}>
            <View style={styles.checklistItem}>
              <Text style={styles.checklistItemText}>Oils spills cleaned</Text>
              <Text style={[styles.checklistStatus, step2Data?.oilSpillsCleaned === 'Yes' ? styles.statusOk : styles.statusNotOk]}>
                {step2Data?.oilSpillsCleaned || 'N/A'}
              </Text>
            </View>
            {step2Data?.oilSpillsCleaned_comments && (
              <Text style={styles.commentsText}>Comments: {step2Data.oilSpillsCleaned_comments}</Text>
            )}
            
            <View style={styles.checklistItem}>
              <Text style={styles.checklistItemText}>Free from chemical spills</Text>
              <Text style={[styles.checklistStatus, step2Data?.freeFromChemicalSpills === 'Yes' ? styles.statusOk : styles.statusNotOk]}>
                {step2Data?.freeFromChemicalSpills || 'N/A'}
              </Text>
            </View>
            {step2Data?.freeFromChemicalSpills_comments && (
              <Text style={styles.commentsText}>Comments: {step2Data.freeFromChemicalSpills_comments}</Text>
            )}
            
            <View style={styles.checklistItem}>
              <Text style={styles.checklistItemText}>Sump closed</Text>
              <Text style={[styles.checklistStatus, step2Data?.sumpClosed === 'Yes' ? styles.statusOk : styles.statusNotOk]}>
                {step2Data?.sumpClosed || 'N/A'}
              </Text>
            </View>
            {step2Data?.sumpClosed_comments && (
              <Text style={styles.commentsText}>Comments: {step2Data.sumpClosed_comments}</Text>
            )}
            
            <View style={styles.checklistItem}>
              <Text style={styles.checklistItemText}>Free from other waste</Text>
              <Text style={[styles.checklistStatus, step2Data?.freeFromOtherWaste === 'Yes' ? styles.statusOk : styles.statusNotOk]}>
                {step2Data?.freeFromOtherWaste || 'N/A'}
              </Text>
            </View>
            {step2Data?.freeFromOtherWaste_comments && (
              <Text style={styles.commentsText}>Comments: {step2Data.freeFromOtherWaste_comments}</Text>
            )}
            
            <View style={styles.checklistItem}>
              <Text style={styles.checklistItemText}>Surrounding area</Text>
              <Text style={[styles.checklistStatus, step2Data?.surroundingArea === 'Yes' ? styles.statusOk : styles.statusNotOk]}>
                {step2Data?.surroundingArea || 'N/A'}
              </Text>
            </View>
            {step2Data?.surroundingArea_comments && (
              <Text style={styles.commentsText}>Comments: {step2Data.surroundingArea_comments}</Text>
            )}
          </View>
        </View>

        {/* Personnel Information Input */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Personnel Information</Text>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Supervisor Name *</Text>
            <TextInput
              style={styles.input}
              placeholder="Enter supervisor name"
              value={step2Data?.supervisorName || ''}
              onChangeText={(text) => handleInputChange('supervisorName', text)}
            />
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Supervisor Date *</Text>
            <TouchableOpacity
              style={styles.dateTimeInput}
              onPress={() => handleDatePress('supervisorDate')}
            >
              <Text style={[
                styles.dateTimeInputText,
                !step2Data?.supervisorDate && styles.dateTimePlaceholderText
              ]}>
                {formatDate(step2Data?.supervisorDate)}
              </Text>
              <MaterialIcons name="calendar-today" size={20} color="#015185" />
            </TouchableOpacity>
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Supervisor Signature *</Text>
            {signatures.supervisor.signatureStatus && signatures.supervisor.signatureData ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: signatures.supervisor.signatureData }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={() => openSignatureModal('supervisor')}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={() => openSignatureModal('supervisor')}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>HSE Officer Name *</Text>
            <TextInput
              style={styles.input}
              placeholder="Enter HSE officer name"
              value={step2Data?.hseOfficerName || ''}
              onChangeText={(text) => handleInputChange('hseOfficerName', text)}
            />
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>HSE Officer Date *</Text>
            <TouchableOpacity
              style={styles.dateTimeInput}
              onPress={() => handleDatePress('hseOfficerDate')}
            >
              <Text style={[
                styles.dateTimeInputText,
                !step2Data?.hseOfficerDate && styles.dateTimePlaceholderText
              ]}>
                {formatDate(step2Data?.hseOfficerDate)}
              </Text>
              <MaterialIcons name="calendar-today" size={20} color="#015185" />
            </TouchableOpacity>
          </View>
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>HSE Officer Signature *</Text>
            {signatures.hseOfficer.signatureStatus && signatures.hseOfficer.signatureData ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: signatures.hseOfficer.signatureData }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={() => openSignatureModal('hseOfficer')}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={() => openSignatureModal('hseOfficer')}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>
        </View>



        {/* Submit Section */}
        <View style={styles.submitSection}>
          <Text style={styles.submitTitle}>Ready to Submit</Text>
          <Text style={styles.submitDescription}>
            Please ensure all required fields are filled and both signatures are completed before submitting. Once submitted, this report will be saved and cannot be modified.
          </Text>
        </View>
      </ScrollView>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.backButtonBottom} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.backButtonTextBottom}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity 
          style={[styles.submitButton, isSubmitting && styles.submitButtonDisabled]} 
          onPress={handleSubmit}
          disabled={isSubmitting}
        >
          {isSubmitting ? (
            <ActivityIndicator size="small" color="#fff" />
          ) : (
            <>
              <Text style={styles.submitButtonText}>Submit</Text>
              <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
            </>
          )}
        </TouchableOpacity>
      </View>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={new Date()}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
          onConfirm={handleDateConfirm}
          onCancel={handleDateCancel}
        />
      )}

      {/* Time Picker Modal */}
      {showTimePicker && (
        <DateTimePicker
          value={new Date()}
          mode="time"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleTimeChange}
          onConfirm={handleTimeConfirm}
          onCancel={handleTimeCancel}
        />
      )}

      {/* PDF Generation Loading Modal */}
      <Modal
        visible={isGeneratingPDF}
        transparent
        animationType="fade"
        onRequestClose={() => {}}
      >
        <View style={styles.loadingModalOverlay}>
          <View style={styles.loadingModalContent}>
            <ActivityIndicator size="large" color="#015185" />
            <Text style={styles.loadingModalText}>Generating PDF...</Text>
            <Text style={styles.loadingModalSubtext}>Please wait while we create your report</Text>
          </View>
        </View>
      </Modal>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              HSE Inspection Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={async () => {
                  setShowSuccessModal(false);
                  // Ensure AsyncStorage is cleared before navigation
                  await clearAsyncStorage();
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Signature Modal */}
      <Modal
        visible={signatureModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={closeSignatureModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>
                {currentSignatureField === 'supervisor' 
                  ? 'Supervisor Signature' 
                  : 'HSE Officer Signature'}
              </Text>
              <TouchableOpacity onPress={closeSignatureModal}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>
            
            <View style={styles.signatureCanvasContainer}>
              <SignatureCanvas
                ref={signatureRef}
                onOK={handleSignatureData}
                onEmpty={() => {
                  console.log('Signature canvas is empty');
                  capturedSignature.current = null;
                }}
                onBegin={() => console.log('User started drawing')}
                onEnd={() => console.log('User finished drawing')}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={`
                  .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
                  .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
                  canvas { height: 100% !important; width: 100% !important; }
                `}
                penColor="#000000"
                backgroundColor="#ffffff"
                minStrokeWidth={2}
                maxStrokeWidth={4}
                onError={(error) => {
                  console.error('SignatureCanvas WebView error:', error);
                  Alert.alert('Error', 'Failed to load signature canvas.');
                }}
              />
            </View>
            
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={styles.modalCancelButton}
                onPress={closeSignatureModal}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalClearButton}
                onPress={handleSignatureClear}
              >
                <Text style={styles.modalClearButtonText}>Clear</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalConfirmButton}
                onPress={handleSignatureConfirm}
              >
                <Text style={styles.modalConfirmButtonText}>Confirm</Text>
              </TouchableOpacity>
            </View>
          </View>
    </View>
      </Modal>
      
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: 20,
    paddingTop: 20,
  },
  stepHeader: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    marginBottom: 20,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  stepTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 8,
    textAlign: 'center',
  },
  stepDescription: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    lineHeight: 20,
  },
  scrollContent: {
    paddingBottom: 100,
  },

  section: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    marginBottom: 20,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingBottom: 10,
  },

  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 20,
    paddingVertical: 15,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  backButtonBottom: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  backButtonTextBottom: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginLeft: 8,
  },
  submitButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 8,
    backgroundColor: '#4CAF50',
  },
  submitButtonDisabled: {
    backgroundColor: '#ccc',
  },
  submitButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#fff',
    marginRight: 8,
  },
  errorText: {
    fontSize: 18,
    color: 'red',
    textAlign: 'center',
    marginTop: 100,
    marginBottom: 20,
  },
  errorSubText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginBottom: 30,
    lineHeight: 20,
  },
  backButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 8,
    marginBottom: 15,
  },
  backButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  inputContainer: {
    marginBottom: 20,
  },
  inputLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    paddingHorizontal: 15,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  signatureButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#0078D4',
    backgroundColor: '#fff',
  },
  signatureButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#0078D4',
    marginLeft: 8,
  },
  // Signature styles
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
  },
  signatureImage: {
    width: width * 0.5,
    height: height * 0.12,
    borderRadius: 10,
    backgroundColor: '#f5f5f5',
  },
  resignButton: {
    backgroundColor: '#015185',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
  },
  resignButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  signatureBox: {
    height: height * 0.15,
    backgroundColor: '#f5f5f5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#ddd',
    borderStyle: 'dashed',
  },
  signaturePlaceholder: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  // Modal styles
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '95%',
    maxHeight: '90%',
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 20,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
  },
  signatureCanvasContainer: {
    alignItems: 'center',
    marginBottom: 20,
    height: height * 0.4,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 10,
    overflow: 'hidden',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    marginRight: 5,
  },
  modalCancelButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalClearButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#FF6B6B',
    marginHorizontal: 5,
  },
  modalClearButtonText: {
    color: '#FF6B6B',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalConfirmButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    backgroundColor: '#015185',
    marginLeft: 5,
  },
  modalConfirmButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  submitSection: {
    backgroundColor: '#e8f5e8',
    margin: 15,
    padding: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#4caf50',
  },
  submitTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#2e7d32',
    marginBottom: 10,
  },
  submitDescription: {
    fontSize: 14,
    color: '#2e7d32',
    lineHeight: 20,
  },
  // Date and time input styles
  dateTimeInput: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    height: 40,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    backgroundColor: '#fff',
  },
  dateTimeInputText: {
    flex: 1,
    fontSize: 14,
    color: '#333',
  },
  dateTimePlaceholderText: {
    color: '#888',
  },
  // Loading modal styles
  loadingModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 30,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  loadingModalText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginTop: 15,
    marginBottom: 8,
  },
  loadingModalSubtext: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
  },
  // Footer button styles
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  // Success modal styles
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 30,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 15,
    marginBottom: 10,
  },
  successModalText: {
    fontSize: 16,
    color: '#333',
    textAlign: 'center',
    marginBottom: 25,
    lineHeight: 22,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 8,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  // Checklist review styles
  checklistReview: {
    marginTop: 10,
  },
  checklistItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  checklistItemText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  checklistStatus: {
    fontSize: 16,
    fontWeight: '600',
    paddingHorizontal: 12,
    paddingVertical: 4,
    borderRadius: 4,
    minWidth: 60,
    textAlign: 'center',
  },
  statusOk: {
    backgroundColor: '#4CAF50',
    color: '#fff',
  },
  statusNotOk: {
    backgroundColor: '#FF4B2B',
    color: '#fff',
  },
  commentsText: {
    fontSize: 14,
    color: '#666',
    fontStyle: 'italic',
    marginTop: 5,
    marginLeft: 20,
    paddingLeft: 10,
    borderLeftWidth: 2,
    borderLeftColor: '#0078D4',
  },
  refreshButton: {
    backgroundColor: '#007bff',
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    marginTop: 10,
    alignSelf: 'center',
  },
  refreshButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
});

export default HSE_132_step3;