import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Dimensions,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');

const HSE101_024Step2 = ({ setActiveStep = () => {} }) => {
  const navigation = useNavigation();
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [checklistData, setChecklistData] = useState(null);
  const [inductionCategories, setInductionCategories] = useState({});

  useEffect(() => {
    loadSelectedChecklist();
  }, []);

  const loadSelectedChecklist = async () => {
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      if (savedHse) {
        const hseData = JSON.parse(savedHse);
        setSelectedChecklist(hseData.selectedChecklist);
        
        // Load the checklist data based on document_id
        const hse101_024Data = require('./HSE101_024Step3.json');
        
        if (hseData.selectedChecklist?.document_id === 'HSE-024') {
          // Load HSE-024 specific data
          const checklistItem = hse101_024Data.find(item => item.Checklist_024_101);
          if (checklistItem) {
            setChecklistData(checklistItem);
            // Initialize induction categories for HSE-024
            if (checklistItem.induction_categories?.checklist) {
              const initialCategories = {};
              checklistItem.induction_categories.checklist.forEach(category => {
                initialCategories[category] = false;
              });
              setInductionCategories(initialCategories);
            }
          }
        } else if (hseData.selectedChecklist?.document_id === 'HSE-101') {
          // Load HSE-101 specific data
          const checklistItem = hse101_024Data.find(item => item.document);
          if (checklistItem) {
            setChecklistData(checklistItem);
            // Initialize induction categories for HSE-101
            if (checklistItem.induction_categories?.checklist) {
              const initialCategories = {};
              checklistItem.induction_categories.checklist.forEach(category => {
                initialCategories[category] = false;
              });
              setInductionCategories(initialCategories);
            }
          }
        }
      }
    } catch (error) {
      console.error('Error loading selected checklist:', error);
      Alert.alert('Error', 'Failed to load checklist data.');
    }
  };

  const handleCategoryToggle = (category) => {
    setInductionCategories(prev => ({
      ...prev,
      [category]: !prev[category]
    }));
  };

  const handleNext = async () => {
    try {
      // Save the inspection results
      const inspectionResults = {
        checklistTitle: selectedChecklist?.document_name,
        documentId: selectedChecklist?.document_id,
        checklistData,
        inductionCategories,
        timestamp: new Date().toISOString()
      };
      
      await AsyncStorage.setItem('hseInspectionResults', JSON.stringify(inspectionResults));
      setActiveStep(3);
    } catch (error) {
      console.error('Error saving inspection results:', error);
      Alert.alert('Error', 'Failed to save inspection results.');
    }
  };

  const renderSafetyFeatures = () => {
    if (!checklistData?.safety_features) return null;
    
    return (
      <View style={styles.section}>
        <Text style={styles.sectionTitle}>Safety Features</Text>
        {checklistData.safety_features.map((feature, index) => (
          <View key={index} style={styles.bulletItem}>
            <Text style={styles.bullet}>•</Text>
            <Text style={styles.bulletText}>{feature.description}</Text>
          </View>
        ))}
      </View>
    );
  };

  const renderPPERequirements = () => {
    if (!checklistData?.personal_protective_equipment) return null;
    
    return (
      <View style={styles.section}>
        <Text style={styles.sectionTitle}>Personal Protective Equipment</Text>
        <Text style={styles.statementText}>{checklistData.personal_protective_equipment.notes}</Text>
        
        {/* Show mandatory PPE items */}
        {checklistData.personal_protective_equipment.mandatory && (
          <>
            <Text style={styles.subsectionTitle}>Mandatory PPE:</Text>
            {checklistData.personal_protective_equipment.mandatory.map((item, index) => (
              <View key={index} style={styles.bulletItem}>
                <Text style={styles.bullet}>•</Text>
                <Text style={styles.bulletText}>{item}</Text>
              </View>
            ))}
          </>
        )}
        
        {/* Show additional PPE items */}
        {checklistData.personal_protective_equipment.additional && (
          <>
            <Text style={styles.subsectionTitle}>Additional PPE:</Text>
            {checklistData.personal_protective_equipment.additional.map((item, index) => (
              <View key={index} style={styles.bulletItem}>
                <Text style={styles.bullet}>•</Text>
                <Text style={styles.bulletText}>{item}</Text>
              </View>
            ))}
          </>
        )}
      </View>
    );
  };

  const renderNoGoAreas = () => {
    if (!checklistData?.no_go_areas) return null;
    
    return (
      <View style={styles.section}>
        <Text style={styles.sectionTitle}>No-Go Areas</Text>
        <Text style={styles.sectionDescription}>
          Applicable to: {checklistData.no_go_areas.applicable_to}
        </Text>
        {checklistData.no_go_areas.restrictions.map((restriction, index) => (
          <View key={index} style={styles.bulletItem}>
            <Text style={styles.bullet}>•</Text>
            <Text style={styles.bulletText}>{restriction}</Text>
          </View>
        ))}
      </View>
    );
  };

  const renderWorkshopRules = () => {
    if (!checklistData?.workshop_area_rules) return null;
    
    return (
      <View style={styles.section}>
        <Text style={styles.sectionTitle}>Workshop Area Rules</Text>
        <Text style={styles.sectionDescription}>
          Applicable to: {checklistData.workshop_area_rules.applicable_to}
        </Text>
        {checklistData.workshop_area_rules.restrictions.map((rule, index) => (
          <View key={index} style={styles.bulletItem}>
            <Text style={styles.bullet}>•</Text>
            <Text style={styles.bulletText}>{rule}</Text>
          </View>
        ))}
      </View>
    );
  };

  const renderInductionCategories = () => {
    if (!checklistData?.induction_categories) return null;
    
    return (
      <View style={styles.inductionSection}>
        <Text style={styles.inductionTitle}>Induction Categories</Text>
        <Text style={styles.statementText}>
          {checklistData.induction_categories.notes}
        </Text>
        {checklistData.induction_categories.checklist.map((category, index) => (
          <TouchableOpacity
            key={index}
            style={styles.categoryItem}
            onPress={() => handleCategoryToggle(category)}
            activeOpacity={0.7}
          >
            <View style={[
              styles.checkbox,
              inductionCategories[category] && styles.checkboxChecked
            ]}>
              {inductionCategories[category] && (
                <View style={styles.checkboxInner} />
              )}
            </View>
            <Text style={styles.categoryText}>{category}</Text>
          </TouchableOpacity>
        ))}
      </View>
    );
  };

  if (!selectedChecklist || !checklistData) {
    return (
      <View style={styles.container}>
        <Text style={styles.loadingText}>Loading checklist...</Text>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <View style={styles.header}>
        <TouchableOpacity onPress={() => setActiveStep(1)} style={styles.headerBackButton}>
          <MaterialIcons name="arrow-back" size={24} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>{selectedChecklist.document_name}</Text>
        <View style={styles.placeholder} />
      </View>

      <ScrollView style={styles.scrollView} showsVerticalScrollIndicator={false}>
        <View style={styles.content}>
          <View style={styles.checklistInfo}>
            <Text style={styles.checklistTitle}>{selectedChecklist.document_name}</Text>
            <Text style={styles.checklistId}>Document ID: {selectedChecklist.document_id}</Text>
          </View>

          <View style={styles.section}>
            <Text style={styles.sectionTitle}>Safety Statement</Text>
            <Text style={styles.statementText}>
              {checklistData.safety_statement || 'TITAN DRILLING strives to maintain a safe and accident-free workplace. Your co-operation will be appreciated.'}
            </Text>
          </View>

          {renderPPERequirements()}
          {renderSafetyFeatures()}
          {renderNoGoAreas()}
          {renderWorkshopRules()}
          {renderInductionCategories()}
          
          <View style={styles.section}>
            <Text style={styles.sectionTitle}>Induction Scope</Text>
            <Text style={styles.statementText}>
              {checklistData.induction_scope || 'This induction is for visitors who have not undergone a full TITAN DRILLING induction.'}
            </Text>
          </View>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={[styles.button, styles.backButton]} onPress={() => setActiveStep(1)}>
          <Text style={[styles.buttonText, styles.backButtonText]}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={[styles.button, styles.nextButton]} onPress={handleNext}>
          <Text style={[styles.buttonText, styles.nextButtonText]}>Next</Text>
        </TouchableOpacity>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  scrollView: {
    flex: 1,
  },
  content: {
    padding: 20,
  },
  header: {
    backgroundColor: '#015185',
    padding: 15,
    alignItems: 'center',
    flexDirection: 'row',
    justifyContent: 'center',
    position: 'relative',
  },
  headerBackButton: {
    position: 'absolute',
    left: 15,
    top: 15,
    zIndex: 1,
  },
  headerTitle: {
    color: '#fff',
    fontSize: 18,
    fontWeight: 'bold',
    textAlign: 'center',
  },
  placeholder: {
    width: 34,
  },
  checklistInfo: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    marginBottom: 20,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  checklistId: {
    fontSize: 14,
    color: '#666',
  },
  section: {
    backgroundColor: '#fff',
    marginBottom: 15,
    padding: 15,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  sectionTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    lineHeight: 22,
  },
  sectionDescription: {
    fontSize: 14,
    color: '#666',
    marginBottom: 15,
    fontStyle: 'italic',
  },
  statementText: {
    fontSize: 14,
    color: '#333',
    marginBottom: 10,
    lineHeight: 20,
  },
  itemContainer: {
    marginBottom: 15,
    paddingBottom: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  itemTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  itemDescription: {
    fontSize: 14,
    color: '#666',
    lineHeight: 18,
  },
  subsection: {
    marginTop: 15,
  },
  subsectionTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginTop: 10,
    marginBottom: 8,
  },
  bulletItem: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginBottom: 8,
    paddingLeft: 5,
  },
  bullet: {
    fontSize: 16,
    color: '#015185',
    marginRight: 8,
    marginTop: 2,
  },
  bulletText: {
    fontSize: 14,
    color: '#333',
    flex: 1,
    lineHeight: 20,
  },
  checkboxItem: {
    marginBottom: 12,
  },
  checkboxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  checkbox: {
    width: 24,
    height: 24,
    borderRadius: 4,
    borderWidth: 2,
    borderColor: '#015185',
    backgroundColor: '#fff',
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: 12,
  },
  checkboxChecked: {
    backgroundColor: '#015185',
  },
  checkboxLabel: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  inductionSection: {
    backgroundColor: '#fff',
    marginBottom: 15,
    padding: 15,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  inductionTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  categoryItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  categoryText: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
  },
  checkbox: {
    width: 24,
    height: 24,
    borderRadius: 4,
    borderWidth: 2,
    borderColor: '#015185',
    justifyContent: 'center',
    alignItems: 'center',
  },
  checkboxChecked: {
    backgroundColor: '#015185',
  },
  checkboxInner: {
    width: 12,
    height: 12,
    backgroundColor: '#fff',
    borderRadius: 2,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 15,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  button: {
    flex: 1,
    paddingVertical: 12,
    borderRadius: 8,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  backButton: {
    backgroundColor: '#f0f0f0',
  },
  nextButton: {
    backgroundColor: '#015185',
  },
  buttonText: {
    fontSize: 16,
    fontWeight: '600',
  },
  backButtonText: {
    color: '#015185',
  },
  nextButtonText: {
    color: '#fff',
  },
  loadingText: {
    fontSize: 16,
    textAlign: 'center',
    marginTop: 50,
    color: '#666',
  },
});

export default HSE101_024Step2;