import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Platform,
  StatusBar,
  Modal,
  Animated,
} from 'react-native';
import Pdf from 'react-native-pdf';
import LinearGradient from 'react-native-linear-gradient';
import RNFS from 'react-native-fs';
import Toast from 'react-native-toast-message';
import { PermissionsAndroid, Share } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import ReactNativeBlobUtil from 'react-native-blob-util'; // Import react-native-blob-util

const { width, height } = Dimensions.get('window');

const HSEViewPDF = ({ route, navigation }) => {
  const { pdfUrl, onGoBack } = route.params || {};
  const [localPdfPath, setLocalPdfPath] = useState(null);
  const [loading, setLoading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    console.log('PDF URL:', pdfUrl); // Debug the URL
    if (pdfUrl && (pdfUrl.startsWith('http://') || pdfUrl.startsWith('https://'))) {
      downloadPdfForViewing();
    }
    return () => {
      if (localPdfPath) {
        RNFS.unlink(localPdfPath).catch((err) => console.log('Error deleting temp file:', err));
      }
    };
  }, [pdfUrl]);

  useEffect(() => {
    if (loading) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [loading]);

  const requestStoragePermission = async () => {
    if (Platform.OS !== 'android') return true;

    try {
      if (Platform.Version >= 33) {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES,
          {
            title: 'Media Permission Required',
            message: 'This app needs access to your media to view and download PDFs.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } else {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
          {
            title: 'Storage Permission Required',
            message: 'This app needs access to your storage to download the PDF.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      }
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const downloadPdfForViewing = async () => {
    if (!pdfUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF URL found.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    setLoading(true);
    try {
      const hasPermission = await requestStoragePermission();
      if (!hasPermission) {
        throw new Error('Storage permission denied');
      }

      const tempPath = `${RNFS.DocumentDirectoryPath}/temp_report_${Date.now()}.pdf`;
      console.log('Attempting to download PDF to:', tempPath);

      const options = {
        fromUrl: pdfUrl,
        toFile: tempPath,
      };
      const result = await RNFS.downloadFile(options).promise;
      if (result.statusCode === 200) {
        setLocalPdfPath(tempPath);
        console.log('PDF downloaded for viewing to:', tempPath);
      } else {
        throw new Error(`Failed to download PDF for viewing. Status code: ${result.statusCode}`);
      }
    } catch (error) {
      console.error('Error downloading PDF for viewing:', error);
      let errorMessage = 'Failed to download PDF for viewing.';
      if (error.message.includes('permission denied')) {
        errorMessage = 'Storage permission denied. Please grant permission to view the PDF.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else {
        errorMessage = `Internal storage error: ${error.message}`;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 3000,
      });
      navigation.goBack();
    } finally {
      setLoading(false);
    }
  };

  const handleDownloadPDF = async () => {
    if (!pdfUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF URL found.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    try {
      const hasPermission = await requestStoragePermission();
      if (!hasPermission && Platform.OS === 'android') {
        throw new Error('Storage permission denied');
      }

      const fileName = `report_${Date.now()}.pdf`;
      let downloadDest;

      if (Platform.OS === 'android') {
        // On Android, use Download Manager to save to Downloads folder
        downloadDest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${fileName}`;
        const config = {
          fileCache: true,
          path: downloadDest,
          addAndroidDownloads: {
            useDownloadManager: true, // Use Android Download Manager
            notification: true, // Show notification
            title: fileName,
            description: 'Downloading PDF...',
            mime: 'application/pdf',
            mediaScannable: true, // Make file visible in Downloads app
          },
        };

        const res = await ReactNativeBlobUtil.config(config).fetch('GET', pdfUrl);
        console.log('PDF downloaded to:', res.path());

        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'PDF downloaded to Downloads folder.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      } else {
        // On iOS, save to DocumentDirectoryPath and offer to share
        downloadDest = `${ReactNativeBlobUtil.fs.dirs.DocumentDir}/${fileName}`;
        const config = {
          fileCache: true,
          path: downloadDest,
        };

        const res = await ReactNativeBlobUtil.config(config).fetch('GET', pdfUrl);
        console.log('PDF downloaded to:', res.path());

        // Share the file on iOS to allow saving to Files app
        await Share.share({
          url: `file://${res.path()}`,
          message: 'Here is your downloaded PDF.',
        });

        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'PDF downloaded and ready to share.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      }
    } catch (error) {
      console.error('Error downloading PDF:', error);
      let errorMessage = 'Failed to download PDF.';
      if (error.message.includes('permission denied')) {
        errorMessage = 'Storage permission denied. Please grant permission to download.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else {
        errorMessage = `Download error: ${error.message}`;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 3000,
      });
    }
  };

  return (
    <View style={styles.container}>
      {loading ? (
        <Modal visible={loading} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>Downloading PDF...</Text>
              </View>
            </View>
          </View>
        </Modal>
      ) : localPdfPath ? (
        <>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
            <View style={styles.headerContent}>
              <TouchableOpacity onPress={() => (typeof onGoBack === 'function' ? onGoBack() : navigation.goBack())}>
                <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>PDF Viewer</Text>
            </View>
          </LinearGradient>
          <Pdf
            source={{ uri: Platform.OS === 'android' ? `file://${localPdfPath}` : localPdfPath, cache: false }}
            style={styles.pdf}
            onError={(error) => {
              console.error('PDF rendering error:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to load PDF.',
                position: 'bottom',
                visibilityTime: 3000,
              });
              navigation.goBack();
            }}
            onLoadComplete={(numberOfPages, filePath) => {
              console.log(`PDF loaded with ${numberOfPages} pages at ${filePath}`);
            }}
          />
          <View style={styles.buttonContainer}>
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.downloadButtonGradient}>
              <TouchableOpacity style={styles.downloadButton} onPress={handleDownloadPDF}>
                <MaterialIcons name="download" size={width * 0.05} color="#fff" />
                <Text style={styles.downloadButtonText}>Download PDF</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        </>
      ) : (
        <View style={styles.errorContainer}>
          <Text style={styles.errorText}>Invalid or unavailable PDF URL</Text>
          <TouchableOpacity onPress={() => (typeof onGoBack === 'function' ? onGoBack() : navigation.goBack())}>
            <Text style={styles.closeText}>Go Back</Text>
          </TouchableOpacity>
        </View>
      )}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  pdf: {
    flex: 1,
    width: width,
    height: height - (width * 0.2 + (StatusBar.currentHeight || width * 0.1)),
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '5%',
    width: '100%',
    alignItems: 'center',
  },
  downloadButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  downloadButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  downloadButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  errorContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: width * 0.05,
  },
  errorText: {
    fontSize: width * 0.045,
    color: '#FF0000',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  closeText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});


export default HSEViewPDF