import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  StatusBar,
  ScrollView,
  TouchableOpacity,
  Dimensions,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';

// Import step components for HSE-043, HSE-044, HSE-045, HSE-126
import Hsemodule43step1 from './Hsemodule43step1';
import Hsemodule43step2 from './Hsemodule43step2';
import Hsemodule43step3 from './Hsemodule43step3';

// Import step components for HSE-041, HSE-042
import HSE41_42_step2 from './HSE41_42/HSE41_42_step2';
import HSE41_42_step3 from './HSE41_42/HSE41_42_step3';

// Import step components for HSE-101, HSE-024
import HSE101_024Step2 from './HSE101_024/HSE101_024Step2';
import HSE101_024Step3 from './HSE101_024/HSE101_024Step3';

// Import step components for HSE-020
import HSE_020Step2 from './HSE_020/HSE_020Step2';
import HSE_020Step3 from './HSE_020/HSE_020Step3';

const { height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = height * 0.2 - 36;

const Hsemoduleheader43 = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const [activeStep, setActiveStep] = useState(1);
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [steps, setSteps] = useState([]);

  useEffect(() => {
    // Get selected checklist from route params or AsyncStorage
    const getSelectedChecklist = async () => {
      try {
        const savedData = await AsyncStorage.getItem('hseStep1');
        if (savedData) {
          const data = JSON.parse(savedData);
          if (data.selectedChecklist) {
            setSelectedChecklist(data.selectedChecklist);
            setupSteps(data.selectedChecklist);
          } else {
            // If no checklist data, set up default steps
            setupDefaultSteps();
          }
        } else {
          // If no saved data at all, set up default steps
          setupDefaultSteps();
        }
      } catch (error) {
        console.error('Error loading selected checklist:', error);
        // If error occurs, set up default steps
        setupDefaultSteps();
      }
    };

    getSelectedChecklist();
  }, []);

  const setupDefaultSteps = () => {
    console.log('Setting up default steps'); // Debug log
    // Default steps for HSE-043, HSE-044, HSE-045, HSE-126
    const stepsArray = [
      {
        id: 1,
        title: 'Step 1',
        subtitle: 'HSE Information',
        component: <Hsemodule43step1 setActiveStep={setActiveStep} />,
      },
      {
        id: 2,
        title: 'Step 2',
        subtitle: 'HSE Assessment',
        component: <Hsemodule43step2 setActiveStep={setActiveStep} />,
      },
      {
        id: 3,
        title: 'Step 3',
        subtitle: 'HSE Review',
        component: (
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: '#fff' }}>
            <Text style={{ fontSize: 20, color: '#015185' }}>Step 3 - HSE Review</Text>
            <Text style={{ fontSize: 16, color: '#666', marginTop: 10 }}>Loading review component...</Text>
            <Hsemodule43step3 setActiveStep={setActiveStep} />
          </View>
        ),
      },
    ];
    console.log('Setting default steps:', stepsArray);
    console.log('Step 3 component type:', typeof Hsemodule43step3);
    setSteps(stepsArray);
  };

  // Reload data when screen is focused
  useFocusEffect(
    React.useCallback(() => {
      const reloadData = async () => {
        try {
          const savedData = await AsyncStorage.getItem('hseStep1');
          if (savedData) {
            const data = JSON.parse(savedData);
            if (data.selectedChecklist) {
              setSelectedChecklist(data.selectedChecklist);
              setupSteps(data.selectedChecklist);
            } else {
              setupDefaultSteps();
            }
          } else {
            setupDefaultSteps();
          }
        } catch (error) {
          console.error('Error reloading data:', error);
          setupDefaultSteps();
        }
      };
      reloadData();
    }, [])
  );

  const setupSteps = (checklist) => {
    const documentId = checklist?.document_id;
    console.log('Setting up steps for document ID:', documentId); // Debug log
    
    if (documentId === 'HSE-041' || documentId === 'HSE-042') {
      console.log('Using HSE41_42 components for document ID:', documentId); // Debug log
      // For HSE-041 and HSE-042, use 3 steps with specific components
      const stepsArray = [
        {
          id: 1,
          title: 'Step 1',
          subtitle: 'HSE Information',
          component: <Hsemodule43step1 setActiveStep={setActiveStep} />,
        },
        {
          id: 2,
          title: 'Step 2',
          subtitle: 'HSE Assessment',
          component: <HSE41_42_step2 setActiveStep={setActiveStep} />,
        },
        {
          id: 3,
          title: 'Step 3',
          subtitle: 'HSE Review',
          component: (
            <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: '#fff' }}>
              <Text style={{ fontSize: 20, color: '#015185' }}>Step 3 - HSE Review (41/42)</Text>
              <Text style={{ fontSize: 16, color: '#666', marginTop: 10 }}>Loading review component...</Text>
              <HSE41_42_step3 setActiveStep={setActiveStep} />
            </View>
          ),
        },
      ];
      console.log('Setting HSE41_42 steps:', stepsArray);
      setSteps(stepsArray);
    } else if (documentId === 'HSE-101' || documentId === 'HSE-024') {
      console.log('Using HSE101_024 components for document ID:', documentId); // Debug log
      // For HSE-101 and HSE-024, use 3 steps with specific components
      const stepsArray = [
        {
          id: 1,
          title: 'Step 1',
          subtitle: 'HSE Information',
          component: <Hsemodule43step1 setActiveStep={setActiveStep} />,
        },
        {
          id: 2,
          title: 'Step 2',
          subtitle: 'Visitor Induction',
          component: <HSE101_024Step2 setActiveStep={setActiveStep} />,
        },
        {
          id: 3,
          title: 'Step 3',
          subtitle: 'Final Summary',
          component: (
            <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: '#fff' }}>
              <Text style={{ fontSize: 20, color: '#015185' }}>Step 3 - Final Summary (101/024)</Text>
              <Text style={{ fontSize: 16, color: '#666', marginTop: 10 }}>Loading summary component...</Text>
              <HSE101_024Step3 setActiveStep={setActiveStep} />
            </View>
          ),
        },
      ];
      console.log('Setting HSE101_024 steps:', stepsArray);
      setSteps(stepsArray);
    } else if (documentId === 'HSE-020') {
      console.log('Using HSE_020 components for document ID:', documentId); // Debug log
      // For HSE-020, use 3 steps with specific components
      const stepsArray = [
        {
          id: 1,
          title: 'Step 1',
          subtitle: 'HSE Information',
          component: <Hsemodule43step1 setActiveStep={setActiveStep} />,
        },
        {
          id: 2,
          title: 'Step 2',
          subtitle: 'Task Observation Details',
          component: <HSE_020Step2 setActiveStep={setActiveStep} />,
        },
        {
          id: 3,
          title: 'Step 3',
          subtitle: 'Review & Submit',
          component: <HSE_020Step3 setActiveStep={setActiveStep} />,
        },
      ];
      console.log('Setting HSE_020 steps:', stepsArray);
      setSteps(stepsArray);
    } else {
      console.log('Using standard HSE components for document ID:', documentId); // Debug log
      // For HSE-043, HSE-044, HSE-045, HSE-126, use 3 steps with standard components
      const stepsArray = [
        {
          id: 1,
          title: 'Step 1',
          subtitle: 'HSE Information',
          component: <Hsemodule43step1 setActiveStep={setActiveStep} />,
        },
        {
          id: 2,
          title: 'Step 2',
          subtitle: 'HSE Assessment',
          component: <Hsemodule43step2 setActiveStep={setActiveStep} />,
        },
        {
          id: 3,
          title: 'Step 3',
          subtitle: 'HSE Review',
          component: <Hsemodule43step3 setActiveStep={setActiveStep} />,
        },
      ];
      console.log('Setting standard HSE steps:', stepsArray);
      setSteps(stepsArray);
    }
  };

  // Handle checklist selection from navigation
  useEffect(() => {
    if (route.params?.selectedChecklist) {
      console.log('Received selected checklist from navigation:', route.params.selectedChecklist.document_id); // Debug log
      setSelectedChecklist(route.params.selectedChecklist);
      setupSteps(route.params.selectedChecklist);
      
      // Also save to AsyncStorage to ensure persistence
      const saveChecklistData = async () => {
        try {
          const savedHse = await AsyncStorage.getItem('hseStep1');
          let hseData = savedHse ? JSON.parse(savedHse) : {};
          hseData.selectedChecklist = route.params.selectedChecklist;
          await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
        } catch (error) {
          console.error('Error saving checklist data:', error);
        }
      };
      
      saveChecklistData();
    }
  }, [route.params?.selectedChecklist]);

  // Monitor activeStep changes and ensure correct components are loaded
  useEffect(() => {
    console.log('Active step changed to:', activeStep);
    console.log('Selected checklist:', selectedChecklist?.document_id);
    console.log('Steps length:', steps.length);
    
    if (selectedChecklist && steps.length > 0) {
      console.log('Current steps configuration:', steps.map(s => ({ id: s.id, title: s.title })));
      console.log('Active step component:', steps.find((step) => step.id === activeStep)?.component ? 'Found' : 'Not found');
    } else if (activeStep > 1 && steps.length === 0) {
      console.log('No steps available but activeStep > 1, setting up default steps');
      setupDefaultSteps();
    }
  }, [activeStep, selectedChecklist, steps]);

  // Function to handle step changes
  const handleStepChange = async (stepId) => {
    console.log('Changing to step:', stepId, 'for document ID:', selectedChecklist?.document_id); // Debug log
    
    // If moving to step 2, reload the checklist data to ensure we have the latest
    if (stepId === 2) {
      try {
        const savedData = await AsyncStorage.getItem('hseStep1');
        if (savedData) {
          const data = JSON.parse(savedData);
          if (data.selectedChecklist && data.selectedChecklist.document_id !== selectedChecklist?.document_id) {
            console.log('Updating checklist data for step 2:', data.selectedChecklist.document_id);
            setSelectedChecklist(data.selectedChecklist);
            setupSteps(data.selectedChecklist);
          }
        }
      } catch (error) {
        console.error('Error reloading checklist data:', error);
      }
    }
    
    setActiveStep(stepId);
  };

  console.log('Rendering header with steps length:', steps.length, 'activeStep:', activeStep);

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <TouchableOpacity
          style={styles.backButton}
          onPress={() => navigation.goBack()}
        >
          <MaterialIcons name="arrow-back" size={24} color="#fff" />
          <Text style={styles.backButtonText}>Back</Text>
        </TouchableOpacity>
        
        {/* Debug info - show document ID */}
        {selectedChecklist && (
          <View style={styles.debugInfo}>
            <Text style={styles.debugText}>
              Document ID: {selectedChecklist.document_id} | Step: {activeStep}
            </Text>
          </View>
        )}
        
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.stepsScrollContent}
        >
          {steps.map((step, index) => {
            const isActive = activeStep === step.id;
            const isCompleted = activeStep > step.id;
            const isClickable = step.id === 1 || step.id < activeStep; // Allow going back to previous steps
            return (
              <TouchableOpacity
                key={step.id}
                style={styles.stepWrapper}
                onPress={() => isClickable && handleStepChange(step.id)}
                disabled={!isClickable}
              >
                <View
                  style={[
                    styles.stepCircle,
                    isActive && styles.activeStepCircle,
                    isCompleted && styles.completedStepCircle,
                  ]}
                >
                  {isCompleted ? (
                    <MaterialIcons name="check" size={20} color="#fff" />
                  ) : (
                    <Text style={[styles.stepNumber, isActive && styles.activeStepNumber]}>
                      {step.id}
                    </Text>
                  )}
                </View>
                <View style={styles.stepTextContainer}>
                  <Text style={[styles.stepTitle, isActive && styles.activeStepTitle]}>
                    {step.title}
                  </Text>
                  <Text style={[styles.stepSubtitle, isActive && styles.activeStepSubtitle]}>
                    {step.subtitle}
                  </Text>
                </View>
                {index < steps.length - 1 && (
                  <View
                    style={[styles.stepConnector, isCompleted && styles.completedStepConnector]}
                  />
                )}
              </TouchableOpacity>
            );
          })}
        </ScrollView>
      </LinearGradient>
      <View style={styles.mainContent}>
        {steps.length > 0 ? (
          (() => {
            const currentStep = steps.find((step) => step.id === activeStep);
            console.log('Current step found:', currentStep ? `Step ${currentStep.id}` : 'Not found');
            console.log('Active step:', activeStep);
            console.log('Available steps:', steps.map(s => s.id));
            
            if (currentStep && currentStep.component) {
              try {
                console.log('Rendering step component:', currentStep.id);
                return currentStep.component;
              } catch (error) {
                console.error('Error rendering step component:', error);
                return (
                  <View style={styles.fallbackContainer}>
                    <Text style={styles.fallbackText}>Error loading step {activeStep}</Text>
                    <Text style={styles.debugText}>Error: {error.message}</Text>
                  </View>
                );
              }
            } else {
              console.log('Step component not found, showing fallback');
              return (
                <View style={styles.fallbackContainer}>
                  <Text style={styles.fallbackText}>Loading step {activeStep} component...</Text>
                  <Text style={styles.debugText}>Available steps: {steps.map(s => s.id).join(', ')}</Text>
                </View>
              );
            }
          })()
        ) : (
          <View style={styles.fallbackContainer}>
            <Text style={styles.fallbackText}>Loading checklist data...</Text>
          </View>
        )}
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    paddingBottom: 15,
  },
  backButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 20,
    paddingTop: StatusBar.currentHeight || 50,
    paddingBottom: 10,
  },
  backButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 8,
  },
  stepsScrollContent: {
    paddingHorizontal: 20,
    alignItems: 'center',
  },
  stepWrapper: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 5,
  },
  stepCircle: {
    width: 30,
    height: 30,
    borderRadius: 15,
    backgroundColor: '#eee',
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#ccc',
  },
  activeStepCircle: {
    borderColor: '#fff',
    backgroundColor: '#fff',
  },
  completedStepCircle: {
    backgroundColor: '#4CAF50',
    borderColor: '#4CAF50',
  },
  stepNumber: {
    color: '#888',
    fontWeight: 'bold',
  },
  activeStepNumber: {
    color: '#025383',
  },
  stepTextContainer: {
    marginHorizontal: 10,
  },
  stepTitle: {
    fontSize: 14,
    color: '#ccc',
    fontWeight: 'bold',
  },
  activeStepTitle: {
    color: '#fff',
  },
  stepSubtitle: {
    fontSize: 12,
    color: '#ccc',
  },
  activeStepSubtitle: {
    color: '#fff',
  },
  stepConnector: {
    width: 30,
    height: 2,
    backgroundColor: '#ccc',
  },
  completedStepConnector: {
    backgroundColor: '#4CAF50',
  },
  mainContent: {
    flex: 1,
  },
  debugInfo: {
    position: 'absolute',
    top: StatusBar.currentHeight + 60, // Adjust based on header height
    left: 20,
    backgroundColor: 'rgba(0,0,0,0.5)',
    padding: 5,
    borderRadius: 5,
  },
  debugText: {
    color: '#fff',
    fontSize: 12,
  },
  fallbackContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f0f0f0', // A light background for the fallback
  },
  fallbackText: {
    fontSize: 18,
    color: '#333',
    fontStyle: 'italic',
  },
});

export default Hsemoduleheader43;