
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  ScrollView,
  Alert,
  Modal,
  FlatList,
  Image,
  Animated,
  TextInput,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = height * 0.12;
const SEARCH_HEADER_HEIGHT = height * 0.22;

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const DROPDOWN_OPTIONS = [
  { label: 'View PDF Report', value: 'view-pdf', icon: <MaterialIcons name="picture-as-pdf" size={22} color="#015185" /> },
  { label: 'Delete Inspection', value: 'delete', icon: <MaterialIcons name="delete" size={22} color="#e74c3c" /> },
];

const HSEInspectionHome = () => {
  const navigation = useNavigation();
  const [savedInspections, setSavedInspections] = useState([]);
  const [filteredInspections, setFilteredInspections] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedInspection, setSelectedInspection] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [showSearch, setShowSearch] = useState(false);

  useEffect(() => {
    loadSavedInspections();
  }, []);

  // Filter inspections based on search query
  useEffect(() => {
    if (searchQuery.trim() === '') {
      setFilteredInspections(savedInspections);
    } else {
      const filtered = savedInspections.filter(inspection => {
        const query = searchQuery.toLowerCase();
        
        // Search by document ID
        const documentId = inspection.step1?.selectedChecklist?.document_id || '';
        if (documentId.toLowerCase().includes(query)) return true;
        
        // Search by document name
        const documentName = inspection.step1?.selectedChecklist?.document_name || '';
        if (documentName.toLowerCase().includes(query)) return true;
        
        // Search by request number
        const requestNumber = inspection.step1?.requestNumber || inspection.pdfData?.request_number || '';
        if (requestNumber.toLowerCase().includes(query)) return true;
        
        // Search by location
        const location = inspection.location || '';
        if (location.toLowerCase().includes(query)) return true;
        
        // Search by rig
        const rig = inspection.rig || '';
        if (rig.toLowerCase().includes(query)) return true;
        
        // Search by date
        const date = inspection.createdAt ? new Date(inspection.createdAt).toLocaleDateString() : '';
        if (date.toLowerCase().includes(query)) return true;
        
        return false;
      });
      setFilteredInspections(filtered);
    }
  }, [searchQuery, savedInspections]);

  const loadSavedInspections = async () => {
    try {
      setSyncModalVisible(true);
      setLoading(true);
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not authenticated. Please log in again.',
          position: 'top',
        });
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      
      const userData = userSnapshot.val();
      if (!userData) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not found in admin database.',
          position: 'top',
        });
        return;
      }

      const userId = Object.keys(userData)[0];
      const inspectionsSnapshot = await database()
        .ref(`/GlobalHSEInspections/admins/${userId}`)
        .once('value');

      const inspectionsData = inspectionsSnapshot.val();
      if (inspectionsData) {
        const inspectionsArray = Object.entries(inspectionsData).map(([id, data]) => ({
          id,
          ...data,
        }));
        
        // Sort inspections by creation date in descending order (newest first)
        inspectionsArray.sort((a, b) => {
          const dateA = a.createdAt ? new Date(a.createdAt).getTime() : 0;
          const dateB = b.createdAt ? new Date(b.createdAt).getTime() : 0;
          return dateB - dateA; // Descending order (newest first)
        });
        
        setSavedInspections(inspectionsArray);
      } else {
        setSavedInspections([]);
      }
    } catch (error) {
      console.error('Error loading saved inspections:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load saved inspections.',
        position: 'top',
      });
    } finally {
      setLoading(false);
      setSyncModalVisible(false);
    }
  };

  const handleStartInspection = () => {
    navigation.navigate('Hsemoduleheader43');
  };

  const handleGoBack = () => {
    navigation.navigate('MainApp');
  };

  const handleViewPDF = (inspection) => {
    if (inspection.pdfDownloadUrl) {
      navigation.navigate('HSEViewPDF', {
        pdfUrl: inspection.pdfDownloadUrl,
        onGoBack: () => {
          // Refresh the list when returning from PDF viewer
          loadSavedInspections();
        }
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not available for this inspection.',
        position: 'top',
      });
    }
  };

  const handleDeleteInspection = (inspection) => {
    Alert.alert(
      'Delete Inspection',
      'Are you sure you want to delete this inspection? This action cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const userEmail = await AsyncStorage.getItem('userEmail');
              const userSnapshot = await database()
                .ref('/Globalusers/admins')
                .orderByChild('email')
                .equalTo(userEmail)
                .once('value');
              
              const userData = userSnapshot.val();
              const userId = Object.keys(userData)[0];

              await database()
                .ref(`/GlobalHSEInspections/admins/${userId}/${inspection.id}`)
                .remove();

              Toast.show({
                type: 'success',
                text1: 'Success',
                text2: 'Inspection deleted successfully.',
                position: 'top',
              });

              // Refresh the list
              loadSavedInspections();
            } catch (error) {
              console.error('Error deleting inspection:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to delete inspection.',
                position: 'top',
              });
            }
          },
        },
      ],
      { cancelable: true }
    );
  };

  const handleDeleteAllReports = () => {
    Alert.alert(
      'Delete All Reports',
      'Are you sure you want to delete ALL HSE inspection reports? This action cannot be undone and will permanently remove all saved inspections.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete All',
          style: 'destructive',
          onPress: async () => {
            try {
              setSyncModalVisible(true);
              const userEmail = await AsyncStorage.getItem('userEmail');
              const userSnapshot = await database()
                .ref('/Globalusers/admins')
                .orderByChild('email')
                .equalTo(userEmail)
                .once('value');
              
              const userData = userSnapshot.val();
              const userId = Object.keys(userData)[0];

              // Delete all reports for this user
              await database()
                .ref(`/GlobalHSEInspections/admins/${userId}`)
                .remove();

              Toast.show({
                type: 'success',
                text1: 'Success',
                text2: 'All inspection reports deleted successfully.',
                position: 'top',
              });

              // Refresh the list
              loadSavedInspections();
            } catch (error) {
              console.error('Error deleting all reports:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to delete all reports.',
                position: 'top',
              });
            } finally {
              setSyncModalVisible(false);
            }
          },
        },
      ],
      { cancelable: true }
    );
  };

  const handleDropdownAction = async (action, inspection) => {
    try {
      switch (action) {
        case 'view-pdf':
          handleViewPDF(inspection);
          break;
        case 'delete':
          handleDeleteInspection(inspection);
          break;
        default:
          break;
      }
    } catch (error) {
      console.error('Error processing action:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to process action.',
        position: 'top',
      });
    }
  };

  const toggleSearch = () => {
    setShowSearch(!showSearch);
    if (!showSearch) {
      setSearchQuery('');
    }
  };

  const clearSearch = () => {
    setSearchQuery('');
    setShowSearch(false);
  };

  const renderInspectionCard = ({ item, index }) => {
    const dropdownPosition = index > filteredInspections.length - 4 ? 'top' : 'auto';
    
    // Get request number from step1 data or pdfData
    const requestNumber = item.step1?.requestNumber || item.pdfData?.request_number || 'N/A';
    // Get location and project from step1 data
    const location = item.step1?.location || item.location || 'N/A';
    const project = item.step1?.project || 'N/A';
    
    return (
      <View style={styles.inspectionCard}>
        <View style={styles.cardHeader}>
          <View style={styles.cardLeft}>
            <View style={styles.cardIconContainer}>
              <MaterialIcons name="assignment" size={24} color="#3481BC" />
            </View>
            <View style={styles.cardInfo}>
              <Text style={styles.cardTitle}>
                {item.step1?.selectedChecklist?.document_name || 'HSE Inspection'}
              </Text>
              
              {/* Request Number displayed as subtitle */}
              <Text style={styles.requestSubtitle}>
                Request: {requestNumber}
              </Text>
              
              <Text style={styles.cardDate}>
                {item.createdAt ? new Date(item.createdAt).toLocaleDateString() : 'Unknown Date'}
              </Text>
              

            </View>
          </View>
          <View style={styles.dropdownContainer}>
            <Dropdown
              style={styles.dropdown}
              data={DROPDOWN_OPTIONS}
              labelField="label"
              valueField="value"
              value={null}
              onChange={async (option) => {
                await new Promise(resolve => setTimeout(resolve, 50));
                handleDropdownAction(option.value, item);
              }}
              renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
              renderItem={(option) => (
                <View style={styles.dropdownItem}>
                  {option.icon}
                  <Text
                    style={[
                      styles.dropdownText,
                      { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                    ]}
                  >
                    {option.label}
                  </Text>
                </View>
              )}
              placeholder=""
              showsVerticalScrollIndicator={false}
              dropdownPosition={dropdownPosition}
              containerStyle={styles.dropdownContainerStyle}
              itemTextStyle={styles.dropdownItemText}
              selectedTextStyle={{ display: 'none' }}
            />
          </View>
        </View>
        
        {item.pdfDownloadUrl && (
          <View style={styles.pdfIndicator}>
            <MaterialIcons name="picture-as-pdf" size={16} color="#3481BC" />
            <Text style={styles.pdfText}>PDF Available</Text>
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Enhanced Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: showSearch ? SEARCH_HEADER_HEIGHT : TOTAL_HEADER_HEIGHT }]}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity style={styles.backButton} onPress={handleGoBack}>
              <MaterialIcons name="arrow-back" size={24} color="#fff" />
            </TouchableOpacity>
            <View style={styles.headerTitleContainer}>
              <Text style={styles.headerTitle}>HSE Inspection</Text>
              <Text style={styles.headerSubtitle}>Health, Safety & Environment</Text>
            </View>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.headerButton} onPress={toggleSearch}>
              <MaterialIcons name={showSearch ? "close" : "search"} size={24} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity style={styles.headerButton} onPress={handleDeleteAllReports}>
              <MaterialIcons name="delete-sweep" size={24} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity style={styles.headerButton} onPress={loadSavedInspections}>
              <MaterialIcons name="refresh" size={24} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
        
        {/* Search Bar */}
        {showSearch && (
          <View style={styles.searchContainer}>
            <View style={styles.searchInputContainer}>
              <MaterialIcons name="search" size={20} color="#666" style={styles.searchIcon} />
              <TextInput
                style={styles.searchInput}
                placeholder="Search by document ID, name, request number, location, rig, or date..."
                placeholderTextColor="#999"
                value={searchQuery}
                onChangeText={setSearchQuery}
                autoFocus={true}
              />
              {searchQuery.length > 0 && (
                <TouchableOpacity onPress={clearSearch} style={styles.clearButton}>
                  <MaterialIcons name="close" size={18} color="#666" />
                </TouchableOpacity>
              )}
            </View>
            {searchQuery.length > 0 && (
              <Text style={styles.searchResultsText}>
                {filteredInspections.length} result{filteredInspections.length !== 1 ? 's' : ''} found
              </Text>
            )}
          </View>
        )}
      </LinearGradient>

      {/* Main Content */}
      <ScrollView style={styles.scrollView} showsVerticalScrollIndicator={false}>
        <View style={styles.content}>
          {/* Stats Section */}
          <View style={styles.statsContainer}>
            <View style={styles.statCard}>
              <View style={styles.statIconContainer}>
                <MaterialIcons name="assignment" size={24} color="#3481BC" />
              </View>
              <View style={styles.statContent}>
                <Text style={styles.statNumber}>
                  {showSearch && searchQuery.length > 0 ? filteredInspections.length : savedInspections.length}
                </Text>
                <Text style={styles.statLabel}>
                  {showSearch && searchQuery.length > 0 ? 'Filtered' : 'Total'} Inspections
                </Text>
              </View>
            </View>
            <View style={styles.statCard}>
              <View style={styles.statIconContainer}>
                <MaterialIcons name="picture-as-pdf" size={24} color="#3481BC" />
              </View>
              <View style={styles.statContent}>
                <Text style={styles.statNumber}>
                  {(showSearch && searchQuery.length > 0 ? filteredInspections : savedInspections).filter(item => item.pdfDownloadUrl).length}
                </Text>
                <Text style={styles.statLabel}>PDF Reports</Text>
              </View>
            </View>
          </View>

          {/* Saved Inspections Section */}
          <View style={styles.sectionContainer}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>
                {showSearch && searchQuery.length > 0 ? 'Search Results' : 'Saved Inspections'}
              </Text>
              {showSearch && searchQuery.length > 0 && (
                <TouchableOpacity onPress={clearSearch} style={styles.clearSearchButton}>
                  <MaterialIcons name="close" size={16} color="#666" />
                  <Text style={styles.clearSearchText}>Clear</Text>
                </TouchableOpacity>
              )}
            </View>
            
            {loading ? (
              <View style={styles.loadingContainer}>
                <MaterialIcons name="hourglass-empty" size={48} color="#ccc" />
                <Text style={styles.loadingText}>Loading inspections...</Text>
              </View>
            ) : (showSearch && searchQuery.length > 0 ? filteredInspections : savedInspections).length > 0 ? (
              <FlatList
                data={showSearch && searchQuery.length > 0 ? filteredInspections : savedInspections}
                keyExtractor={(item) => item.id}
                renderItem={renderInspectionCard}
                showsVerticalScrollIndicator={false}
                scrollEnabled={false}
              />
            ) : (
              <View style={styles.emptyContainer}>
                <View style={styles.emptyIconContainer}>
                  <MaterialIcons name={showSearch && searchQuery.length > 0 ? "search" : "assignment"} size={64} color="#ccc" />
                </View>
                <Text style={styles.emptyTitle}>
                  {showSearch && searchQuery.length > 0 ? 'No Search Results' : 'No Saved Inspections'}
                </Text>
                <Text style={styles.emptySubtext}>
                  {showSearch && searchQuery.length > 0 
                    ? `No inspections found matching "${searchQuery}"`
                    : 'Start your first HSE inspection to see it here'
                  }
                </Text>
                {!showSearch && (
                  <TouchableOpacity style={styles.emptyButton} onPress={handleStartInspection}>
                    <MaterialIcons name="add" size={20} color="#3481BC" />
                    <Text style={styles.emptyButtonText}>Start First Inspection</Text>
                  </TouchableOpacity>
                )}
              </View>
            )}
          </View>
        </View>
      </ScrollView>

      {/* Enhanced Bottom Button */}
      <View style={styles.bottomContainer}>
        <TouchableOpacity style={styles.startButton} onPress={handleStartInspection}>
          <LinearGradient
            colors={['#3481BC', '#025383']}
            style={styles.gradientButton}
            start={{ x: 0, y: 0 }}
            end={{ x: 1, y: 0 }}
          >
            <MaterialIcons name="play-arrow" size={24} color="#fff" />
            <Text style={styles.startButtonText}>Start New HSE Inspection</Text>
          </LinearGradient>
        </TouchableOpacity>
      </View>

      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Loading Data" />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  header: {
    paddingTop: StatusBar.currentHeight || 50,
    paddingBottom: 20,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    flex: 1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 10,
  },
  headerButton: {
    padding: 8,
    borderRadius: 6,
    backgroundColor: 'rgba(255, 255, 255, 0.1)',
  },
  backButton: {
    marginRight: 15,
    padding: 5,
  },
  headerTitleContainer: {
    flex: 1,
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#fff',
    marginBottom: 2,
  },
  headerSubtitle: {
    fontSize: 14,
    color: '#e3f2fd',
    opacity: 0.9,
  },
  searchContainer: {
    paddingHorizontal: 20,
    marginTop: 15,
  },
  searchInputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'rgba(255, 255, 255, 0.95)',
    borderRadius: 25,
    paddingHorizontal: 15,
    paddingVertical: 8,
  },
  searchIcon: {
    marginRight: 10,
  },
  searchInput: {
    flex: 1,
    fontSize: 16,
    color: '#333',
    paddingVertical: 8,
  },
  clearButton: {
    padding: 5,
    marginLeft: 10,
  },
  searchResultsText: {
    color: '#fff',
    fontSize: 12,
    marginTop: 8,
    textAlign: 'center',
    opacity: 0.9,
  },
  refreshButton: {
    padding: 5,
  },
  scrollView: {
    flex: 1,
  },
  content: {
    padding: 16,
  },
  statsContainer: {
    flexDirection: 'row',
    marginBottom: 24,
    gap: 12,
  },
  statCard: {
    flex: 1,
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    flexDirection: 'row',
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  statIconContainer: {
    width: 48,
    height: 48,
    borderRadius: 24,
    backgroundColor: '#f0f8ff',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 12,
  },
  statContent: {
    flex: 1,
  },
  statNumber: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: 2,
  },
  statLabel: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
  },
  sectionContainer: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  sectionHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  clearSearchButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 6,
    backgroundColor: '#f0f0f0',
    borderRadius: 16,
  },
  clearSearchText: {
    fontSize: 12,
    color: '#666',
    marginLeft: 4,
    fontWeight: '500',
  },
  loadingContainer: {
    alignItems: 'center',
    padding: 40,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
    marginTop: 12,
  },
  emptyContainer: {
    alignItems: 'center',
    padding: 40,
  },
  emptyIconContainer: {
    width: 120,
    height: 120,
    borderRadius: 60,
    backgroundColor: '#f8f9fa',
    alignItems: 'center',
    justifyContent: 'center',
    marginBottom: 16,
  },
  emptyTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 8,
  },
  emptySubtext: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginBottom: 24,
    lineHeight: 20,
  },
  emptyButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 20,
    paddingVertical: 12,
    borderRadius: 24,
    borderWidth: 1,
    borderColor: '#3481BC',
  },
  emptyButtonText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#3481BC',
    marginLeft: 8,
  },
  inspectionCard: {
    backgroundColor: '#f8f9fa',
    borderRadius: 12,
    padding: 20,
    marginBottom: 12,
    borderWidth: 1,
    borderColor: '#e9ecef',
    marginHorizontal: -4,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
  },
  cardLeft: {
    flexDirection: 'row',
    flex: 1,
  },
  cardIconContainer: {
    width: 52,
    height: 52,
    borderRadius: 26,
    backgroundColor: '#fff',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  cardInfo: {
    flex: 1,
  },
  cardTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 4,
  },
  requestSubtitle: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
    marginBottom: 8,
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 6,
    alignSelf: 'flex-start',
    borderWidth: 1,
    borderColor: '#e3f2fd',
  },
  cardDate: {
    fontSize: 15,
    color: '#666',
    marginBottom: 10,
  },
  cardDetails: {
    flexDirection: 'row',
    gap: 20,
  },
  detailItem: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  detailText: {
    fontSize: 13,
    color: '#666',
    marginLeft: 6,
  },
  dropdownContainer: {
    justifyContent: 'center',
    alignItems: 'center',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainerStyle: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  pdfIndicator: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    alignSelf: 'flex-start',
    marginTop: 12,
  },
  pdfText: {
    fontSize: 12,
    color: '#3481BC',
    fontWeight: '500',
    marginLeft: 4,
  },
  bottomContainer: {
    padding: 20,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#eee',
  },
  startButton: {
    borderRadius: 12,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  gradientButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: 18,
  },
  startButtonText: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#fff',
    marginLeft: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },

});

export default HSEInspectionHome;