import React, { useState, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  TextInput,
  Platform,
  Dimensions,
  Modal
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const HSE_071step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [formData, setFormData] = useState({});
  const [hseId, setHseId] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [showNoteModal, setShowNoteModal] = useState(false);
  const [currentNoteItem, setCurrentNoteItem] = useState(null);
  const [noteText, setNoteText] = useState('');
  const [notes, setNotes] = useState({});
  const [expandedSections, setExpandedSections] = useState({
    incomingShiftRequirements: false,
    equipmentChecklist: false,
    standardProcedures: false
  });
  const [combinedInputs, setCombinedInputs] = useState([]);

  useEffect(() => {
    loadHseData();
  }, []);

  // Reload notes when screen is focused
  useFocusEffect(
    useCallback(() => {
      if (hseId) {
        loadNotes();
      }
    }, [hseId])
  );

  const loadHseData = async () => {
    try {
      console.log('Loading HSE-071 data...');
      setIsLoading(true);
      const savedData = await AsyncStorage.getItem('hseStep1');
      
      if (savedData) {
        const data = JSON.parse(savedData);
        setSelectedChecklist(data.selectedChecklist);
        setHseId(data.id);
        await loadFormData(data.id);
      } else {
        console.log('No saved data found');
      }
    } catch (error) {
      console.error('Error loading HSE data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load HSE data.', position: 'top' });
    } finally {
      setIsLoading(false);
    }
  };

  const loadFormData = async (currentHseId) => {
    try {
      const savedFormData = await AsyncStorage.getItem(`hse071FormData_${currentHseId}`);
      if (savedFormData) {
        setFormData(JSON.parse(savedFormData));
      } else {
        // Initialize with empty form data based on JSON structure
        setFormData({
          Supervisor: '',
          Operator: '',
          Geologist: '',
          'Vehicle ID': '',
          'Ori Tool Serial ID': '',
          'Controller Serial ID': '',
          'SS Survey Dip': '',
          'Hole ID': '',
          'Hole Depth': '',
          'Shift Meters': '',
          'Bit Meters': '',
          'Bit ID': '',
          'Month Safety Topic': '',
          'Daily Safety Topic': '',
          'Azi': '',
          'Date': '',
          'Shift': 'Day',
          'Rig': '',
          'Engine Hours': '',
          'Casing': {
            'PW': '',
            'PQ': '',
            'HQ': ''
          },
          'P.T.O': '',
          'Incoming Shift Requirements': {
            'Water Swivel': '',
            'Whipchecks': '',
            'Hoist Plug': '',
            'NQ Head Assembly': '',
            'HQ Head Assembly': '',
            'PQ Head Assembly': '',
            'Rig & Site Presentation': '',
            'Barricade Straight & Level': '',
            'Gyro #': '',
            'Gyro Controller #': '',
            'Gyro Charger': ''
          },
          'Equipment Checklist': {},
          'Standard Procedures': []
        });
      }
    } catch (error) {
      console.error('Error loading form data:', error);
    }
  };

  const loadNotes = async () => {
    try {
      if (!hseId) return;
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`hse071Notes_${hseId}`) || '{}');
      setNotes(savedNotes);
    } catch (error) {
      console.error('Error loading notes:', error);
    }
  };

  const handleInputChange = async (field, value) => {
    const newFormData = { ...formData, [field]: value };
    setFormData(newFormData);
    
    try {
      await AsyncStorage.setItem(`hse071FormData_${hseId}`, JSON.stringify(newFormData));
    } catch (error) {
      console.error('Error saving form data:', error);
    }
  };

  const handleCasingChange = async (casingType, value) => {
    const newFormData = {
      ...formData,
      Casing: {
        ...formData.Casing,
        [casingType]: value
      }
    };
    setFormData(newFormData);
    
    try {
      await AsyncStorage.setItem(`hse071FormData_${hseId}`, JSON.stringify(newFormData));
    } catch (error) {
      console.error('Error saving casing data:', error);
    }
  };

  const handleIncomingShiftRequirementChange = async (item, value) => {
    console.log('Updating incoming shift requirement:', item, value);
    const newFormData = {
      ...formData,
      'Incoming Shift Requirements': {
        ...formData['Incoming Shift Requirements'],
        [item]: value
      }
    };
    console.log('New form data:', newFormData);
    setFormData(newFormData);
    
    try {
      await AsyncStorage.setItem(`hse071FormData_${hseId}`, JSON.stringify(newFormData));
      console.log('Saved to AsyncStorage successfully');
    } catch (error) {
      console.error('Error saving incoming shift requirement data:', error);
    }
  };

  const handleEquipmentChecklistChange = async (item, value) => {
    const newFormData = {
      ...formData,
      'Equipment Checklist': {
        ...formData['Equipment Checklist'],
        [item]: value
      }
    };
    setFormData(newFormData);
    
    try {
      await AsyncStorage.setItem(`hse071FormData_${hseId}`, JSON.stringify(newFormData));
    } catch (error) {
      console.error('Error saving equipment checklist data:', error);
    }
  };

  const addCombinedInput = () => {
    if (combinedInputs.length >= 5) {
      Toast.show({
        type: 'info',
        text1: 'Maximum Limit Reached',
        text2: 'You can add up to 5 entries only.',
        position: 'top',
      });
      return;
    }
    
    setCombinedInputs([...combinedInputs, { 
      id: Date.now(), 
      incomingShiftRequirement: '', 
      shiftConcern: '' 
    }]);
  };

  const removeCombinedInput = (id) => {
    // Don't allow removal if it's the only entry
    if (combinedInputs.length <= 1) {
      Toast.show({
        type: 'info',
        text1: 'Cannot Remove',
        text2: 'At least one entry is required.',
        position: 'top',
      });
      return;
    }

    Alert.alert(
      'Remove Entry',
      'Are you sure you want to remove this entry?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Remove',
          style: 'destructive',
          onPress: () => {
            setCombinedInputs(prev => prev.filter(item => item.id !== id));
          },
        },
      ]
    );
  };

  const updateCombinedInput = (id, field, value) => {
    setCombinedInputs(prev => prev.map(item => 
      item.id === id ? { ...item, [field]: value } : item
    ));
  };

  const handleChecklistToggle = async (checklistType, item) => {
    const currentList = formData[checklistType] || [];
    const newList = currentList.includes(item)
      ? currentList.filter(i => i !== item)
      : [...currentList, item];
    
    const newFormData = { ...formData, [checklistType]: newList };
    setFormData(newFormData);
    
    try {
      await AsyncStorage.setItem(`hse071FormData_${hseId}`, JSON.stringify(newFormData));
    } catch (error) {
      console.error('Error saving checklist data:', error);
    }
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections(prev => ({
      ...prev,
      [sectionKey]: !prev[sectionKey]
    }));
  };

  const handleNotePress = async (field) => {
    setCurrentNoteItem(field);
    setNoteText(notes[field] || '');
    setShowNoteModal(true);
  };

  const saveNote = async () => {
    if (currentNoteItem) {
      const newNotes = { ...notes, [currentNoteItem]: noteText };
      setNotes(newNotes);
      try {
        await AsyncStorage.setItem(`hse071Notes_${hseId}`, JSON.stringify(newNotes));
        Toast.show({ type: 'success', text1: 'Note Saved', text2: 'Note saved successfully.', position: 'top' });
      } catch (error) {
        console.error('Error saving note:', error);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save note.', position: 'top' });
      }
      setShowNoteModal(false);
      setCurrentNoteItem(null);
      setNoteText('');
    }
  };

  const validateForm = () => {
    // Check if at least one incoming shift requirement is filled
    const hasIncomingShiftRequirements = Object.values(formData['Incoming Shift Requirements'] || {}).some(value => value && value.trim() !== '');
    
    // Check if at least one equipment checklist item is selected
    const hasEquipmentSelections = Object.values(formData['Equipment Checklist'] || {}).some(value => value === 'Yes' || value === 'No');
    
    // Check if at least one combined input is filled
    const hasCombinedInputs = combinedInputs.some(input => 
      (input.incomingShiftRequirement && input.incomingShiftRequirement.trim() !== '') ||
      (input.shiftConcern && input.shiftConcern.trim() !== '')
    );
    
    // Allow next if at least one of the main sections has data
    if (!hasIncomingShiftRequirements && !hasEquipmentSelections && !hasCombinedInputs) {
      Toast.show({
        type: 'error',
        text1: 'No Data Entered',
        text2: 'Please fill at least one field in any section before proceeding.',
        position: 'top'
      });
      return false;
    }
    
    return true;
  };

  const handleNext = async () => {
    if (!validateForm()) {
      return;
    }

    try {
      const step2Data = {
        id: hseId,
        checklistId: selectedChecklist?.document_id,
        checklistName: selectedChecklist?.document_name,
        formData: formData,
        combinedInputs: combinedInputs,
        notes: notes,
        completedAt: new Date().toISOString(),
      };

      await AsyncStorage.setItem('hseStep2', JSON.stringify(step2Data));
      Toast.show({
        type: 'success',
        text1: 'Data Saved',
        text2: 'Form data saved successfully.',
        position: 'top',
      });
      
      setTimeout(() => {
        setActiveStep(3);
      }, 1000);
    } catch (error) {
      console.error('Error saving step 2 data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data.',
        position: 'top',
      });
    }
  };

  const handleBackPress = () => {
    setActiveStep(1);
  };

  const renderInputField = (label, value, onChangeText, placeholder = '', keyboardType = 'default') => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TextInput
        style={styles.textInput}
        value={value}
        onChangeText={onChangeText}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#999"
      />
    </View>
  );

  const renderCasingInputs = () => (
    <View style={styles.sectionContainer}>
      <Text style={styles.sectionTitle}>Casing</Text>
      <View style={styles.casingContainer}>
        {renderInputField('PW', formData.Casing?.PW || '', (value) => handleCasingChange('PW', value))}
        {renderInputField('PQ', formData.Casing?.PQ || '', (value) => handleCasingChange('PQ', value))}
        {renderInputField('HQ', formData.Casing?.HQ || '', (value) => handleCasingChange('HQ', value))}
      </View>
    </View>
  );

  const renderIncomingShiftRequirements = () => {
    const items = [
      'Water Swivel',
      'Whipchecks',
      'Hoist Plug',
      'NQ Head Assembly',
      'HQ Head Assembly',
      'PQ Head Assembly',
      'Rig & Site Presentation',
      'Barricade Straight & Level',
      'Gyro #',
      'Gyro Controller #',
      'Gyro Charger'
    ];

    return (
      <View style={styles.sectionContainer}>
        <TouchableOpacity
          style={styles.sectionHeader}
          onPress={() => toggleSection('incomingShiftRequirements')}
        >
          <View style={styles.sectionHeaderContent}>
            <AntDesign
              name={expandedSections.incomingShiftRequirements ? 'caretup' : 'caretdown'}
              size={20}
              color={expandedSections.incomingShiftRequirements ? 'red' : 'black'}
            />
            <Text style={styles.sectionHeaderText}>Shift Requirements</Text>
          </View>
        </TouchableOpacity>
        
        {expandedSections.incomingShiftRequirements && (
          <View style={styles.sectionContent}>
            {items.map((item, index) => (
              <View key={index} style={styles.centeredItem}>
                <View style={styles.itemContainer}>
                  <Text style={styles.itemText}>
                    {index + 1}. {item}
                  </Text>
                  <View style={styles.itemRow}>
                    <TextInput
                      style={styles.dropdownInput}
                      value={formData['Incoming Shift Requirements']?.[item] || ''}
                      onChangeText={(value) => {
                        // Only allow numeric input and limit to 5 digits
                        const numericValue = value.replace(/[^0-9]/g, '');
                        if (numericValue.length <= 5) {
                          handleIncomingShiftRequirementChange(item, numericValue);
                        }
                      }}
                      placeholder="0"
                      placeholderTextColor="#999"
                      keyboardType="numeric"
                      maxLength={5}
                      textAlign="center"
                      autoCapitalize="none"
                      autoCorrect={false}
                    />
                  </View>
                </View>
                <View style={styles.itemBorder} />
              </View>
            ))}
          </View>
        )}
      </View>
    );
  };

  const renderCombinedInputs = () => {
    return (
      <View style={styles.sectionContainer}>
        <Text style={styles.sectionTitle}>Incoming Shift Requirements & Shift Concerns</Text>
        <View style={styles.inputListContainer}>
          {combinedInputs.map((item, index) => (
            <View key={item.id} style={styles.combinedInputContainer}>
              <View style={styles.inputItemHeader}>
                <Text style={styles.inputItemLabel}>Entry {index + 1}</Text>
                <TouchableOpacity
                  style={styles.removeButton}
                  onPress={() => removeCombinedInput(item.id)}
                >
                  <MaterialIcons name="remove-circle" size={20} color="#ff4444" />
                </TouchableOpacity>
              </View>
              
              <View style={styles.combinedInputRow}>
                <View style={styles.combinedInputField}>
                  <Text style={styles.combinedInputLabel}>Incoming Shift Requirement</Text>
                  <TextInput
                    style={styles.combinedInputTextInput}
                    value={item.incomingShiftRequirement}
                    onChangeText={(text) => updateCombinedInput(item.id, 'incomingShiftRequirement', text)}
                    placeholder="Enter incoming shift requirement..."
                    placeholderTextColor="#999"
                    multiline
                    textAlignVertical="top"
                    autoCapitalize="none"
                    autoCorrect={false}
                  />
                </View>
                
                <View style={styles.combinedInputField}>
                  <Text style={styles.combinedInputLabel}>Shift Concern</Text>
                  <TextInput
                    style={styles.combinedInputTextInput}
                    value={item.shiftConcern}
                    onChangeText={(text) => updateCombinedInput(item.id, 'shiftConcern', text)}
                    placeholder="Enter shift concern..."
                    placeholderTextColor="#999"
                    multiline
                    textAlignVertical="top"
                    autoCapitalize="none"
                    autoCorrect={false}
                  />
                </View>
              </View>
            </View>
          ))}
          <TouchableOpacity 
            style={[
              styles.addButton, 
              combinedInputs.length >= 5 && styles.addButtonDisabled
            ]} 
            onPress={addCombinedInput}
            disabled={combinedInputs.length >= 5}
          >
            <MaterialIcons 
              name="add-circle" 
              size={20} 
              color={combinedInputs.length >= 5 ? "#ccc" : "#0078D4"} 
            />
            <Text style={[
              styles.addButtonText,
              combinedInputs.length >= 5 && styles.addButtonTextDisabled
            ]}>
              Add More ({combinedInputs.length}/5)
            </Text>
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  const renderEquipmentChecklist = () => {
    const items = [
      'Fire Extinguisher x3 & Signage',
      'Emergency Assembly Sign',
      'Reverse Parking Sign',
      'Safety Meeting Register',
      'Pre Start Check List',
      'SWP File',
      'Emergency Response Procedure',
      'MSDS File',
      'Stretcher',
      'Spill Kit',
      '1st Aid Box & Signage',
      'Tape Measure',
      'Safety Harness x2',
      'Admin Pen',
      'Marker Pen',
      '50ltr Red Bin',
      '50ltr Yellow Bin',
      '50ltr Green Bin',
      'Rig Tool Box Presentation',
      '36" Wrench x2',
      '24" Wrench x1',
      'Innertube Spanner x2',
      'Outertube Spanner x1',
      'Overshot Safety Chain',
      'Grease Pump',
      'Spirit Level',
      'Rod Stand',
      'Rod Storage Stands',
      'Core Barrel Stand',
      'Rod Handlers',
      'Strap Rod Handler',
      'Innertube Work Stand',
      'Core Pump Out Stand',
      'Ori Marker',
      'Core Breaking Chisel',
      'Hammer',
      'Sump Entrance Sign',
      'Life Jacket',
      'Chemical Storage',
      'Bypass Canon & Whipchecks',
      'Mixing Jug x2',
      'Marsh Funnel',
      'Mud Mix Ratio',
      'Viscosity',
      'Water Return',
      'Blue Wash Buckets x4',
      'Site Water Storage',
      'Rake',
      'Shovel',
      'Axe',
      'Pick',
      'Rig Diesel',
      'Petrol Consumption',
      'Core Lifter Consumption',
      'PQ Scrap Rod',
      'HQ Scrap Rod',
      'NQ Scrap Rod',
      'PQ Good Rod',
      'HQ Good Rod',
      'NQ Good Rod'
    ];

    return (
      <View style={styles.sectionContainer}>
        <TouchableOpacity
          style={styles.sectionHeader}
          onPress={() => toggleSection('equipmentChecklist')}
        >
          <View style={styles.sectionHeaderContent}>
            <AntDesign
              name={expandedSections.equipmentChecklist ? 'caretup' : 'caretdown'}
              size={20}
              color={expandedSections.equipmentChecklist ? 'red' : 'black'}
            />
            <Text style={styles.sectionHeaderText}>Equipment Checklist</Text>
          </View>
        </TouchableOpacity>
        
        {expandedSections.equipmentChecklist && (
          <View style={styles.sectionContent}>
            {items.map((item, index) => (
              <View key={index} style={styles.centeredItem}>
                <View style={styles.itemContainer}>
                  <Text style={styles.itemText}>
                    {index + 1}. {item}
                  </Text>
                  <View style={styles.itemRow}>
                    <View style={styles.optionsContainer}>
                      <TouchableOpacity
                        style={[
                          styles.optionButton,
                          formData['Equipment Checklist']?.[item] === 'Yes' && styles.selectedOptionButton,
                          formData['Equipment Checklist']?.[item] === 'Yes' && styles.okButton,
                        ]}
                        onPress={() => handleEquipmentChecklistChange(item, 'Yes')}
                      >
                        <Text style={[
                          styles.optionText,
                          formData['Equipment Checklist']?.[item] === 'Yes' && styles.optionTextSelected,
                        ]}>
                          Yes
                        </Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={[
                          styles.optionButton,
                          formData['Equipment Checklist']?.[item] === 'No' && styles.selectedOptionButton,
                          formData['Equipment Checklist']?.[item] === 'No' && styles.rejectButton,
                        ]}
                        onPress={() => handleEquipmentChecklistChange(item, 'No')}
                      >
                        <Text style={[
                          styles.optionText,
                          formData['Equipment Checklist']?.[item] === 'No' && styles.optionTextSelected,
                        ]}>
                          No
                        </Text>
                      </TouchableOpacity>
                    </View>
                  </View>
                </View>
                <View style={styles.itemBorder} />
              </View>
            ))}
          </View>
        )}
      </View>
    );
  };

  const renderChecklist = (title, items, checklistType) => {
    if (checklistType === 'Standard Procedures') {
      const standardProcedures = [
        'Grease head assembly each run',
        'Grease chuck every 3rd run',
        'Grease water swivel every 3rd run',
        'Grease hoist plug before pulling rods',
        'Note engine rpm during each run',
        'Note downhole pressure each run',
        'Note drill head rpm during each run',
        'Note weight on bit during each run',
        'Note pull back during each run',
        'Note turnaround time at end of run',
        'Note formation on each run',
        'Note muds viscocity during each run',
        'Note drill bit meters after each run',
        'Note depth of core grinding',
        'Compare core grinding to above notes'
      ];

      return (
        <View style={styles.sectionContainer}>
          <TouchableOpacity
            style={styles.sectionHeader}
            onPress={() => toggleSection('standardProcedures')}
          >
            <View style={styles.sectionHeaderContent}>
              <AntDesign
                name={expandedSections.standardProcedures ? 'caretup' : 'caretdown'}
                size={20}
                color={expandedSections.standardProcedures ? 'red' : 'black'}
              />
              <Text style={styles.sectionHeaderText}>{title}</Text>
            </View>
          </TouchableOpacity>
          
          {expandedSections.standardProcedures && (
            <View style={styles.sectionContent}>
              {standardProcedures.map((item, index) => (
                <View key={index} style={styles.centeredItem}>
                  <View style={styles.itemContainer}>
                    <Text style={styles.itemText}>
                      {index + 1}. {item}
                    </Text>
                  </View>
                  <View style={styles.itemBorder} />
                </View>
              ))}
            </View>
          )}
        </View>
      );
    }

    return (
      <View style={styles.sectionContainer}>
        <TouchableOpacity
          style={styles.sectionHeader}
          onPress={() => toggleSection(checklistType === 'Standard Procedures' ? 'standardProcedures' : 'other')}
        >
          <View style={styles.sectionHeaderContent}>
            <AntDesign
              name={expandedSections.standardProcedures ? 'caretup' : 'caretdown'}
              size={20}
              color={expandedSections.standardProcedures ? 'red' : 'black'}
            />
            <Text style={styles.sectionHeaderText}>{title}</Text>
          </View>
        </TouchableOpacity>
        
        {expandedSections.standardProcedures && (
          <ScrollView style={styles.checklistContainer}>
            {items.map((item, index) => (
              <TouchableOpacity
                key={index}
                style={styles.checklistItem}
                onPress={() => handleChecklistToggle(checklistType, item)}
              >
                <MaterialIcons
                  name={formData[checklistType]?.includes(item) ? 'check-box' : 'check-box-outline-blank'}
                  size={24}
                  color={formData[checklistType]?.includes(item) ? '#0078D4' : '#666'}
                />
                <Text style={styles.checklistItemText}>{item}</Text>
              </TouchableOpacity>
            ))}
          </ScrollView>
        )}
      </View>
    );
  };

  if (isLoading) {
    return (
      <View style={styles.loadingContainer}>
        <Text style={styles.loadingText}>Loading HSE-071 form...</Text>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <Text style={styles.checklistTitle}>DD Supervisor Shift Report</Text>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>
            Tap on section to expand
          </Text>
        </View>

        {/* Basic Information */}
        <View style={styles.sectionContainer}>
          <Text style={styles.sectionTitle}>Basic Information</Text>
          <Text style={styles.infoText}>All form fields have been moved to Step 3 for better organization.</Text>
        </View>

        {/* Incoming Shift Requirements */}
        {renderIncomingShiftRequirements()}

        {/* Combined Inputs */}
        {renderCombinedInputs()}

        {/* Equipment Checklist */}
        {renderEquipmentChecklist()}

        {/* Standard Procedures */}
        {renderChecklist('Standard Procedures', [], 'Standard Procedures')}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Note Modal */}
      <Modal visible={showNoteModal} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note for {currentNoteItem}</Text>
            <TextInput
              style={styles.noteInput}
              value={noteText}
              onChangeText={setNoteText}
              placeholder="Enter your note here..."
              multiline
              numberOfLines={4}
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity style={styles.modalButton} onPress={() => setShowNoteModal(false)}>
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.saveButton]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.saveButtonText]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  checklistTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginTop: 18,
    marginBottom: 4,
  },
  sectionContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    marginBottom: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 16,
  },
  infoText: {
    fontSize: 14,
    color: '#666',
    fontStyle: 'italic',
    textAlign: 'center',
    padding: 20,
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  naButton: {
    backgroundColor: '#FFA500',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    marginTop: 10,
  },
  inputContainer: {
    marginBottom: 16,
  },
  inputLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  dropdownInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    backgroundColor: '#fff',
    width: '40%',
    textAlign: 'center',
    minHeight: 45,
  },
  inputListContainer: {
    marginTop: 12,
  },
  inputItemContainer: {
    marginBottom: 16,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 12,
  },
  inputItemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  inputItemLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
  },
  removeButton: {
    padding: 4,
  },
  inputItemTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    padding: 10,
    fontSize: 14,
    backgroundColor: '#fff',
    minHeight: 60,
    textAlignVertical: 'top',
  },
  combinedInputContainer: {
    marginBottom: 16,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 12,
  },
  combinedInputRow: {
    flexDirection: 'row',
    gap: 12,
  },
  combinedInputField: {
    flex: 1,
  },
  combinedInputLabel: {
    fontSize: 12,
    fontWeight: '500',
    color: '#666',
    marginBottom: 4,
  },
  combinedInputTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    padding: 8,
    fontSize: 13,
    backgroundColor: '#fff',
    minHeight: 50,
    textAlignVertical: 'top',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: 12,
    borderWidth: 1,
    borderColor: '#0078D4',
    borderRadius: 8,
    backgroundColor: '#fff',
    marginTop: 8,
  },
  addButtonText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#0078D4',
    fontWeight: '500',
  },
  addButtonDisabled: {
    borderColor: '#ccc',
    backgroundColor: '#f5f5f5',
  },
  addButtonTextDisabled: {
    color: '#ccc',
  },
  checklistContainer: {
    maxHeight: 200,
    marginTop: 12,
  },
  checklistItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 8,
    paddingHorizontal: 4,
  },
  checklistItemText: {
    fontSize: 14,
    color: '#333',
    marginLeft: 12,
    flex: 1,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    width: '80%',
    maxWidth: 400,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 16,
  },
  noteInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    minHeight: 100,
    textAlignVertical: 'top',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: 16,
  },
  modalButton: {
    flex: 1,
    padding: 12,
    borderRadius: 8,
    alignItems: 'center',
    marginHorizontal: 4,
  },
  saveButton: {
    backgroundColor: '#0078D4',
  },
  modalButtonText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '500',
  },
  saveButtonText: {
    color: '#fff',
  },
});

export default HSE_071step2;