import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  Dimensions,
  Modal,
  TextInput,
  Platform,
  Share,
  Linking,
  Animated,
  Clipboard,
  KeyboardAvoidingView,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { Buffer } from 'buffer';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_072step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [step1Data, setStep1Data] = useState(null);
  const [step2Data, setStep2Data] = useState(null);
  const [formData, setFormData] = useState({});
  const [isLoading, setIsLoading] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [uploadProgress, setUploadProgress] = useState(0);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [showDataPreviewModal, setShowDataPreviewModal] = useState(false);
  const [formattedData, setFormattedData] = useState('');
  const isMounted = useRef(true);

  useEffect(() => {
    isMounted.current = true;
    loadData();
    return () => {
      isMounted.current = false;
    };
  }, []);

  const loadData = async () => {
    try {
      setIsLoading(true);
      
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      if (savedStep1) {
        setStep1Data(JSON.parse(savedStep1));
      }

      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        setStep2Data(JSON.parse(savedStep2));
        setFormData(JSON.parse(savedStep2).formData || {});
      }
    } catch (error) {
      console.error('Error loading data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load data.', position: 'top' });
    } finally {
      setIsLoading(false);
    }
  };

  const handleBackPress = () => {
    setActiveStep(2);
  };

  const formatDataForPreview = () => {
    try {
      const allData = {
        step1: step1Data,
        step2: step2Data,
        step3: formData,
        timestamp: new Date().toISOString(),
        reportType: 'HSE-071',
        reportTitle: 'DD Supervisor Shift Report'
      };

      const formattedString = JSON.stringify(allData, null, 2);
      setFormattedData(formattedString);
      setShowDataPreviewModal(true);
    } catch (error) {
      console.error('Error formatting data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to format data for preview.',
        position: 'top',
      });
    }
  };

  const copyToClipboard = async () => {
    try {
      await Clipboard.setString(formattedData);
      Toast.show({
        type: 'success',
        text1: 'Copied!',
        text2: 'Data copied to clipboard successfully.',
        position: 'top',
      });
    } catch (error) {
      console.error('Error copying to clipboard:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to copy data to clipboard.',
        position: 'top',
      });
    }
  };

  const handleInputChange = async (field, value) => {
    const newFormData = { ...formData, [field]: value };
    setFormData(newFormData);
    
    try {
      const step2Data = await AsyncStorage.getItem('hseStep2');
      if (step2Data) {
        const updatedStep2 = JSON.parse(step2Data);
        updatedStep2.formData = newFormData;
        await AsyncStorage.setItem('hseStep2', JSON.stringify(updatedStep2));
      }
    } catch (error) {
      console.error('Error saving form data:', error);
    }
  };

  const handleCasingChange = async (casingType, value) => {
    const newFormData = {
      ...formData,
      Casing: {
        ...formData.Casing,
        [casingType]: value
      }
    };
    setFormData(newFormData);
    
    try {
      const step2Data = await AsyncStorage.getItem('hseStep2');
      if (step2Data) {
        const updatedStep2 = JSON.parse(step2Data);
        updatedStep2.formData = newFormData;
        await AsyncStorage.setItem('hseStep2', JSON.stringify(updatedStep2));
      }
    } catch (error) {
      console.error('Error saving casing data:', error);
    }
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setSelectedDate(selectedDate);
      const formattedDate = selectedDate.toLocaleDateString('en-GB');
      handleInputChange('Date', formattedDate);
    }
  };

  const validateForm = () => {
    const requiredFields = ['Supervisor', 'Operator', 'Date', 'Shift', 'Rig'];
    const missingFields = requiredFields.filter(field => !formData[field]);
    
    if (missingFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.join(', ')}`,
        position: 'top'
      });
      return false;
    }
    return true;
  };

  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      const checklistData = step1.selectedChecklist || {};
      const sections = step2.sections || [];
      const selections = step2.selections || {};
      const notes = step2.notes || {};
      const countInputs = step2.countInputs || {};
      const combinedInputs = step2.combinedInputs || [];
      const incomingShiftRequirements = step2.formData?.['Incoming Shift Requirements'] || {};
      const equipmentChecklist = step2.formData?.['Equipment Checklist'] || {};

      // Generate incoming shift requirements rows
      let incomingShiftRequirementsRows = '';
      const shiftRequirementItems = [
        'Water Swivel', 'Whipchecks', 'Hoist Plug', 'NQ Head Assembly', 
        'HQ Head Assembly', 'PQ Head Assembly', 'Rig & Site Presentation', 
        'Barricade Straight & Level', 'Gyro #', 'Gyro Controller #', 'Gyro Charger'
      ];
      
      shiftRequirementItems.forEach((item, index) => {
        const value = incomingShiftRequirements[item] || '';
        if (value.trim() !== '') {
          incomingShiftRequirementsRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item}</td>
              <td>${value}</td>
            </tr>
          `;
        }
      });

      // Generate equipment checklist rows
      let equipmentChecklistRows = '';
      const equipmentItems = [
        'Fire Extinguisher x3 & Signage', 'Emergency Assembly Sign', 'Reverse Parking Sign',
        'Safety Meeting Register', 'Pre Start Check List', 'SWP File', 'Emergency Response Procedure',
        'MSDS File', 'Stretcher', 'Spill Kit', '1st Aid Box & Signage', 'Tape Measure',
        'Safety Harness x2', 'Admin Pen', 'Marker Pen', '50ltr Red Bin', '50ltr Yellow Bin',
        '50ltr Green Bin', 'Rig Tool Box Presentation', '36" Wrench x2', '24" Wrench x1',
        'Innertube Spanner x2', 'Outertube Spanner x1', 'Overshot Safety Chain', 'Grease Pump',
        'Spirit Level', 'Rod Stand', 'Rod Storage Stands', 'Core Barrel Stand', 'Rod Handlers',
        'Strap Rod Handler', 'Innertube Work Stand', 'Core Pump Out Stand', 'Ori Marker',
        'Core Breaking Chisel', 'Hammer', 'Sump Entrance Sign', 'Life Jacket', 'Chemical Storage',
        'Bypass Canon & Whipchecks', 'Mixing Jug x2', 'Marsh Funnel', 'Mud Mix Ratio',
        'Viscosity', 'Water Return', 'Blue Wash Buckets x4', 'Site Water Storage', 'Rake',
        'Shovel', 'Axe', 'Pick', 'Rig Diesel', 'Petrol Consumption', 'Core Lifter Consumption',
        'PQ Scrap Rod', 'HQ Scrap Rod', 'NQ Scrap Rod', 'PQ Good Rod', 'HQ Good Rod', 'NQ Good Rod'
      ];

      equipmentItems.forEach((item, index) => {
        const selection = equipmentChecklist[item] || '';
        if (selection === 'Yes' || selection === 'No') {
          equipmentChecklistRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item}</td>
              <td style="text-align: center;">${selection === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${selection === 'No' ? '✓' : ''}</td>
            </tr>
          `;
        }
      });

      let combinedInputsRows = '';
      if (combinedInputs && combinedInputs.length > 0) {
        combinedInputs.forEach((input, index) => {
          combinedInputsRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${input.incomingShiftRequirement || ''}</td>
              <td>${input.shiftConcern || ''}</td>
            </tr>
          `;
        });
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>DD Supervisor Shift Report</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 0;
              box-sizing: border-box;
            }
            .container {
              width: 100%;
              max-width: 800px;
              margin: 20px auto;
              border: 1px solid #000;
              box-sizing: border-box;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: center;
              background-color: #f0f0f0;
              padding: 10px;
              border-bottom: 1px solid #000;
            }
            .header .logo {
              width: 100px;
              height: 100px;
            }
            .header .details table {
              border-collapse: collapse;
              width: 100%;
            }
            .header .details td {
              padding: 2px 0;
              text-align: left;
            }
            .main-title {
              text-align: center;
              font-size: 24px;
              font-weight: bold;
              color: #d32f2f;
              margin: 20px 0;
              text-transform: uppercase;
            }
            .form-section {
              display: grid;
              grid-template-columns: 1fr 1fr 1fr;
              padding: 10px;
              border-bottom: 1px solid #000;
              gap: 20px;
            }
            .form-section div {
              padding: 5px;
            }
            .form-section label {
              font-weight: bold;
              display: block;
              margin-bottom: 5px;
            }
            .form-section input {
              width: 100%;
              padding: 5px;
              border: 1px solid #ccc;
              border-radius: 3px;
            }
            .requirements-section {
              padding: 10px;
              border-bottom: 1px solid #000;
            }
            .requirements-section h3 {
              margin: 0 0 10px 0;
              font-size: 16px;
              font-weight: bold;
            }
            .requirements-section textarea {
              width: 100%;
              min-height: 100px;
              padding: 10px;
              border: 1px solid #ccc;
              border-radius: 3px;
              resize: vertical;
            }
            .table {
              width: 100%;
              border-collapse: collapse;
              margin-top: 10px;
            }
            .table th,
            .table td {
              padding: 8px;
              border: 1px solid #000;
              text-align: left;
            }
            .table th {
              background-color: #d3d3d3;
              font-weight: bold;
            }
            .table .section-header {
              background-color: #a9a9a9;
              font-weight: bold;
            }
            .equipment-section {
              margin-top: 20px;
            }
            .equipment-section h3 {
              background-color: #f0f0f0;
              padding: 10px;
              margin: 0;
              border-bottom: 1px solid #000;
            }
            .procedures-section {
              margin-top: 20px;
            }
            .procedures-section h3 {
              background-color: #f0f0f0;
              padding: 10px;
              margin: 0;
              border-bottom: 1px solid #000;
            }
            .procedures-list {
              padding: 10px;
              border: 1px solid #000;
            }
            .procedures-list ol {
              margin: 0;
              padding-left: 20px;
            }
            .procedures-list li {
              margin-bottom: 5px;
            }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <div class="Logo">
                <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" width="200">
              </div>
              <div class="details">
                <table border="1">
                  <tr><td>Document name: ${checklistData.document_name || 'HSE-FOR-8.1.1 DD Supervisor Shift Report'}</td></tr>
                  <tr><td>Document number: ${checklistData.document_id || 'HSE-071'}</td></tr>
                  <tr><td>Version: ${checklistData.version || '1'}</td></tr>
                  <tr><td>Revision date: ${checklistData.version_date || '01/07/2021'}</td></tr>
                  <tr><td>Approved by: ${checklistData.approved_by || 'CEO'}</td></tr>
                  <tr><td>Request #: ${step1.requestNumber || 'N/A'}</td></tr>
                </table>
              </div>
            </div>
            
            <div class="main-title">DD SUPERVISOR SHIFT REPORT</div>
            
            <div class="form-section">
              <div>
                <label>Supervisor:</label>
                <input type="text" value="${step3.Supervisor || ''}" readonly>
              </div>
              <div>
                <label>Hole ID:</label>
                <input type="text" value="${step3['Hole ID'] || ''}" readonly>
              </div>
              <div>
                <label>Date:</label>
                <input type="text" value="${step3.Date || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>Operator:</label>
                <input type="text" value="${step3.Operator || ''}" readonly>
              </div>
              <div>
                <label>Hole Depth:</label>
                <input type="text" value="${step3['Hole Depth'] || ''}" readonly>
              </div>
              <div>
                <label>Shift:</label>
                <input type="text" value="${step3.Shift || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>Geologist:</label>
                <input type="text" value="${step3.Geologist || ''}" readonly>
              </div>
              <div>
                <label>Shift Meters:</label>
                <input type="text" value="${step3['Shift Meters'] || ''}" readonly>
              </div>
              <div>
                <label>Rig:</label>
                <input type="text" value="${step3.Rig || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>Vehicle ID:</label>
                <input type="text" value="${step3['Vehicle ID'] || ''}" readonly>
              </div>
              <div>
                <label>Bit Meters:</label>
                <input type="text" value="${step3['Bit Meters'] || ''}" readonly>
              </div>
              <div>
                <label>Engine Hours:</label>
                <input type="text" value="${step3['Engine Hours'] || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>Ori Tool Serial ID:</label>
                <input type="text" value="${step3['Ori Tool Serial ID'] || ''}" readonly>
              </div>
              <div>
                <label>Bit ID:</label>
                <input type="text" value="${step3['Bit ID'] || ''}" readonly>
              </div>
              <div>
                <label>Casing PW:</label>
                <input type="text" value="${step3.Casing?.PW || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>Ori Tool Serial ID:</label>
                <input type="text" value="${step3['Ori Tool Serial ID'] || ''}" readonly>
              </div>
              <div>
                <label>Month Safety Topic:</label>
                <input type="text" value="${step3['Month Safety Topic'] || ''}" readonly>
              </div>
              <div>
                <label>Casing PQ:</label>
                <input type="text" value="${step3.Casing?.PQ || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>Controller Serial ID:</label>
                <input type="text" value="${step3['Controller Serial ID'] || ''}" readonly>
              </div>
              <div>
                <label>Daily Safety Topic:</label>
                <input type="text" value="${step3['Daily Safety Topic'] || ''}" readonly>
              </div>
              <div>
                <label>Casing HQ:</label>
                <input type="text" value="${step3.Casing?.HQ || ''}" readonly>
              </div>
            </div>
            
            <div class="form-section">
              <div>
                <label>SS Survey Dip:</label>
                <input type="text" value="${step3['SS Survey Dip'] || ''}" readonly>
              </div>
              <div>
                <label>Azi:</label>
                <input type="text" value="${step3.Azi || ''}" readonly>
              </div>
              <div>
                <label>P.T.O:</label>
                <input type="text" value="${step3['P.T.O'] || ''}" readonly>
              </div>
            </div>
            
            <div class="requirements-section">
              <h3>Incoming Shift Requirements</h3>
              <table class="table">
                <tr>
                  <th>No.</th>
                  <th>Item</th>
                  <th>Details</th>
                </tr>
                ${incomingShiftRequirementsRows}
              </table>
            </div>
            
            <div class="requirements-section">
              <h3>Incoming Shift Requirements & Shift Concerns</h3>
              <table class="table">
                <tr>
                  <th>No.</th>
                  <th>Incoming Shift Requirement</th>
                  <th>Shift Concern</th>
                </tr>
                ${combinedInputsRows}
              </table>
            </div>
            
            <div class="equipment-section">
              <h3>Equipment Checklist</h3>
              <table class="table">
                <tr>
                  <th>No.</th>
                  <th>Equipment/Item</th>
                  <th>Yes</th>
                  <th>No</th>
                </tr>
                ${equipmentChecklistRows}
              </table>
            </div>
            
            <div class="procedures-section">
              <h3>Standard Procedures</h3>
              <div class="procedures-list">
                <ol>
                  <li>Grease head assembly each run</li>
                  <li>Grease chuck every 3rd run</li>
                  <li>Grease water swivel every 3rd run</li>
                  <li>Grease hoist plug before pulling rods</li>
                  <li>Note engine rpm during each run</li>
                  <li>Note downhole pressure each run</li>
                  <li>Note drill head rpm during each run</li>
                  <li>Note weight on bit during each run</li>
                  <li>Note pull back during each run</li>
                  <li>Note turnaround time at end of run</li>
                  <li>Note formation on each run</li>
                  <li>Note muds viscosity during each run</li>
                  <li>Note drill bit meters after each run</li>
                  <li>Note depth of core grinding</li>
                  <li>Compare core grinding to above notes</li>
                </ol>
              </div>
            </div>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `HSE_071_Report_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath, onProgress) => {
    try {
      const reference = storage().ref(storagePath);
      const fileData = await RNFS.readFile(filePath, 'base64');
      const chunkSize = 1024 * 1024; // 1MB chunks
      const buffer = Buffer.from(fileData, 'base64');
      const totalSize = buffer.length;
      let uploadedBytes = 0;

      const chunks = [];
      for (let i = 0; i < totalSize; i += chunkSize) {
        chunks.push(buffer.slice(i, i + chunkSize));
      }

      const uploadTask = reference.put(buffer);
      uploadTask.on('state_changed', (snapshot) => {
        uploadedBytes = snapshot.bytesTransferred;
        const progress = (uploadedBytes / totalSize) * 100;
        onProgress(Math.min(progress, 100));
      });

      await uploadTask;
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      await AsyncStorage.removeItem(`hseResults_${hseId}`);
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Navigate to HSEViewPDF screen
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
        }
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      return;
    }

    setSyncModalVisible(true);
    setUploadProgress(0);

    try {
      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      
      if (!savedStep1 || !savedStep2) {
        throw new Error('Previous step data (hseStep1 or hseStep2) is missing');
      }
      
      const step1 = JSON.parse(savedStep1);
      const step2 = JSON.parse(savedStep2);

      const newReportId = Date.now().toString();
      setReportId(newReportId);

      const reportData = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        step1,
        step2,
        step3: formData,
      };

      // Split reportData into smaller chunks
      const chunkSize = 100 * 1024; // 100KB per chunk
      const reportDataString = JSON.stringify(reportData);
      const reportBuffer = Buffer.from(reportDataString);
      const totalSize = reportBuffer.length;
      const chunks = [];
      for (let i = 0; i < totalSize; i += chunkSize) {
        chunks.push(reportBuffer.slice(i, i + chunkSize));
      }

      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${newReportId}`);
      
      // Save report metadata first
      await dbRef.set({
        id: newReportId,
        userId,
        createdAt: reportData.createdAt,
        status: 'uploading',
      });
      setUploadProgress(10); // Initial progress for metadata

      // Upload data chunks
      for (let i = 0; i < chunks.length; i++) {
        const chunkPath = `/GlobalHSEInspections/admins/${userId}/${newReportId}/chunks/chunk_${i}`;
        await database().ref(chunkPath).set(chunks[i].toString('base64'));
        const progress = 10 + ((i + 1) / (chunks.length + 2)) * 40; // 40% for data chunks
        if (isMounted.current) {
          setUploadProgress(progress);
        }
      }

      // Generate PDF
      const pdfPath = await generatePDFWithHtml(reportData);
      const storagePath = `GlobalHSEInspections/admins/${userId}/reports/HSE_071_Report_${newReportId}.pdf`;
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath, (progress) => {
        const pdfProgress = 50 + (progress / 2); // 50% for PDF upload
        if (isMounted.current) {
          setUploadProgress(pdfProgress);
        }
      });

      // Update report with PDF URL and final status
      await dbRef.update({
        pdfDownloadUrl,
        status: 'completed',
      });
      setUploadProgress(100);

      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));
      await flushStepData(newReportId);

      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'DD Supervisor Shift Report submitted successfully.',
          position: 'top',
        });
      }
    } catch (error) {
      console.error('Error in handleSubmit:', error);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to submit report.',
          position: 'top',
        });
        Alert.alert('Submit Error', `Failed to submit report.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false });
      }
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const renderInputField = (label, value, onChangeText, placeholder = '', keyboardType = 'default') => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TextInput
        style={styles.textInput}
        value={value}
        onChangeText={onChangeText}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#999"
      />
    </View>
  );

  const renderCasingInputs = () => (
    <View style={styles.sectionContainer}>
      <Text style={styles.sectionTitle}>Casing</Text>
      <View style={styles.casingContainer}>
        <View style={styles.casingInputWrapper}>
          {renderInputField('PW', formData.Casing?.PW || '', (value) => handleCasingChange('PW', value))}
        </View>
        <View style={styles.casingInputWrapper}>
          {renderInputField('PQ', formData.Casing?.PQ || '', (value) => handleCasingChange('PQ', value))}
        </View>
        <View style={styles.casingInputWrapper}>
          {renderInputField('HQ', formData.Casing?.HQ || '', (value) => handleCasingChange('HQ', value))}
        </View>
      </View>
    </View>
  );

  if (isLoading) {
    return (
      <View style={styles.loadingContainer}>
        <Text style={styles.loadingText}>Loading form data...</Text>
      </View>
    );
  }

  return (
    <KeyboardAvoidingView style={styles.container} behavior={Platform.OS === 'ios' ? 'padding' : 'height'}>
      <ScrollView 
        style={styles.scrollView} 
        showsVerticalScrollIndicator={false}
        contentContainerStyle={styles.scrollContent}
      >
        <View style={styles.content}>
          <Text style={styles.headerTitle}>DD Supervisor Shift Report - Form Data</Text>
          
          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Personnel Information</Text>
            {renderInputField('Supervisor', formData.Supervisor, (value) => handleInputChange('Supervisor', value), 'Enter supervisor name')}
            {renderInputField('Operator', formData.Operator, (value) => handleInputChange('Operator', value), 'Enter operator name')}
            {renderInputField('Geologist', formData.Geologist, (value) => handleInputChange('Geologist', value), 'Enter geologist name')}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Equipment Information</Text>
            {renderInputField('Vehicle ID', formData['Vehicle ID'], (value) => handleInputChange('Vehicle ID', value), 'Enter vehicle ID')}
            {renderInputField('Ori Tool Serial ID', formData['Ori Tool Serial ID'], (value) => handleInputChange('Ori Tool Serial ID', value), 'Enter Ori Tool Serial ID')}
            {renderInputField('Controller Serial ID', formData['Controller Serial ID'], (value) => handleInputChange('Controller Serial ID', value), 'Enter Controller Serial ID')}
            {renderInputField('SS Survey Dip', formData['SS Survey Dip'], (value) => handleInputChange('SS Survey Dip', value), 'Enter SS Survey Dip')}
            {renderInputField('Bit ID', formData['Bit ID'], (value) => handleInputChange('Bit ID', value), 'Enter Bit ID')}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Hole Information</Text>
            {renderInputField('Hole ID', formData['Hole ID'], (value) => handleInputChange('Hole ID', value), 'Enter Hole ID')}
            {renderInputField('Hole Depth', formData['Hole Depth'], (value) => handleInputChange('Hole Depth', value), 'Enter hole depth', 'numeric')}
            {renderInputField('Shift Meters', formData['Shift Meters'], (value) => handleInputChange('Shift Meters', value), 'Enter shift meters', 'numeric')}
            {renderInputField('Bit Meters', formData['Bit Meters'], (value) => handleInputChange('Bit Meters', value), 'Enter bit meters', 'numeric')}
            {renderInputField('Azi', formData.Azi, (value) => handleInputChange('Azi', value), 'Enter Azi', 'numeric')}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Safety Topics</Text>
            {renderInputField('Month Safety Topic', formData['Month Safety Topic'], (value) => handleInputChange('Month Safety Topic', value), 'Enter monthly safety topic')}
            {renderInputField('Daily Safety Topic', formData['Daily Safety Topic'], (value) => handleInputChange('Daily Safety Topic', value), 'Enter daily safety topic')}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Date and Shift</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.inputLabel}>Date</Text>
              <TouchableOpacity style={styles.datePickerButton} onPress={showDatePickerModal}>
                <Text style={styles.datePickerText}>{formData.Date || 'Select Date'}</Text>
                <MaterialIcons name="calendar-today" size={20} color="#0078D4" />
              </TouchableOpacity>
            </View>
            <View style={styles.shiftContainer}>
              <Text style={styles.inputLabel}>Shift</Text>
              <View style={styles.shiftButtons}>
                <TouchableOpacity
                  style={[styles.shiftButton, formData.Shift === 'Day' && styles.shiftButtonActive]}
                  onPress={() => handleInputChange('Shift', 'Day')}
                >
                  <Text style={[styles.shiftButtonText, formData.Shift === 'Day' && styles.shiftButtonTextActive]}>Day</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.shiftButton, formData.Shift === 'Night' && styles.shiftButtonActive]}
                  onPress={() => handleInputChange('Shift', 'Night')}
                >
                  <Text style={[styles.shiftButtonText, formData.Shift === 'Night' && styles.shiftButtonTextActive]}>Night</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Rig and Additional Information</Text>
            {renderInputField('Rig', formData.Rig, (value) => handleInputChange('Rig', value), 'Enter rig information')}
            {renderInputField('Engine Hours', formData['Engine Hours'], (value) => handleInputChange('Engine Hours', value), 'Enter engine hours', 'numeric')}
            {renderInputField('P.T.O', formData['P.T.O'], (value) => handleInputChange('P.T.O', value), 'Enter P.T.O information')}
          </View>

          {renderCasingInputs()}
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.previewButton} onPress={formatDataForPreview}>
          <MaterialIcons name="visibility" size={16} color="#FF9800" />
          <Text style={styles.previewButtonText}>Preview</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, syncModalVisible && styles.disabledButton]}
          onPress={handleSubmit}
          disabled={syncModalVisible}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
        />
      )}

      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              RC Supervisor Shift Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={showDataPreviewModal} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.dataPreviewModalContent}>
            <View style={styles.dataPreviewHeader}>
              <Text style={styles.dataPreviewTitle}>All Data Preview</Text>
              <TouchableOpacity 
                style={styles.closeButton}
                onPress={() => setShowDataPreviewModal(false)}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <ScrollView style={styles.dataPreviewScrollView}>
              <Text style={styles.dataPreviewText}>{formattedData}</Text>
            </ScrollView>
            <View style={styles.dataPreviewButtons}>
              <TouchableOpacity 
                style={styles.copyButton}
                onPress={copyToClipboard}
              >
                <MaterialIcons name="content-copy" size={20} color="#fff" />
                <Text style={styles.copyButtonText}>Copy to Clipboard</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.closeDataButton}
                onPress={() => setShowDataPreviewModal(false)}
              >
                <Text style={styles.closeDataButtonText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Generating Report" />
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 120,
  },
  content: {
    padding: 20,
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 20,
  },
  sectionContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    marginBottom: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 16,
  },
  inputContainer: {
    marginBottom: 16,
  },
  inputLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  dateContainer: {
    marginBottom: 16,
  },
  datePickerButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    backgroundColor: '#fff',
  },
  datePickerText: {
    fontSize: 16,
    color: '#333',
  },
  shiftContainer: {
    marginBottom: 16,
  },
  shiftButtons: {
    flexDirection: 'row',
    marginTop: 8,
  },
  shiftButton: {
    flex: 1,
    padding: 12,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    marginHorizontal: 4,
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  shiftButtonActive: {
    backgroundColor: '#0078D4',
    borderColor: '#0078D4',
  },
  shiftButtonText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '500',
  },
  shiftButtonTextActive: {
    color: '#fff',
  },
  casingContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    gap: 10,
  },
  casingInputWrapper: {
    flex: 1,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  previewButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  previewButtonText: {
    color: '#FF9800',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dataPreviewModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    width: '90%',
    maxWidth: 600,
    maxHeight: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  dataPreviewHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  dataPreviewTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
  },
  closeButton: {
    padding: 4,
  },
  dataPreviewScrollView: {
    maxHeight: 400,
    padding: 16,
  },
  dataPreviewText: {
    fontSize: 12,
    color: '#333',
    fontFamily: Platform.OS === 'ios' ? 'Courier' : 'monospace',
    lineHeight: 16,
  },
  dataPreviewButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    padding: 16,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  copyButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#0078D4',
    paddingHorizontal: 16,
    paddingVertical: 8,
    borderRadius: 6,
  },
  copyButtonText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#fff',
    fontWeight: '500',
  },
  closeDataButton: {
    paddingHorizontal: 16,
    paddingVertical: 8,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#666',
  },
  closeDataButtonText: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
});

export default HSE_072step3; 