const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require('cors');
const fetch = require('node-fetch');

const app = express();
const PORT = process.env.LOGBOOK_PORT || 5012;

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB connection - reuse existing connection if available
// Check if mongoose is already connected
if (mongoose.connection.readyState === 0) {
  // Using shared MongoDB connection from dbConnection.js
console.log('✅ Using shared MongoDB connection');
} else {
  console.log('✅ Logbook Handler: Using existing MongoDB connection');
}

// Counter Schema for generating sequential entry numbers
const CounterSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  sequence_value: { type: Number, default: 0 }
});

// Use existing Counter model if already compiled, otherwise create new one
const Counter = mongoose.models.Counter || mongoose.model('Counter', CounterSchema);

// Logbook Entry Schema
const LogbookEntrySchema = new mongoose.Schema({
  id: { type: String, required: true, unique: true },
  entryNumber: { type: String, required: true },
  country: { type: String, required: true },
  project: { type: String, required: true },
  selectedEquipment: { type: String, required: true },
  equipmentId: { type: String },
  currentMeterReading: { type: Number },
  equipmentStatus: { type: String, required: true },
  startReading: { type: String, required: true },
  endReading: { type: String, required: true },
  startDateTime: { type: Date, required: true },
  endDateTime: { type: Date, required: true },
  fuelOpening: { type: String },
  fuelConsumption: { type: String },
  fuelUnit: { type: String, default: 'KG' },
  operatorName: { type: String },
  workLocation: { type: String },
  workType: { type: String },
  workDescription: { type: String },
  materialDescription: { type: String },
  materialName: { type: String },
  materialQuantity: { type: String },
  materialUnit: { type: String, default: 'Ton' },
  issueReport: { type: String },
  issueName: { type: String },
  issueDetails: { type: String },
  adminUid: { type: String, required: true },
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
});

// Add index for faster queries
LogbookEntrySchema.index({ adminUid: 1, createdAt: -1 });
LogbookEntrySchema.index({ entryNumber: 1 });
LogbookEntrySchema.index({ country: 1, project: 1 });

const LogbookEntry = mongoose.model('GlobalLogbook', LogbookEntrySchema);

// Function to get next sequence value
async function getNextSequenceValue(sequenceName) {
  const sequenceDocument = await Counter.findByIdAndUpdate(
    sequenceName,
    { $inc: { sequence_value: 1 } },
    { new: true, upsert: true }
  );
  return sequenceDocument.sequence_value;
}

// Function to generate entry number
async function generateEntryNumber(country, project) {
  try {
    // Get country code
    let countryCode = '';
    if (country === 'Zambia') {
      countryCode = 'ZMB';
    } else if (country === 'DRC') {
      countryCode = 'DRC';
    } else {
      countryCode = country.substring(0, 3).toUpperCase();
    }

    // Get project code (first 3 letters)
    const projectCode = project.substring(0, 3).toUpperCase();

    // Get next sequence number
    const nextNumber = await getNextSequenceValue('logbookEntryNo');
    const paddedNumber = nextNumber.toString().padStart(3, '0');

    return `${countryCode}-${projectCode}-LBE-${paddedNumber}`;
  } catch (error) {
    console.error('Error generating entry number:', error);
    throw error;
  }
}

// Helper function to send logbook notifications
const sendLogbookNotifications = async (logbookData, createdByName, creatorEmail) => {
  try {
    console.log('🔔 Sending logbook notifications...');
    
    const notificationPayload = {
      logbookData: {
        _id: logbookData._id?.toString() || logbookData.id,
        id: logbookData.id,
        entryNumber: logbookData.entryNumber,
        selectedEquipment: logbookData.selectedEquipment || 'N/A',
        equipmentStatus: logbookData.equipmentStatus || 'N/A',
        operatorName: logbookData.operatorName || '',
        workType: logbookData.workType || '',
        workLocation: logbookData.workLocation || '',
        workDescription: logbookData.workDescription || '',
        startReading: logbookData.startReading,
        endReading: logbookData.endReading,
        fuelConsumption: logbookData.fuelConsumption || '',
        fuelUnit: logbookData.fuelUnit || 'KG',
        country: logbookData.country,
        project: logbookData.project,
      },
      createdByName,
      creatorEmail,
    };
    
    const response = await fetch('http://api.titandrillingzm.com:5026/api/logbook-notifications/new-entry', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(notificationPayload),
    });
    
    const result = await response.json();
    
    if (result.success) {
      console.log('✅ Logbook notifications sent successfully!');
      console.log('📊 Recipients:', result.recipients);
    } else {
      console.error('❌ Failed to send notifications:', result.error);
    }
    
    return result;
  } catch (error) {
    console.error('❌ Error sending logbook notifications:', error);
    return { success: false, error: error.message };
  }
};

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ 
    status: 'ok', 
    service: 'Logbook Handler',
    port: PORT,
    mongodb: mongoose.connection.readyState === 1 ? 'connected' : 'disconnected'
  });
});

// GET next entry number (preview)
app.get('/logbook/next-entry-number', async (req, res) => {
  try {
    const { country, project } = req.query;
    
    if (!country || !project) {
      return res.status(400).json({ 
        success: false, 
        message: 'Country and project are required' 
      });
    }

    const entryNumber = await generateEntryNumber(country, project);
    res.json({ success: true, entryNumber });
  } catch (error) {
    console.error('Error getting next entry number:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Failed to generate entry number',
      error: error.message 
    });
  }
});

// POST - Create new logbook entry
app.post('/logbook', async (req, res) => {
  try {
    const logbookData = req.body;
    
    console.log('📝 Creating new logbook entry:', {
      country: logbookData.country,
      project: logbookData.project,
      equipment: logbookData.selectedEquipment,
      adminUid: logbookData.adminUid
    });

    // Validate required fields
    if (!logbookData.country || !logbookData.project) {
      return res.status(400).json({ 
        success: false, 
        message: 'Country and project are required' 
      });
    }

    // Generate entry number if not provided
    if (!logbookData.entryNumber) {
      logbookData.entryNumber = await generateEntryNumber(
        logbookData.country, 
        logbookData.project
      );
    }

    // Generate ID if not provided
    if (!logbookData.id) {
      logbookData.id = Date.now().toString();
    }

    // Set timestamps
    logbookData.createdAt = logbookData.createdAt || new Date().toISOString();
    logbookData.updatedAt = new Date().toISOString();

    // Create the logbook entry
    const newEntry = new LogbookEntry(logbookData);
    await newEntry.save();

    console.log('✅ Logbook entry created successfully:', newEntry.entryNumber);

    // Send notifications in background (fire-and-forget)
    setImmediate(async () => {
      try {
        // Get creator name and email from adminUid - using different database
        const GlobalUserSchema = new mongoose.Schema({}, { strict: false, collection: 'GlobalUsers' });
        let GlobalUser;
        try {
          GlobalUser = mongoose.models.GlobalUser || mongoose.model('GlobalUser', GlobalUserSchema);
        } catch (error) {
          GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
        }
        
        // Use titan_drilling database for users
        const titanDrillingDB = mongoose.connection.useDb('titan_drilling');
        const GlobalUserModel = titanDrillingDB.model('GlobalUser', GlobalUserSchema);
        
        const creator = await GlobalUserModel.findOne({ _id: logbookData.adminUid }).select('name email');
        const createdByName = creator?.name || 'Unknown User';
        const creatorEmail = creator?.email || '';
        
        console.log('📧 Sending logbook notifications for creator:', createdByName);
        await sendLogbookNotifications(newEntry.toObject(), createdByName, creatorEmail);
      } catch (notifError) {
        console.error('❌ Error sending logbook notifications:', notifError);
      }
    });

    res.status(201).json({
      success: true,
      message: 'Logbook entry created successfully',
      data: newEntry
    });
  } catch (error) {
    console.error('❌ Error creating logbook entry:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create logbook entry',
      error: error.message
    });
  }
});

// GET - Get all logbook entries for a user
app.get('/logbook/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    
    console.log('📋 Fetching logbook entries for user:', adminUid);

    const entries = await LogbookEntry.find({ adminUid })
      .sort({ createdAt: -1 })
      .lean();

    console.log(`✅ Found ${entries.length} logbook entries`);

    res.json({
      success: true,
      count: entries.length,
      data: entries
    });
  } catch (error) {
    console.error('❌ Error fetching logbook entries:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch logbook entries',
      error: error.message
    });
  }
});

// GET - Get single logbook entry by ID
app.get('/logbook/entry/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    const entry = await LogbookEntry.findOne({ id }).lean();

    if (!entry) {
      return res.status(404).json({
        success: false,
        message: 'Logbook entry not found'
      });
    }

    res.json({
      success: true,
      data: entry
    });
  } catch (error) {
    console.error('❌ Error fetching logbook entry:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch logbook entry',
      error: error.message
    });
  }
});

// PUT - Update logbook entry
app.put('/logbook/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;
    
    console.log('📝 Updating logbook entry:', id);

    // Set updated timestamp
    updateData.updatedAt = new Date().toISOString();

    const updatedEntry = await LogbookEntry.findOneAndUpdate(
      { id },
      updateData,
      { new: true, runValidators: true }
    );

    if (!updatedEntry) {
      return res.status(404).json({
        success: false,
        message: 'Logbook entry not found'
      });
    }

    console.log('✅ Logbook entry updated successfully');

    res.json({
      success: true,
      message: 'Logbook entry updated successfully',
      data: updatedEntry
    });
  } catch (error) {
    console.error('❌ Error updating logbook entry:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update logbook entry',
      error: error.message
    });
  }
});

// DELETE - Delete logbook entry
app.delete('/logbook/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    console.log('🗑️ Deleting logbook entry:', id);

    const deletedEntry = await LogbookEntry.findOneAndDelete({ id });

    if (!deletedEntry) {
      return res.status(404).json({
        success: false,
        message: 'Logbook entry not found'
      });
    }

    console.log('✅ Logbook entry deleted successfully');

    res.json({
      success: true,
      message: 'Logbook entry deleted successfully',
      data: deletedEntry
    });
  } catch (error) {
    console.error('❌ Error deleting logbook entry:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete logbook entry',
      error: error.message
    });
  }
});

// GET - Get logbook statistics
app.get('/logbook/stats/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    
    const totalEntries = await LogbookEntry.countDocuments({ adminUid });
    
    const entriesByCountry = await LogbookEntry.aggregate([
      { $match: { adminUid } },
      { $group: { _id: '$country', count: { $sum: 1 } } }
    ]);

    const entriesByProject = await LogbookEntry.aggregate([
      { $match: { adminUid } },
      { $group: { _id: '$project', count: { $sum: 1 } } }
    ]);

    res.json({
      success: true,
      data: {
        totalEntries,
        byCountry: entriesByCountry,
        byProject: entriesByProject
      }
    });
  } catch (error) {
    console.error('❌ Error fetching logbook statistics:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch statistics',
      error: error.message
    });
  }
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('❌ Server error:', err);
  res.status(500).json({
    success: false,
    message: 'Internal server error',
    error: err.message
  });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`✅ Logbook Handler Server running on port ${PORT}`);
  console.log(`📡 API endpoint: http://api.titandrillingzm.com:${PORT}`);
});

module.exports = app;

