// MongoDB Configuration for Job Cards
// Using external IP for mobile app access
const JOBCARD_API_BASE_URL = 'http://168.231.113.219:5005';

// Helper function to build full URL
const buildURL = (endpoint) => {
  return `${JOBCARD_API_BASE_URL}${endpoint}`;
};

// Helper function for API calls
const apiCall = async (url, options = {}) => {
  try {
    console.log('🌐 Making API call to:', url);
    
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 45000); // 45 second timeout for better reliability
    
    const fetchOptions = {
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache',
        ...options.headers,
      },
      signal: controller.signal,
      timeout: 45000,
      ...options,
    };

    // Remove Content-Type for FormData uploads
    if (options.body && options.body instanceof FormData) {
      delete fetchOptions.headers['Content-Type'];
    }

    const response = await fetch(url, fetchOptions);

    clearTimeout(timeoutId);

    if (!response.ok) {
      const errorText = await response.text();
      console.error('❌ Server response error:', {
        status: response.status,
        statusText: response.statusText,
        body: errorText
      });
      throw new Error(`HTTP error! status: ${response.status} - ${errorText}`);
    }

    const data = await response.json();
    console.log('✅ API call successful:', url);
    return data;
  } catch (error) {
    console.error('❌ API call error:', error);
    if (error.name === 'AbortError') {
      throw new Error('Request timeout - please check your network connection');
    }
    if (error.message.includes('Network request failed')) {
      throw new Error('Network request failed - please check your internet connection and server availability');
    }
    throw error;
  }
};

// ========== JOB CARD API FUNCTIONS ==========

// Get all job cards for a user
export const getJobCards = async (adminUid, params = {}) => {
  try {
    const queryParams = new URLSearchParams({
      page: params.page || 1,
      limit: params.limit || 50,
      ...params
    });
    
    const url = buildURL(`/jobcards/${adminUid}?${queryParams}`);
    console.log('🔍 Fetching job cards from:', url);
    
    const response = await apiCall(url);
    return response;
  } catch (error) {
    console.error('Error fetching job cards:', error);
    throw error;
  }
};

// Get all job cards (global view)
export const getAllJobCards = async (params = {}) => {
  try {
    const queryParams = new URLSearchParams({
      page: params.page || 1,
      limit: params.limit || 10000,
      ...params
    });
    
    const url = buildURL(`/jobcards/all?${queryParams}`);
    console.log('🔍 Fetching all job cards from:', url);
    
    const response = await apiCall(url);
    return response;
  } catch (error) {
    console.error('Error fetching all job cards:', error);
    throw error;
  }
};

// Get a specific job card by ID
export const getJobCard = async (jobCardId) => {
  try {
    const url = buildURL(`/jobcards/report/${jobCardId}`);
    console.log('🔍 Fetching job card from:', url);
    
    const response = await apiCall(url);
    return response;
  } catch (error) {
    console.error('Error fetching job card:', error);
    throw error;
  }
};

// Create a new job card
export const createJobCard = async (adminUid, jobCardData) => {
  try {
    const url = buildURL('/jobcards');
    console.log('🔧 Creating job card:', url);
    console.log('🔧 JobCardData type:', typeof jobCardData);
    console.log('🔧 Items type:', typeof jobCardData?.items);
    console.log('🔧 Items is array:', Array.isArray(jobCardData?.items));
    
    const requestBody = {
      adminUid,
      jobCardData
    };
    
    console.log('🔧 Request body before stringify:', {
      adminUid,
      jobCardDataType: typeof jobCardData,
      itemsType: typeof jobCardData?.items,
      itemsLength: jobCardData?.items?.length
    });
    
    const response = await apiCall(url, {
      method: 'POST',
      body: JSON.stringify(requestBody)
    });
    
    return response;
  } catch (error) {
    console.error('Error creating job card:', error);
    throw error;
  }
};

// Update a job card (using original endpoint with adminUid)
export const updateJobCard = async (jobCardId, updateData) => {
  try {
    // Get adminUid from the job card data or use AsyncStorage
    let adminUid = updateData.adminUid;
    if (!adminUid) {
      const AsyncStorage = require('@react-native-async-storage/async-storage').default;
      adminUid = await AsyncStorage.getItem('userUid');
    }
    
    if (!adminUid) {
      throw new Error('Admin UID not available for job card update');
    }
    
    const url = buildURL(`/jobcards/${adminUid}/${jobCardId}`);
    console.log('🔧 Updating job card:', url);
    
    const response = await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(updateData)
    });
    
    return response;
  } catch (error) {
    console.error('Error updating job card:', error);
    throw error;
  }
};

// Update a job card (original - requires adminUid and jobCardId)
export const updateJobCardWithAdmin = async (adminUid, jobCardId, updateData) => {
  try {
    const url = buildURL(`/jobcards/${adminUid}/${jobCardId}`);
    console.log('🔧 Updating job card with admin:', url);
    
    const response = await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(updateData)
    });
    
    return response;
  } catch (error) {
    console.error('Error updating job card with admin:', error);
    throw error;
  }
};

// Delete a job card
export const deleteJobCard = async (adminUid, jobCardId) => {
  try {
    const url = buildURL(`/jobcards/${adminUid}/${jobCardId}`);
    console.log('🗑️ Deleting job card:', url);
    
    const response = await apiCall(url, {
      method: 'DELETE'
    });
    
    return response;
  } catch (error) {
    console.error('Error deleting job card:', error);
    throw error;
  }
};

// Upload job card PDF
export const uploadJobCardPdf = async (adminUid, formData) => {
  try {
    const url = buildURL(`/jobcards/upload-pdf/${adminUid}`);
    console.log('📤 Uploading job card PDF to:', url);
    
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 90000); // 90 second timeout for file uploads
    
    const fetchOptions = {
      method: 'POST',
      body: formData,
      signal: controller.signal,
      headers: {
        'Accept': 'application/json',
        'Cache-Control': 'no-cache',
      },
      // Don't set Content-Type, let fetch set it automatically for FormData
    };
    
    console.log('📤 Starting PDF upload with options:', {
      url,
      hasFormData: !!formData,
      timeout: '90s'
    });
    
    const response = await fetch(url, fetchOptions);

    clearTimeout(timeoutId);

    if (!response.ok) {
      const errorText = await response.text();
      console.error('❌ PDF upload server error:', {
        status: response.status,
        statusText: response.statusText,
        body: errorText
      });
      throw new Error(`HTTP error! status: ${response.status} - ${errorText}`);
    }

    const data = await response.json();
    console.log('✅ PDF upload successful:', url);
    console.log('📊 Upload response data:', data);
    return data;
  } catch (error) {
    console.error('❌ Error uploading job card PDF:', error);
    if (error.name === 'AbortError') {
      throw new Error('Upload timeout - please check your network connection');
    }
    if (error.message.includes('Network request failed')) {
      throw new Error('Network request failed - please check your internet connection');
    }
    throw error;
  }
};

// Upload job card signature
export const uploadJobCardSignature = async (adminUid, formData) => {
  try {
    const url = buildURL(`/jobcards/upload-signature/${adminUid}`);
    console.log('📤 Uploading job card signature to:', url);
    
    const response = await fetch(url, {
      method: 'POST',
      body: formData,
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error uploading job card signature:', error);
    throw error;
  }
};

// Get job card PDF download URL
export const getJobCardPdfDownloadUrl = (fileName, adminUid) => {
  return buildURL(`/jobcards/download-file/${adminUid}/${fileName}`);
};

// Get job card signature download URL
export const getJobCardSignatureDownloadUrl = (fileName, adminUid) => {
  return buildURL(`/jobcards/download-file/${adminUid}/${fileName}`);
};

// Get job card statistics
export const getJobCardStats = async (adminUid, params = {}) => {
  try {
    const queryParams = new URLSearchParams(params);
    const url = buildURL(`/jobcards/${adminUid}/stats?${queryParams}`);
    console.log('📊 Fetching job card stats from:', url);
    
    const response = await apiCall(url);
    return response;
  } catch (error) {
    console.error('Error fetching job card stats:', error);
    throw error;
  }
};

// ========== PERMISSION CHECK FUNCTIONS ==========

// Check if user has permission to create job cards
export const canCreateJobCard = (permissions) => {
  return Array.isArray(permissions) && permissions.includes('create');
};

// Check if user has permission to edit job cards
export const canEditJobCard = (permissions) => {
  return Array.isArray(permissions) && permissions.includes('edit');
};

// Check if user has permission to delete job cards
export const canDeleteJobCard = (permissions) => {
  return Array.isArray(permissions) && permissions.includes('delete');
};

// Check if user has permission to view only their own job cards
export const canViewOnlyMine = (permissions) => {
  return Array.isArray(permissions) && permissions.includes('onlyMineView');
};

// Check if user has permission to view all job cards
export const canViewAll = (permissions) => {
  return Array.isArray(permissions) && permissions.includes('viewAll');
};

// Get user job card permissions from user data
export const getUserJobCardPermissions = (userData) => {
  if (!userData || !userData.modules) return [];
  
  const jobCardModule = userData.modules.find(
    (module) => (module.module || '').toLowerCase() === 'jobcard'
  );
  
  return jobCardModule ? (jobCardModule.permissions || []) : [];
};

// Check if user has any job card permissions
export const hasJobCardPermissions = (permissions) => {
  return Array.isArray(permissions) && permissions.length > 0;
};

// ========== EXPORT DEFAULT OBJECT ==========

const JobCardAPI = {
  // API functions
  getJobCards,
  getAllJobCards,
  getJobCard,
  createJobCard,
  updateJobCard,
  deleteJobCard,
  uploadJobCardPdf,
  uploadJobCardSignature,
  getJobCardPdfDownloadUrl,
  getJobCardSignatureDownloadUrl,
  getJobCardStats,
  
  // Permission functions
  canCreateJobCard,
  canEditJobCard,
  canDeleteJobCard,
  canViewOnlyMine,
  canViewAll,
  getUserJobCardPermissions,
  hasJobCardPermissions,
  
  // Configuration
  API_BASE_URL: JOBCARD_API_BASE_URL,
  buildURL,
  apiCall
};

export default JobCardAPI;
