import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Modal,
  Alert,
  Dimensions,
  Animated,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import HSE_077_JSON from './HSE_077_json.json';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Generating PDF Report</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_077_Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [requestNumber, setRequestNumber] = useState('');
  const [reportId, setReportId] = useState('');
  const [formData, setFormData] = useState({
    dateOfInspection: '',
    safetyOfficer: '',
    timeOfInspection: '',
    siteSupervisor: '',
    actionToBeTaken: '',
    commitmentDate: '',
    recommendation: '',
  });
  const isMounted = useRef(true);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [showTimePicker, setShowTimePicker] = useState(false);
  const [dateValue, setDateValue] = useState(new Date());
  const [timeValue, setTimeValue] = useState(new Date());
  const [showCommitmentPicker, setShowCommitmentPicker] = useState(false);
  const [commitmentValue, setCommitmentValue] = useState(new Date());
  const [tempDateValue, setTempDateValue] = useState(new Date());
  const [tempTimeValue, setTempTimeValue] = useState(new Date());
  const [tempCommitmentValue, setTempCommitmentValue] = useState(new Date());

  useEffect(() => {
    loadSavedData();
    loadStep1Data();
    // initialize defaults for date/time if empty
    setFormData((prev) => ({
      ...prev,
      dateOfInspection: prev.dateOfInspection || formatDate(new Date()),
      timeOfInspection: prev.timeOfInspection || formatTime(new Date()),
    }));
    return () => {
      isMounted.current = false;
    };
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse077Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData.formData || formData);
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        formData,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse077Step3', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const loadStep1Data = async () => {
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      if (savedHse) {
        const hseData = JSON.parse(savedHse);
        if (hseData.requestNumber) {
          setRequestNumber(hseData.requestNumber);
        }
      }
    } catch (error) {
      console.error('Error loading step 1 data:', error);
    }
  };

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
    setTimeout(() => saveData(), 500);
  };

  const formatDate = (date) => {
    try {
      const d = typeof date === 'string' ? new Date(date) : date;
      return d.toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric' });
    } catch {
      return '';
    }
  };

  const formatTime = (date) => {
    try {
      const d = typeof date === 'string' ? new Date(date) : date;
      return d.toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit', hour12: false });
    } catch {
      return '';
    }
  };

  const handleSubmit = async () => {
    if (!formData.safetyOfficer || !formData.siteSupervisor) {
      Alert.alert('Required Fields', 'Please fill in all required fields before submitting.');
      return;
    }

    try {
      await saveData();
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Error in handleSubmit:', error);
      Alert.alert('Error', 'Failed to submit. Please try again.');
    }
  };

  const handleBackToHome = () => {
    setShowSuccessModal(false);
    setActiveStep(1);
  };

  const deepCleanObject = (obj) => {
    if (typeof obj !== 'object' || obj === null) return obj;
    if (Array.isArray(obj)) return obj.map(deepCleanObject);
    const cleaned = {};
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      cleaned[key] = typeof value === 'object' && value !== null ? deepCleanObject(value) : String(value || '');
    });
    return cleaned;
  };

  const getAdminUserId = async () => {
    const userId = await AsyncStorage.getItem('userUid');
    if (!userId) throw new Error('User ID not found. Please log in again.');
    return userId;
  };

  const incrementReportCounter = async () => {
    try {
      const response = await HSEInspectionAPI.incrementCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error incrementing report counter in MongoDB:', error);
      return '001';
    }
  };

  const uploadPDFToMongoDB = async (filePath, userId) => {
    try {
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_077_${userId}_${Date.now()}.pdf`
      };
      const response = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      if (!response.success) throw new Error('Failed to upload PDF to MongoDB');
      return response.downloadUrl;
    } catch (error) {
      console.error('❌ Error uploading PDF to MongoDB:', error);
      throw error;
    }
  };

  const calculateSafetySummary = (step2) => {
    const sections = step2?.sections || [];
    const selections = step2?.selections || {};
    let totalSafe = 0;
    let totalUnsafe = 0;
    let totalNA = 0;
    sections.forEach((section) => {
      (section.subItems || []).forEach((item) => {
        const s = selections[item.key];
        if (s === 'Safe' || s === 'OK' || s === 'Okay') totalSafe += 1;
        else if (s === 'Unsafe' || s === 'Faulty' || s === 'Not Ok') totalUnsafe += 1;
        else if (s === 'N/A' || s === 'Not Applicable') totalNA += 1;
      });
    });
    const totalObservations = totalSafe + totalUnsafe + totalNA;
    const percentageSafe = totalObservations > 0 ? Math.round((totalSafe / totalObservations) * 100) : 0;
    return { totalSafe, totalUnsafe, totalNA, totalObservations, percentageSafe };
  };

  const generatePDFWithHtml = async (report) => {
    const step1 = report.step1 || {};
    const step2 = report.step2 || {};
    const step3 = report.step3 || {};
    const sections = step2.sections || [];
    const selections = step2.selections || {};
    const notes = step2.notes || {};

    const summary = calculateSafetySummary(step2);

    let checklistRows = '';
    if (sections.length > 0) {
      sections.forEach((section) => {
        if (section.subItems && section.subItems.length > 0) {
          checklistRows += `
            <tr>
              <td colspan="5" class="section-header">${section.name || 'SECTION'}</td>
            </tr>
          `;
          section.subItems.forEach((item) => {
            const selection = selections[item.key] || '';
            const comment = notes[item.key] || '';
            const isNA = selection === 'N/A' || selection === 'Not Applicable';
            const isSafe = selection === 'Safe' || selection === 'OK' || selection === 'Okay';
            const isUnsafe = selection === 'Unsafe' || selection === 'Faulty' || selection === 'Not Ok';
            checklistRows += `
              <tr>
                <td>${item.itemName || 'Unknown Item'}</td>
                <td style="text-align:center;">${isNA ? '✓' : ''}</td>
                <td style="text-align:center;">${isSafe ? '✓' : ''}</td>
                <td style="text-align:center;">${isUnsafe ? '✓' : ''}</td>
                <td>${comment}</td>
              </tr>
            `;
          });
        }
      });
    }

    const html = `
      <!DOCTYPE html>
      <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0" />
          <title>HSE-077 Lifting Equipment Inspection</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
            .container { width: 100%; max-width: 800px; margin: 20px auto; border: 1px solid #000; }
            .header { display:flex; justify-content: space-between; align-items:center; background:#f0f0f0; padding:10px; border-bottom:1px solid #000; }
            .details table { border-collapse: collapse; width: 100%; }
            .details td { padding: 2px 4px; }
            .inspection-details { display:grid; grid-template-columns: 1fr 1fr; padding:6px; border-bottom:1px solid #000; gap:4px; }
            .table { width:100%; border-collapse: collapse; }
            .table th, .table td { border:1px solid #000; padding:8px; }
            .table th { background:#d3d3d3; }
            .section-header { background:#a9a9a9; font-weight:bold; }
            .summary { margin-top: 10px; }
            .summary th { background:#d3d3d3; }
            .percentage { background:#e8f5e8; font-weight:bold; color:#2e7d32; }
            .additional { margin-top: 18px; }
            .additional th { background:#d3d3d3; }
            .totals-row td { font-weight: bold; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <div class="logo"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="160"/></div>
              <div class="details">
                <table>
                  <tr><td>Document name : ${HSE_077_JSON.document_name || 'Lifting Equipment Inspection'}</td></tr>
                  <tr><td>Document number : ${HSE_077_JSON.document_id || 'HSE-077'}</td></tr>
                  <tr><td>Version : ${HSE_077_JSON.version || '1'}</td></tr>
                  <tr><td>Revision date : ${HSE_077_JSON.version_date || new Date().toLocaleDateString()}</td></tr>
                  <tr><td>Approved by : ${HSE_077_JSON.approved_by || 'HSM'}</td></tr>
                  <tr><td>Request # : ${report.id || 'N/A'}</td></tr>
                </table>
              </div>
            </div>
            <div class="inspection-details">
              <div>DATE OF INSPECTION</div><div>${step3.dateOfInspection || ''}</div>
              <div>SAFETY OFFICER</div><div>${step3.safetyOfficer || ''}</div>
              <div>TIME OF INSPECTION</div><div>${step3.timeOfInspection || ''}</div>
              <div>SITE SUPERVISOR</div><div>${step3.siteSupervisor || ''}</div>
            </div>
            <table class="table">
              <tr>
                <th>LIFTING EQUIPMENT INSPECTION CHECK LIST</th>
                <th>Not Applicable</th>
                <th>Safe</th>
                <th>Unsafe</th>
                <th>Comments</th>
              </tr>
              ${checklistRows}
              <tr class="totals-row">
                <td>TOTAL</td>
                <td style="text-align:center;">${summary.totalNA}</td>
                <td style="text-align:center;">${summary.totalSafe}</td>
                <td style="text-align:center;">${summary.totalUnsafe}</td>
                <td></td>
              </tr>
            </table>
            <div class="summary">
              <table class="table">
                <tr>
                  <th>Total Observations</th>
                  <th>Safe</th>
                  <th>Unsafe</th>
                  <th>% Safe</th>
                </tr>
                <tr>
                  <td style="text-align:center;">${summary.totalObservations}</td>
                  <td style="text-align:center;">${summary.totalSafe}</td>
                  <td style="text-align:center;">${summary.totalUnsafe}</td>
                  <td class="percentage" style="text-align:center;">${summary.percentageSafe}%</td>
                </tr>
              </table>
            </div>
            <div class="additional">
              <table class="table">
                <tr>
                  <th>Action to be taken</th>
                  <th>Commitment Date</th>
                  <th>Recommendation</th>
                </tr>
                <tr>
                  <td>${step3.actionToBeTaken || ''}</td>
                  <td>${step3.commitmentDate || ''}</td>
                  <td>${step3.recommendation || ''}</td>
                </tr>
              </table>
            </div>
          </div>
        </body>
      </html>
    `;

    const file = await RNHTMLtoPDF.convert({ html, fileName: report.id || `HSE_077_${Date.now()}`, directory: 'Documents' });
    return file.filePath;
  };

  const handleGenerateReport = async () => {
    if (!formData.safetyOfficer || !formData.siteSupervisor) {
      Alert.alert('Required Fields', 'Please fill in Safety Officer and Site Supervisor.');
      return;
    }
    try {
      setSyncModalVisible(true);
      setUploadProgress(5);
      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hse077Step2');
      if (!savedStep1 || !savedStep2) throw new Error('Previous step data not found.');
      const step1 = JSON.parse(savedStep1);
      const step2 = JSON.parse(savedStep2);

      const newReportId = step1.requestNumber || `HSE-077-${await incrementReportCounter()}`;
      setReportId(newReportId);

      const step1Data = deepCleanObject(step1);
      const step2Data = deepCleanObject(step2);
      const step3Data = deepCleanObject(formData);

      const reportData = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        step1: step1Data,
        step2: step2Data,
        step3: step3Data,
      };

      // Generate PDF
      console.log('📄 Generating PDF for HSE-077...');
      const pdfPath = await generatePDFWithHtml({ id: newReportId, step1: step1Data, step2: step2Data, step3: step3Data });
      console.log('✅ PDF generated at:', pdfPath);
      setUploadProgress(40);

      // Upload PDF to MongoDB
      console.log('📤 Uploading PDF to MongoDB...');
      const pdfDownloadUrl = await uploadPDFToMongoDB(pdfPath, userId);
      console.log('✅ PDF uploaded:', pdfDownloadUrl);
      setUploadProgress(70);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      const completeReportData = {
        _id: newReportId,
        userId,
        requestNumber: newReportId,
        status: 'completed',
        step1: step1Data,
        step2: step2Data,
        step3: step3Data,
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId
      };

      const saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
      if (!saveResponse.success) throw new Error('Failed to save report to MongoDB');
      console.log('✅ Report saved to MongoDB');
      setUploadProgress(90);

      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const summary = calculateSafetySummary(step2);
        
        const notificationPayload = {
          hseInspectionData: {
            _id: newReportId,
            id: newReportId,
            requestNumber: newReportId,
            checklistName: 'HSE-077 Lifting Equipment Inspection',
            equipmentName: step1.selectedEquipment?.equipmentName || '',
            country: step1.country || '',
            project: step1.project || '',
            percentageSafe: summary.percentageSafe || 0,
            totalSafe: summary.totalSafe || 0,
            totalUnsafe: summary.totalUnsafe || 0,
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('http://168.231.113.219:5028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }

      // Delete temp PDF
      await RNFS.unlink(pdfPath).catch(() => {});

      // Increment global counter after success
      await incrementReportCounter();
      setUploadProgress(100);

      // Clear local step data
      try {
        await AsyncStorage.removeItem('hseStep1');
        await AsyncStorage.removeItem('hse077Step2');
        await AsyncStorage.removeItem('hse077Step3');
      } catch (e) {
        console.warn('Failed clearing local step data', e);
      }

      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        Toast.show({ type: 'success', text1: 'Success', text2: 'Report saved and PDF generated.', position: 'bottom' });
      }
    } catch (error) {
      console.error('HSE_077 generate report error:', error);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({ type: 'error', text1: 'Error', text2: error.message || 'Failed to generate report.', position: 'bottom' });
        Alert.alert('Error', error.message || 'Failed to generate report.');
      }
    }
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.headingText}>Lifting Equipment Inspection - Final Review</Text>
        
        <View style={styles.requestNumberContainer}>
          <Text style={styles.requestNumberLabel}>Request #</Text>
          <Text style={styles.requestNumberValue}>{requestNumber || 'Loading...'}</Text>
        </View>

        <View style={styles.formSection}>
          <Text style={styles.sectionTitle}>Inspection Details</Text>
          
          <View style={styles.inputContainerFull}>
            <Text style={styles.label}>Date of Inspection *</Text>
            <TouchableOpacity style={styles.pickerButton} onPress={() => {
              setTempDateValue(dateValue);
              setShowDatePicker(true);
            }}>
              <Text style={styles.pickerButtonText}>{formatDate(dateValue) || formData.dateOfInspection}</Text>
              <MaterialIcons name="calendar-today" size={20} color="#015185" />
            </TouchableOpacity>
          </View>

          <View style={styles.inputContainerFull}>
            <Text style={styles.label}>Time of Inspection *</Text>
            <TouchableOpacity style={styles.pickerButton} onPress={() => {
              setTempTimeValue(timeValue);
              setShowTimePicker(true);
            }}>
              <Text style={styles.pickerButtonText}>{formatTime(timeValue) || formData.timeOfInspection}</Text>
              <MaterialIcons name="access-time" size={20} color="#015185" />
            </TouchableOpacity>
          </View>

          <View style={styles.inputContainerFull}>
            <Text style={styles.label}>Safety Officer *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.safetyOfficer}
              onChangeText={(value) => handleInputChange('safetyOfficer', value)}
              placeholder="Enter safety officer name"
            />
          </View>

          <View style={styles.inputContainerFull}>
            <Text style={styles.label}>Site Supervisor *</Text>
            <TextInput
              style={styles.textInput}
              value={formData.siteSupervisor}
              onChangeText={(value) => handleInputChange('siteSupervisor', value)}
              placeholder="Enter supervisor name"
            />
          </View>

          {/** Equipment and Location are populated from Step 1 in PDF */}
        </View>

        <View style={styles.formSection}>
          <Text style={styles.sectionTitle}>Action & Recommendations</Text>
          
          <View style={styles.inputContainer}>
            <Text style={styles.label}>Action to be taken</Text>
            <TextInput
              style={[styles.textInput, styles.textArea]}
              value={formData.actionToBeTaken}
              onChangeText={(value) => handleInputChange('actionToBeTaken', value)}
              placeholder="Describe actions required to address unsafe conditions"
              multiline
              numberOfLines={4}
            />
          </View>

          <View style={styles.inputContainerFull}>
            <Text style={styles.label}>Commitment Date</Text>
            <TouchableOpacity style={styles.pickerButton} onPress={() => {
              setTempCommitmentValue(commitmentValue);
              setShowCommitmentPicker(true);
            }}>
              <Text style={styles.pickerButtonText}>{formData.commitmentDate || formatDate(commitmentValue)}</Text>
              <MaterialIcons name="calendar-today" size={20} color="#015185" />
            </TouchableOpacity>
          </View>

          <View style={styles.inputContainerFull}>
            <Text style={styles.label}>Recommendation</Text>
            <TextInput
              style={[styles.textInput, styles.textArea]}
              value={formData.recommendation}
              onChangeText={(value) => handleInputChange('recommendation', value)}
              placeholder="Enter recommendation"
              multiline
              numberOfLines={4}
            />
          </View>
        </View>

        {/** Signature not required */}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(2)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.submitButton} onPress={handleGenerateReport} disabled={syncModalVisible}>
          <Text style={styles.submitButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon1} name="check" size={16} color="#fff" />
        </TouchableOpacity>
      </View>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <View style={styles.successIconContainer}>
              <MaterialIcons name="check-circle" size={60} color="#4CAF50" />
            </View>
            <Text style={styles.successTitle}>Report Generated Successfully!</Text>
            <Text style={styles.successMessage}>Your lifting equipment inspection report has been saved and a PDF is available.</Text>
            <View style={{ flexDirection: 'row' }}>
              <TouchableOpacity style={styles.backButton} onPress={handleBackToHome}>
                <Text style={styles.backButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.backButton, { backgroundColor: '#4CAF50', marginLeft: 10 }]}
                onPress={async () => {
                  try {
                    const response = await HSEInspectionAPI.getHSEInspectionById(reportId || requestNumber);
                    if (!response.success || !response.data) {
                      Alert.alert('PDF not ready', 'Report not found in MongoDB.');
                      return;
                    }
                    const pdfUrl = response.data.pdfDownloadUrl;
                    if (!pdfUrl) {
                      Alert.alert('PDF not ready', 'PDF URL not found.');
                      return;
                    }
                    setShowSuccessModal(false);
                    navigation.navigate('HSEViewPDF', { pdfUrl, onGoBack: () => navigation.navigate('HSEInspectionHome') });
                  } catch (e) {
                    console.error('❌ Error opening PDF from MongoDB:', e);
                    Alert.alert('Error', 'Unable to open PDF from MongoDB.');
                  }
                }}
              >
                <Text style={styles.backButtonText}>View Report</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      {/* iOS Date Picker Modal */}
      {Platform.OS === 'ios' && showDatePicker && (
        <Modal
          visible={showDatePicker}
          animationType="slide"
          transparent={true}
          onRequestClose={() => setShowDatePicker(false)}
          presentationStyle="pageSheet"
        >
          <View style={styles.iosDatePickerBackground}>
            <View style={styles.iosDatePickerContainer}>
              <View style={styles.iosDatePickerHeader}>
                <TouchableOpacity onPress={() => setShowDatePicker(false)}>
                  <Text style={styles.iosDatePickerButton}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity onPress={() => {
                  setDateValue(tempDateValue);
                  handleInputChange('dateOfInspection', formatDate(tempDateValue));
                  setShowDatePicker(false);
                }}>
                  <Text style={[styles.iosDatePickerButton, styles.iosDatePickerDoneButton]}>Done</Text>
                </TouchableOpacity>
              </View>
              <DateTimePicker
                value={tempDateValue}
                mode="date"
                display="spinner"
                onChange={(event, selectedDate) => {
                  if (selectedDate) setTempDateValue(selectedDate);
                }}
                style={styles.iosDatePicker}
              />
            </View>
          </View>
        </Modal>
      )}

      {/* Android Date Picker */}
      {Platform.OS === 'android' && showDatePicker && (
        <DateTimePicker
          value={dateValue}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => {
            setShowDatePicker(false);
            if (selectedDate) {
              setDateValue(selectedDate);
              handleInputChange('dateOfInspection', formatDate(selectedDate));
            }
          }}
        />
      )}

      {/* iOS Time Picker Modal */}
      {Platform.OS === 'ios' && showTimePicker && (
        <Modal
          visible={showTimePicker}
          animationType="slide"
          transparent={true}
          onRequestClose={() => setShowTimePicker(false)}
          presentationStyle="pageSheet"
        >
          <View style={styles.iosDatePickerBackground}>
            <View style={styles.iosDatePickerContainer}>
              <View style={styles.iosDatePickerHeader}>
                <TouchableOpacity onPress={() => setShowTimePicker(false)}>
                  <Text style={styles.iosDatePickerButton}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity onPress={() => {
                  setTimeValue(tempTimeValue);
                  handleInputChange('timeOfInspection', formatTime(tempTimeValue));
                  setShowTimePicker(false);
                }}>
                  <Text style={[styles.iosDatePickerButton, styles.iosDatePickerDoneButton]}>Done</Text>
                </TouchableOpacity>
              </View>
              <DateTimePicker
                value={tempTimeValue}
                mode="time"
                display="spinner"
                onChange={(event, selectedTime) => {
                  if (selectedTime) setTempTimeValue(selectedTime);
                }}
                style={styles.iosDatePicker}
              />
            </View>
          </View>
        </Modal>
      )}

      {/* Android Time Picker */}
      {Platform.OS === 'android' && showTimePicker && (
        <DateTimePicker
          value={timeValue}
          mode="time"
          display="default"
          onChange={(event, selectedTime) => {
            setShowTimePicker(false);
            if (selectedTime) {
              setTimeValue(selectedTime);
              handleInputChange('timeOfInspection', formatTime(selectedTime));
            }
          }}
        />
      )}

      {/* iOS Commitment Date Picker Modal */}
      {Platform.OS === 'ios' && showCommitmentPicker && (
        <Modal
          visible={showCommitmentPicker}
          animationType="slide"
          transparent={true}
          onRequestClose={() => setShowCommitmentPicker(false)}
          presentationStyle="pageSheet"
        >
          <View style={styles.iosDatePickerBackground}>
            <View style={styles.iosDatePickerContainer}>
              <View style={styles.iosDatePickerHeader}>
                <TouchableOpacity onPress={() => setShowCommitmentPicker(false)}>
                  <Text style={styles.iosDatePickerButton}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity onPress={() => {
                  setCommitmentValue(tempCommitmentValue);
                  handleInputChange('commitmentDate', formatDate(tempCommitmentValue));
                  setShowCommitmentPicker(false);
                }}>
                  <Text style={[styles.iosDatePickerButton, styles.iosDatePickerDoneButton]}>Done</Text>
                </TouchableOpacity>
              </View>
              <DateTimePicker
                value={tempCommitmentValue}
                mode="date"
                display="spinner"
                onChange={(event, selectedDate) => {
                  if (selectedDate) setTempCommitmentValue(selectedDate);
                }}
                style={styles.iosDatePicker}
              />
            </View>
          </View>
        </Modal>
      )}

      {/* Android Commitment Date Picker */}
      {Platform.OS === 'android' && showCommitmentPicker && (
        <DateTimePicker
          value={commitmentValue}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => {
            setShowCommitmentPicker(false);
            if (selectedDate) {
              setCommitmentValue(selectedDate);
              handleInputChange('commitmentDate', formatDate(selectedDate));
            }
          }}
        />
      )}
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message={`Processing... ${uploadProgress}%`} />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: 15,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    marginVertical: 20,
  },
  requestNumberContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    paddingHorizontal: 15,
    paddingVertical: 12,
    borderRadius: 8,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  requestNumberLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  requestNumberValue: {
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
  },
  formSection: {
    marginBottom: 25,
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
  },
  inputRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    gap: 15,
  },
  inputContainer: {
    flex: 1,
    marginBottom: 15,
  },
  inputContainerFull: {
    width: '100%',
    marginBottom: 15,
  },
  label: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 5,
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
  },
  textArea: {
    minHeight: 80,
    textAlignVertical: 'top',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 12,
    backgroundColor: '#fff',
  },
  pickerButtonText: {
    fontSize: 16,
    color: '#333',
  },
  signatureButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#e8f4fd',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#015185',
  },
  signatureButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 10,
  },
  submitButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    // backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    paddingBottom: 18,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  submitButtonText: {
    // color: '#fff',
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginRight: 8,
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 2,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 30,
    alignItems: 'center',
    width: width * 0.85,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
    width: width * 0.8,
  },
  syncModalText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  syncIcondiv: {
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginBottom: 8,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  successIconContainer: {
    marginBottom: 20,
  },
  successTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 15,
  },
  successMessage: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    lineHeight: 22,
    marginBottom: 25,
  },
  backButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 30,
    borderRadius: 6,
    alignItems: 'center',
  },
  backButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  iosDatePickerBackground: {
    flex: 1,
    justifyContent: 'flex-end',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  iosDatePickerContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    paddingBottom: 20,
  },
  iosDatePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  iosDatePickerButton: {
    fontSize: 17,
    color: '#015185',
    fontWeight: '600',
  },
  iosDatePickerDoneButton: {
    fontWeight: 'bold',
  },
  iosDatePicker: {
    height: 200,
    backgroundColor: '#fff',
  },
});

export default HSE_077_Step3;