
import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Modal,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');
const { height } = Dimensions.get('window');

const HSE_124Step2 = ({ setActiveStep }) => {
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [notes, setNotes] = useState({});
  const [inspectionId, setInspectionId] = useState('hse124');
  const [noteModal, setNoteModal] = useState({
    visible: false,
    itemKey: '',
    currentNote: '',
  });

  useEffect(() => {
    loadSavedData();
    setupChecklistData();
  }, []);

  const setupChecklistData = () => {
    const checklistData = [
      {
        key: 'donkey_engine',
        name: 'DONKEY ENGINE',
        subItems: [
          { key: 'engineOil', itemName: 'Engine oil (CHANGED/LEVEL)', inputType: 'Yes/No' },
          { key: 'coolant', itemName: 'Coolant (LEAKS/LEVEL)', inputType: 'Yes/No' },
          { key: 'airFilter1', itemName: 'Air filter #1 (CLEANED HOUSING/CHANGED)', inputType: 'Yes/No' },
          { key: 'airFilter2', itemName: 'Air filter #2 (CLEANED HOUSING/ROTATED)', inputType: 'Yes/No' },
          { key: 'fuelFilter', itemName: 'Fuel filter (CHANGED)', inputType: 'Yes/No' },
          { key: 'fuelSystem', itemName: 'Fuel system (LEAKS)', inputType: 'Yes/No' },
          { key: 'vBelts', itemName: 'V belts (CONDITION/TENSION)', inputType: 'Yes/No' },
          { key: 'radiator', itemName: 'Radiator (CLEANED/PRESSURE TESTED)', inputType: 'Yes/No' },
          { key: 'radiatorCap', itemName: 'Radiator cap (INSPECTED)', inputType: 'Yes/No' },
          { key: 'engineMountings', itemName: 'Engine mountings (WORN/SECURED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'every_200_hrs',
        name: 'EVERY 2000 HRS',
        subItems: [
          { key: 'valveClearance', itemName: 'Valve clearance (ADJUSTED/CHECKED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'electrical',
        name: 'ELECTRICAL',
        subItems: [
          { key: 'batteries', itemName: 'Batteries (POLE CONDITION/JELLED)', inputType: 'Yes/No' },
          { key: 'dateEngraved', itemName: 'Date engraved', inputType: 'Yes/No' },
          { 
            key: 'voltageAtCrankingDonkey', 
            itemName: 'Voltage at cranking (DONKEY)', 
            inputType: 'VoltageInput',
            voltageType: 'DONKEY'
          },
          { 
            key: 'voltageAtCrankingTruck', 
            itemName: 'Voltage at cranking (TRUCK)', 
            inputType: 'VoltageInput',
            voltageType: 'TRUCK',
            positions: ['1', '2', 'COMBINED']
          },
          { key: 'alternator', itemName: 'Alternator (MOUNTING/CHECKED)', inputType: 'Yes/No' },
          { key: 'starter', itemName: 'Starter (MOUNTING/TESTED)', inputType: 'Yes/No' },
          { key: 'lights', itemName: 'Lights (WORKING/WIRING)', inputType: 'Yes/No' },
          { 
            key: 'chargingSystemDonke', 
            itemName: 'Charging system (CHARGING/WIRES/FUSES)', 
            inputType: 'VoltageInput',
            voltageType: 'DONKEY',
            positions: ['V RUNNING', 'V NOT RUNNING']
          },
          { 
            key: 'chargingSystemTruck', 
            itemName: 'Charging system (CHARGING/WIRES/FUSES)', 
            inputType: 'VoltageInput',
            voltageType: 'TRUCK',
            positions: ['V RUNNING', 'V NOT RUNNING']
          },
          { key: 'rpmVoltTempOilGauge', itemName: 'RPM/Volt/Temp/Oil gauge (WORKING/CALIBRATED)', inputType: 'Yes/No' },
          { key: 'eStops', itemName: 'E stops (CHECKED/TESTED)', inputType: 'Yes/No' },
          { 
            key: 'engineProtectionCutO', 
            itemName: 'Engine protection cut (OIL PRESSURE TEST)', 
            inputType: 'Yes/No'
          },
          { 
            key: 'engineProtectionTemp', 
            itemName: 'Engine protection cut (TEMPERATURE TEST)', 
            inputType: 'Yes/No'
          },
          { key: 'horn', itemName: 'Horn (CHECKED/TESTED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'marst',
        name: 'MARST',
        subItems: [
          { key: 'keeperSpannerRam', itemName: 'Keeper spanner/ram (CONDITION/PINS/LEAKS)', inputType: 'Yes/No' },
          { 
            key: 'breakOutRam', 
            itemName: 'Break out ram (PINS/LEAKS)', 
            inputType: 'RamPositions',
            positions: ['RAM 1', 'RAM 2', 'RAM 3']
          },
          { key: 'breakOutRotationMoto', itemName: 'Break out rotation motor (SECURED/LEAKS)', inputType: 'Yes/No' },
          { key: 'traverseRam', itemName: 'Traverse ram (LEAKS/PINS/WHEELS)', inputType: 'Yes/No' },
          { key: 'marstRaisingRams', itemName: 'Marst raising rams (LEAKS/PINS/BUSHES/PRES HOLD)', inputType: 'Yes/No' },
          { key: 'headTiltRam', itemName: 'Head tilt ram (LEAKS/PINS/BUSHES/PRES HOLD)', inputType: 'Yes/No' },
          { key: 'hoseArmSwivel', itemName: '3" Hose arm/swivel (SECURED/INSPECTED)', inputType: 'Yes/No' },
          { key: 'tableHalfMoons', itemName: 'Table half moons (CHANGED/CONDITION)', inputType: 'Yes/No' },
          { key: 'tableDeckBush', itemName: 'Table deck bush (CHANGED/CONDITION)', inputType: 'Yes/No' },
          { key: 'greaseNipples', itemName: 'Grease nipples (GREASED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'drill_head',
        name: 'DRILL HEAD',
        subItems: [
          { key: 'gearbox', itemName: 'Gearbox (PACKED WITH GREASE)', inputType: 'Yes/No' },
          { key: 'rotationMotors', itemName: 'Rotation motors (SECURE/LEAKS)', inputType: 'Yes/No' },
          { key: 'wearCollar', itemName: 'Wear collar (INSPECTED/CHANGED)', inputType: 'Yes/No' },
          { key: 'bullshaft', itemName: 'Bullshaft (PLAY/WEAR)', inputType: 'Yes/No' },
          { key: 'subsaver', itemName: 'Subsaver (CONDITION)', inputType: 'Yes/No' },
          { key: 'headInnertube', itemName: 'Head innertube (PLAY/WEAR)', inputType: 'Yes/No' },
          { key: 'deflectorBox', itemName: 'Deflector box (WEAR/BLOCKED)', inputType: 'Yes/No' },
          { key: 'cycloneHoses', itemName: 'Cyclone hoses (CONDITION)', inputType: 'Yes/No' },
          { key: 'airHoses', itemName: '2" Air hoses (CONDITION)', inputType: 'Yes/No' },
          { key: 'safetySocks', itemName: 'Safety socks (CONDITION CLIPS)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'control_panel',
        name: 'CONTROL PANEL',
        subItems: [
          { key: 'gauges', itemName: 'Gauges (LEAKS/CALIBRATED)', inputType: 'Yes/No' },
          { key: 'levers', itemName: 'Levers (STIFF/PLAY/LEAKS)', inputType: 'Yes/No' },
          { key: 'dripTray', itemName: 'Drip tray (PRESENT/CLEAN)', inputType: 'Yes/No' },
          { key: 'soakBags', itemName: 'Soak bags (PRESENT/CLEAN)', inputType: 'Yes/No' },
          { key: 'quickCouplers', itemName: 'Quick couplers (CLEAN/LEAKS/INSPECTED)', inputType: 'Yes/No' },
          { key: 'jacklegs', itemName: 'Jacklegs (PRESS HOLD/LEAKS)', inputType: 'Yes/No' },
          { key: 'hydraulicHoses', itemName: 'Hydraulic hoses (LEAKS/INSPECTED)', inputType: 'Yes/No' },
          { key: 'fireExtinguisher', itemName: 'Fire extinguisher (INSPECT/DATE OF EXPIRY)', inputType: 'Yes/No' },
        ]
      }
    ];

    setSections(checklistData);
    
    // Initialize expanded sections
    const initialExpanded = {};
    checklistData.forEach((section) => {
      initialExpanded[section.key] = false;
    });
    setExpandedSections(initialExpanded);
  };

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse124Step2');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setSelections(parsedData.selections || {});
        setNotes(parsedData.notes || {});
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        selections,
        notes,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse124Step2', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Yes/No':
        return ['YES/OK', 'NO/DEF'];
      case 'VoltageInput':
      case 'RamPositions':
        return ['YES/OK', 'NO/DEF'];
      default:
        return ['YES/OK', 'NO/DEF'];
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = {
        ...prev,
        [itemKey]: option,
      };
      saveData();
      return updatedSelections;
    });
  };

  const openNoteModal = (itemKey) => {
    setNoteModal({
      visible: true,
      itemKey,
      currentNote: notes[itemKey] || '',
    });
  };

  const saveNote = () => {
    const { itemKey, currentNote } = noteModal;
    if (currentNote.trim()) {
      setNotes(prev => ({
        ...prev,
        [itemKey]: currentNote.trim()
      }));
      saveData();
    }
    setNoteModal({ visible: false, itemKey: '', currentNote: '' });
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({
      ...prev,
      [sectionKey]: !prev[sectionKey],
    }));
  };

  const renderItem = (item, index) => {
    const selectedOption = selections[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {item.itemName}
          </Text>
          <View style={styles.itemRow}>
            <View style={styles.optionsContainer}>
              {options.map((option) => {
                const isSelected = selectedOption === option;
                return (
                  <TouchableOpacity
                    key={option}
                    style={[
                      styles.optionButton,
                      isSelected && styles.selectedOptionButton,
                      isSelected && option === 'YES/OK' && styles.okButton,
                      isSelected && option === 'NO/DEF' && styles.rejectButton,
                    ]}
                    onPress={() => handleSelectOption(item.key, option)}
                  >
                    <Text
                      style={[
                        styles.optionText,
                        isSelected && styles.optionTextSelected,
                      ]}
                    >
                      {option}
                    </Text>
                  </TouchableOpacity>
                );
              })}
            </View>
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => openNoteModal(item.key)}
              >
                <MaterialIcons name="note-add" size={24} color="#333" />
              </TouchableOpacity>
            </View>
          </View>
          
          {/* Voltage input fields */}
          {item.inputType === 'VoltageInput' && item.positions && (
            <View style={styles.voltageContainer}>
              <Text style={styles.voltageLabel}>Voltage readings:</Text>
              <View style={styles.voltageGrid}>
                {item.positions.map((position) => {
                  const voltageKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                  const voltageValue = selections[voltageKey] || '';
                  
                  return (
                    <View key={position} style={styles.voltageItem}>
                      <Text style={styles.voltagePositionLabel}>{position}:</Text>
                      <TextInput
                        style={styles.voltageInput}
                        value={voltageValue}
                        onChangeText={(value) => handleSelectOption(voltageKey, value)}
                        placeholder="Enter voltage"
                        placeholderTextColor="#999"
                        keyboardType="numeric"
                      />
                      <Text style={styles.voltageUnit}>VOLTS</Text>
                    </View>
                  );
                })}
              </View>
            </View>
          )}

          {/* Position-specific input fields for RAMs */}
          {item.inputType === 'RamPositions' && item.positions && (
            <View style={styles.positionsContainer}>
              <Text style={styles.positionsLabel}>Check RAM positions:</Text>
              <View style={styles.positionsGrid}>
                {item.positions.map((position) => {
                  const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                  const positionValue = selections[positionKey] || '';
                  
                  return (
                    <View key={position} style={styles.positionItem}>
                      <Text style={styles.positionLabel}>{position}:</Text>
                      <View style={styles.positionOptions}>
                        {['OK', 'CHECK', 'REPLACE'].map((option) => {
                          const isSelected = positionValue === option;
                          return (
                            <TouchableOpacity
                              key={option}
                              style={[
                                styles.positionOptionButton,
                                isSelected && styles.positionOptionSelected,
                                isSelected && option === 'OK' && styles.okButton,
                                isSelected && option === 'CHECK' && styles.checkButton,
                                isSelected && option === 'REPLACE' && styles.replaceButton,
                              ]}
                              onPress={() => handleSelectOption(positionKey, option)}
                            >
                              <Text
                                style={[
                                  styles.positionOptionText,
                                  isSelected && styles.positionOptionTextSelected,
                                ]}
                              >
                                {option}
                              </Text>
                            </TouchableOpacity>
                          );
                        })}
                      </View>
                    </View>
                  );
                })}
              </View>
            </View>
          )}

          {itemNote && (
            <Text style={styles.noteText}>
              Note: {itemNote}
            </Text>
          )}
        </View>
        <View style={styles.itemBorder} />
      </View>
    );
  };

  const handleNext = () => {
    setActiveStep(3);
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.checklistTitle}>RC Drill Rig Workshop Checklist</Text>

        {sections.length > 0 ? (
          sections.map((section) => {
            // Calculate selected items count for this section
            const sectionItems = section.subItems || [];
            const selectedItemsCount = sectionItems.filter(item => selections[item.key]).length;
            const totalItemsCount = sectionItems.length;
            
            return (
              <View key={section.key}>
                <TouchableOpacity
                  style={styles.sectionHeader}
                  onPress={() => toggleSection(section.key)}
                >
                  <View style={styles.sectionHeaderContent}>
                    <MaterialIcons
                      name={expandedSections[section.key] ? 'expand-less' : 'expand-more'}
                      size={20}
                      color={expandedSections[section.key] ? '#015185' : '#333'}
                    />
                    <View style={styles.sectionHeaderTextContainer}>
                      <Text style={styles.sectionHeaderText}>{section.name}</Text>
                      <Text style={styles.sectionItemCount}>
                        ({selectedItemsCount}/{totalItemsCount} selected)
                      </Text>
                    </View>
                  </View>
                </TouchableOpacity>
                {expandedSections[section.key] && (
                  <View style={styles.sectionContent}>
                    {section.subItems.length > 0 ? (
                      section.subItems.map((item, index) => renderItem(item, index))
                    ) : (
                      <Text style={styles.emptySectionText}>No inspection items</Text>
                    )}
                  </View>
                )}
              </View>
            );
          })
        ) : (
          <Text style={styles.emptySectionText}>No sections available</Text>
        )}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(1)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Note Modal */}
      <Modal
        visible={noteModal.visible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note</Text>
            <TextInput
              style={styles.modalTextInput}
              value={noteModal.currentNote}
              onChangeText={(value) => setNoteModal(prev => ({ ...prev, currentNote: value }))}
              placeholder="Enter your note here..."
              multiline
              numberOfLines={4}
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.modalButtonPrimary]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.modalButtonTextPrimary]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionHeaderTextContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  sectionItemCount: {
    fontSize: 14,
    color: '#666',
    marginLeft: 5,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  iconsContainer: {
    flexDirection: 'row',
  },
  iconButton: {
    padding: 10,
  },
  noteText: {
    fontSize: 14,
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 20,
    width: width * 0.9,
    maxHeight: height * 0.6,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 100,
    textAlignVertical: 'top',
    marginBottom: 20,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginHorizontal: 5,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#fff',
  },
  modalButtonPrimary: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  modalButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  modalButtonTextPrimary: {
    color: '#fff',
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  // New styles for voltage and position inputs
  voltageContainer: {
    marginTop: 10,
    paddingHorizontal: 10,
    paddingVertical: 8,
    backgroundColor: '#ECF0F3',
    borderRadius: 5,
  },
  voltageLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 5,
  },
  voltageGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
  },
  voltageItem: {
    width: '48%', // Two columns
    marginBottom: 10,
  },
  voltagePositionLabel: {
    fontSize: 13,
    color: '#666',
    marginBottom: 3,
  },
  voltageInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    paddingHorizontal: 8,
    paddingVertical: 6,
    fontSize: 14,
    backgroundColor: '#fff',
    color: '#333',
  },
  voltageUnit: {
    fontSize: 12,
    color: '#666',
    marginTop: 3,
  },
  positionsContainer: {
    marginTop: 15,
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  positionsLabel: {
    fontSize: 15,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 12,
    textAlign: 'center',
  },
  positionsGrid: {
    flexDirection: 'column',
    gap: 12,
  },
  positionItem: {
    width: '100%',
    marginBottom: 8,
  },
  positionLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
    marginBottom: 8,
    textAlign: 'center',
  },
  positionOptions: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    backgroundColor: '#fff',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#ddd',
    padding: 8,
  },
  positionOptionButton: {
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 6,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#ECF0F3',
    borderWidth: 1,
    borderColor: '#ddd',
  },
  positionOptionSelected: {
    backgroundColor: '#333',
    borderColor: '#333',
  },
  positionOptionText: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  positionOptionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  checkButton: {
    backgroundColor: '#FFD700', // Gold color for CHECK
  },
  replaceButton: {
    backgroundColor: '#FF4B2B', // Red color for REPLACE
  },
});

export default HSE_124Step2;