# Titan Drilling FCM Notification System Setup

This document provides complete setup instructions for implementing Firebase Cloud Messaging (FCM) notifications in the Titan Drilling app.

## Features Implemented

1. **Automatic FCM Token Generation**: Generates and stores FCM tokens for each user
2. **Database Storage**: Saves FCM tokens to Firebase Realtime Database under `/Globalusers/admins/{uid}/NotificationFCMtoken`
3. **Dynamic Notifications**: Send notifications to all users via PHP API
4. **Real-time Status**: Shows online/offline status and user information
5. **Error Handling**: Comprehensive error handling for network issues and API failures

## Files Overview

- `FirebaseFCMNotificaion.js` - React Native component for FCM functionality
- `titanNotificationsFCM.php` - PHP API for sending notifications to all users
- `firebase_config_example.php` - Example configuration file
- `README_FCM_Setup.md` - This setup guide

## Setup Instructions

### 1. Firebase Console Setup

1. Go to [Firebase Console](https://console.firebase.google.com/)
2. Create a new project or select your existing Titan Drilling project
3. Enable the following services:
   - **Authentication** (Email/Password)
   - **Realtime Database**
   - **Cloud Messaging**

### 2. Get Firebase Credentials

1. Go to **Project Settings** > **General**
2. Copy the following values:
   - **Web API Key**
   - **Database URL**
   - **Project ID**

3. Go to **Project Settings** > **Cloud Messaging**
4. Copy the **Server Key** (this is your FCM Server Key)

### 3. Update PHP Configuration

1. Open `titanNotificationsFCM.php`
2. Replace the placeholder values with your actual Firebase credentials:

```php
$firebaseConfig = [
    'apiKey' => 'YOUR_ACTUAL_FIREBASE_API_KEY',
    'databaseURL' => 'YOUR_ACTUAL_FIREBASE_DATABASE_URL',
    'projectId' => 'YOUR_ACTUAL_FIREBASE_PROJECT_ID'
];

$fcmServerKey = 'YOUR_ACTUAL_FCM_SERVER_KEY';
```

### 4. Upload PHP File

1. Upload `titanNotificationsFCM.php` to your server at `https://owninfotech.com/`
2. Ensure PHP has the `curl` extension enabled
3. Set proper file permissions (644 or 755)

### 5. Firebase Database Rules

Set up security rules in your Firebase Realtime Database:

```json
{
  "rules": {
    "Globalusers": {
      "admins": {
        "$uid": {
          ".read": "auth != null && auth.uid == $uid",
          ".write": "auth != null && auth.uid == $uid",
          "NotificationFCMtoken": {
            ".validate": "newData.isString() && newData.val().length > 0"
          }
        }
      }
    },
    "notifications": {
      ".read": "auth != null",
      ".write": "auth != null"
    }
  }
}
```

### 6. Test the Setup

#### Test PHP API Connection
```bash
curl -X GET "https://owninfotech.com/titanNotificationsFCM.php?action=test_connection"
```

Expected response:
```json
{
  "success": true,
  "message": "PHP API is working correctly",
  "timestamp": "2024-01-01 12:00:00",
  "server": "owninfotech.com"
}
```

#### Check FCM Tokens
```bash
curl -X GET "https://owninfotech.com/titanNotificationsFCM.php?action=check_tokens"
```

#### Send Test Notification
```bash
curl -X POST "https://owninfotech.com/titanNotificationsFCM.php" \
  -H "Content-Type: application/json" \
  -d '{
    "action": "send_to_all",
    "title": "Test Notification",
    "body": "This is a test notification from Titan Drilling",
    "senderId": "test_user"
  }'
```

## How It Works

### 1. FCM Token Generation and Storage

When a user clicks "Get FCM Token & Save to Database":

1. App requests notification permissions
2. Generates FCM token using Firebase Messaging
3. Saves token to Firebase database at `/Globalusers/admins/{userId}/NotificationFCMtoken`
4. Also stores device information and timestamp

### 2. Dynamic Notification Sending

When a user clicks "Send to All Users":

1. App sends POST request to PHP API with notification details
2. PHP API fetches all FCM tokens from Firebase database
3. Sends push notification to each token individually
4. Logs the notification result to Firebase
5. Returns success/failure statistics

### 3. Database Structure

```
/Globalusers
  /admins
    /{userId}
      - NotificationFCMtoken: "fcm_token_string"
      - lastTokenUpdate: "2024-01-01T12:00:00.000Z"
      - deviceInfo: {
          platform: "ios" | "android",
          version: "version_number",
          timestamp: "2024-01-01T12:00:00.000Z"
        }
      - other user data...

/notifications (optional)
  - notification logs with results
```

## Usage in App

### 1. Navigate to FCM Screen

The FCM notification screen is accessible from the HomeScreen under "HR Apply Application" section.

### 2. Generate FCM Token

1. Click "Get FCM Token & Save to Database"
2. Grant notification permissions if prompted
3. Token will be generated and saved to Firebase automatically

### 3. Send Notifications

1. Enter notification title and message
2. Click "Send to All Users"
3. Notification will be sent to all registered users' devices

## Error Handling

The system handles various error scenarios:

- **Network Issues**: Shows offline mode and queues actions
- **Permission Denied**: Alerts user about notification permissions
- **API Failures**: Shows specific error messages
- **Invalid Tokens**: Logs failed deliveries and continues with others

## Security Considerations

1. **FCM Server Key**: Never expose in client-side code
2. **Database Rules**: Use Firebase security rules to protect data
3. **Input Validation**: All inputs are validated in PHP
4. **HTTPS**: Use HTTPS for all API calls
5. **Rate Limiting**: Consider implementing rate limiting for production

## Troubleshooting

### Common Issues

1. **"No FCM tokens found"**
   - Ensure users have generated FCM tokens first
   - Check Firebase database structure

2. **"Permission denied"**
   - Verify Firebase database rules
   - Check API key and database URL

3. **"Network error"**
   - Check internet connection
   - Verify PHP file is accessible
   - Ensure curl extension is enabled

4. **"Invalid server key"**
   - Verify FCM server key is correct
   - Check Firebase project settings

### Debug Steps

1. Test PHP API connection first
2. Check FCM tokens in database
3. Verify Firebase credentials
4. Check server logs for errors
5. Test with a single device first

## Support

For technical support or questions about this implementation, please refer to:

- Firebase Documentation: https://firebase.google.com/docs
- FCM Documentation: https://firebase.google.com/docs/cloud-messaging
- React Native Firebase: https://rnfirebase.io/

## Version History

- **v1.0**: Initial implementation with FCM token storage and dynamic notifications
- Features: Token generation, database storage, PHP API, error handling 