import React, { useState, useEffect } from 'react';
import { View, Text, TextInput, TouchableOpacity, StyleSheet, Dimensions, Modal, ActivityIndicator } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', ...props }) => (
  <View style={inputStyles.container}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && (
        <Text style={inputStyles.requiredError}> *</Text>
      )}
    </Text>
    <TextInput
      style={[
        inputStyles.input,
        showError ? inputStyles.inputError : null,
        props.multiline ? styles.descriptionInput : null,
      ]}
      value={value}
      onChangeText={onChangeText}
      {...props}
      placeholderTextColor="#888"
    />
    {showError && (
      <Text style={inputStyles.errorText}>{errorMessage}</Text>
    )}
  </View>
);

const Inspection_info_Checklist = ({ checklistData, setChecklistData, setActiveStep }) => {
  const [checklistTitle, setChecklistTitle] = useState(checklistData?.title || '');
  const [description, setDescription] = useState(checklistData?.description || '');
  const [showErrors, setShowErrors] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [hasSaved, setHasSaved] = useState(false);
  const [lastSaveTime, setLastSaveTime] = useState(0);
  const navigation = useNavigation();

  useEffect(() => {
    setChecklistTitle(checklistData?.title || '');
    setDescription(checklistData?.description || '');
  }, [checklistData]);

  const handleNext = async () => {
    // Prevent multiple saves
    if (isSaving || hasSaved) {
      console.log('⚠️ Save already in progress or completed');
      return;
    }

    // Add debounce mechanism to prevent rapid-fire requests
    const currentTime = Date.now();
    const timeSinceLastSave = currentTime - lastSaveTime;
    
    if (timeSinceLastSave < 2000) { // 2 seconds debounce
      console.log('⚠️ Too soon since last save, skipping duplicate call');
      return;
    }

    setShowErrors(true);
    if (!checklistTitle.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Checklist Title is required.',
      });
      return;
    }

    setIsSaving(true);
    setHasSaved(true);
    setLastSaveTime(currentTime);
    
    const updatedData = {
      ...checklistData,
      id: checklistData?.id || Date.now().toString(),
      title: checklistTitle.trim() || 'Untitled Checklist',
      description: description.trim(),
      items: checklistData?.items || [],
      createdAt: checklistData?.createdAt || new Date().toISOString(),
      updatedAt: new Date().toISOString(),
    };

    try {
      console.log('💾 Updating local checklist info:', updatedData.title);
      setChecklistData(updatedData);
      setActiveStep(2);
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Checklist info updated.',
      });
    } catch (error) {
      console.error('Error updating checklist info:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to update checklist info.',
      });
      setHasSaved(false); // Reset on error
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    navigation.goBack();
  };

  return (
    <View style={styles.container}>
      <Modal
        transparent={true}
        animationType="fade"
        visible={isSaving}
        onRequestClose={() => {}}
      >
        <View style={styles.modalContainer}>
          <View style={styles.loaderContent}>
            <ActivityIndicator size="large" color="#0078D4" />
            <Text style={styles.loaderText}>Saving Checklist...</Text>
          </View>
        </View>
      </Modal>

      <CustomInputField
        label="Checklist Title"
        value={checklistTitle}
        onChangeText={setChecklistTitle}
        placeholder="Enter checklist title"
        isRequired
        showError={showErrors && !checklistTitle.trim()}
        errorMessage="Checklist Title is required"
      />

      <CustomInputField
        label="Description"
        value={description}
        onChangeText={setDescription}
        placeholder="Enter description"
        multiline
        numberOfLines={4}
      />

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleCancel}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
          disabled={isSaving}
        >
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <Toast />
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
    alignSelf: 'center',
    width: width * 0.92,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
    textAlign: 'left',
    alignSelf: 'flex-start',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
    width: '100%',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    padding: 20,
    backgroundColor: '#fff',
    alignItems: 'center',
  },
  descriptionInput: {
    height: 160,
    textAlignVertical: 'top',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  modalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  loaderContent: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    alignItems: 'center',
  },
  loaderText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
});

export default Inspection_info_Checklist;