import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  Modal,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const MONGODB_API_URL = 'http://168.231.113.219:5004';

const Create_Custom_c = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [checklists, setChecklists] = useState([]);
  const [filteredChecklists, setFilteredChecklists] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [userData, setUserData] = useState(null);
  const [userId, setUserId] = useState(null);

  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const userDataString = await AsyncStorage.getItem('userData');
        if (userDataString) {
          const parsedUserData = JSON.parse(userDataString);
          setUserData(parsedUserData);
          setUserId(parsedUserData.uid);
          console.log('✅ User data loaded:', parsedUserData.uid);
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User not logged in.',
          });
        }
      } catch (error) {
        console.error('❌ Error loading user data:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load user data.',
        });
      }
    };

    fetchUserData();
  }, []);

  useEffect(() => {
    if (isFocused && userId) {
      loadChecklists();
    }
  }, [isFocused, userId]);

  const loadChecklists = async () => {
    if (!userId) {
      console.log('⚠️ No user ID available');
      return;
    }

    setIsLoading(true);
    try {
      console.log('📋 Fetching ALL custom checklists (including from other users)');
      
      // Fetch ALL custom checklists, not just the user's
      const response = await fetch(`${MONGODB_API_URL}/checklists/custom/all`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const result = await response.json();
      console.log('📋 Fetch result:', result);

      if (result.success && result.data) {
        setChecklists(result.data);
        setFilteredChecklists(result.data);
        
        // Save to AsyncStorage for offline access
        await AsyncStorage.setItem('customChecklists', JSON.stringify(result.data));
        
        // Count how many are created by current user
        const myChecklists = result.data.filter(item => item.createdBy === userId);
        
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: `${result.data.length} total checklists (${myChecklists.length} by you)`,
        });
      } else {
        console.log('⚠️ No checklists found or fetch failed');
        setChecklists([]);
        setFilteredChecklists([]);
      }
    } catch (error) {
      console.error('❌ Error loading checklists:', error);
      
      // Try to load from AsyncStorage as fallback
      try {
        const cachedData = await AsyncStorage.getItem('customChecklists');
        if (cachedData) {
          const parsedData = JSON.parse(cachedData);
          setChecklists(parsedData);
          setFilteredChecklists(parsedData);
          Toast.show({
            type: 'info',
            text1: 'Offline Mode',
            text2: 'Showing cached checklists.',
          });
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Failed to load checklists.',
          });
        }
      } catch (cacheError) {
        console.error('❌ Error loading from cache:', cacheError);
      }
    } finally {
      setIsLoading(false);
    }
  };

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text.trim()) {
      const filtered = checklists.filter((item) =>
        (item.title || '').toLowerCase().includes(text.toLowerCase()) ||
        (item.description || '').toLowerCase().includes(text.toLowerCase())
      );
      setFilteredChecklists(filtered);
    } else {
      setFilteredChecklists(checklists);
    }
  };

  const handleDeleteChecklist = async (checklistId) => {
    if (!userId) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'User not logged in.',
      });
      return;
    }

    try {
      console.log('🗑️ Deleting checklist:', checklistId);
      
      const response = await fetch(`${MONGODB_API_URL}/checklists/custom/${userId}/${checklistId}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const result = await response.json();
      console.log('🗑️ Delete result:', result);

      if (result.success) {
        const updatedChecklists = checklists.filter((item) => item._id !== checklistId && item.id !== checklistId);
        setChecklists(updatedChecklists);
        setFilteredChecklists(updatedChecklists);
        await AsyncStorage.setItem('customChecklists', JSON.stringify(updatedChecklists));
        
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Checklist deleted successfully.',
        });
      } else {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: result.error || 'Failed to delete checklist.',
        });
      }
    } catch (error) {
      console.error('❌ Error deleting checklist:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to delete checklist.',
      });
    } finally {
      setDropdownVisible(null);
    }
  };

  const truncateDescription = (description) => {
    if (!description) return 'No description';
    const words = description.split(' ');
    return words.length > 4 ? `${words.slice(0, 4).join(' ')}...` : description;
  };

  const renderChecklist = ({ item }) => {
    const isCreatedByMe = item.createdBy === userId;
    
    return (
      <View style={styles.checklistItem}>
        <TouchableOpacity
          style={styles.checklistTitleContainer}
          onPress={() =>
            navigation.navigate('CreatechecklistHeader', { checklist: item })
          }
        >
          <View style={styles.titleRow}>
            <Text style={styles.checklistTitle}>{item.title || 'Untitled Checklist'}</Text>
            {isCreatedByMe && (
              <View style={styles.creatorBadge}>
                <Text style={styles.creatorBadgeText}>me</Text>
              </View>
            )}
          </View>
          <Text style={styles.checklistDescription}>
            {truncateDescription(item.description)}
          </Text>
        <View style={styles.itemsContainer}>
          <Text style={styles.itemsLabel}>Items:</Text>
          {item.items?.length > 0 ? (
            item.items.map((section, index) => (
              <View key={index} style={styles.sectionContainer}>
                <Text style={styles.sectionTitle}>{section.text || `Section ${index + 1}`}</Text>
                {section.subItems?.length > 0 ? (
                  <FlatList
                    data={section.subItems}
                    keyExtractor={(subItem, subIndex) => `${index}-${subIndex}`}
                    renderItem={({ item: subItem }) => (
                      <Text style={styles.subItemText}>
                        • {subItem.itemName || 'Unnamed Item'}
                      </Text>
                    )}
                    style={styles.subItemsList}
                  />
                ) : (
                  <Text style={styles.noSubItemsText}>No inspection items</Text>
                )}
              </View>
            ))
          ) : (
            <Text style={styles.noItemsText}>No sections available</Text>
          )}
        </View>
        </TouchableOpacity>
        {isCreatedByMe && (
          <View style={styles.menuContainer}>
            <TouchableOpacity
              style={styles.menuButton}
              onPress={(e) => {
                e.stopPropagation();
                setDropdownVisible(dropdownVisible === (item._id || item.id) ? null : (item._id || item.id));
              }}
            >
              <MaterialIcons name="more-vert" size={20} color="#0078D4" />
            </TouchableOpacity>
            {dropdownVisible === (item._id || item.id) && (
              <View style={styles.dropdown}>
                <TouchableOpacity
                  style={styles.dropdownItem}
                  activeOpacity={0.7}
                  onPress={(e) => {
                    e.stopPropagation();
                    navigation.navigate('CreatechecklistHeader', { checklist: item });
                    setDropdownVisible(null);
                  }}
                >
                  <MaterialIcons name="edit" size={18} color="#0078D4" style={styles.dropdownIcon} />
                  <Text style={styles.dropdownText}>Edit Checklist</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={styles.dropdownItem}
                  activeOpacity={0.7}
                  onPress={(e) => {
                    e.stopPropagation();
                    handleDeleteChecklist(item._id || item.id);
                  }}
                >
                  <MaterialIcons name="delete" size={18} color="#dc3545" style={styles.dropdownIcon} />
                  <Text style={styles.dropdownText}>Delete Checklist</Text>
                </TouchableOpacity>
              </View>
            )}
          </View>
        )}
      </View>
    );
  };

  if (!userId) {
    return (
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <TouchableOpacity onPress={() => navigation.goBack()}>
                <MaterialIcons name="chevron-left" size={28} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>All Custom Checklists</Text>
            </View>
          </View>
        </LinearGradient>
        <View style={styles.mainContent}>
          <ActivityIndicator size="large" color="#015185" />
          <Text style={styles.errorText}>Loading user data...</Text>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      {/* Loader Modal */}
      <Modal
        transparent={true}
        animationType="fade"
        visible={isLoading}
        onRequestClose={() => {}}
      >
        <View style={styles.modalContainer}>
          <View style={styles.loaderContent}>
            <ActivityIndicator size="large" color="#0078D4" />
            <Text style={styles.loaderText}>Loading Checklists...</Text>
          </View>
        </View>
      </Modal>

      {/* Header with Gradient */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="chevron-left" size={28} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>All Custom Checklists</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search checklists..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <TouchableOpacity 
        style={styles.listContainer}
        activeOpacity={1}
        onPress={() => setDropdownVisible(null)}
      >
        <FlatList
          data={filteredChecklists}
          keyExtractor={(item) => (item._id || item.id || Math.random()).toString()}
          renderItem={renderChecklist}
          ListEmptyComponent={
            <View style={styles.emptyContainer}>
              <MaterialIcons name="assignment" size={64} color="#ccc" />
              <Text style={styles.emptyText}>
                {searchQuery ? 'No checklists match your search.' : 'No checklists found.'}
              </Text>
              <Text style={styles.emptySubtext}>
                {searchQuery ? 'Try a different search term.' : 'Create your first custom checklist!'}
              </Text>
            </View>
          }
          style={styles.list}
        />
      </TouchableOpacity>

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.startButton}
            onPress={() => navigation.navigate('CreatechecklistHeader', { checklist: null })}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>Create New Checklist</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Toast Configuration */}
      <Toast position="bottom" />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.06,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  listContainer: {
    flex: 1,
  },
  list: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  checklistItem: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    paddingVertical: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  checklistTitleContainer: {
    flex: 1,
  },
  titleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 8,
  },
  checklistTitle: {
    fontSize: width * 0.045,
    color: '#333',
    fontWeight: '500',
    flex: 1,
  },
  creatorBadge: {
    backgroundColor: '#0078D4',
    paddingHorizontal: 8,
    paddingVertical: 3,
    borderRadius: 12,
    marginLeft: 6,
  },
  creatorBadgeText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  checklistDescription: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: 4,
  },
  itemsContainer: {
    marginTop: width * 0.02,
  },
  itemsLabel: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
    marginBottom: width * 0.01,
  },
  sectionContainer: {
    marginBottom: width * 0.02,
  },
  sectionTitle: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '500',
    marginBottom: width * 0.01,
  },
  subItemsList: {
    marginLeft: width * 0.04,
  },
  subItemText: {
    fontSize: width * 0.035,
    color: '#555',
    marginBottom: width * 0.005,
  },
  noSubItemsText: {
    fontSize: width * 0.035,
    color: '#888',
    marginLeft: width * 0.04,
  },
  noItemsText: {
    fontSize: width * 0.035,
    color: '#888',
  },
  menuContainer: {
    position: 'relative',
  },
  menuButton: {
    padding: width * 0.02,
  },
  dropdown: {
    position: 'absolute',
    right: 0,
    top: 40,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 3,
    width: width * 0.4,
    zIndex: 1000,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownIcon: {
    marginRight: 10,
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '400',
  },
  emptyContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.1,
    paddingHorizontal: width * 0.05,
  },
  emptyText: {
    fontSize: width * 0.045,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.03,
    fontWeight: '600',
  },
  emptySubtext: {
    fontSize: width * 0.035,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.02,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  modalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  loaderContent: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    alignItems: 'center',
  },
  loaderText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
  mainContent: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  errorText: {
    fontSize: 16,
    color: 'red',
    textAlign: 'center',
  },
});

export default Create_Custom_c;