import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Modal,
  FlatList,
  KeyboardAvoidingView,
  Platform,
  PermissionsAndroid,
  Animated,
  Alert,
  Dimensions,
  Image,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DropDownPicker from 'react-native-dropdown-picker';
// Removed Firebase auth import - now using local storage for authentication
import Geolocation from '@react-native-community/geolocation';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import DateTimePicker from '@react-native-community/datetimepicker';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');

// Function to get current week dates
const getCurrentWeekDates = (startDate) => {
  const weekDates = [];
  for (let i = 0; i < 7; i++) {
    const date = new Date(startDate);
    date.setDate(startDate.getDate() + i);
    weekDates.push(date);
  }
  return weekDates;
};

// Function to format date to day name
const formatDayName = (date) => {
  return date.toLocaleDateString('en-US', { weekday: 'short' }).charAt(0);
};

// Custom Input Field Component
const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', ...props }) => (
  <View style={inputStyles.container}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && (
        <Text style={inputStyles.requiredError}> *</Text>
      )}
    </Text>
    <TextInput
      style={[
        inputStyles.input,
        showError ? inputStyles.inputError : null,
      ]}
      value={value}
      onChangeText={onChangeText}
      {...props}
      placeholderTextColor="#888"
    />
    {showError && (
      <Text style={inputStyles.errorText}>{errorMessage}</Text>
    )}
  </View>
);

// SyncModal Component
const SyncModal = ({ visible, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Add this helper function near the top (after imports):
function formatMeterReading(value) {
  if (!value) return '';
  const str = value.toString();
  if (str.length <= 6) return str;
  return `${str.slice(0, 2)}-${str.slice(-4)}`;
}

const Daily_Ins_InfoStep1 = ({ setActiveStep, equipment }) => {
  const navigation = useNavigation();
  const route = useRoute();
  const [country, setCountry] = useState(null);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);
  const [project, setProject] = useState(null);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);
  const [inspector, setInspector] = useState('');
  const [checklistModalVisible, setChecklistModalVisible] = useState(false);
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [checklists, setChecklists] = useState([]);
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [checklistSearchQuery, setChecklistSearchQuery] = useState('');
  const [isChecklistSearchActive, setIsChecklistSearchActive] = useState(false);
  const [selectedTab, setSelectedTab] = useState('ALL');
  const [selectedChecklistId, setSelectedChecklistId] = useState(null);
  const [checklistTitle, setChecklistTitle] = useState('');
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [meterReading, setMeterReading] = useState('');
  const [currentMeterReading, setCurrentMeterReading] = useState('');
  const [requestNumber, setRequestNumber] = useState('');
  const [equipmentSearchQuery, setEquipmentSearchQuery] = useState('');
  const [isEquipmentSearchActive, setIsEquipmentSearchActive] = useState(false);
  const [showErrors, setShowErrors] = useState(false);
  const [countryError, setCountryError] = useState(false);
  const [projectError, setProjectError] = useState(false);
  const [inspectorError, setInspectorError] = useState(false);
  const [equipmentError, setEquipmentError] = useState(false);
  const [checklistError, setChecklistError] = useState(false);
  const [coordinates, setCoordinates] = useState('');
  const [gpsAddress, setGpsAddress] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [isEquipmentLoading, setIsEquipmentLoading] = useState(false);
  const equipmentSearchInputRef = useRef(null);
  const isMounted = useRef(true);
  const [weekStartDate, setWeekStartDate] = useState(() => {
    const today = new Date();
    const currentDay = today.getDay();
    const startOfWeek = new Date(today);
    startOfWeek.setDate(today.getDate() - currentDay + 1);
    startOfWeek.setHours(0, 0, 0, 0);
    return startOfWeek;
  });
  const weekDates = getCurrentWeekDates(weekStartDate);
  const [selectedDayIndex, setSelectedDayIndex] = useState(() => {
    const today = new Date();
    return weekDates.findIndex(date => date.toDateString() === today.toDateString());
  });
  const [inspectionDate, setInspectionDate] = useState(new Date());
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [hasExistingReport, setHasExistingReport] = useState(false);

  const today = new Date();
  const currentDayIndex = today.getDay() === 0 ? 6 : today.getDay() - 1;
  const [selectedDay] = useState(currentDayIndex);

  const getCountryCode = (country) => {
    switch (country) {
      case 'Zambia': return 'ZAM';
      case 'DRC': return 'DRC';
      default: return 'XXX';
    }
  };

  const getProjectCode = (project) => {
    switch (project) {
      case 'Kobold': return 'KOB';
      case 'Kalumbila': return 'KAL';
      case 'Kansanshi': return 'KAN';
      case 'Kimiteto': return 'KIM';
      case 'Mimosa': return 'MIM';
      case 'Musompo': return 'MUS';
      case 'IME': return 'IME';
      default: return 'XXX';
    }
  };

  const getEquipmentCode = (equipment) => {
    if (!equipment) return 'XX';
    switch (equipment.equipmentCategory) {
      case 'Light Vehicle': return 'LV';
      case 'Heavy Vehicle': return 'HV';
      case 'Machinery': return 'MA';
      case 'Workshop Machinery': return 'WM';
      case 'Container': return 'CN';
      case 'Rig': return 'RG';
      default: return 'EQ';
    }
  };

  // Helper function to get user ID from local storage
  const getUserIdFromStorage = async () => {
    try {
      const storedUser = await AsyncStorage.getItem('userData');
      if (storedUser) {
        const userData = JSON.parse(storedUser);
        const userId = userData.uid || userData.id;
        console.log('👤 User ID found in local storage:', userId);
        return userId;
      }
      
      // Check for other possible user data keys
      const altUserData = await AsyncStorage.getItem('user');
      if (altUserData) {
        const userData = JSON.parse(altUserData);
        const userId = userData.uid || userData.id;
        console.log('👤 User ID found in alternative storage:', userId);
        return userId;
      }
      
      console.log('👤 No user data found in local storage');
      return null;
    } catch (error) {
      console.error('Error getting user ID from storage:', error);
      return null;
    }
  };

  const generateRequestNumber = async (selectedEquipment, country, project) => {
    try {
      if (!country || !project) {
        return 'XXX-XXX-DR-001';
      }
      const countryCode = getCountryCode(country);
      const projectCode = getProjectCode(project);
      let serialNumber = 1;

      // Generate a unique request number based on timestamp
      // Since we don't have a centralized counter in MongoDB, use timestamp-based approach
      const timestamp = Date.now();
      const timeBasedSerial = (timestamp % 999) + 1; // Use last 3 digits of timestamp
      serialNumber = timeBasedSerial;
      
      const formattedNumber = `${countryCode}-${projectCode}-DR-${String(serialNumber).padStart(3, '0')}`;
      console.log('Generated request number:', formattedNumber);
      return formattedNumber;
    } catch (error) {
      console.error('Error generating request number:', error);
      return 'XXX-XXX-DR-001';
    }
  };

  const saveInspectionData = async (data) => {
    try {
      await AsyncStorage.setItem('inspectionStep1', JSON.stringify(data));
      if (data.selectedEquipment) {
        await AsyncStorage.setItem('globalSelectedEquipment', JSON.stringify(data.selectedEquipment));
      }
      if (data.selectedChecklistId && data.selectedChecklist) {
        await AsyncStorage.setItem('selectedChecklistData', JSON.stringify(data.selectedChecklist));
      }
      // Remove Firebase database calls - only save to AsyncStorage for now
      // Data will be submitted to database from step 3
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data locally.',
        position: 'top',
      });
    }
  };

  // Removed syncDataToFirebase function and related useEffect
  // Data will be synced to database from step 3 only

  // Load previous reports for a specific date, equipment, and checklist
  const loadPreviousReportsForDate = async (date, equipmentId, checklistId) => {
    try {
      // Check for user authentication via local storage instead of Firebase
      const userId = await getUserIdFromStorage();
      
      if (!userId) {
        // Skip loading previous reports if no user data available
        console.log('👤 No user data found, skipping previous reports loading');
        return;
      }

      const dateStr = date.toISOString().split('T')[0];
      let reports = [];

      // Check online database
      const isOnline = await NetInfo.fetch().then((state) => state.isConnected);
      if (isOnline) {
        try {
          // Fetch inspections from MongoDB
          const inspectionsResponse = await fetch(`http://168.231.113.219:5008/inspections/${userId}`);
          if (inspectionsResponse.ok) {
            const inspectionsData = await inspectionsResponse.json();
            if (inspectionsData.success && inspectionsData.data) {
              reports = inspectionsData.data.filter((inspection) => {
                const reportDate = new Date(inspection.step1?.inspectionDate || inspection.createdAt);
                const reportDateStr = reportDate.toISOString().split('T')[0];
                return inspection.step1?.equipmentId === equipmentId && 
                       inspection.step1?.selectedChecklistId === checklistId &&
                       reportDateStr === dateStr;
              });
            }
          }
        } catch (error) {
          console.warn('Error fetching inspections from MongoDB:', error);
        }
      }

      // Check local storage
      const storedReports = await AsyncStorage.getItem('dailyInspections');
      if (storedReports) {
        const localReports = JSON.parse(storedReports).filter((report) => {
          const reportDate = new Date(report.step1?.inspectionDate || report.step3?.createdAt);
          const reportDateStr = reportDate.toISOString().split('T')[0];
          return report.equipmentId === equipmentId && 
                 report.step1?.selectedChecklistId === checklistId &&
                 reportDateStr === dateStr;
        });
        reports = [...reports, ...localReports];
      }

      // Store previous reports for use in PDF generation
      if (reports.length > 0) {
        await AsyncStorage.setItem('previousReportsForDate', JSON.stringify(reports));
        console.log(`Found ${reports.length} previous reports for date ${dateStr}`);
        setHasExistingReport(true);
        
        // Show warning if report exists for this date
        Toast.show({
          type: 'info',
          text1: 'Report Exists',
          text2: `A report already exists for ${date.toLocaleDateString()}. You can continue to update it.`,
          position: 'top',
          visibilityTime: 3000,
        });
      } else {
        await AsyncStorage.removeItem('previousReportsForDate');
        setHasExistingReport(false);
      }
    } catch (error) {
      console.error('Error loading previous reports:', error);
    }
  };

  const loadEquipmentData = async () => {
    setIsEquipmentLoading(true);
    try {
      const isOnline = await NetInfo.fetch().then((state) => state.isConnected);

      let data = [];
      if (isOnline) {
        try {
          // Fetch equipment from MongoDB
          console.log('🚛 Fetching equipment from MongoDB...');
          console.log('📡 Equipment endpoint: http://168.231.113.219:5001/EquipmentsDataJSON/GlobalEquipmentJsonData');
          
          const response = await fetch('http://168.231.113.219:5001/EquipmentsDataJSON/GlobalEquipmentJsonData');
          
          console.log('📡 Equipment response status:', response.status);
          
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }

          const result = await response.json();
          console.log('🚛 Equipment response structure:', {
            success: result.success,
            hasData: !!result.data,
            dataLength: result.data ? result.data.length : 0,
            count: result.count,
            firstItem: result.data && result.data[0] ? {
              id: result.data[0].id,
              equipmentName: result.data[0].equipmentName,
              category: result.data[0].category,
              status: result.data[0].status
            } : null
          });
          
          if (result.success && result.data) {
            data = Array.isArray(result.data) ? result.data : [];
            data = data.filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index);
            data.sort((a, b) => {
              const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
              const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
              return dateB - dateA;
            });
            await AsyncStorage.setItem('equipment', JSON.stringify(data));
            console.log(`✅ Successfully loaded ${data.length} equipment items from MongoDB`);
          } else {
            throw new Error(result.error || 'Failed to fetch equipment data');
          }
        } catch (mongodbError) {
          console.error('MongoDB error:', mongodbError);
          // Fallback to local storage if MongoDB fails
          const storedEquipment = await AsyncStorage.getItem('equipment');
          data = storedEquipment ? JSON.parse(storedEquipment) : [];
          Toast.show({ 
            type: 'info', 
            text1: 'Offline Mode', 
            text2: 'Loaded equipment from local storage.', 
            position: 'top' 
          });
        }
      } else {
        const storedEquipment = await AsyncStorage.getItem('equipment');
        data = storedEquipment ? JSON.parse(storedEquipment) : [];
        data.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        Toast.show({ 
          type: 'info', 
          text1: 'Offline', 
          text2: 'Loaded equipment from local storage.', 
          position: 'top' 
        });
      }
      
      // Always set the equipment data, even if empty
      setFirebaseEquipment(data);
      
    } catch (error) {
      console.error('Error loading equipment:', error);
      // Ensure we always have some data to show
      try {
        const storedEquipment = await AsyncStorage.getItem('equipment');
        const parsedEquipment = storedEquipment ? JSON.parse(storedEquipment) : [];
        setFirebaseEquipment(parsedEquipment);
      } catch (storageError) {
        console.error('Error loading from storage:', storageError);
        setFirebaseEquipment([]);
      }
      
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load equipment. Please try again.',
        position: 'top',
      });
    } finally {
      setIsEquipmentLoading(false);
      setEquipmentSearchQuery('');
      setIsEquipmentSearchActive(false);
    }
  };

  // Load initial data and set inspectionDate to selected day
  useEffect(() => {
    const loadData = async () => {
      try {
        await loadEquipmentData();
        // Set inspectionDate to the selected day from week bar (not always today)
        const selectedDate = weekDates[selectedDayIndex];
        setInspectionDate(selectedDate);
        const inspectionData = {
          id: Date.now().toString(),
          requestNumber,
          country,
          project,
          inspector,
          selectedEquipment,
          meterReading,
          currentMeterReading,
          selectedChecklistId,
          checklistTitle,
          coordinates,
          gpsAddress,
          inspectionDate: selectedDate.toISOString(),
          createdAt: new Date().toISOString(),
        };
        await AsyncStorage.setItem('inspectionStep1', JSON.stringify(inspectionData));
        setRequestNumber('W01001');
      } catch (error) {
        console.error('Error loading data:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load initial data. Please try again.',
          position: 'top',
        });
      }
    };

    loadData();

    return () => {
      isMounted.current = false;
    };
  }, [selectedDayIndex]); // Add selectedDayIndex as dependency

  // When user selects a day in the week bar, update inspectionDate and save to AsyncStorage
  const handleDaySelect = async (idx) => {
    setSelectedDayIndex(idx);
    const selectedDate = weekDates[idx];
    setInspectionDate(selectedDate);
    
    // Save to AsyncStorage immediately
    const savedInspection = await AsyncStorage.getItem('inspectionStep1');
    let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};
    inspectionData.inspectionDate = selectedDate.toISOString();
    await AsyncStorage.setItem('inspectionStep1', JSON.stringify(inspectionData));
    
    // Load previous reports for this date and equipment if available
    if (selectedEquipment && selectedChecklistId) {
      await loadPreviousReportsForDate(selectedDate, selectedEquipment.id, selectedChecklistId);
    }
    
    // Show confirmation of selected date
    Toast.show({
      type: 'info',
      text1: 'Date Selected',
      text2: `Inspection will be for ${selectedDate.toLocaleDateString()}`,
      position: 'top',
      visibilityTime: 2000,
    });
  };

  useEffect(() => {
    const updateRequestNumber = async () => {
      if (country && project) {
        const newRequestNumber = await generateRequestNumber(selectedEquipment, country, project);
        setRequestNumber(newRequestNumber);
      }
    };
    updateRequestNumber();
  }, [selectedEquipment, country, project]);

  useEffect(() => {
    const inspectionData = {
      id: Date.now().toString(),
      requestNumber,
      country,
      project,
      inspector,
      selectedEquipment,
      meterReading,
      currentMeterReading,
      selectedChecklistId,
      checklistTitle,
      coordinates,
      gpsAddress,
      inspectionDate: inspectionDate.toISOString(),
      createdAt: new Date().toISOString(),
    };
    saveInspectionData(inspectionData);
  }, [
    country,
    project,
    inspector,
    selectedEquipment,
    meterReading,
    currentMeterReading,
    selectedChecklistId,
    checklistTitle,
    coordinates,
    gpsAddress,
    requestNumber,
    inspectionDate,
  ]);

  // Debug useEffect to monitor checklists state changes
  useEffect(() => {
    console.log('🔄 Checklists state changed:', {
      length: checklists.length,
      modalVisible: checklistModalVisible,
      firstItem: checklists[0] ? checklists[0].title : 'No items'
    });
  }, [checklists, checklistModalVisible]);

  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }
    if (!country || (project && !projectItems.some((item) => item.value === project))) {
      setProject(null);
    }
  }, [country]);

  const loadChecklists = async () => {
    try {
      setSyncModalVisible(true);

      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: 'Cannot fetch checklists. Loading from local storage.',
          position: 'top',
        });
        const storedChecklists = await AsyncStorage.getItem('checklists');
        const parsedChecklists = storedChecklists ? JSON.parse(storedChecklists) : [];
        setChecklists(parsedChecklists);
        setSyncModalVisible(false);
        return;
      }

      // Remove authentication requirement for checklist loading
      console.log('📋 Loading checklists without authentication requirement');

      // Skip fetching used checklist IDs since we removed authentication requirement
      let usedChecklistIds = new Set();
      console.log('📋 Skipping used checklist fetch - no authentication required');

      // Fetch checklists from MongoDB
      console.log('🔍 Fetching checklists from MongoDB...');
      console.log('📡 Making request to: http://168.231.113.219:5008/checklists');
      
      const checklistsResponse = await fetch('http://168.231.113.219:5008/checklists', {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json',
        },
        timeout: 10000, // 10 second timeout
      });
      
      console.log('📡 Checklists response status:', checklistsResponse.status);
      console.log('📡 Checklists response headers:', checklistsResponse.headers);
      
      if (!checklistsResponse.ok) {
        const errorText = await checklistsResponse.text();
        console.error('❌ HTTP error response:', errorText);
        Toast.show({
          type: 'error',
          text1: 'MongoDB error',
          text2: `Error: HTTP error! status: ${checklistsResponse.status}`,
          position: 'top',
        });
        throw new Error(`HTTP error! status: ${checklistsResponse.status} - ${errorText}`);
      }

      const checklistsData = await checklistsResponse.json();
      console.log('📋 Checklists data received:', {
        success: checklistsData.success,
        dataLength: checklistsData.data ? checklistsData.data.length : 0,
        data: checklistsData.data ? checklistsData.data.slice(0, 2) : null // Log first 2 items
      });
      
      console.log('📋 Full response structure:', {
        success: checklistsData.success,
        count: checklistsData.count,
        hasData: !!checklistsData.data,
        dataType: Array.isArray(checklistsData.data) ? 'Array' : typeof checklistsData.data
      });

      if (!checklistsData.success || !checklistsData.data || checklistsData.data.length === 0) {
        console.warn('⚠️ No checklists found in database');
        Toast.show({
          type: 'info',
          text1: 'No Checklists',
          text2: 'No checklists found in the database. Please upload checklists first.',
          position: 'top',
        });
        setChecklists([]);
        setSyncModalVisible(false);
        return;
      }

      // Format checklists
      console.log('🔧 Starting to format checklists...');
      console.log('🔧 Raw checklists data sample:', checklistsData.data[0]);
      
      const formattedChecklists = checklistsData.data.map((checklist, index) => {
        console.log(`🔧 Formatting checklist ${index}:`, {
          id: checklist.id,
          hasFormTitle: !!checklist.formTitle,
          hasInspectionItems: !!checklist.inspectionItems,
          inspectionItemsLength: checklist.inspectionItems ? checklist.inspectionItems.length : 0
        });
        
        if (!checklist || !checklist.formTitle || !checklist.inspectionItems) {
          console.warn(`⚠️ Invalid checklist at index ${index}:`, {
            hasChecklist: !!checklist,
            hasFormTitle: !!checklist?.formTitle,
            hasInspectionItems: !!checklist?.inspectionItems
          });
          return null;
        }
        const isUsed = usedChecklistIds.has(checklist.id);
        
        return {
          id: checklist.id,
          title: checklist.formTitle,
          description: checklist.document?.name || checklist.formTitle,
          documentNumber: checklist.document?.number || 'N/A',
          items: checklist.inspectionItems.map((item, index) => ({
            text: item || `Item ${index + 1}`,
            subItems: [{ 
              itemName: item || `Item ${index + 1}`, 
              inputType: 'Ok/Reject/N/A', 
              description: '', 
              instruction: '', 
              sortOrder: index + 1, 
              isInspectionRequired: false 
            }],
          })),
          inspectionItems: checklist.inspectionItems,
          status: 'Available',
          isUsed,
          imageUrl: null, // MongoDB doesn't store images in this structure
        };
      }).filter(checklist => checklist !== null);
      
      console.log('🔧 Formatted checklists count:', formattedChecklists.length);
      console.log('🔧 Sample formatted checklist:', formattedChecklists[0]);

      // Save to local storage
      await AsyncStorage.setItem('checklists', JSON.stringify(formattedChecklists));
      console.log('💾 Saved checklists to AsyncStorage');
      
      setChecklists(formattedChecklists);
      console.log('📱 Set checklists state with:', formattedChecklists.length, 'items');
      console.log('📱 State update completed');
      
      console.log(`✅ Successfully loaded ${formattedChecklists.length} checklists from MongoDB`);
    } catch (error) {
      console.error('Error loading checklists:', error);
      let errorMessage = 'Failed to load checklists.';
      
      if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
        errorMessage = 'Network error. Loading from local storage.';
        try {
          const storedChecklists = await AsyncStorage.getItem('checklists');
          const parsedChecklists = storedChecklists ? JSON.parse(storedChecklists) : [];
          setChecklists(parsedChecklists);
        } catch (storageError) {
          console.error('Error loading from storage:', storageError);
          setChecklists([]);
        }
      } else {
        // Try to load from local storage as fallback
        try {
          const storedChecklists = await AsyncStorage.getItem('checklists');
          const parsedChecklists = storedChecklists ? JSON.parse(storedChecklists) : [];
          setChecklists(parsedChecklists);
          errorMessage = 'Using cached checklists. Please check your connection.';
        } catch (storageError) {
          console.error('Error loading from storage:', storageError);
          setChecklists([]);
        }
      }
      
      Toast.show({
        type: error.message.includes('Network request failed') ? 'info' : 'warning',
        text1: error.message.includes('Network request failed') ? 'Offline' : 'Connection Issue',
        text2: errorMessage,
        position: 'top',
        visibilityTime: 4000,
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  const getFilteredEquipment = () => {
    let allEquipment = firebaseEquipment;

    if (selectedTab !== 'ALL') {
      allEquipment = allEquipment.filter((item) => {
        switch (selectedTab) {
          case 'VEHICLE': return ['Light Vehicle', 'Heavy Vehicle'].includes(item.equipmentCategory);
          case 'MACHINERY': return ['Machinery', 'Workshop Machinery'].includes(item.equipmentCategory);
          case 'CONTAINER': return item.equipmentCategory === 'Container';
          case 'RIG': return item.equipmentCategory === 'Rig';
          default: return true;
        }
      });
    }

    if (!equipmentSearchQuery.trim()) return allEquipment;
    return allEquipment.filter((item) =>
      [item.equipmentName, item.mainCategory, item.model].some((field) =>
        field?.toLowerCase().includes(equipmentSearchQuery.toLowerCase())
      )
    );
  };

  const handleSelectPress = async () => {
    // Open modal immediately for better user experience
    setEquipmentModalVisible(true);
    
    try {
      // Load equipment data in the background
      await loadEquipmentData();
    } catch (error) {
      console.error('Error loading equipment data:', error);
      Toast.show({ 
        type: 'error', 
        text1: 'Error', 
        text2: 'Failed to load equipment data. Please try again.', 
        position: 'top' 
      });
      // Modal is already open, so user can still see cached data or try again
    }
  };

  const handleChecklistSelectPress = async () => {
    console.log('🔍 Opening checklist modal...');
    console.log('📋 Current checklists state before loading:', checklists.length);
    
    // Open modal first for better UX
    setChecklistModalVisible(true);
    
    // Load checklists in background
    try {
      await loadChecklists();
      console.log('📋 Checklists loaded successfully');
    } catch (error) {
      console.error('❌ Error loading checklists:', error);
    }
  };

  const handleBackPress = () => {
    Alert.alert(
      'Exit Step',
      'Do you want to exit this step? Your data will be saved.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'OK',
          onPress: () => navigation.goBack(),
          style: 'default',
        },
      ],
      { cancelable: false }
    );
  };

  const requestLocationPermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION,
          {
            title: 'Location Permission',
            message: 'This app needs access to your location to fetch your current address.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        if (granted === PermissionsAndroid.RESULTS.GRANTED) {
          return true;
        } else if (granted === PermissionsAndroid.RESULTS.DENIED) {
          Toast.show({
            type: 'error',
            text1: 'Permission Denied',
            text2: 'Location permission is required to fetch your current location. Please enable it in settings.',
            position: 'top',
          });
          return false;
        } else if (granted === PermissionsAndroid.RESULTS.NEVER_ASK_AGAIN) {
          Toast.show({
            type: 'error',
            text1: 'Permission Blocked',
            text2: 'Location permission is blocked. Please enable it in your device settings.',
            position: 'top',
          });
          return false;
        }
      } else {
        return new Promise((resolve, reject) => {
          Geolocation.requestAuthorization(
            () => resolve(true),
            (error) => {
              console.error('iOS permission error:', error);
              Toast.show({
                type: 'error',
                text1: 'Permission Error',
                text2: 'Unable to access location permission. Please enable it in settings.',
                position: 'top',
              });
              resolve(false);
            }
          );
        });
      }
    } catch (error) {
      console.error('Error requesting location permission:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to request location permission. Please try again.',
        position: 'top',
      });
      return false;
    }
  };

  const fetchAddressFromCoordinates = async (latitude, longitude) => {
    const apiKey = 'AIzaSyDDEh8W8buXtPi7uhhwJx4e5iK7Gb-vbe8';
    const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&key=${apiKey}`;

    try {
      const response = await fetch(url);
      const data = await response.json();

      if (data.status === 'OK' && data.results.length > 0) {
        return data.results[0].formatted_address;
      } else if (data.status === 'REQUEST_DENIED') {
        throw new Error('Request denied: Invalid API key or billing issue.');
      } else if (data.status === 'ZERO_RESULTS') {
        throw new Error('No address found for the given coordinates.');
      } else {
        throw new Error(`Geocoding API error: ${data.status}`);
      }
    } catch (error) {
      console.error('Error fetching address:', error.message);
      Toast.show({
        type: 'error',
        text1: 'Address Fetch Failed',
        text2: error.message || 'Failed to fetch address. Using coordinates only.',
        position: 'top',
      });
      return `Latitude: ${latitude}, Longitude: ${longitude} (Address unavailable)`;
    }
  };

  const handleGetLocation = async () => {
    setSyncModalVisible(true);
    const timeoutId = setTimeout(() => {
      if (isMounted.current && syncModalVisible) {
        setSyncModalVisible(false);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Location fetching timed out. Please try again.',
          position: 'top',
        });
      }
    }, 15000);

    try {
      const hasPermission = await requestLocationPermission();
      if (!hasPermission) {
        throw new Error('Location permission denied');
      }

      await new Promise((resolve, reject) => {
        Geolocation.getCurrentPosition(
          async (position) => {
            const { latitude, longitude } = position.coords;
            const coords = `${latitude}, ${longitude}`;
            setCoordinates(coords);

            const address = await fetchAddressFromCoordinates(latitude, longitude);
            setGpsAddress(address);

            const savedInspection = await AsyncStorage.getItem('inspectionStep1');
            let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};
            inspectionData.coordinates = coords;
            inspectionData.gpsAddress = address;
            await saveInspectionData(inspectionData);

            if (isMounted.current) {
              Toast.show({
                type: 'success',
                text1: 'Location Fetched',
                text2: `Coordinates: ${coords}\nAddress: ${address}`,
                position: 'top',
                visibilityTime: 4000,
              });
            }
            resolve();
          },
          (error) => {
            console.error('Error getting location:', error);
            let errorMessage = '';
            if (error.code === error.PERMISSION_DENIED) {
              errorMessage += 'Please ensure location permissions are granted.';
            } else if (error.code === error.POSITION_UNAVAILABLE) {
              errorMessage += 'Please turn on your GPS.';
            } else if (error.code === error.TIMEOUT) {
              errorMessage += 'The request timed out. Please try again.';
            } else {
              errorMessage += 'An unexpected error occurred.';
            }
            Toast.show({
              type: 'error',
              text1: 'Location Error',
              text2: errorMessage,
              position: 'top',
            });
            reject(error);
          },
          { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 }
        );
      });
    } catch (error) {
      console.error('Error in handleGetLocation:', error);
    } finally {
      clearTimeout(timeoutId);
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const validateChecklistItems = (checklistId) => {
    const checklist = checklists.find((c) => c.id === checklistId);
    if (!checklist || !checklist.items || !Array.isArray(checklist.items)) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'The selected checklist is invalid or empty.',
        position: 'top',
      });
      return false;
    }
    return checklist.items.every(
      (item) =>
        item &&
        typeof item.text === 'string' &&
        Array.isArray(item.subItems) &&
        item.subItems.every(
          (subItem) =>
            subItem &&
            typeof subItem.itemName === 'string' &&
            typeof subItem.inputType === 'string'
        )
    );
  };

  const handleNextPress = async () => {
    setShowErrors(true);
    setCountryError(!country);
    setProjectError(!project);
    setInspectorError(!inspector.trim());
    setEquipmentError(!selectedEquipment);
    setChecklistError(!selectedChecklistId);

    if (!country || !project || !inspector.trim() || !selectedEquipment || !selectedChecklistId) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please fill all required fields.',
        position: 'top',
      });
      return;
    }

    // Check if a report already exists for this date
    const previousReports = await AsyncStorage.getItem('previousReportsForDate');
    if (previousReports) {
      const reports = JSON.parse(previousReports);
      if (reports.length > 0) {
        Alert.alert(
          'Report Exists',
          `A report already exists for ${inspectionDate.toLocaleDateString()}. Do you want to continue and update it?`,
          [
            { text: 'Cancel', style: 'cancel' },
            { text: 'Continue', onPress: () => proceedToNextStep() },
          ]
        );
        return;
      }
    }

    await proceedToNextStep();
  };

  const proceedToNextStep = async () => {

    if (selectedEquipment && meterReading && parseFloat(meterReading) <= parseFloat(selectedEquipment.meterReading)) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: `New meter reading (${meterReading}) must be greater than current meter reading (${selectedEquipment.meterReading || '0'}).`,
        position: 'top',
      });
      return;
    }

    if (!validateChecklistItems(selectedChecklistId)) {
      setChecklistError(true);
      return;
    }

    try {
      // Check for user authentication via local storage instead of Firebase
      let userId = await getUserIdFromStorage();
      
      if (userId) {
        console.log('👤 User authenticated via local storage:', userId);
      } else {
        // Fallback: generate a temporary user ID for offline use
        userId = `temp_user_${Date.now()}`;
        console.log('👤 Using temporary user ID for offline mode:', userId);
      }
      const selectedChecklist = checklists.find((c) => c.id === selectedChecklistId);
      const inspectionData = {
        id: Date.now().toString(),
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        equipmentId: selectedEquipment.id,
        meterReading,
        currentMeterReading: selectedEquipment.meterReading || '0',
        selectedChecklistId,
        selectedChecklist: {
          id: selectedChecklist.id,
          formTitle: selectedChecklist.title,
          inspectionItems: selectedChecklist.inspectionItems,
        },
        checklistTitle,
        coordinates,
        gpsAddress,
        inspectionDate: inspectionDate.toISOString(),
        createdAt: new Date().toISOString(),
      };

      if (selectedEquipment && meterReading && parseFloat(meterReading) !== parseFloat(selectedEquipment.meterReading)) {
        console.log('🔧 Updating equipment meter reading:', {
          equipmentId: selectedEquipment.id,
          oldReading: selectedEquipment.meterReading,
          newReading: meterReading
        });
        
        const updatedEquipment = {
          ...selectedEquipment,
          meterReading: parseFloat(meterReading),
          updatedAt: new Date().toISOString(),
        };

        if (isOnline) {
          // Try to update equipment in MongoDB (non-blocking)
          try {
            const updateResponse = await fetch(`http://168.231.113.219:5001/equipment/${updatedEquipment.id}`, {
              method: 'PUT',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify(updatedEquipment)
            });

            if (updateResponse.ok) {
              const updateResult = await updateResponse.json();
              if (updateResult.success) {
                console.log('✅ Equipment updated in MongoDB successfully');
              } else {
                console.warn('⚠️ Equipment update failed:', updateResult.error);
              }
            } else {
              console.warn('⚠️ Equipment update failed with status:', updateResponse.status);
            }
          } catch (equipmentUpdateError) {
            console.warn('⚠️ Equipment update error (non-blocking):', equipmentUpdateError.message);
            // Don't throw error - just log it and continue
          }
        } else {
          const storedEquipment = await AsyncStorage.getItem('equipment');
          let equipmentList = storedEquipment ? JSON.parse(storedEquipment) : [];
          const index = equipmentList.findIndex((item) => item.id === updatedEquipment.id);
          if (index !== -1) {
            equipmentList[index] = { ...updatedEquipment, pendingSync: true };
          } else {
            equipmentList.push({ ...updatedEquipment, pendingSync: true });
          }
          await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
          const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
          syncQueue.push({ type: 'equipment', data: updatedEquipment });
          await AsyncStorage.setItem('syncQueue', JSON.stringify(syncQueue));
        }

        setFirebaseEquipment((prev) =>
          prev.map((item) => (item.id === updatedEquipment.id ? updatedEquipment : item))
        );
        setSelectedEquipment(updatedEquipment);
        inspectionData.selectedEquipment = updatedEquipment;
      }

      await AsyncStorage.setItem('selectedChecklistIds', JSON.stringify([selectedChecklistId]));
      await saveInspectionData(inspectionData);

      setShowErrors(false);
      setCountryError(false);
      setProjectError(false);
      setInspectorError(false);
      setEquipmentError(false);
      setChecklistError(false);
      setActiveStep(2);
    } catch (error) {
      console.error('Error saving inspection data:', error);
      let errorMessage = 'Failed to save inspection data.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please check your Firebase configuration.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Network error. Data saved locally and will sync when online.';
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'top',
      });
    }
  };

  const handleChecklistSearch = (query) => {
    setChecklistSearchQuery(query);
  };

  const handleEquipmentSearch = (query) => {
    setEquipmentSearchQuery(query);
  };

  const filteredChecklists = checklists.filter((checklist) => {
    console.log('🔍 Filtering checklist:', {
      id: checklist?.id,
      title: checklist?.title,
      hasTitle: !!checklist?.title,
      searchQuery: checklistSearchQuery,
      selectedTab: selectedTab
    });
    
    if (!checklist || !checklist.title) {
      console.log('❌ Filtering out checklist - invalid:', checklist);
      return false;
    }
    const matchesSearch = checklist.title.toLowerCase().includes(checklistSearchQuery.toLowerCase()) ||
                         checklist.documentNumber?.toLowerCase().includes(checklistSearchQuery.toLowerCase());
    if (selectedTab === 'ALL') {
      console.log('✅ ALL tab - matches search:', matchesSearch);
      return matchesSearch;
    } else if (selectedTab === 'Assign') {
      console.log('✅ Assign tab - matches search and is used:', matchesSearch && checklist.isUsed);
      return matchesSearch && checklist.isUsed;
    }
    return matchesSearch;
  });
  
  console.log('🔍 Filtered checklists result:', {
    totalChecklists: checklists.length,
    filteredCount: filteredChecklists.length,
    searchQuery: checklistSearchQuery,
    selectedTab: selectedTab
  });

  const toggleChecklistSelection = async (checklistId, title) => {
    setSelectedChecklistId((prev) => {
      const newId = prev === checklistId ? null : checklistId;
      if (newId) {
        setChecklistTitle(title || 'Untitled Checklist');
      } else {
        setChecklistTitle('');
      }
      AsyncStorage.setItem('selectedChecklistIds', JSON.stringify(newId ? [newId] : []))
        .catch((error) => console.error('Error saving selected checklist:', error));
      return newId;
    });
    setChecklistModalVisible(false);
    
    // Load previous reports for this checklist and date if equipment is already selected
    if (selectedEquipment && inspectionDate) {
      await loadPreviousReportsForDate(inspectionDate, selectedEquipment.id, checklistId);
    }
  };

  const selectEquipment = async (equipment) => {
    setSelectedEquipment(equipment);
    setMeterReading(equipment.meterReading?.toString() || '');
    setEquipmentModalVisible(false);
    // Save immediately to AsyncStorage for summary step
    const savedInspection = await AsyncStorage.getItem('inspectionStep1');
    let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};
    inspectionData.selectedEquipment = equipment;
    inspectionData.equipmentId = equipment.id;
    await AsyncStorage.setItem('inspectionStep1', JSON.stringify(inspectionData));
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const onDateChange = (event, selectedDate) => {
    setShowDatePicker(Platform.OS === 'ios');
    if (selectedDate) {
      setInspectionDate(selectedDate);
      const inspectionData = {
        id: Date.now().toString(),
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        meterReading,
        currentMeterReading,
        selectedChecklistId,
        checklistTitle,
        coordinates,
        gpsAddress,
        inspectionDate: selectedDate.toISOString(),
        createdAt: new Date().toISOString(),
      };
      saveInspectionData(inspectionData);
    }
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity
      style={styles.checklistItem}
      onPress={() => toggleChecklistSelection(item.id, item.title)}
    >
      <View style={styles.checklistItemContent}>
        {item.imageUrl ? (
          <Image
            source={{ uri: item.imageUrl }}
            style={styles.checklistImage}
            resizeMode="cover"
          />
        ) : (
          <View style={styles.checklistImagePlaceholder}>
            <MaterialIcons name="image" size={30} color="#ccc" />
          </View>
        )}
        <View style={styles.checklistTextContainer}>
          <Text style={styles.checklistItemText}>{item.title || 'Untitled Checklist'}</Text>
          <Text style={styles.checklistDescription}>
            {item.description || 'No description available'}
          </Text>
          <Text style={styles.checklistDocumentInfo}>
            Document: {item.documentNumber || 'N/A'} | Items: {item.items?.length || 0} sections
          </Text>
        </View>
        <MaterialIcons
          name={selectedChecklistId === item.id ? 'check-box' : 'check-box-outline-blank'}
          size={24}
          color={selectedChecklistId === item.id ? '#0078D4' : '#666'}
        />
      </View>
    </TouchableOpacity>
  );

  const renderEquipmentItem = ({ item }) => (
    <View style={styles.equipmentRow}>
      <TouchableOpacity style={styles.equipmentInfo} onPress={() => selectEquipment(item)}>
        <View style={styles.equipmentDetailItemheader}>
          <Text style={styles.detailLabelheader}>{item.equipmentName || 'Tyre - Tyre'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Category</Text>
          <Text style={styles.detailValue}>{item.mainCategory || 'Sample Category'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Model</Text>
          <Text style={styles.detailValue}>{item.model || '326F'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Meter Reading</Text>
          <Text style={styles.detailValue}>{item.meterReading || '0'} {item.meterUnit || 'Miles'}</Text>
        </View>
        <View style={styles.equipmentDetailItemNoBorder}>
          <Text style={[styles.detailLabelstatus, { color: getStatusColor(item.status || 'Available') }]}>{item.status || 'Available'}</Text>
        </View>
      </TouchableOpacity>
      <TouchableOpacity style={styles.selectEquipmentButton} onPress={() => selectEquipment(item)}>
        <Text style={styles.selectEquipmentText}>{selectedEquipment && selectedEquipment.id === item.id ? 'Selected' : 'Select'}</Text>
      </TouchableOpacity>
    </View>
  );

  return (
    <View style={styles.container}>
      <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingHorizontal: 10, marginBottom: 5 }}>
        <TouchableOpacity onPress={() => {
          const prevWeek = new Date(weekStartDate);
          prevWeek.setDate(weekStartDate.getDate() - 7);
          setWeekStartDate(prevWeek);
          setSelectedDayIndex(0);
        }}>
          <MaterialIcons name="chevron-left" size={24} color="#015185" />
        </TouchableOpacity>
        <View style={{ alignItems: 'center' }}>
        <Text style={{ fontWeight: 'bold', color: '#015185' }}>
          {weekDates[0].toLocaleDateString()} - {weekDates[6].toLocaleDateString()}
        </Text>
          {hasExistingReport && (
            <Text style={{ fontSize: 12, color: '#e74c3c', marginTop: 2 }}>
              ⚠️ Report exists for selected date
            </Text>
          )}
        </View>
        <TouchableOpacity onPress={() => {
          const nextWeek = new Date(weekStartDate);
          nextWeek.setDate(weekStartDate.getDate() + 7);
          setWeekStartDate(nextWeek);
          setSelectedDayIndex(0);
        }}>
          <MaterialIcons name="chevron-right" size={24} color="#015185" />
        </TouchableOpacity>
      </View>
      <View style={styles.daysBar}>
        {weekDates.map((date, idx) => {
          const isToday = (new Date().toDateString() === date.toDateString());
          return (
            <TouchableOpacity
              key={idx}
              style={[
                styles.dayItem,
                isToday && styles.daySelected,
                isToday && styles.currentDaySelected,
                !isToday && { opacity: 0.5 },
              ]}
              onPress={() => {}}
              disabled={!isToday}
            >
              <Text
                style={[
                  styles.dayText,
                  isToday && styles.dayTextSelected,
                  isToday && styles.currentDayText,
                  isToday && { fontWeight: 'bold', color: '#015185' },
                ]}
              >
                {formatDayName(date)}
              </Text>
              <Text style={styles.dateText}>{date.getDate()}</Text>
            </TouchableOpacity>
          );
        })}
      </View>

      <KeyboardAvoidingView
        style={styles.container}
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
      >
        <ScrollView
          style={styles.mainContent}
          contentContainerStyle={styles.scrollContent}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.requestRow}>
            <Text style={styles.requestLabel}>Request #</Text>
            <Text style={styles.requestValue}>{requestNumber}</Text>
          </View>
          <View style={inputStyles.container}>
            <Text style={inputStyles.label}>
              Country
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <Dropdown
              style={[
                inputStyles.input,
                countryError && showErrors ? inputStyles.inputError : null,
              ]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.dropdownText}
              data={countryItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Country"
              value={country}
              onChange={item => setCountry(item.value)}
              containerStyle={styles.dropdownContainer}
              itemTextStyle={styles.dropdownText}
              activeColor="#e0e0e0"
            />
            {countryError && showErrors && (
              <Text style={inputStyles.errorText}>Country is required</Text>
            )}
          </View>
          <View style={inputStyles.container}>
            <Text style={inputStyles.label}>
              Project
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <Dropdown
              style={[
                inputStyles.input,
                !country && styles.disabledDropdown,
                projectError && showErrors ? inputStyles.inputError : null,
              ]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.dropdownText}
              data={projectItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Project"
              value={project}
              onChange={item => setProject(item.value)}
              containerStyle={styles.dropdownContainer}
              itemTextStyle={styles.dropdownText}
              activeColor="#e0e0e0"
              disable={!country}
            />
            {projectError && showErrors && (
              <Text style={inputStyles.errorText}>Project is required</Text>
            )}
          </View>
          <CustomInputField
            label="Inspector"
            value={inspector}
            onChangeText={setInspector}
            placeholder="Enter Inspector"
            isRequired
            showError={inspectorError && showErrors}
            errorMessage="Inspector is required"
          />
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Equipments</Text>
            <TouchableOpacity onPress={handleSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>
                {selectedEquipment ? 'Change equipment' : 'Select'}
              </Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          {equipmentError && showErrors && (
            <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Equipment is required</Text>
          )}
          {selectedEquipment && (
            <View style={styles.equipmentDetailsBox}>
              <View style={styles.vehicleEquipmentRow}>
                <Text style={styles.vehicleEquipmentText}>
                  Vehicle/Model: {selectedEquipment.vehicleNumber || selectedEquipment.model || 'N/A'}, 
                  Category: {selectedEquipment.mainCategory || 'N/A'}, 
                  Equipment Name: {selectedEquipment.equipmentName || 'N/A'}
                </Text>
              </View>
              <View style={styles.meterReadingContainer}>
                <View style={styles.labelRow}>
                  <Text style={styles.meterReadingLabel}>
                    <Text style={styles.required}>*</Text>Meter Reading
                  </Text>
                </View>
                <View style={[styles.inputRow, showErrors && !meterReading && styles.inputError]}>
                  <TextInput
                    style={styles.meterReadingInput}
                    value={meterReading}
                    onChangeText={(text) => {
                      setMeterReading(text);
                    }}
                    onBlur={() => {
                      if (meterReading) {
                        const currentReading = parseFloat(selectedEquipment.meterReading) || 0;
                        const newReading = parseFloat(meterReading);
                        
                        if (!isNaN(newReading) && newReading < currentReading) {
                          Toast.show({
                            type: 'error',
                            text1: 'Invalid Reading',
                            text2: `Meter reading must be greater than or equal to ${currentReading}`,
                            position: 'top',
                          });
                        }
                      }
                    }}
                    placeholder="Enter meter reading"
                    placeholderTextColor="#888"
                    keyboardType="numeric"
                  />
                  <Text style={styles.meterUnit}>KM</Text>
                </View>
                {showErrors && !meterReading && (
                  <Text style={styles.errorText}>Meter reading is required</Text>
                )}
                <View style={styles.currentMeterRow}>
                  <Text style={styles.currentMeterLabel}>Current Meter Reading: </Text>
                  <Text style={styles.currentMeterValue}>
                    {formatMeterReading(selectedEquipment.meterReading || '0')} Mil
                  </Text>
                </View>
              </View>
            </View>
          )}
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Checklist</Text>
            <TouchableOpacity onPress={handleChecklistSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>
                {checklistTitle ? 'Change checklist' : 'Select'}
              </Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          <View style={styles.selectedChecklistsBox}>
            {checklistTitle ? (
              <Text style={styles.selectedChecklistText}>
                {checklistTitle}
              </Text>
            ) : (
              <Text style={styles.noChecklistText}>No checklist selected</Text>
            )}
          </View>
          {checklistError && showErrors && (
            <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Checklist is required</Text>
          )}
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>GPS Location</Text>
            <TouchableOpacity onPress={handleGetLocation} style={styles.selectButton}>
              <MaterialIcons name="location-on" size={16} color="#0078D4" />
              <Text style={styles.selectText}>Get Location</Text>
            </TouchableOpacity>
          </View>
          <View style={styles.gpsDetailsBox}>
            {coordinates ? (
              <View style={styles.gpsInfoRow}>
                <Text style={styles.gpsLabel}>Coordinates: </Text>
                <Text style={styles.gpsValue}>{coordinates}</Text>
              </View>
            ) : null}
            {gpsAddress ? (
              <View style={styles.gpsInfoRow}>
                <Text style={styles.gpsLabel}>GPS Address: </Text>
                <Text style={styles.gpsValue}>{gpsAddress}</Text>
              </View>
            ) : null}
            {!coordinates && !gpsAddress && (
              <Text style={styles.noGpsText}>Click "Get Location" to fetch your current location</Text>
            )}
          </View>
        </ScrollView>
        <View style={styles.buttonContainer}>
          <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
            <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
            <Text style={styles.cancelButtonText}>Cancel</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.nextButton}
            onPress={handleNextPress}
          >
            <Text style={styles.nextButtonText}>Next</Text>
            <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
          </TouchableOpacity>
        </View>
      </KeyboardAvoidingView>
      <Modal
        visible={checklistModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setChecklistModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.checklistModalContent}>
            <View style={styles.checklistModalHeader}>
              <Text style={styles.checklistModalTitle}>Select Checklist</Text>
              <TouchableOpacity
                onPress={() => setChecklistModalVisible(false)}
                style={styles.modalCloseIcon}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <View style={styles.checklistModalTabs}>
              <TouchableOpacity
                style={[styles.tabButton, selectedTab === 'ALL' && styles.activeTabButton]}
                onPress={() => setSelectedTab('ALL')}
              >
                <Text style={[styles.tabText, selectedTab === 'ALL' && styles.activeTabText]}>
                  ALL ({checklists.length})
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.tabButton, selectedTab === 'Assign' && styles.activeTabButton]}
                onPress={() => setSelectedTab('Assign')}
              >
                <Text style={[styles.tabText, selectedTab === 'Assign' && styles.activeTabText]}>
                  Assign ({checklists.filter(c => c.isUsed).length})
                </Text>
              </TouchableOpacity>
            </View>
            <View style={styles.checklistSearchContainer}>
              {isChecklistSearchActive ? (
                <TextInput
                  style={styles.checklistSearchInput}
                  placeholder="Search checklists..."
                  placeholderTextColor="#888"
                  value={checklistSearchQuery}
                  onChangeText={handleChecklistSearch}
                  autoFocus
                />
              ) : (
                <Text style={styles.checklistSelectText}>Search Checklists</Text>
              )}
              <TouchableOpacity
                onPress={() => {
                  setIsChecklistSearchActive(!isChecklistSearchActive);
                  if (isChecklistSearchActive) setChecklistSearchQuery('');
                }}
              >
                <MaterialIcons
                  name={isChecklistSearchActive ? 'close' : 'search'}
                  size={24}
                  color="#666"
                />
              </TouchableOpacity>
            </View>
            <View style={styles.divider} />
            <FlatList
              data={filteredChecklists}
              keyExtractor={(item) => item.id.toString()}
              renderItem={renderChecklistItem}
              ListEmptyComponent={() => {
                console.log('📋 Rendering empty component:', {
                  checklistsLength: checklists.length,
                  filteredChecklistsLength: filteredChecklists.length,
                  searchQuery: checklistSearchQuery
                });
                return (
                  <View style={styles.checklistEmptyContainer}>
                    <Text style={styles.checklistEmptyText}>
                      {checklists.length === 0 ? 'No checklists available' : 'No checklists match your search'}
                    </Text>
                    {checklists.length === 0 && (
                      <Text style={styles.checklistEmptySubtext}>
                        Please check your connection or contact administrator
                      </Text>
                    )}
                  </View>
                );
              }}
              style={styles.checklistList}
            />
            <View style={styles.divider} />
            <View style={styles.checklistModalFooter}>
              <TouchableOpacity
                onPress={() => {
                  setChecklistModalVisible(false);
                  if (navigation.canGoBack('Checklist_home')) {
                    navigation.navigate('Checklist_home', { checklist: null });
                  } else {
                    Toast.show({
                      type: 'error',
                      text1: 'Navigation Error',
                      text2: 'Checklist creation screen not available.',
                      position: 'top',
                    });
                  }
                }}
              >
                <Text style={styles.createChecklistText}>Create New Checklist</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.checklistCloseButton}
                onPress={() => setChecklistModalVisible(false)}
              >
                <Text style={styles.checklistCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Modal
        visible={equipmentModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setEquipmentModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.equipmentModalContent}>
            <View style={styles.equipmentModalHeader}>
              <Text style={styles.equipmentModalTitle}>Select Equipments</Text>
              <TouchableOpacity
                onPress={() => setEquipmentModalVisible(false)}
                style={styles.modalCloseIcon}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <TouchableOpacity
              style={[
                styles.equipmentSearchContainer,
                isEquipmentSearchActive && styles.equipmentSearchContainerActive,
              ]}
              onPress={() => {
                setIsEquipmentSearchActive(true);
                equipmentSearchInputRef.current?.focus();
              }}
            >
              {isEquipmentSearchActive ? (
                <TextInput
                  ref={equipmentSearchInputRef}
                  style={[
                    styles.equipmentSearchInput,
                    isEquipmentSearchActive && styles.equipmentSearchInputActive,
                  ]}
                  placeholder="Search..."
                  placeholderTextColor="#888"
                  value={equipmentSearchQuery}
                  onChangeText={handleEquipmentSearch}
                  autoFocus={isEquipmentSearchActive}
                />
              ) : (
                <Text style={styles.equipmentSelectText}>Search Equipment</Text>
              )}
              <TouchableOpacity
                onPress={() => {
                  if (isEquipmentSearchActive) {
                    setIsEquipmentSearchActive(false);
                    setEquipmentSearchQuery('');
                    equipmentSearchInputRef.current?.blur();
                  } else {
                    setIsEquipmentSearchActive(true);
                    equipmentSearchInputRef.current?.focus();
                  }
                }}
              >
                <MaterialIcons
                  name={isEquipmentSearchActive ? 'close' : 'search'}
                  size={24}
                  color="#666"
                />
              </TouchableOpacity>
            </TouchableOpacity>
            <View style={styles.divider} />
            {isEquipmentLoading ? (
              <View style={styles.equipmentLoadingContainer}>
                <MaterialIcons name="hourglass-empty" size={30} color="#3481BC" />
                <Text style={styles.equipmentLoadingText}>Loading equipment...</Text>
              </View>
            ) : (
              <FlatList
                data={getFilteredEquipment()}
                keyExtractor={(item) => item.id.toString()}
                renderItem={renderEquipmentItem}
                ListEmptyComponent={<Text style={styles.equipmentEmptyText}>No equipment found</Text>}
                style={styles.equipmentList}
              />
            )}
            <View style={styles.divider} />
            <View style={styles.equipmentModalFooter}>
              <TouchableOpacity
                onPress={() => {
                  setEquipmentModalVisible(false);
                    navigation.navigate('Add_equipment');
                }}
              >
                <Text style={styles.addEquipmentText}>Add your own equipment</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.equipmentCloseButton}
                onPress={() => setEquipmentModalVisible(false)}
              >
                <Text style={styles.equipmentCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message="Fetching Data"
      />
      <Toast />
    </View>
  );
};








// Styles
// Styles
// Styles
const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 130,
  },
  headingContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 10,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  requestRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    marginBottom: 20,
  },
  requestLabel: {
    fontSize: 16,
    color: '#015185',
  },
  requestValue: {
    fontSize: 16,
    color: '#015185',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: 16,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  selectRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 5,
  },
  selectLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectText: {
    fontSize: 16,
    color: '#0078D4',
    marginLeft: 5,
  },
  equipmentDetailsBox: {
    width: '100%',
    backgroundColor: '#f0f0f0',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginBottom: 10,
    borderRadius: 5,
    alignSelf: 'center',
  },
  gpsDetailsBox: {
    width: '100%',
    backgroundColor: '#f0f0f0',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginBottom: 10,
    borderRadius: 5,
    alignSelf: 'center',
  },
  gpsInfoRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: 5,
  },
  gpsLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  gpsValue: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  noGpsText: {
    fontSize: 14,
    color: '#888',
    textAlign: 'center',
  },
  vehicleEquipmentRow: {
    paddingVertical: 10,
  },
  vehicleEquipmentText: {
    fontSize: 16,
    color: '#333',
  },
  meterReadingContainer: {
    padding: 12,
    backgroundColor: '#F9F9F9',
    borderRadius: 6,
    marginTop: 8,
  },
  labelRow: {
    marginBottom: 8,
  },
  meterReadingLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  inputRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 8,
  },
  meterReadingInput: {
    flex: 1,
    fontSize: 16,
    color: '#333',
    padding: 0,
  },
  meterUnit: {
    color: '#666',
    fontSize: 14,
    marginLeft: 8,
  },
  currentMeterRow: {
    flexDirection: 'row',
    marginTop: 8,
    alignItems: 'center',
  },
  currentMeterLabel: {
    fontSize: 13,
    color: '#666',
  },
  currentMeterValue: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  checklistModalContent: {
    width: '90%',
    maxHeight: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  checklistModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  checklistModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  modalCloseIcon: {
    padding: 5,
  },
  checklistModalTabs: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
  },
  tabButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 10,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  activeTabButton: {
    borderBottomColor: '#0078D4',
  },
  tabText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '600',
  },
  activeTabText: {
    color: '#0078D4',
  },
  checklistSearchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
  },
  checklistSelectText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  checklistSearchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  divider: {
    height: 1,
    backgroundColor: '#eee',
    marginVertical: 10,
  },
  checklistList: {
    maxHeight: 200,
  },
  checklistItem: {
    paddingVertical: 10,
  },
  checklistItemContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  checklistItemText: {
    fontSize: 16,
    color: '#333',
    marginLeft: 10,
  },
  checklistEmptyContainer: {
    paddingVertical: 20,
    paddingHorizontal: 10,
    alignItems: 'center',
  },
  checklistEmptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginBottom: 5,
  },
  checklistEmptySubtext: {
    fontSize: 14,
    color: '#aaa',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  checklistModalFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  createChecklistText: {
    fontSize: 16,
    color: '#0078D4',
    fontWeight: '600',
  },
  checklistCloseButton: {
    padding: 10,
  },
  checklistCloseText: {
    fontSize: 16,
    color: '#3481BC',
    fontWeight: '600',
  },
  equipmentModalContent: {
    width: '90%',
    maxHeight: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  equipmentModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  equipmentModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  equipmentSearchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 10,
  },
  equipmentSearchContainerActive: {
    paddingVertical: 2,
  },
  equipmentSelectText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  equipmentSearchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  equipmentSearchInputActive: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  equipmentTabs: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    paddingHorizontal: 15,
    paddingVertical: 10,
  },
  equipmentList: {
    maxHeight: 300,
    paddingHorizontal: 12,
  },
  equipmentRow: {
    marginVertical: 5,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
    paddingRight: 10,
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: '5%',
    paddingRight: '10%',
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingLeft: '5%',
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: '5%',
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatus: {
    fontSize: 14,
    fontWeight: '400',
    width: '60%',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: 18,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: 16,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  selectEquipmentButton: {
    position: 'absolute',
    right: 10,
    bottom: 10,
    backgroundColor: '#0078D4',
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 5,
  },
  selectEquipmentText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  equipmentEmptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  equipmentModalFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  addEquipmentText: {
    fontSize: 16,
    color: '#0078D4',
    fontWeight: '600',
  },
  equipmentCloseButton: {
    padding: 10,
  },
  equipmentCloseText: {
    fontSize: 16,
    color: '#3481BC',
    fontWeight: '600',
  },
  selectedChecklistsBox: {
    width: '94%',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginBottom: 10,
    alignSelf: 'center',
  },
  selectedChecklistText: {
    fontSize: 14,
    color: '#333',
    marginBottom: 5,
  },
  noChecklistText: {
    fontSize: 14,
    color: '#888',
  },
  bottomicon: {
    marginTop: 2,
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  daysBar: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 10,
    paddingVertical: 8,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dayItem: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 8,
    borderRadius: 8,
  },
  daySelected: {
    backgroundColor: '#F8D6D6',
  },
  currentDaySelected: {
    backgroundColor: '#F36B6B',
    opacity: 0.8,
  },
  dayText: {
    color: '#888',
    fontSize: 16,
    fontWeight: '500',
  },
  dayTextSelected: {
    color: '#fff',
  },
  currentDayText: {
    color: '#fff',
  },
  dateText: {
    fontSize: 12,
    color: '#666',
    marginTop: 2,
  },
  equipmentLoadingContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 10,
  },
  equipmentLoadingText: {
    fontSize: 16,
    color: '#3481BC',
    marginLeft: 5,
  },
  checklistDocumentInfo: {
    fontSize: 14,
    color: '#015185',
    marginTop: 5,
    fontWeight: '500',
  },
});

export default Daily_Ins_InfoStep1;