import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  Alert,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import OfflineIncidentHelper from './OfflineIncidentHelper';

const API_BASE_URL = 'http://168.231.113.219:5007';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const IncidentReport = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [incidentReports, setIncidentReports] = useState([]);
  const [filteredReports, setFilteredReports] = useState([]);
  const [archivedReports, setArchivedReports] = useState([]);
  const [filteredArchivedReports, setFilteredArchivedReports] = useState([]);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [selectedReport, setSelectedReport] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0, openUp: false });
  const [confirmDeleteVisible, setConfirmDeleteVisible] = useState(false);
  const [reportToDelete, setReportToDelete] = useState(null);
  const [showArchived, setShowArchived] = useState(false);
  const [syncMessage, setSyncMessage] = useState('Cloud Data Sync in Progress');
  const [reportedByName, setReportedByName] = useState('N/A');
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState({
    countries: [],
    projects: [],
    incidentReportPermissions: [],
  });
  const [currentUserUid, setCurrentUserUid] = useState(null);
  const [hasOnlyMineView, setHasOnlyMineView] = useState(false);
  const [hasCreatePermission, setHasCreatePermission] = useState(false);
  const [hasDeletePermission, setHasDeletePermission] = useState(false);
  const [hasEditPermission, setHasEditPermission] = useState(false);
  const [hasViewPermission, setHasViewPermission] = useState(false);
  const [offlineReportsCount, setOfflineReportsCount] = useState(0);
  const [onlineReportsCount, setOnlineReportsCount] = useState(0);
  const [localStorageSize, setLocalStorageSize] = useState('0 KB');
  const [totalMongoCount, setTotalMongoCount] = useState(0);
  const [isSyncing, setIsSyncing] = useState(false);
  const isSyncingOfflineReports = useRef(false);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const fetchReportedByName = async () => {
    try {
      const userName = await AsyncStorage.getItem('userName');
      if (userName) {
        setReportedByName(userName);
      } else {
        setReportedByName('N/A');
      }
    } catch (error) {
      console.error('Error fetching reported by name:', error);
      setReportedByName('N/A');
    }
  };

  const fetchUserPermissions = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }
      
      setCurrentUserUid(userUid);

      // PRIORITY 1: Try incidentModuleUserInfo first (most specific)
      let loadedUserInfo = null;
      try {
        const cachedIncidentUserInfo = await AsyncStorage.getItem('incidentModuleUserInfo');
        if (cachedIncidentUserInfo) {
          loadedUserInfo = JSON.parse(cachedIncidentUserInfo);
          
          setUserPermissions({
            countries: loadedUserInfo.countries || [],
            projects: loadedUserInfo.projects || [],
            incidentReportPermissions: loadedUserInfo.incidentReportPermissions || [],
          });

          // Set permission flags
          const permissions = loadedUserInfo.incidentReportPermissions || [];
          setHasOnlyMineView(permissions.includes('onlyMineView'));
          setHasCreatePermission(permissions.includes('create'));
          setHasDeletePermission(permissions.includes('delete'));
          setHasEditPermission(permissions.includes('edit'));
          setHasViewPermission(permissions.includes('view'));
          
          console.log('✅ Loaded incident permissions from cache:', permissions);
        }
      } catch (error) {
        console.error('❌ Error loading cached incident permissions:', error.message);
      }
      
      // PRIORITY 2: Fallback to cached reports if no permissions found
      if (!loadedUserInfo) {
        try {
          const cachedReports = await AsyncStorage.getItem('incidentReports');
          if (cachedReports) {
            const reports = JSON.parse(cachedReports);
            if (reports.length > 0 && reports[0].userPermissions) {
              loadedUserInfo = {
                countries: reports[0].userCountries || [],
                projects: reports[0].userProjects || [],
                incidentReportPermissions: reports[0].userPermissions || [],
                employeeNumber: '',
                name: '',
              };
              console.log('✅ Found permissions in cached reports:', loadedUserInfo.incidentReportPermissions);
            }
          }
        } catch (error) {
          console.error('❌ Error loading from cached reports:', error.message);
        }
      }
      
      // PRIORITY 3: Fallback to offline incident reports
      if (!loadedUserInfo) {
        try {
          const offlineReports = await OfflineIncidentHelper.getOfflineReports();
          if (offlineReports.length > 0 && offlineReports[0].userPermissions) {
            loadedUserInfo = {
              countries: offlineReports[0].userCountries || [],
              projects: offlineReports[0].userProjects || [],
              incidentReportPermissions: offlineReports[0].userPermissions || [],
              employeeNumber: '',
              name: '',
            };
            console.log('✅ Found permissions in offline incident reports:', loadedUserInfo.incidentReportPermissions);
          }
        } catch (error) {
          console.error('❌ Error loading from offline reports:', error.message);
        }
      }
      
      // Set permissions immediately if loaded from any cache source
      if (loadedUserInfo) {
        const permissionsObj = {
          countries: loadedUserInfo.countries || [],
          projects: loadedUserInfo.projects || [],
          incidentReportPermissions: loadedUserInfo.incidentReportPermissions || [],
        };
        
        setUserPermissions(permissionsObj);

        // Set permission flags
        const permissions = loadedUserInfo.incidentReportPermissions || [];
        setHasOnlyMineView(permissions.includes('onlyMineView'));
        setHasCreatePermission(permissions.includes('create'));
        setHasDeletePermission(permissions.includes('delete'));
        setHasEditPermission(permissions.includes('edit'));
        setHasViewPermission(permissions.includes('view'));
        
        // Store for next time
        await AsyncStorage.setItem('incidentModuleUserInfo', JSON.stringify(loadedUserInfo));
        await AsyncStorage.setItem('incidentModulePermissions', JSON.stringify(permissions));
        
        console.log('✅ OFFLINE - Incident permissions loaded successfully:');
        console.log('   Permissions:', permissions.join(', ') || 'None');
        console.log('   Countries:', loadedUserInfo.countries?.join(', ') || 'None');
        console.log('   Projects:', loadedUserInfo.projects?.length || 0, 'projects');
      } else {
        console.log('⚠️ NO PERMISSIONS FOUND IN ANY OFFLINE STORAGE!');
      }

      // PRIORITY 4: Fetch fresh from MongoDB if online
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      
      if (!isOnline) {
        console.log('📴 Offline - Using cached permissions only');
        // Return cached permissions
        if (loadedUserInfo) {
          return {
            countries: loadedUserInfo.countries || [],
            projects: loadedUserInfo.projects || [],
            incidentReportPermissions: loadedUserInfo.incidentReportPermissions || [],
          };
        }
        return {
          countries: [],
          projects: [],
          incidentReportPermissions: [],
        };
      }

      // Fetch user data from MongoDB API
      const response = await fetch(`http://168.231.113.219:5003/users/${userUid}`);
      const result = await response.json();

      if (!result.success || !result.data) {
        throw new Error('User data not found.');
      }

      const userData = result.data;

      const incidentReportModule = userData.modules?.find(
        (mod) => mod.module === 'incidentReport'
      );

      const permissions = incidentReportModule?.permissions || [];
      
      setUserPermissions({
        countries: userData.countries || [],
        projects: userData.projects || [],
        incidentReportPermissions: permissions,
      });

      // Set permission flags
      setHasOnlyMineView(permissions.includes('onlyMineView'));
      setHasCreatePermission(permissions.includes('create'));
      setHasDeletePermission(permissions.includes('delete'));
      setHasEditPermission(permissions.includes('edit'));
      setHasViewPermission(permissions.includes('view'));
      
      // CRITICAL: Cache permissions for offline access
      const incidentModuleUserInfo = {
        countries: userData.countries || [],
        projects: userData.projects || [],
        incidentReportPermissions: permissions,
        employeeNumber: userData.employeeNumber || '',
        name: userData.name || '',
      };
      
      await AsyncStorage.setItem('incidentModuleUserInfo', JSON.stringify(incidentModuleUserInfo));
      await AsyncStorage.setItem('incidentModulePermissions', JSON.stringify(permissions));
      console.log('✅ Updated incident module permissions from MongoDB:', permissions);
      
      // Return updated permissions
      return {
        countries: userData.countries || [],
        projects: userData.projects || [],
        incidentReportPermissions: permissions,
      };
      
    } catch (error) {
      console.error('Error fetching user permissions:', error);
      
      // Try to load from cache on error
      try {
        const cachedIncidentUserInfo = await AsyncStorage.getItem('incidentModuleUserInfo');
        if (cachedIncidentUserInfo) {
          const loadedInfo = JSON.parse(cachedIncidentUserInfo);
          
          setUserPermissions({
            countries: loadedInfo.countries || [],
            projects: loadedInfo.projects || [],
            incidentReportPermissions: loadedInfo.incidentReportPermissions || [],
          });
          
          const permissions = loadedInfo.incidentReportPermissions || [];
          setHasOnlyMineView(permissions.includes('onlyMineView'));
          setHasCreatePermission(permissions.includes('create'));
          setHasDeletePermission(permissions.includes('delete'));
          setHasEditPermission(permissions.includes('edit'));
          setHasViewPermission(permissions.includes('view'));
          
          console.log('✅ Fallback: Loaded permissions from cache after error');
          
          return {
            countries: loadedInfo.countries || [],
            projects: loadedInfo.projects || [],
            incidentReportPermissions: permissions,
          };
        }
      } catch (cacheError) {
        console.error('❌ Could not load from cache:', cacheError);
      }
      
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to fetch user permissions.',
        position: 'top',
      });
      setUserPermissions({
        countries: [],
        projects: [],
        incidentReportPermissions: [],
      });
      
      return {
        countries: [],
        projects: [],
        incidentReportPermissions: [],
      };
    }
  };

  const isReportEmpty = (report) => {
    const fields = [
      report.incidentNumber,
      report.incidentDate,
      report.incidentArea,
      report.severity,
      report.incidentCategory,
      report.selectedIncidents && Object.values(report.selectedIncidents).some(Boolean),
    ];
    return fields.every(
      (field) =>
        field === undefined ||
        field === null ||
        field === '' ||
        field === 'N/A' ||
        field === false
    );
  };

  // Helper: filter reports according to current permissions
  const filterReportsByPermissions = (allReports, permissions, onlyMine, userUid) => {
    let filtered = allReports;
    
    console.log('🔍 Filtering reports by permissions:', {
      totalReports: allReports.length,
      countries: permissions.countries,
      projects: permissions.projects,
      onlyMine: onlyMine,
    });
    
    // STEP 1: Filter by country (if user has country restrictions)
    if (permissions.countries && permissions.countries.length > 0) {
      const beforeCountryFilter = filtered.length;
      filtered = filtered.filter((report) => {
        // Show report if it matches ANY of the user's assigned countries
        return permissions.countries.includes(report.country);
      });
      console.log(`📍 Country filter: ${beforeCountryFilter} → ${filtered.length} (allowed: ${permissions.countries.join(', ')})`);
    } else {
      console.log('📍 No country restrictions - showing reports from all countries');
    }
    
    // STEP 2: Filter by project (if user has project restrictions)
    if (permissions.projects && permissions.projects.length > 0) {
      const beforeProjectFilter = filtered.length;
      filtered = filtered.filter((report) => {
        // Show report if it matches ANY of the user's assigned projects
        return permissions.projects.includes(report.project);
      });
      console.log(`📍 Project filter: ${beforeProjectFilter} → ${filtered.length} (allowed: ${permissions.projects.length} projects)`);
    } else {
      console.log('📍 No project restrictions - showing reports from all projects');
    }
    
    // STEP 3: Filter by "onlyMineView" permission (ONLY if explicitly enabled)
    if (onlyMine && userUid) {
      const beforeOnlyMineFilter = filtered.length;
      filtered = filtered.filter((report) => report.adminUid === userUid);
      console.log(`👤 onlyMineView filter: ${beforeOnlyMineFilter} → ${filtered.length} (showing only reports created by user)`);
    } else {
      console.log('👥 Showing reports from ALL users (no onlyMineView restriction)');
    }
    
    console.log(`✅ Final filtered count: ${filtered.length} reports`);
    return filtered;
  };

  // Update report counts (offline vs online)
  const updateReportCounts = async () => {
    try {
      // Get offline pending sync reports (created offline, not yet in MongoDB)
      const offlinePendingReports = await OfflineIncidentHelper.getOfflineReports();
      const pendingCount = offlinePendingReports.length;
      
      // Get cached reports from MongoDB (synced and stored offline)
      const cachedReportsRaw = await AsyncStorage.getItem('incidentReports');
      const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
      const cachedCount = cachedReports.length;
      
      // TOTAL OFFLINE STORED = Pending + Cached (ALL reports available offline)
      const totalOfflineCount = pendingCount + cachedCount;
      setOfflineReportsCount(totalOfflineCount);
      
      // Online in DB count = Just the cached reports (synced to MongoDB)
      setOnlineReportsCount(cachedCount);

      // Fetch total count from MongoDB (all reports in database)
      try {
        const netInfo = await NetInfo.fetch();
        const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
        
        if (isOnline) {
          const response = await fetch(`${API_BASE_URL}/incident-reports`);
          const result = await response.json();
          if (result.success && result.data) {
            setTotalMongoCount(result.data.length);
          }
        }
      } catch (mongoCountError) {
        console.warn('Could not fetch total MongoDB count:', mongoCountError?.message);
      }
      
      // Calculate local storage size
      try {
        let totalSize = 0;
        const keys = await AsyncStorage.getAllKeys();
        const incidentKeys = keys.filter(key => 
          key.includes('incident') || 
          key.includes('Incident') || 
          key.includes('offline')
        );
        
        for (const key of incidentKeys) {
          const value = await AsyncStorage.getItem(key);
          if (value) {
            totalSize += value.length;
          }
        }
        
        // Convert to KB/MB
        const sizeInKB = (totalSize / 1024).toFixed(2);
        const sizeInMB = (totalSize / (1024 * 1024)).toFixed(2);
        const displaySize = totalSize > 1024 * 1024 ? `${sizeInMB} MB` : `${sizeInKB} KB`;
        setLocalStorageSize(displaySize);
      } catch (storageError) {
        // Silent fail
      }
      
    } catch (error) {
      console.error('❌ Count update error:', error.message);
    }
  };

  const fetchIncidentReports = async (showModal = false) => {
    if (showModal) {
      setSyncMessage('Cloud Data Sync in Progress');
      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    }

    try {
      await fetchReportedByName();
      
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Check network status first
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      
      console.log('🌐 Network Status:', isOnline ? 'ONLINE' : 'OFFLINE');

      // Load permissions (this handles both online and offline)
      const fetchedPermissions = await fetchUserPermissions();
      const currentPermissions = fetchedPermissions || userPermissions;
      
      console.log('📋 Loaded Permissions:', {
        countries: currentPermissions.countries,
        projects: currentPermissions.projects?.length || 0,
        permissions: currentPermissions.incidentReportPermissions,
      });

      // If OFFLINE, load from cache immediately
      if (!isOnline) {
        console.log('📴 OFFLINE MODE - Loading from cache...');
        const cachedReportsRaw = await AsyncStorage.getItem('incidentReports');
        const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
        const offlineReports = await OfflineIncidentHelper.getOfflineReports();
        
        console.log('📦 Cached reports:', cachedReports.length);
        console.log('📦 Offline reports:', offlineReports.length);
        
        const formattedOfflineReports = offlineReports.map(report => ({
          ...report,
          offlineCreated: true,
          offlineSyncPending: true,
        }));
        
        const allReports = [...formattedOfflineReports, ...cachedReports];
        
        // Check if onlyMineView is in permissions
        const hasOnlyMineInCurrentPermissions = currentPermissions.incidentReportPermissions?.includes('onlyMineView');
        
        // Apply ALL permission filters (country, project, onlyMineView)
        const filteredAllReports = filterReportsByPermissions(
          allReports,
          currentPermissions,
          hasOnlyMineInCurrentPermissions,
          userUid
        );
        
        const activeReports = filteredAllReports.filter(r => !r.isArchived);
        const archivedReports = filteredAllReports.filter(r => r.isArchived);
        
        // Sort by date
        const sortedActive = activeReports.sort((a, b) => {
          const dateA = new Date(a.createdAt || a.incidentDate || 0);
          const dateB = new Date(b.createdAt || b.incidentDate || 0);
          return dateB.getTime() - dateA.getTime();
        });
        
        const sortedArchived = archivedReports.sort((a, b) => {
          const dateA = new Date(a.createdAt || a.incidentDate || 0);
          const dateB = new Date(b.createdAt || b.incidentDate || 0);
          return dateB.getTime() - dateA.getTime();
        });
        
        setIncidentReports(sortedActive);
        setFilteredReports(sortedActive);
        setArchivedReports(sortedArchived);
        setFilteredArchivedReports(sortedArchived);
        
        await updateReportCounts();
        
        Toast.show({
          type: 'info',
          text1: 'Offline Mode',
          text2: `Loaded ${sortedActive.length} reports from cache`,
          position: 'top',
        });
        
        if (showModal) {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
        }
        return;
      }

      // ONLINE - Fetch from API (GET ALL REPORTS - NO FILTERING)
      console.log('🌐 ONLINE MODE - Fetching ALL reports from MongoDB...');
      const response = await fetch(`${API_BASE_URL}/incident-reports`);
      const result = await response.json();
      
      console.log('📡 API Response:', {
        success: result.success,
        totalReports: result.data?.length || 0,
      });
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to fetch reports');
      }

      const reports = result.data || [];
      console.log('📊 Total reports from DB:', reports.length);
      
      const nonEmptyReports = reports.filter((report) => !isReportEmpty(report));
      console.log('✅ Non-empty reports:', nonEmptyReports.length);
      
      // Check if onlyMineView is in the CURRENT permissions (not state variable)
      const hasOnlyMineInCurrentPermissions = currentPermissions.incidentReportPermissions?.includes('onlyMineView');
      
      console.log('🔍 Permission Check:', {
        stateHasOnlyMineView: hasOnlyMineView,
        currentPermissionsOnlyMine: hasOnlyMineInCurrentPermissions,
        permissions: currentPermissions.incidentReportPermissions,
      });
      
      // Apply ALL permission filters (country, project, onlyMineView)
      let filtered = filterReportsByPermissions(
        nonEmptyReports,
        currentPermissions,
        hasOnlyMineInCurrentPermissions,
        userUid
      );
      
      console.log('📊 Final filtered count:', filtered.length, 'out of', nonEmptyReports.length);
      
      // Sort reports by creation date (newest first)
      const sortedReports = filtered.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime(); // Newest first
      });
      
      const activeReports = sortedReports.filter((report) => !report.isArchived);
      const archivedReports = sortedReports.filter((report) => report.isArchived);
      
      console.log('Active reports:', activeReports.length);
      console.log('Archived reports:', archivedReports.length);
      
      // CRITICAL: Store permissions with each report for offline access
      const reportsWithPermissions = filtered.map(report => ({
        ...report,
        userPermissions: currentPermissions.incidentReportPermissions || [],
        userCountries: currentPermissions.countries || [],
        userProjects: currentPermissions.projects || [],
        syncedAt: new Date().toISOString(),
      }));
      
      console.log('📊 Storing permissions with reports:');
      console.log('   Total reports:', reportsWithPermissions.length);
      console.log('   Permissions per report:', currentPermissions.incidentReportPermissions);
      console.log('   Countries per report:', currentPermissions.countries);
      console.log('   Projects per report:', currentPermissions.projects?.length || 0, 'projects');
      if (reportsWithPermissions.length > 0) {
        console.log('   Sample report permissions:', reportsWithPermissions[0].userPermissions);
      }
      
      // Load offline reports and merge with online
      const offlineReports = await OfflineIncidentHelper.getOfflineReports();
      const formattedOfflineReports = offlineReports.map(report => ({
        ...report,
        offlineCreated: true,
        offlineSyncPending: true,
        // Preserve permissions from offline report if available
        userPermissions: report.userPermissions || currentPermissions.incidentReportPermissions || [],
        userCountries: report.userCountries || currentPermissions.countries || [],
        userProjects: report.userProjects || currentPermissions.projects || [],
      }));
      
      // Merge and deduplicate
      const allActiveReports = [...formattedOfflineReports, ...activeReports];
      const uniqueReportsMap = new Map();
      allActiveReports.forEach(report => {
        const reportId = report.id || report.incidentNumber || report._id;
        if (!uniqueReportsMap.has(reportId)) {
          uniqueReportsMap.set(reportId, report);
        }
      });
      const uniqueActiveReports = Array.from(uniqueReportsMap.values());
      
      setIncidentReports(uniqueActiveReports);
      setFilteredReports(uniqueActiveReports);
      setArchivedReports(archivedReports);
      setFilteredArchivedReports(archivedReports);
      
      // Store reports with permissions
      await AsyncStorage.setItem('incidentReports', JSON.stringify(reportsWithPermissions));
      
      // Update counts
      await updateReportCounts();
    } catch (error) {
      console.error('❌ Error fetching incident reports:', error);
      
      // FALLBACK: Always load from cache on error
      console.log('🔄 Loading from offline cache as fallback...');
      
      try {
        const localReports = await AsyncStorage.getItem('incidentReports');
        const parsedLocalReports = localReports ? JSON.parse(localReports) : [];
        const offlineReports = await OfflineIncidentHelper.getOfflineReports();
        
        console.log('📦 Cache fallback - Cached:', parsedLocalReports.length, 'Offline:', offlineReports.length);
        
        const formattedOfflineReports = offlineReports.map(report => ({
          ...report,
          offlineCreated: true,
          offlineSyncPending: true,
        }));
        
        const allReports = [...formattedOfflineReports, ...parsedLocalReports];
        
        // Get user UID for filtering
        const userUid = await AsyncStorage.getItem('userUid');
        
        // Get permissions from cache
        const cachedIncidentUserInfo = await AsyncStorage.getItem('incidentModuleUserInfo');
        let fallbackPermissions = {
          countries: [],
          projects: [],
          incidentReportPermissions: []
        };
        
        if (cachedIncidentUserInfo) {
          const userInfo = JSON.parse(cachedIncidentUserInfo);
          fallbackPermissions = {
            countries: userInfo.countries || [],
            projects: userInfo.projects || [],
            incidentReportPermissions: userInfo.incidentReportPermissions || []
          };
        }
        
        // Check if onlyMineView is in permissions
        const hasOnlyMineInFallback = fallbackPermissions.incidentReportPermissions?.includes('onlyMineView');
        
        // Apply ALL permission filters (country, project, onlyMineView)
        const filteredAllReports = filterReportsByPermissions(
          allReports,
          fallbackPermissions,
          hasOnlyMineInFallback,
          userUid
        );
        
        const activeLocalReports = filteredAllReports.filter((report) => !report.isArchived);
        const archivedLocalReports = filteredAllReports.filter((report) => report.isArchived);
        
        // Sort by date
        const sortedActive = activeLocalReports.sort((a, b) => {
          const dateA = new Date(a.createdAt || a.incidentDate || 0);
          const dateB = new Date(b.createdAt || b.incidentDate || 0);
          return dateB.getTime() - dateA.getTime();
        });
        
        const sortedArchived = archivedLocalReports.sort((a, b) => {
          const dateA = new Date(a.createdAt || a.incidentDate || 0);
          const dateB = new Date(b.createdAt || b.incidentDate || 0);
          return dateB.getTime() - dateA.getTime();
        });
        
        setIncidentReports(sortedActive);
        setFilteredReports(sortedActive);
        setArchivedReports(sortedArchived);
        setFilteredArchivedReports(sortedArchived);
        
        await updateReportCounts();
        
        if (sortedActive.length > 0) {
          Toast.show({
            type: 'info',
            text1: 'Offline Mode',
            text2: `Loaded ${sortedActive.length} reports from cache`,
            position: 'top',
          });
        } else {
          Toast.show({
            type: 'warning',
            text1: 'No Reports',
            text2: 'No cached reports available. Try syncing when online.',
            position: 'top',
          });
        }
      } catch (cacheError) {
        console.error('❌ Cache load error:', cacheError);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Could not load incident reports.',
          position: 'top',
        });
      }
    } finally {
      if (showModal) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
      }
    }
  };

  // On mount, fetch permissions and reports, then filter
  useFocusEffect(
    React.useCallback(() => {
      const loadAll = async () => {
        // Use the same logic as fetchIncidentReports but without modal
        await fetchIncidentReports(false);
      };
      loadAll();
    }, [])
  );

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filteredActive = incidentReports.filter((report) =>
        [
          report.incidentNumber,
          report.country,
          report.project,
          report.vehicleId,
          report.vehicleName,
          report.personInvolved,
          report.incidentArea,
          report.severity,
          report.incidentCategory,
          ...Object.keys(report.selectedIncidents || {}).filter(
            (key) => report.selectedIncidents[key]
          ),
        ].some(
          (field) =>
            field && field.toLowerCase().includes(text.toLowerCase())
        )
      );
      const filteredArchived = archivedReports.filter((report) =>
        [
          report.incidentNumber,
          report.country,
          report.project,
          report.vehicleId,
          report.vehicleName,
          report.personInvolved,
          report.incidentArea,
          report.severity,
          report.incidentCategory,
          ...Object.keys(report.selectedIncidents || {}).filter(
            (key) => report.selectedIncidents[key]
          ),
        ].some(
          (field) =>
            field && field.toLowerCase().includes(text.toLowerCase())
        )
      );
      
      // Sort search results by creation date (newest first)
      const sortedActive = filteredActive.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      const sortedArchived = filteredArchived.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      setFilteredReports(sortedActive);
      setFilteredArchivedReports(sortedArchived);
    } else {
      setFilteredReports(incidentReports);
      setFilteredArchivedReports(archivedReports);
    }
  };

  // Sync offline reports to MongoDB
  const syncOfflineReports = async () => {
    // Prevent concurrent syncs
    if (isSyncingOfflineReports.current) {
      console.log('⚠️ Offline sync already in progress, skipping...');
      return { success: true, synced: 0, failed: 0 };
    }

    try {
      isSyncingOfflineReports.current = true;
      
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;
      
      console.log('🔒 Starting offline incident reports sync (locked)...');

      // Preload existing cloud incident numbers to compute next numbers per prefix
      let cloudReports = [];
      try {
        const preResp = await fetch(`${API_BASE_URL}/incident-reports`);
        const preJson = await preResp.json();
        if (preJson.success && Array.isArray(preJson.data)) {
          cloudReports = preJson.data;
        }
      } catch (e) {
        console.warn('⚠️ Could not preload cloud reports for numbering:', e?.message || e);
      }

      // Maintain global counter for ALL incidents (not per-prefix)
      let globalNextCounter = 0;
      const batchAssigned = new Set();
      
      // Initialize global counter from ALL cloud reports
      try {
        cloudReports.forEach((r) => {
          const inc = r?.incidentNumber;
          if (inc && typeof inc === 'string' && !inc.includes('-OFFLINE')) {
            const match = inc.match(/INC-(\d+)/);
            if (match) {
              const num = parseInt(match[1], 10);
              if (!isNaN(num)) {
                globalNextCounter = Math.max(globalNextCounter, num);
              }
            }
          }
        });
        globalNextCounter += 1; // Start from next available
      } catch {}
      
      const computeNextGlobal = () => {
        const nextNum = globalNextCounter;
        globalNextCounter += 1; // Increment for next call
        return nextNum;
      };

      // Sync offline reports using OfflineIncidentHelper
      const syncResult = await OfflineIncidentHelper.syncOfflineReports(async (report) => {
        try {
          console.log('📤 Syncing offline incident report:', report.incidentNumber);
          
          // Generate clean request number (remove -OFFLINE suffix)
          let newRequestNumber = report.incidentNumber;
          const oldOfflineId = report.incidentNumber;
          
          if (newRequestNumber && newRequestNumber.includes('-OFFLINE')) {
            newRequestNumber = newRequestNumber.replace('-OFFLINE', '');
            console.log('🧹 Removed -OFFLINE suffix:', report.incidentNumber, '→', newRequestNumber);
          }
          
          // Get next number based on GLOBAL count (not per-prefix)
          try {
            const prefix = newRequestNumber.split('-OFFLINE')[0];
            const match = prefix.match(/^(.*-INC-)/);
            if (match) {
              const basePrefix = match[1];
              const nextNum = computeNextGlobal();
              newRequestNumber = `${basePrefix}${String(nextNum).padStart(3, '0')}`;
              batchAssigned.add(newRequestNumber);
              // Also append to cloudReports list to avoid duplicate with later items in the same batch
              cloudReports.push({ incidentNumber: newRequestNumber });
              console.log('✅ Generated new request number (global):', newRequestNumber);
            }
          } catch (numberError) {
            console.warn('⚠️ Failed to generate new number, using cleaned version:', numberError.message);
          }
          
          // Update report with clean number
          report.incidentNumber = newRequestNumber;
          report.id = newRequestNumber;
          report._id = newRequestNumber;
          if (report.step1) {
            report.step1.incidentNumber = newRequestNumber;
          }

          // Remap stored local PDF key to new id if it exists
          try {
            if (oldOfflineId && oldOfflineId !== newRequestNumber) {
              const oldKey = `incident_pdf_local_${oldOfflineId}`;
              const newKey = `incident_pdf_local_${newRequestNumber}`;
              const existingLocalPdf = await AsyncStorage.getItem(oldKey);
              if (existingLocalPdf) {
                await AsyncStorage.setItem(newKey, existingLocalPdf);
                await AsyncStorage.removeItem(oldKey);
              }
            }
          } catch (remapErr) {
            console.warn('⚠️ Could not remap local PDF key:', remapErr?.message || remapErr);
          }
          
          // Create report in MongoDB
          const createResponse = await fetch(`${API_BASE_URL}/incident-reports`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(report),
          });
          
          const createResult = await createResponse.json();
          
          if (!createResult.success) {
            throw new Error('Failed to create incident in MongoDB');
          }
          
          console.log('✅ Incident created in MongoDB:', newRequestNumber);

          // Source of truth: confirm the stored incidentNumber returned by API
          try {
            const storedNumber = createResult?.data?.incidentNumber || createResult?.data?._id || createResult?.data?.id;
            if (storedNumber && storedNumber !== newRequestNumber) {
              console.log('ℹ️ Server returned different number; adopting:', storedNumber);
              newRequestNumber = storedNumber;
              report.incidentNumber = newRequestNumber;
              report.id = newRequestNumber;
              report._id = newRequestNumber;
              if (report.step1) report.step1.incidentNumber = newRequestNumber;
              batchAssigned.add(newRequestNumber);
              cloudReports.push({ incidentNumber: newRequestNumber });

              // Remap local pdf key if needed
              try {
                if (oldOfflineId && oldOfflineId !== newRequestNumber) {
                  const oldKey = `incident_pdf_local_${oldOfflineId}`;
                  const newKey = `incident_pdf_local_${newRequestNumber}`;
                  const existingLocalPdf = await AsyncStorage.getItem(oldKey);
                  if (existingLocalPdf) {
                    await AsyncStorage.setItem(newKey, existingLocalPdf);
                    await AsyncStorage.removeItem(oldKey);
                  }
                }
              } catch {}
            }
          } catch {}
          
          // Upload PDF if exists
          let uploadedPdfUrl = null;
          if (report.step2?.pdfLocalPath) {
            try {
              const RNFS = require('react-native-fs');
              const fileExists = await RNFS.exists(report.step2.pdfLocalPath);
              
              if (fileExists) {
                const formData = new FormData();
                formData.append('pdf', {
                  uri: Platform.OS === 'android' ? `file://${report.step2.pdfLocalPath}` : report.step2.pdfLocalPath,
                  type: 'application/pdf',
                  name: `incident_${newRequestNumber}.pdf`,
                });
                
                const pdfResponse = await fetch(
                  `${API_BASE_URL}/incident-reports/${userUid}/${newRequestNumber}/upload-pdf`,
                  { method: 'POST', body: formData }
                );
                
                const pdfResult = await pdfResponse.json();
                
                if (pdfResult.success && pdfResult.pdfUrl) {
                  uploadedPdfUrl = pdfResult.pdfUrl;
                  console.log('✅ PDF uploaded:', uploadedPdfUrl);
                }
              }
            } catch (pdfError) {
              console.warn('⚠️ PDF upload failed:', pdfError.message);
            }
          }

          // Send notifications after successful sync (fire-and-forget)
          setTimeout(async () => {
            try {
              const reporterName = report.personInvolved || report.step1?.personInvolved || 'Reporter';
              
              // Get current user's email
              let reporterEmail = '';
              try {
                const userData = await AsyncStorage.getItem('userData');
                if (userData) {
                  const user = JSON.parse(userData);
                  reporterEmail = user.email || '';
                }
              } catch (emailError) {
                console.warn('⚠️ Could not get reporter email:', emailError);
              }
              
              // Prepare incident data for notifications
              const incidentNotificationData = {
                _id: newRequestNumber,
                id: newRequestNumber,
                incidentNumber: newRequestNumber,
                severity: report.severity || report.step1?.severity,
                incidentCategory: report.incidentCategory || report.step1?.incidentCategory,
                country: report.country || report.step1?.country,
                project: report.project || report.step1?.project,
                incidentArea: report.incidentArea || report.step1?.incidentArea,
              };
              
              console.log('🔔 Sending incident notifications for synced report:', newRequestNumber);
              
              const INCIDENT_NOTIFICATIONS_API_URL = 'http://168.231.113.219:5021';
              const controller = new AbortController();
              const timeoutId = setTimeout(() => controller.abort(), 30000);
              
              const response = await fetch(`${INCIDENT_NOTIFICATIONS_API_URL}/api/incident-notifications/new-incident`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                  incidentData: incidentNotificationData,
                  createdByName: reporterName,
                  reporterEmail: reporterEmail,
                  pdfUrl: uploadedPdfUrl,
                }),
                signal: controller.signal,
              });
              
              clearTimeout(timeoutId);
              
              if (response.ok) {
                const result = await response.json();
                console.log(`✅ Notifications sent for ${newRequestNumber}:`, result.recipients?.total || 0, 'users');
              } else {
                console.warn('⚠️ Notification send failed for', newRequestNumber);
              }
            } catch (notifError) {
              console.error('❌ Notification error for synced report:', notifError?.message || notifError);
            }
          }, 500);

          // Update cached incidentReports with the new number/id so UI reflects immediately
          try {
            const cachedRaw = await AsyncStorage.getItem('incidentReports');
            const cached = cachedRaw ? JSON.parse(cachedRaw) : [];
            const replaced = cached.map((r) => {
              const rid = r.id || r.incidentNumber || r._id;
              if (rid === oldOfflineId) {
                return {
                  ...r,
                  id: newRequestNumber,
                  _id: newRequestNumber,
                  incidentNumber: newRequestNumber,
                  step1: { ...(r.step1 || {}), incidentNumber: newRequestNumber },
                  offlineCreated: false,
                  offlineSyncPending: false,
                };
              }
              return r;
            });
            await AsyncStorage.setItem('incidentReports', JSON.stringify(replaced));

            // Also update in-memory lists if present
            const replaceInState = (arr) => arr.map((r) => {
              const rid = r.id || r.incidentNumber || r._id;
              if (rid === oldOfflineId) {
                return {
                  ...r,
                  id: newRequestNumber,
                  _id: newRequestNumber,
                  incidentNumber: newRequestNumber,
                  step1: { ...(r.step1 || {}), incidentNumber: newRequestNumber },
                  offlineCreated: false,
                  offlineSyncPending: false,
                };
              }
              return r;
            });
            setIncidentReports((prev) => replaceInState(prev));
            setFilteredReports((prev) => replaceInState(prev));
          } catch (cacheUpdateErr) {
            console.warn('⚠️ Could not update cached incidentReports:', cacheUpdateErr?.message || cacheUpdateErr);
          }
          
          return { success: true };
        } catch (error) {
          console.error('❌ Sync error for incident report:', error.message);
          return { success: false, error: error.message };
        }
      });
      
      if (syncResult.synced > 0) {
        Toast.show({
          type: 'success',
          text1: 'Sync Complete',
          text2: `${syncResult.synced} offline incident reports synced to MongoDB`,
          position: 'top',
        });
        
        // Reload reports from MongoDB
        await fetchIncidentReports(false);
        await updateReportCounts();
      }
      
      if (syncResult.failed > 0) {
        Toast.show({
          type: 'warning',
          text1: 'Partial Sync',
          text2: `${syncResult.failed} incident reports failed to sync`,
          position: 'top',
        });
      }
      
      console.log('🔓 Offline incident sync completed (unlocked)');
      return syncResult;
    } catch (error) {
      console.error('❌ Sync offline incidents error:', error.message);
      return { success: false, synced: 0, failed: 0 };
    } finally {
      isSyncingOfflineReports.current = false;
    }
  };

  // Handle cloud sync button
  const handleCloudSync = async () => {
    if (isSyncing) return;
    
    try {
      setIsSyncing(true);
      setSyncMessage('Clearing cache and syncing from MongoDB...');
      setSyncModalVisible(true);
      
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
      
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      
      if (!isOnline) {
        Toast.show({
          type: 'error',
          text1: 'Offline',
          text2: 'Cannot sync while offline',
          position: 'top',
        });
        setSyncModalVisible(false);
        setIsSyncing(false);
        return;
      }
      
      // CLEAR CACHE FIRST - Force fresh data from MongoDB
      console.log('🗑️ Clearing cached incident reports...');
      await AsyncStorage.removeItem('incidentReports');
      await AsyncStorage.removeItem('incidentModuleUserInfo');
      console.log('✅ Cache cleared!');
      
      // Sync offline reports
      const syncResult = await syncOfflineReports();
      
      // Refresh from MongoDB with cleared cache
      setSyncMessage('Loading all reports from MongoDB...');
      await fetchIncidentReports(false);
      
      Toast.show({
        type: 'success',
        text1: 'Sync Complete',
        text2: `Synced ${syncResult.synced} reports to MongoDB`,
        position: 'top',
      });
      
    } catch (error) {
      console.error('❌ Cloud sync error:', error);
      Toast.show({
        type: 'error',
        text1: 'Sync Failed',
        text2: 'Could not sync incident reports',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setIsSyncing(false);
    }
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    fetchIncidentReports(true);
    setIsRefreshing(false);
  };

  const handleViewPDF = async (report) => {
    if (!hasViewPermission && !hasEditPermission) {
      Toast.show({
        type: 'error',
        text1: 'Permission Denied',
        text2: 'You do not have permission to view PDF reports.',
        position: 'top',
      });
      return;
    }

    try {
      // Resolve PDF URL from various possible fields and normalize to absolute URL
      const ensureAbsoluteUrl = (value) => {
        if (!value) return null;
        if (typeof value !== 'string') return null;
        const v = value.trim();
        if (!v) return null;
        if (v.startsWith('http://') || v.startsWith('https://') || v.startsWith('file://') || v.startsWith('/')) {
          // Absolute web URL, file URL, or absolute local path
          // If it is a server-relative path like "/uploads/...", make it absolute with API_BASE_URL
          if (v.startsWith('/uploads') || v.startsWith('/public') || v.startsWith('/files')) {
            return `${API_BASE_URL}${v}`;
          }
          return v;
        }
        // Likely a relative path or bare filename
        // Common backends place PDFs under /uploads or serve directly from base
        if (v.endsWith('.pdf')) {
          // Prefer uploads directory if not already present
          const path = v.includes('uploads/') ? v : `uploads/${v}`;
          return `${API_BASE_URL}/${path}`;
        }
        return null;
      };

      let pdfUrl =
        ensureAbsoluteUrl(report?.pdfDownloadUrl) ||
        ensureAbsoluteUrl(report?.pdfUrl) ||
        ensureAbsoluteUrl(report?.step2?.pdfDownloadUrl) ||
        ensureAbsoluteUrl(report?.step2?.pdfUrl) ||
        ensureAbsoluteUrl(report?.pdf) ||
        ensureAbsoluteUrl(report?.filePath);

      // Decide based on connectivity
      let isOnline = false;
      try {
        const net = await NetInfo.fetch();
        isOnline = net.isConnected && net.isInternetReachable !== false;
      } catch {}

      // OFFLINE-FIRST: Use saved local path if available
      const localKeyId = report?.id || report?.incidentNumber;
      if (!isOnline && localKeyId) {
        try {
          const storedLocal = await AsyncStorage.getItem(`incident_pdf_local_${localKeyId}`);
          if (storedLocal) {
            pdfUrl = storedLocal;
          }
        } catch {}
      }

      // If still no URL and report has local pdf path saved
      if (!pdfUrl && report.offlineCreated && report.step2?.pdfLocalPath) {
        const RNFS = require('react-native-fs');
        const fileExists = await RNFS.exists(report.step2.pdfLocalPath);
        if (fileExists) {
          pdfUrl = report.step2.pdfLocalPath;
          console.log('✅ Using offline PDF:', pdfUrl);
          Toast.show({
            type: 'info',
            text1: 'Offline Mode',
            text2: 'Viewing locally stored PDF',
            position: 'top',
          });
        }
      }

      // ONLINE: try to refresh the report from the API to get the latest cloud PDF URL
      if (isOnline && report?.id) {
        try {
          const userUid = await AsyncStorage.getItem('userUid');
          if (userUid) {
            const resp = await fetch(`${API_BASE_URL}/incident-reports/${userUid}/${report.id}`);
            if (resp.ok) {
              const fresh = await resp.json();
              const freshData = fresh?.data || fresh; // support either wrapper or raw
              const cloudUrl =
                ensureAbsoluteUrl(freshData?.pdfDownloadUrl) ||
                ensureAbsoluteUrl(freshData?.pdfUrl) ||
                ensureAbsoluteUrl(freshData?.step2?.pdfDownloadUrl) ||
                ensureAbsoluteUrl(freshData?.step2?.pdfUrl);
              if (cloudUrl && (cloudUrl.startsWith('http://') || cloudUrl.startsWith('https://'))) {
                pdfUrl = cloudUrl; // Prefer cloud URL when present
              }
            }
          }
        } catch (refreshErr) {
          console.warn('PDF refresh from API failed:', refreshErr?.message || refreshErr);
        }
      }
      
      // Try OfflineIncidentHelper for locally cached PDF (only if no cloud URL found)
      if (!pdfUrl && report.id) {
        const offlinePdfPath = await OfflineIncidentHelper.getOfflinePDF(report.id);
        if (offlinePdfPath) {
          pdfUrl = offlinePdfPath;
          console.log('✅ Found offline PDF:', pdfUrl);
        }
      }
      
      console.log('Attempting to view PDF:', {
        reportId: report?.id || report?.incidentNumber,
        pdfUrl: pdfUrl,
        isOffline: !!report.offlineCreated,
      });

      if (!pdfUrl || typeof pdfUrl !== 'string') {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'No PDF available. Generate the report or try syncing first.',
          position: 'top',
        });
        return;
      }

      // Validate PDF URL - accept local paths, absolute URLs, or server-relative paths normalized above
      const isValidUrl =
        typeof pdfUrl === 'string' && (
          pdfUrl.startsWith('https://') ||
          pdfUrl.startsWith('http://') ||
          pdfUrl.startsWith('file://') ||
          pdfUrl.startsWith('/storage') || // Android absolute path
          pdfUrl.startsWith('/data')
        );
      
      if (!isValidUrl) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Invalid PDF URL for this report.',
          position: 'top',
        });
        console.error('Invalid PDF URL:', pdfUrl);
        return;
      }

      // Normalize Android local paths to file:// URIs
      if (Platform.OS === 'android' && (pdfUrl.startsWith('/storage') || pdfUrl.startsWith('/data'))) {
        pdfUrl = `file://${pdfUrl}`;
      }

      console.log('Navigating to PDF viewer with URL:', pdfUrl);
      navigation.navigate('PdfviewPage', {
        pdfUrl: pdfUrl,
        onGoBack: () => navigation.navigate('IncidentReport'),
      });
    } catch (error) {
      console.error('Error navigating to PDF viewer:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to open the PDF. Please try again.',
        position: 'top',
      });
    }
  };

  const handleArchiveReport = async () => {
    if (!selectedReport) return;

    setDropdownVisible(false);
    setSyncMessage(`Archiving Incident Report ${selectedReport.incidentNumber || 'N/A'}`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Archive via MongoDB API
      const response = await fetch(`${API_BASE_URL}/incident-reports/${userUid}/${selectedReport.id}/archive`, {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ isArchived: true }),
      });

      const result = await response.json();
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to archive report');
      }

      const updatedActiveReports = incidentReports.filter(
        (report) => report.id !== selectedReport.id
      );
      const updatedArchivedReports = [
        ...archivedReports,
        { ...selectedReport, isArchived: true },
      ];

      // Sort both lists by creation date (newest first)
      const sortedActive = updatedActiveReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      const sortedArchived = updatedArchivedReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });

      setIncidentReports(sortedActive);
      setFilteredReports(sortedActive);
      setArchivedReports(sortedArchived);
      setFilteredArchivedReports(sortedArchived);

      const allReports = [...updatedActiveReports, ...updatedArchivedReports];
      await AsyncStorage.setItem('incidentReports', JSON.stringify(allReports));

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `Incident Report ${selectedReport.incidentNumber || 'N/A'} archived successfully.`,
        position: 'top',
      });
    } catch (error) {
      console.error('Error archiving incident report:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to archive the incident report. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setSelectedReport(null);
    }
  };

  const handleUnarchiveReport = async (report) => {
    setSyncMessage(`Unarchiving Incident Report ${report.incidentNumber || 'N/A'}`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Unarchive via MongoDB API
      const response = await fetch(`${API_BASE_URL}/incident-reports/${userUid}/${report.id}/archive`, {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ isArchived: false }),
      });

      const result = await response.json();
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to unarchive report');
      }

      const updatedArchivedReports = archivedReports.filter(
        (r) => r.id !== report.id
      );
      const updatedActiveReports = [
        ...incidentReports,
        { ...report, isArchived: false },
      ];

      // Sort both lists by creation date (newest first)
      const sortedActive = updatedActiveReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });
      
      const sortedArchived = updatedArchivedReports.sort((a, b) => {
        const dateA = new Date(a.createdAt || a.incidentDate || 0);
        const dateB = new Date(b.createdAt || b.incidentDate || 0);
        return dateB.getTime() - dateA.getTime();
      });

      setArchivedReports(sortedArchived);
      setFilteredArchivedReports(sortedArchived);
      setIncidentReports(sortedActive);
      setFilteredReports(sortedActive);

      const allReports = [...updatedActiveReports, ...updatedArchivedReports];
      await AsyncStorage.setItem('incidentReports', JSON.stringify(allReports));

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `Incident Report ${report.incidentNumber || 'N/A'} unarchived successfully.`,
        position: 'top',
      });
    } catch (error) {
      console.error('Error unarchiving incident report:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to unarchive the incident report. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const handleDeleteReport = async () => {
    if (!reportToDelete) return;

    if (!hasDeletePermission) {
      Toast.show({
        type: 'error',
        text1: 'Permission Denied',
        text2: 'You do not have permission to delete incident reports.',
        position: 'top',
      });
      setConfirmDeleteVisible(false);
      setReportToDelete(null);
      return;
    }

    setConfirmDeleteVisible(false);
    setDropdownVisible(false);
    setSyncMessage(`Deleting Incident Report ${reportToDelete.incidentNumber || 'N/A'}`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Delete via MongoDB API
      const response = await fetch(`${API_BASE_URL}/incident-reports/${userUid}/${reportToDelete.id}`, {
        method: 'DELETE',
      });

      const result = await response.json();
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to delete report');
      }

      let updatedActiveReports = incidentReports;
      let updatedArchivedReports = archivedReports;
      if (reportToDelete.isArchived) {
        updatedArchivedReports = archivedReports.filter(
          (report) => report.id !== reportToDelete.id
        );
      } else {
        updatedActiveReports = incidentReports.filter(
          (report) => report.id !== reportToDelete.id
        );
      }
      setIncidentReports(updatedActiveReports);
      setFilteredReports(updatedActiveReports);
      setArchivedReports(updatedArchivedReports);
      setFilteredArchivedReports(updatedArchivedReports);

      const allReports = [...updatedActiveReports, ...updatedArchivedReports];
      await AsyncStorage.setItem('incidentReports', JSON.stringify(allReports));

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `Incident Report ${reportToDelete.incidentNumber || 'N/A'} deleted successfully.`,
        position: 'top',
      });
    } catch (error) {
      console.error('Error deleting incident report:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to delete the incident report. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setReportToDelete(null);
    }
  };

  const handleUploadToCloud = async (report) => {
    if (!report) return;

    setDropdownVisible(false);
    setSyncMessage(`Uploading Incident Report ${report.incidentNumber || 'N/A'} to Cloud`);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const isConnected = await NetInfo.fetch();
      if (!isConnected.isConnected) {
        Toast.show({
          type: 'error',
          text1: 'No Internet Connection',
          text2: 'Please check your internet connection and try again.',
          position: 'top',
        });
        return;
      }

      // Upload the report to MongoDB
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      const response = await fetch(`${API_BASE_URL}/incident-reports/${userUid}/${report.id}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(report),
      });

      const result = await response.json();
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to upload report');
      }

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Incident report uploaded to cloud successfully.',
        position: 'top',
      });
    } catch (error) {
      console.error('Error uploading incident report to cloud:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to upload incident report to cloud. Please try again.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const formatDate = (dateValue) => {
    if (!dateValue) {
      console.warn('Date is missing for report:', dateValue);
      return 'N/A';
    }

    let dateObj;
    if (typeof dateValue === 'number') {
      dateObj = new Date(dateValue);
    } else if (typeof dateValue === 'string') {
      dateObj = new Date(dateValue);
    } else {
      console.warn('Invalid date format:', dateValue);
      return 'N/A';
    }

    if (isNaN(dateObj.getTime())) {
      console.warn('Invalid date:', dateValue);
      return 'N/A';
    }

    try {
      // Try with timeZone, but fallback if error
      return dateObj.toLocaleString('en-IN', {
        day: '2-digit',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        hour12: true,
        timeZone: 'Asia/Kolkata',
      });
    } catch (e) {
      // Fallback: no timeZone
      return dateObj.toLocaleString('en-IN', {
        day: '2-digit',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        hour12: true,
      });
    }
  };

  const getSelectedIncidentsList = (selectedIncidents) => {
    if (!selectedIncidents || typeof selectedIncidents !== 'object') {
      return 'N/A';
    }
    const selected = Object.keys(selectedIncidents).filter(
      (key) => selectedIncidents[key]
    );
    return selected.length > 0 ? selected.join(', ') : 'None';
  };

  const canPerformActionOnReport = (report) => {
    if (!report) return false; // Prevents crash if report is null/undefined
    // If user has onlyMineView permission, they can only perform actions on their own reports
    if (hasOnlyMineView && currentUserUid) {
      return report.adminUid === currentUserUid;
    }
    // If no onlyMineView restriction, user can perform actions on any report they have permission for
    return true;
  };

  const renderIncidentReport = ({ item }) => {
    const hasAnyPermission = hasViewPermission || hasEditPermission || hasDeletePermission;
    const canPerformAction = canPerformActionOnReport(item);
    
    return (
      <View style={styles.reportCard}>
        <View style={styles.cardHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.cardTitle}>{String(item.incidentNumber || 'N/A')}</Text>
            
            {/* Offline Sync Pending Badge */}
            {item.offlineSyncPending && (
              <View style={styles.offlineBadge}>
                <MaterialIcons name="cloud-upload" size={12} color="#fff" />
                <Text style={styles.offlineBadgeText}>SYNC</Text>
              </View>
            )}
          </View>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Date</Text>
          <Text style={styles.cardValue}>{String(formatDate(item.incidentDate))}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Area</Text>
          <Text style={styles.cardValue}>{String(item.incidentArea || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Severity</Text>
          <Text style={styles.cardValue}>{String(item.severity || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Involves Equipment</Text>
          <Text style={styles.cardValue}>{item.involvesEquipment ? 'Yes' : 'No'}</Text>
        </View>
        {item.involvesEquipment && (
          <>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment #</Text>
              <Text style={styles.cardValue}>{String(item.vehicleId || 'N/A')}</Text>
            </View>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment Name</Text>
              <Text style={styles.cardValue}>{String(item.vehicleName || 'N/A')}</Text>
            </View>
          </>
        )}
        {/* Created At row with 3-dots menu at the right */}
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Created At</Text>
          <View style={{ flexDirection: 'row', alignItems: 'center', flex: 1, justifyContent: 'space-between' }}>
            <Text style={styles.cardValue}>{String(formatDate(item.createdAt))}</Text>
            {hasAnyPermission && canPerformAction && (
              <TouchableOpacity
                onPress={(event) => {
                  setSelectedReport(item);
                  const { pageX, pageY } = event.nativeEvent;
                  const dropdownHeight = 280;
                  const screenHeight = Dimensions.get('window').height;
                  let top = pageY + 2;
                  let openUp = false;
                  if (pageY + 2 + dropdownHeight + 20 > screenHeight) {
                    top = pageY - dropdownHeight - 2;
                    openUp = true;
                  }
                  setDropdownPosition({ x: pageX - 200 + 24, y: top, openUp });
                  setDropdownVisible(true);
                }}
                style={styles.menuButton}
              >
                <MaterialIcons name="more-vert" size={24} color="#333" />
              </TouchableOpacity>
            )}
          </View>
        </View>
      </View>
    );
  };

  const renderArchivedReport = ({ item }) => {
    return (
      <View style={[styles.reportCard, { backgroundColor: '#F5F5F5' }]}>
        <View style={styles.cardHeader}>
          <Text style={styles.cardTitle}>{String(item.incidentNumber || 'N/A')}</Text>
          <TouchableOpacity
            onPress={() => handleUnarchiveReport(item)}
            style={styles.menuButton}
          >
            <MaterialIcons name="unarchive" size={24} color="#3481BC" />
          </TouchableOpacity>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Date</Text>
          <Text style={styles.cardValue}>{String(formatDate(item.incidentDate))}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Country</Text>
          <Text style={styles.cardValue}>{String(item.country || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Project</Text>
          <Text style={styles.cardValue}>{String(item.project || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Area</Text>
          <Text style={styles.cardValue}>{String(item.incidentArea || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Severity</Text>
          <Text style={styles.cardValue}>{String(item.severity || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Incident Category</Text>
          <Text style={styles.cardValue}>{String(item.incidentCategory || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Types of Event</Text>
          <Text style={styles.cardValue}>{String(getSelectedIncidentsList(item.selectedIncidents))}</Text>
        </View>
        {item.selectedIncidents?.['Other'] && (
          <View style={styles.cardRow}>
            <Text style={styles.cardLabel}>Other Incident Note</Text>
            <Text style={styles.cardValue}>{String(item.otherIncidentNote || 'N/A')}</Text>
          </View>
        )}
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Involves Equipment</Text>
          <Text style={styles.cardValue}>{item.involvesEquipment ? 'Yes' : 'No'}</Text>
        </View>
        {item.involvesEquipment && (
          <>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment #</Text>
              <Text style={styles.cardValue}>{String(item.vehicleId || 'N/A')}</Text>
            </View>
            <View style={styles.cardRow}>
              <Text style={styles.cardLabel}>Equipment Name</Text>
              <Text style={styles.cardValue}>{String(item.vehicleName || 'N/A')}</Text>
            </View>
          </>
        )}
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Person Involved</Text>
          <Text style={styles.cardValue}>{String(item.personInvolved || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Coordinates</Text>
          <Text style={styles.cardValue}>{String(item.coordinates || 'N/A')}</Text>
        </View>
        <View style={styles.cardRow}>
          <Text style={styles.cardLabel}>Created At</Text>
          <Text style={styles.cardValue}>{String(formatDate(item.createdAt))}</Text>
        </View>
      </View>
    );
  };

  const getDropdownData = (report) => {
    if (!report) return [];
    const canPerformAction = canPerformActionOnReport(report);
    
    return [
      ...(hasViewPermission || hasEditPermission ? [{
        label: 'View Report',
        value: 'view',
        icon: 'picture-as-pdf',
        action: () => {
          if (selectedReport) {
            handleViewPDF(selectedReport); // Pass entire report object
          }
        },
      }] : []),
      {
        label: 'Upload to Cloud',
        value: 'upload',
        icon: 'cloud-upload',
        action: () => {
          if (selectedReport) {
            handleUploadToCloud(selectedReport);
          }
        },
      },
      ...(hasDeletePermission && canPerformAction ? [{
        label: 'Delete',
        value: 'delete',
        icon: 'delete',
        action: () => {
          if (selectedReport) {
            setReportToDelete(selectedReport);
            setConfirmDeleteVisible(true);
          }
        },
      }] : []),
    ];
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.reset({
              index: 0,
              routes: [
                { name: 'MainApp' },
              ],
            })}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Incident Reports</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity 
              style={[styles.iconButton, isSyncing && styles.syncingButton]} 
              onPress={handleCloudSync}
              disabled={isSyncing}
            >
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={[styles.refreshIcon, isSyncing && styles.syncingIcon]}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={async () => {
                // Reload permissions before showing modal
                console.log('🔄 Refreshing permissions from storage...');
                await fetchUserPermissions();
                await new Promise(resolve => setTimeout(resolve, 300)); // Wait for state update
                setPermissionsModalVisible(true);
              }}
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by title or type..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => {
              Toast.show({
                type: 'info',
                text1: 'Filter',
                text2: 'Filter functionality not implemented yet.',
                position: 'top',
              });
            }}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.archiveButton}
            onPress={() => setShowArchived(!showArchived)}
          >
            <MaterialIcons 
              name={showArchived ? "archive" : "unarchive"} 
              size={width * 0.06} 
              color="#fff" 
            />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      
      {/* Report Status Counter Cards */}
      <View style={styles.statusCounterContainer}>
        <TouchableOpacity 
          style={styles.statusCounterCard}
          onPress={async () => {
            try {
              const pendingReports = await OfflineIncidentHelper.getOfflineReports();
              const cachedReportsRaw = await AsyncStorage.getItem('incidentReports');
              const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
              
              // Get permissions from cached reports
              let storedPermissions = 'None';
              let storedCountries = 'None';
              let storedProjects = 'None';
              
              if (cachedReports.length > 0 && cachedReports[0].userPermissions) {
                storedPermissions = cachedReports[0].userPermissions.join(', ');
                storedCountries = cachedReports[0].userCountries?.join(', ') || 'None';
                storedProjects = cachedReports[0].userProjects?.length > 3 
                  ? `${cachedReports[0].userProjects.slice(0, 3).join(', ')}... (+${cachedReports[0].userProjects.length - 3} more)`
                  : cachedReports[0].userProjects?.join(', ') || 'None';
              } else if (userPermissions.incidentReportPermissions?.length > 0) {
                storedPermissions = userPermissions.incidentReportPermissions.join(', ');
                storedCountries = userPermissions.countries?.join(', ') || 'None';
                storedProjects = userPermissions.projects?.length > 3 
                  ? `${userPermissions.projects.slice(0, 3).join(', ')}... (+${userPermissions.projects.length - 3} more)`
                  : userPermissions.projects?.join(', ') || 'None';
              }
              
              Alert.alert(
                '📱 Total Offline Storage',
                `TOTAL Available Offline: ${offlineReportsCount} reports\n\n` +
                `BREAKDOWN:\n` +
                `✓ Cached from MongoDB: ${cachedReports.length}\n` +
                `✓ Pending Sync: ${pendingReports.length}\n\n` +
                `PERMISSIONS STORED:\n` +
                `✓ ${storedPermissions}\n\n` +
                `COUNTRIES:\n` +
                `✓ ${storedCountries}\n\n` +
                `PROJECTS:\n` +
                `✓ ${storedProjects}\n\n` +
                `All reports & PDFs accessible offline!`,
                [{ text: 'OK' }]
              );
            } catch (e) {
              console.error(e);
            }
          }}
        >
          <MaterialIcons name="cloud-off" size={24} color="#FF9800" />
          <View style={styles.statusCounterContent}>
            <Text style={styles.statusCounterLabel}>Offline Stored</Text>
            <Text style={styles.statusCounterValue}>{offlineReportsCount}</Text>
          </View>
        </TouchableOpacity>
        
      

        <TouchableOpacity 
          style={styles.statusCounterCard}
          onPress={async () => {
            try {
              Alert.alert(
                '📊 Total in MongoDB',
                `TOTAL INCIDENT REPORTS:\n` +
                `✓ All reports in database: ${totalMongoCount}\n\n` +
                `This count includes all incident reports across all users in MongoDB.\n\n` +
                `Updated when online.`,
                [{ text: 'OK' }]
              );
            } catch (e) {
              console.error(e);
            }
          }}
        >
          <MaterialIcons name="assessment" size={24} color="#9C27B0" />
          <View style={styles.statusCounterContent}>
            <Text style={styles.statusCounterLabel}>Total in DB</Text>
            <Text style={styles.statusCounterValue}>{totalMongoCount}</Text>
          </View>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.statusCounterCard}
          onPress={async () => {
            try {
              const keys = await AsyncStorage.getAllKeys();
              const incidentKeys = keys.filter(key => 
                key.includes('incident') || 
                key.includes('Incident') || 
                key.includes('offline')
              );
              
              // Get permissions info
              const cachedIncidentUserInfo = await AsyncStorage.getItem('incidentModuleUserInfo');
              let permissionsStored = 'No';
              if (cachedIncidentUserInfo) {
                const userInfoData = JSON.parse(cachedIncidentUserInfo);
                permissionsStored = `Yes (${userInfoData.incidentReportPermissions?.length || 0} permissions)`;
              }
              
              Alert.alert(
                '💾 Storage Details',
                `STORAGE SIZE:\n` +
                `✓ Total Size: ${localStorageSize}\n` +
                `✓ Storage Keys: ${incidentKeys.length}\n\n` +
                `LOCATION:\n` +
                `✓ AsyncStorage (Shared)\n` +
                `✓ Type: Persistent\n\n` +
                `PERMISSIONS STORED:\n` +
                `✓ ${permissionsStored}\n\n` +
                `DATA INCLUDES:\n` +
                `• Incident reports\n` +
                `• User permissions\n` +
                `• Countries & Projects\n` +
                `• PDFs (local)\n` +
                `• Images (local)\n\n` +
                `All data accessible offline!`,
                [{ text: 'OK' }]
              );
            } catch (e) {
              console.error(e);
            }
          }}
        >
          <MaterialIcons name="storage" size={24} color="#2196F3" />
          <View style={styles.statusCounterContent}>
            <Text style={styles.statusCounterLabel}>Storage</Text>
            <Text style={styles.statusCounterValue}>{localStorageSize}</Text>
          </View>
        </TouchableOpacity>
      </View>
      
      <View style={styles.mainContent}>
        {filteredReports.length === 0 && (!showArchived || filteredArchivedReports.length === 0) ? (
          <>
            <Text style={styles.defaultText}>You don't have any incident reports</Text>
            <View style={styles.questionContainer}>
              <MaterialIcons name="help-outline" size={width * 0.05} color="#333" />
              <Text style={styles.questionText}>How To Create Incident Report?</Text>
            </View>
          </>
        ) : (
          <>
            <FlatList
              data={filteredReports}
              renderItem={renderIncidentReport}
              keyExtractor={(item) => `active-${item.id}`}
              contentContainerStyle={styles.reportList}
              refreshing={isRefreshing}
              onRefresh={handleRefresh}
            />
            {showArchived && filteredArchivedReports.length > 0 && (
              <>
                <Text style={styles.sectionHeader}>Archived Reports</Text>
                <FlatList
                  data={filteredArchivedReports}
                  renderItem={renderArchivedReport}
                  keyExtractor={(item) => `archived-${item.id}`}
                  contentContainerStyle={styles.reportList}
                />
              </>
            )}
          </>
        )}
        <Modal
          visible={dropdownVisible}
          transparent={true}
          animationType="none"
          onRequestClose={() => setDropdownVisible(false)}
        >
          <TouchableOpacity
            style={{ flex: 1, backgroundColor: 'transparent' }}
            onPress={() => setDropdownVisible(false)}
            activeOpacity={1}
          >
            <View
              style={[
                styles.dropdownContainer,
                {
                  position: 'absolute',
                  top: dropdownPosition.y,
                  left: dropdownPosition.x,
                  width: 220,
                  zIndex: 9999,
                },
              ]}
            >
              {selectedReport && getDropdownData(selectedReport).map((option, idx) => (
                <TouchableOpacity
                  key={idx}
                  style={styles.dropdownItem}
                  onPress={() => {
                    option.action();
                    if (option.label !== 'Delete') {
                      setDropdownVisible(false);
                    }
                  }}
                  activeOpacity={0.8}
                >
                  <MaterialIcons
                    name={option.icon}
                    size={22}
                    color={option.value === 'delete' ? '#e74c3c' : '#015185'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                    ]}
                  >
                    {option.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
          </TouchableOpacity>
        </Modal>
        <Modal
          visible={confirmDeleteVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setConfirmDeleteVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.confirmModalContent}>
              <Text style={styles.confirmModalTitle}>Confirm Deletion</Text>
              <Text style={styles.confirmModalText}>
                Are you sure you want to delete the incident report{' '}
                <Text style={styles.boldText}>
                  {reportToDelete?.incidentNumber || 'N/A'}
                </Text>
                ? This action cannot be undone.
              </Text>
              <View style={styles.modalButtons}>
                <TouchableOpacity
                  style={[styles.modalButton, styles.cancelButton]}
                  onPress={() => {
                    setConfirmDeleteVisible(false);
                    setReportToDelete(null);
                    setDropdownVisible(false);
                  }}
                >
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.modalButton, styles.deleteButton]}
                  onPress={handleDeleteReport}
                >
                  <Text style={[styles.modalButtonText, { color: '#fff' }]}>Delete</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
        <Modal
          visible={permissionsModalVisible}
          transparent={true}
          animationType="slide"
          onRequestClose={() => setPermissionsModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.modalContent}>
              <Text style={styles.modalTitle}>User Permissions</Text>
              <View style={styles.userInfoItem}>
                <Text style={styles.userInfoLabel}>Countries:</Text>
                <Text style={styles.userInfoValue}>
                  {userPermissions?.countries && Array.isArray(userPermissions.countries) && userPermissions.countries.length > 0 
                    ? userPermissions.countries.join(', ') 
                    : 'N/A'}
                </Text>
              </View>
              <View style={styles.userInfoItem}>
                <Text style={styles.userInfoLabel}>Projects:</Text>
                <Text style={styles.userInfoValue}>
                  {userPermissions?.projects && Array.isArray(userPermissions.projects) && userPermissions.projects.length > 0 
                    ? userPermissions.projects.join(', ') 
                    : 'N/A'}
                </Text>
              </View>
              <View style={styles.userInfoItem}>
                <Text style={styles.userInfoLabel}>Incident Report Permissions:</Text>
                <Text style={styles.userInfoValue}>
                  {userPermissions?.incidentReportPermissions && Array.isArray(userPermissions.incidentReportPermissions) && userPermissions.incidentReportPermissions.length > 0 
                    ? userPermissions.incidentReportPermissions.join(', ') 
                    : 'None'}
                </Text>
              </View>
              <View style={styles.userInfoItem}>
                <Text style={styles.userInfoLabel}>Permission Details:</Text>
                <Text style={styles.userInfoValue}>
                  Create: {hasCreatePermission ? 'Yes' : 'No'}{'\n'}
                  View: {hasViewPermission ? 'Yes' : 'No'}{'\n'}
                  Edit: {hasEditPermission ? 'Yes' : 'No'}{'\n'}
                  Delete: {hasDeletePermission ? 'Yes' : 'No'}{'\n'}
                  Only Mine View: {hasOnlyMineView ? 'Yes' : 'No'}
                </Text>
              </View>
              
              {/* Debug: Show Filtering Details */}
              <TouchableOpacity 
                style={[styles.modalCloseButton, { backgroundColor: '#FF9800', paddingHorizontal: 20, paddingVertical: 10, borderRadius: 6, marginBottom: 10 }]}
                onPress={async () => {
                  try {
                    // Get all reports from cache
                    const cachedRaw = await AsyncStorage.getItem('incidentReports');
                    const allCachedReports = cachedRaw ? JSON.parse(cachedRaw) : [];
                    
                    // Count how many reports match each filter
                    let countryMatches = 0;
                    let projectMatches = 0;
                    let bothMatch = 0;
                    let reportsByCountry = {};
                    let reportsByProject = {};
                    
                    allCachedReports.forEach(report => {
                      // Count by country
                      if (!reportsByCountry[report.country]) {
                        reportsByCountry[report.country] = 0;
                      }
                      reportsByCountry[report.country]++;
                      
                      // Count by project
                      if (!reportsByProject[report.project]) {
                        reportsByProject[report.project] = 0;
                      }
                      reportsByProject[report.project]++;
                      
                      // Check if matches user's permissions
                      const matchesCountry = !userPermissions.countries || userPermissions.countries.length === 0 || userPermissions.countries.includes(report.country);
                      const matchesProject = !userPermissions.projects || userPermissions.projects.length === 0 || userPermissions.projects.includes(report.project);
                      
                      if (matchesCountry) countryMatches++;
                      if (matchesProject) projectMatches++;
                      if (matchesCountry && matchesProject) bothMatch++;
                    });
                    
                    const countriesList = Object.entries(reportsByCountry).map(([c, count]) => `${c}: ${count}`).join('\n');
                    const projectsList = Object.entries(reportsByProject).map(([p, count]) => `${p}: ${count}`).join('\n');
                    
                    Alert.alert(
                      '🔍 Filter Debug Info',
                      `TOTAL REPORTS IN DB: ${allCachedReports.length}\n\n` +
                      `YOUR PERMISSIONS:\n` +
                      `Countries: ${userPermissions.countries?.join(', ') || 'ALL'}\n` +
                      `Projects: ${userPermissions.projects?.slice(0, 3).join(', ') || 'ALL'}${userPermissions.projects?.length > 3 ? '...' : ''}\n` +
                      `OnlyMineView: ${hasOnlyMineView ? 'YES' : 'NO'}\n\n` +
                      `REPORTS BY COUNTRY:\n${countriesList}\n\n` +
                      `REPORTS BY PROJECT:\n${projectsList}\n\n` +
                      `FILTER RESULTS:\n` +
                      `✓ Country matches: ${countryMatches}\n` +
                      `✓ Project matches: ${projectMatches}\n` +
                      `✓ BOTH match: ${bothMatch}\n\n` +
                      `SHOWING: ${filteredReports.length} reports`,
                      [{ text: 'OK' }]
                    );
                  } catch (e) {
                    console.error('Filter debug error:', e);
                    Alert.alert('Error', e.message);
                  }
                }}
              >
                <Text style={[styles.modalCloseText, { color: '#fff' }]}>🔍 Show Filter Debug</Text>
              </TouchableOpacity>
              
              {/* Debug: Check Storage Button */}
              <TouchableOpacity 
                style={[styles.modalCloseButton, { backgroundColor: '#ffd205', paddingHorizontal: 20, paddingVertical: 10, borderRadius: 6, marginBottom: 10 }]}
                onPress={async () => {
                  try {
                    console.log('🔍 Checking AsyncStorage for incident permissions...');
                    
                    const keys = await AsyncStorage.getAllKeys();
                    const relevantKeys = keys.filter(k => 
                      k.includes('incident') || 
                      k.includes('Incident') || 
                      k.includes('permission') ||
                      k.includes('userData')
                    );
                    
                    console.log('📦 Relevant storage keys:', relevantKeys);
                    
                    // Check each key
                    for (const key of relevantKeys) {
                      const value = await AsyncStorage.getItem(key);
                      if (value) {
                        try {
                          const parsed = JSON.parse(value);
                          if (parsed.userPermissions || parsed.incidentReportPermissions) {
                            console.log(`✅ Found permissions in ${key}:`, parsed.userPermissions || parsed.incidentReportPermissions);
                          }
                          if (parsed.length && parsed[0]?.userPermissions) {
                            console.log(`✅ Found permissions in first report of ${key}:`, parsed[0].userPermissions);
                          }
                        } catch (e) {
                          // Not JSON
                        }
                      }
                    }
                    
                    // Reload permissions
                    await fetchUserPermissions();
                    await new Promise(resolve => setTimeout(resolve, 500));
                    
                    Alert.alert(
                      '🔍 Storage Check',
                      `Checked ${relevantKeys.length} storage keys.\n\nSee console for details.\n\nPermissions reloaded!`,
                      [{ text: 'OK' }]
                    );
                  } catch (e) {
                    console.error('Storage check error:', e);
                  }
                }}
              >
                <Text style={[styles.modalCloseText, { color: '#333' }]}>🔍 Check Storage & Reload</Text>
              </TouchableOpacity>
              
              <TouchableOpacity style={styles.modalCloseButton} onPress={() => setPermissionsModalVisible(false)}>
                <Text style={styles.modalCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>
        <Modal visible={syncModalVisible} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>{syncMessage}</Text>
              </View>
            </View>
          </View>
        </Modal>
      </View>
      {hasCreatePermission && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('CreateIncidentReport')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Create Incident Report</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  iconContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  filterButton: {
    padding: width * 0.015,
    marginRight: width * 0.015,
  },
  archiveButton: {
    padding: width * 0.015,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  sectionHeader: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#3481BC',
    marginVertical: width * 0.03,
  },
  sectionHeader11: {
    // Empty style for consistency
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  reportCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    marginBottom: 16,
    padding: 0,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    overflow: 'visible',
    elevation: 2,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 12,
    borderBottomWidth: 1,
    borderColor: '#f0f0f0',
  },
  cardTitle: {
    fontSize: 20,
    color: '#3481BC',
    fontWeight: 'bold',
  },
  cloudIconContainer: {
    position: 'relative',
    alignItems: 'center',
    justifyContent: 'center',
  },
  tickContainer: {
    position: 'absolute',
    top: '50%',
    left: '50%',
    transform: [{ translateX: -8 }, { translateY: -8 }],
  },
  menuButton: {
    padding: 4,
  },
  cardRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderColor: '#f0f0f0',
  },
  cardLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    flex: 1,
  },
  cardValue: {
    fontSize: 16,
    color: '#333',
    flex: 1,
    textAlign: 'left',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  confirmModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  confirmModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
  },
  confirmModalText: {
    fontSize: 16,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  boldText: {
    fontWeight: 'bold',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 10,
    alignItems: 'center',
    borderRadius: 5,
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#f0f0f0',
  },
  deleteButton: {
    backgroundColor: '#FF3A3A',
  },
  modalButtonText: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  permissionsModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  permissionsModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: 15,
  },
  permissionsSection: {
    width: '100%',
    marginBottom: 15,
  },
  permissionsLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  permissionsValue: {
    fontSize: 16,
    color: '#333',
    textAlign: 'left',
  },
  noReportsContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 40,
  },
  permissionNote: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 8,
    fontStyle: 'italic',
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  questionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginTop: width * 0.02,
  },
  questionText: {
    fontSize: width * 0.04,
    color: '#666',
    marginLeft: width * 0.02,
  },
  reportList: {
    paddingBottom: width * 0.02,
  },
  // Status Counter Cards
  statusCounterContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.03,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  statusCounterCard: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F5F5F5',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.025,
    marginHorizontal: width * 0.01,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  statusCounterContent: {
    marginLeft: width * 0.02,
    flex: 1,
  },
  statusCounterLabel: {
    fontSize: width * 0.028,
    color: '#666',
    marginBottom: 2,
  },
  statusCounterValue: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
  },
  // Offline Badge
  offlineBadge: {
    marginLeft: width * 0.025,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.008,
    borderRadius: width * 0.01,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ff9800',
  },
  offlineBadgeText: {
    fontSize: width * 0.025,
    color: '#fff',
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  // Syncing states
  syncingButton: {
    opacity: 0.7,
  },
  syncingIcon: {
    opacity: 0.8,
  },
  // Modal styles to match DVIR
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  // UserInfo styles to match DVIR modal
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.015,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
    width: '60%',
  },
});

export default IncidentReport;