import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  ScrollView,
  Dimensions,
  Animated,
  Modal,
  InteractionManager,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { runOnJS } from 'react-native-reanimated';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig.js';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig.js';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, onClose, message = 'loading KPI Data...' }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const Inspection_Kpi_Home = () => {
  const navigation = useNavigation();
  const [selectedTab, setSelectedTab] = useState('Today');
  const tabs = ['Today', '7 Days', '30 Days', '3 Months'];
  const [isLoading, setIsLoading] = useState(false);
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    inspectionPermissions: [],
    employeeNumber: '',
    name: '',
    userUid: '',
  });
  const [allReports, setAllReports] = useState([]);
  const isMounted = useRef(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [hseReportCount, setHseReportCount] = useState(0);
  const [inspectionReportCount, setInspectionReportCount] = useState(0);

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      console.log('🌐 Network status:', {
        isConnected: netInfo.isConnected,
        type: netInfo.type,
        isInternetReachable: netInfo.isInternetReachable
      });
      return netInfo.isConnected && netInfo.isInternetReachable;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const fetchUserInfo = async () => {
    try {
      // Get user data from AsyncStorage (set during login)
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return null;
      }
      const user = JSON.parse(userData);

      let userUid = await AsyncStorage.getItem('userUid');
      let userDataFromStorage = null;

      if (userUid) {
        // Use the user data already parsed from AsyncStorage
        userDataFromStorage = user;
      }

      if (!userDataFromStorage) {
        // Use the user data already parsed from AsyncStorage
        userDataFromStorage = user;
        if (userDataFromStorage && userDataFromStorage.userUid) {
          userUid = userDataFromStorage.userUid;
          await AsyncStorage.setItem('userUid', userUid);
        } else {
          userUid = user.email; // Fallback to email if no userUid
          await AsyncStorage.setItem('userUid', userUid);
        }
      }

      if (!userDataFromStorage) {
        showToast('error', 'Error', 'User data not found.');
        return null;
      }

      const inspectionModule = userDataFromStorage.modules?.find(
        (module) => module.module === 'inspectionReport'
      ) || { permissions: [] };

      const userInfoData = {
        countries: Array.isArray(userDataFromStorage.countries) ? userDataFromStorage.countries : [],
        projects: Array.isArray(userDataFromStorage.projects) ? userDataFromStorage.projects : [],
        inspectionPermissions: Array.isArray(inspectionModule.permissions) ? inspectionModule.permissions : [],
        employeeNumber: userDataFromStorage.employeeNumber || '',
        name: userDataFromStorage.name || '',
        userUid: userUid || user._id || user.email,
      };

      await AsyncStorage.setItem('userInfo', JSON.stringify(userInfoData));
      if (isMounted.current) {
        setUserInfo(userInfoData);
      }
      return userInfoData;
    } catch (error) {
      console.error('Error fetching user info:', error);
      showToast('error', 'Error', 'Failed to fetch user permissions.');
      return null;
    }
  };

  // Background threading for HSE report counting - MongoDB version
  const loadHSEReportCount = useCallback(async () => {
    try {
      // TODO: Implement HSE MongoDB API endpoint
      // For now, setting to 0 until HSE MongoDB migration is complete
      if (isMounted.current) {
        setHseReportCount(0);
      }
      console.log('⚠️ HSE report count from MongoDB not yet implemented');
    } catch (error) {
      console.error('Error loading HSE report count:', error);
    }
  }, []);

  // Background threading for inspection report counting - MongoDB version
  const loadInspectionReportCount = useCallback(async () => {
    try {
      console.log('🔢 Loading total inspection report count from MongoDB...');
      const response = await InspectionAPI.getTotalCount();
      
      if (response.success && response.total !== undefined) {
        const count = response.total;
        console.log('✅ Total inspection reports in MongoDB:', count);
        
        if (isMounted.current) {
          setInspectionReportCount(count);
        }
        
        return count;
      } else {
        console.log('⚠️ Failed to get count from MongoDB, trying fallback...');
        // Fallback to getAllReports
        const fallbackResponse = await InspectionAPI.getAllReports({ page: 1, limit: 1 });
        if (fallbackResponse.success && fallbackResponse.pagination) {
          const count = fallbackResponse.pagination.total || 0;
          console.log('✅ Total inspection reports (fallback):', count);
          
          if (isMounted.current) {
            setInspectionReportCount(count);
          }
          
          return count;
        }
        return 0;
      }
    } catch (error) {
      console.error('Error loading inspection report count:', error);
      return 0;
    }
  }, []);

  // Background threading for data processing - MongoDB version
  const processDataInBackground = useCallback(async (userInfoData, isOnline) => {
    return new Promise((resolve, reject) => {
      // Use InteractionManager to run after interactions are complete
      InteractionManager.runAfterInteractions(async () => {
        try {
          let reportsArray = [];
          
          if (isOnline) {
            console.log('🌐 Background processing - fetching data from MongoDB...');
            
            // Process report counts in background
            setTimeout(() => {
              loadHSEReportCount();
              loadInspectionReportCount();
            }, 100);
            
            // Fetch all inspection reports from MongoDB based on permissions
            let response;
            if (userInfoData.inspectionPermissions.includes('onlyMineView')) {
              // If user has onlyMineView permission, show only their own reports
              console.log('🔒 User has onlyMineView permission - loading only own reports');
              response = await InspectionAPI.getInspections(userInfoData.userUid, { page: 1, limit: 10000 });
            } else if (userInfoData.inspectionPermissions.includes('view')) {
              // If user has view permission, try to show all reports
              console.log('🌐 User has view permission - attempting to load all reports');
              
              try {
                response = await InspectionAPI.getAllReports({ page: 1, limit: 10000 });
                
                // If getAllReports returns empty, try fetching from specific users
                if (!response.success || !response.data || response.data.length === 0) {
                  console.log('🔄 getAllReports returned empty, trying alternative...');
                  response = await InspectionAPI.getInspections(userInfoData.userUid, { page: 1, limit: 10000 });
                }
              } catch (error) {
                console.log('❌ Error loading all reports, falling back to user reports:', error);
                response = await InspectionAPI.getInspections(userInfoData.userUid, { page: 1, limit: 10000 });
              }
            } else {
              console.log('❌ User has no view permissions');
              response = { success: true, data: [] };
            }

            if (response.success && response.data) {
              reportsArray = response.data.map(report => ({
                ...report,
                id: report._id || report.id,
                adminId: report.userId || report.createdBy,
                createdAt: report.createdAt || new Date().toISOString(),
              }));
            }
            
            console.log('✅ Background - Loaded inspection reports from MongoDB:', reportsArray.length);

            // Fetch all job cards from MongoDB
            try {
              console.log('🔄 Fetching job cards from MongoDB...');
              const jobCardsResponse = await JobCardAPI.getAllJobCards();
              let allWorkOrders = [];
              
              if (jobCardsResponse.success && jobCardsResponse.data) {
                allWorkOrders = jobCardsResponse.data.map(jobCard => ({
                  ...jobCard,
                  id: jobCard._id || jobCard.id,
                  adminId: jobCard.adminUid || jobCard.userId,
                  createdAt: jobCard.createdAt || new Date().toISOString(),
                  requestNumber: jobCard.requestNumber || jobCard.title,
                  status: jobCard.status || 'Unknown',
                }));
                console.log('✅ Loaded job cards from MongoDB:', allWorkOrders.length);
              }

              // Corrective actions are now embedded in inspection reports
              console.log('🔍 Extracting corrective actions from reports...');
              
              // Merge job card and corrective action info into reports
              reportsArray = reportsArray.map((report) => {
                const workOrder = allWorkOrders.find(
                  (wo) => wo.linkInspection?.id === report.id || 
                          wo.linkInspection?._id === report.id || 
                          wo.serviceScheduleId === report.id
                );
                const correctiveAction = report.correctiveActionData || null;
                
                return {
                  ...report,
                  workOrder: workOrder || null,
                  correctiveAction: correctiveAction,
                };
              });

              console.log('✅ Merged job cards and corrective actions into reports');
            } catch (jobCardError) {
              console.error('❌ Error fetching job cards:', jobCardError);
              // Continue with reports only - set corrective actions from embedded data
              reportsArray = reportsArray.map((report) => ({
                ...report,
                workOrder: null,
                correctiveAction: report.correctiveActionData || null,
              }));
            }

            // Filter reports by user permissions
            const beforeFilter = reportsArray.length;
            
            if (userInfoData.countries.length === 0 || userInfoData.projects.length === 0) {
              console.log('⚠️ User has no countries/projects permissions - showing ALL reports for debugging');
            } else {
              reportsArray = reportsArray.filter(
                (report) => {
                  if (!report || !report.step1) {
                    return false;
                  }
                  
                  const hasCountryPermission = userInfoData.countries.includes(report.step1?.country);
                  const hasProjectPermission = userInfoData.projects.includes(report.step1?.project);
                  
                  return hasCountryPermission && hasProjectPermission;
                }
              );
            }
            
            console.log('✅ Background - Filtered by permissions:', {
              before: beforeFilter,
              after: reportsArray.length
            });

            // Save with user-specific key
            try {
              await AsyncStorage.setItem(`inspectionReports_${userInfoData.userUid}`, JSON.stringify(reportsArray));
              console.log('💾 Background - Data saved to AsyncStorage for user:', userInfoData.userUid);
            } catch (storageError) {
              console.warn('⚠️ Failed to save data to AsyncStorage:', storageError);
            }
          } else {
            console.log('📱 Background - Offline mode - loading from cache...');
            try {
              const savedReports = await AsyncStorage.getItem(`inspectionReports_${userInfoData.userUid}`);
              reportsArray = savedReports ? JSON.parse(savedReports) : [];
              console.log('📱 Background - Loaded from cache:', reportsArray.length, 'reports');
              
              if (reportsArray.length === 0) {
                const genericReports = await AsyncStorage.getItem('inspectionReports');
                if (genericReports) {
                  reportsArray = JSON.parse(genericReports);
                  console.log('📱 Background - Loaded from generic cache:', reportsArray.length, 'reports');
                }
              }
            } catch (storageError) {
              console.warn('⚠️ Failed to load from AsyncStorage:', storageError);
              reportsArray = [];
            }
            
            // Filter by countries and projects only
            const beforeOfflineFilter = reportsArray.length;
            reportsArray = reportsArray.filter(
              (report) =>
                report?.step1?.country && userInfoData.countries.includes(report.step1.country) &&
                report?.step1?.project && userInfoData.projects.includes(report.step1.project)
            );
            
            console.log('📱 Background - Offline filtering result:', {
              before: beforeOfflineFilter,
              after: reportsArray.length
            });
          }

          resolve(reportsArray);
        } catch (error) {
          reject(error);
        }
      });
    });
  }, [loadHSEReportCount, loadInspectionReportCount]);

  const loadData = async () => {
    setIsLoading(true);
    let didTimeout = false;
    const timeoutId = setTimeout(() => {
      didTimeout = true;
      setIsLoading(false);
      showToast('error', 'Timeout', 'Loading KPI data took too long. Please check your internet connection.');
    }, 30000);
    
    try {
      console.log('🔄 Starting data load for KPI...');
      const userInfoData = await fetchUserInfo();
      if (!userInfoData) {
        console.log('❌ No user info data available');
        if (!didTimeout) setIsLoading(false);
        return;
      }
      console.log('✅ User info loaded:', { 
        userId: userInfoData.userUid, 
        countries: userInfoData.countries.length, 
        projects: userInfoData.projects.length 
      });
      
      const userId = userInfoData.userUid;
      if (!userId) {
        showToast('error', 'Error', 'User ID not found');
        if (!didTimeout) setIsLoading(false);
        return;
      }

      const isOnline = await checkNetworkStatus();
      
      if (!isOnline) {
        console.log('📱 No internet connection - loading from cache only');
        showToast('info', 'Offline Mode', 'Loading data from local storage...');
      }
      
      // Process data in background thread
      const reportsArray = await processDataInBackground(userInfoData, isOnline);

      if (isMounted.current && !didTimeout) {
        if (reportsArray && reportsArray.length > 0) {
          // Sort by date descending
          const sortedReports = reportsArray.sort((a, b) => {
            const dateA = new Date(a.createdAt);
            const dateB = new Date(b.createdAt);
            return dateB - dateA;
          });
          
          console.log('✅ Final reports loaded for KPI:', sortedReports.length);
          console.log('✅ Sample reports:', sortedReports.slice(0, 3).map(r => ({
            requestNumber: r.step1?.requestNumber,
            country: r.step1?.country,
            project: r.step1?.project,
            createdAt: r.createdAt,
            hasWorkOrder: !!r.workOrder,
            hasCorrectiveAction: !!r.correctiveAction
          })));
          
          setAllReports(sortedReports);
          setIsLoading(false);
        } else {
          setAllReports([]);
          setIsLoading(false);
          console.log('⚠️ No reports found for KPI');
        }
      }
    } catch (error) {
      console.error('❌ Error loading KPI data:', error);
      console.error('❌ Error details:', {
        message: error.message,
        code: error.code,
        stack: error.stack?.substring(0, 200)
      });
      
      // Show more specific error messages
      let errorMessage = 'Unknown error occurred';
      if (error.message.includes('timeout')) {
        errorMessage = 'Database connection timeout. Please check your internet connection and try again.';
      } else if (error.message.includes('network')) {
        errorMessage = 'Network error. Please check your internet connection.';
      } else if (error.message.includes('permission')) {
        errorMessage = 'Permission denied. Please contact administrator.';
      } else {
        errorMessage = `Database error: ${error.message || 'Unknown error'}`;
      }
      
      showToast('error', 'Error loading data', errorMessage);
      if (!didTimeout) setIsLoading(false);
    } finally {
      clearTimeout(timeoutId);
    }
  };

  useEffect(() => {
    // Clear any old cached data on mount to ensure fresh load
    const clearOldCache = async () => {
      try {
        // Remove generic key (old cache)
        await AsyncStorage.removeItem('inspectionReports');
      } catch (e) {
        console.warn('Could not clear old cache:', e);
      }
    };
    
    clearOldCache();
    
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected && isMounted.current) {
        loadData();
      }
    });

    loadData();

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, []);

  // Enhanced KPI calculation
  const calculateKPIs = () => {
    const now = new Date();
    const timeRanges = {
      Today: 1,
      '7 Days': 7,
      '30 Days': 30,
      '3 Months': 90,
    };
    const days = timeRanges[selectedTab];
    
    // Fix: Don't mutate the date object
    const cutoffDate = new Date();
    cutoffDate.setDate(cutoffDate.getDate() - days);
    cutoffDate.setHours(0, 0, 0, 0); // Start of day

    const filteredReports = allReports.filter((report) => {
      if (!report || !report.createdAt) return false;
      const reportDate = new Date(report.createdAt);
      if (isNaN(reportDate.getTime())) {
        console.warn('⚠️ Invalid date found:', report.createdAt, 'for report:', report.id);
        return false;
      }
      const isInRange = reportDate >= cutoffDate;
      if (!isInRange) {
        console.log('📅 Report filtered out:', report.step1?.requestNumber, 'Date:', report.createdAt, 'Cutoff:', cutoffDate.toISOString());
      }
      return isInRange;
    });

    // Deployed with BOTH job card AND corrective action
    const deployedWithBoth = filteredReports.filter(
      (report) => report.step3?.equipmentStatus === 'Deployed' && report.workOrder && report.correctiveAction
    );
    const deployedCount = deployedWithBoth.length;
    const deployedRequestNumbers = deployedWithBoth.map(r => r.step1?.requestNumber || r.id);

    // Inspection reports linked to job card
    const inspectionLinkedJobCard = filteredReports.filter(
      (report) => report.workOrder
    );
    const inspectionLinkedJobCardCount = inspectionLinkedJobCard.length;

    // Of those, how many have corrective action
    const inspectionLinkedJobCardWithCA = inspectionLinkedJobCard.filter(
      (report) => report.correctiveAction
    ).length;

    const inspectionCount = filteredReports.length;
    // --- Replace vehicleTestKitCount with total unique equipment used ---
    const equipmentNumbers = filteredReports
      .map(r => r.step1?.selectedEquipment?.equipmentNumber)
      .filter(Boolean);
    const uniqueEquipmentNumbers = Array.from(new Set(equipmentNumbers));
    const totalEquipmentUsed = uniqueEquipmentNumbers.length;
    // --- End replacement ---
    // const vehicleTestKitCount = filteredReports.filter(
    //   (report) => report.step1?.selectedEquipment?.equipmentType === 'Vehicle Test Kit'
    // ).length;
    const correctiveActionCount = filteredReports.filter(
      (report) => report.correctiveAction
    ).length;
    const breakdownCount = filteredReports.filter(
      (report) => report.step3?.equipmentStatus === 'Breakdown'
    ).length;
    const repairItemCount = filteredReports.filter(
      (report) => report.correctiveAction?.actionType === 'Repair'
    ).length;
    const replaceItemCount = filteredReports.filter(
      (report) => report.correctiveAction?.actionType === 'Replace'
    ).length;

    return {
      inspectionCount,
      // vehicleTestKitCount,
      totalEquipmentUsed, // <-- new
      correctiveActionCount,
      breakdownCount,
      repairItemCount,
      replaceItemCount,
      deployedCount,
      deployedRequestNumbers,
      inspectionLinkedJobCardCount,
      inspectionLinkedJobCardWithCA,
    };
  };

  const kpis = calculateKPIs();

  const renderDeployedActionButton = (count, requestNumbers) => (
    <View style={styles.actionButton}>
      <Text style={styles.actionButtonTitle}>Deployed (With Job Card & CA)</Text>
      <Text style={styles.actionButtonCount}>{count}</Text>
      {count > 0 && (
        <Text style={styles.actionButtonSubtitle} numberOfLines={2}>
          {requestNumbers.join(', ')}
        </Text>
      )}
    </View>
  );

  const renderActionButton = (title, count, onPress = null) => (
    <TouchableOpacity 
      style={styles.actionButton}
      onPress={onPress}
      activeOpacity={onPress ? 0.7 : 1}
    >
      <Text style={styles.actionButtonTitle}>{title}</Text>
      <Text style={styles.actionButtonCount}>{count}</Text>
    </TouchableOpacity>
  );

  const renderInspectionItem = (label, count, onPress = null) => (
    <TouchableOpacity 
      style={styles.inspectionItem}
      onPress={onPress}
      activeOpacity={onPress ? 0.7 : 1}
    >
      <Text style={styles.inspectionLabel}>{label}</Text>
      <Text style={styles.inspectionCount}>{count}</Text>
    </TouchableOpacity>
  );

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      
      {/* Header */}
      <View style={styles.headerContainer}>
        <View style={styles.header}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()} style={styles.backButton}>
              <MaterialIcons name="arrow-back" size={24} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Inspection KPIs</Text>
          </View>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <TouchableOpacity onPress={loadData} style={{ padding: 5 }}>
              <MaterialIcons name="refresh" size={24} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={() => setModalVisible(true)} style={{ padding: 5, marginLeft: 8 }}>
              <MaterialIcons name="lock" size={24} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </View>

      {/* Permission Modal */}
      <Modal visible={modalVisible} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Permissions</Text>
            <Text style={styles.modalLabel}>Inspection Report Permissions:</Text>
            <Text style={styles.modalValue}>{userInfo.inspectionPermissions && userInfo.inspectionPermissions.length > 0 ? userInfo.inspectionPermissions.join(', ') : 'None'}</Text>
            <Text style={styles.modalLabel}>Countries:</Text>
            <Text style={styles.modalValue}>{userInfo.countries && userInfo.countries.length > 0 ? userInfo.countries.join(', ') : 'None'}</Text>
            <Text style={styles.modalLabel}>Projects:</Text>
            <Text style={styles.modalValue}>{userInfo.projects && userInfo.projects.length > 0 ? userInfo.projects.join(', ') : 'None'}</Text>
            <TouchableOpacity style={styles.closeButton} onPress={() => setModalVisible(false)}>
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Main Content */}
      <ScrollView style={styles.mainContent}>
        {/* Date Filter Tabs */}
        <View style={styles.tabsContainer}>
          {tabs.map((tab) => (
            <TouchableOpacity
              key={tab}
              style={[styles.tab, selectedTab === tab && styles.selectedTab]}
              onPress={() => setSelectedTab(tab)}
            >
              <Text
                style={[styles.tabText, selectedTab === tab && styles.selectedTabText]}
              >
                {tab}
              </Text>
            </TouchableOpacity>
          ))}
        </View>

        {/* Action Buttons Grid */}
        <View style={styles.actionButtonsContainer}>
          {(() => {
            const cutoffDate = new Date();
            const timeRanges = {
              Today: 1,
              '7 Days': 7,
              '30 Days': 30,
              '3 Months': 90,
            };
            const days = timeRanges[selectedTab];
            cutoffDate.setDate(cutoffDate.getDate() - days);
            cutoffDate.setHours(0, 0, 0, 0);
            
            const filteredReports = allReports.filter((report) => {
              const reportDate = new Date(report.createdAt);
              return reportDate >= cutoffDate;
            });
            const deployedWithBoth = filteredReports.filter(
              report => report.correctiveAction && report.workOrder
            );
            return renderDeployedActionButton(
              deployedWithBoth.length,
              deployedWithBoth.map(r => r.step1?.requestNumber || r.id)
            );
          })()}
          {renderActionButton('Inspection Report', kpis.inspectionCount, () => navigation.navigate('Inspection_Report_Home'))}
          {renderActionButton('Breakdown', kpis.breakdownCount, () => navigation.navigate('Inspection_Report_Home'))}
          {renderActionButton('Repair Item', kpis.repairItemCount, () => navigation.navigate('Inspection_Report_Home'))}
          {renderActionButton('Replace Item', kpis.replaceItemCount, () => navigation.navigate('Inspection_Report_Home'))}
          
          {/* HSE Report Button */}
          <TouchableOpacity 
            style={styles.hseReportButton}
            onPress={() => navigation.navigate('View_inspection_KPIs', { type: 'hse' })}
          >
            <View style={styles.hseReportContent}>
              <MaterialIcons name="assignment" size={32} color="#015185" />
              <Text style={styles.hseReportTitle}>HSE Reports</Text>
              <Text style={styles.hseReportCount}>{hseReportCount}</Text>
              <Text style={styles.hseReportSubtitle}>View All HSE Reports</Text>
            </View>
          </TouchableOpacity>

          {/* Inspection Report Count Button */}
          <TouchableOpacity 
            style={styles.inspectionReportButton}
            onPress={() => navigation.navigate('View_inspection_KPIs', { type: 'inspection' })}
          >
            <View style={styles.inspectionReportContent}>
              <MaterialIcons name="assessment" size={32} color="#0055A5" />
              <Text style={styles.inspectionReportTitle}>Inspection Reports</Text>
              <Text style={styles.inspectionReportCount}>{inspectionReportCount}</Text>
              <Text style={styles.inspectionReportSubtitle}>Total Reports in System</Text>
            </View>
          </TouchableOpacity>
        </View>

        {/* List of deployed reports with both Job Card and Corrective Action */}
        <View style={{ marginVertical: 16, alignItems: 'center' }}>
          <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 15 }}>
            Deployed with Job Card & Corrective Action: {kpis.deployedCount}
          </Text>
          {kpis.deployedRequestNumbers.length > 0 && (
            <Text style={{ color: '#444', fontSize: 13, marginTop: 4, textAlign: 'center' }}>
              {kpis.deployedRequestNumbers.join(', ')}
            </Text>
          )}
        </View>

        {/* Inspection & Compliance Section */}
        <View style={styles.inspectionSection}>
          <Text style={styles.sectionTitle}>Inspection & Compliance</Text>
          {renderInspectionItem('Inspection', kpis.inspectionCount, () => navigation.navigate('Inspection_Report_Home'))}
          {renderInspectionItem('Total Equipment Used', kpis.totalEquipmentUsed)}
          {renderInspectionItem(
            'Inspection Linked Job Card',
            `${kpis.inspectionLinkedJobCardCount} (CA: ${kpis.inspectionLinkedJobCardWithCA})`,
            () => navigation.navigate('Inspection_Report_Home')
          )}
          {renderInspectionItem('Corrective Action', kpis.correctiveActionCount, () => navigation.navigate('Inspection_Report_Home'))}
        </View>

        {/* List all inspection reports and show if they have corrective action and job card */}
        <View style={{ marginHorizontal: 15, marginBottom: 20 }}>
          <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 15, marginBottom: 8 }}>
            Total Inspection Reports ({selectedTab}): {kpis.inspectionCount}
          </Text>
          <Text style={{ fontWeight: 'bold', color: '#2ecc40', fontSize: 15, marginBottom: 8 }}>
            Reports with both Corrective Action & Job Card: {kpis.deployedCount}
          </Text>
          {(() => {
            const cutoffDate = new Date();
            const timeRanges = {
              Today: 1,
              '7 Days': 7,
              '30 Days': 30,
              '3 Months': 90,
            };
            const days = timeRanges[selectedTab];
            cutoffDate.setDate(cutoffDate.getDate() - days);
            cutoffDate.setHours(0, 0, 0, 0);
            
            const filteredReports = allReports.filter((report) => {
              const reportDate = new Date(report.createdAt);
              return reportDate >= cutoffDate;
            });
            
            return (
              <>
                {filteredReports.length === 0 ? (
                  <Text style={{ color: '#888', fontSize: 13 }}>No inspection reports in this range.</Text>
                ) : (
                  filteredReports.map((report, idx) => {
                    const hasBoth = report.correctiveAction && report.workOrder;
                    return (
                      <View
                        key={report.id || idx}
                        style={{
                          marginBottom: 8,
                          padding: 8,
                          backgroundColor: '#F5F5F5',
                          borderRadius: 6,
                          borderWidth: hasBoth ? 2 : 0,
                          borderColor: hasBoth ? '#2ecc40' : 'transparent',
                          flexDirection: 'row',
                          alignItems: 'center',
                        }}
                      >
                        <View style={{ flex: 1 }}>
                          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                            <Text style={{ color: '#015185', fontWeight: 'bold', marginRight: 6 }}>
                              #{report.step1?.requestNumber || report.id}
                            </Text>
                            {hasBoth && (
                              <MaterialIcons name="check-circle" size={18} color="#2ecc40" style={{ marginRight: 6 }} />
                            )}
                          </View>
                          <Text style={{ color: report.correctiveAction ? '#2ecc40' : '#e74c3c', fontSize: 13 }}>
                            Corrective Action: {report.correctiveAction ? 'Yes' : 'No'}
                          </Text>
                          <Text style={{ color: report.workOrder ? '#0078D4' : '#e74c3c', fontSize: 13 }}>
                            Job Card: {report.workOrder ? 'Yes' : 'No'}
                          </Text>
                          {report.workOrder?.status && (
                            <Text style={{ color: '#0078D4', fontSize: 12 }}>
                              Job Card Status: {report.workOrder.status}
                            </Text>
                          )}
                          {/* New details section */}
                          <View style={{ marginTop: 4 }}>
                            {report.step1?.selectedEquipment?.equipmentNumber && (
                              <Text style={{ color: '#015185', fontSize: 12 }}>
                                Equipment No: {report.step1.selectedEquipment.equipmentNumber}
                              </Text>
                            )}
                            {report.workOrder?.requestNumber && (
                              <Text style={{ color: '#0078D4', fontSize: 12 }}>
                                Job Card Req No: {report.workOrder.requestNumber}
                              </Text>
                            )}
                            {report.correctiveAction?.requestNumber && (
                              <Text style={{ color: '#2ecc40', fontSize: 12 }}>
                                Corrective Action No: {report.correctiveAction.requestNumber}
                              </Text>
                            )}
                            {report.step1?.country && (
                              <Text style={{ color: '#888', fontSize: 12 }}>
                                Country: {report.step1.country}
                              </Text>
                            )}
                          </View>
                        </View>
                      </View>
                    );
                  })
                )}
              </>
            );
          })()}
        </View>
      </ScrollView>
      <Toast />
      <SyncModal visible={isLoading} onClose={() => setIsLoading(false)} message="Loading KPI Data..." />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#015185',
  },
  headerContainer: {
    paddingTop: StatusBar.currentHeight || 44,
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingVertical: 15,
    marginTop: 10,
    borderBottomWidth: 1,
    borderBottomColor: 'rgba(255, 255, 255, 0.1)',
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  backButton: {
    padding: 5,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  mainContent: {
    flex: 1,
    backgroundColor: '#fff',
    borderTopLeftRadius: 0,
    borderTopRightRadius: 0,
  },
  tabsContainer: {
    flexDirection: 'row',
    backgroundColor: '#F5F5F5',
    marginHorizontal: 15,
    borderRadius: 8,
    padding: 4,
    marginTop: 20,
  },
  tab: {
    flex: 1,
    paddingVertical: 8,
    alignItems: 'center',
    borderRadius: 6,
  },
  selectedTab: {
    backgroundColor: '#015185',
  },
  tabText: {
    fontSize: 12,
    color: '#666',
  },
  selectedTabText: {
    color: '#fff',
  },
  actionButtonsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    paddingHorizontal: 15,
    marginBottom: 20,
    paddingTop: width * 0.07,
  },
  actionButton: {
    width: '48%',
    backgroundColor: '#F5F5F5',
    padding: 15,
    margin: '1%',
    borderRadius: 8,
    alignItems: 'center',
  },
  actionButtonTitle: {
    fontSize: 14,
    color: '#015185',
    marginBottom: 5,
  },
  actionButtonCount: {
    fontSize: 24,
    color: '#015185',
    fontWeight: 'bold',
  },
  actionButtonSubtitle: {
    fontSize: 12,
    color: '#888',
    marginTop: 2,
  },
  inspectionSection: {
    marginHorizontal: 15,
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 14,
    color: '#015185',
    fontWeight: 'bold',
    marginBottom: 15,
  },
  inspectionItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#F0F0F0',
  },
  inspectionLabel: {
    fontSize: 14,
    color: '#333',
  },
  inspectionCount: {
    fontSize: 14,
    color: '#015185',
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '80%',
    alignItems: 'flex-start',
    elevation: 5,
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    alignSelf: 'center',
    width: '100%',
    textAlign: 'center',
  },
  modalLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 10,
  },
  modalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 4,
  },
  closeButton: {
    alignSelf: 'center',
    marginTop: 18,
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 24,
    paddingVertical: 8,
  },
  closeButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  hseReportButton: {
    width: '98%',
    backgroundColor: '#e3f2fd',
    padding: 20,
    margin: '1%',
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#015185',
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  hseReportContent: {
    alignItems: 'center',
  },
  hseReportTitle: {
    fontSize: 18,
    color: '#015185',
    fontWeight: 'bold',
    marginTop: 8,
  },
  hseReportCount: {
    fontSize: 36,
    color: '#015185',
    fontWeight: 'bold',
    marginVertical: 8,
  },
  hseReportSubtitle: {
    fontSize: 12,
    color: '#666',
    marginTop: 4,
  },
  inspectionReportButton: {
    width: '98%',
    backgroundColor: '#f0f8ff',
    padding: 20,
    margin: '1%',
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#0055A5',
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  inspectionReportContent: {
    alignItems: 'center',
  },
  inspectionReportTitle: {
    fontSize: 18,
    color: '#0055A5',
    fontWeight: 'bold',
    marginTop: 8,
  },
  inspectionReportCount: {
    fontSize: 36,
    color: '#0055A5',
    fontWeight: 'bold',
    marginVertical: 8,
  },
  inspectionReportSubtitle: {
    fontSize: 12,
    color: '#666',
    marginTop: 4,
  },
});

export default Inspection_Kpi_Home;