import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  Share,
  Modal,
  Alert,
  Animated,
  ActivityIndicator,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect, useIsFocused, useRoute } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { Dropdown } from 'react-native-element-dropdown';
import { getEmailStatistics } from './ModulesNotificationEmail';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig.js';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig.js';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';
import * as checklistData from './GlobalInspectionChecklistdata.json';
import OfflineDataHelper from './OfflineDataHelper';

// Error Boundary Component
class ErrorBoundary extends React.Component {
  state = { hasError: false };

  static getDerivedStateFromError(error) {
    return { hasError: true };
  }

  render() {
    if (this.state.hasError) {
      return (
        <View style={styles.container}>
          <View style={styles.offlineContainer}>
            <MaterialIcons name="wifi-off" size={60} color="#e74c3c" />
            <Text style={styles.offlineTitle}>No Internet Connection</Text>
            <Text style={styles.offlineMessage}>
              Please check your internet connection and try again.
            </Text>
            <TouchableOpacity 
              style={styles.retryButton}
              onPress={() => this.setState({ hasError: false })}
            >
              <Text style={styles.retryButtonText}>Retry</Text>
            </TouchableOpacity>
          </View>
        </View>
      );
    }
    return this.props.children;
  }
}

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const DROPDOWN_ITEM_HEIGHT = width * 0.13;

const DROPDOWN_OPTIONS = [
  { label: 'View Report', value: 'view', icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" /> },
  { label: 'Share', value: 'share', icon: <MaterialIcons name="share" size={22} color="#2196f3" /> },
  { label: 'Create Job Card', value: 'create-work-order', icon: <MaterialIcons name="build" size={22} color="#607D8B" /> },
  { label: 'Corrective Action', value: 'corrective', icon: <MaterialIcons name="radio-button-checked" size={22} color="#4CAF50" /> },
  { label: 'Archive Inspection', value: 'archive', icon: <MaterialIcons name="archive" size={22} color="#2196f3" /> },
  { label: 'Risk Assessment', value: 'risk-assessment', icon: <MaterialIcons name="warning" size={22} color="#FF9800" /> },
  { label: 'Delete', value: 'delete', icon: <MaterialIcons name="delete" size={22} color="#e74c3c" /> },
];

const SyncModal = ({ visible, onClose, message = 'Loading Reports', isOffline = false }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>{isOffline ? '📴 Offline Mode' : 'Please Wait'}</Text>
          <View style={styles.syncIcondiv}>
            {isOffline ? (
              <MaterialIcons name="cloud-off" size={30} color="#FF9800" style={{ marginRight: 10 }} />
            ) : (
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            )}
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const UserInfoModal = ({ visible, onClose, userInfo }) => {
  return (
    <Modal visible={visible} transparent animationType="slide" onRequestClose={onClose}>
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>User Information</Text>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Countries:</Text>
            <Text style={styles.userInfoValue}>
              {userInfo?.countries && Array.isArray(userInfo.countries) && userInfo.countries.length > 0 
                ? userInfo.countries.join(', ') 
                : 'N/A'}
            </Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Projects:</Text>
            <Text style={styles.userInfoValue}>
              {userInfo?.projects && Array.isArray(userInfo.projects) && userInfo.projects.length > 0 
                ? userInfo.projects.join(', ') 
                : 'N/A'}
            </Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Inspection Permissions:</Text>
            <Text style={styles.userInfoValue}>
              {userInfo?.inspectionPermissions && Array.isArray(userInfo.inspectionPermissions) && userInfo.inspectionPermissions.length > 0 
                ? userInfo.inspectionPermissions.join(', ') 
                : 'None'}
            </Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Employee Number:</Text>
            <Text style={styles.userInfoValue}>{userInfo?.employeeNumber || 'N/A'}</Text>
          </View>
          <View style={styles.userInfoItem}>
            <Text style={styles.userInfoLabel}>Name:</Text>
            <Text style={styles.userInfoValue}>{userInfo?.name || 'N/A'}</Text>
          </View>
          <TouchableOpacity style={styles.modalCloseButton} onPress={onClose}>
            <Text style={styles.modalCloseText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );
};

const Inspection_Report_Home = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('');
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [allReports, setAllReports] = useState([]);
  const [reports, setReports] = useState([]);
  const [selectedReport, setSelectedReport] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [userInfoModalVisible, setUserInfoModalVisible] = useState(false);
  const [isProcessingAction, setIsProcessingAction] = useState(false);
  const [jobCardsMap, setJobCardsMap] = useState({}); // Map inspection IDs to job cards
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    inspectionPermissions: [],
    employeeNumber: '',
    name: '',
  });
  const flatListRef = useRef(null);
  const isMounted = useRef(true);
  const [loadingUserInfo, setLoadingUserInfo] = useState(true);
  const PAGE_SIZE = 6;
  const [page, setPage] = useState(1);
  const [lastOnlineStatus, setLastOnlineStatus] = useState(true);
  const [isSyncing, setIsSyncing] = useState(false);
  const [lastSyncTime, setLastSyncTime] = useState(null);
  const [hasShownOfflineToast, setHasShownOfflineToast] = useState(false);
  const [hasShownOnlineToast, setHasShownOnlineToast] = useState(false);
  const [hasCompletedInitialLoad, setHasCompletedInitialLoad] = useState(false);
  const [offlineReportsCount, setOfflineReportsCount] = useState(0);
  const [onlineReportsCount, setOnlineReportsCount] = useState(0);
  const [localStorageSize, setLocalStorageSize] = useState('0 KB');

  const isSyncingOfflineReports = useRef(false); // Lock to prevent concurrent syncs

  // Handle notification data from deep linking
  useEffect(() => {
    if (route.params?.notificationData) {
      const notificationData = route.params.notificationData;
      console.log('Inspection report notification data received:', notificationData);
      
      // Show a toast message to indicate the notification was received
      Toast.show({
        type: 'info',
        text1: 'Inspection Report Notification',
        text2: 'Opening inspection report: ' + (notificationData.requestNumber || 'Unknown'),
        position: 'top',
        visibilityTime: 3000,
      });
      
      // You can add additional logic here to highlight or filter to the specific report
      // For now, we'll just show the notification and let the user browse normally
    }
  }, [route.params?.notificationData]);
  const [hasMore, setHasMore] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const [emailStats, setEmailStats] = useState(null);

  const filterOptions = ['Location', 'Inspector', 'Equipment', 'Project'];

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      // Check both isConnected AND isInternetReachable for accurate detection
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      return isOnline;
    } catch (error) {
      console.error('❌ Network check error:', error.message);
      return false;
    }
  };

  const loadEmailStatistics = async () => {
    try {
      const stats = await getEmailStatistics();
      if (isMounted.current) {
        setEmailStats(stats);
      }
    } catch (error) {
      console.error('Error loading email statistics:', error);
    }
  };

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const queueOperation = async (operation) => {
    try {
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      const pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push(operation);
      await AsyncStorage.setItem('pendingOperations', JSON.stringify(pendingOperations));
      showToast('info', 'Offline Mode', 'Operation queued for sync when online.');
    } catch (error) {
      console.error('Error queuing operation:', error);
      showToast('error', 'Error', 'Failed to queue operation.');
    }
  };

  const fetchUserInfo = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;
      
      let loadedUserInfo = null;
      
      // PRIORITY 1: Try inspectionModuleUserInfo first (most specific)
      try {
        const cachedInspectionUserInfo = await AsyncStorage.getItem('inspectionModuleUserInfo');
        if (cachedInspectionUserInfo) {
          loadedUserInfo = JSON.parse(cachedInspectionUserInfo);
          if (isMounted.current && loadedUserInfo) {
            setUserInfo(loadedUserInfo);
            console.log('✅ Loaded permissions from inspectionModuleUserInfo:', loadedUserInfo.inspectionPermissions);
          }
        }
      } catch (error) {
        console.error('❌ Error loading inspectionModuleUserInfo:', error.message);
      }
      
      // PRIORITY 2: Fallback to full userData
      if (!loadedUserInfo) {
        try {
          const cachedUserData = await AsyncStorage.getItem('userData');
          if (cachedUserData) {
            const userData = JSON.parse(cachedUserData);
            const inspectionModule = userData.modules?.find((module) => module.module === 'inspectionReport');
            
            const userInfoData = {
                countries: Array.isArray(userData.countries) ? userData.countries : [],
                projects: Array.isArray(userData.projects) ? userData.projects : [],
                inspectionPermissions: inspectionModule ? (Array.isArray(inspectionModule.permissions) ? inspectionModule.permissions : []) : [],
                employeeNumber: userData.employeeNumber || '',
                name: userData.name || '',
            };
            
            if (isMounted.current) {
              setUserInfo(userInfoData);
              loadedUserInfo = userInfoData;
              console.log('✅ Loaded permissions from userData:', userInfoData.inspectionPermissions);
              
              // Store for next time
              await AsyncStorage.setItem('inspectionModulePermissions', JSON.stringify(userInfoData.inspectionPermissions));
              await AsyncStorage.setItem('inspectionModuleUserInfo', JSON.stringify(userInfoData));
            }
          }
        } catch (error) {
          console.error('❌ Error loading userData:', error.message);
        }
      }
      
      // PRIORITY 3: Final fallback - restore from cached reports
      if (!loadedUserInfo) {
        try {
          const cachedReports = await AsyncStorage.getItem('inspectionReports');
          if (cachedReports) {
            const reports = JSON.parse(cachedReports);
            if (reports.length > 0 && reports[0].userPermissions) {
              const firstReport = reports[0];
          const userInfoData = {
                countries: firstReport.userCountries || [],
                projects: firstReport.userProjects || [],
                inspectionPermissions: firstReport.userPermissions || [],
                employeeNumber: '',
                name: '',
              };
              
              if (isMounted.current && userInfoData.inspectionPermissions.length > 0) {
          setUserInfo(userInfoData);
                loadedUserInfo = userInfoData;
                console.log('✅ Restored permissions from cached reports:', userInfoData.inspectionPermissions);
                
                // Store for next time
                await AsyncStorage.setItem('inspectionModulePermissions', JSON.stringify(userInfoData.inspectionPermissions));
                await AsyncStorage.setItem('inspectionModuleUserInfo', JSON.stringify(userInfoData));
              }
            }
          }
        } catch (error) {
          console.error('❌ Error loading from cached reports:', error.message);
        }
      }
      
      // If we loaded from cache, log success
      if (loadedUserInfo) {
        console.log('✅ UserInfo loaded from cache - Offline mode ready');
      } else {
        console.log('⚠️ No cached userInfo found');
      }
      
      // Check if online - only try MongoDB if online
      const onlineStatus = await checkNetworkStatus();
      
      if (!onlineStatus) {
        // Offline - cache already loaded (or not)
        console.log('📴 Offline - Using cached data only');
        return;
      }
      
      // Online - try to fetch from MongoDB (max 2 attempts) to update cache
      console.log('🌐 Online - Fetching fresh data from MongoDB...');
      let attempts = 0;
      const MAX_ATTEMPTS = 2;
      
      while (attempts < MAX_ATTEMPTS) {
        attempts++;
        try {
          const response = await MongoAPI.getUserById(userId);
          
          if (response.success && response.data) {
            const userData = response.data;
          const inspectionModule = userData.modules?.find((module) => module.module === 'inspectionReport');
          
            const userInfoData = {
              countries: Array.isArray(userData.countries) ? userData.countries : [],
              projects: Array.isArray(userData.projects) ? userData.projects : [],
              inspectionPermissions: inspectionModule ? (Array.isArray(inspectionModule.permissions) ? inspectionModule.permissions : []) : [],
              employeeNumber: userData.employeeNumber || '',
              name: userData.name || '',
            };
            
            if (isMounted.current) {
            setUserInfo(userInfoData);
              console.log('✅ Updated permissions from MongoDB:', userInfoData.inspectionPermissions);
              
              // Update cache
              await AsyncStorage.setItem('userData', JSON.stringify(userData));
              
              // Store inspection module permissions separately
              await AsyncStorage.setItem('inspectionModulePermissions', JSON.stringify(userInfoData.inspectionPermissions));
              await AsyncStorage.setItem('inspectionModuleUserInfo', JSON.stringify(userInfoData));
            }
            break; // Success, exit loop
        } else {
            if (attempts < MAX_ATTEMPTS) {
              await new Promise(resolve => setTimeout(resolve, 1000));
        }
      }
    } catch (error) {
          console.error(`❌ MongoDB fetch attempt ${attempts} failed:`, error.message);
          if (attempts < MAX_ATTEMPTS) {
            await new Promise(resolve => setTimeout(resolve, 1000));
          }
        }
      }
    } catch (error) {
      console.error('❌ User info error:', error.message);
    } finally {
      if (isMounted.current) {
        setLoadingUserInfo(false);
      }
    }
  };

  const syncOfflineReports = async () => {
    if (!(await checkNetworkStatus())) return;
    
    // Prevent concurrent syncs
    if (isSyncingOfflineReports.current) {
      console.log('⚠️ Offline sync already in progress, skipping...');
      return { success: true, synced: 0, failed: 0 };
    }

    try {
      isSyncingOfflineReports.current = true;
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;
      
      console.log('🔒 Starting offline reports sync (locked)...');

      // Sync offline reports using OfflineDataHelper
      const syncResult = await OfflineDataHelper.syncOfflineReports(async (report) => {
        try {
          console.log('📤 Syncing offline report:', report.requestNumber || report._id);
          
          // Generate dynamic request number based on current MongoDB count
          let newRequestNumber = report.requestNumber;
          
          // CRITICAL: Remove "-OFFLINE" suffix if present
          if (newRequestNumber && newRequestNumber.includes('-OFFLINE')) {
            newRequestNumber = newRequestNumber.replace('-OFFLINE', '');
            console.log('🧹 Removed -OFFLINE suffix from:', report.requestNumber, '→', newRequestNumber);
          }
          
          try {
            console.log('🔢 Generating new request number from MongoDB count...');
            const countResponse = await InspectionAPI.getTotalCount();
            let nextNumber = 1;
            
            if (countResponse.success && countResponse.total !== undefined) {
              nextNumber = countResponse.total + 1;
              console.log('📊 Current total in MongoDB:', countResponse.total);
              console.log('📈 New report will be:', nextNumber);
            } else {
              // Fallback to getAllReports
              const fallbackResponse = await InspectionAPI.getAllReports({ page: 1, limit: 1 });
              if (fallbackResponse.success && fallbackResponse.pagination) {
                nextNumber = (fallbackResponse.pagination.total || 0) + 1;
              }
            }
            
            // Generate proper request number (clean, without -OFFLINE)
            const country = report.step1?.country || 'Unknown';
            const project = report.step1?.project || 'Unknown';
            const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
            
            let projectCode = 'XXX';
            if (project) {
              const projectMap = {
                'lubumbashi': 'LUB',
                'musompo': 'MUS',
                'kansanshi': 'KAN',
                'kalumbila': 'KAL',
                'kimteto': 'KIM',
                'kobold': 'KOB',
                'fqm exploration': 'FQM',
                'mimosa': 'MIM',
                'ime': 'IME',
                'kamoa': 'KAM',
              };
              projectCode = projectMap[project.toLowerCase()] || project.slice(0, 3).toUpperCase();
            }
            
            // Generate clean request number (no -OFFLINE suffix)
            newRequestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}`;
            console.log('✅ Generated clean request number:', newRequestNumber);
            console.log('   Old (offline):', report.requestNumber);
            console.log('   New (MongoDB):', newRequestNumber);
            
            // Update report with new clean request number
            report.requestNumber = newRequestNumber;
            if (report.step1) {
              report.step1.requestNumber = newRequestNumber;
            }
          } catch (numberError) {
            console.warn('⚠️ Failed to generate dynamic request number:', numberError.message);
            // Use cleaned version (without -OFFLINE) if generation fails
            report.requestNumber = newRequestNumber;
            if (report.step1) {
              report.step1.requestNumber = newRequestNumber;
            }
          }
          
          // Check if report already exists in MongoDB to prevent duplicates
          try {
            const existingReport = await InspectionAPI.getInspectionById(report._id || report.id);
            if (existingReport.success && existingReport.data) {
              console.log('⚠️ Report already exists in MongoDB, skipping:', report._id || report.id);
              return { success: true, skipped: true, message: 'Report already synced' };
            }
          } catch (checkError) {
            // Report doesn't exist, proceed with creation
            console.log('✅ Report not found in MongoDB, creating...');
          }
          
          // Create the inspection in MongoDB with updated request number
          const response = await InspectionAPI.createInspection(userId, report);
          
          if (response.success) {
            console.log('✅ Report created in MongoDB with number:', newRequestNumber);
            console.log('✅ Report ID:', response.data?._id || report._id);
            
            // If report has local PDF, upload it to MongoDB
            if (report.step3?.pdfLocalPath) {
              try {
                const RNFS = require('react-native-fs');
                
                // Check if PDF file exists
                const fileExists = await RNFS.exists(report.step3.pdfLocalPath);
                if (!fileExists) {
                  console.warn('⚠️ PDF file not found at:', report.step3.pdfLocalPath);
                  return { success: true, warning: 'Report synced but PDF file not found' };
                }
                
                console.log('📎 Uploading PDF from:', report.step3.pdfLocalPath);
                
                const formData = new FormData();
                formData.append('pdf', {
                  uri: Platform.OS === 'android' ? `file://${report.step3.pdfLocalPath}` : report.step3.pdfLocalPath,
                  type: 'application/pdf',
                  name: `inspection_${report._id || report.id}.pdf`,
                });
                
                // Upload PDF and get download URL
                const pdfResponse = await InspectionAPI.uploadPdf(userId, formData);
                
                if (pdfResponse.success && pdfResponse.downloadUrl) {
                  console.log('✅ PDF uploaded successfully:', pdfResponse.downloadUrl);
                  
                  // Update the report in MongoDB with PDF download URL
                  try {
                    const reportId = response.data?._id || report._id || report.id;
                    await InspectionAPI.updateInspection(userId, reportId, {
                      'step3.pdfDownloadUrl': pdfResponse.downloadUrl
                    });
                    console.log('✅ Report updated with PDF URL in MongoDB');
                  } catch (updateError) {
                    console.warn('⚠️ Failed to update report with PDF URL:', updateError.message);
                  }
                  
                  // Send notifications and emails for synced offline report
                  setTimeout(() => {
                    (async () => {
                      try {
                        console.log('🔔 Sending notifications for synced offline inspection report...');
                        console.log('📎 Using PDF URL:', pdfResponse.downloadUrl);
                        
                        // Get inspector email from report data
                        let inspectorEmail = null;
                        try {
                          const userDataStr = await AsyncStorage.getItem('userData');
                          if (userDataStr) {
                            const userData = JSON.parse(userDataStr);
                            inspectorEmail = userData.email || null;
                          }
                        } catch (err) {
                          console.warn('⚠️ Could not fetch inspector email:', err);
                        }
                        
                        // Calculate faulty items count
                        const sections = report.step2?.sections || [];
                        const selections = report.step2?.selections || {};
                        const faultyCount = sections.flatMap((section) =>
                          (section.subItems || []).filter((item) => 
                            selections[item.key] === 'Faulty' || 
                            selections[item.key] === 'Reject' || 
                            selections[item.key] === 'Not Ok'
                          )
                        ).length;
                        
                        const notificationData = {
                          inspectionData: {
                            _id: report._id || report.id,
                            requestNumber: newRequestNumber,
                            equipmentName: report.step1?.selectedEquipment?.equipmentName || 'N/A',
                            country: report.step1?.country,
                            project: report.step1?.project,
                            faultyItems: faultyCount,
                          },
                          createdByName: report.step1?.inspector || 'Inspector',
                          inspectorEmail: inspectorEmail,
                          pdfUrl: pdfResponse.downloadUrl // Include PDF URL for email attachment
                        };
                        
                        // Call Inspection Notifications service
                        const notifResponse = await fetch('http://168.231.113.219:5019/api/inspection-notifications/new-inspection', {
                          method: 'POST',
                          headers: { 'Content-Type': 'application/json' },
                          body: JSON.stringify(notificationData),
                          timeout: 10000,
                        });
                        
                        if (notifResponse.ok) {
                          const notifResult = await notifResponse.json();
                          console.log('✅ Inspection notifications sent successfully for synced report');
                          console.log(`  📱 Push: ${notifResult.results?.push?.sent || 0} device(s)`);
                          console.log(`  📧 Emails to Recipients: ${notifResult.results?.emails?.sent || 0}`);
                          console.log(`  📧 Inspector Email: ${notifResult.results?.inspectorEmail?.success ? 'Sent' : 'Failed'}`);
                          console.log(`  💾 MongoDB: ${notifResult.results?.mongodb?.saved || 0} notification(s)`);
                        } else {
                          console.error('❌ Failed to send inspection notifications for synced report');
                        }
                      } catch (notifError) {
                        console.error('❌ Error sending inspection notifications:', notifError);
                        // Don't fail sync if notifications fail
                      }
                    })();
                  }, 500);
                  
                  return { success: true, pdfUrl: pdfResponse.downloadUrl };
                } else {
                  console.warn('⚠️ PDF upload response missing download URL');
                  return { success: true, warning: 'Report synced but PDF URL not returned' };
                }
              } catch (pdfError) {
                console.error('❌ PDF upload failed:', pdfError.message);
                return { success: true, warning: 'Report synced but PDF upload failed' };
              }
            }
            
            // If no PDF to upload, still send notifications
            // Send notifications for synced offline report (without PDF)
            setTimeout(() => {
              (async () => {
                try {
                  console.log('🔔 Sending notifications for synced offline inspection report (no PDF)...');
                  
                  // Get inspector email from report data
                  let inspectorEmail = null;
                  try {
                    const userDataStr = await AsyncStorage.getItem('userData');
                    if (userDataStr) {
                      const userData = JSON.parse(userDataStr);
                      inspectorEmail = userData.email || null;
                    }
                  } catch (err) {
                    console.warn('⚠️ Could not fetch inspector email:', err);
                  }
                  
                  // Calculate faulty items count
                  const sections = report.step2?.sections || [];
                  const selections = report.step2?.selections || {};
                  const faultyCount = sections.flatMap((section) =>
                    (section.subItems || []).filter((item) => 
                      selections[item.key] === 'Faulty' || 
                      selections[item.key] === 'Reject' || 
                      selections[item.key] === 'Not Ok'
                    )
                  ).length;
                  
                  const notificationData = {
                    inspectionData: {
                      _id: report._id || report.id,
                      requestNumber: newRequestNumber,
                      equipmentName: report.step1?.selectedEquipment?.equipmentName || 'N/A',
                      country: report.step1?.country,
                      project: report.step1?.project,
                      faultyItems: faultyCount,
                    },
                    createdByName: report.step1?.inspector || 'Inspector',
                    inspectorEmail: inspectorEmail,
                    pdfUrl: null // No PDF available
                  };
                  
                  // Call Inspection Notifications service
                  const notifResponse = await fetch('http://168.231.113.219:5019/api/inspection-notifications/new-inspection', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(notificationData),
                    timeout: 10000,
                  });
                  
                  if (notifResponse.ok) {
                    const notifResult = await notifResponse.json();
                    console.log('✅ Inspection notifications sent successfully for synced report');
                    console.log(`  📱 Push: ${notifResult.results?.push?.sent || 0} device(s)`);
                    console.log(`  📧 Emails to Recipients: ${notifResult.results?.emails?.sent || 0}`);
                    console.log(`  📧 Inspector Email: ${notifResult.results?.inspectorEmail?.success ? 'Sent' : 'Failed'}`);
                    console.log(`  💾 MongoDB: ${notifResult.results?.mongodb?.saved || 0} notification(s)`);
                  } else {
                    console.error('❌ Failed to send inspection notifications for synced report');
                  }
                } catch (notifError) {
                  console.error('❌ Error sending inspection notifications:', notifError);
                  // Don't fail sync if notifications fail
                }
              })();
            }, 500);
            
            return { success: true };
          } else {
            throw new Error('Failed to create inspection in MongoDB');
          }
        } catch (error) {
          console.error('❌ Sync error for report:', error.message);
          return { success: false, error: error.message };
        }
      });
      
      if (syncResult.synced > 0) {
        showToast('success', 'Sync Complete', `${syncResult.synced} offline reports & PDFs synced to MongoDB`);
        console.log('📊 Sync Summary:', {
          synced: syncResult.synced,
          failed: syncResult.failed,
          total: syncResult.synced + syncResult.failed
        });
        
        // CRITICAL: Reload reports from MongoDB to update UI and remove SYNC badges
        console.log('🔄 Reloading reports from MongoDB to update UI...');
        try {
          await loadReports(false); // Reload without showing modal
          await updateReportCounts(); // Update offline/online counts
          console.log('✅ UI refreshed - SYNC badges removed');
        } catch (refreshError) {
          console.error('❌ Failed to refresh UI after sync:', refreshError.message);
        }
      }
      
      if (syncResult.failed > 0) {
        showToast('warning', 'Partial Sync', `${syncResult.failed} reports failed to sync`);
      }
      
      console.log('🔓 Offline sync completed (unlocked)');
      return syncResult;
    } catch (error) {
      console.error('❌ Sync offline error:', error.message);
      return { success: false, synced: 0, failed: 0 };
    } finally {
      isSyncingOfflineReports.current = false; // Always unlock
    }
  };

  // Auto-cleanup reports without created dates AND duplicate requestNumbers
  const cleanupReportsWithoutDates = async () => {
    try {
      console.log('🧹 Starting automatic cleanup of invalid reports...');
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;
      
      // Check both local cache and MongoDB
      const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
      const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
      
      let deletedCount = 0;
      const reportsToDelete = [];
      
      // STRATEGY 1: Find reports without createdAt date
      console.log('🔍 Checking for reports without dates...');
      for (const report of cachedReports) {
        if (!report.createdAt && !report.step3?.createdAt && !report.offlineCreatedAt) {
          console.log('🗑️ Found report without date:', report.id || report._id, report.step1?.requestNumber || report.requestNumber);
          reportsToDelete.push({
            report,
            reason: 'Missing date'
          });
        }
      }
      
      // STRATEGY 2: Find duplicate requestNumbers (keep the one with date, delete others)
      console.log('🔍 Checking for duplicate requestNumbers...');
      const requestNumberMap = new Map();
      
      for (const report of cachedReports) {
        const requestNumber = report.step1?.requestNumber || report.requestNumber;
        
        if (requestNumber) {
          if (!requestNumberMap.has(requestNumber)) {
            requestNumberMap.set(requestNumber, []);
          }
          requestNumberMap.get(requestNumber).push(report);
        }
      }
      
      // Check for duplicates
      for (const [requestNumber, reports] of requestNumberMap.entries()) {
        if (reports.length > 1) {
          console.log(`⚠️ Found ${reports.length} reports with same requestNumber: ${requestNumber}`);
          
          // Sort: Reports WITH dates first, then by date (newest first)
          const sorted = reports.sort((a, b) => {
            const aDate = a.createdAt || a.step3?.createdAt || a.offlineCreatedAt;
            const bDate = b.createdAt || b.step3?.createdAt || b.offlineCreatedAt;
            
            // If one has date and other doesn't, prefer the one with date
            if (aDate && !bDate) return -1;
            if (!aDate && bDate) return 1;
            
            // If both have dates, prefer newer
            if (aDate && bDate) {
              return new Date(bDate) - new Date(aDate);
            }
            
            // If neither has date, keep first one
            return 0;
          });
          
          // Keep the first one (best candidate), delete the rest
          const toKeep = sorted[0];
          const toDelete = sorted.slice(1);
          
          console.log(`✅ Keeping: ${toKeep.id || toKeep._id} (Date: ${toKeep.createdAt || 'N/A'})`);
          
          for (const duplicate of toDelete) {
            console.log(`🗑️ Deleting duplicate: ${duplicate.id || duplicate._id} (Date: ${duplicate.createdAt || 'N/A'})`);
            
            // Check if already marked for deletion
            const alreadyMarked = reportsToDelete.find(item => 
              (item.report.id || item.report._id) === (duplicate.id || duplicate._id)
            );
            
            if (!alreadyMarked) {
              reportsToDelete.push({
                report: duplicate,
                reason: `Duplicate requestNumber: ${requestNumber}`
              });
            }
          }
        }
      }
      
      if (reportsToDelete.length === 0) {
        console.log('✅ No invalid or duplicate reports found');
        return;
      }
      
      console.log(`🧹 Found ${reportsToDelete.length} invalid reports to delete:`);
      
      // Log all reports to delete with reasons
      reportsToDelete.forEach(item => {
        console.log(`  - ${item.report.id || item.report._id}: ${item.reason}`);
      });
      
      // Delete from MongoDB if online
      const onlineStatus = await checkNetworkStatus();
      if (onlineStatus) {
        for (const item of reportsToDelete) {
          try {
            const reportId = item.report.id || item.report._id;
            const requestNumber = item.report.step1?.requestNumber || item.report.requestNumber || 'Unknown';
            console.log(`🗑️ Deleting from MongoDB: ${reportId} (${requestNumber}) - Reason: ${item.reason}`);
            
            const response = await InspectionAPI.deleteInspectionById(reportId);
            if (response.success) {
              deletedCount++;
              console.log(`✅ Deleted from MongoDB: ${reportId}`);
            }
          } catch (error) {
            console.warn('⚠️ Failed to delete from MongoDB:', error.message);
          }
        }
      }
      
      // Delete from local storage
      for (const item of reportsToDelete) {
        const reportId = item.report.id || item.report._id;
        try {
          await AsyncStorage.removeItem(`report_${reportId}`);
          console.log(`✅ Deleted from local storage: ${reportId}`);
        } catch (error) {
          console.warn('⚠️ Failed to delete from local storage:', error.message);
        }
      }
      
      // Update cached reports list (remove deleted reports)
      const deletedIds = new Set(reportsToDelete.map(item => item.report.id || item.report._id));
      const updatedReports = cachedReports.filter(report => {
        const reportId = report.id || report._id;
        return !deletedIds.has(reportId);
      });
      
      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));
      
      console.log(`✅ Cleanup complete: Deleted ${reportsToDelete.length} invalid reports`);
      console.log(`   - Reports without dates: ${reportsToDelete.filter(r => r.reason === 'Missing date').length}`);
      console.log(`   - Duplicate requestNumbers: ${reportsToDelete.filter(r => r.reason.includes('Duplicate')).length}`);
      
      if (reportsToDelete.length > 0) {
        const withoutDates = reportsToDelete.filter(r => r.reason === 'Missing date').length;
        const duplicates = reportsToDelete.filter(r => r.reason.includes('Duplicate')).length;
        
        let message = '';
        if (withoutDates > 0 && duplicates > 0) {
          message = `${withoutDates} without dates, ${duplicates} duplicates deleted`;
        } else if (withoutDates > 0) {
          message = `${withoutDates} reports without dates deleted`;
        } else if (duplicates > 0) {
          message = `${duplicates} duplicate reports deleted`;
        }
        
        showToast('info', 'Auto Cleanup', message);
        
        // Reload reports to update UI (skip cleanup to prevent infinite loop)
        console.log('🔄 Reloading reports after cleanup...');
        await loadReports(false, true);
      }
      
    } catch (error) {
      console.error('❌ Cleanup error:', error.message);
    }
  };

  const syncPendingEquipmentUpdates = async () => {
    if (!(await checkNetworkStatus())) return;

    try {
      const equipmentRaw = await AsyncStorage.getItem('equipment');
      if (!equipmentRaw) return;
      
      const equipmentList = JSON.parse(equipmentRaw);
      const pendingEquipment = equipmentList.filter(item => item.pendingSync === true);
      
      if (pendingEquipment.length === 0) return;
      
      const EquipmentAPI = require('../../ConfigMongo/EquipmentMongoConfig').default;
      
      let synced = 0;
      
      for (const equipment of pendingEquipment) {
        try {
          await EquipmentAPI.updateEquipment(equipment.id, {
            meterReading: equipment.meterReading,
            updatedAt: equipment.updatedAt,
          });
          
          // Remove pendingSync flag
          const index = equipmentList.findIndex(item => item.id === equipment.id);
          if (index !== -1) {
            equipmentList[index].pendingSync = false;
          }
          
          synced++;
        } catch (error) {
          // Keep pendingSync flag for retry
        }
      }
      
      // Save updated equipment list
      await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
    } catch (error) {
      console.error('❌ Equipment sync error:', error.message);
    }
  };

  const syncPendingOperations = async () => {
    if (!(await checkNetworkStatus())) return;

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'User not authenticated.');
        return;
      }

      const response = await MongoAPI.getUserById(userId);
      if (!response.success || !response.data) {
        showToast('error', 'Authentication Error', 'User not found in admin database.');
        return;
      }

      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      if (!pendingOperationsRaw) return;

      const pendingOperations = JSON.parse(pendingOperationsRaw);
      if (pendingOperations.length === 0) return;

      let updatedReports = [];
      for (const operation of pendingOperations) {
        try {
          if (operation.type === 'delete') {
            await InspectionAPI.deleteInspection(operation.userId, operation.reportId);
            await AsyncStorage.removeItem(`report_${operation.reportId}`);
          } else if (operation.type === 'create') {
            const createResponse = await InspectionAPI.createInspection(operation.userId, operation.report);
            if (createResponse.success) {
              await AsyncStorage.setItem(`report_${operation.report.id}`, JSON.stringify(operation.report));
              updatedReports.push(operation.report);
            }
          }
        } catch (error) {
          continue;
        }
      }

      if (isMounted.current) {
        setAllReports((prev) => {
          const allReports = [...prev, ...updatedReports].filter((report, index, self) =>
            index === self.findIndex((r) => r.id === report.id)
          );
          return allReports;
        });
      }
      await AsyncStorage.setItem('pendingOperations', JSON.stringify([]));
      showToast('success', 'Sync Complete', 'Offline operations synced successfully.');
    } catch (error) {
      console.error('❌ Pending ops error:', error.message);
    }
  };

  const loadReports = async (showModal = false, skipCleanup = false) => {
    if (!isMounted.current) return;
    
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;

      // Step 1: INSTANT LOAD FROM CACHE (works offline & online)
        const cachedReports = await AsyncStorage.getItem('inspectionReports');
        const cachedSyncTime = await AsyncStorage.getItem('inspectionReportsLastSync');
      const offlineReports = await OfflineDataHelper.getOfflineReports();
      
      let reportsArray = cachedReports ? JSON.parse(cachedReports) : [];
      
      // CRITICAL: If userInfo permissions missing, restore from first cached report
      if (reportsArray.length > 0 && (!userInfo.inspectionPermissions || userInfo.inspectionPermissions.length === 0)) {
        const firstReport = reportsArray[0];
        if (firstReport.userPermissions && firstReport.userPermissions.length > 0) {
          const restoredUserInfo = {
            countries: firstReport.userCountries || userInfo.countries || [],
            projects: firstReport.userProjects || userInfo.projects || [],
            inspectionPermissions: firstReport.userPermissions || [],
            employeeNumber: userInfo.employeeNumber || '',
            name: userInfo.name || '',
          };
          
          setUserInfo(restoredUserInfo);
          setLoadingUserInfo(false);
          
          console.log('🔄 Restored permissions from report:', firstReport.userPermissions);
          console.log('🔄 Restored countries:', firstReport.userCountries);
          console.log('🔄 Restored projects:', firstReport.userProjects?.length, 'projects');
          
          // Also update cache
          await AsyncStorage.setItem('inspectionModulePermissions', JSON.stringify(firstReport.userPermissions));
          await AsyncStorage.setItem('inspectionModuleUserInfo', JSON.stringify(restoredUserInfo));
        }
      }
      
      // Append offline reports
      if (offlineReports.length > 0) {
        const formattedOfflineReports = offlineReports.map(report => ({
          ...report,
          id: report._id || report.id,
          adminId: report.userId || report.createdBy,
          createdAt: report.createdAt || report.offlineCreatedAt,
          offlineCreated: true,
          offlineSyncPending: true,
          // Preserve permissions from offline report if available
          userPermissions: report.userPermissions || userInfo.inspectionPermissions || [],
          userCountries: report.userCountries || userInfo.countries || [],
          userProjects: report.userProjects || userInfo.projects || [],
        }));
        reportsArray = [...reportsArray, ...formattedOfflineReports];
      }
      
      // CRITICAL: Remove duplicates by ID
      const uniqueReportsMap = new Map();
      reportsArray.forEach(report => {
        const reportId = report.id || report._id;
        if (!uniqueReportsMap.has(reportId)) {
          uniqueReportsMap.set(reportId, report);
        }
      });
      reportsArray = Array.from(uniqueReportsMap.values());
      
      // Show cache data instantly
      setAllReports(reportsArray);
      if (cachedSyncTime) setLastSyncTime(cachedSyncTime);
      await updateReportCounts();
      
      console.log('📊 Loaded reports summary:');
      console.log(`   Cached from MongoDB: ${cachedReports ? JSON.parse(cachedReports).length : 0}`);
      console.log(`   Offline pending: ${offlineReports.length}`);
      console.log(`   TOTAL in allReports: ${reportsArray.length}`);
      if (reportsArray.length > 0) {
        console.log(`   First report: ${reportsArray[0].step1?.requestNumber || reportsArray[0].id}`);
        console.log(`   Report country: ${reportsArray[0].step1?.country}`);
        console.log(`   Report project: ${reportsArray[0].step1?.project}`);
        console.log(`   Has permissions: ${!!reportsArray[0].userPermissions}`);
        if (reportsArray[0].userPermissions) {
          console.log(`   Report permissions: ${reportsArray[0].userPermissions.join(', ')}`);
          console.log(`   Report countries: ${reportsArray[0].userCountries?.join(', ')}`);
          console.log(`   Report projects: ${reportsArray[0].userProjects?.length} projects`);
        }
      } else {
        console.log('⚠️ NO REPORTS LOADED FROM CACHE!');
      }
      
      // Step 2: Check if online
      const onlineStatus = await checkNetworkStatus();
      setIsOnline(onlineStatus);
      
      if (!onlineStatus) {
        // OFFLINE - use cache only, exit
        console.log('📴 Offline mode - Using cached data only');
        return;
      }
      
      // Step 3: If online, try MongoDB (max 2 attempts)
      if (!userInfo || !userInfo.inspectionPermissions || userInfo.inspectionPermissions.length === 0) {
        return; // UserInfo not ready, cache already shown
      }
      
      if (showModal) setSyncModalVisible(true);
      setIsRefreshing(true);
      
      let response = null;
      let attempts = 0;
      const MAX_ATTEMPTS = 2;
      
      while (attempts < MAX_ATTEMPTS && !response) {
        attempts++;
        try {
      if (userInfo.inspectionPermissions.includes('onlyMineView')) {
        response = await InspectionAPI.getInspections(userId, { page: 1, limit: 1000 });
      } else if (userInfo.inspectionPermissions.includes('view')) {
          response = await InspectionAPI.getAllReports({ page: 1, limit: 1000 });
          
          if (!response.success || !response.data || response.data.length === 0) {
              response = await InspectionAPI.getInspections('test_proadmin_amit_001', { page: 1, limit: 1000 });
            }
          } else {
            break; // No permissions
          }
          
          // Check if response is valid
          if (response && response.success && response.data) {
            break; // Got valid response, exit loop
          } else {
            response = null; // Reset for retry
          }
        } catch (error) {
          console.error(`❌ MongoDB attempt ${attempts}/${MAX_ATTEMPTS} failed:`, error.message);
          response = null;
          if (attempts < MAX_ATTEMPTS) {
            await new Promise(resolve => setTimeout(resolve, 1000)); // Wait 1s before retry
          }
        }
      }
      
      // Step 4: If MongoDB succeeded, update cache
      if (response && response.success && response.data) {
        const mongoReports = response.data.map(report => ({
            ...report,
            id: report._id,
            adminId: report.userId || report.createdBy,
            createdAt: report.createdAt || new Date().toISOString(),
          // CRITICAL: Store permissions with each report for offline access
          userPermissions: userInfo.inspectionPermissions || [],
          userCountries: userInfo.countries || [],
          userProjects: userInfo.projects || [],
          syncedAt: new Date().toISOString(),
          syncedBy: userId,
        }));
        
        // Store in AsyncStorage
        await AsyncStorage.setItem('inspectionReports', JSON.stringify(mongoReports));
        
        for (const report of mongoReports) {
          try {
            await AsyncStorage.setItem(`report_${report.id}`, JSON.stringify(report));
          } catch (error) {
            // Silent fail
          }
        }
        
          const syncTime = new Date().toISOString();
          await AsyncStorage.setItem('inspectionReportsLastSync', syncTime);
        await AsyncStorage.setItem('inspectionSyncMetadata', JSON.stringify({
          lastSyncTime: syncTime,
          totalReports: mongoReports.length,
          syncedBy: userId,
          syncSource: 'MongoDB',
          isOnline: true
        }));
          setLastSyncTime(syncTime);
          
        // Update UI with fresh data (preserve permissions)
        let finalReports = [
          ...mongoReports,
          ...offlineReports.map(r => ({
            ...r,
            id: r._id || r.id,
            adminId: r.userId || r.createdBy,
            createdAt: r.createdAt || r.offlineCreatedAt,
            offlineCreated: true,
            offlineSyncPending: true,
            // Preserve permissions from offline report or use current userInfo
            userPermissions: r.userPermissions || userInfo.inspectionPermissions || [],
            userCountries: r.userCountries || userInfo.countries || [],
            userProjects: r.userProjects || userInfo.projects || [],
          }))
        ];
        
        // CRITICAL: Remove duplicates by ID
        const uniqueReportsMap = new Map();
        finalReports.forEach(report => {
          const reportId = report.id || report._id;
          if (!uniqueReportsMap.has(reportId)) {
            uniqueReportsMap.set(reportId, report);
          }
        });
        finalReports = Array.from(uniqueReportsMap.values());
        
        setAllReports(finalReports);
        await updateReportCounts();
        
          if (showModal) {
          showToast('success', 'Synced', `${finalReports.length} reports synced & stored offline`);
        }
      }
      
      // Load job cards if online
      if (onlineStatus) {
        await loadJobCardsForInspections();
      }
      
      // Auto-cleanup reports without dates (run in background, don't block) - ONLY if not skipped
      if (!skipCleanup) {
        setTimeout(() => {
          cleanupReportsWithoutDates().catch(err => {
            console.error('❌ Background cleanup failed:', err);
          });
        }, 1000);
      }
      
    } catch (error) {
      console.error('❌ Load reports error:', error.message);
    } finally {
      if (isMounted.current) {
        setIsRefreshing(false);
        if (showModal) setSyncModalVisible(false);
      }
    }
  };

  // Update report counts (offline vs online)
  const updateReportCounts = async () => {
    try {
      // Get offline pending sync reports (created offline, not yet in MongoDB)
      const offlinePendingReports = await OfflineDataHelper.getOfflineReports();
      const pendingCount = offlinePendingReports.length;
      
      // Get cached reports from MongoDB (synced and stored offline)
      const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
      const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
      const cachedCount = cachedReports.length;
      
      // TOTAL OFFLINE STORED = Pending + Cached (ALL reports available offline)
      const totalOfflineCount = pendingCount + cachedCount;
      setOfflineReportsCount(totalOfflineCount);
      
      // Online in DB count = Just the cached reports (synced to MongoDB)
      setOnlineReportsCount(cachedCount);
      
      // Calculate local storage size
      try {
        let totalSize = 0;
        const keys = await AsyncStorage.getAllKeys();
        const inspectionKeys = keys.filter(key => 
          key.includes('inspection') || 
          key.includes('report') || 
          key.includes('offline')
        );
        
        for (const key of inspectionKeys) {
          const value = await AsyncStorage.getItem(key);
          if (value) {
            totalSize += value.length;
          }
        }
        
        // Convert to KB/MB
        const sizeInKB = (totalSize / 1024).toFixed(2);
        const sizeInMB = (totalSize / (1024 * 1024)).toFixed(2);
        const displaySize = totalSize > 1024 * 1024 ? `${sizeInMB} MB` : `${sizeInKB} KB`;
        setLocalStorageSize(displaySize);
      } catch (storageError) {
        // Silent fail
      }
      
    } catch (error) {
      console.error('❌ Count update error:', error.message);
    }
  };

  // Format last sync time
  const getLastSyncText = () => {
    if (!lastSyncTime) return 'Never synced';
    
    const now = new Date();
    const syncDate = new Date(lastSyncTime);
    const diffInMinutes = Math.floor((now - syncDate) / (1000 * 60));
    
    if (diffInMinutes < 1) return 'Just now';
    if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
    
    const diffInHours = Math.floor(diffInMinutes / 60);
    if (diffInHours < 24) return `${diffInHours}h ago`;
    
    const diffInDays = Math.floor(diffInHours / 24);
    return `${diffInDays}d ago`;
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (!isMounted.current) return;
      
      const currentOnlineStatus = state.isConnected;
      const wasOffline = !lastOnlineStatus;
      const isNowOnline = currentOnlineStatus;
      
      setIsOnline(currentOnlineStatus);
      
      // Show toast notifications for status changes
      if (wasOffline && isNowOnline) {
        // Device came back online
        console.log('🌐 Connection restored');
        showToast('success', 'Online', 'Connection restored - Use cloud sync to upload reports');
        setLastOnlineStatus(true);
        setHasShownOnlineToast(true);
        setHasShownOfflineToast(false);
        
        // Reload data from MongoDB only (NO AUTO-SYNC of offline reports)
        setTimeout(async () => {
          try {
            console.log('🌐 Reloading data from MongoDB (no auto-sync)');
            await fetchUserInfo();
            await loadReports(false); // Reload from MongoDB without syncing
            showToast('info', 'Data Refreshed', 'Use cloud button to sync offline reports');
          } catch (error) {
            console.error('Reload error:', error);
          }
        }, 1000);
      } else if (!wasOffline && !isNowOnline) {
        // Device went offline
        console.log('📴 Connection lost - Switching to offline mode');
        showToast('info', 'Offline', 'No internet connection');
        setLastOnlineStatus(false);
        setHasShownOfflineToast(true);
        setHasShownOnlineToast(false);
      }
    });

    const loadInitialData = async () => {
      try {
        // Check network status correctly
        const netInfo = await NetInfo.fetch();
        const connected = netInfo.isConnected && netInfo.isInternetReachable !== false;
        setIsOnline(connected);
        setLastOnlineStatus(connected);
        
        // Show appropriate loading modal
      setSyncModalVisible(true);
      
        if (!connected) {
          // OFFLINE MODE: Show special offline loader
          console.log('═══════════════════════════════════════');
          console.log('📴 OFFLINE MODE - Loading from cache...');
          console.log('═══════════════════════════════════════');
          
          // Step 1: Load ALL cached data sources
          const userId = await AsyncStorage.getItem('userUid');
          
          // Load permissions from ALL sources
          let permissionsLoaded = false;
          let loadedUserInfoData = null;
          
          // Try inspectionModuleUserInfo
          const cachedInspectionUserInfo = await AsyncStorage.getItem('inspectionModuleUserInfo');
          if (cachedInspectionUserInfo) {
            loadedUserInfoData = JSON.parse(cachedInspectionUserInfo);
            permissionsLoaded = true;
            console.log('✅ Found permissions in inspectionModuleUserInfo:', loadedUserInfoData.inspectionPermissions);
          }
          
          // Try userData
          if (!permissionsLoaded) {
            const cachedUserData = await AsyncStorage.getItem('userData');
            if (cachedUserData) {
              const userData = JSON.parse(cachedUserData);
              const inspectionModule = userData.modules?.find((module) => module.module === 'inspectionReport');
              
              loadedUserInfoData = {
                countries: Array.isArray(userData.countries) ? userData.countries : [],
                projects: Array.isArray(userData.projects) ? userData.projects : [],
                inspectionPermissions: inspectionModule ? (Array.isArray(inspectionModule.permissions) ? inspectionModule.permissions : []) : [],
                employeeNumber: userData.employeeNumber || '',
                name: userData.name || '',
              };
              permissionsLoaded = true;
              console.log('✅ Found permissions in userData:', loadedUserInfoData.inspectionPermissions);
            }
          }
          
          // Try cached reports
          if (!permissionsLoaded) {
            const cachedReports = await AsyncStorage.getItem('inspectionReports');
            if (cachedReports) {
              const reports = JSON.parse(cachedReports);
              if (reports.length > 0 && reports[0].userPermissions) {
                loadedUserInfoData = {
                  countries: reports[0].userCountries || [],
                  projects: reports[0].userProjects || [],
                  inspectionPermissions: reports[0].userPermissions || [],
                  employeeNumber: '',
                  name: '',
                };
                permissionsLoaded = true;
                console.log('✅ Found permissions in cached reports:', loadedUserInfoData.inspectionPermissions);
              }
            }
          }
          
          // Set permissions immediately and force update
          if (loadedUserInfoData && permissionsLoaded) {
            setUserInfo(loadedUserInfoData);
            setLoadingUserInfo(false);
            
            // Store back to cache to ensure it's available
            await AsyncStorage.setItem('inspectionModulePermissions', JSON.stringify(loadedUserInfoData.inspectionPermissions));
            await AsyncStorage.setItem('inspectionModuleUserInfo', JSON.stringify(loadedUserInfoData));
            
            console.log('✅ OFFLINE - Permissions loaded successfully:');
            console.log('   Permissions:', loadedUserInfoData.inspectionPermissions?.join(', '));
            console.log('   Countries:', loadedUserInfoData.countries?.join(', '));
            console.log('   Projects:', loadedUserInfoData.projects?.length, 'projects');
            console.log('   Name:', loadedUserInfoData.name || 'N/A');
            console.log('   Employee #:', loadedUserInfoData.employeeNumber || 'N/A');
          } else {
            console.log('⚠️ NO PERMISSIONS FOUND IN OFFLINE STORAGE!');
            console.log('⚠️ Checking storage keys...');
            
            // Debug: Check what keys exist
            const allKeys = await AsyncStorage.getAllKeys();
            const relevantKeys = allKeys.filter(k => 
              k.includes('user') || 
              k.includes('inspection') || 
              k.includes('permission')
            );
            console.log('📦 Available storage keys:', relevantKeys);
          }
          
          // Step 2: Auto-cleanup reports without dates FIRST
          console.log('🧹 Running auto-cleanup on page load...');
          await cleanupReportsWithoutDates();
          
          // Step 3: Load reports from cache
          await loadReports(false);
          await updateReportCounts();
          
          // Step 4: Calculate actual counts
          const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
          const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
          const offlinePendingReports = await OfflineDataHelper.getOfflineReports();
          const totalReportsCount = cachedReports.length + offlinePendingReports.length;
          const permissionsCount = loadedUserInfoData?.inspectionPermissions?.length || 0;
          
          // Step 4: Hide modal after loading
          setSyncModalVisible(false);
          setHasCompletedInitialLoad(true);
          
          // Step 5: Show completion toast
          if (!hasShownOfflineToast) {
            showToast('success', 'Offline Mode', `${totalReportsCount} reports & ${permissionsCount} permissions loaded`);
            setHasShownOfflineToast(true);
          }
          
          console.log('═══════════════════════════════════════');
          console.log(`📊 INSPECTION HOME LOADED | 📴 Offline`);
          console.log(`📋 Permissions: ${loadedUserInfoData?.inspectionPermissions?.join(', ') || 'NONE'}`);
          console.log(`🌍 Countries: ${loadedUserInfoData?.countries?.join(', ') || 'NONE'}`);
          console.log(`📁 Projects: ${loadedUserInfoData?.projects?.length || 0} projects`);
          console.log(`📊 Cached Reports: ${cachedReports.length}`);
          console.log(`📊 Pending Reports: ${offlinePendingReports.length}`);
          console.log(`📊 TOTAL Reports: ${totalReportsCount}`);
          console.log('═══════════════════════════════════════');
          
        } else {
          // ONLINE MODE: Normal loading
          await fetchUserInfo();
          await new Promise((resolve) => setTimeout(resolve, 200));
          
          // Auto-cleanup reports without dates FIRST (before loading)
          console.log('🧹 Running auto-cleanup on page load (online)...');
          await cleanupReportsWithoutDates();
          
          await loadReports(false);
          
          // Hide modal - data is now visible
          setSyncModalVisible(false);
          setHasCompletedInitialLoad(true);
          
          console.log('═══════════════════════════════════════');
          console.log(`📊 INSPECTION HOME LOADED | 🌐 Online`);
          console.log('═══════════════════════════════════════');
          
          // NO BACKGROUND SYNC ON INITIAL LOAD
          // User must manually sync using the cloud button or refresh
          console.log('ℹ️ Use the cloud sync button to upload offline reports');
        }
      } catch (error) {
        console.error('❌ Initial load error:', error.message);
        
        // Cache fallback
        try {
          await fetchUserInfo();
          await loadReports(false);
          await updateReportCounts();
        } catch (cacheError) {
          // Silent fail
        }
      } finally {
          setSyncModalVisible(false);
        setHasCompletedInitialLoad(true);
      }
    };
    
    loadInitialData();

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, [lastOnlineStatus]);

  // Reload data when returning to inspection module page
  useFocusEffect(
    useCallback(() => {
      if (!hasCompletedInitialLoad) return;
      
      const reloadOnFocus = async () => {
        console.log('🔄 Inspection page focused - Reloading data from cache only...');
        
        // Reload user info and permissions from cache first
        await fetchUserInfo();
        
        // Auto-cleanup reports without dates when user returns to page
        console.log('🧹 Running auto-cleanup on page focus...');
        await cleanupReportsWithoutDates();
        
        // Reload reports from cache ONLY (no sync on focus)
        await loadReports(false);
        await updateReportCounts();
        
        console.log('✅ Data & permissions reloaded from cache (no auto-sync)');
      };
      
      reloadOnFocus();
    }, [hasCompletedInitialLoad])
  );

  useEffect(() => {
    const filterReports = async () => {
      // If no reports, skip
      if (!allReports || allReports.length === 0) {
        setReports([]);
        return;
      }
      
      // CRITICAL: Filter out reports without created dates AND duplicates (auto-cleanup)
      const reportsWithDates = allReports.filter(report => 
        report.createdAt || report.step3?.createdAt || report.offlineCreatedAt
      );
      
      // Remove duplicate requestNumbers (keep the newest/best one)
      const requestNumberMap = new Map();
      const uniqueReports = [];
      
      for (const report of reportsWithDates) {
        const requestNumber = report.step1?.requestNumber || report.requestNumber;
        
        if (!requestNumber) {
          // No request number, keep it anyway
          uniqueReports.push(report);
          continue;
        }
        
        if (!requestNumberMap.has(requestNumber)) {
          // First occurrence of this requestNumber
          requestNumberMap.set(requestNumber, report);
          uniqueReports.push(report);
        } else {
          // Duplicate found - check which one to keep
          const existing = requestNumberMap.get(requestNumber);
          const existingDate = existing.createdAt || existing.step3?.createdAt || existing.offlineCreatedAt;
          const currentDate = report.createdAt || report.step3?.createdAt || report.offlineCreatedAt;
          
          // If current has date and existing doesn't, replace
          if (currentDate && !existingDate) {
            const index = uniqueReports.findIndex(r => 
              (r.id || r._id) === (existing.id || existing._id)
            );
            if (index !== -1) {
              uniqueReports[index] = report;
              requestNumberMap.set(requestNumber, report);
            }
          } else if (currentDate && existingDate) {
            // Both have dates, keep newer one
            if (new Date(currentDate) > new Date(existingDate)) {
              const index = uniqueReports.findIndex(r => 
                (r.id || r._id) === (existing.id || existing._id)
              );
              if (index !== -1) {
                uniqueReports[index] = report;
                requestNumberMap.set(requestNumber, report);
              }
            }
          }
          // If existing has date and current doesn't, keep existing (do nothing)
        }
      }
      
      const reportsToDisplay = uniqueReports;
      
      // If userInfo not loaded yet, show ALL reports without filtering (offline mode)
      if (!userInfo || !userInfo.inspectionPermissions || userInfo.inspectionPermissions.length === 0) {
        console.log('⏸️ UserInfo not ready - showing ALL reports without filtering');
        console.log(`   Total reports to show: ${reportsToDisplay.length}`);
        
        // Show all reports anyway (for offline mode)
        const sorted = reportsToDisplay.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        const paginated = sorted.slice(0, page * PAGE_SIZE);
        setReports(paginated);
        setHasMore(sorted.length > paginated.length);
        console.log(`   Displayed ${paginated.length} reports without filtering`);
        return;
      }
      
      let userUid = await AsyncStorage.getItem('userUid');
      let filtered = reportsToDisplay;
      
      console.log('🔍 Filtering reports:');
      console.log(`   Total reports (unique, with dates): ${reportsToDisplay.length}`);
      console.log(`   Total reports (all): ${allReports.length}`);
      console.log(`   User permissions: ${userInfo.inspectionPermissions?.join(', ')}`);
      console.log(`   User countries: ${userInfo.countries?.join(', ')}`);
      console.log(`   User projects: ${userInfo.projects?.length} projects`);
      console.log(`   Is Online: ${isOnline}`);
      
      // If user has onlyMineView permission, we already filtered by userId in the API call
      // If user has view permission, we need to filter by their countries and projects
      // BUT: Skip filtering when offline to show all cached reports
      if (isOnline && userInfo.inspectionPermissions.includes('view') && !userInfo.inspectionPermissions.includes('onlyMineView')) {
        // Filter reports based on country and project matching for all users' reports
        const beforeCount = filtered.length;
        filtered = reportsToDisplay.filter(
          (report) => {
            const countryMatch = userInfo.countries.includes(report.step1?.country);
            const projectMatch = userInfo.projects.includes(report.step1?.project);
            return countryMatch && projectMatch;
          }
        );
        console.log(`   Filtered by country/project: ${filtered.length}/${beforeCount}`);
      } else if (!isOnline) {
        console.log(`   ⚠️ OFFLINE MODE: Skipping country/project filtering to show all cached reports`);
      }
      // For onlyMineView, we already have only the user's reports, so no additional filtering needed

      if (searchQuery.trim() !== '') {
        filtered = filtered.filter((report) => {
          if (selectedFilter === 'Location') {
            return report.step1?.country?.toLowerCase().includes(searchQuery.toLowerCase());
          } else if (selectedFilter === 'Inspector') {
            return report.step1?.inspector?.toLowerCase().includes(searchQuery.toLowerCase());
          } else if (selectedFilter === 'Equipment') {
            return report.step1?.selectedEquipment?.equipmentName?.toLowerCase().includes(searchQuery.toLowerCase());
          } else if (selectedFilter === 'Project') {
            return report.step1?.project?.toLowerCase().includes(searchQuery.toLowerCase());
          }
          return (
            report.step1?.country?.toLowerCase().includes(searchQuery.toLowerCase()) ||
            report.step1?.inspector?.toLowerCase().includes(searchQuery.toLowerCase()) ||
            report.step1?.selectedEquipment?.equipmentName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
            report.step1?.project?.toLowerCase().includes(searchQuery.toLowerCase())
          );
        });
      }

      const sorted = filtered.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
      const paginated = sorted.slice(0, page * PAGE_SIZE);
      setReports(paginated);
      setHasMore(sorted.length > paginated.length);
      
      console.log(`✅ Final filtered reports: ${paginated.length}`);
    };
    filterReports();
  }, [allReports, userInfo, searchQuery, selectedFilter, page, isOnline]);

  const handleLoadMore = () => {
    if (!loadingMore && hasMore) {
      setLoadingMore(true);
      setTimeout(() => {
        setPage((prev) => prev + 1);
        setLoadingMore(false);
      }, 300);
    }
  };

  const handleRefresh = async () => {
    if (!isMounted.current) return;
    
    console.log('🔄 Manual refresh triggered');
    
    const onlineStatus = await checkNetworkStatus();
    if (!onlineStatus) {
      // OFFLINE: Just reload from cache
      showToast('info', 'Offline Mode', 'Reloading from cache');
      setSyncModalVisible(true);
      setPage(1);
      
      try {
        await fetchUserInfo();
        await loadReports(false);
        await updateReportCounts();
        setSearchQuery('');
        setSelectedFilter('');
      } catch (error) {
        console.error('❌ Offline refresh failed:', error);
      } finally {
        setSyncModalVisible(false);
      }
      return;
    }
    
    // ONLINE: Full refresh from MongoDB
    setSyncModalVisible(true);
    setIsRefreshing(true);
    setPage(1);
    
    // Clear current reports to force fresh load
    setReports([]);
    setAllReports([]);
    
    try {
      // Load fresh reports
      await loadReports(true);
      setSearchQuery('');
      setSelectedFilter('');
    } catch (error) {
      console.error('❌ Manual refresh failed:', error);
      showToast('error', 'Refresh Failed', 'Failed to refresh reports');
    } finally {
      setSyncModalVisible(false);
    }
  };

  // Enhanced sync function for cloud sync button - Wrapped in useCallback for auto-sync
  const handleCloudSync = useCallback(async () => {
    if (isSyncing) return;
    if (!isMounted.current) return;
    
    // Check userInfo is loaded
    if (!userInfo || !userInfo.inspectionPermissions) {
      showToast('error', 'Not Ready', 'User information not loaded. Please wait...');
      return;
    }
    
    try {
      setIsSyncing(true);
      setSyncModalVisible(true);
      
      // Check if online
      const onlineStatus = await checkNetworkStatus();
      setIsOnline(onlineStatus);
      
      if (!onlineStatus) {
        showToast('error', 'Offline', 'Cannot sync while offline');
        setIsSyncing(false);
        setSyncModalVisible(false);
        return;
      }

      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Error', 'No user is logged in');
        setIsSyncing(false);
        setSyncModalVisible(false);
        return;
      }

      // Sync pending operations first
      console.log('🔄 Step 1: Syncing pending operations...');
      await syncPendingOperations();
      
      console.log('🔄 Step 2: Syncing equipment updates...');
      await syncPendingEquipmentUpdates();
      
      console.log('🔄 Step 3: Syncing offline reports with PDFs...');
      // Wait if another sync is in progress
      let attempts = 0;
      while (isSyncingOfflineReports.current && attempts < 10) {
        console.log('⏳ Waiting for ongoing sync to complete...');
        await new Promise(resolve => setTimeout(resolve, 500));
        attempts++;
      }
      
      const offlineSyncResult = await syncOfflineReports();
      
      if (offlineSyncResult && offlineSyncResult.synced > 0) {
        console.log(`✅ Synced ${offlineSyncResult.synced} offline reports with PDFs to MongoDB`);
      }
      
      // Load fresh reports from MongoDB based on permissions
      let response;
      if (userInfo.inspectionPermissions.includes('onlyMineView')) {
        response = await InspectionAPI.getInspections(userId, { page: 1, limit: 1000 });
      } else if (userInfo.inspectionPermissions.includes('view')) {
        try {
          response = await InspectionAPI.getAllReports({ page: 1, limit: 1000 });
          
          if (!response.success || !response.data || response.data.length === 0) {
            const knownUserResponse = await InspectionAPI.getInspections('test_proadmin_amit_001', { page: 1, limit: 1000 });
            if (knownUserResponse.success && knownUserResponse.data && knownUserResponse.data.length > 0) {
              response = knownUserResponse;
            }
          }
        } catch (error) {
          response = await InspectionAPI.getInspections('test_proadmin_amit_001', { page: 1, limit: 1000 });
        }
      } else {
        response = { success: true, data: [] };
      }
      
      let reportsArray = [];
      
      if (response.success && response.data) {
        reportsArray = response.data.map(report => ({
          ...report,
          id: report._id,
          adminId: report.userId || report.createdBy,
          createdAt: report.createdAt || new Date().toISOString(),
          // CRITICAL: Store permissions with each report for offline access
          userPermissions: userInfo.inspectionPermissions || [],
          userCountries: userInfo.countries || [],
          userProjects: userInfo.projects || [],
          syncedAt: new Date().toISOString(),
          syncedBy: userId,
        }));
        
        // Store in offline shared storage
        await AsyncStorage.setItem('inspectionReports', JSON.stringify(reportsArray));
        
        // Store each report individually with permissions
        for (const report of reportsArray) {
          try {
            await AsyncStorage.setItem(`report_${report.id}`, JSON.stringify(report));
          } catch (error) {
            // Silent fail
          }
        }
        
        // Store sync metadata
        const syncTime = new Date().toISOString();
        const syncMetadata = {
          lastSyncTime: syncTime,
          totalReports: reportsArray.length,
          syncedBy: userId,
          syncSource: 'MongoDB',
          isOnline: true
        };
        await AsyncStorage.setItem('inspectionReportsLastSync', syncTime);
        await AsyncStorage.setItem('inspectionSyncMetadata', JSON.stringify(syncMetadata));
        setLastSyncTime(syncTime);
      }

      // Load offline reports and append (preserve permissions)
      const offlineReports = await OfflineDataHelper.getOfflineReports();
      if (offlineReports.length > 0) {
        const formattedOfflineReports = offlineReports.map(report => ({
          ...report,
          id: report._id || report.id,
          adminId: report.userId || report.createdBy,
          createdAt: report.createdAt || report.offlineCreatedAt,
          offlineCreated: true,
          offlineSyncPending: true,
          // Preserve permissions from offline report or use current userInfo
          userPermissions: report.userPermissions || userInfo.inspectionPermissions || [],
          userCountries: report.userCountries || userInfo.countries || [],
          userProjects: report.userProjects || userInfo.projects || [],
        }));
        reportsArray = [...reportsArray, ...formattedOfflineReports];
      }
      
      // CRITICAL: Remove duplicates by ID before updating state
      const uniqueReportsMap = new Map();
      reportsArray.forEach(report => {
        const reportId = report.id || report._id;
        if (!uniqueReportsMap.has(reportId)) {
          uniqueReportsMap.set(reportId, report);
        }
      });
      reportsArray = Array.from(uniqueReportsMap.values());
      
      // Update state
      setAllReports(reportsArray);
      await loadJobCardsForInspections();
      await fetchUserInfo();
      await loadEmailStatistics();
      await updateReportCounts();
      
      // Show result with permissions information
      const storedCount = response.data?.length || 0;
      const totalOffline = storedCount + offlineReports.length;
      
      // Get stored permissions info
      const permissionsInfo = userInfo.inspectionPermissions?.join(', ') || 'None';
      const countriesInfo = userInfo.countries?.join(', ') || 'None';
      const projectsInfo = userInfo.projects?.length > 3 
        ? `${userInfo.projects.slice(0, 3).join(', ')}... (+${userInfo.projects.length - 3} more)` 
        : userInfo.projects?.join(', ') || 'None';
      
      console.log('═══════════════════════════════════════');
      console.log(`✅ SYNC COMPLETE | ${totalOffline} reports offline`);
      console.log(`📋 Permissions: ${permissionsInfo}`);
      console.log('═══════════════════════════════════════');
      
      showToast('success', 'Synced', `${totalOffline} reports now available offline`);
      
      // Show detailed sync result
      const offlineSyncedCount = offlineSyncResult?.synced || 0;
      const offlineFailedCount = offlineSyncResult?.failed || 0;
      
      Alert.alert(
        '✅ Sync Successful',
        `MongoDB ↔ Offline Storage\n\n` +
        `REPORTS SYNCED:\n` +
        `✓ ${storedCount} reports downloaded from MongoDB\n` +
        `✓ ${offlineSyncedCount} offline reports uploaded to MongoDB\n` +
        (offlineFailedCount > 0 ? `⚠ ${offlineFailedCount} offline reports failed to sync\n` : '') +
        `✓ ${offlineReports.length} reports still pending sync\n` +
        `✓ TOTAL Offline: ${totalOffline} reports\n` +
        `✓ Storage size: ${localStorageSize}\n\n` +
        `PERMISSIONS STORED:\n` +
        `✓ ${permissionsInfo}\n\n` +
        `COUNTRIES:\n` +
        `✓ ${countriesInfo}\n\n` +
        `PROJECTS:\n` +
        `✓ ${projectsInfo}\n\n` +
        `All ${totalOffline} reports & PDFs accessible offline!`,
        [{ text: 'OK' }]
      );
      
    } catch (error) {
      console.error('❌ Error during cloud sync:', error);
      showToast('error', 'Sync Failed', 'Could not sync from MongoDB');
    } finally {
      if (isMounted.current) {
        setIsSyncing(false);
        setSyncModalVisible(false);
      }
    }
  }, [isSyncing, userInfo, syncPendingOperations, loadJobCardsForInspections, fetchUserInfo, loadEmailStatistics, showToast]);

  const handleSearch = (text) => {
    setSearchQuery(text);
    setPage(1);
  };

  // Helper function to properly encode PDF URLs
  const encodePdfUrl = (url) => {
    if (!url) return null;
    
    // If it's a local file path, don't encode
    if (url.startsWith('file://') || url.startsWith('/')) {
      return url;
    }
    
    // If it's an HTTP URL with special characters, encode the path parts
    try {
      const urlObj = new URL(url);
      const pathParts = urlObj.pathname.split('/');
      const encodedParts = pathParts.map((part, index) => {
        // Don't encode the first empty part or route segments
        if (index === 0 || part === 'inspections' || part === 'download-file') {
          return part;
        }
        // Encode email addresses and filenames
        return encodeURIComponent(part);
      });
      urlObj.pathname = encodedParts.join('/');
      return urlObj.toString();
    } catch (e) {
      // If URL parsing fails, return original
      console.warn('⚠️ Failed to encode URL:', e.message);
      return url;
    }
  };

  const handleViewReport = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    if (!userInfo.inspectionPermissions.includes('view') && !userInfo.inspectionPermissions.includes('onlyMineView')) {
      showToast('error', 'Permission Denied', 'You do not have permission to view reports.');
      return;
    }

    try {
      const adminId = report.userId || report.createdBy || report.adminId;
      if (!adminId) {
        showToast('error', 'Error', 'Report admin ID not found.');
        return;
      }

      let pdfUrl = null;
        
      if (isOnline) {
        // Online - fetch from MongoDB
        console.log('🌐 Online: Fetching report from MongoDB');
        let response;
        if (userInfo.inspectionPermissions.includes('onlyMineView')) {
          response = await InspectionAPI.getInspection(adminId, report.id);
        } else {
          response = await InspectionAPI.getInspectionById(report.id);
        }
        
        if (response.success && response.data) {
          pdfUrl = response.data.step3?.pdfDownloadUrl;
          console.log('📄 PDF URL from MongoDB (raw):', pdfUrl);
          
          // Encode URL if it contains special characters
          if (pdfUrl) {
            pdfUrl = encodePdfUrl(pdfUrl);
            console.log('📄 PDF URL (encoded):', pdfUrl);
          }
          
          if (!pdfUrl) {
            // Try AsyncStorage for local PDF path (stored with request number key)
            try {
              const requestNum = response.data.requestNumber || response.data.step1?.requestNumber || report.requestNumber || report.step1?.requestNumber;
              if (requestNum) {
                const pdfPathKey = `inspection_pdf_local_${requestNum}`;
                const storedPdfPath = await AsyncStorage.getItem(pdfPathKey);
                if (storedPdfPath) {
                  const RNFS = require('react-native-fs');
                  const fileExists = await RNFS.exists(storedPdfPath);
                  if (fileExists) {
                    pdfUrl = Platform.OS === 'android' ? `file://${storedPdfPath}` : storedPdfPath;
                    console.log('✅ Using local PDF from AsyncStorage:', pdfUrl);
                    showToast('info', 'Local PDF', 'Viewing locally stored PDF');
                  }
                }
              }
            } catch (asyncError) {
              console.warn('⚠️ AsyncStorage PDF lookup failed:', asyncError);
            }
            
            // Try local PDF as fallback (for offline-created reports not yet synced)
            if (!pdfUrl && report.step3?.pdfLocalPath) {
              const RNFS = require('react-native-fs');
              const fileExists = await RNFS.exists(report.step3.pdfLocalPath);
              if (fileExists) {
                pdfUrl = report.step3.pdfLocalPath;
                console.log('📱 Using local PDF (not yet synced):', pdfUrl);
                showToast('info', 'Local PDF', 'Viewing locally stored PDF (not yet uploaded)');
              }
            }
            
            if (!pdfUrl) {
              showToast('error', 'No PDF Available', 'This report does not have a PDF generated yet.');
              return;
            }
          }
        } else {
          showToast('error', 'Database Error', 'Failed to fetch report from database.');
          return;
        }
      } else {
        // Offline - check local storage
        console.log('📴 Offline: Looking for local PDF');
        
        // Priority 1: Check AsyncStorage with request number key
        try {
          const requestNum = report.requestNumber || report.step1?.requestNumber;
          if (requestNum) {
            const pdfPathKey = `inspection_pdf_local_${requestNum}`;
            const storedPdfPath = await AsyncStorage.getItem(pdfPathKey);
            if (storedPdfPath) {
              const RNFS = require('react-native-fs');
              const fileExists = await RNFS.exists(storedPdfPath);
              if (fileExists) {
                pdfUrl = Platform.OS === 'android' ? `file://${storedPdfPath}` : storedPdfPath;
                console.log('✅ Found PDF in AsyncStorage:', pdfUrl);
              }
            }
          }
        } catch (asyncError) {
          console.warn('⚠️ AsyncStorage check failed:', asyncError);
        }
        
        // Priority 2: Check if it's an offline-created report with local PDF
        if (!pdfUrl && report.offlineCreated && report.step3?.pdfLocalPath) {
          const RNFS = require('react-native-fs');
          const fileExists = await RNFS.exists(report.step3.pdfLocalPath);
          if (fileExists) {
            pdfUrl = report.step3.pdfLocalPath;
            console.log('✅ Found offline PDF at:', pdfUrl);
          }
        }
        
        // Priority 3: Check OfflineDataHelper
        if (!pdfUrl) {
          const offlinePdfPath = await OfflineDataHelper.getOfflinePDF(report.id);
          if (offlinePdfPath) {
            pdfUrl = offlinePdfPath;
            console.log('✅ Found PDF in OfflineDataHelper:', pdfUrl);
          }
        }
        
        // Priority 4: Check AsyncStorage for synced report
        if (!pdfUrl) {
          const reportRaw = await AsyncStorage.getItem(`report_${report.id}`);
          if (reportRaw) {
            const reportData = JSON.parse(reportRaw);
            pdfUrl = reportData.step3?.pdfLocalPath || reportData.step3?.pdfDownloadUrl;
            if (pdfUrl) {
              pdfUrl = encodePdfUrl(pdfUrl);
            }
            console.log('✅ Found PDF in AsyncStorage report data:', pdfUrl);
          }
        }
      }

      if (!pdfUrl) {
        showToast('error', 'Error', isOnline ? 'PDF URL not available in database.' : 'PDF not available offline.');
        return;
      }
      
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored report.');
      }

      console.log('🚀 Opening PDF with URL:', pdfUrl);
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('❌ View PDF error:', error.message);
      
      let errorMessage = 'Could not fetch the PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection.';
      } else if (error.message.includes('404')) {
        errorMessage = 'Report not found in database.';
      }
      
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewCorrectivePDF = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    try {
      let pdfUrl = null;
      
      if (isOnline) {
        // Try to get corrective action data from MongoDB
        const reportId = report._id || report.id;
        console.log('🔍 Fetching corrective action for report:', reportId);
        const response = await InspectionAPI.getCorrectiveAction(reportId);
        console.log('📡 Corrective action response:', response);
        
        if (response.success && response.data?.pdfDownloadUrl) {
          pdfUrl = response.data.pdfDownloadUrl;
          console.log('✅ Found corrective action PDF in MongoDB (raw):', pdfUrl);
        } else {
          console.log('⚠️ No PDF URL found in corrective action data');
        }
      }
      
      // Fallback to report data
      if (!pdfUrl) {
        pdfUrl = report.correctiveAction?.pdfDownloadUrl || report.correctiveActionData?.pdfDownloadUrl;
        console.log('📱 Fallback PDF URL from report data (raw):', pdfUrl);
      }
      
      if (!pdfUrl) {
        showToast('error', 'Error', 'No Corrective Action PDF found for this report.');
        return;
      }

      // Encode URL to handle special characters
      pdfUrl = encodePdfUrl(pdfUrl);
      console.log('📄 Corrective PDF URL (encoded):', pdfUrl);

      // Add userId parameter to corrective action PDF URLs if missing
      if (pdfUrl.includes('/corrective-actions/download-pdf/') && !pdfUrl.includes('userId=')) {
        const separator = pdfUrl.includes('?') ? '&' : '?';
        pdfUrl = `${pdfUrl}${separator}userId=${userInfo?.userId || 'test_proadmin_amit_001'}`;
        console.log('🔧 Added userId to corrective action URL:', pdfUrl);
      }

      console.log('🚀 Navigating to PDF viewer with URL:', pdfUrl);
      
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored corrective action.');
      }
      
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        userId: userInfo?.userId || 'test_proadmin_amit_001',
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching corrective PDF URL:', error);
      let errorMessage = 'Could not fetch the corrective PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewRiskAssessmentPDF = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }
    try {
      const adminId = report.adminId;
      if (!adminId) {
        showToast('error', 'Error', 'Report admin ID not found.');
        return;
      }
      let pdfUrl = null;
      if (isOnline) {
        // TODO: Implement MongoDB risk assessment lookup
        pdfUrl = report.riskAssessment?.step3?.pdfDownloadUrl;
      } else {
        // No offline cache yet for risk assessment PDFs
        pdfUrl = report.riskAssessment?.step3?.pdfDownloadUrl || null;
      }
      if (!pdfUrl) {
        showToast('error', 'Error', isOnline ? 'Risk Assessment PDF not available.' : 'Risk Assessment PDF not available offline.');
        return;
      }
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching Risk Assessment PDF URL:', error);
      let errorMessage = 'Could not fetch the Risk Assessment PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the Risk Assessment PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewJobCardPDF = async (jobCard) => {
    if (!jobCard?.pdfDownloadUrl) {
      showToast('error', 'Error', 'No Job Card PDF URL found.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      let pdfUrl = jobCard.pdfDownloadUrl;
      
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored job card.');
      }

      console.log('🚀 Navigating to PDF viewer with Job Card URL:', pdfUrl);
      
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching job card PDF URL:', error);
      let errorMessage = 'Could not fetch the job card PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleViewJobCardPDFFromReport = async (report) => {
    const jobCard = jobCardsMap[report.id] || report.workOrder;
    if (jobCard) {
      await handleViewJobCardPDF(jobCard);
    } else {
      showToast('error', 'Error', 'No job card found for this inspection.');
    }
  };

  const handleViewJobCard = async (report) => {
    const jobCard = jobCardsMap[report.id] || report.workOrder;
    if (jobCard) {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      if (!isOnline) {
        showToast('error', 'Error', 'Offline mode. Please connect to the internet to view the job card.');
        return;
      }

      navigation.navigate('WorkerDashboardhome', {
        workOrderId: jobCard._id || jobCard.id,
        adminUid: jobCard.adminUid || userId,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } else {
      showToast('error', 'Error', 'No job card found for this inspection.');
    }
  };

  const handleViewWorkOrderPDF = async (report) => {
    if (!report?.workOrder?.pdfDownloadUrl) {
      showToast('error', 'Error', 'No Work Order PDF URL found for this report.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }
      let pdfUrl = null;
      if (isOnline) {
        // TODO: Implement MongoDB work order lookup
        pdfUrl = report.workOrder?.pdfDownloadUrl;
      } else {
        const reportRaw = await AsyncStorage.getItem(`report_${report.id}`);
        if (reportRaw) {
          const report = JSON.parse(reportRaw);
          pdfUrl = report.workOrder?.pdfDownloadUrl;
        }
      }

      if (!pdfUrl) {
        showToast('error', 'Error', isOnline ? 'PDF URL not available for this work order.' : 'Work Order PDF not available offline.');
        return;
      }

      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Viewing locally stored work order.');
      }

      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error fetching work order PDF URL:', error);
      let errorMessage = 'Could not fetch the work order PDF URL. Please try again.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the PDF.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleShare = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    try {
      let shareContent = null;
      if (isOnline && report.step3?.pdfDownloadUrl) {
        shareContent = {
          url: report.step3.pdfDownloadUrl,
          title: 'Inspection Report ' + report.step1.requestNumber,
          message: 'Inspection Report ' + report.step1.requestNumber + ' generated on ' + formatDate(report.step3?.createdAt),
        };
      } else {
        const reportRaw = await AsyncStorage.getItem(`report_${report.id}`);
        if (reportRaw) {
          const report = JSON.parse(reportRaw);
          shareContent = {
            message: 'Inspection Report ' + report.step1.requestNumber + '\nLocation: ' + report.step1.country + '\nProject: ' + report.step1.project + '\nInspector: ' + report.step1.inspector + '\nEquipment: ' + (report.step1.selectedEquipment?.equipmentName || 'N/A') + '\nCreated: ' + formatDate(report.step3?.createdAt),
          };
        }
      }

      if (!shareContent) {
        showToast('error', 'Error', 'Unable to share report. Data not available.');
        return;
      }

      await Share.share(shareContent);
      showToast('success', 'Success', 'Report shared successfully.');
    } catch (error) {
      console.error('Error sharing report:', error);
      showToast('error', 'Error', 'Failed to share report.');
    }
  };

  const loadJobCardsForInspections = async () => {
    try {
      const response = await JobCardAPI.getAllJobCards();
      
      if (response.success && response.data) {
        const jobCardsMap = {};
        
        response.data.forEach(jobCard => {
          const inspectionId = jobCard.linkInspection?.id || 
                              jobCard.linkInspection?._id ||
                              jobCard.serviceScheduleId ||
                              jobCard.inspectionId;
                              
          if (inspectionId) {
            jobCardsMap[inspectionId] = jobCard;
          }
        });
        
        setJobCardsMap(jobCardsMap);
      }
    } catch (error) {
      console.error('❌ Job cards error:', error.message);
    }
  };

  const checkExistingJobCard = async (inspectionId) => {
    try {
      const response = await JobCardAPI.getAllJobCards();
      if (response.success && response.data) {
        const existingJobCard = response.data.find(jobCard => 
          jobCard.linkInspection?.id === inspectionId || 
          jobCard.serviceScheduleId === inspectionId
        );
        return existingJobCard;
      }
      return null;
    } catch (error) {
      console.error('Error checking existing job card:', error);
      return null;
    }
  };

  const handleCreateWorkOrder = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    try {
      if (!isOnline) {
        showToast('error', 'Error', 'Offline mode. Cannot create work order.');
        return;
      }

      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      // Check if job card already exists for this inspection
      const existingJobCard = await checkExistingJobCard(report.id);
      if (existingJobCard) {
        Alert.alert(
          'Job Card Already Exists', 
          `A job card has already been created for this inspection.\n\nJob Card: ${existingJobCard.requestNumber || existingJobCard.title}\nStatus: ${existingJobCard.status}\nCreated: ${new Date(existingJobCard.createdAt).toLocaleDateString()}`,
          [
            { text: 'OK', style: 'default' },
            { 
              text: 'View Job Card', 
              onPress: () => {
                navigation.navigate('WorkerDashboardhome', {
                  workOrderId: existingJobCard._id || existingJobCard.id,
                  adminUid: existingJobCard.adminUid || userId,
                });
              }
            }
          ]
        );
        return;
      }

      const response = await MongoAPI.getUserById(userId);
      if (!response.success || !response.data) {
        throw new Error('User not found in admin database');
      }

      navigation.navigate('CreateInsWorkOrder', {
        selectedReport: {
          ...report,
          // Ensure all inspection data is passed
          step1: report.step1 || {},
          step2: report.step2 || {},
          step3: report.step3 || {},
        },
        selectedEquipment: report.step1?.selectedEquipment || report.step1?.equipment,
        onRefresh: handleRefresh,
        userId,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error checking work order existence:', error);
      showToast('error', 'Error', 'Failed to verify work order status.');
    }
  };

  const handleCorrectiveAction = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    if (!report.step2?.selections || !report.step2?.sections) {
      Alert.alert('Error', 'Report is missing required data for corrective action.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      // Note: Corrective action checking would need to be implemented in the backend
      // For now, we'll proceed with navigation
      navigation.navigate('CorrectiveactionHeader', {
        report,
        isUpdate: !!report.correctiveAction,
        userId,
        onRefresh: handleRefresh,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error checking corrective action:', error);
      showToast('error', 'Error', 'Failed to load corrective action data.');
    }
  };

  const handleArchive = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }
    Alert.alert('Archive', 'This feature is not yet implemented.');
  };

  const handleDelete = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }

    Alert.alert(
      'Delete Report',
      `Are you sure you want to delete Inspection #${report.step1?.requestNumber || report.id}?`,
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
            const userId = await AsyncStorage.getItem('userUid');
            if (!userId) {
              showToast('error', 'Authentication Error', 'No user is logged in.');
              return;
            }
            if (isOnline) {
              // Use delete-by-id endpoint for cross-user deletion (works for all users with delete permission)
              const response = await InspectionAPI.deleteInspectionById(report.id);
              if (response.success) {
                await AsyncStorage.removeItem(`report_${report.id}`);
                showToast('success', 'Success', 'Report deleted successfully.');
                
                // Reload the home page to refresh data
                console.log('🔄 Reloading inspection home after deletion...');
                await handleRefresh();
              } else {
                showToast('error', 'Error', response.error || 'Failed to delete report.');
                return;
              }
            } else {
                await queueOperation({
                  type: 'delete',
                  reportId: report.id,
                  userId,
                });
                await AsyncStorage.removeItem(`report_${report.id}`);
                showToast('info', 'Offline Mode', 'Deletion queued and will sync when online.');
                
                // Remove from state immediately in offline mode
                if (isMounted.current) {
                  setReports((prev) => prev.filter((r) => r.id !== report.id));
                  setAllReports((prev) => prev.filter((r) => r.id !== report.id));
                }
              }
          } catch (error) {
            console.error('Error deleting report:', error);
            let errorMessage = 'Failed to delete report';
            if (!isOnline) {
              errorMessage = 'Offline mode. Deletion will sync when online.';
            } else if (error.message) {
              errorMessage = error.message;
            }
            showToast('error', 'Error', errorMessage);
          }
          },
        },
      ]
    );
  };

  const handleRiskAssessment = async (report) => {
    if (!report) {
      showToast('error', 'Error', 'No report selected.');
      return;
    }
    navigation.navigate('Hazards_step1', {
      report,
      onGoBack: () => navigation.navigate('Inspection_Report_Home'),
    });
  };

  const handleWorkOrderNavigation = async (workOrderId, adminId) => {
    if (!workOrderId) {
      showToast('error', 'Error', 'Work order ID is missing.');
      return;
    }

    if (!adminId) {
      showToast('error', 'Error', 'Admin ID is missing.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }
      if (!isOnline) {
        showToast('error', 'Error', 'Offline mode. Please connect to the internet to view the work order.');
        return;
      }

      // Note: Work order checking would need to be implemented in the backend
      // For now, we'll proceed with navigation
      navigation.navigate('WorkerDashboardhome', {
        workOrderId,
        adminUid: adminId,
        onGoBack: () => navigation.navigate('Inspection_Report_Home'),
      });
    } catch (error) {
      console.error('Error navigating to work order:', error);
      let errorMessage = 'Failed to load work order details.';
      if (!isOnline) {
        errorMessage = 'Offline mode. Please connect to the internet to view the work order.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleDropdownAction = async (action, report) => {
    try {
      setIsProcessingAction(true);
      await new Promise(resolve => setTimeout(resolve, 100));
      
      switch (action) {
        case 'view':
          await handleViewReport(report);
          break;
        case 'share':
          await handleShare(report);
          break;
        case 'create-work-order':
          await handleCreateWorkOrder(report);
          break;
        case 'corrective':
          await handleCorrectiveAction(report);
          break;
        case 'archive':
          await handleArchive(report);
          break;
        case 'view-risk-assessment':
          await handleViewRiskAssessmentPDF(report);
          break;
        case 'risk-assessment':
          await handleRiskAssessment(report);
          break;
        case 'view-job-card':
          await handleViewJobCard(report);
          break;
        case 'view-job-card-pdf':
          await handleViewJobCardPDFFromReport(report);
          break;
        case 'delete':
          await handleDelete(report);
          break;
        default:
          break;
      }
    } catch (error) {
      console.error('Error processing action:', error);
      showToast('error', 'Error', 'Failed to process action.');
    } finally {
      setIsProcessingAction(false);
    }
  };

  const formatDate = (isoString) => {
    if (!isoString) return 'N/A';
    const date = new Date(isoString);
    return date.toLocaleString('en-US', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
    }).replace(',', '');
  };

  const renderReportItem = ({ item, index }) => {
    const dropdownPosition = index > reports.length - 4 ? 'top' : 'auto';
    let filteredOptions = DROPDOWN_OPTIONS.filter(option => {
      if (option.value === 'delete' && !userInfo.inspectionPermissions.includes('delete')) {
        return false;
      }
      return true;
    });

    // If a Risk Assessment PDF exists, show "View Risk Assessment" and hide the create entry
    const raPdf = item?.riskAssessment?.step3?.pdfDownloadUrl;
    if (raPdf) {
      // Keep existing "Risk Assessment" (for updates) and add a dedicated view option
      filteredOptions.push({
        label: 'View Risk Assessment',
        value: 'view-risk-assessment',
        icon: <MaterialIcons name="picture-as-pdf" size={22} color="#FF9800" />,
      });
    }

    // If a job card exists, add job card-related options
    const jobCard = jobCardsMap[item.id] || item.workOrder;
    if (jobCard) {
      filteredOptions.push({
        label: 'View Job Card',
        value: 'view-job-card',
        icon: <MaterialIcons name="build" size={22} color="#2ecc40" />,
      });
      if (jobCard.pdfDownloadUrl) {
        filteredOptions.push({
          label: 'Job Card PDF',
          value: 'view-job-card-pdf',
          icon: <MaterialIcons name="picture-as-pdf" size={22} color="#3498db" />,
        });
      }
    }

    return (
      <View style={styles.reportItem}>
        <View style={styles.reportHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.reportTitle}>{item.step1?.requestNumber || 'N/A'}</Text>
            
            {/* Offline Sync Pending Indicator */}
            {item.offlineSyncPending && (
              <View style={[styles.offlineBadge]}>
                <MaterialIcons name="cloud-upload" size={12} color="#fff" />
                <Text style={styles.offlineBadgeText}>SYNC</Text>
              </View>
            )}
            
            {/* Job Card Status Indicator */}
            {(() => {
              const jobCard = jobCardsMap[item.id] || item.workOrder;
              if (jobCard) {
                return (
                  <View style={[styles.jobCardIndicator, { 
                    backgroundColor: jobCard.status === 'Completed' ? '#2ecc40' : 
                                   jobCard.status === 'In Progress' ? '#f39c12' : '#3498db'
                  }]}>
                    <MaterialIcons name="build" size={12} color="#fff" />
                  </View>
                );
              }
              return null;
            })()}
          </View>
          <TouchableOpacity style={styles.statusButton}>
            <Text style={styles.statusButtonText}>{item.step3?.equipmentStatus || 'N/A'}</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.reportDetails}>
          <View style={{ flexDirection: 'row', alignItems: 'stretch' }}>
            <View style={{ flex: 1 }}>
              <View style={styles.detailItem}>
                <Text style={styles.detailLabel}>Inspector</Text>
                <Text style={[styles.detailValue, styles.detailValueLeft]}>{item.step1?.inspector || 'N/A'}</Text>
              </View>
              <View style={styles.detailItem}>
                <Text style={styles.detailLabel}>Location</Text>
                <Text style={[styles.detailValue, styles.detailValueLeft]}>{item.step1?.country || 'N/A'}</Text>
              </View>
            </View>
            <View style={{ justifyContent: 'center', alignItems: 'center', width: 70 }}>
              <View style={styles.faultBadgeContainer}>
                <Text style={styles.faultBadgeText}>{item.step3?.faultyItems || 0}</Text>
                <Text style={styles.faultBadgeLabel}>Fault</Text>
              </View>
            </View>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Project</Text>
            <Text style={styles.detailValue}>{item.step1?.project || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValue}>{item.step1?.selectedEquipment?.equipmentName || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Corrective Action</Text>
            <Text style={[styles.detailValue, { color: (item.correctiveActionData?.pdfDownloadUrl || item.correctiveActionData?.mechanicSignatureUri || item.correctiveActionData?.operatorSignatureUri) ? '#2ecc40' : '#e74c3c' }]}>
              {(item.correctiveActionData?.pdfDownloadUrl || item.correctiveActionData?.mechanicSignatureUri || item.correctiveActionData?.operatorSignatureUri) ? 'Taken' : 'Not taken'}
            </Text>
          </View>
          {(item.correctiveActionData?.pdfDownloadUrl || item.correctiveActionData?.mechanicSignatureUri || item.correctiveActionData?.operatorSignatureUri) && (
            <View style={styles.detailItem}>
              <Text style={styles.detailLabel}>Corrective Action No</Text>
              <TouchableOpacity onPress={() => handleDropdownAction('corrective', item)}>
                <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>CA-{item.step1?.requestNumber || (item._id || item.id)}</Text>
              </TouchableOpacity>
            </View>
          )}
          {(item.correctiveActionData?.pdfDownloadUrl) && (
            <View style={styles.detailItem}>
              <Text style={styles.detailLabel}>Corrective PDF</Text>
              <TouchableOpacity onPress={() => handleViewCorrectivePDF(item)}>
                <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>View PDF</Text>
              </TouchableOpacity>
            </View>
          )}
          {/* Auto-show job card information from MongoDB */}
          {(() => {
            const jobCard = jobCardsMap[item.id] || item.workOrder;
            if (jobCard) {
              return (
                <>
                  <View style={styles.detailItem}>
                    <Text style={styles.detailLabel}>Job Card</Text>
                    <Text style={[styles.detailValue, { color: '#2ecc40', fontWeight: 'bold' }]}>
                      {jobCard.requestNumber || jobCard.title || 'Created'}
                    </Text>
                  </View>
                  {jobCard.status && (
                    <View style={styles.detailItem}>
                      <Text style={styles.detailLabel}>Job Card Status</Text>
                      <Text style={[styles.detailValue, { 
                        color: jobCard.status === 'Completed' ? '#2ecc40' : 
                               jobCard.status === 'In Progress' ? '#f39c12' : '#3498db',
                        fontWeight: 'bold'
                      }]}>
                        {jobCard.status}
                      </Text>
                    </View>
                  )}
                  {jobCard.pdfDownloadUrl && (
                    <View style={styles.detailItem}>
                      <Text style={styles.detailLabel}>Job Card PDF</Text>
                      <TouchableOpacity onPress={() => handleViewJobCardPDF(jobCard)}>
                        <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>View PDF</Text>
                      </TouchableOpacity>
                    </View>
                  )}
                  <View style={styles.detailItem}>
                    <Text style={styles.detailLabel}>Job Card Track</Text>
                    <TouchableOpacity onPress={() => {
                      handleWorkOrderNavigation(
                        jobCard._id || jobCard.id, 
                        jobCard.adminUid || jobCard.adminId
                      );
                    }}>
                      <Text style={[styles.detailValue, { color: '#0078D4', textDecorationLine: 'underline' }]}>Track Job Card</Text>
                    </TouchableOpacity>
                  </View>
                </>
              );
            }
            return null;
          })()}
          <View style={styles.detailItem}>
            <Text style={[styles.detailLabel, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.detailValue}>{formatDate(item.step3?.createdAt)}</Text>
              <Dropdown
                style={styles.dropdown}
                data={filteredOptions}
                labelField="label"
                valueField="value"
                value={null}
                onChange={async (option) => {
                  await new Promise(resolve => setTimeout(resolve, 50));
                  handleDropdownAction(option.value, item);
                }}
                renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
        </View>
      </View>
    );
  };

  return (
    <ErrorBoundary>
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <TouchableOpacity
                onPress={() => {
                  Alert.alert(
                    'Go Back',
                    'Are you sure you want to go back? Your changes may not be saved.',
                    [
                      { text: 'No', style: 'cancel' },
                      {
                        text: 'Yes',
                        onPress: () => navigation.reset({
                          index: 0,
                          routes: [{ name: 'MainApp' }],
                        }),
                        style: 'destructive',
                      },
                    ]
                  );
                }}
              >
                <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>Inspection Report</Text>
            </View>
            <View style={styles.headerRight}>
              {!isOnline && (
                <View style={styles.offlineIndicator}>
                  <MaterialIcons name="wifi-off" size={width * 0.05} color="#ffd205" />
                </View>
              )}
              <TouchableOpacity 
                style={[styles.iconButton, isSyncing && styles.syncingButton]} 
                onPress={handleCloudSync}
                disabled={isSyncing}
              >
                <Image
                  source={require('../../../Images/adminhome/equp/cloud_1.png')}
                  style={[styles.icons12, isSyncing && styles.syncingIcon]}
                />
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={async () => {
                  // Reload permissions before showing modal
                  if (!isOnline) {
                    console.log('🔄 Refreshing permissions from offline storage...');
                    await fetchUserInfo();
                    await new Promise(resolve => setTimeout(resolve, 300)); // Wait for state update
                  }
                  setUserInfoModalVisible(true);
                }}
              >
                <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
            </View>
          </View>
          <View style={styles.searchContainer}>
            <View style={styles.searchWrapper}>
              <TextInput
                style={styles.searchInput}
                placeholder={'Search' + (selectedFilter ? ' by ' + selectedFilter : '...')}
                placeholderTextColor="#888"
                value={searchQuery}
                onChangeText={handleSearch}
              />
              <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
            </View>
            <TouchableOpacity
              style={styles.filterButton}
              onPress={() => setFilterModalVisible(true)}
            >
              <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </LinearGradient>
        
        {/* Report Status Counter */}
        <View style={styles.statusCounterContainer}>
          <TouchableOpacity 
            style={styles.statusCounterCard}
            onPress={async () => {
              try {
                const pendingReports = await OfflineDataHelper.getOfflineReports();
                const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
                const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
                const metadata = await AsyncStorage.getItem('inspectionSyncMetadata');
                const meta = metadata ? JSON.parse(metadata) : {};
                
                // Get permissions from cached reports or userInfo
                let storedPermissions = 'None';
                let storedCountries = 'None';
                let storedProjects = 'None';
                
                if (cachedReports.length > 0 && cachedReports[0].userPermissions) {
                  storedPermissions = cachedReports[0].userPermissions.join(', ');
                  storedCountries = cachedReports[0].userCountries?.join(', ') || 'None';
                  storedProjects = cachedReports[0].userProjects?.length > 3 
                    ? `${cachedReports[0].userProjects.slice(0, 3).join(', ')}... (+${cachedReports[0].userProjects.length - 3} more)`
                    : cachedReports[0].userProjects?.join(', ') || 'None';
                } else if (userInfo.inspectionPermissions?.length > 0) {
                  storedPermissions = userInfo.inspectionPermissions.join(', ');
                  storedCountries = userInfo.countries?.join(', ') || 'None';
                  storedProjects = userInfo.projects?.length > 3 
                    ? `${userInfo.projects.slice(0, 3).join(', ')}... (+${userInfo.projects.length - 3} more)`
                    : userInfo.projects?.join(', ') || 'None';
                }
                
                Alert.alert(
                  '📱 Total Offline Storage',
                  `TOTAL Available Offline: ${offlineReportsCount} reports\n\n` +
                  `BREAKDOWN:\n` +
                  `✓ Cached from MongoDB: ${cachedReports.length} (with PDFs)\n` +
                  `✓ Pending Sync: ${pendingReports.length} (with local PDFs)\n\n` +
                  `PERMISSIONS STORED:\n` +
                  `✓ ${storedPermissions}\n\n` +
                  `COUNTRIES:\n` +
                  `✓ ${storedCountries}\n\n` +
                  `PROJECTS:\n` +
                  `✓ ${storedProjects}\n\n` +
                  `SYNC INFO:\n` +
                  `Last Sync: ${meta.lastSyncTime ? new Date(meta.lastSyncTime).toLocaleString() : 'Never'}\n` +
                  `Storage: AsyncStorage (Shared)\n` +
                  `Status: ✅ All Accessible Offline\n\n` +
                  `📄 All reports & PDFs viewable offline!\n` +
                  `🔄 Pending reports will sync when online!`,
                  [{ text: 'OK' }]
                );
              } catch (e) {
                console.error(e);
              }
            }}
          >
            <MaterialIcons name="cloud-off" size={24} color="#FF9800" />
            <View style={styles.statusCounterContent}>
              <Text style={styles.statusCounterLabel}>Offline Stored</Text>
              <Text style={styles.statusCounterValue}>{offlineReportsCount}</Text>
            </View>
          </TouchableOpacity>
          
          <TouchableOpacity 
            style={styles.statusCounterCard}
            onPress={async () => {
              try {
                const metadata = await AsyncStorage.getItem('inspectionSyncMetadata');
                const meta = metadata ? JSON.parse(metadata) : {};
                
                // Get permissions from cached reports
                const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
                const cachedReports = cachedReportsRaw ? JSON.parse(cachedReportsRaw) : [];
                
                let storedPermissions = 'None';
                let storedCountries = 'None';
                let storedProjects = 'None';
                
                if (cachedReports.length > 0 && cachedReports[0].userPermissions) {
                  storedPermissions = cachedReports[0].userPermissions.join(', ');
                  storedCountries = cachedReports[0].userCountries?.join(', ') || 'None';
                  storedProjects = cachedReports[0].userProjects?.length > 3 
                    ? `${cachedReports[0].userProjects.slice(0, 3).join(', ')}... (+${cachedReports[0].userProjects.length - 3} more)`
                    : cachedReports[0].userProjects?.join(', ') || 'None';
                } else if (userInfo.inspectionPermissions?.length > 0) {
                  storedPermissions = userInfo.inspectionPermissions.join(', ');
                  storedCountries = userInfo.countries?.join(', ') || 'None';
                  storedProjects = userInfo.projects?.length > 3 
                    ? `${userInfo.projects.slice(0, 3).join(', ')}... (+${userInfo.projects.length - 3} more)`
                    : userInfo.projects?.join(', ') || 'None';
                }
                
                Alert.alert(
                  '🌐 Online Database Info',
                  `MONGODB REPORTS:\n` +
                  `✓ Stored in MongoDB: ${onlineReportsCount} reports\n` +
                  `✓ Last Synced: ${meta.lastSyncTime ? new Date(meta.lastSyncTime).toLocaleString() : 'Never'}\n` +
                  `✓ Synced By: ${meta.syncedBy || 'Unknown'}\n\n` +
                  `PERMISSIONS SYNCED:\n` +
                  `✓ ${storedPermissions}\n\n` +
                  `COUNTRIES:\n` +
                  `✓ ${storedCountries}\n\n` +
                  `PROJECTS:\n` +
                  `✓ ${storedProjects}\n\n` +
                  `STORAGE:\n` +
                  `✓ Stored in: Offline Shared Storage\n` +
                  `✓ Status: ✅ Accessible Offline\n\n` +
                  `All reports & permissions cached locally!`,
                  [{ text: 'OK' }]
                );
              } catch (e) {
                console.error(e);
              }
            }}
          >
            <MaterialIcons name="cloud-done" size={24} color="#4CAF50" />
            <View style={styles.statusCounterContent}>
              <Text style={styles.statusCounterLabel}>Online in DB</Text>
              <Text style={styles.statusCounterValue}>{onlineReportsCount}</Text>
            </View>
          </TouchableOpacity>
          
          <TouchableOpacity 
            style={styles.statusCounterCard}
            onPress={async () => {
              try {
                const keys = await AsyncStorage.getAllKeys();
                const inspectionKeys = keys.filter(key => 
                  key.includes('inspection') || 
                  key.includes('report') || 
                  key.includes('offline')
                );
                
                // Get permissions info
                const cachedInspectionUserInfo = await AsyncStorage.getItem('inspectionModuleUserInfo');
                let permissionsStored = 'No';
                if (cachedInspectionUserInfo) {
                  const userInfoData = JSON.parse(cachedInspectionUserInfo);
                  permissionsStored = `Yes (${userInfoData.inspectionPermissions?.length || 0} permissions)`;
                }
                
                Alert.alert(
                  '💾 Storage Details',
                  `STORAGE SIZE:\n` +
                  `✓ Total Size: ${localStorageSize}\n` +
                  `✓ Storage Keys: ${inspectionKeys.length}\n\n` +
                  `LOCATION:\n` +
                  `✓ AsyncStorage (Shared)\n` +
                  `✓ Type: Persistent\n\n` +
                  `PERMISSIONS STORED:\n` +
                  `✓ ${permissionsStored}\n\n` +
                  `DATA INCLUDES:\n` +
                  `• Inspection reports\n` +
                  `• User permissions\n` +
                  `• Countries & Projects\n` +
                  `• Sync metadata\n` +
                  `• Offline queue\n\n` +
                  `All data accessible offline!`,
                  [{ text: 'OK' }]
                );
              } catch (e) {
                console.error(e);
              }
            }}
          >
            <MaterialIcons name="storage" size={24} color="#2196F3" />
            <View style={styles.statusCounterContent}>
              <Text style={styles.statusCounterLabel}>Storage</Text>
              <Text style={styles.statusCounterValue}>{localStorageSize}</Text>
            </View>
          </TouchableOpacity>
        </View>
        
        <View style={styles.mainContent}>
          {loadingUserInfo ? (
            <Text style={styles.defaultText}>Loading...</Text>
          ) : false ? ( // TEMPORARY: Always allow viewing for testing
            <View>
              <Text style={styles.defaultText}>You do not have permission to view inspection records.</Text>
              <Text style={[styles.defaultText, { fontSize: 12, marginTop: 10 }]}>
                Debug: Permissions: {JSON.stringify(userInfo.inspectionPermissions)}
              </Text>
            </View>
          ) : reports.length > 0 ? (
            <>
              <FlatList
                ref={flatListRef}
                data={reports}
                keyExtractor={(item) => item.id.toString()}
                renderItem={renderReportItem}
                ListEmptyComponent={<Text style={styles.defaultText}>No inspection records found</Text>}
                contentContainerStyle={styles.reportList}
                refreshing={isRefreshing}
                onRefresh={handleRefresh}
                onEndReached={handleLoadMore}
                onEndReachedThreshold={0.5}
                ListFooterComponent={loadingMore && hasMore ? <View style={{ padding: 16 }}><ActivityIndicator size="small" color="#3481BC" /></View> : null}
              />
              {/* Debug: Show job card information */}
              {(() => {
                const reportsWithJobCards = reports.filter(report => jobCardsMap[report.id]);
                const countWithJobCards = reportsWithJobCards.length;
                return (
                  <View style={{ marginVertical: 16, alignItems: 'center', backgroundColor: '#f0f8ff', padding: 10, borderRadius: 8, marginHorizontal: 16 }}>
                    <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 15 }}>
                      Job Cards Found: {countWithJobCards}
                    </Text>
                    {countWithJobCards > 0 && (
                      <Text style={{ color: '#444', fontSize: 13, marginTop: 4, textAlign: 'center' }}>
                        {reportsWithJobCards.map(r => `${r.step1?.requestNumber || r.id} (${jobCardsMap[r.id]?.requestNumber || 'Unknown'})`).join(', ')}
                      </Text>
                    )}
                    <Text style={{ color: '#666', fontSize: 12, marginTop: 4 }}>
                      Total Job Cards in Map: {Object.keys(jobCardsMap).length}
                    </Text>
                    <TouchableOpacity 
                      style={{ backgroundColor: '#ffd205', paddingHorizontal: 16, paddingVertical: 8, borderRadius: 6, marginTop: 8 }}
                      onPress={() => {
                        console.log('🔄 Manual job card refresh triggered');
                        loadJobCardsForInspections();
                      }}
                    >
                      <Text style={{ color: '#333', fontSize: 12, fontWeight: 'bold' }}>Refresh Job Cards</Text>
                    </TouchableOpacity>
                  </View>
                );
              })()}
            </>
          ) : (
            <>
              <Text style={styles.defaultText}>
                {!isOnline && offlineReportsCount > 0 
                  ? `${offlineReportsCount} offline reports available` 
                  : "You don't have any inspection records"}
              </Text>
              {!isOnline && offlineReportsCount > 0 && (
                <TouchableOpacity 
                  style={styles.retryButton}
                  onPress={async () => {
                    console.log('🔄 Manual offline reload triggered');
                    setSyncModalVisible(true);
                    setPage(1);
                    setSearchQuery('');
                    setSelectedFilter('');
                    
                    try {
                      // Step 1: Load user permissions first
                      console.log('📋 Step 1: Loading user info...');
                      await fetchUserInfo();
                      await new Promise(resolve => setTimeout(resolve, 300)); // Wait for state update
                      
                      // Step 2: Force load all offline reports
                      console.log('📥 Step 2: Loading offline reports...');
                      const cachedReportsRaw = await AsyncStorage.getItem('inspectionReports');
                      const offlineReports = await OfflineDataHelper.getOfflineReports();
                      
                      let allLoadedReports = [];
                      
                      // Add cached reports
                      if (cachedReportsRaw) {
                        const cached = JSON.parse(cachedReportsRaw);
                        allLoadedReports = [...cached];
                        console.log(`✅ Loaded ${cached.length} cached reports from MongoDB`);
                        if (cached.length > 0) {
                          console.log('   First cached report:', cached[0].step1?.requestNumber || cached[0].id);
                        }
                      } else {
                        console.log('⚠️ No cached reports found in AsyncStorage');
                      }
                      
                      // Add offline pending reports
                      if (offlineReports.length > 0) {
                        const formatted = offlineReports.map(report => ({
                          ...report,
                          id: report._id || report.id,
                          adminId: report.userId || report.createdBy,
                          createdAt: report.createdAt || report.offlineCreatedAt,
                          offlineCreated: true,
                          offlineSyncPending: true,
                        }));
                        allLoadedReports = [...allLoadedReports, ...formatted];
                        console.log(`✅ Loaded ${offlineReports.length} offline pending reports`);
                        if (formatted.length > 0) {
                          console.log('   First offline report:', formatted[0].step1?.requestNumber || formatted[0].id);
                        }
                      } else {
                        console.log('⚠️ No offline pending reports found');
                      }
                      
                      console.log(`📊 TOTAL REPORTS LOADED: ${allLoadedReports.length}`);
                      
                      // Step 3: Force update states
                      if (allLoadedReports.length > 0) {
                        // Sort by date (newest first)
                        const sorted = allLoadedReports.sort((a, b) => 
                          new Date(b.createdAt) - new Date(a.createdAt)
                        );
                        
                        // Set both allReports and reports directly for immediate display
                        setAllReports(sorted);
                        
                        // Also directly set the paginated reports for immediate display
                        const paginated = sorted.slice(0, PAGE_SIZE);
                        setReports(paginated);
                        setHasMore(sorted.length > paginated.length);
                        
                        console.log(`📋 Displaying ${paginated.length} of ${sorted.length} reports`);
                        
                        // Force update counts
                        await updateReportCounts();
                        
                        showToast('success', 'Loaded', `${allLoadedReports.length} offline reports loaded successfully!`);
                        console.log('✅ Offline reports loaded and displayed');
                      } else {
                        showToast('warning', 'No Reports', 'No offline reports found in storage');
                        console.log('⚠️ No reports found in offline storage');
                      }
                      
                    } catch (error) {
                      console.error('❌ Offline reload failed:', error);
                      showToast('error', 'Error', 'Failed to load offline reports: ' + error.message);
                    } finally {
                      setSyncModalVisible(false);
                    }
                  }}
                >
                  <MaterialIcons name="refresh" size={width * 0.06} color="#333" />
                  <Text style={styles.retryButtonText}>Load Offline Reports</Text>
                </TouchableOpacity>
              )}
              {(isOnline || offlineReportsCount === 0) && (
                <View style={styles.questionContainer}>
                  <MaterialIcons name="help-outline" size={width * 0.05} color="#333" />
                  <Text style={styles.questionText}>How To Create Inspection?</Text>
                </View>
              )}
            </>
          )}
        </View>
        {userInfo.inspectionPermissions.includes('create') && (
          <View style={styles.buttonContainer}>
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
              <TouchableOpacity
                style={styles.startButton}
                onPress={async () => {
                  try {
                    const userId = await AsyncStorage.getItem('userUid');
                    if (!userId) {
                      showToast('error', 'Authentication Error', 'No user is logged in.');
                      return;
                    }
                    await AsyncStorage.removeItem('inspectionStep1');
                    await AsyncStorage.setItem('equipmentStoragePath', 'GlobalEquipments/admins/' + userId);
                    navigation.navigate('InsreportHeader', {
                      onGoBack: () => navigation.navigate('Inspection_Report_Home'),
                    });
                  } catch (error) {
                    console.error('Error initiating new inspection:', error);
                    showToast('error', 'Error', 'Failed to start new inspection.');
                  }
                }}
              >
                <Ionicons name="add" size={width * 0.06} color="#fff" />
                <Text style={styles.buttonText}>Start New Inspection</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        )}
        
        {/* MongoDB Checklist Upload Button */}
        {userInfo.inspectionPermissions.includes('create') && (
          <View style={styles.checklistUploadContainer}>
            <LinearGradient colors={['#00B894', '#00A085']} style={styles.checklistButtonGradient}>
              <TouchableOpacity
                style={styles.checklistButton}
                onPress={async () => {
                  try {
                    setSyncModalVisible(true);
                    
                    console.log('📤 Uploading checklists to MongoDB...');
                    console.log('📦 Total checklists to upload:', Object.keys(checklistData).length);
                    
                    const response = await InspectionAPI.uploadAllChecklists(checklistData, 'admin');
                    
                    if (response.success) {
                      showToast('success', 'Success', `✅ Uploaded ${response.data.processed} checklists with header & footer inputs!`);
                      console.log('✅ Checklist upload result:', response);
                    } else {
                      showToast('error', 'Upload Failed', response.error || 'Failed to upload checklists');
                    }
                  } catch (error) {
                    console.error('❌ Error uploading checklists:', error);
                    showToast('error', 'Upload Error', 'Failed to upload checklists to MongoDB');
                  } finally {
                    setSyncModalVisible(false);
                  }
                }}
              >
                <MaterialIcons name="cloud-upload" size={width * 0.05} color="#fff" />
                <Text style={styles.checklistButtonText}>Upload Checklists to MongoDB</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        )}
        <Modal
          visible={filterModalVisible}
          transparent={true}
          animationType="slide"
          onRequestClose={() => setFilterModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.modalContent}>
              <Text style={styles.modalTitle}>Filter By</Text>
              {filterOptions.map((filter) => (
                <TouchableOpacity
                  key={filter}
                  style={styles.modalOption}
                  onPress={() => {
                    setSelectedFilter(filter);
                    setFilterModalVisible(false);
                  }}
                >
                  <Text style={styles.modalOptionText}>{filter}</Text>
                </TouchableOpacity>
              ))}
              <TouchableOpacity
                style={styles.modalCloseButton}
                onPress={() => setFilterModalVisible(false)}
              >
                <Text style={styles.modalCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>
        <UserInfoModal
          visible={userInfoModalVisible}
          onClose={() => setUserInfoModalVisible(false)}
          userInfo={userInfo}
        />
        <SyncModal 
          visible={syncModalVisible} 
          onClose={() => setSyncModalVisible(false)} 
          message={
            !isOnline && !isSyncing ? "Loading from Offline Storage..." : 
            isSyncing ? "Syncing from MongoDB & storing offline..." : 
            "Loading Reports..."
          }
          isOffline={!isOnline && !isSyncing}
        />
        <SyncModal visible={isProcessingAction} message="Processing Action..." />
        <Toast />
      </View>
    </ErrorBoundary>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.013,
    marginLeft: width * 0.035,
  },
  icons12: {
    width: 30,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  questionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.04,
    borderRadius: width * 0.012,
    marginTop: width * 0.025,
  },
  questionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.025,
  },
  reportList: {
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  reportItem: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  reportHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  reportTitle: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '500',
  },
  offlineBadge: {
    marginLeft: width * 0.025,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.008,
    borderRadius: width * 0.01,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ff9800',
  },
  offlineBadgeText: {
    fontSize: width * 0.025,
    color: '#fff',
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  jobCardIndicator: {
    marginLeft: width * 0.025,
    paddingHorizontal: width * 0.015,
    paddingVertical: width * 0.008,
    borderRadius: width * 0.01,
    flexDirection: 'row',
    alignItems: 'center',
  },
  statusButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.012,
    border: '2px solid #015185',
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
  },
  reportDetails: {
    paddingHorizontal: width * 0.05,
  },
  detailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  detailValueLeft: {
    textAlign: 'left',
    width: '60%',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalOption: {
    paddingVertical: width * 0.025,
  },
  modalOptionText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  faultBadgeContainer: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    marginLeft: width * 0.03,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.01,
    minWidth: width * 0.09,
    minHeight: width * 0.07,
  },
  faultBadgeText: {
    color: '#e74c3c',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    textAlign: 'center',
    lineHeight: width * 0.05,
  },
  faultBadgeLabel: {
    color: '#e74c3c',
    fontSize: width * 0.03,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: -2,
  },
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.015,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
    width: '60%',
  },
  emailStatsContainer: {
    backgroundColor: '#fff',
    marginHorizontal: width * 0.04,
    marginTop: width * 0.025,
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  emailStatsHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    backgroundColor: '#f8f9fa',
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  emailStatsTitle: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    marginLeft: width * 0.025,
  },
  emailStatsContent: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  emailStatItem: {
    flex: 1,
    alignItems: 'center',
  },
  emailStatLabel: {
    fontSize: width * 0.032,
    color: '#666',
    marginBottom: width * 0.01,
    textAlign: 'center',
  },
  emailStatValue: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    textAlign: 'center',
  },
  checklistUploadContainer: {
    position: 'absolute',
    bottom: '5%',
    width: '100%',
    alignItems: 'center',
  },
  checklistButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  checklistButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
  },
  checklistButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncingButton: {
    opacity: 0.7,
  },
  syncingIcon: {
    opacity: 0.8,
  },
  offlineContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: width * 0.08,
    backgroundColor: '#fff',
  },
  offlineTitle: {
    fontSize: width * 0.06,
    fontWeight: 'bold',
    color: '#e74c3c',
    marginTop: width * 0.05,
    marginBottom: width * 0.03,
    textAlign: 'center',
  },
  offlineMessage: {
    fontSize: width * 0.045,
    color: '#666',
    textAlign: 'center',
    marginBottom: width * 0.02,
    lineHeight: width * 0.06,
  },
  offlineSubMessage: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.02,
    fontStyle: 'italic',
  },
  retryButton: {
    backgroundColor: '#ffd205',
    paddingHorizontal: width * 0.08,
    paddingVertical: width * 0.025,
    borderRadius: width * 0.012,
    marginTop: width * 0.05,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  retryButtonText: {
    color: '#333',
    fontSize: width * 0.045,
    fontWeight: 'bold',
    textAlign: 'center',
    marginLeft: width * 0.02,
  },
  offlineIndicator: {
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    borderRadius: width * 0.025,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.01,
    marginRight: width * 0.02,
  },
  statusCounterContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.03,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  statusCounterCard: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F5F5F5',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.025,
    marginHorizontal: width * 0.01,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  statusCounterContent: {
    marginLeft: width * 0.02,
    flex: 1,
  },
  statusCounterLabel: {
    fontSize: width * 0.028,
    color: '#666',
    marginBottom: 2,
  },
  statusCounterValue: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
  },
});

export default Inspection_Report_Home;