import React from 'react';
import { View, Text, TouchableOpacity, StyleSheet, Alert } from 'react-native';
import { 
  sendInspectionEmails, 
  getEmailStatistics, 
  retryFailedEmails,
  emailService,
  InspectionEmailUtil
} from './index';

/**
 * Test component for the Email Notification System
 * This component demonstrates how to use all the email functionality
 */
const EmailTest = () => {
  
  // Sample inspection report data for testing
  const sampleReportData = {
    id: 'test_inspection_123',
    step1: {
      requestNumber: 'ZAM-KOB-IR-001',
      country: 'Zambia',
      project: 'Kobold',
      inspector: 'John Doe',
      selectedEquipment: {
        equipmentName: 'Drill Rig A',
        model: 'DR-2000',
        fleetNumber: 'F001'
      }
    },
    step2: {
      checklistTitle: 'Daily Rig Prestart Checklist',
      selections: {
        'item_1': 'Okay',
        'item_2': 'Faulty',
        'item_3': 'N/A'
      },
      faultyItems: 1
    },
    step3: {
      equipmentStatus: 'Maintenance Required',
      faultyItems: 1,
      overallCondition: 'Good',
      priority: 'Medium',
      safeToUse: 'Yes',
      additionalNote: 'Minor maintenance required on hydraulic system',
      createdAt: new Date().toISOString()
    }
  };

  // Test sending inspection emails
  const testSendInspectionEmails = async () => {
    try {
      Alert.alert('Test', 'Sending test inspection emails...');
      
      const result = await sendInspectionEmails(
        sampleReportData,
        'https://example.com/test-report.pdf',
        'https://example.com/test-signature.png'
      );
      
      Alert.alert('Success', 'Test emails sent successfully!');
      console.log('Email test result:', result);
      
    } catch (error) {
      Alert.alert('Error', `Failed to send test emails: ${error.message}`);
      console.error('Email test error:', error);
    }
  };

  // Test getting email statistics
  const testGetEmailStatistics = async () => {
    try {
      Alert.alert('Test', 'Fetching email statistics...');
      
      const stats = await getEmailStatistics();
      
      Alert.alert('Statistics', 
        `Total: ${stats.total}\n` +
        `Sent: ${stats.sent}\n` +
        `Failed: ${stats.failed}\n` +
        `Queued: ${stats.queued}\n` +
        `Success Rate: ${stats.successRate}%`
      );
      
      console.log('Email statistics:', stats);
      
    } catch (error) {
      Alert.alert('Error', `Failed to get statistics: ${error.message}`);
      console.error('Statistics test error:', error);
    }
  };

  // Test retrying failed emails
  const testRetryFailedEmails = async () => {
    try {
      Alert.alert('Test', 'Retrying failed emails...');
      
      const retryCount = await retryFailedEmails();
      
      Alert.alert('Retry Result', `Retried ${retryCount} failed emails`);
      console.log('Retry test result:', retryCount);
      
    } catch (error) {
      Alert.alert('Error', `Failed to retry emails: ${error.message}`);
      console.error('Retry test error:', error);
    }
  };

  // Test direct service methods
  const testDirectService = async () => {
    try {
      Alert.alert('Test', 'Testing direct service methods...');
      
      // Test getting current user
      const currentUser = await emailService.getCurrentUser();
      console.log('Current user:', currentUser);
      
      // Test getting supervisors
      const supervisors = await emailService.getSupervisorsByCountryAndProject('Zambia', 'Kobold');
      console.log('Supervisors found:', supervisors.length);
      
      Alert.alert('Service Test', 
        `Current user: ${currentUser?.name || 'Unknown'}\n` +
        `Supervisors found: ${supervisors.length}`
      );
      
    } catch (error) {
      Alert.alert('Error', `Service test failed: ${error.message}`);
      console.error('Service test error:', error);
    }
  };

  // Test email queue status
  const testQueueStatus = () => {
    const queueLength = emailService.emailQueue.length;
    const isProcessing = emailService.isProcessing;
    
    Alert.alert('Queue Status', 
      `Queue length: ${queueLength}\n` +
      `Processing: ${isProcessing ? 'Yes' : 'No'}`
    );
    
    console.log('Queue status:', { queueLength, isProcessing });
  };

  // Test InspectionEmailUtil functions
  const testEmailUtils = () => {
    try {
      const testEmail = 'test@example.com';
      const testPhone = '+1234567890';
      const testEmployeeNumber = 'EMP001';
      
      const emailValid = InspectionEmailUtil.isValidEmail(testEmail);
      const phoneValid = InspectionEmailUtil.isValidPhone(testPhone);
      const employeeValid = InspectionEmailUtil.isValidEmployeeNumber(testEmployeeNumber);
      
      const sanitizedContent = InspectionEmailUtil.sanitizeEmailContent('<script>alert("test")</script>Hello World');
      
      Alert.alert('Email Utils Test', 
        `Email Validation: ${emailValid ? 'Valid' : 'Invalid'}\n` +
        `Phone Validation: ${phoneValid ? 'Valid' : 'Invalid'}\n` +
        `Employee Number: ${employeeValid ? 'Valid' : 'Invalid'}\n` +
        `Sanitized Content: ${sanitizedContent}`
      );
      
      console.log('Email utils test:', {
        emailValid,
        phoneValid,
        employeeValid,
        sanitizedContent
      });
      
    } catch (error) {
      Alert.alert('Error', `Email utils test failed: ${error.message}`);
      console.error('Email utils test error:', error);
    }
  };

  return (
    <View style={styles.container}>
      <Text style={styles.title}>Email Notification System Test</Text>
      <Text style={styles.subtitle}>Test all email functionality</Text>
      
      <View style={styles.buttonContainer}>
        <TouchableOpacity 
          style={styles.testButton} 
          onPress={testSendInspectionEmails}
        >
          <Text style={styles.buttonText}>Test Send Emails</Text>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.testButton} 
          onPress={testGetEmailStatistics}
        >
          <Text style={styles.buttonText}>Get Email Statistics</Text>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.testButton} 
          onPress={testRetryFailedEmails}
        >
          <Text style={styles.buttonText}>Retry Failed Emails</Text>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.testButton} 
          onPress={testDirectService}
        >
          <Text style={styles.buttonText}>Test Service Methods</Text>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.testButton} 
          onPress={testQueueStatus}
        >
          <Text style={styles.buttonText}>Check Queue Status</Text>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.testButton} 
          onPress={testEmailUtils}
        >
          <Text style={styles.buttonText}>Test Email Utils</Text>
        </TouchableOpacity>
      </View>
      
      <View style={styles.infoContainer}>
        <Text style={styles.infoTitle}>Test Data:</Text>
        <Text style={styles.infoText}>Request: {sampleReportData.step1.requestNumber}</Text>
        <Text style={styles.infoText}>Country: {sampleReportData.step1.country}</Text>
        <Text style={styles.infoText}>Project: {sampleReportData.step1.project}</Text>
        <Text style={styles.infoText}>Equipment: {sampleReportData.step1.selectedEquipment.equipmentName}</Text>
        <Text style={styles.infoText}>Status: {sampleReportData.step3.equipmentStatus}</Text>
        <Text style={styles.infoText}>Faulty Items: {sampleReportData.step3.faultyItems}</Text>
      </View>
      
      <Text style={styles.note}>
        Note: This is a test component. In production, emails are sent automatically 
        when inspection reports are generated.
      </Text>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    padding: 20,
    backgroundColor: '#f5f5f5',
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 10,
  },
  subtitle: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    marginBottom: 30,
  },
  buttonContainer: {
    marginBottom: 30,
  },
  testButton: {
    backgroundColor: '#015185',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  buttonText: {
    color: 'white',
    fontSize: 16,
    fontWeight: '600',
    textAlign: 'center',
  },
  infoContainer: {
    backgroundColor: 'white',
    padding: 20,
    borderRadius: 8,
    marginBottom: 20,
    elevation: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 2,
  },
  infoTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
  },
  infoText: {
    fontSize: 14,
    color: '#333',
    marginBottom: 5,
  },
  note: {
    fontSize: 12,
    color: '#666',
    textAlign: 'center',
    fontStyle: 'italic',
    lineHeight: 18,
  },
});

export default EmailTest;
