import React, { useState, useRef } from 'react';
import { View, Text, StyleSheet, Dimensions, ScrollView, TouchableOpacity, Modal, Platform, KeyboardAvoidingView, TextInput, Animated, Image } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import RiskManagementHeader from './RiskManagementHeader';

const { width } = Dimensions.get('window');

const FIELDS = [
  { key: 'fleetNameNumber', label: 'Fleet Name / Number:' },
  { key: 'mechanic', label: 'Mechanic:' },
  { key: 'registrationNumber', label: 'Registration Number:' },
  { key: 'electrician', label: 'Electrician:' },
  { key: 'odometerReadingHours', label: 'Odometer Reading / Hours:' },
  { key: 'boilermaker', label: 'Boilermaker:' },
  // jobCardNo is rendered conditionally only when a Job Card exists
  { key: 'jobCardNo', label: 'Job Card #:' },
];

const MiniRiskAssessment_step3 = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const [currentStep, setCurrentStep] = useState(3);
  // Prepare local form state (auto-filled from Inspection where possible)
  const [form, setForm] = useState({
    fleetNameNumber: '',
    mechanic: '',
    registrationNumber: '',
    electrician: '',
    odometerReadingHours: '',
    boilermaker: '',
    jobCardNo: '',
  });
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [processingModalVisible, setProcessingModalVisible] = useState(false);
  const [pdfUrl, setPdfUrl] = useState(null);
  const [hasGenerated, setHasGenerated] = useState(false);
  const isExitingRef = useRef(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const scrollRef = useRef(null);
  const fieldOffsetsRef = useRef({});
  const [info, setInfo] = useState({
    inspectionRequestNumber: '',
    inspectionId: '',
    jobCardRequestNumber: '',
    jobCardId: '',
    inspector: '',
  });

  // Load saved step2 rows and step1 hazard selections for context if needed
  const incomingRows = Array.isArray(route.params?.rows) ? route.params.rows : [];
  const linkedReport = route.params?.report || null;

  // Helper to derive best fleet number
  const getBestFleetValue = (equipment) => {
    if (!equipment) return '';
  return (
      equipment.fleetNumber ||
      equipment.fleet_no ||
      equipment.fleetNo ||
      equipment.fleet ||
      equipment.vehicleNumber ||
      equipment.registrationNumber ||
      equipment.registration ||
      equipment.assetNumber ||
      equipment.equipmentId ||
      equipment.id ||
      ''
    );
  };

  React.useEffect(() => {
    (async () => {
      try {
        const saved = await AsyncStorage.getItem('riskMgmt_step3_form');
        if (saved) {
          const parsed = JSON.parse(saved);
          setForm((prev) => ({ ...prev, ...parsed }));
        }
        // If linked to a report, load existing record from Firebase and prefill
        if (linkedReport && linkedReport.adminId && linkedReport.id) {
          try {
            const ref = database().ref(`/GlobalRiskAssessments/admins/${linkedReport.adminId}/${linkedReport.id}`);
            const snap = await ref.once('value');
            const data = snap.val();
            if (data?.step3?.form) setForm((prev) => ({ ...prev, ...data.step3.form }));
          } catch (e) {}
        }

        // Build top info and auto-fill from inspection (route report first, then local cache)
        let step1 = null;
        if (linkedReport?.step1) {
          step1 = linkedReport.step1;
        } else {
          const step1Raw = await AsyncStorage.getItem('inspectionStep1');
          step1 = step1Raw ? JSON.parse(step1Raw) : null;
        }

        const equipment = step1?.selectedEquipment || null;
        const bestFleet = getBestFleetValue(equipment);
        const regNo = equipment?.registrationNumber || equipment?.vehicleNumber || bestFleet || '';
        const odo = (equipment?.meterReading != null ? String(equipment.meterReading) : (step1?.meterReading != null ? String(step1.meterReading) : ''));

        setForm((prev) => ({
          ...prev,
          fleetNameNumber: prev.fleetNameNumber || bestFleet,
          registrationNumber: prev.registrationNumber || regNo,
          odometerReadingHours: prev.odometerReadingHours || odo,
          jobCardNo: prev.jobCardNo || (linkedReport?.workOrder?.requestNumber || linkedReport?.workOrder?.id || ''),
        }));

        setInfo({
          inspectionRequestNumber: step1?.requestNumber || linkedReport?.step1?.requestNumber || '',
          inspectionId: linkedReport?.id || '',
          jobCardRequestNumber: linkedReport?.workOrder?.requestNumber || '',
          jobCardId: linkedReport?.workOrder?.id || '',
          inspector: step1?.inspector || '',
        });
      } catch (e) {}
    })();
  }, []);

  // Prevent navigating back to Step 2 after a report has been generated
  React.useEffect(() => {
    const unsub = navigation.addListener('beforeRemove', (e) => {
      if (!hasGenerated) return; // allow normal back before generation
      if (isExitingRef.current) return; // avoid loops on our own reset
      e.preventDefault();
      isExitingRef.current = true;
      navigation.reset({
        index: 1,
        routes: [
          { name: 'MainApp' },
          { name: 'Inspection_Report_Home' },
        ],
      });
    });
    return unsub;
  }, [navigation, hasGenerated]);

  // Spinner rotation for processing modal
  React.useEffect(() => {
    if (processingModalVisible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.stopAnimation(() => rotateAnim.setValue(0));
    }
  }, [processingModalVisible]);

  const buildPdfHtml = (record) => {
    const step1 = record.step1 || {};
    const step2 = record.step2 || {};
    const step3 = record.step3 || {};
    const hazardRiskSelections = step1?.hazardRiskSelections || {};
    const selectedHazardIds = step1?.selectedHazardIds || [];

    // Minimal labels (fallbacks when not available)
    const HAZARD_LABELS = {
      '1': 'Abrasive Surface', '2': 'Confined Space', '3': 'Dust', '4': 'Electricity LV', '5': 'Electricity HV', '6': 'Elevated Work Area',
      '7': 'Falling Objects', '8': 'Flying Objects', '9': 'Fumes', '10': 'Electrocution', '11': 'Entanglement', '12': 'Falls',
      '13': 'Fire/Explosion', '14': 'Flammable Liquid', '15': 'Hazardous Chemicals', '16': 'Heat/Cold', '17': 'Hot Surface', '18': 'Manual Handling',
      '19': 'Moving Machinery', '20': 'Noise', '21': 'Pressure Systems', '22': 'Sharp Objects', '23': 'Slip/Trip', '24': 'Vibration', '25': 'Unexpected Movement'
    };
    const RISK_LABELS = {
      r1: 'Amputation', r2: 'Burns', r3: 'Caught Between', r4: 'Choking', r5: 'Contact With / Between (Pinch)', r6: 'Contamination – Soil',
      r7: 'Contamination – Oil', r8: 'Crush', r9: 'Drowning', r10: 'Electrocution', r11: 'Explosion', r12: 'Fall From Height', r13: 'Fire',
      r14: 'Hearing Loss', r15: 'Inhalation', r16: 'Laceration', r17: 'Penetrated By', r18: 'Poisoning', r19: 'Slip/Trip', r20: 'Struck By', r21: 'Suffocation', r22: 'Vibration Injury', r23: 'Eye Injury', r24: 'Back Injury', r25: 'Ergonomic Injury', r26: 'Other'
    };

    const hazardsRowsHtml = selectedHazardIds.map((hid) => {
      const risksMap = hazardRiskSelections[hid] || {};
      const chosen = Object.keys(risksMap).filter((rid) => risksMap[rid]);
      const riskNames = chosen.map((rid) => RISK_LABELS[rid] || rid).join(', ');
      const hazardName = HAZARD_LABELS[hid] || hid;
      return `
        <tr>
          <td>${hazardName}</td>
          <td>${riskNames || '—'}</td>
        </tr>
      `;
    }).join('');

    const rows = Array.isArray(step2.rows) ? step2.rows : [];
    const step2RowsHtml = rows.map((row) => `
      <tr>
        <td>${(row.step || '').toString()}</td>
        <td>${(row.hazard || '').toString()}</td>
        <td>${(row.assess || '').toString()}</td>
        <td>${(row.controls || '').toString()}</td>
      </tr>
    `).join('');

    const f = step3.form || {};
    const formRowsHtml = `
      <tr><td>Fleet Name / Number</td><td>${f.fleetNameNumber || ''}</td></tr>
      <tr><td>Mechanic</td><td>${f.mechanic || ''}</td></tr>
      <tr><td>Registration Number</td><td>${f.registrationNumber || ''}</td></tr>
      <tr><td>Electrician</td><td>${f.electrician || ''}</td></tr>
      <tr><td>Odometer Reading / Hours</td><td>${f.odometerReadingHours || ''}</td></tr>
      <tr><td>Boilermaker</td><td>${f.boilermaker || ''}</td></tr>
      ${f.jobCardNo ? `<tr><td>Job Card #</td><td>${f.jobCardNo}</td></tr>` : ''}
    `;

    const infoHtml = `
      <tr><td>Inspection Request #</td><td>${record.linkInspection?.requestNumber || '—'}</td></tr>
      <tr><td>Inspection ID</td><td>${record.id || '—'}</td></tr>
      <tr><td>Inspector</td><td>${info.inspector || step1?.inspector || '—'}</td></tr>
      ${info.jobCardRequestNumber || info.jobCardId ? `<tr><td>Job Card</td><td>${(info.jobCardRequestNumber || '') + (info.jobCardId ? ` (ID: ${info.jobCardId})` : '')}</td></tr>` : ''}
    `;

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
        <meta charset="UTF-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1.0" />
        <title>Risk Assessment</title>
        <style>
          body { font-family: Arial, sans-serif; margin: 20px; color: #222; }
          h1 { text-align: center; font-size: 20px; margin-bottom: 10px; }
          .section { margin-top: 16px; }
          .section h2 { font-size: 14px; margin-bottom: 6px; color: #015185; }
          table { width: 100%; border-collapse: collapse; font-size: 11px; }
          td, th { border: 1px solid #333; padding: 6px; text-align: left; }
          th { background: #f2f6fb; }

          /* Static header styles */
          .header-static { display: flex; align-items: flex-start; justify-content: space-between; margin-bottom: 14px; }
          .header-left { display: flex; align-items: center; gap: 10px; }
          .header-logo { width: 50px; height: 50px; object-fit: contain; }
          .company-wrap { display: flex; flex-direction: column; }
          .company-title { font-size: 18px; font-weight: 800; letter-spacing: .5px; }
          .company-subtitle { font-size: 16px; font-weight: 700; margin-top: 6px; text-transform: uppercase; }
          .doc-info { width: 48%; }
          .doc-info table { width: 100%; border: 1px solid #222; border-collapse: collapse; font-size: 11px; }
          .doc-info td { border: 1px solid #222; padding: 4px 6px; }
          .doc-info td:first-child { font-weight: bold; background-color: #f5f5f5; width: 40%; }
        </style>
      </head>
      <body>
        <div class="header-static">
          <div class="header-left">
            <img class="header-logo" src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" />
           
          </div>
          <div class="doc-info">
            <table>
              <tr><td>Document Name</td><td>HSEC-FOR-8.1.1 Workshop Job Card</td></tr>
              <tr><td>Document ID</td><td>HSE-032</td></tr>
              <tr><td>Version Date</td><td>Date: 04/03/2019</td></tr>
              <tr><td>Version</td><td>1</td></tr>
              <tr><td>Approved</td><td>General Manager</td></tr>
            </table>
          </div>
        </div>

        <h1> RISK ASSESSMENT</h1>

        <div class="section">
       
          <table>
            ${infoHtml}
          </table>
        </div>

        <div class="section">
         
          <table>
            ${formRowsHtml}
          </table>
        </div>

        <div class="section">
          <h2>HAZARDS AND RISKS ASSOCIATED WITH THE WORK TO BE CARRIED OUT</h2>
          <table>
            <tr><th>Hazard</th><th>Selected Risks</th></tr>
            ${hazardsRowsHtml || '<tr><td colspan="2">No hazards selected</td></tr>'}
          </table>
        </div>

        <div class="section">
          <h2>Risk Assessment</h2>
          <table>
            <tr><th>Steps</th><th>Hazards</th><th>Assess the Risk</th><th>Controls</th></tr>
            ${step2RowsHtml || '<tr><td colspan="4">No rows</td></tr>'}
          </table>
        </div>
      </body>
      </html>
    `;
  };

  const uploadFileToFirebaseStorage = async (localPath, storagePath) => {
    const ref = storage().ref(storagePath);
    await ref.putFile(localPath);
    const url = await ref.getDownloadURL();
    return url;
  };

  const handleSubmit = async () => {
    try {
      setProcessingModalVisible(true);
      await AsyncStorage.setItem('riskMgmt_step3_form', JSON.stringify(form));
      // Compose payload
      const step1Raw = await AsyncStorage.getItem('riskMgmt_step1');
      const step1 = step1Raw ? JSON.parse(step1Raw) : null;
      const step2RowsRaw = await AsyncStorage.getItem('riskMgmt_step2_rows');
      const step2Rows = step2RowsRaw ? JSON.parse(step2RowsRaw) : incomingRows;
      const fullStep1Raw = await AsyncStorage.getItem('inspectionStep1');
      const fullStep1 = fullStep1Raw ? JSON.parse(fullStep1Raw) : null;

      // Determine admin/report context
      let meta = { adminId: null, reportId: null };
      const metaSaved = await AsyncStorage.getItem('riskMgmt_meta');
      if (metaSaved) {
        try { meta = JSON.parse(metaSaved) || meta; } catch {}
      }
      if (linkedReport) {
        meta = { adminId: linkedReport.adminId || meta.adminId, reportId: linkedReport.id || meta.reportId };
      }
      // Fallback if meta missing: derive from logged-in user and inspectionStep1 id
      if (!meta.adminId || !meta.reportId) {
        try {
          const userUid = await AsyncStorage.getItem('userUid');
          const fullStep1Raw = await AsyncStorage.getItem('inspectionStep1');
          const fullStep1 = fullStep1Raw ? JSON.parse(fullStep1Raw) : null;
          if (userUid && fullStep1?.id) {
            meta = { adminId: userUid, reportId: fullStep1.id };
            await AsyncStorage.setItem('riskMgmt_meta', JSON.stringify(meta));
          }
        } catch (e) {}
      }

      const riskRecordId = `risk_${Date.now()}`;

      // Build record
      const record = {
        id: meta.reportId,
        adminId: meta.adminId,
        riskRecordId,
        linkInspection: {
          id: meta.reportId,
          requestNumber: fullStep1?.requestNumber || linkedReport?.step1?.requestNumber || '',
        },
        step1: step1 || {},
        step2: { rows: step2Rows },
        step3: { form, createdAt: new Date().toISOString() },
      };

      // Save to Firebase under GlobalRiskAssessments
      const id = meta.reportId || `risk_${Date.now()}`;
      const adminId = meta.adminId || 'unassigned';
      const dbRef = database().ref(`/GlobalRiskAssessments/admins/${adminId}/${id}`);
      await dbRef.set(record);

      // Generate PDF and upload
      const html = buildPdfHtml(record);
      const pdf = await RNHTMLtoPDF.convert({ html, fileName: `Mini_Risk_Assessment_${id}`, directory: 'Documents' });
      const uploadedUrl = await uploadFileToFirebaseStorage(
        pdf.filePath,
        `GlobalRiskAssessments/admins/${adminId}/${id}/Mini_Risk_Assessment_${id}.pdf`
      );

      // Save PDF URL back to record
      await dbRef.child('step3').update({ pdfDownloadUrl: uploadedUrl, updatedAt: new Date().toISOString() });

      // Clear local caches now that it is uploaded
      await AsyncStorage.removeItem('riskMgmt_step1');
      await AsyncStorage.removeItem('riskMgmt_step2_rows');
      await AsyncStorage.removeItem('riskMgmt_step3_form');
      await AsyncStorage.removeItem('riskMgmt_meta');

      setPdfUrl(uploadedUrl);
      setSuccessModalVisible(true);
      setHasGenerated(true);
    } catch (e) {
      // Show a simple failure modal
      setPdfUrl(null);
      setSuccessModalVisible(true);
    }
    setProcessingModalVisible(false);
  };

  const handleOk = () => {
    setSuccessModalVisible(false);
    navigation.reset({
      index: 1,
      routes: [
        { name: 'MainApp' },
        { name: 'Inspection_Report_Home' },
      ],
    });
  };

  const handleViewPdf = () => {
    if (!pdfUrl) {
      setSuccessModalVisible(false);
      return;
    }
    setSuccessModalVisible(false);
    navigation.navigate('PdfviewPage', {
      pdfUrl,
      onGoBack: () =>
        navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'Inspection_Report_Home' },
          ],
        }),
    });
  };

  return (
    <View style={styles.container}>
      <RiskManagementHeader currentStep={currentStep} setCurrentStep={setCurrentStep} />
      <Text style={styles.pageHeading}>Mini Risk Assessment</Text>
      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? 90 : 0}
      >
        <ScrollView
          ref={scrollRef}
          style={{ flex: 1 }}
          contentContainerStyle={{ paddingBottom: 140 }}
          keyboardShouldPersistTaps="handled"
          keyboardDismissMode="on-drag"
        >
          {/* Top info hidden as requested */}

          <View style={styles.formCard}>
            {FIELDS.map((f, idx) => {
              // Skip Job Card # input when no Job Card exists
              if (f.key === 'jobCardNo' && !(info.jobCardRequestNumber || info.jobCardId)) return null;
              return (
                <View
                  key={f.key}
                  style={[styles.fieldGroup, idx === FIELDS.length - 1 && styles.lastRow]}
                  onLayout={(e) => {
                    fieldOffsetsRef.current[f.key] = e.nativeEvent.layout.y;
                  }}
                >
                  <Text style={styles.fieldLabel}>{f.label}</Text>
                  <TextInput
                    value={form[f.key]}
                    onChangeText={(t) => setForm((prev) => ({ ...prev, [f.key]: t }))}
                    onFocus={() => {
                      const y = fieldOffsetsRef.current[f.key] || 0;
                      setTimeout(() => {
                        if (scrollRef.current && typeof scrollRef.current.scrollTo === 'function') {
                          scrollRef.current.scrollTo({ y: Math.max(0, y - 20), animated: true });
                        }
                      }, 220);
                    }}
                    style={styles.fieldInput}
                    placeholder="Enter here"
                    placeholderTextColor="#8fb7d8"
                    underlineColorAndroid="transparent"
                    returnKeyType="done"
                  />
                </View>
              );
            })}
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <View style={styles.buttonBar}>
        <TouchableOpacity
          style={styles.backBtn}
          onPress={() => {
            if (hasGenerated) {
              navigation.reset({
                index: 1,
                routes: [
                  { name: 'MainApp' },
                  { name: 'Inspection_Report_Home' },
                ],
              });
            } else {
              navigation.goBack();
            }
          }}
        >
          <MaterialIcons name="chevron-left" size={18} color="#015185" />
          <Text style={styles.backText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.submitBtn} onPress={handleSubmit}>
          <Text style={styles.submitText}>Submit</Text>
          <MaterialIcons name="check" size={18} color="#015185" />
        </TouchableOpacity>
      </View>

      <Modal visible={processingModalVisible} animationType="fade" transparent>
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Please Wait</Text>
            <View style={styles.processingRow}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[
                  styles.syncIcon,
                  { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] },
                ]}
              />
              <Text style={styles.processingText}>Generating Report</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={successModalVisible}
        animationType="fade"
        transparent
        onRequestClose={handleOk}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Success</Text>
            <Text style={styles.modalMsg}>Mini Risk Assessment saved.</Text>
            <View style={{ flexDirection: 'row', marginTop: 10 }}>
              <TouchableOpacity style={[styles.okBtn, { backgroundColor: '#e0e0e0', marginRight: 10 }]} onPress={handleOk}>
                <Text style={[styles.okText, { color: '#015185' }]}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.okBtn} onPress={handleViewPdf}>
                <Text style={styles.okText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  pageHeading: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 10,
    marginLeft: 20,
    marginBottom: 8,
  },
  formCard: {
    marginHorizontal: 12,
    borderWidth: 1,
    borderColor: '#bdbdbd',
    borderRadius: 8,
    overflow: 'hidden',
    backgroundColor: '#fff',
    paddingHorizontal: 10,
    paddingTop: 6,
    paddingBottom: 4,
  },
  fieldGroup: {
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#bdbdbd',
  },
  lastRow: {
    borderBottomWidth: 0,
  },
  fieldLabel: {
    color: '#015185',
    fontWeight: '700',
    fontSize: 13,
    marginBottom: 6,
  },
  fieldInput: {
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 6,
    paddingHorizontal: 10,
    paddingVertical: 10,
    color: '#015185',
    fontSize: 14,
    backgroundColor: '#fff',
  },
  buttonBar: {
    position: 'absolute',
    left: 0,
    right: 0,
    bottom: 0,
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  backBtn: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  backText: {
    color: '#015185',
    fontWeight: '600',
    fontSize: 16,
    marginLeft: 4,
  },
  submitBtn: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  submitText: {
    color: '#015185',
    fontWeight: '600',
    fontSize: 16,
    marginRight: 6,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    paddingHorizontal: 30,
    paddingVertical: 18,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
  },
  modalMsg: {
    fontSize: 16,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  processingRow: {
    marginTop: 6,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 6,
    paddingHorizontal: 4,
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 6,
  },
  processingText: {
    fontSize: 16,
    color: '#333',
  },
  okBtn: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 30,
    borderRadius: 5,
  },
  okText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
});

export default MiniRiskAssessment_step3;