import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  Alert,
  TouchableWithoutFeedback,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

// MongoDB API URL
const MONGODB_LOGBOOK_API_URL = 'http://168.231.113.219:5012';

const LogbookList = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [logBookEntries, setLogBookEntries] = useState([]);
  const [filteredEntries, setFilteredEntries] = useState([]);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [selectedItem, setSelectedItem] = useState(null);
  const [isOnline, setIsOnline] = useState(true);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    logBookPermissions: [],
  });
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const isMounted = useRef(true);

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notifications
  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Queue operation for offline sync
  const queueOperation = async (operation) => {
    try {
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingLogBookOperations');
      const pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push(operation);
      await AsyncStorage.setItem('pendingLogBookOperations', JSON.stringify(pendingOperations));
      showToast('info', 'Offline Mode', 'Operation queued for sync when online.');
    } catch (error) {
      console.error('Error queuing operation:', error);
      showToast('error', 'Error', 'Failed to queue operation.');
    }
  };

  // Sync pending operations to Firebase
  const syncPendingOperations = async () => {
    if (!isMounted.current) return;
    if (!(await checkNetworkStatus())) return;

    try {
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingLogBookOperations');
      if (!pendingOperationsRaw) return;

      const pendingOperations = JSON.parse(pendingOperationsRaw);
      if (pendingOperations.length === 0) return;

      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Authentication Error', 'User not authenticated.');
        return;
      }

      let updatedEntries = [...logBookEntries];
      for (const operation of pendingOperations) {
        try {
          if (operation.type === 'create') {
            const entryRef = database().ref(`/GlobalLogbook/admins/${userUid}/${operation.entry.id}`);
            await entryRef.set(operation.entry);
            updatedEntries = updatedEntries.filter((entry) => entry.id !== operation.entry.id);
            updatedEntries.push(operation.entry);
            await AsyncStorage.setItem(`logbook_${operation.entry.id}`, JSON.stringify(operation.entry));
          } else if (operation.type === 'delete') {
            const entryRef = database().ref(`/GlobalLogbook/admins/${userUid}/${operation.entryId}`);
            await entryRef.remove();
            updatedEntries = updatedEntries.filter((entry) => entry.id !== operation.entryId);
            await AsyncStorage.removeItem(`logbook_${operation.entryId}`);
          }
        } catch (error) {
          console.error(`Error syncing operation ${operation.type}:`, error);
          continue;
        }
      }

      if (isMounted.current) {
        setLogBookEntries(updatedEntries.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt)));
        setFilteredEntries(updatedEntries.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt)));
      }
      await AsyncStorage.setItem('pendingLogBookOperations', JSON.stringify([]));
      showToast('success', 'Sync Complete', 'Offline operations synced successfully.');
    } catch (error) {
      console.error('Error syncing pending operations:', error);
      showToast('error', 'Sync Error', 'Failed to sync operations. Will retry when online.');
    }
  };

  const fetchLogBookEntries = async () => {
    if (!isMounted.current) return;
    
    setSyncModalVisible(true);
    setIsRefreshing(true);
    setIsLoading(true);
    
    // Add timeout to prevent modal from getting stuck
    const timeoutId = setTimeout(() => {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setIsRefreshing(false);
        showToast('error', 'Timeout', 'Request timed out. Please try again.');
      }
    }, 30000); // 30 seconds timeout
    
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        if (isMounted.current) {
          setLogBookEntries([]);
          setFilteredEntries([]);
          setSyncModalVisible(false);
          setIsRefreshing(false);
        }
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      console.log('📋 Fetching logbook entries for user:', userUid);

      let entriesArray = [];
      
      // Fetch user data first
      if (isOnline) {
        try {
          console.log('Fetching user data from MongoDB...');
          const response = await MongoAPI.getUserById(userUid);
          
          if (response.success && response.data) {
            const userDataFromDB = response.data;
            
            // Normalize modules to array and extract Logbook permissions
            const modules = Array.isArray(userDataFromDB.modules) ? userDataFromDB.modules : 
                            (userDataFromDB.modules ? Object.values(userDataFromDB.modules) : []);
            
            // Find Logbook module
            const logBookModule = modules.find(module => 
              module.module && (
                module.module === 'logbook' ||
                module.module === 'Logbook' ||
                module.module.toLowerCase() === 'logbook'
              )
            );
            
            const userDataObj = {
              countries: userDataFromDB.countries || [],
              projects: userDataFromDB.projects || [],
              logBookPermissions: logBookModule?.permissions || [],
            };
            
            console.log('Logbook permissions:', userDataObj.logBookPermissions);
            
            if (isMounted.current) {
              setUserData(userDataObj);
            }
            
            await AsyncStorage.setItem('userData', JSON.stringify(userDataObj));
          } else {
            if (isMounted.current) {
              setUserData({ countries: [], projects: [], logBookPermissions: [] });
            }
            showToast('error', 'Error', 'User data not found.');
          }
        } catch (error) {
          console.error('Error fetching user data:', error);
          showToast('error', 'Error', 'Failed to load user data.');
        }

        // Fetch logbook entries from MongoDB
        try {
          console.log('Fetching logbook entries from MongoDB...');
          const response = await fetch(`${MONGODB_LOGBOOK_API_URL}/logbook/${userUid}`);
          
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }
          
          const result = await response.json();
          
          if (result.success && Array.isArray(result.data)) {
            entriesArray = result.data;
            console.log(`✅ Found ${entriesArray.length} logbook entries`);
            
            // Store in AsyncStorage for offline access
            await AsyncStorage.setItem('logbookEntries', JSON.stringify(entriesArray));
          } else {
            console.log('No logbook entries found');
            entriesArray = [];
          }
        } catch (error) {
          console.error('Error fetching logbook entries:', error);
          showToast('error', 'Error', 'Failed to load logbook entries.');
          
          // Try to load from AsyncStorage
          const cachedEntries = await AsyncStorage.getItem('logbookEntries');
          if (cachedEntries) {
            entriesArray = JSON.parse(cachedEntries);
          }
        }
      } else {
        // Load from AsyncStorage when offline
        const cachedEntries = await AsyncStorage.getItem('logbookEntries');
        if (cachedEntries) {
          entriesArray = JSON.parse(cachedEntries);
          showToast('info', 'Offline', 'Showing cached logbook entries.');
        }
      }

      if (isMounted.current) {
        setLogBookEntries(entriesArray);
        setFilteredEntries(entriesArray);
      }
    } catch (error) {
      console.error('Error in fetchLogBookEntries:', error);
      showToast('error', 'Error', 'Failed to load data.');
    } finally {
      clearTimeout(timeoutId);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setIsRefreshing(false);
        setIsLoading(false);
      }
    }
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (isMounted.current) {
        setIsOnline(state.isConnected);
        if (state.isConnected) {
          syncPendingOperations();
        }
      }
    });

    checkNetworkStatus().then((connected) => {
      if (isMounted.current) {
        setIsOnline(connected);
        if (connected) {
          syncPendingOperations();
        }
      }
    });

    fetchLogBookEntries();

    return () => {
      isMounted.current = false;
      unsubscribe();
      // Clean up animation
      rotateAnim.setValue(0);
    };
  }, []);

  useFocusEffect(
    React.useCallback(() => {
      if (isMounted.current) {
        fetchLogBookEntries();
      }
    }, [])
  );

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filtered = logBookEntries.filter(
        (entry) =>
          entry.selectedEquipment?.toLowerCase().includes(text.toLowerCase()) ||
          entry.entryNumber?.toLowerCase().includes(text.toLowerCase()) ||
          entry.operatorName?.toLowerCase().includes(text.toLowerCase()) ||
          entry.workLocation?.toLowerCase().includes(text.toLowerCase()) ||
          entry.workType?.toLowerCase().includes(text.toLowerCase()) ||
          entry.country?.toLowerCase().includes(text.toLowerCase()) ||
          entry.project?.toLowerCase().includes(text.toLowerCase())
      );
      setFilteredEntries(filtered);
    } else {
      setFilteredEntries(logBookEntries);
    }
  };

  const handleRefresh = () => {
    if (!isMounted.current) return;
    
    setIsRefreshing(true);
    setSyncModalVisible(true);
    setIsLoading(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    fetchLogBookEntries().finally(() => {
      if (isMounted.current) {
        rotateAnim.setValue(0);
      }
    });
  };

  const handleActionPress = (item) => {
    setSelectedItem(item);
    setDropdownVisible(dropdownVisible === item.id ? null : item.id);
  };

  const handleAction = async (action) => {
    if (!selectedItem) return;

    if (action === 'Edit') {
      navigation.navigate('LogBookEntry', { entry: selectedItem });
    } else if (action === 'Delete') {
      Alert.alert(
        'Confirm Deletion',
        `Are you sure you want to delete Entry# ${selectedItem.entryNumber}?`,
        [
          {
            text: 'Cancel',
            style: 'cancel',
          },
          {
            text: 'Delete',
            style: 'destructive',
            onPress: async () => {
              try {
                console.log('🗑️ Deleting logbook entry:', selectedItem.id);

                // Delete from MongoDB
                const response = await fetch(`${MONGODB_LOGBOOK_API_URL}/logbook/${selectedItem.id}`, {
                  method: 'DELETE',
                  headers: {
                    'Content-Type': 'application/json',
                  },
                });

                if (!response.ok) {
                  throw new Error(`Failed to delete entry: ${response.status}`);
                }

                const result = await response.json();
                
                if (result.success) {
                  console.log('✅ Entry deleted successfully');
                  
                  // Update local state
                  let updatedEntries = logBookEntries.filter((entry) => entry.id !== selectedItem.id);
                  
                  if (isMounted.current) {
                    setLogBookEntries(updatedEntries);
                    setFilteredEntries(updatedEntries);
                  }
                  
                  // Remove from AsyncStorage
                  await AsyncStorage.removeItem(`logbook_${selectedItem.id}`);
                  
                  // Update cached entries
                  await AsyncStorage.setItem('logbookEntries', JSON.stringify(updatedEntries));
                  
                  showToast('success', 'Success', 'Entry deleted successfully.');
                } else {
                  throw new Error(result.message || 'Failed to delete entry');
                }
              } catch (error) {
                console.error('Error deleting entry:', error);
                let errorMessage = 'Failed to delete entry.';
                if (error.message.includes('Failed to fetch')) {
                  errorMessage = 'Network error. Please check your connection.';
                }
                showToast('error', 'Error', errorMessage);
              }
            },
          },
        ]
      );
    }
    setDropdownVisible(null);
  };

  const DROPDOWN_OPTIONS = [
    {
      label: 'Edit',
      value: 'Edit',
      icon: <MaterialIcons name="edit" size={width * 0.05} color="#015185" />,
    },
    {
      label: 'Delete',
      value: 'Delete',
      icon: <MaterialIcons name="delete" size={width * 0.05} color="#FF0000" />,
    },
  ];

  const formatTime = (isoString) => {
    if (!isoString) return '-';
    const date = new Date(isoString);
    return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', hour12: true });
  };

  const renderLogItem = ({ item, index }) => (
    <View style={[styles.logItem, dropdownVisible === item.id && { zIndex: 999999 }]}>
      <View style={styles.logHeader}>
        <Text style={styles.logTitle}>Entry#: {item.entryNumber}</Text>
        <View style={styles.statusContainer}>
          <Text style={styles.statusText}>{item.equipmentStatus || 'In Maintenance'}</Text>
        </View>
      </View>
      <Text style={styles.logDetail}>Vehicle: {item.selectedEquipment || '-'}</Text>
      <Text style={styles.logDetail}>Country: {item.country || '-'}</Text>
      <Text style={styles.logDetail}>Project: {item.project || '-'}</Text>
      <View style={styles.table}>
        <View style={styles.tableRow}>
          <View style={[styles.tableCell, styles.headerCell, styles.firstHeaderCell, { flex: 1 }]}>
            <Text style={styles.tableHeaderText}></Text>
          </View>
          <View style={[styles.tableCell, styles.headerCell, { flex: 2 }]}>
            <Text style={styles.tableHeaderText}>Start</Text>
          </View>
          <View style={[styles.tableCell, styles.headerCell, styles.lastHeaderCell, { flex: 2 }]}>
            <Text style={styles.tableHeaderText}>End</Text>
          </View>
        </View>
        <View style={styles.tableRow}>
          <View style={[styles.tableCell, styles.regularCell, { flex: 1 }]}>
            <Text style={styles.tableCellText}>Meter Reading</Text>
          </View>
          <View style={[styles.tableCell, styles.regularCell, { flex: 2 }]}>
            <Text style={styles.tableCellText}>{item.startReading || '-'}</Text>
          </View>
          <View style={[styles.tableCell, styles.lastColumnCell, { flex: 2 }]}>
            <Text style={styles.tableCellText}>{item.endReading || '-'}</Text>
          </View>
        </View>
        <View style={styles.tableRow}>
          <View style={[styles.tableCell, styles.regularCell, { flex: 1 }]}>
            <Text style={styles.tableCellText}>Time</Text>
          </View>
          <View style={[styles.tableCell, styles.regularCell, { flex: 2 }]}>
            <Text style={styles.tableCellText}>{formatTime(item.startDateTime)}</Text>
          </View>
          <View style={[styles.tableCell, styles.lastColumnCell, { flex: 2 }]}>
            <Text style={styles.tableCellText}>{formatTime(item.endDateTime)}</Text>
          </View>
        </View>
        <View style={[styles.tableRow, styles.lastRow]}>
          <View style={[styles.tableCell, styles.regularCell, { flex: 1 }]}>
            <Text style={styles.tableCellText}>Fuel</Text>
          </View>
          <View style={[styles.tableCell, styles.regularCell, { flex: 2 }]}>
            <Text style={styles.tableCellText}>{item.fuelOpening || '-'}</Text>
          </View>
          <View style={[styles.tableCell, styles.lastColumnCell, { flex: 2 }]}>
            <Text style={styles.tableCellText}>{item.fuelConsumption || '-'}</Text>
          </View>
        </View>
      </View>
      <Text style={styles.logDetail}>Material: {item.materialName || '-'}</Text>
      <Text style={styles.logDetail}>Issue: {item.issueReport || '-'}</Text>
      <View style={styles.footerContainer}>
        <Text style={styles.logDetail}>
          Created Date: {item.createdAt ? new Date(item.createdAt).toLocaleString() : '-'}
        </Text>
        <View style={styles.menuContainer}>
          <Dropdown
            style={styles.dropdown}
            data={DROPDOWN_OPTIONS}
            labelField="label"
            valueField="value"
            value={null}
            onChange={(option) => handleAction(option.value)}
            renderRightIcon={() => (
              <MaterialIcons name="more-vert" size={width * 0.05} color="#333" />
            )}
            renderItem={(option) => (
              <View style={styles.dropdownItem}>
                {option.icon}
                <Text
                  style={[
                    styles.dropdownText,
                    option.value === 'Delete' ? { color: '#FF0000' } : { color: '#015185' },
                  ]}
                >
                  {option.label}
                </Text>
              </View>
            )}
            placeholder={''}
            showsVerticalScrollIndicator={false}
            dropdownPosition={'auto'}
            containerStyle={styles.dropdownContainer}
            itemTextStyle={styles.dropdownItemText}
            selectedTextStyle={{ display: 'none' }}
          />
        </View>
      </View>
    </View>
  );

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.navigate('MainApp')}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Logbook</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionModalVisible(true)}
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
        </View>
      </LinearGradient>

      <FlatList
        data={filteredEntries}
        renderItem={renderLogItem}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.listContent}
        refreshing={isRefreshing}
        onRefresh={handleRefresh}
        ListEmptyComponent={
          <Text style={styles.emptyText}>No logbook entries found.</Text>
        }
      />

      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.startButton}
            onPress={() => navigation.replace('LogBookEntry')}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>New Entry</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.loadingModalContainer}>
          <View style={styles.loadingModalContent}>
            <Text style={styles.loadingModalText}>Please Wait</Text>
            <View style={styles.loadingIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.loadingModalTextadd}>Processing of cloud data syncing</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={styles.permissionModalOverlay}>
          <View style={styles.permissionModalContent}>
            <Text style={styles.permissionModalTitle}>User Permissions</Text>
            
            <Text style={styles.permissionModalLabel}>Logbook Permissions:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.logBookPermissions && userData.logBookPermissions.length > 0 ? userData.logBookPermissions.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Countries:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.countries && userData.countries.length > 0 ? userData.countries.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Projects:</Text>
            <Text style={styles.permissionModalValue}>
              {userData.projects && userData.projects.length > 0 ? userData.projects.join(', ') : 'None'}
            </Text>
            
            <TouchableOpacity
              style={styles.permissionCloseButton}
              onPress={() => setPermissionModalVisible(false)}
            >
              <Text style={styles.permissionCloseButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#ECF0F3',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  loadingModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  loadingIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  loadingModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  listContent: {
    padding: width * 0.04,
    paddingBottom: width * 0.4,
  },
  logItem: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ccc',
    position: 'relative',
  },
  logHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  logTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#000',
  },
  logDetail: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.01,
  },
  table: {
    marginVertical: width * 0.02,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: width * 0.01,
    overflow: 'hidden',
  },
  tableRow: {
    flexDirection: 'row',
    borderBottomWidth: 1,
    borderBottomColor: '#ccc',
  },
  tableCell: {
    borderRightWidth: 1,
    borderRightColor: '#ccc',
    padding: width * 0.02,
  },
  headerCell: {
    backgroundColor: '#f0f0f0',
    borderRightWidth: 0,
    borderLeftWidth: 0,
  },
  firstHeaderCell: {
    borderLeftWidth: 0,
    borderRightWidth: 0,
  },
  lastHeaderCell: {
    borderRightWidth: 0,
  },
  regularCell: {
    borderRightWidth: 1,
    borderRightColor: '#ccc',
  },
  lastColumnCell: {
    borderRightWidth: 0,
  },
  lastRow: {
    borderBottomWidth: 0,
  },
  tableHeaderText: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#333',
    textAlign: 'center',
  },
  tableCellText: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
  },
  footerContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginTop: width * 0.02,
    position: 'relative',
  },
  menuContainer: {
    position: 'relative',
  },
  dropdown: {
    position: 'absolute',
    top: -width * 0.02,
    right: width * 0.06,
    backgroundColor: 'transparent',
    borderRadius: width * 0.02,
    minWidth: width * 0.3,
    zIndex: 999999,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginLeft: width * 0.02,
  },
  statusContainer: {
    marginTop: width * 0.02,
    alignSelf: 'flex-end',
    backgroundColor: '#00C4B4',
    borderRadius: width * 0.05,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.01,
  },
  statusText: {
    fontSize: width * 0.035,
    color: '#fff',
    fontWeight: 'bold',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  dropdownContainer: {
    // Add any additional styles for the dropdown container
  },
  dropdownItemText: {
    // Add any additional styles for the dropdown item text
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  permissionSection: {
    marginBottom: width * 0.04,
    width: '100%',
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  permissionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.02,
    marginBottom: width * 0.01,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    fontStyle: 'italic',
  },
  buttonWrapper: {
    width: '100%',
    alignItems: 'flex-end',
    marginTop: 10,
  },
  closeButton: {
    backgroundColor: '#025383',
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  filterModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  filterModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'flex-start',
  },
  permissionModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '80%',
    alignItems: 'flex-start',
    elevation: 5,
  },
  permissionModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    alignSelf: 'center',
    width: '100%',
    textAlign: 'center',
  },
  permissionModalLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 10,
  },
  permissionModalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 4,
  },
  permissionCloseButton: {
    alignSelf: 'center',
    marginTop: 18,
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 24,
    paddingVertical: 8,
  },
  permissionCloseButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
});

export default LogbookList;