import React, { useState, useEffect, useRef, useCallback, useMemo } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  Easing, // <-- Add this import
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import OfflineRequestMaintenanceHelper from './OfflineRequestMaintenanceHelper';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

// MongoDB API URLs
const MONGODB_REQUEST_MAINTENANCE_API_URL = 'http://168.231.113.219:5009';
const REQUEST_MAINTENANCE_NOTIFICATIONS_API_URL = 'http://168.231.113.219:5022';

const DROPDOWN_OPTIONS = [
  {
    label: 'Approve',
    value: 'approve',
    icon: <MaterialIcons name="check-circle" size={22} color="#3481BC" />,
  },

  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#C2185B" />,
  },
];

// Define priority icons and colors
const priorityIcons = {
  Low: { icon: 'arrow-downward', color: '#4ECB71' }, // Green
  Medium: { icon: 'priority-high', color: '#FFC107' }, // Yellow
  High: { icon: 'warning', color: '#FF5722' }, // Orange
  Emergency: { icon: 'error', color: '#C2185B' }, // Red
};

// Define status badge colors
const statusColors = {
  Requested: '#3481BC', // Blue
  Approved: '#4ECB71',  // Green
  Rejected: '#C2185B',  // Red
};

// SyncModal Component
const SyncModal = ({ visible, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    let animation;
    if (visible) {
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 900,
          useNativeDriver: true,
          easing: Easing.linear, // <-- Use Easing.linear
        })
      );
      animation.start();
    } else {
      rotateAnim.setValue(0);
      if (animation) animation.stop();
    }
    return () => {
      if (animation) animation.stop && animation.stop();
    };
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const RequestmaintenanceHome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [maintenanceRequests, setMaintenanceRequests] = useState([]);
  const [filteredRequests, setFilteredRequests] = useState([]);
  const [userId, setUserId] = useState(null);
  const [selectedRequest, setSelectedRequest] = useState(null);
  const [deleteConfirmVisible, setDeleteConfirmVisible] = useState(false);
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    requestMaintenancePermissions: [],
    name: '',
    email: '',
  });

  // Add state for dropdown menu
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [dropdownRequest, setDropdownRequest] = useState(null);
  
  // Offline state tracking
  const [isOnline, setIsOnline] = useState(true);
  const [offlineRequestsCount, setOfflineRequestsCount] = useState(0);
  const [onlineRequestsCount, setOnlineRequestsCount] = useState(0);
  const [localStorageSize, setLocalStorageSize] = useState('0 KB');
  const [isSyncing, setIsSyncing] = useState(false);
  const isSyncingOfflineRequests = useRef(false);
  const [offlineUpdateIds, setOfflineUpdateIds] = useState([]);

  // Memoized permission checks to prevent re-renders
  const hasOnlyMineView = useMemo(() => {
    return userData.requestMaintenancePermissions.some(p => 
      p && typeof p === 'string' && p.toLowerCase() === 'onlymineview'
    );
  }, [userData.requestMaintenancePermissions]);

  const hasPermission = useCallback((permission) => {
    return userData.requestMaintenancePermissions.includes(permission);
  }, [userData.requestMaintenancePermissions]);

  const hasCreatePermission = useMemo(() => hasPermission('create'), [hasPermission]);
  const hasEditPermission = useMemo(() => hasPermission('edit'), [hasPermission]);
  const hasViewPermission = useMemo(() => hasPermission('view'), [hasPermission]);
  const hasDeletePermission = useMemo(() => hasPermission('delete'), [hasPermission]);
  const hasApprovePermission = useMemo(() => hasPermission('approve'), [hasPermission]);
  const hasUpdatePermission = useMemo(() => hasPermission('update'), [hasPermission]);

  // Filter dropdown options based on permissions
  const filteredDropdownOptions = useMemo(() => {
    return DROPDOWN_OPTIONS.filter(option => {
      switch (option.value) {
        case 'approve':
          return hasApprovePermission || hasUpdatePermission;
        case 'edit':
          return hasEditPermission;
        case 'view':
          return hasViewPermission;
        case 'delete':
          return hasDeletePermission;
        default:
          return true;
      }
    });
  }, [hasEditPermission, hasViewPermission, hasDeletePermission, hasApprovePermission, hasUpdatePermission]);

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      const online = netInfo.isConnected && netInfo.isInternetReachable !== false;
      setIsOnline(online);
      return online;
    } catch (error) {
      console.error('Error checking network status:', error);
      setIsOnline(false);
      return false;
    }
  };
  
  // Update request counts (offline vs online)
  const updateRequestCounts = async () => {
    try {
      // Get offline pending sync requests (created offline, not yet in MongoDB)
      const offlinePendingRequests = await OfflineRequestMaintenanceHelper.getOfflineRequests();
      const pendingCount = offlinePendingRequests.length;
      
      // Get cached requests from MongoDB (synced and stored offline)
      const cachedRequestsRaw = await AsyncStorage.getItem('globalMaintenanceRequests');
      const cachedRequests = cachedRequestsRaw ? JSON.parse(cachedRequestsRaw) : [];
      const cachedCount = cachedRequests.length;
      
      // TOTAL OFFLINE STORED = Pending + Cached (ALL requests available offline)
      const totalOfflineCount = pendingCount + cachedCount;
      setOfflineRequestsCount(totalOfflineCount);
      
      // Online in DB count = Just the cached requests (synced to MongoDB)
      setOnlineRequestsCount(cachedCount);
      
      // Load offline update IDs (approve/reject pending sync)
      const offlineUpdates = await OfflineRequestMaintenanceHelper.getOfflineUpdates();
      const updateIds = offlineUpdates.map(u => u.id);
      setOfflineUpdateIds(updateIds);
      console.log('📋 Offline update IDs:', updateIds);
      
      // Calculate local storage size
      try {
        let totalSize = 0;
        const keys = await AsyncStorage.getAllKeys();
        const maintenanceKeys = keys.filter(key => 
          key.includes('maintenance') || 
          key.includes('request') || 
          key.includes('offline')
        );
        
        for (const key of maintenanceKeys) {
          const value = await AsyncStorage.getItem(key);
          if (value) {
            totalSize += value.length;
          }
        }
        
        // Convert to KB/MB
        const sizeInKB = (totalSize / 1024).toFixed(2);
        const sizeInMB = (totalSize / (1024 * 1024)).toFixed(2);
        const displaySize = totalSize > 1024 * 1024 ? `${sizeInMB} MB` : `${sizeInKB} KB`;
        setLocalStorageSize(displaySize);
      } catch (storageError) {
        // Silent fail
      }
      
    } catch (error) {
      console.error('❌ Count update error:', error.message);
    }
  };

  // Show toast notification
  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };
  
  // Sync offline requests to MongoDB
  const syncOfflineRequests = async () => {
    if (!(await checkNetworkStatus())) return { success: true, synced: 0, failed: 0 };
    
    // Prevent concurrent syncs
    if (isSyncingOfflineRequests.current) {
      console.log('⚠️ Offline sync already in progress, skipping...');
      return { success: true, synced: 0, failed: 0 };
    }

    try {
      isSyncingOfflineRequests.current = true;
      
      if (!userId) return { success: false, synced: 0, failed: 0 };
      
      console.log('🔒 Starting offline maintenance requests sync (locked)...');

      // Sync offline requests using OfflineRequestMaintenanceHelper
      const syncResult = await OfflineRequestMaintenanceHelper.syncOfflineRequests(async (request) => {
        try {
          console.log('📤 Syncing offline maintenance request:', request.requestNumber);
          
          // Generate dynamic request number based on current MongoDB count
          let newRequestNumber = request.requestNumber;
          
          // CRITICAL: Remove "-OFFLINE" suffix if present
          if (newRequestNumber && newRequestNumber.includes('-OFFLINE')) {
            newRequestNumber = newRequestNumber.replace('-OFFLINE', '');
            console.log('🧹 Removed -OFFLINE suffix from:', request.requestNumber, '→', newRequestNumber);
          }
          
          try {
            console.log('🔢 Generating new request number from MongoDB count...');
            const response = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance`);
            if (response.ok) {
              const result = await response.json();
              if (result.success && result.data && result.data.length > 0) {
                const sortedRequests = result.data.sort((a, b) => 
                  new Date(b.createdAt) - new Date(a.createdAt)
                );
                const lastNumber = sortedRequests[0].requestNumber;
                
                // Extract number from last request
                const match = lastNumber.match(/-RM-(\d+)/);
                const nextNumber = match ? parseInt(match[1], 10) + 1 : 1;
                
                // Generate clean request number
                const country = request.country || 'Unknown';
                const project = request.project || 'Unknown';
                const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
                const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
                
                newRequestNumber = `${countryCode}-${projectCode}-RM-${String(nextNumber).padStart(3, '0')}`;
                console.log('✅ Generated clean request number:', newRequestNumber);
                console.log('   Old (offline):', request.requestNumber);
                console.log('   New (MongoDB):', newRequestNumber);
                
                // Update request with new clean request number
                request.requestNumber = newRequestNumber;
              }
            }
          } catch (numberError) {
            console.warn('⚠️ Failed to generate dynamic request number:', numberError.message);
            // Use cleaned version (without -OFFLINE) if generation fails
            request.requestNumber = newRequestNumber;
          }
          
          // Check if request already exists in MongoDB to prevent duplicates
          try {
            const existingRequest = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance/${request.id}`);
            if (existingRequest.ok) {
              const existingResult = await existingRequest.json();
              if (existingResult.success && existingResult.data) {
                console.log('⚠️ Request already exists in MongoDB, skipping:', request.id);
                return { success: true, skipped: true, message: 'Request already synced' };
              }
            }
          } catch (checkError) {
            // Request doesn't exist, proceed with creation
            console.log('✅ Request not found in MongoDB, creating...');
          }
          
          // Create the maintenance request in MongoDB with updated request number
          const createResponse = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(request),
          });
          
          if (!createResponse.ok) {
            throw new Error(`MongoDB save failed: ${createResponse.status}`);
          }
          
          const createResult = await createResponse.json();
          
          if (createResult.success) {
            console.log('✅ Request created in MongoDB:', newRequestNumber);
            
            // Send notifications for synced offline request
            setTimeout(() => {
              (async () => {
                try {
                  console.log('🔔 Sending notifications for synced offline maintenance request...');
                  
                  const notificationData = {
                    maintenanceData: {
                      _id: request.id,
                      id: request.id,
                      requestNumber: newRequestNumber,
                      workOrderTitle: request.workOrderTitle,
                      equipmentName: request.equipment?.equipmentName || 'N/A',
                      priority: request.priority || 'Low',
                      requestType: request.requestType || 'General',
                      country: request.country,
                      project: request.project,
                      description: request.description || '',
                      tasksCount: request.tasks?.length || 0,
                    },
                    createdByName: userData.name || 'User',
                    requesterEmail: userData.email || '',
                  };
                  
                  const notifResponse = await fetch(`${REQUEST_MAINTENANCE_NOTIFICATIONS_API_URL}/api/maintenance-notifications/new-request`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(notificationData),
                    timeout: 10000,
                  });
                  
                  if (notifResponse.ok) {
                    const notifResult = await notifResponse.json();
                    console.log('✅ Maintenance notifications sent successfully for synced request');
                    console.log(`  📱 Push: ${notifResult.results?.push?.sent || 0} device(s)`);
                    console.log(`  📧 Emails: ${notifResult.results?.emails?.sent || 0}`);
                    console.log(`  💾 MongoDB: ${notifResult.results?.mongodb?.saved || 0} notification(s)`);
                  } else {
                    console.error('❌ Failed to send maintenance notifications for synced request');
                  }
                } catch (notifError) {
                  console.error('❌ Error sending maintenance notifications:', notifError);
                  // Don't fail sync if notifications fail
                }
              })();
            }, 500);
            
            return { success: true };
          } else {
            throw new Error('Failed to create maintenance request in MongoDB');
          }
        } catch (error) {
          console.error('❌ Sync error for request:', error.message);
          return { success: false, error: error.message };
        }
      });
      
      if (syncResult.synced > 0) {
        showToast('success', 'Sync Complete', `${syncResult.synced} offline requests synced to MongoDB`);
        console.log('📊 Sync Summary:', {
          synced: syncResult.synced,
          failed: syncResult.failed,
          total: syncResult.synced + syncResult.failed
        });
        
        // Reload requests from MongoDB to update UI and remove SYNC badges
        console.log('🔄 Reloading requests from MongoDB to update UI...');
        try {
          await fetchMaintenanceRequests();
          await updateRequestCounts();
          console.log('✅ UI refreshed - SYNC badges removed');
        } catch (refreshError) {
          console.error('❌ Failed to refresh UI after sync:', refreshError.message);
        }
      }
      
      if (syncResult.failed > 0) {
        showToast('warning', 'Partial Sync', `${syncResult.failed} requests failed to sync`);
      }
      
      console.log('🔓 Offline sync completed (unlocked)');
      return syncResult;
    } catch (error) {
      console.error('❌ Sync offline error:', error.message);
      return { success: false, synced: 0, failed: 0 };
    } finally {
      isSyncingOfflineRequests.current = false; // Always unlock
    }
  };

  // Sync offline updates (approve/reject) to MongoDB
  const syncOfflineUpdates = async () => {
    if (!(await checkNetworkStatus())) return { success: true, synced: 0, failed: 0 };

    try {
      const syncResult = await OfflineRequestMaintenanceHelper.syncOfflineUpdates(async (update) => {
        try {
          const { id, updatedRequest } = update;
          const response = await fetch(`${MONGODB_REQUEST_MAINTENANCE_API_URL}/request-maintenance/${id}`, {
            method: 'PUT',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(updatedRequest),
          });
          if (!response.ok) {
            return { success: false };
          }
          const result = await response.json();
          if (result.success) {
            // update local cache entry
            try {
              const stored = await AsyncStorage.getItem('globalMaintenanceRequests');
              let list = stored ? JSON.parse(stored) : [];
              list = list.map((r) => (r.id === id ? { ...r, ...updatedRequest } : r));
              await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(list));
            } catch {}
            return { success: true };
          }
          return { success: false };
        } catch (e) {
          return { success: false };
        }
      });

      if (syncResult.synced > 0) {
        showToast('success', 'Synced', `${syncResult.synced} approvals updated to MongoDB`);
        await fetchMaintenanceRequests();
        await updateRequestCounts();
      }
      if (syncResult.failed > 0) {
        showToast('warning', 'Partial', `${syncResult.failed} approvals failed to sync`);
      }
      return syncResult;
    } catch (e) {
      return { success: false, synced: 0, failed: 0 };
    }
  };

  // Fetch user UID and data from AsyncStorage and MongoDB (with offline support)
  useEffect(() => {
    const loadUserData = async () => {
      try {
        const uid = await AsyncStorage.getItem('userUid');
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (!uid || !userEmail) {
          showToast('error', 'Error', 'User not logged in. Please log in again.');
          navigation.replace('AdminLogin');
          return;
        }

        setUserId(uid);
        
        let loadedUserData = null;

        // PRIORITY 1: Try requestMaintenanceModuleUserInfo first (most specific)
        try {
          const cachedRequestMaintenanceUserInfo = await AsyncStorage.getItem('requestMaintenanceModuleUserInfo');
          if (cachedRequestMaintenanceUserInfo) {
            loadedUserData = JSON.parse(cachedRequestMaintenanceUserInfo);
            setUserData(loadedUserData);
            console.log('✅ Loaded permissions from requestMaintenanceModuleUserInfo:', loadedUserData.requestMaintenancePermissions);
          }
        } catch (error) {
          console.error('❌ Error loading requestMaintenanceModuleUserInfo:', error.message);
        }

        // PRIORITY 2: Fallback to full userData
        if (!loadedUserData) {
          try {
            const cachedUserData = await AsyncStorage.getItem('userData');
            if (cachedUserData) {
              const userData = JSON.parse(cachedUserData);
              const requestMaintenanceModule = userData.modules?.find((module) => module.module === 'requestMaintenance');
              
              const userInfoData = {
                countries: Array.isArray(userData.countries) ? userData.countries : [],
                projects: Array.isArray(userData.projects) ? userData.projects : [],
                requestMaintenancePermissions: requestMaintenanceModule ? (Array.isArray(requestMaintenanceModule.permissions) ? requestMaintenanceModule.permissions : []) : [],
                name: userData.name || userData.fullName || 'Admin',
                email: userData.email || userEmail,
              };
              
              setUserData(userInfoData);
              loadedUserData = userInfoData;
              console.log('✅ Loaded permissions from userData:', userInfoData.requestMaintenancePermissions);
              
              // Store for next time
              await AsyncStorage.setItem('requestMaintenanceModuleUserInfo', JSON.stringify(userInfoData));
            }
          } catch (error) {
            console.error('❌ Error loading userData:', error.message);
          }
        }
        
        // PRIORITY 3: Final fallback - restore from cached requests
        if (!loadedUserData) {
          try {
            const cachedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
            if (cachedRequests) {
              const requests = JSON.parse(cachedRequests);
              if (requests.length > 0 && requests[0].userPermissions) {
                const firstRequest = requests[0];
                const userInfoData = {
                  countries: firstRequest.userCountries || [],
                  projects: firstRequest.userProjects || [],
                  requestMaintenancePermissions: firstRequest.userPermissions || [],
                  name: '',
                  email: '',
                };
                
                if (userInfoData.requestMaintenancePermissions.length > 0) {
                  setUserData(userInfoData);
                  loadedUserData = userInfoData;
                  console.log('✅ Restored permissions from cached requests:', userInfoData.requestMaintenancePermissions);
                  
                  // Store for next time
                  await AsyncStorage.setItem('requestMaintenanceModuleUserInfo', JSON.stringify(userInfoData));
                }
              }
            }
          } catch (error) {
            console.error('❌ Error loading from cached requests:', error.message);
          }
        }
        
        // If we loaded from cache, log success
        if (loadedUserData) {
          console.log('✅ UserInfo loaded from cache - Offline mode ready');
        } else {
          console.log('⚠️ No cached userInfo found');
        }

        // Check if online - only try MongoDB if online
        const onlineStatus = await checkNetworkStatus();
        
        if (!onlineStatus) {
          // Offline - cache already loaded (or not)
          console.log('📴 Offline - Using cached data only');
          return;
        }
        
        // Online - try to fetch from MongoDB to update cache
        console.log('🌐 Online - Fetching fresh data from MongoDB...');
        try {
          const response = await fetch(`http://168.231.113.219:5009/admin-users/uid/${uid}`);
          
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }
          
          const result = await response.json();
          
          if (result.success && result.data) {
            const data = result.data;

            // Extract requestMaintenance permissions (case sensitive)
            const requestMaintenanceModule = data.modules?.find(mod => mod.module === 'requestMaintenance');
            const requestMaintenancePermissions = requestMaintenanceModule?.permissions || [];

            const freshUserData = {
              countries: data.countries || [],
              projects: data.projects || [],
              requestMaintenancePermissions: requestMaintenancePermissions,
              name: data.name || data.fullName || 'Admin',
              email: data.email || userEmail,
            };
            
            setUserData(freshUserData);
            console.log('✅ Updated permissions from MongoDB:', freshUserData.requestMaintenancePermissions);
            
            // Update cache
            await AsyncStorage.setItem('userData', JSON.stringify(data));
            await AsyncStorage.setItem('requestMaintenanceModuleUserInfo', JSON.stringify(freshUserData));
          }
        } catch (error) {
          console.error('❌ MongoDB fetch failed:', error.message);
          console.log('📴 Using cached permissions (MongoDB unavailable)');
        }
      } catch (error) {
        console.error('❌ User data error:', error.message);
      }
    };

    loadUserData();
  }, [navigation]);
  
  // Network status monitoring
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      const currentOnlineStatus = state.isConnected && state.isInternetReachable !== false;
      const wasOffline = !isOnline;
      const isNowOnline = currentOnlineStatus;
      
      setIsOnline(currentOnlineStatus);
      
      // Show toast notifications for status changes
      if (wasOffline && isNowOnline) {
        // Device came back online
        console.log('🌐 Connection restored');
        showToast('success', 'Online', 'Connection restored - Use cloud sync to upload requests');
        
        // Reload data from MongoDB only (NO AUTO-SYNC of offline requests)
        setTimeout(async () => {
          try {
            console.log('🌐 Reloading data from MongoDB (no auto-sync)');
            await fetchMaintenanceRequests();
            showToast('info', 'Data Refreshed', 'Use cloud button to sync offline requests');
          } catch (error) {
            console.error('Reload error:', error);
          }
        }, 1000);
      } else if (!wasOffline && !isNowOnline) {
        // Device went offline
        console.log('📴 Connection lost - Switching to offline mode');
        showToast('info', 'Offline', 'No internet connection - Using cached data');
      }
    });

    return () => {
      unsubscribe();
    };
  }, [isOnline, fetchMaintenanceRequests]);

  // Memoized filter function
  const filterRequestsByPermissions = useCallback((requests) => {
    if (!requests || requests.length === 0) return [];

    let filteredRequests = requests;

    // If user has onlyMineView permission, show only their own requests
    if (hasOnlyMineView) {
      filteredRequests = requests.filter(request => request.userId === userId);
    } else {
      // Filter by user's assigned countries and projects
      if (userData.countries.length > 0 || userData.projects.length > 0) {
        filteredRequests = requests.filter(request => {
          const matchesCountry = userData.countries.length === 0 || 
            (request.country && userData.countries.includes(request.country));
          const matchesProject = userData.projects.length === 0 || 
            (request.project && userData.projects.includes(request.project));
          return matchesCountry && matchesProject;
        });
      }
    }

    return filteredRequests;
  }, [userData.countries, userData.projects, userId, hasOnlyMineView]);

  // Fetch maintenance requests
  const fetchMaintenanceRequests = useCallback(async () => {
    if (!userId) return;
    setSyncModalVisible(true);
    try {
      const isOnline = await checkNetworkStatus();
      let requests = [];

      if (isOnline) {
        // If user has onlyMineView permission, fetch only their requests
        if (hasOnlyMineView) {
          const response = await fetch(`http://168.231.113.219:5009/request-maintenance/user/${userId}`);
          
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }
          
          const result = await response.json();
          requests = result.success ? result.data : [];
        } else {
          // Fetch all requests and filter by permissions
          const response = await fetch('http://168.231.113.219:5009/request-maintenance');
          
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }
          
          const result = await response.json();
          requests = result.success ? result.data : [];
        }
        
        // Add user permissions to each request for offline access
        requests = requests.map(req => ({
          ...req,
          userPermissions: userData.requestMaintenancePermissions || [],
          userCountries: userData.countries || [],
          userProjects: userData.projects || [],
          syncedAt: new Date().toISOString(),
        }));
        
        await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));
        showToast('success', 'Success', 'Maintenance requests synced successfully.');
      } else {
        const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
        requests = storedRequests ? JSON.parse(storedRequests) : [];
        showToast('info', 'Offline', 'Using offline data. Will sync when online.');
      }
      
      // Load offline requests and append
      const offlineRequests = await OfflineRequestMaintenanceHelper.getOfflineRequests();
      if (offlineRequests.length > 0) {
        const formattedOfflineRequests = offlineRequests.map(req => ({
          ...req,
          offlineCreated: true,
          offlineSyncPending: true,
        }));
        requests = [...requests, ...formattedOfflineRequests];
        console.log(`📱 Loaded ${offlineRequests.length} offline pending requests`);
      }

      // Sort requests by requestNumber descending (latest first)
      requests = requests.slice().sort((a, b) => {
        // Extract the numeric part from requestNumber (e.g., ZAM-KAL-RM-002 => 2)
        const getNum = (req) => {
          const match = (req.requestNumber || '').match(/(\d+)$/);
          return match ? parseInt(match[1], 10) : 0;
        };
        return getNum(b) - getNum(a);
      });

      // Apply permission-based filtering
      const filteredRequests = filterRequestsByPermissions(requests);
      setMaintenanceRequests(filteredRequests);
      setFilteredRequests(filteredRequests);
      
      // Update counts
      await updateRequestCounts();
    } catch (error) {
      console.error('Error fetching maintenance requests:', error);
      let errorMessage = 'Failed to fetch maintenance requests.';
      if (error.message.includes('404')) {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.message.includes('NETWORK_ERROR') || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Showing locally stored data.';
        const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
        const cachedRequests = storedRequests ? JSON.parse(storedRequests) : [];
        const offlineRequests = await OfflineRequestMaintenanceHelper.getOfflineRequests();
        const allRequests = [...cachedRequests, ...offlineRequests.map(req => ({
          ...req,
          offlineCreated: true,
          offlineSyncPending: true,
        }))];
        const filteredRequests = filterRequestsByPermissions(allRequests);
        setMaintenanceRequests(filteredRequests);
        setFilteredRequests(filteredRequests);
      }
      showToast('error', 'Error', errorMessage);
      await updateRequestCounts();
    } finally {
      setSyncModalVisible(false);
    }
  }, [userId, filterRequestsByPermissions, hasOnlyMineView, userData]);

  // Auto-refresh on screen focus (hard refresh every time)
  useFocusEffect(
    useCallback(() => {
      setSyncModalVisible(true);
      fetchMaintenanceRequests().finally(() => {
        setSyncModalVisible(false);
      });
    }, [fetchMaintenanceRequests, userId])
  );

  // Handle search
  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text.trim()) {
      const filtered = maintenanceRequests.filter(
        (request) =>
          (request.workOrderTitle?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.requestType?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.equipment?.equipmentName?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.requestNumber?.toLowerCase() || '').includes(text.toLowerCase())
      );
      setFilteredRequests(filtered);
    } else {
      setFilteredRequests(maintenanceRequests);
    }
  };

  // Handle refresh/sync
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    fetchMaintenanceRequests().finally(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
    });
  };
  
  // Cloud sync button handler
  const handleCloudSync = useCallback(async () => {
    if (isSyncing) return;
    
    try {
      setIsSyncing(true);
      setSyncModalVisible(true);
      
      // Check if online
      const onlineStatus = await checkNetworkStatus();
      
      if (!onlineStatus) {
        showToast('error', 'Offline', 'Cannot sync while offline');
        setIsSyncing(false);
        setSyncModalVisible(false);
        return;
      }

      if (!userId) {
        showToast('error', 'Error', 'No user is logged in');
        setIsSyncing(false);
        setSyncModalVisible(false);
        return;
      }

      console.log('🔄 Starting cloud sync for maintenance requests...');
      
      // Sync offline requests
      const offlineSyncResult = await syncOfflineRequests();
      // Sync offline approvals/updates
      const offlineUpdatesResult = await syncOfflineUpdates();
      
      if (offlineSyncResult && offlineSyncResult.synced > 0) {
        console.log(`✅ Synced ${offlineSyncResult.synced} offline requests to MongoDB`);
      }
      
      // Reload fresh requests from MongoDB
      await fetchMaintenanceRequests();
      
      showToast('success', 'Synced', `${offlineRequestsCount} requests now available offline`);
      
    } catch (error) {
      console.error('❌ Error during cloud sync:', error);
      showToast('error', 'Sync Failed', 'Could not sync from MongoDB');
    } finally {
      setIsSyncing(false);
      setSyncModalVisible(false);
    }
  }, [isSyncing, userId, syncOfflineRequests, fetchMaintenanceRequests, offlineRequestsCount]);

  // Defensive navigation handlers
  const handleApprove = (request) => {
    console.log('🔄 handleApprove called with request:', {
      request: request,
      requestId: request?.id,
      hasApprovePermission,
      hasUpdatePermission
    });
    
    if (request && request.id) {
      if (hasApprovePermission || hasUpdatePermission) {
        setDropdownVisible(false);
        setTimeout(() => {
          try {
            console.log('🚀 Navigating to RequestMApproveWO with requestId:', request.id);
            navigation.navigate('RequestMApproveWO', { requestId: request.id });
            console.log('✅ Navigation call completed');
          } catch (err) {
            console.error('❌ Navigation to RequestMApproveWO failed:', err, request);
            showToast('error', 'Navigation Error', 'Failed to open approval page.');
          }
        }, 100);
      } else {
        console.log('❌ No permission to approve');
        showToast('error', 'Permission Denied', 'You do not have permission to approve maintenance requests.');
      }
    } else {
      console.warn('❌ Invalid request for approve:', request);
      showToast('error', 'Invalid Request', 'Invalid request data for approval.');
    }
  };

  const handleEdit = (request) => {
    if (request && request.id && hasEditPermission) {
      setSyncModalVisible(false);
      setDropdownVisible(false);
      setDeleteConfirmVisible(false);
      setPermissionsModalVisible(false);
      setTimeout(() => {
        try {
          navigation.navigate('UpdateRequestMaintenance', { requestId: request.id });
        } catch (err) {
          console.error('Navigation to UpdateRequestMaintenance failed:', err, request);
        }
      }, 100);
    } else {
      console.warn('Invalid request for edit:', request);
    }
  };

  const handleView = (request) => {
    if (request && request.id && hasViewPermission) {
      setDropdownVisible(false);
      setTimeout(() => {
        try {
          navigation.navigate('ViewMRequest', { requestId: request.id });
        } catch (err) {
          console.error('Navigation to ViewMRequest failed:', err, request);
        }
      }, 100);
    } else {
      console.warn('No view permission or invalid request', request);
    }
  };

  const handleDelete = (request) => {
    if (request && request.id && hasDeletePermission) {
      setSelectedRequest(request);
      setDeleteConfirmVisible(true);
    } else {
      console.warn('Invalid request for delete:', request);
    }
  };

  const handleDropdownAction = async (action, request) => {
    // Defensive: close all modals before any navigation
    setSyncModalVisible(false);
    setDropdownVisible(false);
    setDeleteConfirmVisible(false);
    setPermissionsModalVisible(false);
    try {
      switch (action) {
        case 'approve':
          await handleApprove(request);
          break;
        case 'edit':
          await handleEdit(request);
          break;
        case 'view':
          await handleView(request);
          break;
        case 'delete':
          await handleDelete(request);
          break;
        default:
          break;
      }
    } catch (error) {
      console.error('Error processing action:', error);
      showToast('error', 'Error', 'Failed to process action.');
    }
  };

  const confirmDelete = async () => {
    if (!selectedRequest || !userId || !hasDeletePermission) return;

    setDeleteConfirmVisible(false);
    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();

      const updatedRequests = maintenanceRequests.filter(
        (request) => request.id !== selectedRequest.id
      );
      setMaintenanceRequests(updatedRequests);
      setFilteredRequests(updatedRequests);

      await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(updatedRequests));

      if (isOnline) {
        const response = await fetch(`http://168.231.113.219:5009/request-maintenance/${selectedRequest.id}`, {
          method: 'DELETE',
          headers: {
            'Content-Type': 'application/json',
          },
        });

        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        
        if (!result.success) {
          throw new Error(result.error || 'Failed to delete request');
        }

        showToast('success', 'Deleted', 'Maintenance request deleted successfully.');
      } else {
        showToast('info', 'Offline', 'Request deleted locally. Will sync when online.');
      }
    } catch (error) {
      console.error('Error deleting maintenance request:', error);
      let errorMessage = 'Failed to delete maintenance request.';
      if (error.message.includes('404')) {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.message.includes('NETWORK_ERROR') || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Deletion saved locally and will sync when online.';
      }
      showToast('error', 'Error', errorMessage);
      await fetchMaintenanceRequests();
    } finally {
      setSyncModalVisible(false);
      setSelectedRequest(null);
    }
  };

  const cancelDelete = () => {
    setDeleteConfirmVisible(false);
    setSelectedRequest(null);
  };

  // Memoized empty state message
  const emptyStateMessage = useMemo(() => {
    return hasOnlyMineView 
      ? 'No maintenance requests found for your account.' 
      : 'No maintenance requests found for your assigned countries/projects.';
  }, [hasOnlyMineView]);

  // Render each maintenance request
  const renderMaintenanceRequest = useCallback(({ item }) => {
    const status = item.status || 'Requested';
    const statusColor = statusColors[status] || statusColors['Requested'];
    
    // Check if this request has pending offline updates (approve/reject)
    const hasPendingUpdate = offlineUpdateIds.includes(item.id);

    return (
      <View style={styles.cardContainer}>
        {/* Top Row: Icon, WO number, Status */}
        <View style={styles.cardTopRow}>
          <MaterialIcons name="local-fire-department" size={22} color="#4ECB71" style={{ marginRight: 6 }} />
          <Text style={styles.woNumber}>{item.requestNumber || 'WO-NA'}</Text>
          
          {/* Offline Sync Pending Indicator - NEW REQUEST */}
          {item.offlineSyncPending && (
            <View style={styles.offlineBadge}>
              <MaterialIcons name="cloud-upload" size={12} color="#fff" />
              <Text style={styles.offlineBadgeText}>SYNC</Text>
            </View>
          )}
          
          {/* Offline Update Pending Indicator - APPROVE/REJECT */}
          {hasPendingUpdate && !item.offlineSyncPending && (
            <View style={styles.offlineUpdateBadge}>
              <MaterialIcons name="cloud-queue" size={12} color="#fff" />
              <Text style={styles.offlineUpdateBadgeText}>UPDATE</Text>
            </View>
          )}
          
          <View style={[styles.statusBadge, { backgroundColor: statusColor }]}> 
            <Text style={styles.statusText}>{status}</Text>
          </View>
        </View>
        {/* Table Rows */}
        <View style={styles.tableRow}>
          <Text style={styles.tableLabel}>Title</Text>
          <Text style={styles.tableValue}>{item.workOrderTitle || 'N/A'}</Text>
        </View>
        <View style={styles.tableRow}>
          <Text style={styles.tableLabel}>Equipments #</Text>
          <Text style={styles.tableValue}>{item.equipment?.equipmentName || 'N/A'}</Text>
        </View>
        <View style={styles.tableRow}>
          <Text style={styles.tableLabel}>Assign</Text>
          <Text style={styles.tableValue}>
            {item.assignTo?.map(a => a.fullName || a.name).join(', ') || 'Unassigned'}
          </Text>
        </View>
        <View style={[styles.tableRow, { alignItems: 'center' }]}> 
          <Text style={styles.tableLabel}>Create Date</Text>
          <Text style={styles.tableValue}>
            {item.createdAt
              ? new Date(item.createdAt).toLocaleString('en-GB', {
                  day: '2-digit',
                  month: 'short',
                  hour: '2-digit',
                  minute: '2-digit',
                  hour12: true,
                }).replace(',', '')
              : 'N/A'}
          </Text>
          <TouchableOpacity
            style={styles.menuButton}
            onPress={() => {
              setSelectedRequest(item);
              setDropdownVisible(true);
            }}
          >
            <MaterialIcons name="more-vert" size={22} color="#333" />
          </TouchableOpacity>
        </View>
      </View>
    );
  }, [filteredDropdownOptions, handleDropdownAction, offlineUpdateIds]);

  const scale = (size) => (width / 375) * size;

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={scale(28)} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Request Maintenance</Text>
          </View>
          <View style={styles.headerRight}>
            {!isOnline && (
              <View style={styles.offlineIndicator}>
                <MaterialIcons name="wifi-off" size={width * 0.05} color="#ffd205" />
              </View>
            )}
            <TouchableOpacity 
              style={[styles.iconButton, isSyncing && styles.syncingButton]} 
              onPress={handleCloudSync}
              disabled={isSyncing}
            >
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={[styles.refreshIcon, isSyncing && styles.syncingIcon]}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionsModalVisible(true)}
            >
              <MaterialIcons name="lock" size={scale(24)} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by title, type, vehicle..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={scale(20)} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={scale(24)} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      
      {/* Status Counter Widgets */}
      <View style={styles.statusCounterContainer}>
        <TouchableOpacity 
          style={styles.statusCounterCard}
          onPress={() => {
            showToast('info', 'Offline Storage', `${offlineRequestsCount} requests available offline`);
          }}
        >
          <MaterialIcons name="cloud-off" size={24} color="#FF9800" />
          <View style={styles.statusCounterContent}>
            <Text style={styles.statusCounterLabel}>Offline Stored</Text>
            <Text style={styles.statusCounterValue}>{offlineRequestsCount}</Text>
          </View>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.statusCounterCard}
          onPress={() => {
            showToast('info', 'Online Database', `${onlineRequestsCount} requests in MongoDB`);
          }}
        >
          <MaterialIcons name="cloud-done" size={24} color="#4CAF50" />
          <View style={styles.statusCounterContent}>
            <Text style={styles.statusCounterLabel}>Online in DB</Text>
            <Text style={styles.statusCounterValue}>{onlineRequestsCount}</Text>
          </View>
        </TouchableOpacity>
        
        <TouchableOpacity 
          style={styles.statusCounterCard}
          onPress={() => {
            showToast('info', 'Storage Size', `Total: ${localStorageSize}`);
          }}
        >
          <MaterialIcons name="storage" size={24} color="#2196F3" />
          <View style={styles.statusCounterContent}>
            <Text style={styles.statusCounterLabel}>Storage</Text>
            <Text style={styles.statusCounterValue}>{localStorageSize}</Text>
          </View>
        </TouchableOpacity>
      </View>

      {/* Main Content */}
      <FlatList
        data={filteredRequests.filter(item => item && item.id)}
        renderItem={renderMaintenanceRequest}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.mainContent}
        ListEmptyComponent={
          <Text style={styles.defaultText}>{emptyStateMessage}</Text>
        }
        removeClippedSubviews={true}
        maxToRenderPerBatch={10}
        windowSize={10}
        initialNumToRender={10}
      />

      {/* Bottom Button - Only show if user has create permission */}
      {hasCreatePermission && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('CreateRequestMaintenance')}
            >
              <Ionicons name="add" size={scale(24)} color="#fff" />
              <Text style={styles.buttonText}>New Request</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}

      {/* Sync Modal */}
      <SyncModal
        visible={syncModalVisible}
        message={deleteConfirmVisible ? 'Deleting Request...' : 'Syncing Data...'}
      />

      {/* Delete Confirmation Modal */}
      <Modal
        visible={deleteConfirmVisible}
        transparent
        animationType="fade"
        onRequestClose={cancelDelete}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.confirmModalContent}>
            <Text style={styles.confirmModalText}>Are you sure?</Text>
            <Text style={styles.confirmModalSubText}>
              Do you really want to delete this maintenance request?
            </Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={cancelDelete}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.deleteButton]}
                onPress={confirmDelete}
              >
                <Text style={styles.deleteButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Dropdown Action Menu Modal */}
      <Modal
        visible={dropdownVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setDropdownVisible(false)}
      >
        <TouchableOpacity
          style={styles.dropdownOverlay}
          activeOpacity={1}
          onPress={() => setDropdownVisible(false)}
        >
          <View style={styles.dropdownMenuContainer}>
            <Text style={styles.dropdownMenuTitle}>Select Action</Text>
            {filteredDropdownOptions.map((option) => (
              <TouchableOpacity
                key={option.value}
                style={styles.dropdownMenuItem}
                onPress={() => {
                  setDropdownVisible(false);
                  handleDropdownAction(option.value, selectedRequest);
                }}
              >
                {option.icon}
                <Text style={styles.dropdownMenuText}>{option.label}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.dropdownMenuCloseButton}
              onPress={() => setDropdownVisible(false)}
            >
              <Text style={styles.dropdownMenuCloseText}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </TouchableOpacity>
      </Modal>

      {/* Permissions Modal */}
      <Modal
        animationType="slide"
        transparent={true}
        visible={permissionsModalVisible}
        onRequestClose={() => setPermissionsModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{
            width: '85%',
            backgroundColor: '#fff',
            borderRadius: 18,
            padding: 28,
            alignItems: 'center',
            shadowColor: '#000',
            shadowOffset: { width: 0, height: 2 },
            shadowOpacity: 0.2,
            shadowRadius: 8,
            elevation: 8,
            maxHeight: '80%',
          }}>
            <Text style={{
              fontSize: 22,
              fontWeight: 'bold',
              color: '#025383',
              marginBottom: 18,
            }}>
              User Details
            </Text>
            <ScrollView style={{ width: '100%' }} contentContainerStyle={{ paddingBottom: 16 }} showsVerticalScrollIndicator={true}>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Name</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: '#025383', marginTop: 2 }}>{userData.name}</Text>
              </View>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Countries</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: userData.countries.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData.countries.length > 0 ? userData.countries.join(', ') : 'No countries assigned'}</Text>
              </View>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Projects</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: userData.projects.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData.projects.length > 0 ? userData.projects.join(', ') : 'No projects assigned'}</Text>
              </View>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Request Maintenance Permissions</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: userData.requestMaintenancePermissions.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData.requestMaintenancePermissions.length > 0 ? userData.requestMaintenancePermissions.join(', ') : 'No permissions assigned'}</Text>
              </View>
            </ScrollView>
            <TouchableOpacity
              style={{
                marginTop: 8,
                backgroundColor: '#025383',
                borderRadius: 8,
                paddingVertical: 10,
                paddingHorizontal: 32,
              }}
              onPress={() => setPermissionsModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={{
                color: '#fff',
                fontWeight: 'bold',
                fontSize: 16,
              }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  requestCard: {
    backgroundColor: '#f9f9f9',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 2,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  woNumber: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#3481BC',
    marginLeft: 6,
  },
  statusBadge: {
    borderRadius: 4,
    paddingHorizontal: 8,
    paddingVertical: 2,
    marginLeft: 10,
  },
  statusText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
  },
  menuButton: {
    padding: 4,
    hitSlop: { top: 10, bottom: 10, left: 10, right: 10 },
  },
  cardRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 4,
  },
  cardLabel: {
    color: '#333',
    fontWeight: '500',
    fontSize: width * 0.035,
    width: '30%',
  },
  cardValue: {
    color: '#222',
    fontSize: width * 0.035,
    fontWeight: '400',
    flex: 1,
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 14,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: width * 0.05,
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: 10,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  confirmModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    alignItems: 'center',
  },
  confirmModalText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  confirmModalSubText: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: width * 0.01,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  deleteButton: {
    backgroundColor: '#C2185B',
  },
  modalButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  deleteButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 20,
  },
  modalScroll: {
    maxHeight: Dimensions.get('window').height * 0.5,
    width: '100%',
  },
  modalSection: {
    marginBottom: 20,
  },
  modalSectionTitle: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 6,
    marginRight: 8,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
  },
  noItemsText: {
    fontSize: 14,
    color: '#888',
    marginTop: 8,
  },
  modalCloseButton: {
    marginTop: 20,
    paddingVertical: 10,
    paddingHorizontal: 20,
    backgroundColor: '#ff4444',
    borderRadius: 5,
  },
  modalCloseText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  modernModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.25)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modernModalContent: {
    width: '90%',
    maxWidth: 400,
    height: '60%', // Set fixed height to 60% of the screen
    backgroundColor: '#fff',
    borderRadius: 22,
    padding: 24,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.18,
    shadowRadius: 12,
    elevation: 8,
    alignItems: 'center',
    justifyContent: 'flex-start', // Ensure content starts at the top
  },
  modernModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#222',
    marginBottom: 18,
    textAlign: 'center',
  },
  section: {
    width: '100%',
    marginBottom: 24,
  },
  sectionTitle: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 17,
    marginBottom: 4,
  },
  sectionDivider: {
    height: 2,
    backgroundColor: '#e0e6ed',
    marginBottom: 12,
    borderRadius: 1,
  },
  chipRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    alignItems: 'center',
    marginBottom: 0,
  },
  chip: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f4f8fb',
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#3481BC',
    paddingHorizontal: 14,
    paddingVertical: 8,
    margin: 6,
  },
  chipText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
  },
  chipEmpty: {
    fontSize: 14,
    color: '#bbb',
    margin: 6,
    fontStyle: 'italic',
  },
  closeButton: {
    backgroundColor: '#ff4444',
    borderRadius: 24,
    paddingHorizontal: 32,
    paddingVertical: 12,
    marginTop: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 4,
  },
  closeButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    textAlign: 'center',
    letterSpacing: 0.5,
  },
  userInfoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  userInfoLabel: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  userInfoValue: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: '600',
  },
  summaryRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  summaryLabel: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  summaryValue: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: '600',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownItemText: {
    fontSize: 14,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  cardContainer: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginBottom: 16,
    overflow: 'hidden',
    paddingHorizontal: 0,
    paddingVertical: 0,
  },
  cardTopRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
    paddingHorizontal: 16,
    paddingTop: 12,
    paddingBottom: 8,
    backgroundColor: '#fff',
  },
  woNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#3481BC',
    marginLeft: 4,
    flex: 0,
  },
  statusBadge: {
    marginLeft: 'auto',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 2,
    backgroundColor: '#3481BC',
    alignSelf: 'center',
  },
  statusText: {
    color: '#fff',
    fontSize: 13,
    fontWeight: 'bold',
  },
  tableRow: {
    flexDirection: 'row',
    alignItems: 'center',
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
    paddingHorizontal: 16,
    paddingVertical: 10,
    backgroundColor: '#fff',
  },
  tableLabel: {
    width: 110,
    color: '#333',
    fontWeight: '500',
    fontSize: 15,
  },
  tableValue: {
    flex: 1,
    color: '#222',
    fontSize: 15,
    fontWeight: '400',
    textAlign: 'right',
  },
  menuButton: {
    marginLeft: 8,
    padding: 4,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: 30,
    height: width * 0.06,
  },
  syncingButton: {
    opacity: 0.7,
  },
  syncingIcon: {
    opacity: 0.8,
  },
  offlineIndicator: {
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    borderRadius: width * 0.025,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.01,
    marginRight: width * 0.02,
  },
  offlineBadge: {
    marginLeft: width * 0.025,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.008,
    borderRadius: width * 0.01,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ff9800',
  },
  offlineBadgeText: {
    fontSize: width * 0.025,
    color: '#fff',
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  offlineUpdateBadge: {
    marginLeft: width * 0.025,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.008,
    borderRadius: width * 0.01,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#2196F3',
  },
  offlineUpdateBadgeText: {
    fontSize: width * 0.025,
    color: '#fff',
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  statusCounterContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.03,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  statusCounterCard: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F5F5F5',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.025,
    marginHorizontal: width * 0.01,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  statusCounterContent: {
    marginLeft: width * 0.02,
    flex: 1,
  },
  statusCounterLabel: {
    fontSize: width * 0.028,
    color: '#666',
    marginBottom: 2,
  },
  statusCounterValue: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  dropdownMenuContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    width: '85%',
    maxWidth: 400,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 8,
    elevation: 8,
  },
  dropdownMenuTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    textAlign: 'center',
  },
  dropdownMenuItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 14,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  dropdownMenuText: {
    fontSize: width * 0.042,
    color: '#333',
    marginLeft: 12,
    fontWeight: '500',
  },
  dropdownMenuCloseButton: {
    marginTop: 12,
    paddingVertical: 12,
    paddingHorizontal: 20,
    backgroundColor: '#f0f0f0',
    borderRadius: 8,
    alignItems: 'center',
  },
  dropdownMenuCloseText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '600',
  },
});

export default RequestmaintenanceHome;