import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  FlatList,
  Modal,
  Animated,
  ActivityIndicator,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width, height } = Dimensions.get('window');

const SyncModal = ({ visible, message = 'Loading Reports' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const InspectionsOpened = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const isMounted = useRef(true);

  // Date states - Default to last 14 days
  const [fromDate, setFromDate] = useState(() => {
    const date = new Date();
    date.setDate(date.getDate() - 14);
    date.setHours(0, 0, 0, 0);
    return date;
  });
  const [toDate, setToDate] = useState(() => {
    const date = new Date();
    date.setHours(23, 59, 59, 999);
    return date;
  });
  const [showFromDatePicker, setShowFromDatePicker] = useState(false);
  const [showToDatePicker, setShowToDatePicker] = useState(false);

  // Data states
  const [allInspections, setAllInspections] = useState([]);
  const [jobCardsMap, setJobCardsMap] = useState({});
  const [closedInspections, setClosedInspections] = useState([]); // Job card status = 'complete'
  const [openedInspections, setOpenedInspections] = useState([]); // Job card status = anything else
  
  // UI states
  const [activeTab, setActiveTab] = useState('opened'); // 'opened' or 'closed'
  const [loading, setLoading] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  
  // User info
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    inspectionPermissions: [],
  });

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const formatDate = (date) => {
    if (!date) return 'N/A';
    const d = new Date(date);
    return d.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    });
  };

  const formatDateTime = (isoString) => {
    if (!isoString) return 'N/A';
    const date = new Date(isoString);
    return date.toLocaleString('en-US', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
    }).replace(',', '');
  };

  // Get status color
  const getJobCardStatusColor = (status) => {
    const statusLower = (status || '').toLowerCase();
    if (statusLower === 'complete') return '#2ecc40';
    if (statusLower === 'in_progress') return '#f39c12';
    if (statusLower === 'hold') return '#e74c3c';
    if (statusLower === 'assigned') return '#3498db';
    if (statusLower === 'cancelled') return '#95a5a6';
    return '#666';
  };

  // Get status display name
  const getJobCardStatusDisplay = (status) => {
    const statusLower = (status || '').toLowerCase();
    if (statusLower === 'complete') return 'Completed';
    if (statusLower === 'in_progress') return 'In Progress';
    if (statusLower === 'hold') return 'On Hold';
    if (statusLower === 'assigned') return 'Assigned';
    if (statusLower === 'cancelled') return 'Cancelled';
    return status || 'Unknown';
  };

  // Fetch user info
  const fetchUserInfo = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      const cachedUserData = await AsyncStorage.getItem('userData');
      if (cachedUserData) {
        const userData = JSON.parse(cachedUserData);
        const inspectionModule = userData.modules?.find((module) => module.module === 'inspectionReport');
        
        if (isMounted.current) {
          setUserInfo({
            countries: Array.isArray(userData.countries) ? userData.countries : [],
            projects: Array.isArray(userData.projects) ? userData.projects : [],
            inspectionPermissions: inspectionModule ? (Array.isArray(inspectionModule.permissions) ? inspectionModule.permissions : []) : [],
          });
        }
      }
    } catch (error) {
      console.error('Error fetching user info:', error);
    }
  };

  // Load job cards
  const loadJobCards = async () => {
    try {
      console.log('🔄 Loading job cards...');
      const response = await JobCardAPI.getAllJobCards();
      
      if (response.success && response.data) {
        const jobCardsMap = {};
        
        response.data.forEach(jobCard => {
          const inspectionId = jobCard.linkInspection?.id || 
                              jobCard.linkInspection?._id ||
                              jobCard.serviceScheduleId ||
                              jobCard.inspectionId;
                              
          if (inspectionId) {
            jobCardsMap[inspectionId] = jobCard;
          }
        });
        
        setJobCardsMap(jobCardsMap);
        console.log(`✅ Loaded ${response.data.length} job cards, ${Object.keys(jobCardsMap).length} linked`);
      }
    } catch (error) {
      console.error('Error loading job cards:', error);
    }
  };

  // Load inspections
  const loadInspections = async () => {
    if (!isMounted.current) return;
    
    try {
      setLoading(true);
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      // Fetch inspections based on permissions
      let response;
      if (userInfo.inspectionPermissions.includes('onlyMineView')) {
        response = await InspectionAPI.getInspections(userId, { page: 1, limit: 1000 });
      } else if (userInfo.inspectionPermissions.includes('view')) {
        try {
          response = await InspectionAPI.getAllReports({ page: 1, limit: 1000 });
          
          if (!response.success || !response.data || response.data.length === 0) {
            const knownUserResponse = await InspectionAPI.getInspections('test_proadmin_amit_001', { page: 1, limit: 1000 });
            if (knownUserResponse.success && knownUserResponse.data) {
              response = knownUserResponse;
            }
          }
        } catch (error) {
          response = await InspectionAPI.getInspections('test_proadmin_amit_001', { page: 1, limit: 1000 });
        }
      } else {
        setAllInspections([]);
        setLoading(false);
        return;
      }

      if (response.success && response.data) {
        const inspections = response.data.map(report => ({
          ...report,
          id: report._id,
          adminId: report.userId || report.createdBy,
          createdAt: report.createdAt || new Date().toISOString(),
        }));
        
        setAllInspections(inspections);
        console.log(`✅ Loaded ${inspections.length} inspections`);
      }
    } catch (error) {
      console.error('Error loading inspections:', error);
      showToast('error', 'Error', 'Failed to load inspections');
    } finally {
      if (isMounted.current) {
        setLoading(false);
      }
    }
  };

  // Filter inspections by date and job card status
  useEffect(() => {
    if (!allInspections.length || !Object.keys(jobCardsMap).length) return;

    // Filter inspections that have job cards
    const inspectionsWithJobCards = allInspections.filter(inspection => {
      const createdDate = new Date(inspection.createdAt);
      const isInDateRange = createdDate >= fromDate && createdDate <= toDate;
      
      // Must have a job card
      const hasJobCard = !!jobCardsMap[inspection.id];
      
      // Apply country/project filter if user has view permission (not onlyMineView)
      let matchesPermissions = true;
      if (userInfo.inspectionPermissions.includes('view') && !userInfo.inspectionPermissions.includes('onlyMineView')) {
        const matchesCountry = userInfo.countries.includes(inspection.step1?.country);
        const matchesProject = userInfo.projects.includes(inspection.step1?.project);
        matchesPermissions = matchesCountry && matchesProject;
      }
      
      return isInDateRange && hasJobCard && matchesPermissions;
    });

    // Separate by job card status
    const closed = inspectionsWithJobCards.filter(inspection => {
      const jobCard = jobCardsMap[inspection.id];
      return jobCard && jobCard.status && jobCard.status.toLowerCase() === 'complete';
    });

    const opened = inspectionsWithJobCards.filter(inspection => {
      const jobCard = jobCardsMap[inspection.id];
      return jobCard && (!jobCard.status || jobCard.status.toLowerCase() !== 'complete');
    });

    setClosedInspections(closed);
    setOpenedInspections(opened);

    console.log(`📊 Filtered results: ${closed.length} closed job cards, ${opened.length} opened job cards`);
  }, [allInspections, jobCardsMap, fromDate, toDate, userInfo]);

  // Initial load
  useEffect(() => {
    const loadData = async () => {
      await fetchUserInfo();
      await loadJobCards();
      await loadInspections();
    };

    if (isFocused) {
      loadData();
    }

    return () => {
      isMounted.current = false;
    };
  }, [isFocused]);

  // Reload when dates change
  useEffect(() => {
    if (userInfo.inspectionPermissions.length > 0) {
      loadInspections();
    }
  }, [fromDate, toDate]);

  // Handle view PDF
  const handleViewPDF = async (inspection) => {
    if (!inspection) {
      showToast('error', 'Error', 'No inspection selected.');
      return;
    }

    try {
      const reportId = inspection._id || inspection.id;
      const adminId = inspection.userId || inspection.createdBy || inspection.adminId;

      let response;
      if (userInfo.inspectionPermissions.includes('onlyMineView')) {
        response = await InspectionAPI.getInspection(adminId, reportId);
      } else {
        response = await InspectionAPI.getInspectionById(reportId);
      }

      if (response.success && response.data) {
        const pdfUrl = response.data.step3?.pdfDownloadUrl;
        
        if (!pdfUrl) {
          showToast('error', 'No PDF Available', 'This inspection does not have a PDF generated yet.');
          return;
        }

        navigation.navigate('PdfviewPage', {
          pdfUrl,
          onGoBack: () => navigation.goBack(),
        });
      } else {
        showToast('error', 'Database Error', 'Failed to fetch inspection from database.');
      }
    } catch (error) {
      console.error('Error viewing PDF:', error);
      showToast('error', 'Error', 'Could not fetch the PDF URL. Please try again.');
    }
  };

  // Handle view job card PDF
  const handleViewJobCardPDF = async (inspection) => {
    const jobCard = jobCardsMap[inspection.id];
    
    if (!jobCard?.pdfDownloadUrl) {
      showToast('error', 'Error', 'No Job Card PDF URL found.');
      return;
    }

    try {
      navigation.navigate('PdfviewPage', {
        pdfUrl: jobCard.pdfDownloadUrl,
        onGoBack: () => navigation.goBack(),
      });
    } catch (error) {
      console.error('Error viewing job card PDF:', error);
      showToast('error', 'Error', 'Could not view job card PDF.');
    }
  };

  // Handle view corrective action PDF
  const handleViewCorrectivePDF = async (inspection) => {
    try {
      const reportId = inspection._id || inspection.id;
      const response = await InspectionAPI.getCorrectiveAction(reportId);
      
      let pdfUrl = null;
      if (response.success && response.data?.pdfDownloadUrl) {
        pdfUrl = response.data.pdfDownloadUrl;
      } else {
        pdfUrl = inspection.correctiveActionData?.pdfDownloadUrl;
      }
      
      if (!pdfUrl) {
        showToast('error', 'Error', 'No Corrective Action PDF found for this inspection.');
        return;
      }

      // Add userId parameter if missing
      if (pdfUrl.includes('/corrective-actions/download-pdf/') && !pdfUrl.includes('userId=')) {
        const separator = pdfUrl.includes('?') ? '&' : '?';
        pdfUrl = `${pdfUrl}${separator}userId=${inspection.userId || inspection.adminId}`;
      }

      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.goBack(),
      });
    } catch (error) {
      console.error('Error viewing corrective PDF:', error);
      showToast('error', 'Error', 'Could not fetch corrective action PDF.');
    }
  };

  // Handle navigate to job card
  const handleNavigateToJobCard = async (inspection) => {
    const jobCard = jobCardsMap[inspection.id];
    
    if (!jobCard) {
      showToast('error', 'Error', 'No job card found for this inspection.');
      return;
    }

    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      navigation.navigate('WorkerDashboardhome', {
        workOrderId: jobCard._id || jobCard.id,
        adminUid: jobCard.adminUid || userId,
        onGoBack: () => navigation.goBack(),
      });
    } catch (error) {
      console.error('Error navigating to job card:', error);
      showToast('error', 'Error', 'Failed to load job card details.');
    }
  };

  // Render inspection item
  const renderInspectionItem = ({ item }) => {
    const jobCard = jobCardsMap[item.id];
    const hasCorrectiveAction = item.correctiveActionData?.pdfDownloadUrl || 
                                 item.correctiveActionData?.mechanicSignatureUri || 
                                 item.correctiveActionData?.operatorSignatureUri;

    if (!jobCard) return null; // Should not happen as we filter

    return (
      <View style={styles.inspectionCard}>
        <View style={styles.cardHeader}>
          <View style={styles.cardTitleRow}>
            <Text style={styles.cardTitle}>{item.step1?.requestNumber || 'N/A'}</Text>
            <View style={[styles.statusBadge, { 
              backgroundColor: item.step3?.equipmentStatus === 'Available' ? '#4CAF50' : '#FF9800' 
            }]}>
              <Text style={styles.statusText}>{item.step3?.equipmentStatus || 'N/A'}</Text>
            </View>
          </View>
          <Text style={styles.cardDate}>{formatDateTime(item.createdAt)}</Text>
        </View>

        <View style={styles.cardContent}>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Inspector:</Text>
            <Text style={styles.detailValue}>{item.step1?.inspector || 'N/A'}</Text>
          </View>

          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Location:</Text>
            <Text style={styles.detailValue}>{item.step1?.country || 'N/A'}</Text>
          </View>

          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Project:</Text>
            <Text style={styles.detailValue}>{item.step1?.project || 'N/A'}</Text>
          </View>

          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Equipment:</Text>
            <Text style={styles.detailValue}>{item.step1?.selectedEquipment?.equipmentName || 'N/A'}</Text>
          </View>

          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Faulty Items:</Text>
            <Text style={[styles.detailValue, { color: '#e74c3c', fontWeight: 'bold' }]}>
              {item.step3?.faultyItems || 0}
            </Text>
          </View>

          {/* Job Card Information - Always present */}
          <View style={styles.divider} />
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Job Card:</Text>
            <TouchableOpacity onPress={() => handleNavigateToJobCard(item)}>
              <Text style={[styles.detailValue, { color: '#0055A5', textDecorationLine: 'underline', fontWeight: 'bold' }]}>
                {jobCard.requestNumber || jobCard.title || 'Created'}
              </Text>
            </TouchableOpacity>
          </View>

          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Job Card Status:</Text>
            <View style={[styles.jobCardStatusBadge, { backgroundColor: getJobCardStatusColor(jobCard.status) }]}>
              <Text style={styles.jobCardStatusText}>
                {getJobCardStatusDisplay(jobCard.status)}
              </Text>
            </View>
          </View>

          {jobCard.priority && (
            <View style={styles.detailRow}>
              <Text style={styles.detailLabel}>Priority:</Text>
              <Text style={[styles.detailValue, { 
                color: jobCard.priority === 'Emergency' ? '#e74c3c' : 
                       jobCard.priority === 'High' ? '#f39c12' : 
                       jobCard.priority === 'Medium' ? '#3498db' : '#95a5a6',
                fontWeight: '600'
              }]}>
                {jobCard.priority}
              </Text>
            </View>
          )}

          {jobCard.dueDate && (
            <View style={styles.detailRow}>
              <Text style={styles.detailLabel}>Due Date:</Text>
              <Text style={styles.detailValue}>{formatDate(jobCard.dueDate)}</Text>
            </View>
          )}

          {/* Corrective Action */}
          {hasCorrectiveAction && (
            <>
              <View style={styles.divider} />
              <View style={styles.detailRow}>
                <Text style={styles.detailLabel}>Corrective Action:</Text>
                <Text style={[styles.detailValue, { color: '#2ecc40', fontWeight: 'bold' }]}>
                  Taken
                </Text>
              </View>
            </>
          )}
        </View>

        {/* Action Buttons */}
        <View style={styles.cardActions}>
          <TouchableOpacity 
            style={[styles.actionButton, { backgroundColor: '#0055A5' }]}
            onPress={() => handleViewPDF(item)}
          >
            <MaterialIcons name="picture-as-pdf" size={16} color="#fff" />
            <Text style={styles.actionButtonText}>Inspection PDF</Text>
          </TouchableOpacity>

          {jobCard?.pdfDownloadUrl && (
            <TouchableOpacity 
              style={[styles.actionButton, { backgroundColor: '#00B894' }]}
              onPress={() => handleViewJobCardPDF(item)}
            >
              <MaterialIcons name="build" size={16} color="#fff" />
              <Text style={styles.actionButtonText}>Job Card PDF</Text>
            </TouchableOpacity>
          )}

          {hasCorrectiveAction && (
            <TouchableOpacity 
              style={[styles.actionButton, { backgroundColor: '#ffd205' }]}
              onPress={() => handleViewCorrectivePDF(item)}
            >
              <MaterialIcons name="check-circle" size={16} color="#333" />
              <Text style={[styles.actionButtonText, { color: '#333' }]}>CA PDF</Text>
            </TouchableOpacity>
          )}

          <TouchableOpacity 
            style={[styles.actionButton, { backgroundColor: '#545d6b' }]}
            onPress={() => handleNavigateToJobCard(item)}
          >
            <MaterialIcons name="track-changes" size={16} color="#fff" />
            <Text style={styles.actionButtonText}>Track Job Card</Text>
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  const currentData = activeTab === 'closed' ? closedInspections : openedInspections;

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Job Cards Status</Text>
          <View style={{ width: 24 }} />
        </View>
      </LinearGradient>

      {/* Date Range Selector */}
      <View style={styles.dateRangeContainer}>
        <Text style={styles.dateRangeTitle}>Date Range Filter</Text>
        
        <View style={styles.datePickersRow}>
          <View style={styles.datePickerContainer}>
            <Text style={styles.dateLabel}>From Date</Text>
            <TouchableOpacity 
              style={styles.dateButton}
              onPress={() => setShowFromDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#0055A5" />
              <Text style={styles.dateButtonText}>{formatDate(fromDate)}</Text>
            </TouchableOpacity>
          </View>

          <View style={styles.datePickerContainer}>
            <Text style={styles.dateLabel}>To Date</Text>
            <TouchableOpacity 
              style={styles.dateButton}
              onPress={() => setShowToDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#0055A5" />
              <Text style={styles.dateButtonText}>{formatDate(toDate)}</Text>
            </TouchableOpacity>
          </View>
        </View>

        {/* Date Pickers */}
        {showFromDatePicker && (
          <DateTimePicker
            value={fromDate}
            mode="date"
            display={Platform.OS === 'ios' ? 'spinner' : 'default'}
            onChange={(event, selectedDate) => {
              setShowFromDatePicker(false);
              if (selectedDate) {
                selectedDate.setHours(0, 0, 0, 0);
                setFromDate(selectedDate);
              }
            }}
          />
        )}

        {showToDatePicker && (
          <DateTimePicker
            value={toDate}
            mode="date"
            display={Platform.OS === 'ios' ? 'spinner' : 'default'}
            onChange={(event, selectedDate) => {
              setShowToDatePicker(false);
              if (selectedDate) {
                selectedDate.setHours(23, 59, 59, 999);
                setToDate(selectedDate);
              }
            }}
          />
        )}
      </View>

      {/* Tabs */}
      <View style={styles.tabsContainer}>
        <TouchableOpacity
          style={[styles.tab, activeTab === 'opened' && styles.activeTab]}
          onPress={() => setActiveTab('opened')}
        >
          <View style={styles.tabContent}>
            <MaterialIcons 
              name="work" 
              size={20} 
              color={activeTab === 'opened' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'opened' && styles.activeTabText]}>
              Opened Job Cards ({openedInspections.length})
            </Text>
          </View>
        </TouchableOpacity>

        <TouchableOpacity
          style={[styles.tab, activeTab === 'closed' && styles.activeTab]}
          onPress={() => setActiveTab('closed')}
        >
          <View style={styles.tabContent}>
            <MaterialIcons 
              name="check-circle" 
              size={20} 
              color={activeTab === 'closed' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'closed' && styles.activeTabText]}>
              Closed Job Cards ({closedInspections.length})
            </Text>
          </View>
        </TouchableOpacity>
      </View>

      {/* Results Summary */}
      <View style={styles.summaryContainer}>
        <MaterialIcons 
          name={activeTab === 'opened' ? 'work-outline' : 'check-circle-outline'} 
          size={18} 
          color={activeTab === 'opened' ? '#f39c12' : '#2ecc40'} 
        />
        <Text style={styles.summaryText}>
          {activeTab === 'opened' 
            ? `Showing ${currentData.length} inspection${currentData.length !== 1 ? 's' : ''} with opened job cards`
            : `Showing ${currentData.length} inspection${currentData.length !== 1 ? 's' : ''} with completed job cards`
          }
        </Text>
      </View>

      {/* Legend */}
      <View style={styles.legendContainer}>
        <Text style={styles.legendTitle}>Job Card Status Legend:</Text>
        <View style={styles.legendRow}>
          <View style={styles.legendItem}>
            <View style={[styles.legendDot, { backgroundColor: '#2ecc40' }]} />
            <Text style={styles.legendText}>Completed</Text>
          </View>
          <View style={styles.legendItem}>
            <View style={[styles.legendDot, { backgroundColor: '#f39c12' }]} />
            <Text style={styles.legendText}>In Progress</Text>
          </View>
          <View style={styles.legendItem}>
            <View style={[styles.legendDot, { backgroundColor: '#3498db' }]} />
            <Text style={styles.legendText}>Assigned</Text>
          </View>
          <View style={styles.legendItem}>
            <View style={[styles.legendDot, { backgroundColor: '#e74c3c' }]} />
            <Text style={styles.legendText}>On Hold</Text>
          </View>
          <View style={styles.legendItem}>
            <View style={[styles.legendDot, { backgroundColor: '#95a5a6' }]} />
            <Text style={styles.legendText}>Cancelled</Text>
          </View>
        </View>
      </View>

      {/* Content */}
      <View style={styles.content}>
        {loading ? (
          <View style={styles.loadingContainer}>
            <ActivityIndicator size="large" color="#0055A5" />
            <Text style={styles.loadingText}>Loading inspections...</Text>
          </View>
        ) : currentData.length > 0 ? (
          <FlatList
            data={currentData}
            keyExtractor={(item) => item.id.toString()}
            renderItem={renderInspectionItem}
            contentContainerStyle={styles.listContent}
            showsVerticalScrollIndicator={false}
          />
        ) : (
          <View style={styles.emptyContainer}>
            <MaterialIcons 
              name={activeTab === 'opened' ? 'work-off' : 'check-circle'} 
              size={60} 
              color="#ccc" 
            />
            <Text style={styles.emptyText}>
              No inspections with {activeTab === 'opened' ? 'opened' : 'completed'} job cards found
            </Text>
            <Text style={styles.emptySubtext}>
              Try adjusting the date range or check back later
            </Text>
          </View>
        )}
      </View>

      <SyncModal visible={loading} message="Loading Inspections..." />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  header: {
    paddingTop: StatusBar.currentHeight || 40,
    paddingBottom: 16,
    paddingHorizontal: 16,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
  },
  dateRangeContainer: {
    backgroundColor: '#fff',
    padding: 16,
    marginHorizontal: 16,
    marginTop: 16,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  dateRangeTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 12,
  },
  datePickersRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  datePickerContainer: {
    flex: 1,
    marginHorizontal: 4,
  },
  dateLabel: {
    fontSize: 12,
    color: '#666',
    marginBottom: 6,
    fontWeight: '500',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingVertical: 10,
    paddingHorizontal: 12,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#0055A5',
  },
  dateButtonText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#0055A5',
    fontWeight: '500',
  },
  tabsContainer: {
    flexDirection: 'row',
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    borderRadius: 8,
    overflow: 'hidden',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  tab: {
    flex: 1,
    paddingVertical: 14,
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
  },
  activeTab: {
    backgroundColor: '#0055A5',
  },
  tabContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  tabText: {
    fontSize: 13,
    color: '#666',
    fontWeight: '600',
    marginLeft: 6,
  },
  activeTabText: {
    color: '#fff',
  },
  summaryContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderRadius: 6,
    borderLeftWidth: 4,
    borderLeftColor: '#ffd205',
  },
  summaryText: {
    fontSize: 13,
    color: '#666',
    fontWeight: '500',
    marginLeft: 8,
  },
  legendContainer: {
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    padding: 12,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  legendTitle: {
    fontSize: 12,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  legendRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 12,
  },
  legendItem: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  legendDot: {
    width: 10,
    height: 10,
    borderRadius: 5,
    marginRight: 4,
  },
  legendText: {
    fontSize: 11,
    color: '#666',
  },
  content: {
    flex: 1,
    marginTop: 12,
  },
  listContent: {
    paddingHorizontal: 16,
    paddingBottom: 20,
  },
  inspectionCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    marginBottom: 12,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    overflow: 'hidden',
  },
  cardHeader: {
    backgroundColor: '#E9E9E9',
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  cardTitleRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 6,
  },
  cardTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
  },
  statusBadge: {
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderRadius: 4,
  },
  statusText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  cardDate: {
    fontSize: 12,
    color: '#666',
  },
  cardContent: {
    padding: 16,
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 6,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    flex: 1,
  },
  detailValue: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
    flex: 1,
    textAlign: 'right',
  },
  jobCardStatusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 4,
  },
  jobCardStatusText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  divider: {
    height: 1,
    backgroundColor: '#ddd',
    marginVertical: 8,
  },
  cardActions: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    padding: 12,
    backgroundColor: '#f9f9f9',
    gap: 8,
  },
  actionButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 8,
    paddingHorizontal: 10,
    borderRadius: 6,
    marginRight: 6,
    marginBottom: 6,
  },
  actionButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    marginLeft: 4,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 12,
    fontSize: 14,
    color: '#666',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 40,
  },
  emptyText: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    marginTop: 16,
    fontWeight: '500',
  },
  emptySubtext: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    marginTop: 8,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default InspectionsOpened;
