import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  Animated,
  Modal,
  Image,
  Dimensions,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import JobCardAPI from '../../ConfigMongo/JobCardMongoConfig.js';

const { width } = Dimensions.get('window');

const DROPDOWN_OPTIONS = [
  {
    label: 'Edit',
    value: 'edit',
    icon: <MaterialIcons name="edit" size={22} color="#2196f3" />,
  },
  {
    label: 'View PDF',
    value: 'view-pdf',
    icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />,
  },
  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#e74c3c" />,
  },
];

const AllWotab = ({ refreshKey, searchQuery, onRefresh, userCountries = [], userProjects = [], jobcardPermissions = [], userUid }) => {
  const navigation = useNavigation();
  const [workOrders, setWorkOrders] = useState([]);
  const [filteredWorkOrders, setFilteredWorkOrders] = useState([]);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [isDataLoaded, setIsDataLoaded] = useState(false);
  const [shouldRenderReports, setShouldRenderReports] = useState(false);
  const [reportsReady, setReportsReady] = useState(false);

  // Animation for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Animation for dropdown
  const fadeAnim = useRef(new Animated.Value(0)).current;

  // Show toast notifications
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Add this useEffect to show loader and load data only once on mount
  useEffect(() => {
    setSyncModalVisible(true);
    loadWorkOrders();
  }, []);

  // Load work orders from MongoDB
  const loadWorkOrders = async () => {
    try {
      console.log('🔍 Loading job cards from MongoDB...');
      console.log('🔍 User UID:', userUid);
      console.log('🔍 Job card permissions:', jobcardPermissions);
      
      let response;
      
      // Check permissions to determine which API to use
      if (Array.isArray(jobcardPermissions) && jobcardPermissions.includes('onlyMineView') && userUid) {
        // User can only see their own job cards
        console.log('👤 Loading user-specific job cards for:', userUid);
        response = await JobCardAPI.getJobCards(userUid);
      } else {
        // User can see all job cards
        console.log('🌍 Loading all job cards');
        response = await JobCardAPI.getAllJobCards();
      }
      
      console.log('📡 API Response:', response);
      
      if (response.success && response.data) {
        let workOrderList = response.data;
        
        // Filter out empty/default work orders
        const filtered = workOrderList.filter(wo => {
          const title = (wo.title || '').trim().toLowerCase();
          const isUntitled = !title || title.includes('untitled work order');
          const equipmentName = (wo.equipment?.equipmentName || '').trim().toLowerCase();
          const isNoEquipment = !wo.equipment || equipmentName === 'no equipment' || !equipmentName;
          const type = (wo.workOrderType || '').trim().toLowerCase();
          const isTypeNA = !type || type === 'n/a';
          const isUnassigned = !wo.assignTo || wo.assignTo.length === 0 || (wo.assignTo.map(u => (u.name || '').toLowerCase()).join(', ') === 'unassigned');
          const isAllNA =
            isUntitled && isNoEquipment && isTypeNA && isUnassigned &&
            (!wo.startDate || wo.startDate === 'N/A') &&
            (!wo.dueDate || wo.dueDate === 'N/A') &&
            (!wo.createdAt || wo.createdAt === 'N/A');
          
          return !(isUntitled || isNoEquipment || isTypeNA || isAllNA);
        });
        
        console.log(`✅ Loaded ${filtered.length} job cards from MongoDB`);
        setWorkOrders(filtered);
        handleSearch(searchQuery, filtered);
        setIsDataLoaded(true);
        setShouldRenderReports(true);
        setReportsReady(true);
        setSyncModalVisible(false);
      } else {
        console.log('⚠️ No job cards found or API error');
        setWorkOrders([]);
        handleSearch(searchQuery, []);
        setIsDataLoaded(true);
        setShouldRenderReports(true);
        setReportsReady(true);
        setSyncModalVisible(false);
      }
    } catch (error) {
      console.error('Error loading job cards from MongoDB:', error);
      showToast('error', 'Error', 'Failed to load job cards.');
      setIsDataLoaded(true);
      setShouldRenderReports(false);
      setReportsReady(false);
      setSyncModalVisible(false);
    }
  };

  // Refresh when refreshKey changes (from parent)
  useEffect(() => {
    if (typeof refreshKey !== 'undefined') {
      setIsDataLoaded(false);
      setShouldRenderReports(false);
      setReportsReady(false);
      handleRefresh();
    }
  }, [refreshKey]);

  // Handle search
  const handleSearch = (query, workOrdersList = workOrders) => {
    if (query) {
      const filtered = workOrdersList.filter(
        (wo) =>
          (wo.title || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.equipment?.equipmentName || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.priority || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.assignTo?.some((user) => user.name.toLowerCase().includes(query.toLowerCase())) || false) ||
          (wo.workOrderType || '').toLowerCase().includes(query.toLowerCase())
      );
      setFilteredWorkOrders(filtered);
    } else {
      setFilteredWorkOrders(workOrdersList);
    }
  };

  useEffect(() => {
    handleSearch(searchQuery);
  }, [searchQuery, workOrders]);

  // Handle refresh
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    loadWorkOrders();

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      if (onRefresh) onRefresh();
    }, 2000);
  };

  // Delete work order from MongoDB
  const deleteWorkOrder = async (workOrderId, adminUid) => {
    try {
      console.log('🗑️ Deleting job card:', workOrderId, 'for admin:', adminUid);
      
      const response = await JobCardAPI.deleteJobCard(adminUid, workOrderId);
      
      if (response.success) {
        const updatedWorkOrders = workOrders.filter((wo) => wo._id !== workOrderId);
        setWorkOrders(updatedWorkOrders);
        handleSearch(searchQuery, updatedWorkOrders);
        setDropdownVisible(null);
        showToast('success', 'Success', 'Job Card deleted successfully!');
        console.log('✅ Job card deleted successfully');
      } else {
        throw new Error(response.error || 'Failed to delete job card');
      }
    } catch (error) {
      console.error('Error deleting job card from MongoDB:', error);
      showToast('error', 'Error', 'Failed to delete job card.');
    }
  };

  // Confirm before deleting
  const confirmDeleteWorkOrder = (workOrderId, adminUid) => {
    Alert.alert(
      'Delete Job Card',
      'Are you sure you want to delete this job card? This action cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Delete', style: 'destructive', onPress: () => deleteWorkOrder(workOrderId, adminUid) },
      ],
      { cancelable: true }
    );
  };

  // View PDF
  const handleViewPDF = async (workOrderId) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      console.log('🔍 Fetching job card PDF for:', workOrderId);
      const response = await JobCardAPI.getJobCard(workOrderId);

      if (!response.success || !response.data) {
        showToast('error', 'Error', 'Job Card not found in the database.');
        return;
      }

      const jobCardData = response.data;
      const pdfUrl = jobCardData.pdfDownloadUrl;
      
      if (!pdfUrl) {
        showToast('error', 'Error', 'PDF URL not available for this Job Card.');
        return;
      }

      console.log('📄 PDF URL found:', pdfUrl);
      setDropdownVisible(null);
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        userId: adminUid,
        onGoBack: () => {
          if (onRefresh) onRefresh();
          navigation.goBack();
        },
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      showToast('error', 'Error', 'Could not fetch the PDF URL. Please try again.');
    }
  };

  // Get priority color
  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'Low':
        return '#2ecc71';
      case 'Medium':
        return '#f1c40f';
      case 'High':
        return '#e67e22';
      case 'Emergency':
        return '#e74c3c';
      default:
        return '#666';
    }
  };

  // Format date
  const formatDate = (date) => {
    if (!date) return 'N/A';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Handle dropdown actions
  const handleDropdownAction = async (action, workOrder) => {
    switch (action) {
      case 'edit':
        setDropdownVisible(null);
        navigation.navigate('UpdateCreateWorkOrder', { workOrder });
        break;
      case 'view-pdf':
        handleViewPDF(workOrder._id || workOrder.id);
        break;
      case 'delete':
        confirmDeleteWorkOrder(workOrder._id || workOrder.id, workOrder.adminUid);
        break;
      default:
        break;
    }
  };

  // Toggle dropdown with animation
  const toggleDropdown = (id) => {
    if (dropdownVisible === id) {
      Animated.timing(fadeAnim, {
        toValue: 0,
        duration: 200,
        useNativeDriver: true,
      }).start(() => setDropdownVisible(null));
    } else {
      setDropdownVisible(id);
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 200,
        useNativeDriver: true,
      }).start();
    }
  };

  // Render work order item
  const renderWorkOrder = ({ item, index }) => {
    const title = (item.title || '').trim().toLowerCase();
    const equipmentName = (item.equipment?.equipmentName || '').trim().toLowerCase();
    const type = (item.workOrderType || '').trim().toLowerCase();
    if (!title || title.includes('untitled work order') || !equipmentName || equipmentName === 'no equipment' || !type || type === 'n/a') {
      return null;
    }
    const dropdownPosition = index > workOrders.length - 4 ? 'top' : 'auto';
    
    // Check permissions for dropdown options
    const canEdit = JobCardAPI.canEditJobCard(jobcardPermissions);
    const canDelete = JobCardAPI.canDeleteJobCard(jobcardPermissions);
    
    const filteredDropdownOptions = DROPDOWN_OPTIONS.filter(option => {
      if (option.value === 'edit' && !canEdit) return false;
      if (option.value === 'delete' && !canDelete) return false;
      return true;
    });
    
    return (
      <View style={styles.workOrderItem}>
        <View style={styles.workOrderHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.workOrderTitle}>{item.title || 'Untitled Work Order'}</Text>
          </View>
          <TouchableOpacity style={[styles.statusButton, { backgroundColor: getPriorityColor(item.priority) }]}> 
            <Text style={styles.statusButtonText}>{item.priority || 'N/A'}</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.workOrderDetails}>
          {item.status && (
            <View style={styles.statusBadgeRow}>
              <Text style={styles.statusBadgeText}>{item.status}</Text>
            </View>
          )}
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValue}>{item.equipment?.equipmentName || 'No Equipment'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Type</Text>
            <Text style={styles.detailValue}>{item.workOrderType || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Start Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.startDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Due Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.dueDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Assigned To</Text>
            <Text style={styles.detailValue}>{item.assignTo?.map((user) => user.name).join(', ') || 'Unassigned'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={[styles.detailLabel, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.detailValue}>{formatDate(item.createdAt)}</Text>
              <Dropdown
                style={styles.dropdown}
                data={filteredDropdownOptions}
                labelField="label"
                valueField="value"
                value={null}
                onChange={(option) => handleDropdownAction(option.value, item)}
                renderRightIcon={() => (
                  <MaterialIcons name="more-vert" size={22} color="#333" />
                )}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
          <View style={styles.detailItem}>
            <TouchableOpacity
              style={styles.trackButton}
              onPress={() => navigation.navigate('WorkerDashboardhome', {
                workOrderId: item.id,
                adminUid: item.adminUid,
                onGoBack: () => navigation.navigate('WorkOrdertab'),
              })}
            >
              <Text style={styles.trackButtonText}>Track</Text>
            </TouchableOpacity>
            {item.serviceScheduleId ? (
              <View style={styles.serviceScheduledBadge}>
                <Text style={styles.serviceScheduledText}>Service scheduled</Text>
              </View>
            ) : null}
          </View>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <View style={styles.mainContent}>
        {isDataLoaded && reportsReady && workOrders.length >= 0 ? (
          <FlatList
            data={filteredWorkOrders}
            renderItem={renderWorkOrder}
            keyExtractor={(item) => item.id}
            contentContainerStyle={styles.workOrderList}
            style={styles.flatList}
            ListEmptyComponent={
              <Text style={styles.emptyText}>
                {searchQuery ? 'No matching work orders found' : 'No work orders available'}
              </Text>
            }
            refreshing={isRefreshing}
            onRefresh={handleRefresh}
          />
        ) : (
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center' }}>
            <Text style={styles.emptyText}>Loading reports...</Text>
          </View>
        )}
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  flatList: {
    flex: 1,
  },
  workOrderList: {
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  workOrderItem: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  workOrderHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingTop: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  workOrderTitle: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    minWidth: width * 0.08,
    alignItems: 'center',
    justifyContent: 'center',
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  workOrderDetails: {
    paddingHorizontal: width * 0.05,
  },
  detailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
    backgroundColor: '#fff',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  emptyText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  trackButton: {
    backgroundColor: '#015185',
    paddingVertical: 6,
    paddingHorizontal: 16,
    borderRadius: 6,
    marginTop: 8,
    alignSelf: 'flex-start',
  },
  trackButtonText: {
    color: '#fff',
    fontWeight: 'bold',
  },
  serviceScheduledBadge: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    marginLeft: 12,
    alignSelf: 'center',
  },
  serviceScheduledText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
  statusBadgeRow: {
    backgroundColor: '#2ecc71',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    alignSelf: 'flex-start',
    marginBottom: 8,
    marginTop: 8,
  },
  statusBadgeText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
});

export default AllWotab;