import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  ActivityIndicator,
  Alert,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const UsersAssignToJC = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const [searchQuery, setSearchQuery] = useState('');
  const [teamMembers, setTeamMembers] = useState([]);
  const [adminProfile, setAdminProfile] = useState(null);
  const [userPermissions, setUserPermissions] = useState([]);
  const [hasOnlyMineView, setHasOnlyMineView] = useState(false);
  const [selectedUsers, setSelectedUsers] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  // Get previously selected users from route params
  useEffect(() => {
    if (route.params?.selectedUsers) {
      setSelectedUsers(route.params.selectedUsers);
    }
  }, [route.params]);

  const fetchFilteredTeamMembers = async () => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      console.log('Admin UID:', adminUid);
      if (!adminUid) {
        console.log('No admin UID found');
        return setTeamMembers([]);
      }

      // Fetch admin's data
      const adminRef = database().ref(`/Globalusers/admins/${adminUid}`);
      const adminSnap = await adminRef.once('value');
      if (!adminSnap.exists()) {
        console.log('Admin data not found');
        return setTeamMembers([]);
      }

      const userData = adminSnap.val();
      console.log('User data:', userData);

      // Set admin profile
      const adminProfileData = {
        ...userData,
        id: adminUid,
        name: userData.name || 'N/A',
        fullName: userData.name || 'N/A',
        employeeNumber: userData.employeeNumber || 'N/A',
        projects: userData.projects || [],
        countries: userData.countries || [],
        role: userData.role || 'Operator',
        isBlocked: userData.isBlocked || false,
        password: userData.password || 'N/A',
        profilePic: userData.profilePic || null,
      };
      setAdminProfile(adminProfileData);
      console.log('Admin profile set:', adminProfileData);

      // Get permissions
      const teamEmployModule = userData.modules?.find(m => m.module === 'teamEmploy');
      const permissions = teamEmployModule?.permissions || [];
      const userCountry = userData.countries || [];
      const userProject = userData.projects || [];
      
      console.log('Admin countries:', userCountry);
      console.log('Admin projects:', userProject);
      console.log('Admin permissions:', permissions);

      setUserPermissions(permissions);

      // Check for OnlyMineView permission
      const hasOnlyMineView = permissions.some(p => p.toLowerCase() === 'onlymineview');
      setHasOnlyMineView(hasOnlyMineView);
      
      console.log('Has OnlyMineView permission:', hasOnlyMineView);

      if (hasOnlyMineView) {
        // If user has OnlyMineView permission, show only their profile
        console.log('User has OnlyMineView permission, showing only admin profile');
        setTeamMembers([]); // No other team members
        setIsLoading(false);
        return;
      }

      // Fetch all team members
      console.log('Fetching all admins from Globalusers/admins...');
      const teamMembersRef = database().ref(`/Globalusers/admins`);
      const snapshot = await teamMembersRef.once('value');
      const data = snapshot.val();
      
      let membersArray = [];
      if (data) {
        membersArray = Object.keys(data).map((key) => ({
          ...data[key],
          id: key,
          name: data[key].name || 'N/A',
          fullName: data[key].name || 'N/A',
          employeeNumber: data[key].employeeNumber || 'N/A',
          projects: data[key].projects || [],
          countries: data[key].countries || [],
          role: data[key].role || 'Operator',
          isBlocked: data[key].isBlocked || false,
          password: data[key].password || 'N/A',
          profilePic: data[key].profilePic || null,
        }));
      }

      console.log('Total members found:', membersArray.length);

      // Filter team members by country and project
      let filteredMembers = membersArray.filter(member => {
        // Skip the current admin (will be added separately)
        if (member.id === adminUid) return false;
        
        // Apply country and project filtering
        const countryMatch = userData.countries.length === 0 || 
          (member.countries && member.countries.some(c => userData.countries.includes(c)));
        const projectMatch = userData.projects.length === 0 || 
          (member.projects && member.projects.some(p => userData.projects.includes(p)));
        
        return countryMatch && projectMatch;
      });
      
      console.log('After filtering:', filteredMembers.length, 'members');
      setTeamMembers(filteredMembers);
      setIsLoading(false);

    } catch (error) {
      console.error('Error fetching team members:', error);
      setTeamMembers([]);
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchFilteredTeamMembers();
  }, []);

  const toggleUserSelection = (user) => {
    setSelectedUsers(prev => {
      const isSelected = prev.some(selected => selected.id === user.id);
      if (isSelected) {
        const newSelection = prev.filter(selected => selected.id !== user.id);
        console.log(`Removed user: ${user.fullName || user.name} from selection`);
        return newSelection;
        } else {
        const newSelection = [...prev, user];
        console.log(`Added user: ${user.fullName || user.name} to selection`);
        return newSelection;
      }
    });
  };

  const handleDone = () => {
    // Navigate back to CreateWorkOrder with selected users
    console.log('Sending selected users back to CreateWorkOrder:', selectedUsers);
    console.log('Selected users details:', selectedUsers.map(user => ({
      id: user.id,
      name: user.name,
      fullName: user.fullName,
      employeeNumber: user.employeeNumber,
      role: user.role
    })));
    
    // Show confirmation dialog
    Alert.alert(
      'Confirm Selection',
      `You have selected ${selectedUsers.length} user(s). Do you want to proceed?`,
      [
        {
          text: 'Cancel',
          style: 'cancel',
        },
        {
          text: 'Proceed',
          onPress: () => {
            // Pass data back through route params
            if (route.params?.onUserSelection) {
              // Small delay to ensure navigation completes
              setTimeout(() => {
                route.params.onUserSelection(selectedUsers);
              }, 200);
            }
            
            navigation.goBack();
          },
        },
      ]
    );
  };

  const handleSelectAll = () => {
    const combinedMembers = hasOnlyMineView
      ? [adminProfile ? { ...adminProfile, isSelf: true } : null].filter(Boolean)
      : [
          adminProfile ? { ...adminProfile, isSelf: true } : null,
          ...teamMembers.map(m => ({ ...m, isSelf: false })),
        ].filter(Boolean);

    const filteredMembers = combinedMembers.filter(member =>
      member.fullName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      member.employeeNumber?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      member.role?.toLowerCase().includes(searchQuery.toLowerCase())
    );

    setSelectedUsers(filteredMembers);
  };

  const handleClearAll = () => {
    setSelectedUsers([]);
  };

  // Combine admin profile and team members for FlatList
  const combinedMembers = hasOnlyMineView
      ? [adminProfile ? { ...adminProfile, isSelf: true } : null].filter(Boolean)
      : [
          adminProfile ? { ...adminProfile, isSelf: true } : null,
        ...teamMembers.map(m => ({ ...m, isSelf: false })),
        ].filter(Boolean);

  // Apply search filter
  const filteredMembers = combinedMembers.filter(member =>
    member.fullName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
    member.employeeNumber?.toLowerCase().includes(searchQuery.toLowerCase()) ||
    member.role?.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const renderUserItem = ({ item }) => {
    const isSelf = item.isSelf;
    const isSelected = selectedUsers.some(selected => selected.id === item.id);
    
    return (
      <View style={[
        styles.userItem,
        isSelf ? { borderWidth: 1, borderColor: '#4CAF50', marginBottom: 8 } : {},
        isSelected && styles.userItemSelected
      ]}>
        <View style={styles.userItemLeft}>
          {item.profilePic && (
            <Image
              source={{ uri: item.profilePic.url }}
              style={styles.profileImage}
            />
          )}
          <View style={styles.userInfo}>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Employee Number: </Text>
            {item.employeeNumber || 'N/A'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Name: </Text>
            {item.fullName || 'N/A'}
            {isSelf && <Text style={{ color: '#4CAF50', fontWeight: 'bold' }}> (My Profile)</Text>}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Role: </Text>
            {item.role || 'Operator'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Projects: </Text>
            {item.projects?.length > 0 ? item.projects.join(', ') : 'None'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Country: </Text>
            {item.countries?.length > 0 ? item.countries.join(', ') : 'None'}
          </Text>
            <Text style={styles.userName}>
            <Text style={styles.labelText}>Status: </Text>
            {item.isBlocked ? 'Blocked' : 'Active'}
          </Text>
        </View>
        </View>
        <View style={styles.userItemRight}>
          <View style={styles.roleBadge}>
            <Text style={styles.roleText}>{item.role || 'Operator'}</Text>
          </View>
          <TouchableOpacity 
          style={[
              styles.selectButton,
              isSelected ? styles.selectButtonSelected : styles.selectButtonUnselected
            ]}
            onPress={() => toggleUserSelection(item)}
          >
            <Text style={[
              styles.selectButtonText,
              isSelected ? styles.selectButtonTextSelected : styles.selectButtonTextUnselected
            ]}>
              {isSelected ? 'Selected' : 'Select'}
                  </Text>
                </TouchableOpacity>
          </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Select Users to Assign</Text>
          </View>
          <View style={styles.headerRight}>
            <Text style={styles.selectedCount}>
              {selectedUsers.length} selected
            </Text>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by name, employee number, or role..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
      </LinearGradient>

      <View style={styles.actionButtons}>
        <TouchableOpacity style={styles.actionButton} onPress={handleSelectAll}>
          <MaterialIcons name="select-all" size={width * 0.05} color="#3481BC" />
          <Text style={styles.actionButtonText}>Select All</Text>
            </TouchableOpacity>
        <TouchableOpacity style={styles.actionButton} onPress={handleClearAll}>
          <MaterialIcons name="clear-all" size={width * 0.05} color="#f44336" />
          <Text style={[styles.actionButtonText, { color: '#f44336' }]}>Clear All</Text>
        </TouchableOpacity>
      </View>

      {isLoading ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#3481BC" />
          <Text style={styles.loadingText}>Loading users...</Text>
            </View>
      ) : (
        <FlatList
          data={filteredMembers}
          renderItem={renderUserItem}
          keyExtractor={(item) => item.id + (item.isSelf ? '_self' : '')}
          style={styles.userList}
          ListEmptyComponent={
            <View style={styles.emptyContainer}>
              <MaterialIcons name="people" size={48} color="#ccc" />
              <Text style={styles.emptyText}>No users found</Text>
              <Text style={styles.emptySubText}>
                {hasOnlyMineView ? 
                  'Only your profile is visible due to OnlyMineView permission' : 
                  'No users found for your country/project permissions'
                }
              </Text>
            </View>
          }
        />
      )}

      <View style={styles.bottomContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.doneButton}>
            <TouchableOpacity
            style={styles.doneButtonTouchable}
            onPress={handleDone}
            >
            <MaterialIcons name="check" size={width * 0.06} color="#fff" />
            <Text style={styles.doneButtonText}>
              Done ({selectedUsers.length} selected)
            </Text>
            </TouchableOpacity>
        </LinearGradient>
          </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  headerRight: {
    alignItems: 'center',
  },
  selectedCount: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  searchContainer: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  searchWrapper: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  searchIcon: {
    marginLeft: width * 0.012,
  },
  actionButtons: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.02,
    backgroundColor: '#f5f5f5',
  },
  actionButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginRight: width * 0.02,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  actionButtonText: {
    fontSize: width * 0.035,
    color: '#3481BC',
    marginLeft: width * 0.01,
  },
  userList: {
    flex: 1,
  },
  userItem: {
    flexDirection: 'row',
    padding: width * 0.04,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#fff',
    marginHorizontal: width * 0.02,
    marginVertical: width * 0.01,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  userItemSelected: {
    backgroundColor: '#e8f5e8',
    borderColor: '#4CAF50',
  },
  userItemLeft: {
    flex: 2,
    flexDirection: 'row',
    paddingRight: width * 0.02,
  },
  userItemRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
  },
  profileImage: {
    width: width * 0.1,
    height: width * 0.1,
    borderRadius: width * 0.05,
    marginRight: width * 0.02,
  },
  userInfo: {
    flex: 1,
  },
  userName: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.015,
    lineHeight: width * 0.05,
  },
  labelText: {
    fontWeight: 'bold',
    color: '#333',
  },
  roleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: 12,
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  roleText: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#fff',
  },
  selectButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    borderWidth: 1,
    alignItems: 'center',
    minWidth: width * 0.2,
  },
  selectButtonUnselected: {
    backgroundColor: '#fff',
    borderColor: '#3481BC',
  },
  selectButtonSelected: {
    backgroundColor: '#3481BC',
    borderColor: '#3481BC',
  },
  selectButtonText: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  selectButtonTextUnselected: {
    color: '#3481BC',
  },
  selectButtonTextSelected: {
    color: '#fff',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: width * 0.02,
    fontSize: width * 0.04,
    color: '#666',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: width * 0.1,
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    marginTop: width * 0.02,
  },
  emptySubText: {
    fontSize: width * 0.035,
    color: '#999',
    textAlign: 'center',
    marginTop: width * 0.01,
    paddingHorizontal: width * 0.04,
  },
  bottomContainer: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.02,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#E0E0E0',
  },
  doneButton: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  doneButtonTouchable: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  doneButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
});

export default UsersAssignToJC;