/**
 * OfflineWorkOrderHelper.js
 * Helper module for managing offline data for Work Orders (Job Cards)
 */

import AsyncStorage from '@react-native-async-storage/async-storage';

class OfflineWorkOrderHelper {
  /**
   * Save work order locally for offline sync
   * @param {Object} workOrder - Complete work order data
   * @returns {Promise<boolean>} Success status
   */
  static async saveOfflineWorkOrder(workOrder) {
    try {
      // Get existing offline work orders
      const offlineWorkOrdersRaw = await AsyncStorage.getItem('offlineWorkOrders');
      const offlineWorkOrders = offlineWorkOrdersRaw ? JSON.parse(offlineWorkOrdersRaw) : [];
      
      // Add new work order with offline flag and ensure PDF path is stored
      const offlineWorkOrder = {
        ...workOrder,
        offlineCreated: true,
        offlineSyncPending: true,
        offlineCreatedAt: new Date().toISOString(),
        // Ensure PDF local path is preserved for sync
        pdfLocalPath: workOrder.pdfLocalPath || workOrder.pdfDownloadUrl || '',
      };
      
      offlineWorkOrders.push(offlineWorkOrder);
      
      // Save to AsyncStorage
      await AsyncStorage.setItem('offlineWorkOrders', JSON.stringify(offlineWorkOrders));
      
      console.log('✅ Saved offline work order:', workOrder._id);
      console.log('📄 PDF local path:', offlineWorkOrder.pdfLocalPath || 'None');
      console.log('📊 Total offline work orders:', offlineWorkOrders.length);
      
      return true;
    } catch (error) {
      console.error('❌ Error saving offline work order:', error);
      return false;
    }
  }

  /**
   * Get all offline work orders waiting to be synced
   * @returns {Promise<Array>} Array of offline work orders
   */
  static async getOfflineWorkOrders() {
    try {
      const offlineWorkOrdersRaw = await AsyncStorage.getItem('offlineWorkOrders');
      const offlineWorkOrders = offlineWorkOrdersRaw ? JSON.parse(offlineWorkOrdersRaw) : [];
      return offlineWorkOrders;
    } catch (error) {
      console.error('❌ Error getting offline work orders:', error.message);
      return [];
    }
  }

  /**
   * Get all work orders (both online cached and offline)
   * @returns {Promise<Array>} Array of all work orders
   */
  static async getAllWorkOrders() {
    try {
      // Get cached online work orders
      const cachedWorkOrdersRaw = await AsyncStorage.getItem('globalWorkOrders');
      const cachedWorkOrders = cachedWorkOrdersRaw ? JSON.parse(cachedWorkOrdersRaw) : [];
      
      // Get offline work orders
      const offlineWorkOrders = await this.getOfflineWorkOrders();
      
      // Combine both arrays
      const allWorkOrders = [...cachedWorkOrders, ...offlineWorkOrders];
      
      return allWorkOrders;
    } catch (error) {
      console.error('❌ Error getting all work orders:', error.message);
      return [];
    }
  }

  /**
   * Sync offline work orders to MongoDB
   * @param {Function} uploadCallback - Function to upload work order to MongoDB
   * @returns {Promise<Object>} Sync results
   */
  static async syncOfflineWorkOrders(uploadCallback) {
    try {
      const offlineWorkOrders = await this.getOfflineWorkOrders();
      
      if (offlineWorkOrders.length === 0) {
        console.log('✅ No offline work orders to sync');
        return { success: true, synced: 0, failed: 0 };
      }
      
      console.log('🔄 Starting sync of', offlineWorkOrders.length, 'offline work orders...');
      
      let synced = 0;
      let failed = 0;
      const failedWorkOrders = [];
      
      for (const workOrder of offlineWorkOrders) {
        try {
          // Call the upload callback function
          const result = await uploadCallback(workOrder);
          
          if (result && result.success) {
            synced++;
            console.log('✅ Synced work order:', workOrder._id);
          } else {
            failed++;
            failedWorkOrders.push(workOrder);
            console.error('❌ Failed to sync work order:', workOrder._id);
          }
        } catch (error) {
          failed++;
          failedWorkOrders.push(workOrder);
          console.error('❌ Error syncing work order:', workOrder._id, error);
        }
      }
      
      // Update offline work orders (keep only failed ones)
      await AsyncStorage.setItem('offlineWorkOrders', JSON.stringify(failedWorkOrders));
      
      console.log('✅ Sync complete - Synced:', synced, 'Failed:', failed);
      
      return {
        success: true,
        synced,
        failed,
        total: offlineWorkOrders.length,
      };
    } catch (error) {
      console.error('❌ Error syncing offline work orders:', error);
      return {
        success: false,
        synced: 0,
        failed: 0,
        error: error.message,
      };
    }
  }

  /**
   * Clear all offline data (use with caution)
   */
  static async clearOfflineData() {
    try {
      await AsyncStorage.removeItem('offlineWorkOrders');
      await AsyncStorage.removeItem('offlineWorkOrderUpdates');
      console.log('✅ Cleared offline work orders and updates');
      return true;
    } catch (error) {
      console.error('❌ Error clearing offline data:', error);
      return false;
    }
  }

  // Save an offline update (status change or any PUT)
  static async saveOfflineUpdate(update) {
    try {
      const raw = await AsyncStorage.getItem('offlineWorkOrderUpdates');
      const updates = raw ? JSON.parse(raw) : [];
      const filtered = updates.filter((u) => u.id !== update.id);
      filtered.push({ ...update, queuedAt: new Date().toISOString() });
      await AsyncStorage.setItem('offlineWorkOrderUpdates', JSON.stringify(filtered));
      console.log('✅ Queued offline work order update:', update.id, update.action);
      return true;
    } catch (error) {
      console.error('❌ Error saving offline work order update:', error);
      return false;
    }
  }

  // Get offline updates queue
  static async getOfflineUpdates() {
    try {
      const raw = await AsyncStorage.getItem('offlineWorkOrderUpdates');
      return raw ? JSON.parse(raw) : [];
    } catch (error) {
      console.error('❌ Error getting offline work order updates:', error);
      return [];
    }
  }

  // Sync offline updates with provided uploader (PUT)
  static async syncOfflineUpdates(uploadCallback) {
    try {
      const updates = await this.getOfflineUpdates();
      if (!updates.length) {
        console.log('✅ No offline work order updates to sync');
        return { success: true, synced: 0, failed: 0 };
      }

      console.log('🔄 Starting sync of', updates.length, 'offline work order updates...');
      let synced = 0;
      let failed = 0;
      const failedUpdates = [];

      for (const upd of updates) {
        try {
          const result = await uploadCallback(upd);
          if (result && result.success) {
            synced++;
            console.log('✅ Synced work order update:', upd.id, upd.action);
          } else {
            failed++;
            failedUpdates.push(upd);
            console.error('❌ Failed to sync work order update:', upd.id, upd.action);
          }
        } catch (err) {
          failed++;
          failedUpdates.push(upd);
          console.error('❌ Error syncing work order update:', upd.id, err.message);
        }
      }

      await AsyncStorage.setItem('offlineWorkOrderUpdates', JSON.stringify(failedUpdates));
      console.log('✅ Updates sync complete - Synced:', synced, 'Failed:', failed);
      return { success: true, synced, failed, total: updates.length };
    } catch (error) {
      console.error('❌ Error syncing offline work order updates:', error);
      return { success: false, synced: 0, failed: 0, error: error.message };
    }
  }

  /**
   * Save PDF path for offline viewing
   * @param {string} workOrderId - Work order ID
   * @param {string} pdfPath - Local PDF file path
   */
  static async saveOfflinePDF(workOrderId, pdfPath) {
    try {
      const pdfKey = `workorder_pdf_local_${workOrderId}`;
      await AsyncStorage.setItem(pdfKey, pdfPath);
      console.log('✅ Saved offline PDF path:', pdfKey);
      return true;
    } catch (error) {
      console.error('❌ Error saving offline PDF:', error);
      return false;
    }
  }

  /**
   * Get offline PDF path
   * @param {string} workOrderId - Work order ID
   * @returns {Promise<string|null>} PDF path or null
   */
  static async getOfflinePDF(workOrderId) {
    try {
      const pdfKey = `workorder_pdf_local_${workOrderId}`;
      const pdfPath = await AsyncStorage.getItem(pdfKey);
      return pdfPath;
    } catch (error) {
      console.error('❌ Error getting offline PDF:', error);
      return null;
    }
  }
}

export default OfflineWorkOrderHelper;

