<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use App\Models\User;

class AuthController extends Controller
{
    /**
     * Show the login form
     */
    public function showLoginForm()
    {
        // If user is already logged in, redirect to dashboard
        if (Auth::check()) {
            return redirect()->route('admin.dashboard');
        }
        
        return view('auth.sign-in');
    }

    /**
     * Handle login request - MongoDB Authentication
     */
    public function login(Request $request)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|min:6'
        ], [
            'email.required' => 'Email address is required.',
            'email.email' => 'Please enter a valid email address.',
            'password.required' => 'Password is required.',
            'password.min' => 'Password must be at least 6 characters.'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput($request->except('password'));
        }

        $credentials = $request->only('email', 'password');
        $remember = $request->has('remember');

        try {
            // Authenticate against MongoDB API
            $mongoLoginUrl = env('MONGODB_WEBPROADMIN_LOGIN_URL', 'https://api.titandrillingzm.com:6050/api/webproadmin/login');
            
            $response = Http::timeout(10)->post($mongoLoginUrl, [
                'email' => $credentials['email'],
                'password' => $credentials['password']
            ]);

            if ($response->successful() && $response->json('success')) {
                $mongoUser = $response->json('user');
                
                // Find or create user in local database for session management
                $user = User::where('email', $mongoUser['email'])->first();
                
                if (!$user) {
                    // Create local user for session
                    $user = User::create([
                        'name' => $mongoUser['name'],
                        'email' => $mongoUser['email'],
                        'password' => Hash::make($credentials['password']), // Store hashed password
                        'role' => $mongoUser['role'] ?? 'WebProAdmin',
                        'email_verified_at' => now()
                    ]);
                } else {
                    // Update user info from MongoDB
                    $user->update([
                        'name' => $mongoUser['name'],
                        'role' => $mongoUser['role'] ?? 'WebProAdmin'
                    ]);
                }
                
                // Log the user in
                Auth::login($user, $remember);
                $request->session()->regenerate();
                
                // Store MongoDB user data in session
                $request->session()->put('mongo_user_data', $mongoUser);
                $request->session()->put('mongo_user_uid', $mongoUser['uid']);
                
                // Log successful login
                \Log::info('User logged in successfully via MongoDB', [
                    'email' => $request->email,
                    'uid' => $mongoUser['uid'],
                    'role' => $mongoUser['role'],
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]);

                return redirect()->intended(route('admin.dashboard'))
                    ->with('success', 'Welcome back! You have been logged in successfully.');
            }

            // Log failed login attempt
            \Log::warning('Failed MongoDB login attempt', [
                'email' => $request->email,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'mongo_response' => $response->json()
            ]);

            return redirect()->back()
                ->withErrors(['email' => 'Invalid email or password. Please try again.'])
                ->withInput($request->except('password'));

        } catch (\Exception $e) {
            // Log exception
            \Log::error('MongoDB authentication error', [
                'email' => $request->email,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withErrors(['email' => 'Authentication service is currently unavailable. Please try again later.'])
                ->withInput($request->except('password'));
        }
    }

    /**
     * Handle logout request - MongoDB Logout
     */
    public function logout(Request $request)
    {
        $mongoUserUid = $request->session()->get('mongo_user_uid');
        
        // Log logout
        \Log::info('User logged out', [
            'user_id' => Auth::id(),
            'email' => Auth::user()->email ?? 'Unknown',
            'mongo_uid' => $mongoUserUid,
            'ip' => $request->ip()
        ]);

        // Notify MongoDB API about logout
        if ($mongoUserUid) {
            try {
                $mongoLogoutUrl = env('MONGODB_WEBPROADMIN_LOGOUT_URL', 'https://api.titandrillingzm.com:6050/api/webproadmin/logout');
                
                Http::timeout(5)->post($mongoLogoutUrl, [
                    'uid' => $mongoUserUid
                ]);
            } catch (\Exception $e) {
                \Log::warning('MongoDB logout notification failed', [
                    'uid' => $mongoUserUid,
                    'error' => $e->getMessage()
                ]);
            }
        }

        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login')
            ->with('success', 'You have been logged out successfully.');
    }

    /**
     * Create a default admin user if it doesn't exist
     */
    public function createDefaultAdmin()
    {
        $adminExists = User::where('email', 'admin@titan.com')->exists();
        
        if (!$adminExists) {
            User::create([
                'name' => 'Admin User',
                'email' => 'admin@titan.com',
                'password' => Hash::make('admin123'),
                'role' => 'admin',
                'email_verified_at' => now()
            ]);
            
            \Log::info('Default admin user created', [
                'email' => 'admin@titan.com',
                'password' => 'admin123'
            ]);
            
            return true;
        }
        
        return false;
    }

    /**
     * Handle password change request
     */
    public function changePassword(Request $request)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'new_password' => 'required|min:8'
        ], [
            'new_password.required' => 'New password is required.',
            'new_password.min' => 'New password must be at least 8 characters.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ], 422);
        }

        $user = Auth::user();

        // Update password
        $user->password = Hash::make($request->new_password);
        $user->save();

        // Log password change
        \Log::info('User changed password', [
            'user_id' => $user->id,
            'email' => $user->email,
            'ip' => $request->ip()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully!'
        ]);
    }
}
