<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    /**
     * Show users page
     */
    public function index()
    {
        return view('auth.users');
    }
    
    /**
     * Get all users from MongoDB with Pagination
     */
    public function getAllUsers(Request $request)
    {
        try {
            $page = $request->get('page', 1);
            $limit = $request->get('limit', 20);
            
            // Use the paginated endpoint from MongoDB
            $mongoUrl = env('MONGODB_USERS_API_BASE_URL', 'https://api.titandrillingzm.com:6003') . '/users/paginated';
            
            Log::info('Fetching users from MongoDB with pagination', [
                'url' => $mongoUrl,
                'page' => $page,
                'limit' => $limit
            ]);
            
            $response = Http::timeout(10)->get($mongoUrl, [
                'page' => $page,
                'limit' => $limit
            ]);
            
            if ($response->successful()) {
                $data = $response->json();
                
                Log::info('Users fetched successfully', [
                    'count' => count($data['data'] ?? []),
                    'page' => $data['pagination']['currentPage'] ?? $page,
                    'totalPages' => $data['pagination']['totalPages'] ?? 0
                ]);
                
                return response()->json([
                    'success' => true,
                    'data' => $data['data'] ?? [],
                    'pagination' => [
                        'currentPage' => $data['pagination']['currentPage'] ?? $page,
                        'totalPages' => $data['pagination']['totalPages'] ?? 0,
                        'totalUsers' => $data['pagination']['totalUsers'] ?? 0,
                        'limit' => $data['pagination']['limit'] ?? $limit,
                        'hasNext' => $data['pagination']['hasNext'] ?? false,
                        'hasPrev' => $data['pagination']['hasPrev'] ?? false
                    ],
                    'timestamp' => now()->toISOString()
                ]);
            }
            
            Log::error('Failed to fetch users from MongoDB', ['status' => $response->status()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch users from MongoDB'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in getAllUsers', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Save new user to MongoDB
     */
    public function saveUser(Request $request)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'password' => 'required|string|min:6',
                'role' => 'required|string',
                'employeeNumber' => 'nullable|string|max:255',
                'department' => 'nullable|string|max:255',
                'nrcIdNumber' => 'nullable|string|max:255',
                'mobile' => 'nullable|string|max:20',
                'countries' => 'required|array|min:1',
                'projects' => 'required|array|min:1',
                'modules' => 'required|array|min:1',
                'engagementDate' => 'nullable|date',
                'titanInductionDate' => 'nullable|date',
                'clientInduction' => 'nullable|date',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }
            
            // Generate unique user ID
            $userId = 'user_' . time() . '_' . substr(md5(uniqid()), 0, 8);
            
            // Prepare user data for MongoDB
            $userData = [
                'name' => $request->name,
                'email' => $request->email,
                'password' => $request->password,
                'role' => $request->role,
                'employeeNumber' => $request->employeeNumber,
                'department' => $request->department,
                'nrcIdNumber' => $request->nrcIdNumber,
                'mobile' => $request->mobile,
                'countries' => $request->countries,
                'projects' => $request->projects,
                'modules' => $request->modules,
                'engagementDate' => $request->engagementDate,
                'titanInductionDate' => $request->titanInductionDate,
                'clientInduction' => $request->clientInduction,
                'loginStatus' => false,
                'authCreated' => false,
            ];
            
            $mongoUrl = env('MONGODB_USERS_CREATE_URL', 'https://api.titandrillingzm.com:6003/users/upload');
            
            Log::info('Creating user in MongoDB', [
                'userId' => $userId,
                'email' => $userData['email']
            ]);
            
            $response = Http::timeout(15)->post($mongoUrl, [
                'userId' => $userId,
                'userData' => $userData
            ]);
            
            if ($response->successful()) {
                $result = $response->json();
                
                Log::info('User created successfully', ['userId' => $userId]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'User created successfully',
                    'userId' => $userId,
                    'data' => $result
                ]);
            }
            
            Log::error('Failed to create user in MongoDB', [
                'status' => $response->status(),
                'response' => $response->json()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create user'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in saveUser', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get specific user by ID from MongoDB
     */
    public function getUserById($userId)
    {
        try {
            $mongoUrl = env('MONGODB_USERS_GET_BY_ID_URL', 'https://api.titandrillingzm.com:6003/users') . '/' . $userId;
            
            Log::info('Fetching user by ID from MongoDB', ['userId' => $userId]);
            
            $response = Http::timeout(10)->get($mongoUrl);
            
            if ($response->successful()) {
                $data = $response->json();
                
                return response()->json([
                    'success' => true,
                    'data' => $data['data'] ?? []
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
            
        } catch (\Exception $e) {
            Log::error('Exception in getUserById', [
                'message' => $e->getMessage(),
                'userId' => $userId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Update user in MongoDB
     */
    public function updateUser(Request $request, $userId)
    {
        try {
            $mongoUrl = env('MONGODB_USERS_UPDATE_URL', 'https://api.titandrillingzm.com:6003/users') . '/' . $userId;
            
            Log::info('Updating user in MongoDB', ['userId' => $userId]);
            
            $response = Http::timeout(15)->put($mongoUrl, $request->all());
            
            if ($response->successful()) {
                $result = $response->json();
                
                return response()->json([
                    'success' => true,
                    'message' => 'User updated successfully',
                    'data' => $result
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update user'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in updateUser', [
                'message' => $e->getMessage(),
                'userId' => $userId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Delete user from MongoDB
     */
    public function deleteUser($userId)
    {
        try {
            $mongoUrl = env('MONGODB_USERS_DELETE_URL', 'https://api.titandrillingzm.com:6003/users') . '/' . $userId;
            
            Log::info('Deleting user from MongoDB', ['userId' => $userId]);
            
            $response = Http::timeout(10)->delete($mongoUrl);
            
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'User deleted successfully'
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete user'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in deleteUser', [
                'message' => $e->getMessage(),
                'userId' => $userId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
}
