@extends('welcome')

@section('title', 'Incident KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
@endsection

@section('content')
<div class="incident-kpi-container">
    <!-- Loading Overlay - Matching Inspection Design -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Incident Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Inspection Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Incident KPI Dashboard</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn-header">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refresh-data-btn-header">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="incident-kpi-content">
        <div class="content-box">

            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="severity">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                            <line x1="12" y1="9" x2="12" y2="13"/>
                            <line x1="12" y1="17" x2="12.01" y2="17"/>
                        </svg>
                        Severity Analysis
                    </button>
                    <button class="tab-btn" data-tab="categories">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="7" height="7"/>
                            <rect x="14" y="3" width="7" height="7"/>
                            <rect x="14" y="14" width="7" height="7"/>
                            <rect x="3" y="14" width="7" height="7"/>
                        </svg>
                        Categories
                    </button>
                    <button class="tab-btn" data-tab="location">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/>
                            <circle cx="12" cy="10" r="3"/>
                        </svg>
                        Location & Project
                    </button>
                    <button class="tab-btn" data-tab="trends">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
                        </svg>
                        Trends
                    </button>
                    <button class="tab-btn" data-tab="reports">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                            <polyline points="14 2 14 8 20 8"/>
                        </svg>
                        Reports List
                    </button>
                </div>
                <div class="tab-content">
                    <!-- Overview Tab -->
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <div class="kpi-card">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                        <polyline points="14 2 14 8 20 8"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Incidents</h3>
                                    <div class="kpi-value" id="totalIncidents">0</div>
                                    <div class="kpi-label">Reports Filed</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Critical Incidents</h3>
                                    <div class="kpi-value" id="criticalIncidents">0</div>
                                    <div class="kpi-label">High Priority</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon success">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <polyline points="20 6 9 17 4 12"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Resolved Incidents</h3>
                                    <div class="kpi-value" id="resolvedIncidents">0</div>
                                    <div class="kpi-label">Closed Cases</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon info">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <polyline points="12 6 12 12 16 14"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Pending Incidents</h3>
                                    <div class="kpi-value" id="pendingIncidents">0</div>
                                    <div class="kpi-label">Under Review</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="kpi-cards-grid">
                            <div class="kpi-card">
                                <div class="kpi-icon secondary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                                        <circle cx="9" cy="7" r="4"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Personnel Incidents</h3>
                                    <div class="kpi-value" id="personnelIncidents">0</div>
                                    <div class="kpi-label">Category</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon secondary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Equipment Involved</h3>
                                    <div class="kpi-value" id="equipmentIncidents">0</div>
                                    <div class="kpi-label">Incidents</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Incident Trends</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="incidentTrendsChart"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Status Distribution</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="statusDistributionChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Severity Analysis Tab -->
                    <div id="severity" class="tab-pane">
                        <div class="severity-cards-grid">
                            <div class="severity-card critical">
                                <h4>Critical</h4>
                                <div class="severity-value" id="severityCritical">0</div>
                            </div>
                            <div class="severity-card high">
                                <h4>High</h4>
                                <div class="severity-value" id="severityHigh">0</div>
                            </div>
                            <div class="severity-card medium">
                                <h4>Medium</h4>
                                <div class="severity-value" id="severityMedium">0</div>
                            </div>
                            <div class="severity-card low">
                                <h4>Low</h4>
                                <div class="severity-value" id="severityLow">0</div>
                            </div>
                        </div>
                        
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Severity Distribution</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="severityTrendsChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Incident Categories Tab -->
                    <div id="categories" class="tab-pane">
                        <div class="category-cards-grid">
                            <div class="category-card personnel">
                                <div class="category-icon">
                                    <i class="fas fa-user-injured"></i>
                                </div>
                                <div class="category-info">
                                    <h4>Personnel</h4>
                                    <div class="category-value" id="categoryPersonnel">0</div>
                                </div>
                            </div>
                            <div class="category-card equipment">
                                <div class="category-icon">
                                    <i class="fas fa-cogs"></i>
                                </div>
                                <div class="category-info">
                                    <h4>Equipment</h4>
                                    <div class="category-value" id="categoryEquipment">0</div>
                                </div>
                            </div>
                            <div class="category-card environmental">
                                <div class="category-icon">
                                    <i class="fas fa-leaf"></i>
                                </div>
                                <div class="category-info">
                                    <h4>Environmental</h4>
                                    <div class="category-value" id="categoryEnvironmental">0</div>
                                </div>
                            </div>
                            <div class="category-card vehicle">
                                <div class="category-icon">
                                    <i class="fas fa-truck"></i>
                                </div>
                                <div class="category-info">
                                    <h4>Vehicle</h4>
                                    <div class="category-value" id="categoryVehicle">0</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Incident Categories Distribution</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="categoriesChart"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Top Incident Types</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="incidentTypesChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Location & Project Tab -->
                    <div id="location" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Incidents by Country</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="countryChart"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Incidents by Project</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="projectChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Trends Tab -->
                    <div id="trends" class="tab-pane">
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Monthly Incident Trends</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="monthlyTrendsChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Reports List Tab -->
                    <div id="reports" class="tab-pane">
                        <div class="table-section">
                            <div class="search-container">
                                <input type="text" id="searchInput" class="search-input" placeholder="Search incidents...">
                            </div>
                            <div class="table-wrapper">
                                <table class="incidents-table">
                                    <thead>
                                        <tr>
                                            <th>Incident ID</th>
                                            <th>Date</th>
                                            <th>Type</th>
                                            <th>Severity</th>
                                            <th>Location</th>
                                            <th>Status</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody id="incidentsTableBody">
                                        <tr>
                                            <td colspan="7" class="no-data">Loading incidents...</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>
<script>
// Firebase Configuration
const firebaseConfig = {
    apiKey: "AIzaSyDsA1JohORkg-EdXWTHlFEea8zCqfvig8w",
    authDomain: "titan-drilling-1f8e9.firebaseapp.com",
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com",
    projectId: "titan-drilling-1f8e9",
    storageBucket: "titan-drilling-1f8e9.firebasestorage.app",
    messagingSenderId: "749788854975",
    appId: "1:749788854975:web:71224235d9355be09b640a"
};

// Initialize Firebase
if (!firebase.apps.length) {
    firebase.initializeApp(firebaseConfig);
}

// Global variables
let allIncidents = [];
let filteredIncidents = [];
let currentPeriod = '7days';
let chartInstances = {};
let customStartDate = null;
let customEndDate = null;

// Load data on page load
document.addEventListener('DOMContentLoaded', function() {
    loadIncidentsFromFirebase();
    setupEventListeners();
});

// Show/hide loading state
function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

// Load incidents from Firebase
async function loadIncidentsFromFirebase() {
    try {
        console.log('Loading incidents from Firebase...');
        
        // Show loading state
        showLoadingState(true);
        
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalIncidentReports/admins.json');
        
        if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
        
        const data = await response.json();
        allIncidents = [];
        
        if (data && typeof data === 'object') {
            Object.keys(data).forEach(uid => {
                const userData = data[uid];
                if (userData && typeof userData === 'object') {
                    Object.keys(userData).forEach(incidentId => {
                        const incident = userData[incidentId];
                        if (incident && typeof incident === 'object') {
                            // Process selected incidents to extract types
                            const incidentTypes = [];
                            if (incident.selectedIncidents && typeof incident.selectedIncidents === 'object') {
                                Object.keys(incident.selectedIncidents).forEach(type => {
                                    if (incident.selectedIncidents[type] === true) {
                                        incidentTypes.push(type);
                                    }
                                });
                            }
                            
                            // Create processed incident object
                            const processedIncident = {
                                ...incident,
                                uid,
                                incidentId: incident.id || incident.incidentNumber || incidentId,
                                incidentNumber: incident.incidentNumber || incident.id,
                                incidentTypes: incidentTypes,
                                incidentTypesText: incidentTypes.join(', ') || 'N/A',
                                category: incident.incidentCategory || 'Unknown',
                                location: incident.incidentArea || incident.country || 'Unknown',
                                pdfUrl: incident.step2?.pdfDownloadUrl || '',
                                cause: incident.step2?.incidentCause || '',
                                correctiveAction: incident.step2?.immediateCorrectiveAction || '',
                                equipmentDamage: incident.step2?.equipmentDamage || '',
                                additionalComment: incident.step2?.additionalComment || ''
                            };
                            
                            allIncidents.push(processedIncident);
                        }
                    });
                }
            });
        }
        
        console.log('Loaded incidents:', allIncidents.length);
        console.log('Sample incident data:', allIncidents[0]);
        
        if (allIncidents.length === 0) {
            if (window.showNotification) window.showNotification('No incident data found', 'info');
        } else {
            if (window.showNotification) window.showNotification(`Loaded ${allIncidents.length} incident(s)`, 'success');
        }
        
        filterDataByPeriod();
        showLoadingState(false);
    } catch (error) {
        console.error('Error loading incidents:', error);
        
        // Show error message to user
        document.getElementById('totalIncidents').textContent = 'Error';
        const tbody = document.getElementById('incidentsTableBody');
        if (tbody) {
            tbody.innerHTML = '<tr><td colspan="7" class="no-data" style="color: #dc3545;">Error loading incidents from Firebase. Please check console for details.</td></tr>';
        }
        
        if (window.showNotification) window.showNotification('Error fetching incident data: ' + error.message, 'error');
        showLoadingState(false);
    }
}

// Setup event listeners
function setupEventListeners() {
    document.querySelectorAll('.period-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.period-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            currentPeriod = this.getAttribute('data-period');
            filterDataByPeriod();
        });
    });
    
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
            document.querySelectorAll('.tab-pane').forEach(p => p.classList.remove('active'));
            this.classList.add('active');
            document.getElementById(this.getAttribute('data-tab')).classList.add('active');
        });
    });
    
    // Export button (header)
    const exportBtnHeader = document.getElementById('export-excel-btn-header');
    if (exportBtnHeader) {
        exportBtnHeader.addEventListener('click', () => {
            exportToExcel();
        });
    }
    
    // Refresh button (header)
    const refreshBtnHeader = document.getElementById('refresh-data-btn-header');
    if (refreshBtnHeader) {
        refreshBtnHeader.addEventListener('click', () => {
            if (window.showNotification) window.showNotification('Refreshing data...', 'info');
            loadIncidentsFromFirebase();
        });
    }
    
    // Custom date range functionality
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                if (window.showNotification) window.showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                if (window.showNotification) window.showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentPeriod = 'custom';
            
            filterDataByPeriod();
            if (window.showNotification) window.showNotification(`Showing data from ${new Date(startDate).toLocaleDateString()} to ${new Date(endDate).toLocaleDateString()}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            
            // Switch back to default (7 days)
            currentPeriod = '7days';
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            // Hide custom date range picker
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            filterDataByPeriod();
            if (window.showNotification) window.showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Set max date to today for date inputs
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
    }
    
    // Search functionality
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            filterIncidentsTable(this.value);
        });
    }
}

// Filter incidents table by search term
function filterIncidentsTable(searchTerm) {
    const tbody = document.getElementById('incidentsTableBody');
    const term = searchTerm.toLowerCase();
    
    const searchResults = filteredIncidents.filter(incident => {
        return (incident.incidentNumber && incident.incidentNumber.toLowerCase().includes(term)) ||
               (incident.category && incident.category.toLowerCase().includes(term)) ||
               (incident.severity && incident.severity.toLowerCase().includes(term)) ||
               (incident.country && incident.country.toLowerCase().includes(term)) ||
               (incident.project && incident.project.toLowerCase().includes(term)) ||
               (incident.incidentTypesText && incident.incidentTypesText.toLowerCase().includes(term));
    });
    
    tbody.innerHTML = '';
    
    if (searchResults.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="no-data">No incidents match your search</td></tr>';
        return;
    }
    
    searchResults.forEach(incident => {
        const row = document.createElement('tr');
        const incidentDate = new Date(incident.incidentDate || incident.createdAt);
        const formattedDate = incidentDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
        
        row.innerHTML = `
            <td><strong>${incident.incidentNumber || incident.incidentId || 'N/A'}</strong></td>
            <td>${formattedDate}</td>
            <td>
                <div style="font-weight: 600; color: #0055a5; margin-bottom: 4px;">${incident.category || 'Unknown'}</div>
                <div style="font-size: 11px; color: #666;">${incident.incidentTypesText || 'N/A'}</div>
            </td>
            <td><span class="severity-badge ${(incident.severity || 'Low').toLowerCase()}">${incident.severity || 'Low'}</span></td>
            <td>
                <div style="font-weight: 600;">${incident.country || 'N/A'}</div>
                <div style="font-size: 11px; color: #666;">${incident.project || 'N/A'}</div>
            </td>
            <td><span class="status-badge ${(incident.status || 'Pending').toLowerCase()}">${incident.status || 'Pending'}</span></td>
            <td>
                <button class="btn-view" onclick="viewIncidentDetails('${incident.incidentNumber}')">
                    <i class="fas fa-eye"></i> View
                </button>
            </td>
        `;
        tbody.appendChild(row);
    });
}

// Filter data by period
function filterDataByPeriod() {
    const now = new Date();
    let startDate = new Date();
    let endDate = now;
    
    // Handle custom date range
    if (currentPeriod === 'custom') {
        if (customStartDate && customEndDate) {
            startDate = new Date(customStartDate);
            startDate.setHours(0, 0, 0, 0);
            endDate = new Date(customEndDate);
            endDate.setHours(23, 59, 59, 999);
        } else {
            // If no custom dates selected, return all data
            filteredIncidents = allIncidents;
            updateAllDashboards();
            return;
        }
    } else {
        switch (currentPeriod) {
            case 'today': 
                startDate.setHours(0, 0, 0, 0);
                break;
            case '7days': 
                startDate.setDate(now.getDate() - 7);
                break;
            case '30days': 
                startDate.setDate(now.getDate() - 30);
                break;
            case '90days': 
                startDate.setDate(now.getDate() - 90);
                break;
            default: 
                startDate.setDate(now.getDate() - 7);
        }
    }
    
    filteredIncidents = allIncidents.filter(incident => {
        const incidentDate = new Date(incident.createdAt || incident.incidentDate);
        return incidentDate >= startDate && incidentDate <= endDate;
    });
    
    console.log('Filtered incidents:', filteredIncidents.length);
    updateAllDashboards();
}

// Update all dashboards
function updateAllDashboards() {
    updateOverviewTab();
    updateSeverityTab();
    updateCategoriesTab();
    updateLocationTab();
    updateTrendsTab();
    updateReportsTab();
}

// Update Overview Tab
function updateOverviewTab() {
    const totalIncidents = filteredIncidents.length;
    const criticalIncidents = filteredIncidents.filter(i => i.severity === 'Critical' || i.severity === 'High').length;
    const resolvedIncidents = filteredIncidents.filter(i => i.status === 'Resolved' || i.status === 'Closed').length;
    const pendingIncidents = filteredIncidents.filter(i => i.status === 'Pending' || i.status === 'Under Review' || !i.status).length;
    const personnelIncidents = filteredIncidents.filter(i => i.category === 'Personnel').length;
    const equipmentInvolvedIncidents = filteredIncidents.filter(i => i.involvesEquipment === true).length;
    
    document.getElementById('totalIncidents').textContent = totalIncidents;
    document.getElementById('criticalIncidents').textContent = criticalIncidents;
    document.getElementById('resolvedIncidents').textContent = resolvedIncidents;
    document.getElementById('pendingIncidents').textContent = pendingIncidents;
    document.getElementById('personnelIncidents').textContent = personnelIncidents;
    document.getElementById('equipmentIncidents').textContent = equipmentInvolvedIncidents;
    
    // Update charts
    updateIncidentTrendsChart();
    updateStatusDistributionChart();
}

// Update Incident Trends Chart
function updateIncidentTrendsChart() {
    const dateGroups = {};
    filteredIncidents.forEach(incident => {
        const date = new Date(incident.createdAt || incident.incidentDate).toLocaleDateString();
        dateGroups[date] = (dateGroups[date] || 0) + 1;
    });
    
    const dates = Object.keys(dateGroups).sort();
    const counts = dates.map(d => dateGroups[d]);
    
    updateChart('incidentTrendsChart', dates, counts, ['#dc3545']);
}

// Update Status Distribution Chart
function updateStatusDistributionChart() {
    const statusCounts = {};
    filteredIncidents.forEach(incident => {
        const status = incident.status || 'Unknown';
        statusCounts[status] = (statusCounts[status] || 0) + 1;
    });
    
    const statuses = Object.keys(statusCounts);
    const counts = Object.values(statusCounts);
    
    updateChart('statusDistributionChart', statuses, counts, ['#28a745', '#ffc107', '#dc3545', '#6c757d']);
}

// Update Severity Tab
function updateSeverityTab() {
    const severityCounts = { Critical: 0, High: 0, Medium: 0, Low: 0 };
    
    filteredIncidents.forEach(incident => {
        const severity = incident.severity || 'Low';
        if (severityCounts.hasOwnProperty(severity)) {
            severityCounts[severity]++;
        } else {
            // Handle any other severity levels
            severityCounts.Low++;
        }
    });
    
    document.getElementById('severityCritical').textContent = severityCounts.Critical;
    document.getElementById('severityHigh').textContent = severityCounts.High;
    document.getElementById('severityMedium').textContent = severityCounts.Medium;
    document.getElementById('severityLow').textContent = severityCounts.Low;
    
    // Update severity trends chart
    updateSeverityTrendsChart(severityCounts);
}

// Update Severity Trends Chart
function updateSeverityTrendsChart(severityCounts) {
    const canvas = document.getElementById('severityTrendsChart');
    if (!canvas) return;
    if (chartInstances['severityTrendsChart']) chartInstances['severityTrendsChart'].destroy();
    
    chartInstances['severityTrendsChart'] = new Chart(canvas, {
        type: 'pie',
        data: {
            labels: ['Critical', 'High', 'Medium', 'Low'],
            datasets: [{
                backgroundColor: ['#dc3545', '#fd7e14', '#ffc107', '#28a745'],
                data: [severityCounts.Critical, severityCounts.High, severityCounts.Medium, severityCounts.Low]
            }]
        },
        options: {
            legend: { display: true, position: 'bottom' },
            responsive: true,
            maintainAspectRatio: false
        }
    });
}

// Update Categories Tab
function updateCategoriesTab() {
    const categoryCounts = { Personnel: 0, Equipment: 0, Environmental: 0, Vehicle: 0, Other: 0 };
    const incidentTypeCounts = {};
    
    filteredIncidents.forEach(incident => {
        const category = incident.category || 'Other';
        if (categoryCounts.hasOwnProperty(category)) {
            categoryCounts[category]++;
        } else {
            categoryCounts.Other++;
        }
        
        // Count incident types
        if (incident.incidentTypes && Array.isArray(incident.incidentTypes)) {
            incident.incidentTypes.forEach(type => {
                incidentTypeCounts[type] = (incidentTypeCounts[type] || 0) + 1;
            });
        }
    });
    
    // Update category cards
    document.getElementById('categoryPersonnel').textContent = categoryCounts.Personnel;
    document.getElementById('categoryEquipment').textContent = categoryCounts.Equipment;
    document.getElementById('categoryEnvironmental').textContent = categoryCounts.Environmental;
    document.getElementById('categoryVehicle').textContent = categoryCounts.Vehicle;
    
    // Update categories chart
    const categories = Object.keys(categoryCounts);
    const categoriesData = Object.values(categoryCounts);
    updateChart('categoriesChart', categories, categoriesData, ['#0055A5', '#19A979', '#F5D904', '#ED4A7B', '#6c757d']);
    
    // Update top incident types chart (top 10)
    const sortedTypes = Object.entries(incidentTypeCounts)
        .sort((a, b) => b[1] - a[1])
        .slice(0, 10);
    
    if (sortedTypes.length > 0) {
        const typeLabels = sortedTypes.map(t => t[0]);
        const typeData = sortedTypes.map(t => t[1]);
        updateChart('incidentTypesChart', typeLabels, typeData, ['#945ECF']);
    }
}

// Update Location Tab
function updateLocationTab() {
    const countryCounts = {};
    const projectCounts = {};
    
    filteredIncidents.forEach(incident => {
        const country = incident.country || 'Unknown';
        const project = incident.project || 'Unknown';
        countryCounts[country] = (countryCounts[country] || 0) + 1;
        projectCounts[project] = (projectCounts[project] || 0) + 1;
    });
    
    updateChart('countryChart', Object.keys(countryCounts), Object.values(countryCounts), ['#0055A5']);
    updateChart('projectChart', Object.keys(projectCounts), Object.values(projectCounts), ['#19A979']);
}

// Update Trends Tab
function updateTrendsTab() {
    const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    const monthlyCounts = new Array(12).fill(0);
    
    filteredIncidents.forEach(incident => {
        const month = new Date(incident.createdAt || incident.incidentDate).getMonth();
        monthlyCounts[month]++;
    });
    
    updateChart('monthlyTrendsChart', months, monthlyCounts, ['#945ECF']);
}

// Update Reports Tab
function updateReportsTab() {
    const tbody = document.getElementById('incidentsTableBody');
    tbody.innerHTML = '';
    
    if (filteredIncidents.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="no-data">No incidents found</td></tr>';
        return;
    }
    
    filteredIncidents.forEach(incident => {
        const row = document.createElement('tr');
        const incidentDate = new Date(incident.incidentDate || incident.createdAt);
        const formattedDate = incidentDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
        
        row.innerHTML = `
            <td><strong>${incident.incidentNumber || incident.incidentId || 'N/A'}</strong></td>
            <td>${formattedDate}</td>
            <td>
                <div style="font-weight: 600; color: #0055a5; margin-bottom: 4px;">${incident.category || 'Unknown'}</div>
                <div style="font-size: 11px; color: #666;">${incident.incidentTypesText || 'N/A'}</div>
            </td>
            <td><span class="severity-badge ${(incident.severity || 'Low').toLowerCase()}">${incident.severity || 'Low'}</span></td>
            <td>
                <div style="font-weight: 600;">${incident.country || 'N/A'}</div>
                <div style="font-size: 11px; color: #666;">${incident.project || 'N/A'}</div>
            </td>
            <td><span class="status-badge ${(incident.status || 'Pending').toLowerCase()}">${incident.status || 'Pending'}</span></td>
            <td>
                <button class="btn-view" onclick="viewIncidentDetails('${incident.incidentNumber}')">
                    <i class="fas fa-eye"></i> View
                </button>
            </td>
        `;
        tbody.appendChild(row);
    });
}

// Helper: Update chart
function updateChart(canvasId, labels, data, colors) {
    const canvas = document.getElementById(canvasId);
    if (!canvas) return;
    if (chartInstances[canvasId]) chartInstances[canvasId].destroy();
    
    chartInstances[canvasId] = new Chart(canvas, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{ backgroundColor: colors, data: data }]
        },
        options: {
            legend: { display: false },
            scales: { yAxes: [{ ticks: { beginAtZero: true } }] },
            responsive: true,
            maintainAspectRatio: false
        }
    });
}

// Export to Excel - Complete data export with ALL fields
function exportToExcel() {
    if (filteredIncidents.length === 0) {
        if (window.showNotification) window.showNotification('No data to export', 'error');
        return;
    }
    
    // Helper function to escape CSV values
    const escapeCSV = (value) => {
        if (value === null || value === undefined) return '';
        const str = String(value);
        // Escape double quotes and wrap in quotes if contains comma, newline, or quote
        if (str.includes(',') || str.includes('\n') || str.includes('"')) {
            return `"${str.replace(/"/g, '""')}"`;
        }
        return str;
    };
    
    // Create comprehensive CSV with ALL available fields from Firebase structure
    const headers = [
        'S.No',
        'Incident Number',
        'Incident ID',
        'Incident Date',
        'Created At',
        'Category',
        'Severity',
        'Country',
        'Project',
        'Incident Area',
        'GPS Address',
        'Coordinates',
        'Admin UID',
        'Status',
        // Incident Types (all possible types)
        'Brake Failure',
        'Chemical Spill',
        'Collision',
        'Drowned in Water',
        'Electrical Failure',
        'Engine Failure',
        'Equipment Failure',
        'Fall from elevation',
        'Fell on side',
        'Loss of control',
        'Oil Spill',
        'Other Incident',
        'Person hit by Vehicle',
        'Puncture',
        'Roll over',
        'Steering Failure',
        'Stuck',
        'Tire Burst',
        'Incident Types (Combined)',
        // Equipment/Vehicle Information
        'Involves Equipment',
        'Vehicle Name',
        'Vehicle ID',
        'Person Involved',
        // Step 2 Details
        'Incident Cause',
        'Immediate Corrective Action',
        'Equipment Damage',
        'Additional Comments',
        'Other Incident Note',
        // PDF and Links
        'PDF Download URL',
        'Firebase UID',
        'Database Key'
    ];
    
    let csv = headers.join(',') + '\n';
    
    filteredIncidents.forEach((incident, index) => {
        const incidentDate = new Date(incident.incidentDate || incident.createdAt);
        const formattedDate = incidentDate.toLocaleString('en-US', { 
            year: 'numeric', 
            month: '2-digit', 
            day: '2-digit', 
            hour: '2-digit', 
            minute: '2-digit' 
        });
        
        const createdAtDate = new Date(incident.createdAt);
        const formattedCreatedAt = createdAtDate.toLocaleString('en-US', { 
            year: 'numeric', 
            month: '2-digit', 
            day: '2-digit', 
            hour: '2-digit', 
            minute: '2-digit' 
        });
        
        // Get selected incidents (all types)
        const selectedIncidents = incident.selectedIncidents || {};
        
        const row = [
            index + 1,
            escapeCSV(incident.incidentNumber || ''),
            escapeCSV(incident.incidentId || incident.id || ''),
            escapeCSV(formattedDate),
            escapeCSV(formattedCreatedAt),
            escapeCSV(incident.category || ''),
            escapeCSV(incident.severity || ''),
            escapeCSV(incident.country || ''),
            escapeCSV(incident.project || ''),
            escapeCSV(incident.incidentArea || ''),
            escapeCSV(incident.gpsAddress || ''),
            escapeCSV(incident.coordinates || ''),
            escapeCSV(incident.adminUid || ''),
            escapeCSV(incident.status || 'Pending'),
            // All incident types as Yes/No columns
            selectedIncidents['Brake Failure'] ? 'Yes' : 'No',
            selectedIncidents['Chemical Spill'] ? 'Yes' : 'No',
            selectedIncidents['Collision'] ? 'Yes' : 'No',
            selectedIncidents['Drowned in Water'] ? 'Yes' : 'No',
            selectedIncidents['Electrical Failure'] ? 'Yes' : 'No',
            selectedIncidents['Engine Failure'] ? 'Yes' : 'No',
            selectedIncidents['Equipment Failure'] ? 'Yes' : 'No',
            selectedIncidents['Fall from elevation'] ? 'Yes' : 'No',
            selectedIncidents['Fell on side'] ? 'Yes' : 'No',
            selectedIncidents['Loss of control'] ? 'Yes' : 'No',
            selectedIncidents['Oil Spill'] ? 'Yes' : 'No',
            selectedIncidents['Other'] ? 'Yes' : 'No',
            selectedIncidents['Person hit by Vehicle'] ? 'Yes' : 'No',
            selectedIncidents['Puncture'] ? 'Yes' : 'No',
            selectedIncidents['Roll over'] ? 'Yes' : 'No',
            selectedIncidents['Steering Failure'] ? 'Yes' : 'No',
            selectedIncidents['Stuck'] ? 'Yes' : 'No',
            selectedIncidents['Tire Burst'] ? 'Yes' : 'No',
            escapeCSV(incident.incidentTypesText || ''),
            // Equipment/Vehicle
            incident.involvesEquipment ? 'Yes' : 'No',
            escapeCSV(incident.vehicleName || ''),
            escapeCSV(incident.vehicleId || ''),
            escapeCSV(incident.personInvolved || ''),
            // Step 2 details
            escapeCSV(incident.cause || ''),
            escapeCSV(incident.correctiveAction || ''),
            escapeCSV(incident.equipmentDamage || ''),
            escapeCSV(incident.additionalComment || ''),
            escapeCSV(incident.otherIncidentNote || ''),
            // PDF and metadata
            escapeCSV(incident.pdfUrl || ''),
            escapeCSV(incident.uid || ''),
            escapeCSV(incident.incidentId || '')
        ];
        
        csv += row.join(',') + '\n';
    });
    
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    
    // Generate filename with date range
    const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
    const timestamp = new Date().toISOString().split('T')[0];
    const filename = `Incident_Report_Complete_${dateRangeText}_${timestamp}.csv`;
    
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    window.URL.revokeObjectURL(url);
    
    if (window.showNotification) window.showNotification(`Exported ${filteredIncidents.length} incident(s) with complete data to Excel successfully`, 'success');
}

// Get current date range as text for filename
function getCurrentDateRangeText() {
    if (currentPeriod === 'today') {
        return 'Today';
    } else if (currentPeriod === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentPeriod.replace('days', '')}_Days`;
    }
}

// View incident details in modal
function viewIncidentDetails(incidentNumber) {
    const incident = allIncidents.find(i => i.incidentNumber === incidentNumber);
    if (!incident) {
        alert('Incident not found');
        return;
    }
    
    // Create modal content
    const modalHtml = `
        <div class="incident-modal-overlay" onclick="closeIncidentModal(event)">
            <div class="incident-modal-content" onclick="event.stopPropagation()">
                <div class="incident-modal-header">
                    <h2>Incident Details</h2>
                    <button class="modal-close-btn" onclick="closeIncidentModal()">&times;</button>
                </div>
                <div class="incident-modal-body">
                    <div class="detail-section">
                        <h3>Basic Information</h3>
                        <div class="detail-grid">
                            <div class="detail-item">
                                <span class="detail-label">Incident Number:</span>
                                <span class="detail-value">${incident.incidentNumber}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Date:</span>
                                <span class="detail-value">${new Date(incident.incidentDate).toLocaleString()}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Category:</span>
                                <span class="detail-value">${incident.category}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Severity:</span>
                                <span class="detail-value severity-badge ${incident.severity.toLowerCase()}">${incident.severity}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Country:</span>
                                <span class="detail-value">${incident.country}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Project:</span>
                                <span class="detail-value">${incident.project}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Area:</span>
                                <span class="detail-value">${incident.incidentArea || 'N/A'}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Involves Equipment:</span>
                                <span class="detail-value">${incident.involvesEquipment ? 'Yes' : 'No'}</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="detail-section">
                        <h3>Incident Types</h3>
                        <div class="incident-types-badges">
                            ${incident.incidentTypes.map(type => `<span class="type-badge">${type}</span>`).join('')}
                        </div>
                    </div>
                    
                    ${incident.involvesEquipment && (incident.vehicleName || incident.vehicleId) ? `
                    <div class="detail-section">
                        <h3>Equipment Information</h3>
                        <div class="detail-grid">
                            <div class="detail-item">
                                <span class="detail-label">Vehicle Name:</span>
                                <span class="detail-value">${incident.vehicleName || 'N/A'}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Vehicle ID:</span>
                                <span class="detail-value">${incident.vehicleId || 'N/A'}</span>
                            </div>
                        </div>
                    </div>
                    ` : ''}
                    
                    ${incident.personInvolved ? `
                    <div class="detail-section">
                        <h3>Person Involved</h3>
                        <p class="detail-text">${incident.personInvolved}</p>
                    </div>
                    ` : ''}
                    
                    ${incident.cause || incident.correctiveAction || incident.equipmentDamage || incident.additionalComment ? `
                    <div class="detail-section">
                        <h3>Additional Details</h3>
                        ${incident.cause ? `
                        <div class="detail-item-full">
                            <span class="detail-label">Incident Cause:</span>
                            <p class="detail-text">${incident.cause}</p>
                        </div>
                        ` : ''}
                        ${incident.correctiveAction ? `
                        <div class="detail-item-full">
                            <span class="detail-label">Immediate Corrective Action:</span>
                            <p class="detail-text">${incident.correctiveAction}</p>
                        </div>
                        ` : ''}
                        ${incident.equipmentDamage ? `
                        <div class="detail-item-full">
                            <span class="detail-label">Equipment Damage:</span>
                            <p class="detail-text">${incident.equipmentDamage}</p>
                        </div>
                        ` : ''}
                        ${incident.additionalComment ? `
                        <div class="detail-item-full">
                            <span class="detail-label">Additional Comments:</span>
                            <p class="detail-text">${incident.additionalComment}</p>
                        </div>
                        ` : ''}
                    </div>
                    ` : ''}
                </div>
                <div class="incident-modal-footer">
                    ${incident.pdfUrl ? `
                    <button class="btn-primary" onclick="window.open('${incident.pdfUrl}', '_blank')">
                        <i class="fas fa-file-pdf"></i> View PDF Report
                    </button>
                    ` : ''}
                    <button class="btn-secondary" onclick="closeIncidentModal()">Close</button>
                </div>
            </div>
        </div>
    `;
    
    // Add modal to page
    document.body.insertAdjacentHTML('beforeend', modalHtml);
}

function closeIncidentModal(event) {
    if (event && event.target !== event.currentTarget) return;
    const modal = document.querySelector('.incident-modal-overlay');
    if (modal) modal.remove();
}
</script>

<style>
/* CSS Variables for consistent theming - Matching Inspection KPI */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.incident-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* ===== HEADER STYLES - Matching Inspection Design ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* ===== MAIN CONTENT ===== */
.incident-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin-bottom: 1rem;
    margin: 0;
    overflow: hidden;
}


/* ===== TIME PERIOD SELECTOR ===== */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* ===== MODERN TABS ===== */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

.tab-btn.active::after {
    content: none;
}

/* ===== TAB CONTENT ===== */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* ===== KPI CARDS ===== */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon.success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.kpi-icon.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.kpi-icon.info {
    background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
}

.kpi-icon.secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
}

.kpi-content {
    flex: 1;
}

.kpi-content h3 {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-label {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* ===== CHARTS SECTION ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-container {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-container canvas {
    max-width: 100%;
    max-height: 100%;
}

/* Severity Cards */
.severity-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.severity-card {
    padding: 1.5rem;
    border-radius: var(--radius-md);
    text-align: center;
    color: white;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.severity-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.severity-card.critical { background: #dc3545; }
.severity-card.high { background: #fd7e14; }
.severity-card.medium { background: #ffc107; color: #333; }
.severity-card.low { background: #28a745; }

.severity-card h4 {
    margin: 0 0 0.5rem 0;
    font-size: 0.875rem;
    text-transform: uppercase;
    font-weight: 600;
}

.severity-value {
    font-size: 2rem;
    font-weight: 700;
}

/* Category Cards */
.category-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.category-card {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.category-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.category-icon {
    width: 56px;
    height: 56px;
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.75rem;
    color: white;
}

.category-card.personnel .category-icon { background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); }
.category-card.equipment .category-icon { background: linear-gradient(135deg, #ffc107 0%, #e0a800 100%); }
.category-card.environmental .category-icon { background: linear-gradient(135deg, #28a745 0%, #218838 100%); }
.category-card.vehicle .category-icon { background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%); }

.category-info h4 {
    margin: 0 0 0.5rem 0;
    font-size: 0.875rem;
    font-weight: 600;
    color: var(--text-secondary);
    text-transform: uppercase;
}

.category-value {
    font-size: 2rem;
    font-weight: 700;
    color: var(--text-primary);
}

/* Table */
.table-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
}

.search-container {
    margin-bottom: 1rem;
}

.search-input {
    width: 100%;
    max-width: 300px;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.table-wrapper {
    overflow-x: auto;
}

.incidents-table {
    width: 100%;
    border-collapse: collapse;
}

.incidents-table th,
.incidents-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid var(--border-color);
    font-size: 14px;
}

.incidents-table th {
    background: var(--bg-light);
    font-weight: 600;
    color: var(--text-primary);
}

.incidents-table tbody tr:hover {
    background: var(--bg-light);
}

.severity-badge,
.status-badge {
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.severity-badge.critical { background: #dc3545; color: white; }
.severity-badge.high { background: #fd7e14; color: white; }
.severity-badge.medium { background: #ffc107; color: #333; }
.severity-badge.low { background: #28a745; color: white; }

.status-badge.resolved,
.status-badge.closed { background: #28a745; color: white; }
.status-badge.pending { background: #ffc107; color: #333; }
.status-badge.investigating { background: #0dcaf0; color: #333; }

.btn-view {
    padding: 8px 16px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    cursor: pointer;
    font-size: 13px;
    font-weight: 500;
    transition: var(--transition-base);
    display: inline-flex;
    align-items: center;
    gap: 6px;
}

.btn-view:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.no-data {
    text-align: center;
    padding: 2rem;
    color: var(--text-muted);
    font-style: italic;
}

/* Incident Detail Modal */
.incident-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    animation: fadeIn 0.3s ease;
}

.incident-modal-content {
    background: white;
    border-radius: var(--radius-md);
    max-width: 800px;
    width: 90%;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
}

.incident-modal-header {
    background: var(--primary-color);
    color: white;
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-radius: var(--radius-md) var(--radius-md) 0 0;
}

.incident-modal-header h2 {
    margin: 0;
    font-size: 1.5rem;
}

.modal-close-btn {
    background: none;
    border: none;
    color: white;
    font-size: 2rem;
    cursor: pointer;
    line-height: 1;
    padding: 0;
    width: 32px;
    height: 32px;
}

.modal-close-btn:hover {
    opacity: 0.8;
}

.incident-modal-body {
    padding: 1.5rem;
}

.detail-section {
    margin-bottom: 1.5rem;
    padding-bottom: 1.5rem;
    border-bottom: 1px solid var(--border-color);
}

.detail-section:last-child {
    border-bottom: none;
}

.detail-section h3 {
    color: var(--primary-color);
    font-size: 1.125rem;
    margin: 0 0 1rem 0;
    font-weight: 600;
}

.detail-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.detail-item {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.detail-item-full {
    margin-bottom: 1rem;
}

.detail-label {
    font-size: 0.75rem;
    color: var(--text-muted);
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.detail-value {
    font-size: 0.875rem;
    color: var(--text-primary);
    font-weight: 500;
}

.detail-text {
    font-size: 0.875rem;
    color: var(--text-secondary);
    line-height: 1.6;
    margin: 0.5rem 0 0 0;
    padding: 0.75rem;
    background: var(--bg-primary);
    border-radius: var(--radius-md);
}

.incident-types-badges {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
}

.type-badge {
    background: var(--primary-color);
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 1rem;
    font-size: 0.75rem;
    font-weight: 600;
}

.incident-modal-footer {
    padding: 1.5rem;
    border-top: 1px solid var(--border-color);
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
}

.btn-primary,
.btn-secondary {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: var(--radius-md);
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: var(--transition-base);
}

.btn-primary {
    background: var(--primary-color);
    color: white;
}

.btn-primary:hover {
    background: var(--primary-hover);
}

.btn-secondary {
    background: var(--bg-white);
    color: var(--text-primary);
    border: 2px solid var(--border-color);
}

.btn-secondary:hover {
    background: var(--bg-primary);
}

@keyframes fadeIn {
    from {
        opacity: 0;
    }
    to {
        opacity: 1;
    }
}

/* Loading/Sync Modal - Matching Inspection Design */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 992px) {
    .kpi-cards-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 16px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .incident-kpi-container,
    .incident-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
        margin: 0;
        border-radius: var(--radius-lg);
    }
    
    
    .time-period-selector {
        margin-bottom: 1.5rem;
    }
    
    .period-buttons {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 8px;
    }
    
    .period-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
        text-align: center;
    }
    
    .custom-date-range {
        padding: 1rem;
    }
    
    .date-range-inputs {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .date-input-group {
        width: 100%;
    }
    
    .date-input {
        width: 100%;
        min-width: auto;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        width: 100%;
        justify-content: center;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
        border-bottom: 1px solid var(--border-color);
    }
    
    .tab-btn {
        justify-content: flex-start;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
        border-bottom: 1px solid var(--border-color);
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .kpi-card {
        padding: 16px;
        flex-direction: column;
        text-align: center;
        gap: 12px;
    }
    
    .kpi-icon {
        width: 48px;
        height: 48px;
    }
    
    .kpi-content h3 {
        font-size: 14px;
    }
    
    .kpi-value {
        font-size: 24px;
    }
    
    .kpi-label {
        font-size: 12px;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .chart-card {
        padding: 16px;
    }
    
    .chart-header h3 {
        font-size: 16px;
    }
    
    .chart-container {
        height: 200px;
    }
    
    .severity-cards-grid {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .incident-modal-content {
        width: 95%;
        max-height: 95vh;
    }
    
    .detail-grid {
        grid-template-columns: 1fr;
    }
    
    .incident-modal-footer {
        flex-direction: column;
    }
    
    .btn-primary,
    .btn-secondary {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .header-title {
        font-size: 20px;
    }
    
    .header-right {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
        padding: 10px 12px;
        font-size: 13px;
    }
    
    .content-box {
        padding: 14px;
    }
    
    
    .period-buttons {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 10px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.875rem;
    }
    
    .date-input {
        padding: 10px;
        font-size: 14px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 10px 14px;
        font-size: 13px;
    }
    
    .tab-btn {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 40px;
    }
    
    .kpi-card {
        padding: 14px;
    }
    
    .kpi-icon {
        width: 44px;
        height: 44px;
    }
    
    .kpi-content h3 {
        font-size: 13px;
    }
    
    .kpi-value {
        font-size: 22px;
    }
    
    .kpi-label {
        font-size: 11px;
    }
    
    .chart-container {
        height: 190px;
    }
    
    .chart-card {
        padding: 12px;
    }
    
    .chart-header h3 {
        font-size: 14px;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-title {
        font-size: 18px;
    }
    
    .header-right {
        width: 100%;
    }
    
    .create-btn {
        font-size: 12px;
        padding: 9px 12px;
    }
    
    .content-box {
        padding: 12px;
    }
    
    
    .period-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.75rem;
    }
    
    .date-input {
        padding: 9px;
        font-size: 13px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 9px 12px;
        font-size: 12px;
    }
    
    .tab-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
    }
    
    .kpi-card {
        padding: 12px;
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }
    
    .kpi-icon {
        width: 40px;
        height: 40px;
    }
    
    .kpi-content h3 {
        font-size: 13px;
    }
    
    .kpi-value {
        font-size: 20px;
    }
    
    .kpi-label {
        font-size: 11px;
    }
    
    .chart-card {
        padding: 12px;
    }
    
    .chart-header h3 {
        font-size: 14px;
    }
    
    .chart-container {
        height: 180px;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .header-gradient,
    .period-buttons,
    .tab-navigation {
        display: none !important;
    }
    
    .incident-kpi-container {
        background: white;
    }
    
    .kpi-card,
    .chart-card {
        border: 1px solid #000;
        break-inside: avoid;
    }
}
</style>

<script>
// Incident KPI Functionality - Matching Inspection Design
(function() {
    'use strict';
    
    // Configuration
    const CONFIG = {
        notificationTimeout: 5000,
        selectors: {
            tabButtons: '.tab-btn',
            tabPanes: '.tab-pane',
            periodButtons: '.period-btn',
            kpiCards: '.kpi-card',
            chartContainers: '.chart-container'
        }
    };

    // Cached DOM elements
    let elements = {};
    
    // Initialize DOM elements
    function initElements() {
        elements = {
            tabButtons: document.querySelectorAll(CONFIG.selectors.tabButtons),
            tabPanes: document.querySelectorAll(CONFIG.selectors.tabPanes),
            periodButtons: document.querySelectorAll(CONFIG.selectors.periodButtons),
            kpiCards: document.querySelectorAll(CONFIG.selectors.kpiCards),
            chartContainers: document.querySelectorAll(CONFIG.selectors.chartContainers)
        };
    }

    // Tab switching functionality
    function initTabSwitching() {
        elements.tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');

            // Remove active class from all tabs and panes
                elements.tabButtons.forEach(btn => btn.classList.remove('active'));
                elements.tabPanes.forEach(pane => pane.classList.remove('active'));

            // Add active class to clicked tab and corresponding pane
            this.classList.add('active');
            const targetPane = document.getElementById(targetTab);
            if (targetPane) {
                targetPane.classList.add('active');
            }
        });
    });
    }

    // Time period selector functionality
    function initPeriodSelector() {
        const customDateRange = document.getElementById('custom-date-range');
        
        elements.periodButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all period buttons
                elements.periodButtons.forEach(btn => btn.classList.remove('active'));
            
            // Add active class to clicked button
            this.classList.add('active');
            
            const period = this.getAttribute('data-period');
            console.log('Period changed to:', period);
            
            // Show/hide custom date range picker
            if (period === 'custom') {
                if (customDateRange) {
                    customDateRange.style.display = 'block';
                }
            } else {
                if (customDateRange) {
                    customDateRange.style.display = 'none';
                }
            }
        });
    });
    }

    // KPI card hover effects
    function initKpiCardEffects() {
        elements.kpiCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-4px)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
    }

    // Chart container responsive handling
    function initChartContainers() {
        elements.chartContainers.forEach(container => {
        const canvas = container.querySelector('canvas');
        if (canvas) {
            canvas.style.maxWidth = '100%';
            canvas.style.height = 'auto';
        }
    });
    }

    // Notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close" aria-label="Close notification">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        setTimeout(() => removeNotification(notification), CONFIG.notificationTimeout);
        
        notification.querySelector('.notification-close').addEventListener('click', () => {
            removeNotification(notification);
        });
    }
    
    function removeNotification(notification) {
        if (notification && notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }

    // Add notification styles
    function addNotificationStyles() {
    const notificationStyles = document.createElement('style');
    notificationStyles.textContent = `
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: var(--z-index-notification);
            padding: 16px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            font-family: var(--font-family);
            font-weight: 500;
            max-width: 400px;
            min-width: 300px;
            animation: slideIn 0.3s ease;
            background: white;
        }
        
        .notification-success {
            border-left: 4px solid #28a745;
        }
        
        .notification-error {
            border-left: 4px solid #dc3545;
        }
        
        .notification-info {
            border-left: 4px solid #17a2b8;
        }
        
        .notification-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 1rem;
        }
        
        .notification-content span {
            color: #333;
            font-size: 14px;
        }
        
        .notification-close {
            background: none;
            border: none;
            font-size: 18px;
            color: #999;
            cursor: pointer;
            padding: 0;
            line-height: 1;
            margin-left: 12px;
        }
        
        .notification-close:hover {
            color: #666;
        }
        
        @keyframes slideIn {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
    `;
    document.head.appendChild(notificationStyles);
    }

    // Initialize everything
    function init() {
        initElements();
        initTabSwitching();
        initPeriodSelector();
        initKpiCardEffects();
        initChartContainers();
        addNotificationStyles();
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
    // Make showNotification globally accessible
    window.showNotification = showNotification;
})();
</script>
@endsection

