@extends('welcome')

@section('title', 'Logbook KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
@endsection

@section('content')
<div class="logbook-kpi-container">
    <!-- Loading Overlay - Matching Inspection Design -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Logbook Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Inspection Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Logbook KPI Dashboard</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn-header">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refresh-data-btn-header">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="logbook-kpi-content">
        <div class="content-box">
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="by-vehicle">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                        </svg>
                        By Vehicle
                    </button>
                    <button class="tab-btn" data-tab="by-operator">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                            <circle cx="9" cy="7" r="4"/>
                        </svg>
                        By Operator
                    </button>
                    <button class="tab-btn" data-tab="location">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/>
                            <circle cx="12" cy="10" r="3"/>
                        </svg>
                        Location & Project
                    </button>
                    <button class="tab-btn" data-tab="trends">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
                        </svg>
                        Trends
                    </button>
                    <button class="tab-btn" data-tab="reports">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                            <polyline points="14 2 14 8 20 8"/>
                        </svg>
                        Logbook Entries
                    </button>
                </div>
                <div class="tab-content">
                    <!-- Overview Tab -->
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <div class="kpi-card">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                        <polyline points="14 2 14 8 20 8"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Entries</h3>
                                    <div class="kpi-value" id="totalEntries">0</div>
                                    <div class="kpi-label">Logbook Records</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon success">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Unique Vehicles</h3>
                                    <div class="kpi-value" id="uniqueVehicles">0</div>
                                    <div class="kpi-label">Tracked</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon info">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <polyline points="12 6 12 12 16 14"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Hours Logged</h3>
                                    <div class="kpi-value" id="totalHours">0</div>
                                    <div class="kpi-label">Operating Hours</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card">
                                <div class="kpi-icon warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                                        <circle cx="9" cy="7" r="4"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Active Operators</h3>
                                    <div class="kpi-value" id="activeOperators">0</div>
                                    <div class="kpi-label">Users</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Daily Logbook Entries</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="dailyEntriesChart"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Hours by Vehicle Type</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="vehicleTypeChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- By Vehicle Tab -->
                    <div id="by-vehicle" class="tab-pane">
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Entries by Vehicle</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="vehicleChart"></canvas>
                            </div>
                        </div>
                        
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Operating Hours by Vehicle</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="vehicleHoursChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- By Operator Tab -->
                    <div id="by-operator" class="tab-pane">
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Entries by Operator</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="operatorChart"></canvas>
                            </div>
                        </div>
                        
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Hours by Operator</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="operatorHoursChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Location & Project Tab -->
                    <div id="location" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Entries by Country</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="countryChart"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Entries by Project</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="projectChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Trends Tab -->
                    <div id="trends" class="tab-pane">
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Monthly Logbook Trends</h3>
                            </div>
                            <div class="chart-container">
                                <canvas id="monthlyTrendsChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Logbook Entries Tab -->
                    <div id="reports" class="tab-pane">
                        <div class="table-section">
                            <div class="search-container">
                                <input type="text" id="searchInput" class="search-input" placeholder="Search logbook entries...">
                            </div>
                            <div class="table-wrapper">
                                <table class="logbook-table">
                                    <thead>
                                        <tr>
                                            <th>Entry ID</th>
                                            <th>Date</th>
                                            <th>Vehicle</th>
                                            <th>Operator</th>
                                            <th>Hours</th>
                                            <th>Location</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody id="logbookTableBody">
                                        <tr>
                                            <td colspan="7" class="no-data">Loading logbook entries...</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>
<script>
// Firebase Configuration
const firebaseConfig = {
    apiKey: "AIzaSyDsA1JohORkg-EdXWTHlFEea8zCqfvig8w",
    authDomain: "titan-drilling-1f8e9.firebaseapp.com",
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com",
    projectId: "titan-drilling-1f8e9",
    storageBucket: "titan-drilling-1f8e9.firebasestorage.app",
    messagingSenderId: "749788854975",
    appId: "1:749788854975:web:71224235d9355be09b640a"
};

// Initialize Firebase
if (!firebase.apps.length) {
    firebase.initializeApp(firebaseConfig);
}

// Global variables
let allLogbookEntries = [];
let filteredLogbookEntries = [];
let currentPeriod = '7days';
let chartInstances = {};
let customStartDate = null;
let customEndDate = null;

// Load data on page load
document.addEventListener('DOMContentLoaded', function() {
    loadLogbookFromFirebase();
    setupEventListeners();
});

// Show/hide loading state
function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

// Load logbook entries from Firebase
async function loadLogbookFromFirebase() {
    try {
        console.log('Loading logbook entries from Firebase...');
        
        // Show loading state
        showLoadingState(true);
        
        // Fetch from multiple possible paths
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalLogbook/admins.json');
        
        if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
        
        const data = await response.json();
        allLogbookEntries = [];
        
        if (data && typeof data === 'object') {
            Object.keys(data).forEach(uid => {
                const userData = data[uid];
                if (userData && typeof userData === 'object') {
                    Object.keys(userData).forEach(entryId => {
                        const entry = userData[entryId];
                        if (entry && typeof entry === 'object') {
                            // Process logbook entry
                            const processedEntry = {
                                ...entry,
                                uid,
                                entryId: entry.id || entryId,
                                vehicleName: entry.vehicleName || entry.equipment || entry.vehicleNumber || 'Unknown',
                                operatorName: entry.operatorName || entry.driver || entry.user || 'Unknown',
                                location: entry.location || entry.site || entry.country || 'Unknown',
                                hours: parseFloat(entry.hours || entry.operatingHours || entry.workHours || 0),
                                date: entry.date || entry.entryDate || entry.createdAt,
                                vehicleType: entry.vehicleType || entry.equipmentType || 'Unknown'
                            };
                            
                            allLogbookEntries.push(processedEntry);
                        }
                    });
                }
            });
        }
        
        console.log('Loaded logbook entries:', allLogbookEntries.length);
        console.log('Sample entry:', allLogbookEntries[0]);
        
        if (allLogbookEntries.length === 0) {
            if (window.showNotification) window.showNotification('No logbook data found', 'info');
        } else {
            if (window.showNotification) window.showNotification(`Loaded ${allLogbookEntries.length} logbook entry(ies)`, 'success');
        }
        
        filterDataByPeriod();
        showLoadingState(false);
    } catch (error) {
        console.error('Error loading logbook:', error);
        
        // Show error message to user
        document.getElementById('totalEntries').textContent = 'Error';
        const tbody = document.getElementById('logbookTableBody');
        if (tbody) {
            tbody.innerHTML = '<tr><td colspan="7" class="no-data" style="color: #dc3545;">Error loading logbook from Firebase. Please check console for details.</td></tr>';
        }
        
        if (window.showNotification) window.showNotification('Error fetching logbook data: ' + error.message, 'error');
        showLoadingState(false);
    }
}

// Setup event listeners
function setupEventListeners() {
    document.querySelectorAll('.period-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.period-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            currentPeriod = this.getAttribute('data-period');
            filterDataByPeriod();
        });
    });
    
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
            document.querySelectorAll('.tab-pane').forEach(p => p.classList.remove('active'));
            this.classList.add('active');
            document.getElementById(this.getAttribute('data-tab')).classList.add('active');
        });
    });
    
    // Export button (header)
    const exportBtnHeader = document.getElementById('export-excel-btn-header');
    if (exportBtnHeader) {
        exportBtnHeader.addEventListener('click', () => {
            exportToExcel();
        });
    }
    
    // Refresh button (header)
    const refreshBtnHeader = document.getElementById('refresh-data-btn-header');
    if (refreshBtnHeader) {
        refreshBtnHeader.addEventListener('click', () => {
            if (window.showNotification) window.showNotification('Refreshing data...', 'info');
            loadLogbookFromFirebase();
        });
    }
    
    // Custom date range functionality
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                if (window.showNotification) window.showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                if (window.showNotification) window.showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentPeriod = 'custom';
            
            filterDataByPeriod();
            if (window.showNotification) window.showNotification(`Showing data from ${new Date(startDate).toLocaleDateString()} to ${new Date(endDate).toLocaleDateString()}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            
            // Switch back to default (7 days)
            currentPeriod = '7days';
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            // Hide custom date range picker
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            filterDataByPeriod();
            if (window.showNotification) window.showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Set max date to today for date inputs
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
    }
    
    // Search functionality
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            filterLogbookTable(this.value);
        });
    }
}

// Filter logbook table by search term
function filterLogbookTable(searchTerm) {
    const tbody = document.getElementById('logbookTableBody');
    const term = searchTerm.toLowerCase();
    
    const searchResults = filteredLogbookEntries.filter(entry => {
        return (entry.entryId && entry.entryId.toLowerCase().includes(term)) ||
               (entry.vehicleName && entry.vehicleName.toLowerCase().includes(term)) ||
               (entry.operatorName && entry.operatorName.toLowerCase().includes(term)) ||
               (entry.location && entry.location.toLowerCase().includes(term)) ||
               (entry.country && entry.country.toLowerCase().includes(term)) ||
               (entry.project && entry.project.toLowerCase().includes(term));
    });
    
    tbody.innerHTML = '';
    
    if (searchResults.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="no-data">No entries match your search</td></tr>';
        return;
    }
    
    searchResults.forEach(entry => {
        const row = document.createElement('tr');
        const entryDate = new Date(entry.date);
        const formattedDate = entryDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
        
        row.innerHTML = `
            <td><strong>${entry.entryId || 'N/A'}</strong></td>
            <td>${formattedDate}</td>
            <td>
                <div style="font-weight: 600; color: #0055a5;">${entry.vehicleName || 'N/A'}</div>
                <div style="font-size: 11px; color: #666;">${entry.vehicleType || 'N/A'}</div>
            </td>
            <td>${entry.operatorName || 'N/A'}</td>
            <td><strong>${entry.hours.toFixed(1)}h</strong></td>
            <td>
                <div style="font-weight: 600;">${entry.country || 'N/A'}</div>
                <div style="font-size: 11px; color: #666;">${entry.project || 'N/A'}</div>
            </td>
            <td>
                <button class="btn-view" onclick="viewLogbookDetails('${entry.entryId}')">
                    <i class="fas fa-eye"></i> View
                </button>
            </td>
        `;
        tbody.appendChild(row);
    });
}

// Filter data by period
function filterDataByPeriod() {
    const now = new Date();
    let startDate = new Date();
    let endDate = now;
    
    // Handle custom date range
    if (currentPeriod === 'custom') {
        if (customStartDate && customEndDate) {
            startDate = new Date(customStartDate);
            startDate.setHours(0, 0, 0, 0);
            endDate = new Date(customEndDate);
            endDate.setHours(23, 59, 59, 999);
        } else {
            filteredLogbookEntries = allLogbookEntries;
            updateAllDashboards();
            return;
        }
    } else {
        switch (currentPeriod) {
            case 'today': 
                startDate.setHours(0, 0, 0, 0);
                break;
            case '7days': 
                startDate.setDate(now.getDate() - 7);
                break;
            case '30days': 
                startDate.setDate(now.getDate() - 30);
                break;
            case '90days': 
                startDate.setDate(now.getDate() - 90);
                break;
            default: 
                startDate.setDate(now.getDate() - 7);
        }
    }
    
    filteredLogbookEntries = allLogbookEntries.filter(entry => {
        const entryDate = new Date(entry.date);
        return entryDate >= startDate && entryDate <= endDate;
    });
    
    console.log('Filtered logbook entries:', filteredLogbookEntries.length);
    updateAllDashboards();
}

// Update all dashboards
function updateAllDashboards() {
    updateOverviewTab();
    updateVehicleTab();
    updateOperatorTab();
    updateLocationTab();
    updateTrendsTab();
    updateReportsTab();
}

// Update Overview Tab
function updateOverviewTab() {
    const totalEntries = filteredLogbookEntries.length;
    const uniqueVehicles = new Set(filteredLogbookEntries.map(e => e.vehicleName)).size;
    const totalHours = filteredLogbookEntries.reduce((sum, e) => sum + e.hours, 0);
    const activeOperators = new Set(filteredLogbookEntries.map(e => e.operatorName)).size;
    
    document.getElementById('totalEntries').textContent = totalEntries;
    document.getElementById('uniqueVehicles').textContent = uniqueVehicles;
    document.getElementById('totalHours').textContent = totalHours.toFixed(1);
    document.getElementById('activeOperators').textContent = activeOperators;
    
    // Update charts
    updateDailyEntriesChart();
    updateVehicleTypeChart();
}

// Update Daily Entries Chart
function updateDailyEntriesChart() {
    const dateGroups = {};
    filteredLogbookEntries.forEach(entry => {
        const date = new Date(entry.date).toLocaleDateString();
        dateGroups[date] = (dateGroups[date] || 0) + 1;
    });
    
    const dates = Object.keys(dateGroups).sort();
    const counts = dates.map(d => dateGroups[d]);
    
    updateChart('dailyEntriesChart', dates, counts, ['#0055A5']);
}

// Update Vehicle Type Chart
function updateVehicleTypeChart() {
    const typeGroups = {};
    filteredLogbookEntries.forEach(entry => {
        const type = entry.vehicleType || 'Unknown';
        typeGroups[type] = (typeGroups[type] || 0) + 1;
    });
    
    updateChart('vehicleTypeChart', Object.keys(typeGroups), Object.values(typeGroups), ['#19A979', '#945ECF', '#E8743B', '#0055A5']);
}

// Update Vehicle Tab
function updateVehicleTab() {
    const vehicleCounts = {};
    const vehicleHours = {};
    
    filteredLogbookEntries.forEach(entry => {
        const vehicle = entry.vehicleName;
        vehicleCounts[vehicle] = (vehicleCounts[vehicle] || 0) + 1;
        vehicleHours[vehicle] = (vehicleHours[vehicle] || 0) + entry.hours;
    });
    
    updateChart('vehicleChart', Object.keys(vehicleCounts), Object.values(vehicleCounts), ['#0055A5']);
    updateChart('vehicleHoursChart', Object.keys(vehicleHours), Object.values(vehicleHours), ['#19A979']);
}

// Update Operator Tab
function updateOperatorTab() {
    const operatorCounts = {};
    const operatorHours = {};
    
    filteredLogbookEntries.forEach(entry => {
        const operator = entry.operatorName;
        operatorCounts[operator] = (operatorCounts[operator] || 0) + 1;
        operatorHours[operator] = (operatorHours[operator] || 0) + entry.hours;
    });
    
    updateChart('operatorChart', Object.keys(operatorCounts), Object.values(operatorCounts), ['#945ECF']);
    updateChart('operatorHoursChart', Object.keys(operatorHours), Object.values(operatorHours), ['#E8743B']);
}

// Update Location Tab
function updateLocationTab() {
    const countryCounts = {};
    const projectCounts = {};
    
    filteredLogbookEntries.forEach(entry => {
        const country = entry.country || 'Unknown';
        const project = entry.project || 'Unknown';
        countryCounts[country] = (countryCounts[country] || 0) + 1;
        projectCounts[project] = (projectCounts[project] || 0) + 1;
    });
    
    updateChart('countryChart', Object.keys(countryCounts), Object.values(countryCounts), ['#0055A5']);
    updateChart('projectChart', Object.keys(projectCounts), Object.values(projectCounts), ['#19A979']);
}

// Update Trends Tab
function updateTrendsTab() {
    const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    const monthlyCounts = new Array(12).fill(0);
    
    filteredLogbookEntries.forEach(entry => {
        const month = new Date(entry.date).getMonth();
        monthlyCounts[month]++;
    });
    
    updateChart('monthlyTrendsChart', months, monthlyCounts, ['#945ECF']);
}

// Update Reports Tab
function updateReportsTab() {
    const tbody = document.getElementById('logbookTableBody');
    tbody.innerHTML = '';
    
    if (filteredLogbookEntries.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="no-data">No logbook entries found</td></tr>';
        return;
    }
    
    filteredLogbookEntries.forEach(entry => {
        const row = document.createElement('tr');
        const entryDate = new Date(entry.date);
        const formattedDate = entryDate.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
        
        row.innerHTML = `
            <td><strong>${entry.entryId || 'N/A'}</strong></td>
            <td>${formattedDate}</td>
            <td>
                <div style="font-weight: 600; color: #0055a5;">${entry.vehicleName || 'N/A'}</div>
                <div style="font-size: 11px; color: #666;">${entry.vehicleType || 'N/A'}</div>
            </td>
            <td>${entry.operatorName || 'N/A'}</td>
            <td><strong>${entry.hours.toFixed(1)}h</strong></td>
            <td>
                <div style="font-weight: 600;">${entry.country || 'N/A'}</div>
                <div style="font-size: 11px; color: #666;">${entry.project || 'N/A'}</div>
            </td>
            <td>
                <button class="btn-view" onclick="viewLogbookDetails('${entry.entryId}')">
                    <i class="fas fa-eye"></i> View
                </button>
            </td>
        `;
        tbody.appendChild(row);
    });
}

// Helper: Update chart
function updateChart(canvasId, labels, data, colors) {
    const canvas = document.getElementById(canvasId);
    if (!canvas) return;
    if (chartInstances[canvasId]) chartInstances[canvasId].destroy();
    
    chartInstances[canvasId] = new Chart(canvas, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{ backgroundColor: colors, data: data }]
        },
        options: {
            legend: { display: false },
            scales: { yAxes: [{ ticks: { beginAtZero: true } }] },
            responsive: true,
            maintainAspectRatio: false
        }
    });
}

// Export to Excel - Complete data export
function exportToExcel() {
    if (filteredLogbookEntries.length === 0) {
        if (window.showNotification) window.showNotification('No data to export', 'error');
        return;
    }
    
    // Helper function to escape CSV values
    const escapeCSV = (value) => {
        if (value === null || value === undefined) return '';
        const str = String(value);
        if (str.includes(',') || str.includes('\n') || str.includes('"')) {
            return `"${str.replace(/"/g, '""')}"`;
        }
        return str;
    };
    
    // Create comprehensive CSV with ALL fields
    const headers = [
        'S.No',
        'Entry ID',
        'Date',
        'Vehicle Name',
        'Vehicle Number',
        'Vehicle Type',
        'Operator Name',
        'Operator ID',
        'Operating Hours',
        'Start Time',
        'End Time',
        'Country',
        'Project',
        'Location/Site',
        'GPS Address',
        'Coordinates',
        'Odometer Start',
        'Odometer End',
        'Fuel Used',
        'Notes/Comments',
        'Purpose',
        'Status',
        'Created At',
        'Updated At',
        'Firebase UID',
        'Database Key'
    ];
    
    let csv = headers.join(',') + '\n';
    
    filteredLogbookEntries.forEach((entry, index) => {
        const entryDate = new Date(entry.date);
        const formattedDate = entryDate.toLocaleString('en-US', { 
            year: 'numeric', 
            month: '2-digit', 
            day: '2-digit'
        });
        
        const createdAt = entry.createdAt ? new Date(entry.createdAt).toLocaleString('en-US') : '';
        const updatedAt = entry.updatedAt ? new Date(entry.updatedAt).toLocaleString('en-US') : '';
        
        const row = [
            index + 1,
            escapeCSV(entry.entryId || entry.id || ''),
            escapeCSV(formattedDate),
            escapeCSV(entry.vehicleName || ''),
            escapeCSV(entry.vehicleNumber || ''),
            escapeCSV(entry.vehicleType || ''),
            escapeCSV(entry.operatorName || ''),
            escapeCSV(entry.operatorId || entry.userId || ''),
            escapeCSV(entry.hours || 0),
            escapeCSV(entry.startTime || ''),
            escapeCSV(entry.endTime || ''),
            escapeCSV(entry.country || ''),
            escapeCSV(entry.project || ''),
            escapeCSV(entry.location || entry.site || ''),
            escapeCSV(entry.gpsAddress || ''),
            escapeCSV(entry.coordinates || ''),
            escapeCSV(entry.odometerStart || ''),
            escapeCSV(entry.odometerEnd || ''),
            escapeCSV(entry.fuelUsed || ''),
            escapeCSV(entry.notes || entry.comments || ''),
            escapeCSV(entry.purpose || ''),
            escapeCSV(entry.status || 'Active'),
            escapeCSV(createdAt),
            escapeCSV(updatedAt),
            escapeCSV(entry.uid || ''),
            escapeCSV(entry.entryId || '')
        ];
        
        csv += row.join(',') + '\n';
    });
    
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    
    // Generate filename with date range
    const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
    const timestamp = new Date().toISOString().split('T')[0];
    const filename = `Logbook_Report_Complete_${dateRangeText}_${timestamp}.csv`;
    
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    window.URL.revokeObjectURL(url);
    
    if (window.showNotification) window.showNotification(`Exported ${filteredLogbookEntries.length} logbook entry(ies) with complete data successfully`, 'success');
}

// Get current date range as text for filename
function getCurrentDateRangeText() {
    if (currentPeriod === 'today') {
        return 'Today';
    } else if (currentPeriod === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentPeriod.replace('days', '')}_Days`;
    }
}

// View logbook details in modal
function viewLogbookDetails(entryId) {
    const entry = allLogbookEntries.find(e => e.entryId === entryId);
    if (!entry) {
        alert('Logbook entry not found');
        return;
    }
    
    // Create modal content
    const modalHtml = `
        <div class="logbook-modal-overlay" onclick="closeLogbookModal(event)">
            <div class="logbook-modal-content" onclick="event.stopPropagation()">
                <div class="logbook-modal-header">
                    <h2>Logbook Entry Details</h2>
                    <button class="modal-close-btn" onclick="closeLogbookModal()">&times;</button>
                </div>
                <div class="logbook-modal-body">
                    <div class="detail-section">
                        <h3>Basic Information</h3>
                        <div class="detail-grid">
                            <div class="detail-item">
                                <span class="detail-label">Entry ID:</span>
                                <span class="detail-value">${entry.entryId}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Date:</span>
                                <span class="detail-value">${new Date(entry.date).toLocaleString()}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Vehicle Name:</span>
                                <span class="detail-value">${entry.vehicleName}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Vehicle Type:</span>
                                <span class="detail-value">${entry.vehicleType || 'N/A'}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Operator:</span>
                                <span class="detail-value">${entry.operatorName}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Operating Hours:</span>
                                <span class="detail-value">${entry.hours.toFixed(1)} hours</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Country:</span>
                                <span class="detail-value">${entry.country || 'N/A'}</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">Project:</span>
                                <span class="detail-value">${entry.project || 'N/A'}</span>
                            </div>
                        </div>
                    </div>
                    
                    ${entry.startTime || entry.endTime ? `
                    <div class="detail-section">
                        <h3>Time Information</h3>
                        <div class="detail-grid">
                            ${entry.startTime ? `
                            <div class="detail-item">
                                <span class="detail-label">Start Time:</span>
                                <span class="detail-value">${entry.startTime}</span>
                            </div>
                            ` : ''}
                            ${entry.endTime ? `
                            <div class="detail-item">
                                <span class="detail-label">End Time:</span>
                                <span class="detail-value">${entry.endTime}</span>
                            </div>
                            ` : ''}
                        </div>
                    </div>
                    ` : ''}
                    
                    ${entry.odometerStart || entry.odometerEnd || entry.fuelUsed ? `
                    <div class="detail-section">
                        <h3>Vehicle Metrics</h3>
                        <div class="detail-grid">
                            ${entry.odometerStart ? `
                            <div class="detail-item">
                                <span class="detail-label">Odometer Start:</span>
                                <span class="detail-value">${entry.odometerStart} km</span>
                            </div>
                            ` : ''}
                            ${entry.odometerEnd ? `
                            <div class="detail-item">
                                <span class="detail-label">Odometer End:</span>
                                <span class="detail-value">${entry.odometerEnd} km</span>
                            </div>
                            ` : ''}
                            ${entry.fuelUsed ? `
                            <div class="detail-item">
                                <span class="detail-label">Fuel Used:</span>
                                <span class="detail-value">${entry.fuelUsed} liters</span>
                            </div>
                            ` : ''}
                        </div>
                    </div>
                    ` : ''}
                    
                    ${entry.notes || entry.comments || entry.purpose ? `
                    <div class="detail-section">
                        <h3>Additional Information</h3>
                        ${entry.purpose ? `
                        <div class="detail-item-full">
                            <span class="detail-label">Purpose:</span>
                            <p class="detail-text">${entry.purpose}</p>
                        </div>
                        ` : ''}
                        ${entry.notes || entry.comments ? `
                        <div class="detail-item-full">
                            <span class="detail-label">Notes/Comments:</span>
                            <p class="detail-text">${entry.notes || entry.comments}</p>
                        </div>
                        ` : ''}
                    </div>
                    ` : ''}
                </div>
                <div class="logbook-modal-footer">
                    <button class="btn-secondary" onclick="closeLogbookModal()">Close</button>
                </div>
            </div>
        </div>
    `;
    
    // Add modal to page
    document.body.insertAdjacentHTML('beforeend', modalHtml);
}

function closeLogbookModal(event) {
    if (event && event.target !== event.currentTarget) return;
    const modal = document.querySelector('.logbook-modal-overlay');
    if (modal) modal.remove();
}
</script>

<style>
/* CSS Variables for consistent theming - Matching Inspection KPI */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.logbook-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* ===== HEADER STYLES - Matching Inspection Design ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* ===== MAIN CONTENT ===== */
.logbook-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin: 0;
    overflow: hidden;
}

/* ===== TIME PERIOD SELECTOR ===== */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* ===== MODERN TABS ===== */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

/* ===== TAB CONTENT ===== */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* ===== KPI CARDS ===== */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon.success { background: linear-gradient(135deg, #10b981 0%, #059669 100%); }
.kpi-icon.warning { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); }
.kpi-icon.info { background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%); }
.kpi-icon.secondary { background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); }

.kpi-content {
    flex: 1;
}

.kpi-content h3 {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-label {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* ===== CHARTS SECTION ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-container {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-container canvas {
    max-width: 100%;
    max-height: 100%;
}

/* Table */
.table-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
}

.search-container {
    margin-bottom: 1rem;
}

.search-input {
    width: 100%;
    max-width: 300px;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.table-wrapper {
    overflow-x: auto;
}

.logbook-table {
    width: 100%;
    border-collapse: collapse;
}

.logbook-table th,
.logbook-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid var(--border-color);
    font-size: 14px;
}

.logbook-table th {
    background: var(--bg-light);
    font-weight: 600;
    color: var(--text-primary);
}

.logbook-table tbody tr:hover {
    background: var(--bg-light);
}

.btn-view {
    padding: 8px 16px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    cursor: pointer;
    font-size: 13px;
    font-weight: 500;
    transition: var(--transition-base);
    display: inline-flex;
    align-items: center;
    gap: 6px;
}

.btn-view:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.no-data {
    text-align: center;
    padding: 2rem;
    color: var(--text-muted);
    font-style: italic;
}

/* Logbook Detail Modal */
.logbook-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    animation: fadeIn 0.3s ease;
}

.logbook-modal-content {
    background: white;
    border-radius: var(--radius-md);
    max-width: 800px;
    width: 90%;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
}

.logbook-modal-header {
    background: var(--primary-color);
    color: white;
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-radius: var(--radius-md) var(--radius-md) 0 0;
}

.logbook-modal-header h2 {
    margin: 0;
    font-size: 1.5rem;
}

.modal-close-btn {
    background: none;
    border: none;
    color: white;
    font-size: 2rem;
    cursor: pointer;
    line-height: 1;
    padding: 0;
    width: 32px;
    height: 32px;
}

.modal-close-btn:hover {
    opacity: 0.8;
}

.logbook-modal-body {
    padding: 1.5rem;
}

.detail-section {
    margin-bottom: 1.5rem;
    padding-bottom: 1.5rem;
    border-bottom: 1px solid var(--border-color);
}

.detail-section:last-child {
    border-bottom: none;
}

.detail-section h3 {
    color: var(--primary-color);
    font-size: 1.125rem;
    margin: 0 0 1rem 0;
    font-weight: 600;
}

.detail-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.detail-item {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.detail-item-full {
    margin-bottom: 1rem;
}

.detail-label {
    font-size: 0.75rem;
    color: var(--text-muted);
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.detail-value {
    font-size: 0.875rem;
    color: var(--text-primary);
    font-weight: 500;
}

.detail-text {
    font-size: 0.875rem;
    color: var(--text-secondary);
    line-height: 1.6;
    margin: 0.5rem 0 0 0;
    padding: 0.75rem;
    background: var(--bg-light);
    border-radius: var(--radius-md);
}

.logbook-modal-footer {
    padding: 1.5rem;
    border-top: 1px solid var(--border-color);
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
}

.btn-primary,
.btn-secondary {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: var(--radius-md);
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: var(--transition-base);
}

.btn-primary {
    background: var(--primary-color);
    color: white;
}

.btn-primary:hover {
    background: var(--primary-hover);
}

.btn-secondary {
    background: var(--bg-white);
    color: var(--text-primary);
    border: 2px solid var(--border-color);
}

.btn-secondary:hover {
    background: var(--bg-light);
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

/* Loading Modal */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .logbook-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 576px) {
    .header-right {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

button:focus-visible,
input:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}
</style>

<script>
// Logbook KPI Functionality - Matching Inspection Design
(function() {
    'use strict';
    
    const CONFIG = {
        notificationTimeout: 5000,
        selectors: {
            tabButtons: '.tab-btn',
            tabPanes: '.tab-pane',
            periodButtons: '.period-btn',
            kpiCards: '.kpi-card',
            chartContainers: '.chart-container'
        }
    };

    let elements = {};
    
    function initElements() {
        elements = {
            tabButtons: document.querySelectorAll(CONFIG.selectors.tabButtons),
            tabPanes: document.querySelectorAll(CONFIG.selectors.tabPanes),
            periodButtons: document.querySelectorAll(CONFIG.selectors.periodButtons),
            kpiCards: document.querySelectorAll(CONFIG.selectors.kpiCards),
            chartContainers: document.querySelectorAll(CONFIG.selectors.chartContainers)
        };
    }

    function initTabSwitching() {
        elements.tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');
                elements.tabButtons.forEach(btn => btn.classList.remove('active'));
                elements.tabPanes.forEach(pane => pane.classList.remove('active'));
                this.classList.add('active');
                const targetPane = document.getElementById(targetTab);
                if (targetPane) targetPane.classList.add('active');
            });
        });
    }

    function initPeriodSelector() {
        const customDateRange = document.getElementById('custom-date-range');
        
        elements.periodButtons.forEach(button => {
            button.addEventListener('click', function() {
                elements.periodButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');
                
                const period = this.getAttribute('data-period');
                if (period === 'custom') {
                    if (customDateRange) customDateRange.style.display = 'block';
                } else {
                    if (customDateRange) customDateRange.style.display = 'none';
                }
            });
        });
    }

    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        setTimeout(() => {
            if (notification.parentNode) notification.parentNode.removeChild(notification);
        }, CONFIG.notificationTimeout);
        
        notification.querySelector('.notification-close').addEventListener('click', () => {
            if (notification.parentNode) notification.parentNode.removeChild(notification);
        });
    }

    function addNotificationStyles() {
        const style = document.createElement('style');
        style.textContent = `
            .notification {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 10000;
                padding: 16px;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: var(--font-family);
                max-width: 400px;
                min-width: 300px;
                animation: slideIn 0.3s ease;
                background: white;
            }
            .notification-success { border-left: 4px solid #28a745; }
            .notification-error { border-left: 4px solid #dc3545; }
            .notification-info { border-left: 4px solid #17a2b8; }
            .notification-content {
                display: flex;
                justify-content: space-between;
                align-items: center;
                gap: 1rem;
            }
            .notification-content span {
                color: #333;
                font-size: 14px;
            }
            .notification-close {
                background: none;
                border: none;
                font-size: 18px;
                color: #999;
                cursor: pointer;
                padding: 0;
            }
            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
        `;
        document.head.appendChild(style);
    }

    function init() {
        initElements();
        initTabSwitching();
        initPeriodSelector();
        addNotificationStyles();
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
    window.showNotification = showNotification;
})();
</script>
@endsection

