@extends('welcome')

@section('title', 'Employees KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="employees-kpi-container">
    <!-- Loading Overlay -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Employee Data...</div>
        </div>
    </div>
    
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('dashboard.team_kpi.home') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Employees KPI</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refresh-btn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="employees-kpi-content">
        <div class="content-box">
            <!-- Filter Section -->
            <div class="filter-section">
                <div class="filter-form">
                    <div class="form-group">
                        <label for="role-filter">Filter by Role:</label>
                        <select class="form-select" id="role-filter">
                            <option value="">All Roles</option>
                        </select>
                    </div>
                    <button class="btn-go" id="apply-filter">Apply</button>
                </div>
            </div>
            
            <!-- Data Table -->
            <div class="table-section">
                <div class="table-responsive">
                    <table class="modern-table">
                        <thead>
                            <tr>
                                <th>S.No</th>
                                <th>Employee Number</th>
                                <th>Role</th>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Inspections</th>
                                <th>Job Cards Created</th>
                            </tr>
                        </thead>
                        <tbody id="employees-table-body">
                            <tr>
                                <td colspan="7" class="loading-row">Loading data...</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <!-- Summary Section -->
            <div class="summary-section">
                <div class="summary-card">
                    <h3>Summary</h3>
                    <div class="summary-stats">
                        <div class="summary-item">
                            <span class="summary-label">Total Employees:</span>
                            <span class="summary-value" id="total-employees">0</span>
                        </div>
                        <div class="summary-item">
                            <span class="summary-label">Total Inspections:</span>
                            <span class="summary-value" id="total-inspections">0</span>
                        </div>
                        <div class="summary-item">
                            <span class="summary-label">Total Job Cards:</span>
                            <span class="summary-value" id="total-jobcards">0</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- User Details Modal -->
<div id="user-details-modal" class="user-modal" style="display: none;" onclick="if(event.target === this) closeUserModal()">
    <div class="user-modal-content">
        <div class="user-modal-header">
            <h2 id="modal-user-name">User Details</h2>
            <button class="user-modal-close" onclick="closeUserModal()">&times;</button>
        </div>
        <div class="user-modal-body">
            <div class="user-info-section">
                <div class="user-info-item">
                    <strong>Email:</strong> <span id="modal-user-email">-</span>
                </div>
                <div class="user-info-item">
                    <strong>Role:</strong> <span id="modal-user-role">-</span>
                </div>
                <div class="user-info-item">
                    <strong>Employee Number:</strong> <span id="modal-user-empnum">-</span>
                </div>
            </div>
            
            <div class="user-tabs">
                <button class="user-tab-btn active" onclick="switchUserTab('inspections')">
                    <i class="fas fa-clipboard-check"></i>
                    Inspections (<span id="user-inspections-count">0</span>)
                </button>
                <button class="user-tab-btn" onclick="switchUserTab('jobcards')">
                    <i class="fas fa-briefcase"></i>
                    Job Cards (<span id="user-jobcards-count">0</span>)
                </button>
            </div>
            
            <div id="user-inspections-tab" class="user-tab-content active">
                <div class="tab-actions">
                    <button class="btn-export-user" onclick="exportUserInspections()">
                        <i class="fas fa-file-excel"></i> Export Inspections
                    </button>
                </div>
                <div class="table-responsive">
                    <table class="modern-table">
                        <thead>
                            <tr>
                                <th>S.No</th>
                                <th>Report ID</th>
                                <th>Equipment</th>
                                <th>Status</th>
                                <th>Created Date</th>
                            </tr>
                        </thead>
                        <tbody id="user-inspections-body">
                            <tr><td colspan="5" class="loading-row">Loading...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <div id="user-jobcards-tab" class="user-tab-content">
                <div class="tab-actions">
                    <button class="btn-export-user" onclick="exportUserJobCards()">
                        <i class="fas fa-file-excel"></i> Export Job Cards
                    </button>
                </div>
                <div class="table-responsive">
                    <table class="modern-table">
                        <thead>
                            <tr>
                                <th>S.No</th>
                                <th>Work Order ID</th>
                                <th>Equipment</th>
                                <th>Status</th>
                                <th>Created Date</th>
                            </tr>
                        </thead>
                        <tbody id="user-jobcards-body">
                            <tr><td colspan="5" class="loading-row">Loading...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Firebase SDK -->
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
const firebaseConfig = {
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com"
};

if (!firebase.apps.length) {
    firebase.initializeApp(firebaseConfig);
}

const database = firebase.database();

const ALL_ROLES = [
    'Operations Manager', 'Health & Safety Manager', 'Project Manager', 'Site Manager',
    'Supervisor', 'HrManager', 'FinanceManager', 'Driller', 'Trainee Driller',
    'Junior Driller', 'Coremarker', 'Assistant', 'Workshop Manager', 'Workshop Foreman',
    'Maintenance Manager', 'Mechanical Engineer', 'Boilermaker', 'Welder', 'Mechanic',
    'Machinist', 'Electrician', 'Housekeeper', 'Cook', 'Stores Officer', 'General Worker'
];

let allUsers = [];
let userStats = {};
let selectedRole = '';
let currentUserData = { user: null, inspections: [], jobCards: [] };

function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

async function fetchAllUsers() {
    showLoadingState(true);
    try {
        const adminsRef = database.ref('/Globalusers/admins');
        const snapshot = await adminsRef.once('value');
        const data = snapshot.val();
        
        if (data) {
            allUsers = Object.keys(data).map(id => ({
                id,
                ...data[id]
            }));
            
            // Populate role filter
            populateRoleFilter();
            
            // Fetch stats for each user
            await fetchUserStats();
            
            // Render table
            renderTable();
        }
        
        showLoadingState(false);
        showNotification(`Loaded ${allUsers.length} employees`, 'success');
    } catch (error) {
        console.error('Error fetching users:', error);
        showNotification('Error loading employee data', 'error');
        showLoadingState(false);
    }
}

async function fetchUserStats() {
    const statsPromises = allUsers.map(async (user) => {
        const userId = user.id;
        let inspections = 0;
        let jobCards = 0;
        
        try {
            const inspSnap = await database.ref(`/GlobalInspectionReport/admins/${userId}`).once('value');
            const inspData = inspSnap.val();
            inspections = inspData ? Object.keys(inspData).length : 0;
        } catch (e) {
            console.log('Error fetching inspections for', userId);
        }
        
        try {
            const wcSnap = await database.ref(`/GlobalWorkOrders/admins/${userId}`).once('value');
            const wcData = wcSnap.val();
            if (wcData) {
                jobCards = Object.keys(wcData).filter(k => k !== 'lastRequestNumber').length;
            }
        } catch (e) {
            console.log('Error fetching job cards for', userId);
        }
        
        return { userId, inspections, jobCards };
    });
    
    const results = await Promise.all(statsPromises);
    results.forEach(({ userId, inspections, jobCards }) => {
        userStats[userId] = { inspections, jobCards };
    });
}

function populateRoleFilter() {
    const roleCounts = {};
    allUsers.forEach(user => {
        const role = user.role || 'Unknown';
        roleCounts[role] = (roleCounts[role] || 0) + 1;
    });
    
    const roleFilter = document.getElementById('role-filter');
    roleFilter.innerHTML = '<option value="">All Roles</option>';
    
    Object.keys(roleCounts).sort().forEach(role => {
        const option = document.createElement('option');
        option.value = role;
        option.textContent = `${role} (${roleCounts[role]})`;
        roleFilter.appendChild(option);
    });
}

function renderTable() {
    const tbody = document.getElementById('employees-table-body');
    const filteredUsers = selectedRole 
        ? allUsers.filter(u => (u.role || '') === selectedRole)
        : allUsers;
    
    if (filteredUsers.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="no-data">No employees found</td></tr>';
        updateSummary(filteredUsers);
        return;
    }
    
    tbody.innerHTML = filteredUsers.map((user, index) => `
        <tr class="clickable-row" onclick="showUserDetails('${user.id}', event)">
            <td>${index + 1}</td>
            <td>${user.employeeNumber || '-'}</td>
            <td><span class="role-badge role-${(user.role || 'unknown').toLowerCase().replace(/\s+/g, '-')}">${user.role || 'Employee'}</span></td>
            <td class="name-cell">${user.fullName || user.name || 'Unknown'}</td>
            <td>${user.email || '-'}</td>
            <td class="count-cell">${userStats[user.id]?.inspections || 0}</td>
            <td class="count-cell">${userStats[user.id]?.jobCards || 0}</td>
        </tr>
    `).join('');
    
    updateSummary(filteredUsers);
}

function updateSummary(users) {
    const totalInspections = users.reduce((sum, u) => sum + (userStats[u.id]?.inspections || 0), 0);
    const totalJobCards = users.reduce((sum, u) => sum + (userStats[u.id]?.jobCards || 0), 0);
    
    document.getElementById('total-employees').textContent = users.length;
    document.getElementById('total-inspections').textContent = totalInspections;
    document.getElementById('total-jobcards').textContent = totalJobCards;
}

function exportToExcel() {
    const filteredUsers = selectedRole 
        ? allUsers.filter(u => (u.role || '') === selectedRole)
        : allUsers;
    
    if (filteredUsers.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const excelData = filteredUsers.map((user, index) => ({
        'S.No': index + 1,
        'Employee Number': user.employeeNumber || '-',
        'Role': user.role || 'Employee',
        'Name': user.fullName || user.name || 'Unknown',
        'Email': user.email || '-',
        'Mobile': user.mobile || '-',
        'Department': user.department || '-',
        'Inspections': userStats[user.id]?.inspections || 0,
        'Job Cards Created': userStats[user.id]?.jobCards || 0,
        'Country': Array.isArray(user.countries) ? user.countries.join(', ') : (user.countries || '-'),
        'Project': Array.isArray(user.projects) ? user.projects.join(', ') : (user.projects || '-')
    }));
    
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(excelData);
    ws['!cols'] = [
        { wch: 6 }, { wch: 18 }, { wch: 20 }, { wch: 25 }, { wch: 30 },
        { wch: 15 }, { wch: 20 }, { wch: 12 }, { wch: 18 }, { wch: 20 }, { wch: 20 }
    ];
    XLSX.utils.book_append_sheet(wb, ws, 'Employees');
    
    const summaryData = [
        { 'Metric': 'Total Employees', 'Value': filteredUsers.length },
        { 'Metric': 'Total Inspections', 'Value': filteredUsers.reduce((s, u) => s + (userStats[u.id]?.inspections || 0), 0) },
        { 'Metric': 'Total Job Cards', 'Value': filteredUsers.reduce((s, u) => s + (userStats[u.id]?.jobCards || 0), 0) },
        { 'Metric': 'Filter Applied', 'Value': selectedRole || 'All Roles' },
        { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB') }
    ];
    
    const wsSummary = XLSX.utils.json_to_sheet(summaryData);
    wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    const timestamp = new Date().toISOString().slice(0, 10);
    const filename = `Employees_KPI_${timestamp}.xlsx`;
    XLSX.writeFile(wb, filename);
    
    showNotification(`Exported ${filteredUsers.length} employee(s) successfully`, 'success');
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span>${message}</span>
            <button class="notification-close">&times;</button>
        </div>
    `;
    document.body.appendChild(notification);
    setTimeout(() => notification.remove(), 5000);
    notification.querySelector('.notification-close').addEventListener('click', () => notification.remove());
}

// Show user details modal
async function showUserDetails(userId, event) {
    const user = allUsers.find(u => u.id === userId);
    if (!user) return;
    
    currentUserData.user = user;
    
    // Update modal header
    document.getElementById('modal-user-name').textContent = user.fullName || user.name || 'Unknown';
    document.getElementById('modal-user-email').textContent = user.email || '-';
    document.getElementById('modal-user-role').textContent = user.role || '-';
    document.getElementById('modal-user-empnum').textContent = user.employeeNumber || '-';
    
    // Show modal
    const modal = document.getElementById('user-details-modal');
    modal.style.display = 'flex';
    
    // Position modal near the clicked row
    if (event) {
        const clickY = event.clientY;
        const windowHeight = window.innerHeight;
        const scrollY = window.scrollY || window.pageYOffset;
        
        // Calculate top position
        let topPosition = clickY + scrollY - 100;
        topPosition = Math.max(scrollY + 20, topPosition);
        
        const modalHeight = 600;
        const maxTop = scrollY + windowHeight - modalHeight - 20;
        if (topPosition > maxTop && maxTop > scrollY + 20) {
            topPosition = maxTop;
        }
        
        // Apply positioning
        const modalContent = modal.querySelector('.user-modal-content');
        modalContent.style.position = 'absolute';
        modalContent.style.top = topPosition + 'px';
        modalContent.style.left = '50%';
        modalContent.style.transform = 'translateX(-50%)';
    }
    
    // Fetch user data
    await fetchUserDetailsData(userId);
}

// Fetch user inspections and job cards
async function fetchUserDetailsData(userId) {
    showLoadingState(true);
    
    try {
        // Fetch Inspections
        const inspSnap = await database.ref(`/GlobalInspectionReport/admins/${userId}`).once('value');
        const inspData = inspSnap.val();
        currentUserData.inspections = [];
        
        if (inspData) {
            Object.keys(inspData).forEach(key => {
                const report = inspData[key];
                currentUserData.inspections.push({
                    id: key,
                    reportId: report?.step1?.inspectionNumber || report?.reportId || key,
                    equipment: report?.step1?.selectedEquipment?.equipmentNumber || '-',
                    status: report?.step3?.status || 'Completed',
                    createdAt: report?.step3?.createdAt || report?.step1?.createdAt || '-'
                });
            });
        }
        
        // Fetch Job Cards
        const wcSnap = await database.ref(`/GlobalWorkOrders/admins/${userId}`).once('value');
        const wcData = wcSnap.val();
        currentUserData.jobCards = [];
        
        if (wcData) {
            Object.keys(wcData).forEach(key => {
                if (key === 'lastRequestNumber') return;
                const card = wcData[key];
                currentUserData.jobCards.push({
                    id: key,
                    workOrderId: card?.workOrderNumber || card?.workOrderId || key,
                    equipment: card?.equipment?.equipmentNumber || '-',
                    status: card?.status || 'Pending',
                    createdAt: card?.createdAt || '-'
                });
            });
        }
        
        // Update counts
        document.getElementById('user-inspections-count').textContent = currentUserData.inspections.length;
        document.getElementById('user-jobcards-count').textContent = currentUserData.jobCards.length;
        
        // Render tables
        renderUserInspections();
        renderUserJobCards();
        
        showLoadingState(false);
    } catch (error) {
        console.error('Error fetching user details:', error);
        showNotification('Error loading user details', 'error');
        showLoadingState(false);
    }
}

// Render user inspections table
function renderUserInspections() {
    const tbody = document.getElementById('user-inspections-body');
    
    if (currentUserData.inspections.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="no-data">No inspections found</td></tr>';
        return;
    }
    
    tbody.innerHTML = currentUserData.inspections.map((insp, index) => `
        <tr>
            <td>${index + 1}</td>
            <td>${insp.reportId}</td>
            <td>${insp.equipment}</td>
            <td><span class="status-badge">${insp.status}</span></td>
            <td>${insp.createdAt ? new Date(insp.createdAt).toLocaleString('en-GB') : '-'}</td>
        </tr>
    `).join('');
}

// Render user job cards table
function renderUserJobCards() {
    const tbody = document.getElementById('user-jobcards-body');
    
    if (currentUserData.jobCards.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="no-data">No job cards found</td></tr>';
        return;
    }
    
    tbody.innerHTML = currentUserData.jobCards.map((card, index) => `
        <tr>
            <td>${index + 1}</td>
            <td>${card.workOrderId}</td>
            <td>${card.equipment}</td>
            <td><span class="status-badge">${card.status}</span></td>
            <td>${card.createdAt ? new Date(card.createdAt).toLocaleString('en-GB') : '-'}</td>
        </tr>
    `).join('');
}

// Close user modal
function closeUserModal() {
    const modal = document.getElementById('user-details-modal');
    modal.style.display = 'none';
    
    // Reset positioning
    const modalContent = modal.querySelector('.user-modal-content');
    modalContent.style.position = '';
    modalContent.style.top = '';
    modalContent.style.left = '';
    modalContent.style.transform = '';
    
    currentUserData = { user: null, inspections: [], jobCards: [] };
}

// Switch user tab
function switchUserTab(tab) {
    document.querySelectorAll('.user-tab-btn').forEach(btn => btn.classList.remove('active'));
    document.querySelectorAll('.user-tab-content').forEach(content => content.classList.remove('active'));
    
    if (tab === 'inspections') {
        document.querySelector('.user-tab-btn:nth-child(1)').classList.add('active');
        document.getElementById('user-inspections-tab').classList.add('active');
    } else {
        document.querySelector('.user-tab-btn:nth-child(2)').classList.add('active');
        document.getElementById('user-jobcards-tab').classList.add('active');
    }
}

// Export user inspections
function exportUserInspections() {
    if (currentUserData.inspections.length === 0) {
        showNotification('No inspections to export', 'error');
        return;
    }
    
    const data = currentUserData.inspections.map((insp, index) => ({
        'S.No': index + 1,
        'Report ID': insp.reportId,
        'Equipment': insp.equipment,
        'Status': insp.status,
        'Created Date': insp.createdAt ? new Date(insp.createdAt).toLocaleString('en-GB') : '-'
    }));
    
    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Inspections');
    
    const userName = currentUserData.user?.fullName || currentUserData.user?.name || 'User';
    XLSX.writeFile(wb, `${userName}_Inspections_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Inspections exported successfully', 'success');
}

// Export user job cards
function exportUserJobCards() {
    if (currentUserData.jobCards.length === 0) {
        showNotification('No job cards to export', 'error');
        return;
    }
    
    const data = currentUserData.jobCards.map((card, index) => ({
        'S.No': index + 1,
        'Work Order ID': card.workOrderId,
        'Equipment': card.equipment,
        'Status': card.status,
        'Created Date': card.createdAt ? new Date(card.createdAt).toLocaleString('en-GB') : '-'
    }));
    
    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Job Cards');
    
    const userName = currentUserData.user?.fullName || currentUserData.user?.name || 'User';
    XLSX.writeFile(wb, `${userName}_JobCards_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Job cards exported successfully', 'success');
}

document.addEventListener('DOMContentLoaded', () => {
    fetchAllUsers();
    
    document.getElementById('apply-filter').addEventListener('click', () => {
        selectedRole = document.getElementById('role-filter').value;
        renderTable();
        showNotification(selectedRole ? `Filtered by role: ${selectedRole}` : 'Showing all roles', 'info');
    });
    
    document.getElementById('export-excel-btn').addEventListener('click', exportToExcel);
    document.getElementById('refresh-btn').addEventListener('click', () => {
        showNotification('Refreshing data...', 'info');
        fetchAllUsers();
    });
});
</script>

<style>
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Jost', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

.employees-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
}

.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.employees-kpi-content {
    padding: 30px 20px;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
}

.filter-section {
    margin-bottom: 30px;
}

.filter-form {
    display: flex;
    align-items: end;
    gap: 1rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
}

.form-group {
    flex: 1;
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
    font-size: 14px;
}

.form-select {
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 12px 16px;
    font-size: 14px;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
}

.form-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-go {
    padding: 12px 24px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    min-height: 44px;
}

.btn-go:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
}

.table-section {
    margin-bottom: 30px;
}

.table-responsive {
    overflow-x: auto;
    border-radius: var(--radius-lg);
    border: 1px solid var(--border-light);
}

.modern-table {
    width: 100%;
    border-collapse: collapse;
}

.modern-table thead {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
}

.modern-table th {
    color: white;
    font-weight: 600;
    padding: 16px 12px;
    text-align: center;
    font-size: 14px;
    border-right: 1px solid rgba(255, 255, 255, 0.2);
}

.modern-table th:last-child {
    border-right: none;
}

.modern-table tbody tr {
    border-bottom: 1px solid var(--border-light);
}

.modern-table tbody tr:nth-child(even) {
    background: #fafdff;
}

.modern-table tbody tr:nth-child(odd) {
    background: #eaf3fa;
}

.modern-table tbody tr:hover {
    background: #d0e7f7;
}

.modern-table td {
    padding: 14px 12px;
    text-align: center;
    font-size: 14px;
    color: var(--text-primary);
}

.name-cell {
    color: #2267A2;
    font-weight: 600;
}

.count-cell {
    color: #025383;
    font-weight: 700;
}

.role-badge {
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
    display: inline-block;
}

.role-supervisor {
    background: #e3f2fd;
    color: #1976d2;
}

.role-manager, .role-operations-manager, .role-project-manager {
    background: #f3e5f5;
    color: #7b1fa2;
}

.loading-row, .no-data {
    text-align: center;
    padding: 40px;
    color: var(--text-secondary);
    font-style: italic;
}

.summary-section {
    margin-top: 30px;
}

.summary-card {
    background: #f8f9fa;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 24px;
}

.summary-card h3 {
    color: #025383;
    font-size: 20px;
    font-weight: 700;
    margin: 0 0 16px 0;
}

.summary-stats {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.summary-item {
    display: flex;
    justify-content: space-between;
    font-size: 16px;
}

.summary-label {
    color: #015185;
    font-weight: 600;
}

.summary-value {
    color: #2267A2;
    font-weight: 700;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 10000;
    padding: 16px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    font-family: var(--font-family);
    font-weight: 500;
    max-width: 400px;
    min-width: 300px;
    animation: slideIn 0.3s ease;
    background: white;
}

.notification-success {
    border-left: 4px solid #28a745;
}

.notification-error {
    border-left: 4px solid #dc3545;
}

.notification-info {
    border-left: 4px solid #17a2b8;
}

.notification-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.notification-close {
    background: none;
    border: none;
    font-size: 18px;
    color: #999;
    cursor: pointer;
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

.clickable-row {
    cursor: pointer;
    transition: var(--transition-base);
}

.clickable-row:hover {
    transform: scale(1.01);
    box-shadow: 0 2px 8px rgba(0, 85, 165, 0.15);
    background: #d0e7f7 !important;
}

.status-badge {
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    background: #e3f2fd;
    color: #1976d2;
}

.user-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10002;
    overflow-y: auto;
}

.user-modal-content {
    background: white;
    border-radius: var(--radius-lg);
    width: 90%;
    max-width: 900px;
    max-height: 85vh;
    overflow: hidden;
    box-shadow: var(--shadow-lg);
    margin: 20px auto;
}

.user-modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 30px;
    border-bottom: 1px solid var(--border-light);
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
}

.user-modal-header h2 {
    color: white;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
}

.user-modal-close {
    background: none;
    border: none;
    color: white;
    font-size: 28px;
    cursor: pointer;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: var(--transition-base);
}

.user-modal-close:hover {
    background: rgba(255, 255, 255, 0.2);
}

.user-modal-body {
    padding: 30px;
    overflow-y: auto;
    max-height: calc(85vh - 80px);
}

.user-info-section {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
    padding: 20px;
    background: #f8fafc;
    border-radius: var(--radius-lg);
    border: 1px solid var(--border-light);
}

.user-info-item {
    font-size: 15px;
    color: var(--text-primary);
}

.user-info-item strong {
    color: var(--primary-color);
    font-weight: 600;
    display: inline-block;
    min-width: 140px;
}

.user-tabs {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
    border-bottom: 2px solid var(--border-light);
}

.user-tab-btn {
    padding: 12px 24px;
    background: transparent;
    border: none;
    border-bottom: 3px solid transparent;
    color: var(--text-secondary);
    font-size: 15px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 8px;
}

.user-tab-btn.active {
    color: var(--primary-color);
    border-bottom-color: var(--primary-color);
}

.user-tab-btn:hover:not(.active) {
    color: var(--primary-hover);
    background: #f8fafc;
}

.user-tab-content {
    display: none;
}

.user-tab-content.active {
    display: block;
}

.tab-actions {
    display: flex;
    justify-content: flex-end;
    margin-bottom: 15px;
}

.btn-export-user {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.btn-export-user:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .btn-go {
        width: 100%;
    }
    
    .modern-table {
        font-size: 12px;
    }
    
    .modern-table th,
    .modern-table td {
        padding: 10px 8px;
    }
    
    .user-modal-content {
        width: 95%;
        max-width: 95%;
    }
    
    .user-modal-body {
        padding: 15px;
    }
    
    .user-info-section {
        grid-template-columns: 1fr;
    }
    
    .user-tabs {
        flex-direction: column;
        border-bottom: none;
    }
    
    .user-tab-btn {
        border-bottom: none;
        border-left: 3px solid transparent;
        text-align: left;
    }
    
    .user-tab-btn.active {
        border-left-color: var(--primary-color);
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        gap: 15px;
        align-items: flex-start;
    }
    
    .header-right {
        width: 100%;
        flex-direction: column;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
}
</style>
@endsection

