@extends('welcome')

@section('title', 'Supervisor KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="supervisor-kpi-container">
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Supervisor Data...</div>
        </div>
    </div>
    
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('dashboard.team_kpi.home') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Supervisor KPI</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn">
                    <i class="fas fa-file-excel"></i>
                    Export
                </button>
                <button type="button" class="create-btn" id="refresh-btn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="supervisor-kpi-content">
        <div class="content-box">
            <!-- Mode Selector -->
            <div class="mode-selector-bar">
                <button class="mode-btn active" data-mode="role" id="role-kpi-btn">
                    <i class="fas fa-user-tag"></i>
                    <span>Role KPI</span>
                </button>
                <button class="mode-btn" data-mode="module" id="module-kpi-btn">
                    <i class="fas fa-th-large"></i>
                    <span>Module KPI</span>
                </button>
            </div>
            
            <!-- Role KPI Mode -->
            <div id="role-kpi-content" class="mode-content active">
                <div class="filter-section">
                    <div class="filter-form">
                        <div class="form-group">
                            <label for="role-filter">Filter by Role:</label>
                            <select class="form-select" id="role-filter">
                                <option value="">All Roles</option>
                            </select>
                        </div>
                        <button class="btn-go" id="apply-filter">Apply</button>
                    </div>
                </div>
                
                <div class="table-section">
                    <div class="table-responsive">
                        <table class="modern-table">
                            <thead>
                                <tr>
                                    <th>S.No</th>
                                    <th>Employee Number</th>
                                    <th>Role</th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Inspections</th>
                                    <th>Job Cards</th>
                                </tr>
                            </thead>
                            <tbody id="role-table-body">
                                <tr><td colspan="7" class="loading-row">Loading...</td></tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div class="summary-section">
                    <h3>Summary</h3>
                    <div class="summary-stats">
                        <div class="summary-item">
                            <span class="summary-label">Total Inspections:</span>
                            <span class="summary-value" id="role-total-inspections">0</span>
                        </div>
                        <div class="summary-item">
                            <span class="summary-label">Total Job Cards Created:</span>
                            <span class="summary-value" id="role-total-jobcards">0</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Module KPI Mode -->
            <div id="module-kpi-content" class="mode-content">
                <!-- Module Tabs -->
                <div class="module-tabs">
                    <button class="module-tab-btn active" data-module="inspections">
                        <i class="fas fa-clipboard-check"></i>
                        <span>Total Inspections</span>
                    </button>
                    <button class="module-tab-btn" data-module="jobCards">
                        <i class="fas fa-briefcase"></i>
                        <span>Job Cards</span>
                    </button>
                    <button class="module-tab-btn" data-module="maintenance">
                        <i class="fas fa-wrench"></i>
                        <span>Maintenance</span>
                    </button>
                    <button class="module-tab-btn" data-module="serviceOrders">
                        <i class="fas fa-calendar-alt"></i>
                        <span>Service Orders</span>
                    </button>
                </div>
                
                <!-- Date Range Picker -->
                <div class="date-picker-container">
                    <div class="date-inputs">
                        <div class="date-input-group">
                            <label>From Date:</label>
                            <input type="date" id="from-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label>To Date:</label>
                            <input type="date" id="to-date" class="date-input">
                        </div>
                        <button class="btn-clear-dates" id="clear-dates">Clear</button>
                    </div>
                </div>
                
                <!-- Module Summary -->
                <div class="module-summary" id="module-summary">
                    <div class="summary-loading">
                        <i class="fas fa-spinner fa-spin"></i>
                        <span>Loading data...</span>
                    </div>
                </div>
                
                <!-- Charts -->
                <div class="charts-container">
                    <!-- Trend Chart -->
                    <div class="chart-card">
                        <div class="chart-header">
                            <h3 id="chart-title">Daily Trend</h3>
                            <p class="chart-subtitle" id="date-range-text"></p>
                        </div>
                        <div class="chart-body">
                            <canvas id="trend-chart" height="250"></canvas>
                        </div>
                    </div>
                    
                    <!-- Concentric Progress Chart -->
                    <div class="chart-card">
                        <div class="chart-header">
                            <h3>Performance Overview</h3>
                            <p class="chart-subtitle">Progress indicators for Reports, Equipment, and Users</p>
                        </div>
                        <div class="chart-body">
                            <div class="concentric-chart-container">
                                <div class="concentric-chart">
                                    <svg id="concentric-svg" width="200" height="200"></svg>
                                    <div class="center-label">
                                        <div class="center-value" id="center-value">0</div>
                                        <div class="center-text" id="center-text">Total</div>
                                    </div>
                                </div>
                                <div class="concentric-legend" id="concentric-legend"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- User Details Modal -->
<div id="user-details-modal" class="user-modal" style="display: none;" onclick="if(event.target === this) closeUserModal()">
    <div class="user-modal-content">
        <div class="user-modal-header">
            <h2 id="modal-user-name">User Details</h2>
            <button class="user-modal-close" onclick="closeUserModal()">&times;</button>
        </div>
        <div class="user-modal-body">
            <div class="user-info-section">
                <div class="user-info-item">
                    <strong>Email:</strong> <span id="modal-user-email">-</span>
                </div>
                <div class="user-info-item">
                    <strong>Role:</strong> <span id="modal-user-role">-</span>
                </div>
                <div class="user-info-item">
                    <strong>Employee Number:</strong> <span id="modal-user-empnum">-</span>
                </div>
            </div>
            
            <div class="user-tabs">
                <button class="user-tab-btn active" onclick="switchUserTab('inspections')">
                    <i class="fas fa-clipboard-check"></i>
                    Inspections (<span id="user-inspections-count">0</span>)
                </button>
                <button class="user-tab-btn" onclick="switchUserTab('jobcards')">
                    <i class="fas fa-briefcase"></i>
                    Job Cards (<span id="user-jobcards-count">0</span>)
                </button>
            </div>
            
            <div id="user-inspections-tab" class="user-tab-content active">
                <div class="tab-actions">
                    <button class="btn-export-user" onclick="exportUserInspections()">
                        <i class="fas fa-file-excel"></i> Export Inspections
                    </button>
                </div>
                <div class="table-responsive">
                    <table class="modern-table">
                        <thead>
                            <tr>
                                <th>S.No</th>
                                <th>Report ID</th>
                                <th>Equipment</th>
                                <th>Status</th>
                                <th>Created Date</th>
                            </tr>
                        </thead>
                        <tbody id="user-inspections-body">
                            <tr><td colspan="5" class="loading-row">Loading...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <div id="user-jobcards-tab" class="user-tab-content">
                <div class="tab-actions">
                    <button class="btn-export-user" onclick="exportUserJobCards()">
                        <i class="fas fa-file-excel"></i> Export Job Cards
                    </button>
                </div>
                <div class="table-responsive">
                    <table class="modern-table">
                        <thead>
                            <tr>
                                <th>S.No</th>
                                <th>Work Order ID</th>
                                <th>Equipment</th>
                                <th>Status</th>
                                <th>Created Date</th>
                            </tr>
                        </thead>
                        <tbody id="user-jobcards-body">
                            <tr><td colspan="5" class="loading-row">Loading...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
const firebaseConfig = { databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com" };
if (!firebase.apps.length) firebase.initializeApp(firebaseConfig);
const database = firebase.database();

const ALL_ROLES = ['Operations Manager', 'Health & Safety Manager', 'Project Manager', 'Site Manager', 'Supervisor', 'HrManager', 'FinanceManager', 'Driller', 'Trainee Driller', 'Junior Driller', 'Coremarker', 'Assistant', 'Workshop Manager', 'Workshop Foreman', 'Maintenance Manager', 'Mechanical Engineer', 'Boilermaker', 'Welder', 'Mechanic', 'Machinist', 'Electrician', 'Housekeeper', 'Cook', 'Stores Officer', 'General Worker'];

let allUsers = [];
let filteredUsers = [];
let userStats = {};
let selectedRole = '';
let currentMode = 'role';
let currentModule = 'inspections';
let dateRange = { from: null, to: null };
let moduleData = { inspections: [], jobCards: [], maintenance: [], serviceOrders: [] };
let equipmentCounts = { inspections: 0, jobCards: 0, maintenance: 0, serviceOrders: 0 };
let userDetails = { inspections: [], jobCards: [], maintenance: [], serviceOrders: [] };
let trendChart = null;
let currentUserData = { user: null, inspections: [], jobCards: [] };

function showLoadingState(isLoading) {
    document.getElementById('loading-overlay').style.display = isLoading ? 'flex' : 'none';
}

// Fetch all users
async function fetchAllUsers() {
    showLoadingState(true);
    try {
        const adminsRef = database.ref('/Globalusers/admins');
        const snapshot = await adminsRef.once('value');
        const data = snapshot.val();
        
        if (data) {
            allUsers = Object.keys(data).map(id => ({ id, ...data[id] }));
            filteredUsers = allUsers;
            populateRoleFilter();
            await fetchUserStats();
            renderRoleTable();
        }
        
        showLoadingState(false);
        showNotification(`Loaded ${allUsers.length} users`, 'success');
    } catch (error) {
        console.error('Error:', error);
        showNotification('Error loading data', 'error');
        showLoadingState(false);
    }
}

// Fetch stats for each user (Role KPI)
async function fetchUserStats() {
    const promises = filteredUsers.map(async (user) => {
        const userId = user.id;
        let inspections = 0, jobCards = 0;
        
        try {
            const inspSnap = await database.ref(`/GlobalInspectionReport/admins/${userId}`).once('value');
            inspections = inspSnap.val() ? Object.keys(inspSnap.val()).length : 0;
        } catch (e) {}
        
        try {
            const wcSnap = await database.ref(`/GlobalWorkOrders/admins/${userId}`).once('value');
            const wcData = wcSnap.val();
            jobCards = wcData ? Object.keys(wcData).filter(k => k !== 'lastRequestNumber').length : 0;
        } catch (e) {}
        
        return { userId, inspections, jobCards };
    });
    
    const results = await Promise.all(promises);
    results.forEach(({ userId, inspections, jobCards }) => {
        userStats[userId] = { inspections, jobCards };
    });
}

// Populate role filter
function populateRoleFilter() {
    const roleCounts = {};
    const roleMap = new Map(); // To track case-insensitive duplicates
    
    allUsers.forEach(user => {
        const role = user.role || 'Unknown';
        const roleLower = role.toLowerCase().trim();
        
        // Find if we already have this role (case-insensitive)
        let existingRole = role;
        for (let [key, value] of roleMap.entries()) {
            if (key.toLowerCase().trim() === roleLower) {
                existingRole = value;
                break;
            }
        }
        
        if (!roleMap.has(roleLower)) {
            roleMap.set(roleLower, role);
        }
        
        roleCounts[existingRole] = (roleCounts[existingRole] || 0) + 1;
    });
    
    const roleFilter = document.getElementById('role-filter');
    roleFilter.innerHTML = '<option value="">All Roles</option>';
    
    // Sort and display unique roles
    Object.keys(roleCounts).sort().forEach(role => {
        if (roleCounts[role] > 0) {
            const option = document.createElement('option');
            option.value = role;
            option.textContent = `${role} (${roleCounts[role]})`;
            roleFilter.appendChild(option);
        }
    });
}

// Render Role KPI table
function renderRoleTable() {
    const displayedUsers = selectedRole 
        ? filteredUsers.filter(u => (u.role || '') === selectedRole)
        : filteredUsers;
    
    const tbody = document.getElementById('role-table-body');
    
    if (displayedUsers.length === 0) {
        tbody.innerHTML = '<tr><td colspan="7" class="no-data">No users found</td></tr>';
        updateRoleSummary(displayedUsers);
        return;
    }
    
    tbody.innerHTML = displayedUsers.map((user, index) => {
        const role = user.role || 'Employee';
        const roleClass = role.toLowerCase().replace(/\s+/g, '-');
        
        return `
            <tr class="clickable-row" onclick="showUserDetails('${user.id}', event)">
                <td>${index + 1}</td>
                <td>${user.employeeNumber || '-'}</td>
                <td><span class="role-badge role-${roleClass}">${role}</span></td>
                <td class="name-cell">${user.fullName || user.name || 'Unknown'}</td>
                <td>${user.email || '-'}</td>
                <td class="count-cell">${userStats[user.id]?.inspections || 0}</td>
                <td class="count-cell">${userStats[user.id]?.jobCards || 0}</td>
            </tr>
        `;
    }).join('');
    
    updateRoleSummary(displayedUsers);
}

// Update Role summary
function updateRoleSummary(users) {
    const totalInspections = users.reduce((sum, u) => sum + (userStats[u.id]?.inspections || 0), 0);
    const totalJobCards = users.reduce((sum, u) => sum + (userStats[u.id]?.jobCards || 0), 0);
    
    document.getElementById('role-total-inspections').textContent = totalInspections;
    document.getElementById('role-total-jobcards').textContent = totalJobCards;
}

// Fetch Module data
async function fetchModuleData() {
    if (!dateRange.from || !dateRange.to) {
        showNotification('Please select date range', 'error');
        return;
    }
    
    showLoadingState(true);
    
    try {
        if (currentModule === 'inspections') {
            await fetchInspectionsModule();
        } else if (currentModule === 'jobCards') {
            await fetchJobCardsModule();
        } else if (currentModule === 'maintenance') {
            await fetchMaintenanceModule();
        } else if (currentModule === 'serviceOrders') {
            await fetchServiceOrdersModule();
        }
        
        updateModuleSummary();
        updateTrendChart();
        updateConcentricChart();
        showLoadingState(false);
    } catch (error) {
        console.error('Error:', error);
        showNotification('Error loading module data', 'error');
        showLoadingState(false);
    }
}

// Fetch Inspections from GlobalInspectionReport/admins
async function fetchInspectionsModule() {
    const snapshot = await database.ref('/GlobalInspectionReport/admins').once('value');
    const adminsData = snapshot.val() || {};
    const reports = [];
    const equipmentSet = new Set();
    const userCounts = {};
    
    Object.entries(adminsData).forEach(([userId, reportsObj]) => {
        if (reportsObj && typeof reportsObj === 'object') {
            Object.values(reportsObj).forEach(report => {
                const createdAt = report?.step3?.createdAt || report?.step1?.createdAt;
                if (createdAt) {
                    const reportDate = new Date(createdAt);
                    if (reportDate >= dateRange.from && reportDate <= dateRange.to) {
                        reports.push({ date: reportDate, userId });
                        const eqNum = report?.step1?.selectedEquipment?.equipmentNumber;
                        if (eqNum) equipmentSet.add(eqNum);
                        userCounts[userId] = (userCounts[userId] || 0) + 1;
                    }
                }
            });
        }
    });
    
    moduleData.inspections = reports;
    equipmentCounts.inspections = equipmentSet.size;
    
    const usersSnap = await database.ref('/Globalusers/admins').once('value');
    const allUsersData = usersSnap.val() || {};
    userDetails.inspections = Object.keys(userCounts).map(uid => ({
        id: uid,
        name: allUsersData[uid]?.fullName || allUsersData[uid]?.name || 'Unknown',
        role: allUsersData[uid]?.role || 'Unknown',
        count: userCounts[uid]
    }));
}

// Fetch Job Cards from GlobalWorkOrders/admins
async function fetchJobCardsModule() {
    const snapshot = await database.ref('/GlobalWorkOrders/admins').once('value');
    const adminsData = snapshot.val() || {};
    const jobCardsArr = [];
    const equipmentSet = new Set();
    const userCounts = {};
    
    Object.entries(adminsData).forEach(([userId, cardsObj]) => {
        if (cardsObj && typeof cardsObj === 'object') {
            Object.entries(cardsObj).forEach(([key, card]) => {
                if (key === 'lastRequestNumber') return;
                const createdAt = card?.createdAt;
                if (createdAt) {
                    const cardDate = new Date(createdAt);
                    if (cardDate >= dateRange.from && cardDate <= dateRange.to) {
                        jobCardsArr.push({ date: cardDate, userId });
                        const eqNum = card?.equipment?.equipmentNumber;
                        if (eqNum) equipmentSet.add(eqNum);
                        userCounts[userId] = (userCounts[userId] || 0) + 1;
                    }
                }
            });
        }
    });
    
    moduleData.jobCards = jobCardsArr;
    equipmentCounts.jobCards = equipmentSet.size;
    
    const usersSnap = await database.ref('/Globalusers/admins').once('value');
    const allUsersData = usersSnap.val() || {};
    userDetails.jobCards = Object.keys(userCounts).map(uid => ({
        id: uid,
        name: allUsersData[uid]?.fullName || allUsersData[uid]?.name || 'Unknown',
        role: allUsersData[uid]?.role || 'Unknown',
        count: userCounts[uid]
    }));
}

// Fetch Maintenance from GlobalRequestMaintenancereport (recursive)
async function fetchMaintenanceModule() {
    const snapshot = await database.ref('/GlobalRequestMaintenancereport').once('value');
    const data = snapshot.val() || {};
    const reportsArr = [];
    const equipmentSet = new Set();
    const userCounts = {};
    
    // Recursive extraction
    function extractReports(obj, uid = '', path = []) {
        if (!obj || typeof obj !== 'object') return;
        
        Object.keys(obj).forEach(key => {
            const item = obj[key];
            if (typeof item !== 'object' || item === null) return;
            
            const hasFields = item.title || item.scheduleId || item.createdAt || item.country || item.equipment;
            if (hasFields) {
                const createdAt = item.createdAt || item.dateCreated;
                if (createdAt) {
                    const reportDate = new Date(createdAt);
                    if (reportDate >= dateRange.from && reportDate <= dateRange.to) {
                        const userId = uid || path[0] || 'unknown';
                        reportsArr.push({ date: reportDate, userId });
                        const eqNum = item?.equipment?.equipmentNumber || item?.equipmentNumber;
                        if (eqNum) equipmentSet.add(eqNum);
                        userCounts[userId] = (userCounts[userId] || 0) + 1;
                    }
                }
            } else {
                extractReports(item, uid || key, [...path, key]);
            }
        });
    }
    
    extractReports(data);
    
    moduleData.maintenance = reportsArr;
    equipmentCounts.maintenance = equipmentSet.size;
    
    const usersSnap = await database.ref('/Globalusers/admins').once('value');
    const allUsersData = usersSnap.val() || {};
    userDetails.maintenance = Object.keys(userCounts).map(uid => ({
        id: uid,
        name: allUsersData[uid]?.fullName || allUsersData[uid]?.name || 'Unknown',
        role: allUsersData[uid]?.role || 'Unknown',
        count: userCounts[uid]
    }));
}

// Fetch Service Orders from GlobalServiceScheduled (recursive)
async function fetchServiceOrdersModule() {
    const snapshot = await database.ref('/GlobalServiceScheduled').once('value');
    const data = snapshot.val() || {};
    const ordersArr = [];
    const equipmentSet = new Set();
    const userCounts = {};
    
    // Recursive extraction
    function extractOrders(obj, uid = '', path = []) {
        if (!obj || typeof obj !== 'object') return;
        
        Object.keys(obj).forEach(key => {
            const item = obj[key];
            if (typeof item !== 'object' || item === null) return;
            
            const hasFields = item.title || item.scheduleId || item.scheduledDate || item.serviceType || item.equipment;
            if (hasFields) {
                const createdAt = item.createdAt || item.scheduledDate;
                if (createdAt) {
                    const orderDate = new Date(createdAt);
                    if (orderDate >= dateRange.from && orderDate <= dateRange.to) {
                        const userId = uid || path[0] || 'unknown';
                        ordersArr.push({ date: orderDate, userId });
                        const eqName = item?.equipment?.equipmentName || item?.equipmentName;
                        if (eqName) equipmentSet.add(eqName);
                        userCounts[userId] = (userCounts[userId] || 0) + 1;
                    }
                }
            } else {
                extractOrders(item, uid || key, [...path, key]);
            }
        });
    }
    
    extractOrders(data);
    
    moduleData.serviceOrders = ordersArr;
    equipmentCounts.serviceOrders = equipmentSet.size;
    
    const usersSnap = await database.ref('/Globalusers/admins').once('value');
    const allUsersData = usersSnap.val() || {};
    userDetails.serviceOrders = Object.keys(userCounts).map(uid => ({
        id: uid,
        name: allUsersData[uid]?.fullName || allUsersData[uid]?.name || 'Unknown',
        role: allUsersData[uid]?.role || 'Unknown',
        count: userCounts[uid]
    }));
}

// Update module summary
function updateModuleSummary() {
    const summaryDiv = document.getElementById('module-summary');
    const currentData = moduleData[currentModule] || [];
    const equipmentCount = equipmentCounts[currentModule] || 0;
    const users = userDetails[currentModule] || [];
    
    const moduleTitles = {
        inspections: 'Inspection Reports',
        jobCards: 'Job Cards',
        maintenance: 'Maintenance Reports',
        serviceOrders: 'Service Orders'
    };
    
    const uniqueRoles = [...new Set(users.map(u => u.role))].join(', ') || 'None';
    
    summaryDiv.innerHTML = `
        <div class="summary-item"><strong>Total ${moduleTitles[currentModule]}:</strong> <span class="highlight">${currentData.length}</span></div>
        <div class="summary-item"><strong>Total Equipment Used:</strong> <span class="highlight">${equipmentCount}</span></div>
        <div class="summary-item"><strong>Total Users:</strong> <span class="highlight">${users.length}</span> (${uniqueRoles})</div>
        <div class="summary-item"><strong>Users who created items in selected range:</strong> <span class="highlight">${users.length}</span></div>
        <div class="users-list">
            ${users.map((user, idx) => `
                <div class="user-item">${idx + 1}. ${user.name} (${user.role}): ${user.count} items</div>
            `).join('')}
        </div>
    `;
}

// Update trend chart
function updateTrendChart() {
    if (!dateRange.from || !dateRange.to) return;
    
    const currentData = moduleData[currentModule] || [];
    const dates = getDateArray(dateRange.from, dateRange.to);
    const counts = {};
    
    dates.forEach(date => {
        const dateStr = date.toLocaleDateString('en-GB');
        counts[dateStr] = 0;
    });
    
    currentData.forEach(item => {
        const dateStr = new Date(item.date).toLocaleDateString('en-GB');
        if (counts.hasOwnProperty(dateStr)) {
            counts[dateStr]++;
        }
    });
    
    const labels = Object.keys(counts);
    const data = Object.values(counts);
    
    const ctx = document.getElementById('trend-chart');
    if (trendChart) {
        trendChart.destroy();
    }
    
    trendChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Count',
                data: data,
                backgroundColor: '#A18AFF',
                borderColor: '#8000FF',
                borderWidth: 2
            }, {
                type: 'line',
                label: 'Trend',
                data: data,
                borderColor: '#8000FF',
                backgroundColor: 'transparent',
                borderWidth: 2,
                pointBackgroundColor: '#A18AFF',
                pointRadius: 5,
                fill: false,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            legend: { display: true },
            scales: {
                yAxes: [{ ticks: { beginAtZero: true } }]
            }
        }
    });
    
    document.getElementById('chart-title').textContent = `${getModuleTitle(currentModule)} Daily Trend`;
    document.getElementById('date-range-text').textContent = `${dateRange.from.toLocaleDateString('en-GB')} - ${dateRange.to.toLocaleDateString('en-GB')}`;
}

// Update concentric progress chart
function updateConcentricChart() {
    const currentData = moduleData[currentModule] || [];
    const equipmentCount = equipmentCounts[currentModule] || 0;
    const usersCount = userDetails[currentModule]?.length || 0;
    
    const maxValue = Math.max(currentData.length, equipmentCount, usersCount, 1);
    
    const chartData = [
        { count: currentData.length, percentage: (currentData.length / maxValue) * 100, color: '#2267A2', text: 'Reports' },
        { count: equipmentCount, percentage: (equipmentCount / maxValue) * 100, color: '#D81B60', text: 'Equipment' },
        { count: usersCount, percentage: (usersCount / maxValue) * 100, color: '#2ECC40', text: 'Users' }
    ];
    
    drawConcentricChart(chartData);
    
    const totalValue = currentData.length + equipmentCount + usersCount;
    document.getElementById('center-value').textContent = totalValue;
    document.getElementById('center-text').textContent = 'Total';
    
    const legendDiv = document.getElementById('concentric-legend');
    legendDiv.innerHTML = chartData.map(item => `
        <div class="legend-item">
            <div class="legend-color" style="background: ${item.color}"></div>
            <span class="legend-text">${item.text}: ${item.count}</span>
        </div>
    `).join('');
}

// Draw concentric chart
function drawConcentricChart(data) {
    const svg = document.getElementById('concentric-svg');
    svg.innerHTML = '';
    
    const size = 200;
    const center = size / 2;
    const baseRadius = 80;
    const strokeWidth = 12;
    
    data.forEach((item, index) => {
        const radius = baseRadius - (index * (strokeWidth + 4));
        const circumference = 2 * Math.PI * radius;
        const offset = circumference - (item.percentage / 100) * circumference;
        
        // Background circle
        const bgCircle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
        bgCircle.setAttribute('cx', center);
        bgCircle.setAttribute('cy', center);
        bgCircle.setAttribute('r', radius);
        bgCircle.setAttribute('fill', 'transparent');
        bgCircle.setAttribute('stroke', '#E0E0E0');
        bgCircle.setAttribute('stroke-width', strokeWidth);
        svg.appendChild(bgCircle);
        
        // Progress circle
        const progressCircle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
        progressCircle.setAttribute('cx', center);
        progressCircle.setAttribute('cy', center);
        progressCircle.setAttribute('r', radius);
        progressCircle.setAttribute('fill', 'transparent');
        progressCircle.setAttribute('stroke', item.color);
        progressCircle.setAttribute('stroke-width', strokeWidth);
        progressCircle.setAttribute('stroke-dasharray', circumference);
        progressCircle.setAttribute('stroke-dashoffset', offset);
        progressCircle.setAttribute('stroke-linecap', 'round');
        progressCircle.setAttribute('transform', `rotate(-90 ${center} ${center})`);
        svg.appendChild(progressCircle);
    });
}

// Helper functions
function getDateArray(start, end) {
    const arr = [];
    let dt = new Date(start);
    while (dt <= end) {
        arr.push(new Date(dt));
        dt.setDate(dt.getDate() + 1);
    }
    return arr;
}

function getModuleTitle(module) {
    const titles = {
        inspections: 'Inspections',
        jobCards: 'Job Cards',
        maintenance: 'Maintenance',
        serviceOrders: 'Service Orders'
    };
    return titles[module] || '';
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `<div class="notification-content"><span>${message}</span><button class="notification-close">&times;</button></div>`;
    document.body.appendChild(notification);
    setTimeout(() => notification.remove(), 5000);
    notification.querySelector('.notification-close').addEventListener('click', () => notification.remove());
}

// Show user details modal
async function showUserDetails(userId, event) {
    const user = allUsers.find(u => u.id === userId);
    if (!user) return;
    
    currentUserData.user = user;
    
    // Update modal header
    document.getElementById('modal-user-name').textContent = user.fullName || user.name || 'Unknown';
    document.getElementById('modal-user-email').textContent = user.email || '-';
    document.getElementById('modal-user-role').textContent = user.role || '-';
    document.getElementById('modal-user-empnum').textContent = user.employeeNumber || '-';
    
    // Show modal
    const modal = document.getElementById('user-details-modal');
    modal.style.display = 'flex';
    
    // Position modal near the clicked row
    if (event) {
        const clickY = event.clientY;
        const windowHeight = window.innerHeight;
        const scrollY = window.scrollY || window.pageYOffset;
        
        // Calculate top position - position near the row clicked
        let topPosition = clickY + scrollY - 100; // Offset slightly above click
        
        // Ensure modal doesn't go off-screen (leave 20px padding at top)
        topPosition = Math.max(scrollY + 20, topPosition);
        
        // Also ensure it doesn't extend below viewport
        const modalHeight = 600; // Approximate modal height
        const maxTop = scrollY + windowHeight - modalHeight - 20;
        if (topPosition > maxTop && maxTop > scrollY + 20) {
            topPosition = maxTop;
        }
        
        // Apply positioning
        const modalContent = modal.querySelector('.user-modal-content');
        modalContent.style.position = 'absolute';
        modalContent.style.top = topPosition + 'px';
        modalContent.style.left = '50%';
        modalContent.style.transform = 'translateX(-50%)';
    }
    
    // Fetch user data
    await fetchUserDetailsData(userId);
}

// Fetch user inspections and job cards
async function fetchUserDetailsData(userId) {
    showLoadingState(true);
    
    try {
        // Fetch Inspections
        const inspSnap = await database.ref(`/GlobalInspectionReport/admins/${userId}`).once('value');
        const inspData = inspSnap.val();
        currentUserData.inspections = [];
        
        if (inspData) {
            Object.keys(inspData).forEach(key => {
                const report = inspData[key];
                currentUserData.inspections.push({
                    id: key,
                    reportId: report?.step1?.inspectionNumber || report?.reportId || key,
                    equipment: report?.step1?.selectedEquipment?.equipmentNumber || '-',
                    status: report?.step3?.status || 'Completed',
                    createdAt: report?.step3?.createdAt || report?.step1?.createdAt || '-'
                });
            });
        }
        
        // Fetch Job Cards
        const wcSnap = await database.ref(`/GlobalWorkOrders/admins/${userId}`).once('value');
        const wcData = wcSnap.val();
        currentUserData.jobCards = [];
        
        if (wcData) {
            Object.keys(wcData).forEach(key => {
                if (key === 'lastRequestNumber') return;
                const card = wcData[key];
                currentUserData.jobCards.push({
                    id: key,
                    workOrderId: card?.workOrderNumber || card?.workOrderId || key,
                    equipment: card?.equipment?.equipmentNumber || '-',
                    status: card?.status || 'Pending',
                    createdAt: card?.createdAt || '-'
                });
            });
        }
        
        // Update counts
        document.getElementById('user-inspections-count').textContent = currentUserData.inspections.length;
        document.getElementById('user-jobcards-count').textContent = currentUserData.jobCards.length;
        
        // Render tables
        renderUserInspections();
        renderUserJobCards();
        
        showLoadingState(false);
    } catch (error) {
        console.error('Error fetching user details:', error);
        showNotification('Error loading user details', 'error');
        showLoadingState(false);
    }
}

// Render user inspections table
function renderUserInspections() {
    const tbody = document.getElementById('user-inspections-body');
    
    if (currentUserData.inspections.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="no-data">No inspections found</td></tr>';
        return;
    }
    
    tbody.innerHTML = currentUserData.inspections.map((insp, index) => `
        <tr>
            <td>${index + 1}</td>
            <td>${insp.reportId}</td>
            <td>${insp.equipment}</td>
            <td><span class="status-badge">${insp.status}</span></td>
            <td>${insp.createdAt ? new Date(insp.createdAt).toLocaleString('en-GB') : '-'}</td>
        </tr>
    `).join('');
}

// Render user job cards table
function renderUserJobCards() {
    const tbody = document.getElementById('user-jobcards-body');
    
    if (currentUserData.jobCards.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="no-data">No job cards found</td></tr>';
        return;
    }
    
    tbody.innerHTML = currentUserData.jobCards.map((card, index) => `
        <tr>
            <td>${index + 1}</td>
            <td>${card.workOrderId}</td>
            <td>${card.equipment}</td>
            <td><span class="status-badge">${card.status}</span></td>
            <td>${card.createdAt ? new Date(card.createdAt).toLocaleString('en-GB') : '-'}</td>
        </tr>
    `).join('');
}

// Close user modal
function closeUserModal() {
    const modal = document.getElementById('user-details-modal');
    modal.style.display = 'none';
    
    // Reset positioning
    const modalContent = modal.querySelector('.user-modal-content');
    modalContent.style.position = '';
    modalContent.style.top = '';
    modalContent.style.left = '';
    modalContent.style.transform = '';
    
    currentUserData = { user: null, inspections: [], jobCards: [] };
}

// Switch user tab
function switchUserTab(tab) {
    document.querySelectorAll('.user-tab-btn').forEach(btn => btn.classList.remove('active'));
    document.querySelectorAll('.user-tab-content').forEach(content => content.classList.remove('active'));
    
    if (tab === 'inspections') {
        document.querySelector('.user-tab-btn:nth-child(1)').classList.add('active');
        document.getElementById('user-inspections-tab').classList.add('active');
    } else {
        document.querySelector('.user-tab-btn:nth-child(2)').classList.add('active');
        document.getElementById('user-jobcards-tab').classList.add('active');
    }
}

// Export user inspections
function exportUserInspections() {
    if (currentUserData.inspections.length === 0) {
        showNotification('No inspections to export', 'error');
        return;
    }
    
    const data = currentUserData.inspections.map((insp, index) => ({
        'S.No': index + 1,
        'Report ID': insp.reportId,
        'Equipment': insp.equipment,
        'Status': insp.status,
        'Created Date': insp.createdAt ? new Date(insp.createdAt).toLocaleString('en-GB') : '-'
    }));
    
    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Inspections');
    
    const userName = currentUserData.user?.fullName || currentUserData.user?.name || 'User';
    XLSX.writeFile(wb, `${userName}_Inspections_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Inspections exported successfully', 'success');
}

// Export user job cards
function exportUserJobCards() {
    if (currentUserData.jobCards.length === 0) {
        showNotification('No job cards to export', 'error');
        return;
    }
    
    const data = currentUserData.jobCards.map((card, index) => ({
        'S.No': index + 1,
        'Work Order ID': card.workOrderId,
        'Equipment': card.equipment,
        'Status': card.status,
        'Created Date': card.createdAt ? new Date(card.createdAt).toLocaleString('en-GB') : '-'
    }));
    
    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Job Cards');
    
    const userName = currentUserData.user?.fullName || currentUserData.user?.name || 'User';
    XLSX.writeFile(wb, `${userName}_JobCards_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Job cards exported successfully', 'success');
}

// Export all data (main export button)
function exportAllData() {
    if (currentMode === 'role') {
        exportRoleKPIData();
    } else {
        exportModuleKPIData();
    }
}

// Export Role KPI data
function exportRoleKPIData() {
    const displayedUsers = selectedRole 
        ? filteredUsers.filter(u => (u.role || '') === selectedRole)
        : filteredUsers;
    
    if (displayedUsers.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const data = displayedUsers.map((user, index) => ({
        'S.No': index + 1,
        'Employee Number': user.employeeNumber || '-',
        'Role': user.role || 'Employee',
        'Name': user.fullName || user.name || 'Unknown',
        'Email': user.email || '-',
        'Inspections': userStats[user.id]?.inspections || 0,
        'Job Cards': userStats[user.id]?.jobCards || 0
    }));
    
    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Role KPI');
    
    XLSX.writeFile(wb, `Supervisor_Role_KPI_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Data exported successfully', 'success');
}

// Export Module KPI data
function exportModuleKPIData() {
    if (!dateRange.from || !dateRange.to) {
        showNotification('Please select date range first', 'error');
        return;
    }
    
    const currentData = moduleData[currentModule] || [];
    if (currentData.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const moduleTitles = {
        inspections: 'Inspections',
        jobCards: 'Job Cards',
        maintenance: 'Maintenance',
        serviceOrders: 'Service Orders'
    };
    
    const data = currentData.map((item, index) => ({
        'S.No': index + 1,
        'User ID': item.userId,
        'Date': new Date(item.date).toLocaleString('en-GB'),
        'Module': moduleTitles[currentModule]
    }));
    
    const ws = XLSX.utils.json_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, moduleTitles[currentModule]);
    
    // Add summary sheet
    const summary = [
        { 'Metric': 'Total Items', 'Value': currentData.length },
        { 'Metric': 'Total Equipment', 'Value': equipmentCounts[currentModule] || 0 },
        { 'Metric': 'Total Users', 'Value': userDetails[currentModule]?.length || 0 },
        { 'Metric': 'Date Range', 'Value': `${dateRange.from.toLocaleDateString('en-GB')} - ${dateRange.to.toLocaleDateString('en-GB')}` }
    ];
    const wsSummary = XLSX.utils.json_to_sheet(summary);
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    XLSX.writeFile(wb, `Supervisor_Module_KPI_${moduleTitles[currentModule]}_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Data exported successfully', 'success');
}

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    fetchAllUsers();
    
    // Set default date range (last 7 days)
    const today = new Date();
    const from = new Date(today);
    from.setDate(today.getDate() - 6);
    document.getElementById('from-date').value = from.toISOString().split('T')[0];
    document.getElementById('to-date').value = today.toISOString().split('T')[0];
    
    // Mode switching
    document.querySelectorAll('.mode-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const mode = this.getAttribute('data-mode');
            currentMode = mode;
            
            document.querySelectorAll('.mode-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            document.querySelectorAll('.mode-content').forEach(c => c.classList.remove('active'));
            document.getElementById(`${mode}-kpi-content`).classList.add('active');
            
            if (mode === 'module') {
                const from = new Date(document.getElementById('from-date').value);
                const to = new Date(document.getElementById('to-date').value);
                to.setHours(23, 59, 59, 999);
                dateRange = { from, to };
                fetchModuleData();
            }
        });
    });
    
    // Module tab switching
    document.querySelectorAll('.module-tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const module = this.getAttribute('data-module');
            currentModule = module;
            
            document.querySelectorAll('.module-tab-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            fetchModuleData();
        });
    });
    
    // Role filter
    document.getElementById('apply-filter').addEventListener('click', () => {
        selectedRole = document.getElementById('role-filter').value;
        renderRoleTable();
        showNotification(selectedRole ? `Filtered by: ${selectedRole}` : 'Showing all roles', 'info');
    });
    
    // Date range
    document.getElementById('from-date').addEventListener('change', function() {
        const fromDate = new Date(this.value);
        dateRange.from = fromDate;
        document.getElementById('to-date').min = this.value;
        
        if (dateRange.to && currentMode === 'module') {
            fetchModuleData();
        }
    });
    
    document.getElementById('to-date').addEventListener('change', function() {
        const toDate = new Date(this.value);
        toDate.setHours(23, 59, 59, 999);
        dateRange.to = toDate;
        
        if (dateRange.from && currentMode === 'module') {
            fetchModuleData();
        }
    });
    
    document.getElementById('clear-dates').addEventListener('click', () => {
        document.getElementById('from-date').value = '';
        document.getElementById('to-date').value = '';
        dateRange = { from: null, to: null };
        showNotification('Date range cleared', 'info');
    });
    
    // Refresh
    document.getElementById('refresh-btn').addEventListener('click', () => {
        showNotification('Refreshing data...', 'info');
        if (currentMode === 'role') {
            fetchAllUsers();
        } else {
            fetchModuleData();
        }
    });
    
    // Export button
    document.getElementById('export-excel-btn').addEventListener('click', exportAllData);
});
</script>

<style>
@import url('https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap');
:root{--primary-color:#0055a5;--primary-hover:#003f7d;--text-primary:#212529;--text-secondary:#6c757d;--bg-light:#f8fafc;--bg-white:#ffffff;--border-light:#e2e8f0;--shadow-sm:0 .125rem .25rem rgba(0,0,0,.075);--shadow-md:0 .5rem 1rem rgba(0,0,0,.15);--shadow-lg:0 1rem 3rem rgba(0,0,0,.175);--radius-md:.375rem;--radius-lg:.5rem;--transition-base:all .15s ease-in-out;--font-family:'Jost',-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif}
.supervisor-kpi-container{min-height:100vh;background:var(--bg-light);font-family:var(--font-family)}
.header-gradient{background:#0055a5;padding:20px 0;box-shadow:0 4px 12px rgba(0,85,165,.3)}
.header-content{width:100%;padding:0 20px;display:flex;align-items:center;justify-content:space-between}
.header-left{display:flex;align-items:center;gap:15px}
.back-button{color:white;font-size:20px;text-decoration:none;padding:8px;border-radius:50%;transition:var(--transition-base)}
.back-button:hover{background-color:rgba(255,255,255,.1)}
.header-title{color:white;font-size:28px;font-weight:700;margin:0}
.header-right{display:flex;gap:15px}
.create-btn{background:white;color:var(--primary-color);border:none;padding:12px 24px;border-radius:var(--radius-md);font-size:16px;font-weight:600;cursor:pointer;transition:var(--transition-base);box-shadow:var(--shadow-sm);display:inline-flex;align-items:center;gap:8px}
.create-btn:hover{background:rgba(255,255,255,.9);transform:translateY(-2px);box-shadow:var(--shadow-md)}
.supervisor-kpi-content{padding:30px 20px}
.content-box{background:white;padding:30px;border-radius:var(--radius-lg);box-shadow:var(--shadow-md);border:1px solid var(--border-light)}
.mode-selector-bar{display:flex;justify-content:center;background:#F0F4F8;border-radius:24px;padding:4px;margin-bottom:30px;max-width:400px;margin-left:auto;margin-right:auto}
.mode-btn{flex:1;padding:12px 24px;background:transparent;border:none;border-radius:20px;font-size:16px;font-weight:600;color:#015185;cursor:pointer;transition:var(--transition-base);display:flex;align-items:center;justify-content:center;gap:8px}
.mode-btn.active{background:#0055a5;color:white;box-shadow:0 2px 4px rgba(0,85,165,.3)}
.mode-btn:hover:not(.active){background:rgba(0,85,165,.1)}
.mode-content{display:none}
.mode-content.active{display:block}
.filter-section{margin-bottom:30px}
.filter-form{display:flex;align-items:end;gap:1rem;background:white;border:1px solid var(--border-light);border-radius:var(--radius-lg);padding:1.5rem;box-shadow:var(--shadow-sm)}
.form-group{flex:1;display:flex;flex-direction:column}
.form-group label{font-weight:500;color:var(--text-primary);margin-bottom:8px;font-size:14px}
.form-select{border:1px solid var(--border-light);border-radius:var(--radius-md);padding:12px 16px;font-size:14px;color:var(--text-primary);background:white;font-family:var(--font-family)}
.form-select:focus{outline:none;border-color:var(--primary-color);box-shadow:0 0 0 3px rgba(0,85,165,.1)}
.btn-go{padding:12px 24px;background:var(--primary-color);color:white;border:none;border-radius:var(--radius-md);font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base);min-height:44px}
.btn-go:hover{background:var(--primary-hover);transform:translateY(-2px)}
.module-tabs{display:flex;gap:10px;margin-bottom:20px;flex-wrap:wrap;justify-content:center}
.module-tab-btn{padding:10px 20px;background:#E0E0E0;border:none;border-radius:18px;color:#015185;font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base);display:flex;align-items:center;gap:8px}
.module-tab-btn.active{background:#015185;color:white;box-shadow:0 2px 4px rgba(0,85,165,.3)}
.module-tab-btn:hover:not(.active){background:#d0d0d0}
.date-picker-container{margin-bottom:20px;display:flex;justify-content:center}
.date-inputs{display:flex;gap:15px;align-items:end;flex-wrap:wrap;justify-content:center}
.date-input-group{display:flex;flex-direction:column;gap:8px}
.date-input-group label{font-size:14px;font-weight:500;color:var(--text-primary)}
.date-input{padding:12px 16px;border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-family:var(--font-family);min-width:180px}
.date-input:focus{outline:none;border-color:var(--primary-color);box-shadow:0 0 0 3px rgba(0,85,165,.1)}
.btn-clear-dates{padding:12px 24px;background:white;color:var(--text-primary);border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base)}
.btn-clear-dates:hover{background:var(--bg-light);border-color:var(--primary-color);color:var(--primary-color)}
.module-summary{background:#F5F7FA;border:1px solid #E0E0E0;border-radius:var(--radius-lg);padding:16px;margin-bottom:20px}
.module-summary .summary-item{color:#015185;font-weight:600;margin-bottom:8px;font-size:15px}
.module-summary .highlight{color:#2267A2;font-weight:700}
.module-summary .users-list{margin-top:12px;padding-left:20px}
.module-summary .user-item{color:#444;font-size:13px;margin-bottom:4px}
.summary-loading{text-align:center;color:#015185;font-weight:600;font-size:16px}
.summary-loading i{margin-right:10px}
.charts-container{display:grid;grid-template-columns:repeat(auto-fit,minmax(400px,1fr));gap:20px}
.chart-card{background:white;border:1px solid var(--border-light);border-radius:var(--radius-lg);padding:20px;box-shadow:var(--shadow-sm)}
.chart-card:hover{box-shadow:var(--shadow-md);transform:translateY(-1px)}
.chart-header{margin-bottom:16px;padding-bottom:12px;border-bottom:1px solid var(--border-light)}
.chart-header h3{font-size:18px;font-weight:700;color:#333;margin:0 0 8px 0}
.chart-subtitle{font-size:14px;color:#888;margin:0}
.chart-body{position:relative}
.concentric-chart-container{display:flex;flex-direction:column;align-items:center;gap:20px}
.concentric-chart{position:relative;width:200px;height:200px;display:flex;align-items:center;justify-content:center}
.center-label{position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);text-align:center}
.center-value{font-size:24px;font-weight:700;color:#015185}
.center-text{font-size:14px;color:#015185;font-weight:500}
.concentric-legend{display:flex;flex-wrap:wrap;gap:15px;justify-content:center}
.legend-item{display:flex;align-items:center;gap:8px}
.legend-color{width:12px;height:12px;border-radius:6px}
.legend-text{font-size:12px;font-weight:600;color:#333}
.table-section{margin-top:30px}
.table-section h3{color:#025383;font-size:20px;font-weight:700;margin-bottom:20px}
.table-responsive{overflow-x:auto;border-radius:var(--radius-lg);border:1px solid var(--border-light)}
.modern-table{width:100%;border-collapse:collapse}
.modern-table thead{background:linear-gradient(135deg,#0055a5 0%,#003f7d 100%)}
.modern-table th{color:white;font-weight:600;padding:16px 12px;text-align:center;font-size:14px;border-right:1px solid rgba(255,255,255,.2)}
.modern-table th:last-child{border-right:none}
.modern-table tbody tr{border-bottom:1px solid var(--border-light)}
.modern-table tbody tr:nth-child(even){background:#fafdff}
.modern-table tbody tr:nth-child(odd){background:#eaf3fa}
.modern-table tbody tr:hover{background:#d0e7f7}
.modern-table td{padding:14px 12px;text-align:center;font-size:14px}
.name-cell{color:#2267A2;font-weight:600}
.count-cell{color:#025383;font-weight:700}
.role-badge{padding:4px 12px;border-radius:12px;font-size:12px;font-weight:600;display:inline-block}
.role-supervisor{background:#e3f2fd;color:#1976d2}
.role-manager,.role-operations-manager,.role-project-manager{background:#f3e5f5;color:#7b1fa2}
.loading-row,.no-data{text-align:center;padding:40px;color:var(--text-secondary);font-style:italic}
.summary-section{background:#f8f9fa;border:1px solid var(--border-light);border-radius:var(--radius-lg);padding:24px;margin-top:30px}
.summary-section h3{color:#025383;font-size:20px;font-weight:700;margin:0 0 16px 0}
.summary-stats{display:flex;flex-direction:column;gap:12px}
.summary-item{display:flex;justify-content:space-between;font-size:16px}
.summary-label{color:#015185;font-weight:600}
.summary-value{color:#2267A2;font-weight:700}
@keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}
.sync-modal{position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.5);display:flex;align-items:center;justify-content:center;z-index:10001}
.sync-content{background:white;border-radius:var(--radius-lg);padding:40px;text-align:center;max-width:300px;box-shadow:var(--shadow-lg)}
.sync-icon{width:40px;height:40px;margin:0 auto 20px;font-size:40px;animation:spin 1s linear infinite}
.sync-text{font-size:18px;font-weight:600;margin-bottom:10px}
.sync-subtext{font-size:14px;color:var(--text-secondary)}
.notification{position:fixed;top:20px;right:20px;z-index:10000;padding:16px;border-radius:8px;box-shadow:0 4px 12px rgba(0,0,0,.15);font-family:var(--font-family);max-width:400px;background:white;animation:slideIn .3s}
.notification-success{border-left:4px solid #28a745}
.notification-error{border-left:4px solid #dc3545}
.notification-info{border-left:4px solid #17a2b8}
.notification-content{display:flex;justify-content:space-between;align-items:center}
.notification-close{background:none;border:none;font-size:18px;color:#999;cursor:pointer}
@keyframes slideIn{from{transform:translateX(100%);opacity:0}to{transform:translateX(0);opacity:1}}
.clickable-row{cursor:pointer;transition:var(--transition-base)}
.clickable-row:hover{transform:scale(1.01);box-shadow:0 2px 8px rgba(0,85,165,.15);background:#d0e7f7 !important}
.status-badge{padding:4px 10px;border-radius:12px;font-size:11px;font-weight:600;background:#e3f2fd;color:#1976d2}
.user-modal{position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.5);display:flex;align-items:center;justify-content:center;z-index:10002;overflow-y:auto}
.user-modal-content{background:white;border-radius:var(--radius-lg);width:90%;max-width:900px;max-height:85vh;overflow:hidden;box-shadow:var(--shadow-lg);margin:20px auto}
.user-modal-header{display:flex;justify-content:space-between;align-items:center;padding:20px 30px;border-bottom:1px solid var(--border-light);background:linear-gradient(135deg,#0055a5 0%,#003f7d 100%)}
.user-modal-header h2{color:white;font-size:24px;font-weight:700;margin:0}
.user-modal-close{background:none;border:none;color:white;font-size:28px;cursor:pointer;padding:0;width:32px;height:32px;display:flex;align-items:center;justify-content:center;border-radius:50%;transition:var(--transition-base)}
.user-modal-close:hover{background:rgba(255,255,255,.2)}
.user-modal-body{padding:30px;overflow-y:auto;max-height:calc(85vh - 80px)}
.user-info-section{display:grid;grid-template-columns:repeat(auto-fit,minmax(250px,1fr));gap:20px;margin-bottom:25px;padding:20px;background:#f8fafc;border-radius:var(--radius-lg);border:1px solid var(--border-light)}
.user-info-item{font-size:15px;color:var(--text-primary)}
.user-info-item strong{color:var(--primary-color);font-weight:600;display:inline-block;min-width:140px}
.user-tabs{display:flex;gap:10px;margin-bottom:20px;border-bottom:2px solid var(--border-light)}
.user-tab-btn{padding:12px 24px;background:transparent;border:none;border-bottom:3px solid transparent;color:var(--text-secondary);font-size:15px;font-weight:600;cursor:pointer;transition:var(--transition-base);display:flex;align-items:center;gap:8px}
.user-tab-btn.active{color:var(--primary-color);border-bottom-color:var(--primary-color)}
.user-tab-btn:hover:not(.active){color:var(--primary-hover);background:#f8fafc}
.user-tab-content{display:none}
.user-tab-content.active{display:block}
.tab-actions{display:flex;justify-content:flex-end;margin-bottom:15px}
.btn-export-user{background:linear-gradient(135deg,#10b981 0%,#059669 100%);color:white;border:none;padding:10px 20px;border-radius:var(--radius-md);font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base);display:inline-flex;align-items:center;gap:8px}
.btn-export-user:hover{transform:translateY(-2px);box-shadow:var(--shadow-md)}
@media (max-width:768px){.header-title{font-size:24px}.charts-container{grid-template-columns:1fr}.header-right{flex-direction:column}.create-btn{width:100%;justify-content:center}.filter-form,.date-inputs{flex-direction:column;align-items:stretch}.btn-go,.btn-clear-dates,.date-input{width:100%}.module-tabs{flex-direction:column}.mode-selector-bar{max-width:100%}.user-modal-content{width:95%;max-width:95%}.user-modal-body{padding:15px}.user-info-section{grid-template-columns:1fr}.user-tabs{flex-direction:column;border-bottom:none}.user-tab-btn{border-bottom:none;border-left:3px solid transparent;text-align:left}.user-tab-btn.active{border-left-color:var(--primary-color)}}
@media (max-width:480px){.header-content{flex-direction:column;gap:15px;align-items:flex-start}.header-right{width:100%}.create-btn{width:100%}}
</style>
@endsection
