@extends('welcome')

@section('title', 'DVIR DOT Inspection')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="dvir-inspection-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="javascript:history.back()" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">DVIR DOT Inspection</h1>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="search-icon">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="M21 21l-4.35-4.35"/>
                    </svg>
                    <input type="text" class="search-input" placeholder="Search DVIR reports..." id="searchInput">
                </div>
                <a href="/dvir-dot-inspection/create-dvir-report-step1" class="create-btn">
                    <i class="fas fa-plus"></i>
                    Create New DVIR
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="table-section">
            <div class="table-container">
            <table class="dvir-table" id="dvirTable">
                <thead class="table-header">
                    <tr>
                        <th class="sno-column">S.No</th>
                        <th>Report Number</th>
                        <th>Date</th>
                        <th>Vehicle</th>
                        <th>Location</th>
                        <th>Inspector</th>
                        <th>Status</th>
                        <th>Fault</th>
                        <th class="action-column">Action</th>
                    </tr>
                </thead>
                <tbody class="table-body" id="dvirTableBody">
                @if(isset($paginatedData) && count($paginatedData) > 0)
                    @foreach($paginatedData as $dvir)
                        <tr class="table-row">
                            <td class="sno-cell">{{ $dvir['serial_number'] ?? ($loop->iteration + (($pagination['current_page'] - 1) * $pagination['per_page'])) }}</td>
                            <td class="data-cell">{{ $dvir['reportNumber'] }}</td>
                            <td class="data-cell">
                                @php
                                    try {
                                        $date = \Carbon\Carbon::parse($dvir['date']);
                                        echo $date->format('M d, Y H:i');
                                    } catch (\Exception $e) {
                                        echo is_string($dvir['date']) ? $dvir['date'] : 'Invalid Date';
                                    }
                                @endphp
                            </td>
                            <td class="data-cell">{{ $dvir['vehicle'] }}</td>
                            <td class="data-cell">{{ $dvir['location'] }}</td>
                            <td class="data-cell">{{ $dvir['inspector'] }}</td>
                            <td class="data-cell">
                                @if($dvir['status'] === 'Completed')
                                    <span class="status-badge completed">{{ $dvir['status'] }}</span>
                                @elseif($dvir['status'] === 'Pending')
                                    <span class="status-badge pending">{{ $dvir['status'] }}</span>
                                @elseif($dvir['status'] === 'Issues Found')
                                    <span class="status-badge issues">{{ $dvir['status'] }}</span>
                                @else
                                    <span class="status-badge default">{{ $dvir['status'] }}</span>
                                @endif
                            </td>
                            <td class="data-cell">
                                @if(isset($dvir['faultyItems']) && $dvir['faultyItems'] > 0)
                                    <div class="fault-info">
                                        <span class="fault-badge">{{ $dvir['faultyItems'] }} faults</span>
                                        @if($dvir['fault'] !== 'No faults found')
                                            <span class="fault-details">{{ Str::limit($dvir['fault'], 50) }}</span>
                                        @endif
                                    </div>
                                @else
                                    <span class="no-faults">No faults</span>
                                @endif
                            </td>
                            <td class="action-cell">
                                <div class="action-dropdown">
                                    <button class="btn-action" type="button" id="actionDropdown{{ $dvir['id'] }}" data-bs-toggle="dropdown" aria-expanded="false">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <circle cx="12" cy="12" r="1"/>
                                            <circle cx="19" cy="12" r="1"/>
                                            <circle cx="5" cy="12" r="1"/>
                                        </svg>
                                        <span>Actions</span>
                                    </button>
                                    <ul class="dropdown-menu" aria-labelledby="actionDropdown{{ $dvir['id'] }}">
                                        <li>
                                            <a class="dropdown-item" href="#" onclick="viewDvir('{{ $dvir['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                                    <circle cx="12" cy="12" r="3"/>
                                                </svg>
                                                <span>View</span>
                                            </a>
                                        </li>
                                        <li>
                                            <a class="dropdown-item" href="#" onclick="shareDvir('{{ $dvir['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <circle cx="18" cy="5" r="3"/>
                                                    <circle cx="6" cy="12" r="3"/>
                                                    <circle cx="18" cy="19" r="3"/>
                                                    <line x1="8.59" y1="13.51" x2="15.42" y2="17.49"/>
                                                    <line x1="15.41" y1="6.51" x2="8.59" y2="10.49"/>
                                                </svg>
                                                <span>Share</span>
                                            </a>
                                        </li>
                                        <li>
                                            <a class="dropdown-item" href="#" onclick="createWorkorder('{{ $dvir['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"/>
                                                </svg>
                                                <span>Create Workorder</span>
                                            </a>
                                        </li>
                                        <li>
                                            <a class="dropdown-item" href="#" onclick="viewPdf('{{ $dvir['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                                    <polyline points="14,2 14,8 20,8"/>
                                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                                    <polyline points="10,9 9,9 8,9"/>
                                                </svg>
                                                <span>View PDF</span>
                                            </a>
                                        </li>
                                        <li>
                                            <a class="dropdown-item" href="#" onclick="correctiveAction('{{ $dvir['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <polyline points="20,6 9,17 4,12"/>
                                                </svg>
                                                <span>Corrective Action</span>
                                            </a>
                                        </li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li>
                                            <a class="dropdown-item delete-action" href="#" onclick="deleteDvir('{{ $dvir['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <polyline points="3,6 5,6 21,6"/>
                                                    <path d="M19,6v14a2,2 0 0,1 -2,2H7a2,2 0 0,1 -2,-2V6m3,0V4a2,2 0 0,1 2,-2h4a2,2 0 0,1 2,2v2"/>
                                                </svg>
                                                <span>Delete</span>
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </td>
                        </tr>
                    @endforeach
                @else
                    <tr class="empty-row">
                        <td colspan="9" class="empty-state">
                            <div class="empty-content">
                                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round" class="empty-icon">
                                    <path d="M9 12l2 2 4-4"/>
                                    <path d="M21 12c-1 0-3-1-3-3s2-3 3-3 3 1 3 3-2 3-3 3"/>
                                    <path d="M3 12c1 0 3-1 3-3s-2-3-3-3-3 1-3 3 2 3 3 3"/>
                                </svg>
                                <h3>No DVIR Reports Found</h3>
                                <p>No DVIR data is currently available. Please check your connection or try refreshing the page.</p>
                                <button class="btn-refresh-empty" onclick="refreshData()">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <polyline points="23 4 23 10 17 10"/>
                                        <polyline points="1 20 1 14 7 14"/>
                                        <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"/>
                                    </svg>
                                    <span>Refresh Data</span>
                                </button>
                            </div>
                        </td>
                    </tr>
                @endif
                </tbody>
            </table>
            </div>
        </div>
    </div>

    <!-- Pagination Section -->
    @if(isset($pagination) && $pagination['total_pages'] > 1)
    <div class="pagination-section">
        <div class="pagination-content">
            <div class="pagination-info">
                <span class="pagination-text">
                    Showing {{ $pagination['start_record'] }} to {{ $pagination['end_record'] }} of {{ $pagination['total_records'] }} entries
                </span>
            </div>
            
            <div class="pagination-controls">
                <!-- Previous Button -->
                @if($pagination['has_previous'])
                    <a href="{{ request()->fullUrlWithQuery(['page' => $pagination['previous_page']]) }}" class="pagination-btn prev-btn">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="15,18 9,12 15,6"/>
                        </svg>
                        <span>Previous</span>
                    </a>
                @else
                    <span class="pagination-btn prev-btn disabled">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="15,18 9,12 15,6"/>
                        </svg>
                        <span>Previous</span>
                    </span>
                @endif

                <!-- Page Numbers -->
                <div class="page-numbers">
                    @php
                        $startPage = max(1, $pagination['current_page'] - 2);
                        $endPage = min($pagination['total_pages'], $pagination['current_page'] + 2);
                    @endphp

                    @if($startPage > 1)
                        <a href="{{ request()->fullUrlWithQuery(['page' => 1]) }}" class="page-number">1</a>
                        @if($startPage > 2)
                            <span class="page-ellipsis">...</span>
                        @endif
                    @endif

                    @for($i = $startPage; $i <= $endPage; $i++)
                        @if($i == $pagination['current_page'])
                            <span class="page-number current">{{ $i }}</span>
                        @else
                            <a href="{{ request()->fullUrlWithQuery(['page' => $i]) }}" class="page-number">{{ $i }}</a>
                        @endif
                    @endfor

                    @if($endPage < $pagination['total_pages'])
                        @if($endPage < $pagination['total_pages'] - 1)
                            <span class="page-ellipsis">...</span>
                        @endif
                        <a href="{{ request()->fullUrlWithQuery(['page' => $pagination['total_pages']]) }}" class="page-number">{{ $pagination['total_pages'] }}</a>
                    @endif
                </div>

                <!-- Next Button -->
                @if($pagination['has_next'])
                    <a href="{{ request()->fullUrlWithQuery(['page' => $pagination['next_page']]) }}" class="pagination-btn next-btn">
                        <span>Next</span>
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="9,18 15,12 9,6"/>
                        </svg>
                    </a>
                @else
                    <span class="pagination-btn next-btn disabled">
                        <span>Next</span>
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="9,18 15,12 9,6"/>
                        </svg>
                    </span>
                @endif
            </div>

            <!-- Records per page selector -->
            <div class="records-per-page">
                <label for="perPageSelect">Show:</label>
                <select id="perPageSelect" class="per-page-select" onchange="changePerPage(this.value)">
                    <option value="10" {{ $pagination['per_page'] == 10 ? 'selected' : '' }}>10</option>
                    <option value="25" {{ $pagination['per_page'] == 25 ? 'selected' : '' }}>25</option>
                    <option value="50" {{ $pagination['per_page'] == 50 ? 'selected' : '' }}>50</option>
                    <option value="100" {{ $pagination['per_page'] == 100 ? 'selected' : '' }}>100</option>
                </select>
                <span>entries</span>
            </div>
        </div>
    </div>
    @endif

    <!-- Loading Spinner -->
    <div id="loadingSpinner" class="loading-section d-none">
        <div class="loading-content">
            <div class="spinner">
                <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="spinner-icon">
                    <line x1="12" y1="2" x2="12" y2="6"/>
                    <line x1="12" y1="18" x2="12" y2="22"/>
                    <line x1="4.93" y1="4.93" x2="7.76" y2="7.76"/>
                    <line x1="16.24" y1="16.24" x2="19.07" y2="19.07"/>
                    <line x1="2" y1="12" x2="6" y2="12"/>
                    <line x1="18" y1="12" x2="22" y2="12"/>
                    <line x1="4.93" y1="19.07" x2="7.76" y2="16.24"/>
                    <line x1="16.24" y1="7.76" x2="19.07" y2="4.93"/>
                </svg>
            </div>
            <h3>Loading DVIR Reports</h3>
            <p>Fetching the latest inspection data...</p>
        </div>
    </div>

    <!-- Error Message -->
    <div id="errorMessage" class="error-section d-none">
        <div class="error-content">
            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round" class="error-icon">
                <circle cx="12" cy="12" r="10"/>
                <line x1="15" y1="9" x2="9" y2="15"/>
                <line x1="9" y1="9" x2="15" y2="15"/>
            </svg>
            <h3>Error Loading Data</h3>
            <p id="errorText">An error occurred while loading DVIR reports.</p>
            <button class="btn-retry" onclick="refreshData()">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <polyline points="23 4 23 10 17 10"/>
                    <polyline points="1 20 1 14 7 14"/>
                    <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"/>
                </svg>
                <span>Try Again</span>
            </button>
        </div>
    </div>
</div>

<script>
// Optimized DVIR DOT Inspection Functionality
(function() {
    'use strict';
    
    // Configuration object
    const CONFIG = {
        selectors: {
            searchInput: '#searchInput',
            dvirTableBody: '#dvirTableBody',
            loadingSpinner: '#loadingSpinner',
            dvirTable: '#dvirTable',
            errorMessage: '#errorMessage',
            errorText: '#errorText'
        },
        messages: {
            loading: 'Loading DVIR Reports',
            loadingSubtext: 'Fetching the latest inspection data...',
            error: 'Error Loading Data',
            errorDefault: 'An error occurred while loading DVIR reports.',
            noData: 'No DVIR Reports Found',
            noDataSubtext: 'No DVIR data is currently available. Please check your connection or try refreshing the page.'
        },
        timing: {
            loadingTimeout: 3000
        }
    };
    
    // Cache DOM elements
    let elements = {};
    
    function initElements() {
        elements = {
            searchInput: document.getElementById('searchInput'),
            dvirTableBody: document.getElementById('dvirTableBody'),
            loadingSpinner: document.getElementById('loadingSpinner'),
            dvirTable: document.getElementById('dvirTable'),
            errorMessage: document.getElementById('errorMessage'),
            errorText: document.getElementById('errorText')
        };
    }
    
    // Refresh data function
    function refreshData() {
        showLoading();
        location.reload();
    }
    
    // Search functionality
    function initSearch() {
        if (!elements.searchInput) return;
        
        elements.searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const tableRows = document.querySelectorAll('#dvirTableBody tr');
            
            tableRows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
    }

    // View DVIR details
    function viewDvir(dvirId) {
        console.log('Viewing DVIR:', dvirId);
        showNotification(`Opening DVIR details for report ${dvirId}`, 'info');
    }
    
    // Share DVIR
    function shareDvir(dvirId) {
        console.log('Sharing DVIR:', dvirId);
        showNotification(`Sharing DVIR report ${dvirId}`, 'info');
    }
    
    // Create workorder
    function createWorkorder(dvirId) {
        console.log('Creating workorder for DVIR:', dvirId);
        showNotification(`Creating workorder for DVIR ${dvirId}`, 'success');
    }
    
    // Corrective action
    function correctiveAction(dvirId) {
        console.log('Corrective action for DVIR:', dvirId);
        showNotification(`Opening corrective action for DVIR ${dvirId}`, 'info');
    }
    
    // View PDF
    function viewPdf(dvirId) {
        console.log('Viewing PDF for DVIR:', dvirId);
        const pdfUrl = `/inspection/view-dvir/${dvirId}`;
        window.open(pdfUrl, '_blank');
    }
    
    // Delete DVIR
    function deleteDvir(dvirId) {
        console.log('Deleting DVIR:', dvirId);
        if (confirm('Are you sure you want to delete this DVIR? This action cannot be undone.')) {
            showNotification(`DVIR ${dvirId} has been deleted`, 'warning');
        }
    }

    // Hide error message
    function hideError() {
        if (elements.errorMessage) {
            elements.errorMessage.classList.add('d-none');
        }
    }
    
    // Show loading spinner
    function showLoading() {
        if (elements.loadingSpinner) elements.loadingSpinner.classList.remove('d-none');
        if (elements.dvirTable) elements.dvirTable.classList.add('d-none');
        if (elements.errorMessage) elements.errorMessage.classList.add('d-none');
    }
    
    // Hide loading spinner
    function hideLoading() {
        if (elements.loadingSpinner) elements.loadingSpinner.classList.add('d-none');
        if (elements.dvirTable) elements.dvirTable.classList.remove('d-none');
    }
    
    // Show error message
    function showError(message) {
        if (elements.errorText) elements.errorText.textContent = message;
        if (elements.errorMessage) elements.errorMessage.classList.remove('d-none');
        hideLoading();
    }

    // Test DVIR data fetching
    function testDvirData() {
        showLoading();
        
        const tableRows = document.querySelectorAll('#dvirTableBody tr');
        const hasData = tableRows.length > 0 && !document.querySelector('.tbl-body-no-data-td');
        
        if (hasData) {
            let message = 'DVIR Data Test Successful!\n\n';
            message += `Total Records: ${tableRows.length}\n\n`;
            
            const sampleRows = Array.from(tableRows).slice(0, 3);
            sampleRows.forEach((row, index) => {
                const cells = row.querySelectorAll('td');
                if (cells.length >= 9) {
                    message += `Row ${index + 1}:\n`;
                    message += `- Report Number: ${cells[1]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Date: ${cells[2]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Vehicle: ${cells[3]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Location: ${cells[4]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Inspector: ${cells[5]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Status: ${cells[6]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Fault: ${cells[7]?.textContent?.trim() || 'N/A'}\n`;
                    message += `- Action: ${cells[8]?.textContent?.trim() || 'N/A'}\n\n`;
                }
            });
            
            alert(message);
        } else {
            alert('DVIR Data Test: No data found in the table. Please check the Firebase connection and data structure.');
        }
        
        hideLoading();
    }

    // Test DVIR data structure
    function testDvirStructure() {
        showLoading();
        
        fetch('/vehicles/test-dvir-structure')
            .then(response => response.json())
            .then(data => {
                hideLoading();
                
                if (data.success) {
                    let message = 'DVIR Data Structure Analysis:\n\n';
                    message += `Total Reports: ${data.analysis.total_reports}\n\n`;
                    
                    if (data.analysis.sample_reports && Object.keys(data.analysis.sample_reports).length > 0) {
                        message += 'Sample Report Analysis:\n';
                        Object.keys(data.analysis.sample_reports).forEach(reportId => {
                            const report = data.analysis.sample_reports[reportId];
                            message += `\nReport ID: ${report.report_id}\n`;
                            message += `Has Step1: ${report.has_step1 ? 'Yes' : 'No'}\n`;
                            
                            if (report.has_step1) {
                                message += `Step1 Keys: ${report.step1_keys.join(', ')}\n`;
                            }
                            
                            message += `\nRequest Number Paths:\n`;
                            Object.keys(report.request_number_paths).forEach(path => {
                                message += `- ${path}: ${report.request_number_paths[path]}\n`;
                            });
                            
                            message += `\nVehicle Paths:\n`;
                            Object.keys(report.vehicle_paths).forEach(path => {
                                message += `- ${path}: ${report.vehicle_paths[path]}\n`;
                            });
                            
                            message += `\nLocation Paths:\n`;
                            Object.keys(report.location_paths).forEach(path => {
                                message += `- ${path}: ${report.location_paths[path]}\n`;
                            });
                            
                            message += `\nInspector Paths:\n`;
                            Object.keys(report.inspector_paths).forEach(path => {
                                message += `- ${path}: ${report.inspector_paths[path]}\n`;
                            });
                        });
                    }
                    
                    alert(message);
                } else {
                    showError('DVIR structure test failed: ' + data.message);
                }
            })
            .catch(error => {
                hideLoading();
                showError('Error testing DVIR structure: ' + error.message);
            });
    }

    // Enhanced notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
        notification.style.cssText = `top: 20px; right: 20px; z-index: ${CONFIG.zIndexNotification}; min-width: 300px;`;
        notification.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }

    // Initialize dropdown functionality
    function initializeDropdowns() {
        // Close dropdowns when clicking outside
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.action-dropdown')) {
                const dropdowns = document.querySelectorAll('.dropdown-menu.show');
                dropdowns.forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });

        // Handle dropdown button clicks
        document.addEventListener('click', function(e) {
            if (e.target.closest('.btn-action')) {
                e.preventDefault();
                e.stopPropagation();
                
                const button = e.target.closest('.btn-action');
                const dropdown = button.nextElementSibling;
                
                // Close other dropdowns
                const otherDropdowns = document.querySelectorAll('.dropdown-menu.show');
                otherDropdowns.forEach(otherDropdown => {
                    if (otherDropdown !== dropdown) {
                        otherDropdown.classList.remove('show');
                    }
                });
                
                // Toggle current dropdown
                if (dropdown) {
                    dropdown.classList.toggle('show');
                }
            }
        });

        // Handle keyboard navigation
        document.addEventListener('keydown', function(e) {
            if (e.target.closest('.btn-action') && (e.key === 'Enter' || e.key === ' ')) {
                e.preventDefault();
                e.stopPropagation();
                
                const button = e.target.closest('.btn-action');
                const dropdown = button.nextElementSibling;
                
                if (dropdown) {
                    dropdown.classList.toggle('show');
                }
            }
        });
    }
    
    // Initialize all functionality
    function init() {
        initElements();
        initSearch();
        initializeDropdowns();
        
        // Check if data is loaded
        const tableBody = document.getElementById('dvirTableBody');
        const hasData = tableBody && tableBody.children.length > 0 && 
                       !tableBody.querySelector('.tbl-body-no-data-td');
        
        if (!hasData) {
            showLoading();
            setTimeout(function() {
                if (!hasData) {
                    showError('Failed to load DVIR reports. Please check your connection and try refreshing.');
                }
            }, CONFIG.timing.loadingTimeout);
        }
        
        // Add click event for search input focus
        if (elements.searchInput) {
            elements.searchInput.addEventListener('focus', function() {
                this.select();
            });
        }
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
    // Change records per page
    function changePerPage(perPage) {
        const url = new URL(window.location);
        url.searchParams.set('per_page', perPage);
        url.searchParams.set('page', 1); // Reset to first page
        window.location.href = url.toString();
    }

    // Make functions globally accessible
    window.refreshData = refreshData;
    window.viewDvir = viewDvir;
    window.shareDvir = shareDvir;
    window.createWorkorder = createWorkorder;
    window.correctiveAction = correctiveAction;
    window.viewPdf = viewPdf;
    window.deleteDvir = deleteDvir;
    window.testDvirData = testDvirData;
    window.testDvirStructure = testDvirStructure;
    window.showNotification = showNotification;
    window.changePerPage = changePerPage;
})();
</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.dvir-inspection-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    color: var(--primary-color);
    text-decoration: none;
}

/* Search Container */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #9ca3af;
    z-index: 1;
}

.search-input {
    padding: 10px 12px 10px 40px;
    border: 1px solid #d1d5db;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    width: 300px;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Table Section */
.table-section {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin-bottom: 24px;
}

.table-container {
    overflow-x: auto;
}

.dvir-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.table-header th {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    color: #374151;
    font-weight: 600;
    font-size: 14px;
    padding: 16px 12px;
    text-align: left;
    border-bottom: 1px solid #e2e8f0;
    border-radius: 0;
}

.sno-column {
    min-width: 60px;
    width: 60px;
    text-align: center;
}

.action-column {
    min-width: 120px;
    width: 120px;
}

.table-body td {
    padding: 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
}

.sno-cell {
    text-align: center;
    font-weight: 600;
    color: #6b7280;
    background-color: #f8fafc;
}

.table-row:hover {
    background: #f8fafc;
}

/* Action Cell */
.action-cell {
    text-align: center;
    padding: 12px 8px;
    border-bottom: 1px solid #e2e8f0;
    vertical-align: middle;
    position: relative;
    width: 120px;
}

/* Action Dropdown */
.action-dropdown {
    position: relative;
    display: inline-block;
}

.btn-action {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 12px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #6b7280;
    font-size: 13px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-action:hover {
    background: #f9fafb;
    border-color: #0055a5;
    color: #0055a5;
}

.dropdown-menu {
    border: 1px solid #e2e8f0;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    border-radius: 8px;
    padding: 8px 0;
    min-width: 200px;
    background: white;
    position: absolute;
    top: 100%;
    right: 0;
    z-index: 1000;
    display: none;
    opacity: 0;
    transform: translateY(-10px);
    transition: all 0.2s ease;
}

.dropdown-menu.show {
    display: block;
    opacity: 1;
    transform: translateY(0);
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 10px 16px;
    font-size: 14px;
    color: #374151;
    transition: all 0.2s ease;
    text-decoration: none;
}

.dropdown-item:hover {
    background: #f8fafc;
    color: #0f172a;
    text-decoration: none;
}

.dropdown-item.delete-action {
    color: #dc2626;
}

.dropdown-item.delete-action:hover {
    background: #fef2f2;
    color: #dc2626;
}

.dropdown-divider {
    margin: 8px 0;
    border-color: #e2e8f0;
}

/* Status Badges */
.status-badge {
    display: inline-flex;
    align-items: center;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

.status-badge.completed {
    background: #dcfce7;
    color: #166534;
    border: 1px solid #bbf7d0;
}

.status-badge.pending {
    background: #fef3c7;
    color: #92400e;
    border: 1px solid #fde68a;
}

.status-badge.issues {
    background: #fee2e2;
    color: #991b1b;
    border: 1px solid #fecaca;
}

.status-badge.default {
    background: #f3f4f6;
    color: #374151;
    border: 1px solid #d1d5db;
}

/* Data Cell Styling */
.data-cell {
    color: #374151;
    font-weight: 400;
}

.fault-text {
    color: #dc2626;
    font-weight: 500;
}

.no-fault {
    color: #059669;
    font-weight: 500;
}

.fault-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.fault-badge {
    background-color: #dc2626;
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
    display: inline-block;
    width: fit-content;
}

.fault-details {
    color: #dc2626;
    font-weight: 500;
    font-size: 14px;
}

.no-faults {
    color: #059669;
    font-weight: 500;
}

.action-text {
    color: #0ea5e9;
    font-weight: 500;
}

.no-action {
    color: #6b7280;
    font-weight: 400;
}

/* Empty State */
.empty-state {
    padding: 60px 20px;
    text-align: center;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-icon {
    color: #9ca3af;
}

.empty-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.empty-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

.btn-refresh-empty {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: 1px solid #3b82f6;
    border-radius: 0;
    background: white;
    color: #3b82f6;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-refresh-empty:hover {
    background: #3b82f6;
    color: white;
}

/* Loading Section */
.loading-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    margin-bottom: 24px;
}

.loading-content {
    padding: 60px 20px;
    text-align: center;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.spinner {
    animation: spin 1s linear infinite;
}

.spinner-icon {
    color: #3b82f6;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.loading-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.loading-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

/* Error Section */
.error-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    margin-bottom: 24px;
}

.error-content {
    padding: 60px 20px;
    text-align: center;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.error-icon {
    color: #dc2626;
}

.error-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.error-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

.btn-retry {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: 1px solid #dc2626;
    border-radius: 0;
    background: white;
    color: #dc2626;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-retry:hover {
    background: #dc2626;
    color: white;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .main-content {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .search-input {
        width: 250px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .header-left {
        justify-content: center;
    }
    
    .header-right {
        justify-content: center;
        flex-direction: column;
        gap: 10px;
    }
    
    .search-input {
        width: 100%;
        max-width: 400px;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
    
    .main-content {
        padding: 15px;
    }
}

@media (max-width: 992px) {
    .dvir-inspection-container {
        padding: 12px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
        margin-bottom: 6px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-content {
        padding: 12px 16px;
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: center;
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create,
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort,
    .btn-export {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
        flex: 1;
        min-width: 120px;
        justify-content: center;
    }
    
    .search-section {
        align-self: center;
    }
    
    .search-input {
        width: 100%;
        max-width: 400px;
        padding: 12px 16px;
        font-size: 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .dvir-table {
        min-width: 800px;
    }
    
    .table-header th {
        padding: 12px 8px;
        font-size: 13px;
    }
    
    .table-body td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .sno-column {
        min-width: 50px;
        width: 50px;
    }
    
    .action-column {
        min-width: 100px;
        width: 100px;
    }
    
    .btn-action {
        padding: 6px 10px;
        font-size: 12px;
    }
    
    .dropdown-menu {
        min-width: 180px;
    }
    
    .dropdown-item {
        padding: 8px 12px;
        font-size: 13px;
    }
}

@media (max-width: 768px) {
    .dvir-inspection-container {
        padding: 0;
    }
    
    .dvir-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        margin-bottom: 16px;
    }
    
    .action-content {
        padding: 12px 16px;
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
        width: 100%;
    }
    
    .btn-create {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 10px 8px;
        font-size: 12px;
        min-height: 44px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
    }
    
    .btn-export {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
    }
    
    .search-section {
        width: 100%;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .table-section {
        margin-bottom: 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .dvir-table {
        min-width: 700px;
        font-size: 12px;
    }
    
    .table-header th {
        padding: 10px 6px;
        font-size: 11px;
        background-color: #f8fafc;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table-body td {
        padding: 8px 6px;
        font-size: 11px;
        white-space: nowrap;
    }
    
    .sno-column {
        min-width: 40px;
        width: 40px;
    }
    
    .action-column {
        min-width: 80px;
        width: 80px;
    }
    
    .btn-action {
        padding: 6px 8px;
        font-size: 11px;
        min-width: 32px;
        min-height: 32px;
    }
    
    .dropdown-menu {
        min-width: 160px;
    }
    
    .dropdown-item {
        padding: 8px 12px;
        font-size: 12px;
    }
    
    .status-badge {
        padding: 4px 8px;
        font-size: 10px;
    }
    
    .empty-state {
        padding: 40px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
    .loading-content,
    .error-content {
        padding: 40px 16px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 16px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 13px;
    }
}

@media (max-width: 576px) {
    .dvir-inspection-container {
        padding: 0;
    }
    
    .header-content {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-content {
        padding: 10px 12px;
        gap: 10px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr;
        gap: 6px;
        width: 100%;
    }
    
    .btn-create,
    .btn-export {
        width: 100%;
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        width: 100%;
        padding: 12px 8px;
        font-size: 11px;
        min-height: 48px;
        flex-direction: row;
        justify-content: center;
        gap: 6px;
    }
    
    .search-input {
        padding: 14px 16px;
        font-size: 15px;
        min-height: 52px;
    }
    
    .table-container {
        border: 1px solid #d1d5db;
    }
    
    .dvir-table {
        min-width: 600px;
        font-size: 11px;
    }
    
    .table-header th {
        padding: 8px 4px;
        font-size: 10px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 6px 4px;
        font-size: 10px;
    }
    
    .sno-column {
        min-width: 35px;
        width: 35px;
    }
    
    .action-column {
        min-width: 70px;
        width: 70px;
    }
    
    .btn-action {
        padding: 4px 6px;
        font-size: 10px;
        min-width: 28px;
        min-height: 28px;
    }
    
    .dropdown-menu {
        min-width: 140px;
    }
    
    .dropdown-item {
        padding: 6px 10px;
        font-size: 11px;
    }
    
    .status-badge {
        padding: 3px 6px;
        font-size: 9px;
    }
    
    .empty-state {
        padding: 30px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
    .loading-content,
    .error-content {
        padding: 30px 12px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 14px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 12px;
    }
}

@media (max-width: 480px) {
    .header-content {
        padding: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .action-content {
        padding: 8px;
        gap: 8px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-create,
    .btn-export {
        padding: 10px 6px !important;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 14px 6px;
        font-size: 10px;
        min-height: 52px;
        gap: 4px;
    }
    
    .search-input {
        padding: 16px 12px;
        font-size: 16px;
        min-height: 56px;
    }
    
    .dvir-table {
        min-width: 500px;
        font-size: 10px;
    }
    
    .table-header th {
        padding: 6px 3px;
        font-size: 9px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 4px 3px;
        font-size: 9px;
    }
    
    .sno-column {
        min-width: 30px;
        width: 30px;
    }
    
    .action-column {
        min-width: 60px;
        width: 60px;
    }
    
    .btn-action {
        padding: 3px 4px;
        font-size: 9px;
        min-width: 24px;
        min-height: 24px;
    }
    
    .dropdown-menu {
        min-width: 120px;
    }
    
    .dropdown-item {
        padding: 4px 8px;
        font-size: 10px;
    }
    
    .status-badge {
        padding: 2px 4px;
        font-size: 8px;
    }
    
    .empty-state {
        padding: 20px 8px;
    }
    
    .empty-content h3 {
        font-size: 13px;
    }
    
    .empty-content p {
        font-size: 11px;
    }
    
    .loading-content,
    .error-content {
        padding: 20px 8px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 13px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 11px;
    }
}

@media (max-width: 360px) {
    .header-content {
        padding: 6px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 1px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .action-content {
        padding: 6px;
        gap: 6px;
    }
    
    .action-buttons {
        gap: 3px;
    }
    
    .btn-create,
    .btn-export {
        padding: 18px 8px;
        font-size: 14px;
        font-weight: 700;
        min-height: 60px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 16px 4px;
        font-size: 9px;
        min-height: 56px;
        gap: 2px;
    }
    
    .search-input {
        padding: 18px 8px;
        font-size: 14px;
        min-height: 60px;
    }
    
    .dvir-table {
        min-width: 450px;
        font-size: 9px;
    }
    
    .table-header th {
        padding: 4px 2px;
        font-size: 8px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 3px 2px;
        font-size: 8px;
    }
    
    .action-column {
        min-width: 50px;
        width: 50px;
    }
    
    .btn-action {
        padding: 2px 3px;
        font-size: 8px;
        min-width: 20px;
        min-height: 20px;
    }
    
    .dropdown-menu {
        min-width: 100px;
    }
    
    .dropdown-item {
        padding: 3px 6px;
        font-size: 9px;
    }
    
    .status-badge {
        padding: 1px 2px;
        font-size: 7px;
    }
    
    .empty-state {
        padding: 15px 6px;
    }
    
    .empty-content h3 {
        font-size: 12px;
    }
    
    .empty-content p {
        font-size: 10px;
    }
    
    .loading-content,
    .error-content {
        padding: 15px 6px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 12px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 10px;
    }
}

/* Focus states for accessibility */
.btn-create:focus,
.btn-refresh:focus,
.btn-test:focus,
.btn-structure:focus,
.btn-sort:focus,
.btn-export:focus,
.btn-action:focus,
.search-input:focus,
.btn-refresh-empty:focus,
.btn-retry:focus {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* Pagination Styles */
.pagination-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
    padding: 20px 32px;
}

.pagination-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

.pagination-info {
    flex-shrink: 0;
}

.pagination-text {
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 8px;
    flex-wrap: wrap;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    cursor: pointer;
    transition: var(--transition-base);
}

.pagination-btn:hover:not(.disabled) {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
    text-decoration: none;
}

.pagination-btn.disabled {
    background: var(--bg-primary);
    color: var(--text-light);
    cursor: not-allowed;
    opacity: 0.6;
}

.page-numbers {
    display: flex;
    align-items: center;
    gap: 4px;
    margin: 0 12px;
}

.page-number {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
    padding: 0 8px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    cursor: pointer;
    transition: var(--transition-base);
}

.page-number:hover:not(.current) {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
    text-decoration: none;
}

.page-number.current {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
    cursor: default;
}

.page-ellipsis {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

.records-per-page {
    display: flex;
    align-items: center;
    gap: 8px;
    flex-shrink: 0;
}

.records-per-page label {
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
}

.per-page-select {
    padding: 6px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.per-page-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.records-per-page span {
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

/* Responsive Pagination */
@media (max-width: 768px) {
    .pagination-content {
        flex-direction: column;
        gap: 16px;
        align-items: stretch;
    }
    
    .pagination-controls {
        justify-content: center;
    }
    
    .page-numbers {
        margin: 0 8px;
    }
    
    .page-number {
        min-width: 36px;
        height: 36px;
        font-size: 13px;
    }
    
    .records-per-page {
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .pagination-section {
        padding: 16px 20px;
    }
    
    .pagination-btn {
        padding: 6px 12px;
        font-size: 13px;
    }
    
    .page-number {
        min-width: 32px;
        height: 32px;
        font-size: 12px;
    }
    
    .pagination-text {
        font-size: 13px;
    }
    
    .per-page-select {
        font-size: 13px;
    }
}

/* Smooth animations */
* {
    transition: all 0.2s ease;
}

/* Icon improvements */
.btn-create svg,
.btn-refresh svg,
.btn-test svg,
.btn-structure svg,
.btn-sort svg,
.btn-export svg,
.btn-action svg,
.search-icon,
.dropdown-item svg,
.btn-refresh-empty svg,
.btn-retry svg,
.empty-icon,
.spinner-icon,
.error-icon {
    flex-shrink: 0;
}

/* ===== ENHANCED RESPONSIVE DESIGN ===== */
@media (max-width: 1400px) {
    .dvir-inspection-container {
        padding: 18px;
    }
    
    .dvir-header {
        margin-bottom: 20px;
    }
    
    .header-content {
        padding: 28px;
    }
}

@media (max-width: 1200px) {
    .dvir-inspection-container {
        padding: 16px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .action-bar {
        padding: 20px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 16px;
        align-items: flex-start;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: flex-start;
        flex-wrap: wrap;
        gap: 12px;
    }
}

@media (max-width: 992px) {
    .dvir-inspection-container {
        padding: 14px;
    }
    
    .dvir-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px;
    }
    
    .action-buttons {
        flex-direction: column;
        width: 100%;
    }
    
    .btn-create,
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort,
    .btn-export {
        width: 100%;
        justify-content: center;
    }
    
    .table-responsive {
        font-size: 14px;
    }
    
    .table th,
    .table td {
        padding: 8px;
    }
}

@media (max-width: 768px) {
    .dvir-inspection-container {
        padding: 8px;
    }
    
    .header-content {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 8px;
        margin-bottom: 12px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
        width: 100%;
    }
    
    .btn-create {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 10px 8px;
        font-size: 12px;
        min-height: 44px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
    }
    
    .btn-export {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 16px;
        height: 16px;
    }
    
    .search-container {
        flex-direction: column;
        gap: 8px;
        margin-bottom: 12px;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        border-radius: 0;
    }
    
    .search-btn {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
    }
    
    .table-responsive {
        font-size: 12px;
        border: 1px solid #e2e8f0;
        border-radius: 0;
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .table {
        margin-bottom: 0;
        min-width: 600px;
    }
    
    .table th,
    .table td {
        padding: 8px 4px;
        white-space: nowrap;
        vertical-align: middle;
    }
    
    .table th {
        background-color: #f8fafc;
        font-weight: 600;
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table td {
        font-size: 11px;
    }
    
    .status-badge {
        padding: 4px 8px;
        font-size: 10px;
        border-radius: 0;
    }
    
    .action-btn {
        padding: 6px 8px;
        font-size: 11px;
        min-width: 32px;
        min-height: 32px;
    }
}

@media (max-width: 576px) {
    .dvir-inspection-container {
        padding: 4px;
    }
    
    .header-content {
        padding: 8px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .action-bar {
        padding: 6px;
        margin-bottom: 8px;
    }
    
    .action-content {
        gap: 6px;
    }
    
    .action-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .btn-create,
    .btn-export {
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 12px 8px;
        font-size: 11px;
        min-height: 48px;
        flex-direction: row;
        justify-content: center;
        gap: 6px;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 18px;
        height: 18px;
    }
    
    .search-container {
        gap: 6px;
        margin-bottom: 8px;
    }
    
    .search-input {
        padding: 14px 16px;
        font-size: 15px;
    }
    
    .search-btn {
        padding: 14px 16px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .table-responsive {
        font-size: 11px;
        margin-bottom: 16px;
    }
    
    .table {
        min-width: 500px;
    }
    
    .table th,
    .table td {
        padding: 6px 3px;
    }
    
    .table th {
        font-size: 10px;
        padding: 8px 3px;
    }
    
    .table td {
        font-size: 10px;
    }
    
    .status-badge {
        padding: 3px 6px;
        font-size: 9px;
    }
    
    .action-btn {
        padding: 4px 6px;
        font-size: 10px;
        min-width: 28px;
        min-height: 28px;
    }
}

@media (max-width: 480px) {
    .dvir-inspection-container {
        padding: 2px;
    }
    
    .header-content {
        padding: 6px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 1px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .action-bar {
        padding: 4px;
        margin-bottom: 6px;
    }
    
    .action-content {
        gap: 4px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-create,
    .btn-export {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 14px 6px;
        font-size: 10px;
        min-height: 52px;
        gap: 4px;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 20px;
        height: 20px;
    }
    
    .search-container {
        gap: 4px;
        margin-bottom: 6px;
    }
    
    .search-input {
        padding: 16px 12px;
        font-size: 16px;
    }
    
    .search-btn {
        padding: 16px 12px;
        font-size: 16px;
        min-height: 52px;
    }
    
    .table-responsive {
        font-size: 10px;
        margin-bottom: 12px;
        border: 1px solid #d1d5db;
    }
    
    .table {
        min-width: 450px;
    }
    
    .table th,
    .table td {
        padding: 4px 2px;
    }
    
    .table th {
        font-size: 9px;
        padding: 6px 2px;
        background-color: #f3f4f6;
    }
    
    .table td {
        font-size: 9px;
    }
    
    .status-badge {
        padding: 2px 4px;
        font-size: 8px;
    }
    
    .action-btn {
        padding: 3px 4px;
        font-size: 9px;
        min-width: 24px;
        min-height: 24px;
    }
}

@media (max-width: 360px) {
    .dvir-inspection-container {
        padding: 1px;
    }
    
    .header-content {
        padding: 4px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 0;
    }
    
    .page-title p {
        font-size: 9px;
    }
    
    .action-bar {
        padding: 2px;
        margin-bottom: 4px;
    }
    
    .action-content {
        gap: 2px;
    }
    
    .action-buttons {
        gap: 2px;
    }
    
    .btn-create,
    .btn-export {
        padding: 18px 8px;
        font-size: 14px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 16px 4px;
        font-size: 9px;
        min-height: 56px;
        gap: 2px;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 18px;
        height: 18px;
    }
    
    .search-container {
        gap: 2px;
        margin-bottom: 4px;
    }
    
    .search-input {
        padding: 18px 8px;
        font-size: 14px;
    }
    
    .search-btn {
        padding: 18px 8px;
        font-size: 14px;
        min-height: 56px;
    }
    
    .table-responsive {
        font-size: 9px;
        margin-bottom: 8px;
    }
    
    .table {
        min-width: 400px;
    }
    
    .table th,
    .table td {
        padding: 3px 1px;
    }
    
    .table th {
        font-size: 8px;
        padding: 4px 1px;
    }
    
    .table td {
        font-size: 8px;
    }
    
    .status-badge {
        padding: 1px 2px;
        font-size: 7px;
    }
    
    .action-btn {
        padding: 2px 3px;
        font-size: 8px;
        min-width: 20px;
        min-height: 20px;
    }
}
</style>
@endsection