@extends('welcome')

@section('title', 'Fuel Bunker')

@section('content')
<div class="fuel-bunker-container">
    <div class="fuel-bunker-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Fuel Bunker</h1>
                <p>Manage fuel bunkers, tanks, and refill operations efficiently</p>
            </div>
        </div>
    </div>
    
    <div class="fuel-bunker-content">
        <!-- Fuel Bunker Section -->
        <div class="content-box">
            <div class="action-bar">
                <div class="action-content">
                    <div class="page-title">
                        <h2>Fuel Bunker</h2>
                    </div>
                    <div class="action-buttons">
                        <button class="btn-create" data-bs-toggle="modal" data-bs-target="#new-fuel-bunker">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M12 5v14m-7-7h14"/>
                            </svg>
                            Add New Fuel Bunker
                        </button>
                        <button class="btn-sort refill-history" onclick="scrollToSection('history-section')">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"/>
                                <path d="M21 3v5h-5"/>
                                <path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"/>
                                <path d="M3 21v-5h5"/>
                            </svg>
                            Refill History
                        </button>
                        <button class="btn-export adjustment-history" onclick="scrollToSection('adjustment-section')">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                <polyline points="7,10 12,15 17,10"/>
                                <line x1="12" y1="15" x2="12" y2="3"/>
                            </svg>
                            Adjustment History
                        </button>
                    </div>
                </div>
            </div>
            <div class="table-section">
                <div class="table-container">
                    <table class="fuel-bunker-table">
                        <thead class="table-header">
                            <tr>
                                <th>Action</th>
                                <th>Tank ID</th>
                                <th>Location</th>
                                <th>Description</th>
                                <th>Re-filling Date</th>
                                <th>Tank Size</th>
                                <th>Fuel Level</th>
                                <th>Current Fuel Quantity</th>
                                <th>Avg Unit Cost</th>
                                <th>Fuel Cost</th>
                            </tr>
                        </thead>
                        <tbody class="table-body">
                            <tr class="empty-row">
                                <td colspan="10">
                                    <div class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                                            </svg>
                                            <h3>No Fuel Bunkers Found</h3>
                                            <p>There are currently no fuel bunkers to display. Create your first fuel bunker to get started.</p>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Fuel Trucks Section -->
        <div class="content-box">
            <div class="action-bar">
                <div class="action-content">
                    <div class="page-title">
                        <h2>Fuel Trucks</h2>
                    </div>
                    <div class="action-buttons">
                        <button class="btn-create" data-bs-toggle="modal" data-bs-target="#new-fuel-truck">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M12 5v14m-7-7h14"/>
                            </svg>
                            Add Fuel Truck
                        </button>
                    </div>
                </div>
            </div>
            <div class="table-section">
                <div class="table-container">
                    <table class="fuel-bunker-table">
                        <thead class="table-header">
                            <tr>
                                <th>Action</th>
                                <th>Tank ID</th>
                                <th>Location</th>
                                <th>Description</th>
                                <th>Re-filling Date</th>
                                <th>Tank Size</th>
                                <th>Fuel Level</th>
                                <th>Current Fuel Quantity</th>
                                <th>Avg Unit Cost</th>
                                <th>Fuel Cost</th>
                            </tr>
                        </thead>
                        <tbody class="table-body">
                            <tr class="empty-row">
                                <td colspan="10">
                                    <div class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                                            </svg>
                                            <h3>No Fuel Trucks Found</h3>
                                            <p>There are currently no fuel trucks to display. Create your first fuel truck to get started.</p>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Fuel Bunker History Section -->
        <div class="content-box" id="history-section">
            <div class="action-bar">
                <div class="action-content">
                    <div class="page-title">
                        <h2>Fuel Bunker History</h2>
                    </div>
                    <div class="action-buttons">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search fuel bunker history...">
                            </div>
                        </div>
                        <button class="btn-sort" data-bs-toggle="modal" data-bs-target="#sorting">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M3 6h18"/>
                                <path d="M7 12h14"/>
                                <path d="M10 18h11"/>
                            </svg>
                            Sorting
                        </button>
                        <button class="btn-export">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                <polyline points="7,10 12,15 17,10"/>
                                <line x1="12" y1="15" x2="12" y2="3"/>
                            </svg>
                            Excel Export
                        </button>
                    </div>
                </div>
            </div>
            <div class="table-section">
                <div class="table-container">
                    <table class="fuel-bunker-table">
                        <thead class="table-header">
                            <tr>
                                <th>Action</th>
                                <th>Tank/Truck ID</th>
                                <th>Location</th>
                                <th>Note</th>
                                <th>Re-filling Date</th>
                                <th>Old Quantity</th>
                                <th>Refill Quantity</th>
                                <th>New Quantity</th>
                                <th>Unit Cost</th>
                                <th>Fuel Cost</th>
                                <th>Refill by</th>
                            </tr>
                        </thead>
                        <tbody class="table-body">
                            <tr class="empty-row">
                                <td colspan="11">
                                    <div class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                                            </svg>
                                            <h3>No History Found</h3>
                                            <p>There are currently no fuel bunker history records to display.</p>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Fuel Bunker Adjustment Section -->
        <div class="content-box" id="adjustment-section">
            <div class="action-bar">
                <div class="action-content">
                    <div class="page-title">
                        <h2>Fuel Bunker Adjustment</h2>
                    </div>
                    <div class="action-buttons">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search adjustment history...">
                            </div>
                        </div>
                        <button class="btn-sort" data-bs-toggle="modal" data-bs-target="#truck-sorting">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M3 6h18"/>
                                <path d="M7 12h14"/>
                                <path d="M10 18h11"/>
                            </svg>
                            Sorting
                        </button>
                        <button class="btn-export">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                <polyline points="7,10 12,15 17,10"/>
                                <line x1="12" y1="15" x2="12" y2="3"/>
                            </svg>
                            Excel Export
                        </button>
                    </div>
                </div>
            </div>
            <div class="table-section">
                <div class="table-container">
                    <table class="fuel-bunker-table">
                        <thead class="table-header">
                            <tr>
                                <th>Tank/Truck ID</th>
                                <th>Location</th>
                                <th>Note</th>
                                <th>Date</th>
                                <th>Re-filling Date</th>
                                <th>Old Quantity</th>
                                <th>Adjust Quantity</th>
                                <th>New Quantity</th>
                                <th>Unit Cost</th>
                                <th>Fuel Cost</th>
                                <th>Adjust by</th>
                            </tr>
                        </thead>
                        <tbody class="table-body">
                            <tr class="empty-row">
                                <td colspan="11">
                                    <div class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                                            </svg>
                                            <h3>No Adjustments Found</h3>
                                            <p>There are currently no fuel bunker adjustment records to display.</p>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Optimized Fuel Bunker Management Functionality
(function() {
    'use strict';
    
    // Configuration object
    const CONFIG = {
        selectors: {
            sidebar: '.sidebar',
            searchInputs: '.search-input',
            tables: '.fuel-bunker-table',
            contentBoxes: '.content-box'
        },
        messages: {
            searchPlaceholder: 'Search fuel bunker data...',
            noResults: 'No matching results found',
            error: 'An error occurred while processing your request'
        },
        timing: {
            searchDelay: 300,
            animationDuration: 300
        }
    };
    
    // Cache DOM elements
    let elements = {};
    
    function initElements() {
        elements = {
            sidebar: document.querySelector(CONFIG.selectors.sidebar),
            searchInputs: document.querySelectorAll(CONFIG.selectors.searchInputs),
            tables: document.querySelectorAll(CONFIG.selectors.tables),
            contentBoxes: document.querySelectorAll(CONFIG.selectors.contentBoxes)
        };
    }
    
    // Sidebar toggle functionality
    function toggleSidebar() {
        if (elements.sidebar) {
            elements.sidebar.classList.toggle('collapsed');
        }
    }
    
    // Smooth scroll to section
    function scrollToSection(sectionId) {
        const element = document.getElementById(sectionId);
        if (element) {
            element.scrollIntoView({ 
                behavior: 'smooth',
                block: 'start'
            });
        }
    }
    
    // Enhanced search functionality
    function filterTable(input, tableSelector) {
        const filter = input.value.toLowerCase();
        const table = document.querySelector(tableSelector);
        if (!table) return;
        
        const rows = table.getElementsByTagName('tr');
        let hasResults = false;
        
        for (let i = 1; i < rows.length; i++) {
            const row = rows[i];
            const cells = row.getElementsByTagName('td');
            let found = false;
            
            for (let j = 0; j < cells.length; j++) {
                const cell = cells[j];
                if (cell.textContent.toLowerCase().indexOf(filter) > -1) {
                    found = true;
                    break;
                }
            }
            
            row.style.display = found ? '' : 'none';
            if (found) hasResults = true;
        }
        
        // Show/hide empty state
        const emptyRow = table.querySelector('.empty-row');
        if (emptyRow) {
            emptyRow.style.display = hasResults ? 'none' : '';
        }
    }
    
    // Initialize search functionality
    function initSearch() {
        elements.searchInputs.forEach(input => {
            let searchTimeout;
            input.addEventListener('keyup', function() {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    const tableSelector = this.closest('.content-box').querySelector('.fuel-bunker-table');
                    filterTable(this, tableSelector);
                }, CONFIG.timing.searchDelay);
            });
        });
    }
    
    // Modal functionality
    function initModals() {
        // Fuel Bunker Form
        const fuelBunkerForm = document.getElementById('fuelBunkerForm');
        if (fuelBunkerForm) {
            fuelBunkerForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const requiredFields = ['tankId', 'site', 'tankSize', 'location', 'fuelType'];
                const formData = new FormData(this);
                
                // Check required fields
                const isValid = requiredFields.every(field => formData.get(field));
                
                if (!isValid) {
                    showNotification('Please fill in all required fields', 'error');
                    return;
                }
                
                showNotification('Fuel bunker added successfully!', 'success');
                
                // Close modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('new-fuel-bunker'));
                if (modal) modal.hide();
                
                // Reset form
                this.reset();
            });
        }
        
        // Fuel Truck Form
        const fuelTruckForm = document.getElementById('fuelTruckForm');
        if (fuelTruckForm) {
            fuelTruckForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const requiredFields = ['truckTankId', 'truckSite', 'truckTankSize', 'truckLocation', 'truckFuelType'];
                const formData = new FormData(this);
                
                // Check required fields
                const isValid = requiredFields.every(field => formData.get(field));
                
                if (!isValid) {
                    showNotification('Please fill in all required fields', 'error');
                    return;
                }
                
                showNotification('Fuel truck added successfully!', 'success');
                
                // Close modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('new-fuel-truck'));
                if (modal) modal.hide();
                
                // Reset form
                this.reset();
            });
        }
    }
    
    // Enhanced notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `fuel-notification fuel-notification-${type}`;
        notification.textContent = message;
        
        const colors = {
            success: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
            error: 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
            warning: 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
            info: 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)'
        };
        
        notification.style.cssText = `
            position: fixed;
            top: 80px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 0;
            color: white;
            font-weight: 500;
            font-size: 14px;
            z-index: 9999;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            font-family: 'Inter', sans-serif;
            background: ${colors[type] || colors.info};
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (notification.parentNode) {
                    document.body.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }
    
    // Initialize all functionality
    function init() {
        initElements();
        initSearch();
        initModals();
        
        // Make functions globally accessible
        window.toggleSidebar = toggleSidebar;
        window.scrollToSection = scrollToSection;
        window.showNotification = showNotification;
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
</script>
<!-- Sorting Popup modals -->
<div class="modal" id="sorting">
    <div class="modal-dialog">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header">
                <h4 class="modal-title">Sorting</h4>
            </div>
            <!-- Modal body -->
            <div class="modal-body">
                <form action="">
                    <form action="">
                        <h1>Sort Order</h1>
                        <div class="form-check">
                            <input type="radio" class="form-check-input" id="radio1" name="optradio" value="Ascending" checked>
                            <label class="form-check-label" for="radio1">Ascending</label>
                        </div>
                        <div class="form-check">
                            <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Descending">
                            <label class="form-check-label" for="radio2">Descending</label>
                        </div>
                    </form>
                    <h1>Sort By</h1>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Number" checked>
                        <label class="form-check-label" for="radio2">Tank ID</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Start Date">
                        <label class="form-check-label" for="radio2">Location</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Number">
                        <label class="form-check-label" for="radio2">Note</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio1" name="optradio" value="Report Number">
                        <label class="form-check-label" for="radio1">Re-filling Date</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Start Date">
                        <label class="form-check-label" for="radio2">Quantity</label>
                    </div>
                    <div class="row my-3 align-items-center justify-content-center">
                        <button type="submit" class="save btn btn-danger col-md-4 mx-2">Ok</button>
                        <button type="reset" class="save btn btn-danger col-md-4 mx-2">Reset</button>
                        <!-- Modal footer -->
                        <div class="modal-footer col-md-4 p-0">
                            <button type="button" class="btn btn-danger" data-bs-dismiss="modal">Cancel</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<!-- Add Fuel Bunker Modal -->
<div class="modal fade" id="new-fuel-bunker" tabindex="-1" aria-labelledby="addFuelBunkerModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="addFuelBunkerModalLabel">Add Fuel Bunker</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">&times;</button>
            </div>
            <div class="modal-body">
                <form class="fuel-bunker-form" id="fuelBunkerForm">
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="tankId">Tank ID * :</label>
                            <input type="text" class="form-control" id="tankId" name="tankId" placeholder="Tank001" required>
                    </div>
                        <div class="form-group">
                            <label for="site">Site * :</label>
                            <select class="form-control" id="site" name="site" required>
                                <option value="">Select a site</option>
                                <option value="site1">Site 1</option>
                                <option value="site2">Site 2</option>
                                <option value="site3">Site 3</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="tankSize">Tank Size * :</label>
                            <div class="input-with-select">
                                <input type="number" class="form-control" id="tankSize" name="tankSize" placeholder="Enter tank size" required>
                                <select class="form-control" name="tankSizeUnit">
                                    <option value="gallon">Gallon</option>
                                    <option value="liter">Liter</option>
                                    <option value="barrel">Barrel</option>
                            </select>
                        </div>
                    </div>
                        <div class="form-group">
                            <label for="fuelLevel">Current Fuel Level :</label>
                            <div class="input-with-select">
                                <input type="number" class="form-control" id="fuelLevel" name="fuelLevel" placeholder="Enter fuel level" step="0.01">
                                <select class="form-control" name="fuelLevelUnit">
                                    <option value="gallon">Gallon</option>
                                    <option value="liter">Liter</option>
                                    <option value="barrel">Barrel</option>
                                </select>
                    </div>
                        </div>
                    </div>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="location">Location * :</label>
                            <input type="text" class="form-control" id="location" name="location" placeholder="Enter location" required>
                        </div>
                        <div class="form-group">
                            <label for="fuelType">Fuel Type * :</label>
                            <select class="form-control" id="fuelType" name="fuelType" required>
                                <option value="">Select fuel type</option>
                                <option value="diesel">Diesel</option>
                                <option value="petrol">Petrol</option>
                                <option value="gas">Gas</option>
                                <option value="lpg">LPG</option>
                                <option value="cng">CNG</option>
                                <option value="oil">Oil</option>
                                <option value="adblue">AdBlue</option>
                        </select>
                    </div>
                    </div>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="avgUnitCost">Average Unit Cost :</label>
                            <div class="input-with-select">
                                <input type="number" class="form-control" id="avgUnitCost" name="avgUnitCost" placeholder="Enter unit cost" step="0.01">
                                <select class="form-control" name="costUnit">
                                    <option value="per_gallon">Per Gallon</option>
                                    <option value="per_liter">Per Liter</option>
                                    <option value="per_barrel">Per Barrel</option>
                                </select>
                        </div>
                        </div>
                        <div class="form-group">
                            <label for="lastRefillDate">Last Refill Date :</label>
                            <input type="date" class="form-control" id="lastRefillDate" name="lastRefillDate">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="description">Description :</label>
                        <textarea class="form-control" id="description" name="description" placeholder="Enter description" rows="3"></textarea>
                    </div>
                    
                    <div class="modal-actions">
                        <button type="submit" class="btn-save">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M20 6L9 17l-5-5"/>
                            </svg>
                            Add Fuel Bunker
                        </button>
                        <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M18 6L6 18M6 6l12 12"/>
                            </svg>
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<!-- Add Fuel Truck Modal -->
<div class="modal fade" id="new-fuel-truck" tabindex="-1" aria-labelledby="addFuelTruckModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="addFuelTruckModalLabel">Add Fuel Truck</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">&times;</button>
            </div>
            <div class="modal-body">
                <form class="fuel-truck-form" id="fuelTruckForm">
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="truckTankId">Tank ID * :</label>
                            <input type="text" class="form-control" id="truckTankId" name="truckTankId" placeholder="Truck001" required>
                    </div>
                        <div class="form-group">
                            <label for="truckSite">Site * :</label>
                            <select class="form-control" id="truckSite" name="truckSite" required>
                                <option value="">Select a site</option>
                                <option value="site1">Site 1</option>
                                <option value="site2">Site 2</option>
                                <option value="site3">Site 3</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="truckTankSize">Tank Size * :</label>
                            <div class="input-with-select">
                                <input type="number" class="form-control" id="truckTankSize" name="truckTankSize" placeholder="Enter tank size" required>
                                <select class="form-control" name="truckTankSizeUnit">
                                    <option value="gallon">Gallon</option>
                                    <option value="liter">Liter</option>
                                    <option value="barrel">Barrel</option>
                            </select>
                        </div>
                    </div>
                        <div class="form-group">
                            <label for="truckFuelLevel">Current Fuel Level :</label>
                            <div class="input-with-select">
                                <input type="number" class="form-control" id="truckFuelLevel" name="truckFuelLevel" placeholder="Enter fuel level" step="0.01">
                                <select class="form-control" name="truckFuelLevelUnit">
                                    <option value="gallon">Gallon</option>
                                    <option value="liter">Liter</option>
                                    <option value="barrel">Barrel</option>
                                </select>
                    </div>
                        </div>
                    </div>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="truckLocation">Location * :</label>
                            <input type="text" class="form-control" id="truckLocation" name="truckLocation" placeholder="Enter location" required>
                        </div>
                        <div class="form-group">
                            <label for="truckFuelType">Fuel Type * :</label>
                            <select class="form-control" id="truckFuelType" name="truckFuelType" required>
                                <option value="">Select fuel type</option>
                                <option value="diesel">Diesel</option>
                                <option value="petrol">Petrol</option>
                                <option value="gas">Gas</option>
                                <option value="lpg">LPG</option>
                                <option value="cng">CNG</option>
                                <option value="oil">Oil</option>
                                <option value="adblue">AdBlue</option>
                        </select>
                    </div>
                    </div>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="truckAvgUnitCost">Average Unit Cost :</label>
                            <div class="input-with-select">
                                <input type="number" class="form-control" id="truckAvgUnitCost" name="truckAvgUnitCost" placeholder="Enter unit cost" step="0.01">
                                <select class="form-control" name="truckCostUnit">
                                    <option value="per_gallon">Per Gallon</option>
                                    <option value="per_liter">Per Liter</option>
                                    <option value="per_barrel">Per Barrel</option>
                                </select>
                        </div>
                        </div>
                        <div class="form-group">
                            <label for="truckLastRefillDate">Last Refill Date :</label>
                            <input type="date" class="form-control" id="truckLastRefillDate" name="truckLastRefillDate">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="truckDescription">Description :</label>
                        <textarea class="form-control" id="truckDescription" name="truckDescription" placeholder="Enter description" rows="3"></textarea>
                    </div>
                    
                    <div class="modal-actions">
                        <button type="submit" class="btn-save">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M20 6L9 17l-5-5"/>
                            </svg>
                            Add Fuel Truck
                        </button>
                        <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M18 6L6 18M6 6l12 12"/>
                            </svg>
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<!-- Truck Sorting Popup modals -->
<div class="modal" id="truck-sorting">
    <div class="modal-dialog">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header">
                <h4 class="modal-title">Sorting</h4>
            </div>
            <!-- Modal body -->
            <div class="modal-body">
                <form action="">
                    <form action="">
                        <h1>Sort Order</h1>
                        <div class="form-check">
                            <input type="radio" class="form-check-input" id="radio1" name="optradio" value="Ascending" checked>
                            <label class="form-check-label" for="radio1">Ascending</label>
                        </div>
                        <div class="form-check">
                            <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Descending">
                            <label class="form-check-label" for="radio2">Descending</label>
                        </div>
                    </form>
                    <h1>Sort By</h1>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Number" checked>
                        <label class="form-check-label" for="radio2">Tank ID</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Start Date">
                        <label class="form-check-label" for="radio2">Location</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Vehicle Number">
                        <label class="form-check-label" for="radio2">Note</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio1" name="optradio" value="Report Number">
                        <label class="form-check-label" for="radio1">Date</label>
                    </div>
                    <div class="form-check">
                        <input type="radio" class="form-check-input" id="radio2" name="optradio" value="Start Date">
                        <label class="form-check-label" for="radio2">Quantity</label>
                    </div>
                    <div class="row my-3 align-items-center justify-content-center">
                        <button type="submit" class="save btn btn-danger col-md-4 mx-2">Ok</button>
                        <button type="reset" class="save btn btn-danger col-md-4 mx-2">Reset</button>
                        <!-- Modal footer -->
                        <div class="modal-footer col-md-4 p-0">
                            <button type="button" class="btn btn-danger" data-bs-dismiss="modal">Cancel</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
@endsection

<style>
/* CSS Variables for better maintainability */
:root {
    --primary-color: #0055a5;
    --primary-hover: #004494;
    --success-color: #00B894;
    --success-hover: #00a085;
    --secondary-color: #545d6b;
    --secondary-hover: #3d4449;
    --text-primary: #212529;
    --text-secondary: #495057;
    --text-muted: #6c757d;
    --text-light: #adb5bd;
    --bg-primary: #ffffff;
    --bg-secondary: #f8fafc;
    --bg-light: #f1f5f9;
    --border-color: #e2e8f0;
    --border-light: #f3f4f6;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 6px;
    --radius-md: 8px;
    --radius-lg: 12px;
    --transition-base: all 0.2s ease;
    --transition-slow: all 0.3s ease;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1055;
    --z-index-notification: 9999;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.fuel-bunker-container {
    background: var(--bg-secondary);
}

/* ===== HEADER STYLES ===== */
.fuel-bunker-header {
    background: var(--primary-color);
    border-radius: 0;
    padding: 32px;
    margin-bottom: 24px;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.page-title h1 {
    color: white;
    font-size: 32px;
    font-weight: 700;
    margin: 0 0 8px 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.page-title p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    font-weight: 400;
    margin: 0;
}

/* ===== MAIN CONTENT ===== */
.fuel-bunker-content {
    display: flex;
    flex-direction: column;
    gap: 24px;
}

.content-box {
    background: var(--bg-primary);
    border-radius: 0;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-color);
    overflow: hidden;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: var(--bg-secondary);
    border-bottom: 1px solid var(--border-color);
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

.page-title h2 {
    color: black !important;
    font-weight: 700;
    font-size: 24px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0;
}

.page-title h2 {
    font-size: 20px;
}

/* ===== ACTION BUTTONS ===== */
.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.btn-create,
.btn-sort,
.btn-export {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0 !important;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    text-decoration: none;
}

.btn-create {
    background: var(--success-color);
    color: white;
}

.btn-create:hover {
    background: var(--success-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    color: white;
    text-decoration: none;
}

.btn-sort {
    background: var(--secondary-color);
    color: white;
}

.btn-sort:hover {
    background: var(--secondary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    color: white;
    text-decoration: none;
}

.btn-export {
    background: var(--primary-color);
    color: white;
}

.btn-export:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    color: white;
    text-decoration: none;
}

.btn-cancel {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: #ffffff;
    color: #64748b;
    border: 2px solid #e2e8f0;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel:hover {
    background: #f8fafc;
    border-color: #cbd5e1;
    color: #475569;
}

/* ===== SEARCH SECTION ===== */
.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: var(--bg-primary);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

/* ===== TABLE SECTION ===== */
.table-section {
    padding: 0;
}

.table-container {
    overflow-x: auto;
}

.fuel-bunker-table {
    width: 100%;
    border-collapse: collapse;
    font-family: var(--font-family);
}

.table-header {
    background: linear-gradient(135deg, var(--bg-secondary) 0%, var(--bg-light) 100%);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: var(--text-secondary);
    border-bottom: 2px solid #e5e7eb;
    border-radius: 0;
}

.table-body {
    background: var(--bg-primary);
}

.table-row {
    border-bottom: 1px solid var(--border-light);
    transition: var(--transition-base);
}

.table-row:hover {
    background: var(--bg-secondary);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: var(--text-secondary);
    vertical-align: middle;
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: var(--bg-primary);
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: var(--bg-primary);
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: var(--text-light);
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-secondary);
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: var(--text-muted);
    margin: 0;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .fuel-bunker-container {
        padding: 20px;
    }
    
    .fuel-bunker-header {
        padding: 28px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-content {
        gap: 16px;
    }
    
    .action-buttons {
        gap: 10px;
    }
    
    .btn-create,
    .btn-sort,
    .btn-export {
        padding: 9px 14px;
        font-size: 13px;
    }
}

@media (max-width: 992px) {
    .fuel-bunker-container {
        padding: 16px;
    }
    
    .fuel-bunker-header {
        padding: 24px;
        margin-bottom: 18px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .fuel-bunker-content {
        gap: 20px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .page-title h2 {
        font-size: 18px;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: flex-start;
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create,
    .btn-sort,
    .btn-export {
        padding: 10px 16px;
        font-size: 14px;
        min-height: 44px;
        flex: 1;
        min-width: 120px;
        justify-content: center;
    }
    
    .search-section {
        width: 100%;
        max-width: none;
    }
    
    .search-input {
        padding: 10px 12px 10px 40px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .fuel-bunker-table {
        min-width: 800px;
    }
    
    .table-header th {
        padding: 12px 8px;
        font-size: 13px;
    }
    
    .table-body td {
        padding: 12px 8px;
        font-size: 13px;
    }
}

@media (max-width: 768px) {
    .fuel-bunker-container {
        padding: 0;
    }
    
    .fuel-bunker-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .fuel-bunker-content {
        gap: 16px;
    }
    
    .content-box {
        margin-bottom: 16px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .page-title h2 {
        font-size: 16px;
        margin-bottom: 8px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
        width: 100%;
    }
    
    .btn-create {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
    }
    
    .btn-sort,
    .btn-export {
        padding: 10px 8px;
        font-size: 12px;
        min-height: 44px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
    }
    
    .search-section {
        width: 100%;
        margin-top: 8px;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px 12px 44px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .fuel-bunker-table {
        min-width: 700px;
        font-size: 12px;
    }
    
    .table-header th {
        padding: 10px 6px;
        font-size: 11px;
        background-color: #f8fafc;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table-body td {
        padding: 8px 6px;
        font-size: 11px;
        white-space: nowrap;
    }
    
    .empty-state {
        padding: 40px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
}

@media (max-width: 576px) {
    .fuel-bunker-container {
        padding: 0;
    }
    
    .fuel-bunker-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .fuel-bunker-content {
        gap: 12px;
    }
    
    .content-box {
        margin-bottom: 12px;
    }
    
    .action-bar {
        padding: 10px 12px;
    }
    
    .action-content {
        gap: 10px;
    }
    
    .page-title h2 {
        font-size: 15px;
        margin-bottom: 6px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr;
        gap: 6px;
        width: 100%;
    }
    
    .btn-create,
    .btn-export {
        width: 100%;
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .btn-sort {
        width: 100%;
        padding: 12px 8px;
        font-size: 11px;
        min-height: 48px;
        flex-direction: row;
        justify-content: center;
        gap: 6px;
    }
    
    .search-input {
        padding: 14px 16px 14px 44px;
        font-size: 15px;
        min-height: 52px;
    }
    
    .table-container {
        border: 1px solid #d1d5db;
    }
    
    .fuel-bunker-table {
        min-width: 600px;
        font-size: 11px;
    }
    
    .table-header th {
        padding: 8px 4px;
        font-size: 10px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 6px 4px;
        font-size: 10px;
    }
    
    .empty-state {
        padding: 30px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
}

@media (max-width: 480px) {
    .fuel-bunker-header {
        padding: 8px;
        margin-bottom: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .fuel-bunker-content {
        gap: 8px;
    }
    
    .content-box {
        margin-bottom: 8px;
    }
    
    .action-bar {
        padding: 8px;
    }
    
    .action-content {
        gap: 8px;
    }
    
    .page-title h2 {
        font-size: 14px;
        margin-bottom: 4px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-create,
    .btn-export {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .btn-sort {
        padding: 14px 6px;
        font-size: 10px;
        min-height: 52px;
        gap: 4px;
    }
    
    .search-input {
        padding: 16px 12px 16px 44px;
        font-size: 16px;
        min-height: 56px;
    }
    
    .fuel-bunker-table {
        min-width: 500px;
        font-size: 10px;
    }
    
    .table-header th {
        padding: 6px 3px;
        font-size: 9px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 4px 3px;
        font-size: 9px;
    }
    
    .empty-state {
        padding: 20px 8px;
    }
    
    .empty-content h3 {
        font-size: 13px;
    }
    
    .empty-content p {
        font-size: 11px;
    }
}

@media (max-width: 360px) {
    .fuel-bunker-header {
        padding: 6px;
        margin-bottom: 6px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 1px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .fuel-bunker-content {
        gap: 6px;
    }
    
    .content-box {
        margin-bottom: 6px;
    }
    
    .action-bar {
        padding: 6px;
    }
    
    .action-content {
        gap: 6px;
    }
    
    .page-title h2 {
        font-size: 13px;
        margin-bottom: 3px;
    }
    
    .action-buttons {
        gap: 3px;
    }
    
    .btn-create,
    .btn-export {
        padding: 18px 8px;
        font-size: 14px;
        font-weight: 700;
        min-height: 60px;
    }
    
    .btn-sort {
        padding: 16px 4px;
        font-size: 9px;
        min-height: 56px;
        gap: 2px;
    }
    
    .search-input {
        padding: 18px 8px 18px 44px;
        font-size: 14px;
        min-height: 60px;
    }
    
    .fuel-bunker-table {
        min-width: 450px;
        font-size: 9px;
    }
    
    .table-header th {
        padding: 4px 2px;
        font-size: 8px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 3px 2px;
        font-size: 8px;
    }
    
    .empty-state {
        padding: 15px 6px;
    }
    
    .empty-content h3 {
        font-size: 12px;
    }
    
    .empty-content p {
        font-size: 10px;
    }
}

/* ===== MODAL STYLES ===== */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: 1055;
    display: none;
    justify-content: center;
    align-items: center;
}

.modal.show {
    display: flex !important;
}

.modal.fade .modal-dialog {
    transition: transform 0.3s ease-out;
    transform: translate(0, -50px);
}

.modal.show .modal-dialog {
    transform: none;
}

.modal-dialog {
    width: 90%;
    max-width: 600px;
    margin: 0;
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-content {
    background: #ffffff;
    border: none;
    border-radius: 0;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
}

.modal-header {
    padding: 20px 24px;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: var(--primary-color);
    color: white;
}

.modal-title {
    margin: 0;
    color: white;
    font-size: 20px;
    font-weight: 600;
}

.modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: white;
    padding: 8px;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 0;
    transition: background-color 0.2s;
}

.modal-close:hover {
    background-color: rgba(255, 255, 255, 0.2);
}

.modal-body {
    padding: 24px;
}

/* ===== FORM STYLES ===== */
.fuel-bunker-form,
.fuel-truck-form {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-group label {
    font-size: 0.875rem;
    font-weight: 600;
    color: #374151;
    font-family: 'Inter', sans-serif;
}

.form-control {
    border: 1px solid #d1d5db;
    border-radius: 0;
    padding: 12px 16px;
    font-size: 14px;
    transition: var(--transition-base);
    background: var(--bg-primary);
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-control::placeholder {
    color: #94a3b8;
    font-weight: 400;
}

.input-with-select {
    display: flex;
    gap: 0.5rem;
}

.input-with-select .form-control {
    flex: 1;
}

/* ===== MODAL ACTIONS ===== */
.modal-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid var(--border-color);
}

.btn-save {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--success-color);
    color: white;
    border: 1px solid var(--success-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    text-decoration: none;
}

.btn-save:hover {
    background: var(--success-hover);
    border-color: var(--success-hover);
    transform: translateY(-1px);
    color: white;
    text-decoration: none;
}

.btn-cancel {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--secondary-color);
    color: white;
    border: 1px solid var(--secondary-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    text-decoration: none;
}

.btn-cancel:hover {
    background: var(--secondary-hover);
    border-color: var(--secondary-hover);
    transform: translateY(-1px);
    color: white;
    text-decoration: none;
}

/* ===== MODAL RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .modal-dialog {
        margin: 1rem;
        max-width: calc(100% - 2rem);
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .modal-actions {
        flex-direction: column;
    }
    
    .btn-save,
    .btn-cancel {
        width: 100%;
        justify-content: center;
    }
    }
    
    .modal-content {
        border-radius: 8px;
    }
    
    .modal-header {
        padding: 1rem;
    }
    
    .modal-header h4 {
        font-size: 1.1rem;
    }
    
    .modal-body {
        padding: 1rem;
    }
    
    .form-inline .input-group {
        margin-bottom: 1rem;
        flex-direction: column;
        align-items: stretch;
    }
    
    .form-inline .input-group label {
        margin-bottom: 0.5rem;
        font-size: 0.9rem;
        font-weight: 600;
    }
    
    .form-inline .input-group .form-control,
    .form-inline .input-group .form-select {
        width: 100%;
        padding: 12px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .form-inline .row {
        margin: 0;
    }
    
    .form-inline .row .input-group {
        width: 100% !important;
        margin-bottom: 1rem;
    }
    
    .form-check {
        margin-bottom: 0.75rem;
    }
    
    .form-check-label {
        font-size: 0.9rem;
    }
    
    .btn {
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }
    
    .btn svg {
        width: 20px;
        height: 20px;
    }
}

@media (max-width: 576px) {
    .modal-dialog {
        margin: 0.5rem;
        max-width: calc(100% - 1rem);
    }
    
    .modal-header {
        padding: 0.75rem;
    }
    
    .modal-header h4 {
        font-size: 1rem;
    }
    
    .modal-body {
        padding: 0.75rem;
    }
    
    .form-inline .input-group {
        margin-bottom: 0.75rem;
    }
    
    .form-inline .input-group label {
        font-size: 0.85rem;
    }
    
    .form-inline .input-group .form-control,
    .form-inline .input-group .form-select {
        padding: 10px;
        font-size: 15px;
        min-height: 44px;
    }
    
    .btn {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
    }
    
    .btn svg {
        width: 18px;
        height: 18px;
    }
}

@media (max-width: 480px) {
    .modal-dialog {
        margin: 0.25rem;
        max-width: calc(100% - 0.5rem);
    }
    
    .modal-header {
        padding: 0.5rem;
    }
    
    .modal-header h4 {
        font-size: 0.9rem;
    }
    
    .modal-body {
        padding: 0.5rem;
    }
    
    .form-inline .input-group {
        margin-bottom: 0.5rem;
    }
    
    .form-inline .input-group label {
        font-size: 0.8rem;
    }
    
    .form-inline .input-group .form-control,
    .form-inline .input-group .form-select {
        padding: 8px;
        font-size: 14px;
        min-height: 40px;
    }
    
    .btn {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 40px;
    }
    
    .btn svg {
        width: 16px;
        height: 16px;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== ADDITIONAL REQUEST_MAINTENANCE-STYLE IMPROVEMENTS ===== */
.fuel-bunker-container {
    background: #f8fafc;
}

.content-box {
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
}

.table-header th {
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: black !important;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

/* Force sharp corners for all buttons */
.action-buttons button,
.action-buttons .btn-create,
.action-buttons .btn-sort,
.action-buttons .btn-export,
.refill-history,
.adjustment-history {
    border-radius: 0 !important;
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid var(--primary-color);
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .action-bar,
    .modal {
        display: none !important;
    }
    
    .fuel-bunker-container {
        background: white;
    }
    
    .fuel-bunker-header {
        background: white;
        color: black;
    }
}
</style>