@extends('welcome')

@section('title', 'Document Upload Test')

@section('content')
<main class="document-upload-container" role="main">
    <!-- Header -->
    <header class="upload-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Document Upload Test</h1>
                <p>Test Firebase Storage document upload functionality</p>
            </div>
        </div>
    </header>

    <!-- Upload Section -->
    <section class="upload-section">
        <div class="upload-container">
            <!-- PDF Upload -->
            <div class="upload-card">
                <div class="upload-header">
                    <h3>PDF Upload</h3>
                    <p>Upload PDF documents to Firebase Storage</p>
                </div>
                <div class="upload-content">
                    <input type="file" id="pdfFileInput" accept=".pdf" class="file-input">
                    <button id="uploadPdfBtn" class="upload-btn pdf-btn" disabled>
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                        </svg>
                        Upload PDF
                    </button>
                </div>
                <div id="pdfProgress" class="progress-container" style="display: none;">
                    <div class="progress-bar">
                        <div class="progress-fill" id="pdfProgressFill"></div>
                    </div>
                    <span class="progress-text" id="pdfProgressText">0%</span>
                </div>
                <div id="pdfResult" class="result-container" style="display: none;"></div>
            </div>

            <!-- DOCX Upload -->
            <div class="upload-card">
                <div class="upload-header">
                    <h3>DOCX Upload</h3>
                    <p>Upload Word documents to Firebase Storage</p>
                </div>
                <div class="upload-content">
                    <input type="file" id="docxFileInput" accept=".docx,.doc" class="file-input">
                    <button id="uploadDocxBtn" class="upload-btn docx-btn" disabled>
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                        </svg>
                        Upload DOCX
                    </button>
                </div>
                <div id="docxProgress" class="progress-container" style="display: none;">
                    <div class="progress-bar">
                        <div class="progress-fill" id="docxProgressFill"></div>
                    </div>
                    <span class="progress-text" id="docxProgressText">0%</span>
                </div>
                <div id="docxResult" class="result-container" style="display: none;"></div>
            </div>

            <!-- File List -->
            <div class="upload-card">
                <div class="upload-header">
                    <h3>Uploaded Files</h3>
                    <p>Files stored in Firebase Storage</p>
                </div>
                <div class="upload-content">
                    <button id="refreshFilesBtn" class="refresh-btn">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="23,4 23,10 17,10"></polyline>
                            <polyline points="1,20 1,14 7,14"></polyline>
                            <path d="M20.49,9A9,9,0,0,0,5.64,5.64L1,10m22,4L18.36,18.36A9,9,0,0,1,3.51,15"></path>
                        </svg>
                        Refresh Files
                    </button>
                    <div id="fileList" class="file-list">
                        <!-- Files will be loaded here -->
                    </div>
                </div>
            </div>
        </div>
    </section>
</main>

<!-- Firebase SDK -->
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-storage-compat.js"></script>

<script>
// Firebase configuration
const firebaseConfig = {
    apiKey: "AIzaSyDsA1JohORkg-EdXWTHlFEea8zCqfvig8w",
    authDomain: "titan-drilling-1f8e9.firebaseapp.com",
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com",
    projectId: "titan-drilling-1f8e9",
    storageBucket: "titan-drilling-1f8e9.firebasestorage.app",
    messagingSenderId: "749788854975",
    appId: "1:749788854975:web:71224235d9355be09b640a",
    measurementId: "G-R9L21J6GHD"
};

// Initialize Firebase
firebase.initializeApp(firebaseConfig);
const storage = firebase.storage();

// Firebase Storage path
const STORAGE_PATH = 'GlobalInspectionReport/admins/7fZvZzm2XJbjMyi2uESwv7luROY2/inspectionReports';

// DOM elements
const pdfFileInput = document.getElementById('pdfFileInput');
const docxFileInput = document.getElementById('docxFileInput');
const uploadPdfBtn = document.getElementById('uploadPdfBtn');
const uploadDocxBtn = document.getElementById('uploadDocxBtn');
const refreshFilesBtn = document.getElementById('refreshFilesBtn');
const fileList = document.getElementById('fileList');

// Initialize event listeners
document.addEventListener('DOMContentLoaded', function() {
    // File input change handlers
    pdfFileInput.addEventListener('change', function() {
        uploadPdfBtn.disabled = !this.files[0];
        if (this.files[0]) {
            uploadPdfBtn.textContent = `Upload ${this.files[0].name}`;
        } else {
            uploadPdfBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                    <polyline points="14,2 14,8 20,8"></polyline>
                </svg>
                Upload PDF
            `;
        }
    });

    docxFileInput.addEventListener('change', function() {
        uploadDocxBtn.disabled = !this.files[0];
        if (this.files[0]) {
            uploadDocxBtn.textContent = `Upload ${this.files[0].name}`;
        } else {
            uploadDocxBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                    <polyline points="14,2 14,8 20,8"></polyline>
                </svg>
                Upload DOCX
            `;
        }
    });

    // Upload button handlers
    uploadPdfBtn.addEventListener('click', () => uploadFile('pdf'));
    uploadDocxBtn.addEventListener('click', () => uploadFile('docx'));
    refreshFilesBtn.addEventListener('click', loadFileList);

    // Load initial file list
    loadFileList();
});

// Upload file function
async function uploadFile(type) {
    const fileInput = type === 'pdf' ? pdfFileInput : docxFileInput;
    const file = fileInput.files[0];
    
    if (!file) {
        showNotification('Please select a file first', 'error');
        return;
    }

    const progressContainer = document.getElementById(`${type}Progress`);
    const progressFill = document.getElementById(`${type}ProgressFill`);
    const progressText = document.getElementById(`${type}ProgressText`);
    const resultContainer = document.getElementById(`${type}Result`);

    try {
        // Show progress
        progressContainer.style.display = 'block';
        resultContainer.style.display = 'none';

        // Create file path
        const timestamp = Date.now();
        const fileExtension = file.name.split('.').pop();
        const fileName = `${type.toUpperCase()}_${timestamp}.${fileExtension}`;
        const filePath = `${STORAGE_PATH}/${fileName}`;

        console.log('Uploading file:', filePath);

        // Create storage reference
        const storageRef = storage.ref();
        const fileRef = storageRef.child(filePath);

        // Upload file with progress monitoring
        const uploadTask = fileRef.put(file, {
            contentType: file.type,
            customMetadata: {
                uploadedAt: new Date().toISOString(),
                originalName: file.name,
                fileType: type,
                uploadedBy: 'admin'
            }
        });

        // Monitor upload progress
        uploadTask.on('state_changed',
            (snapshot) => {
                const progress = (snapshot.bytesTransferred / snapshot.totalBytes) * 100;
                progressFill.style.width = progress + '%';
                progressText.textContent = Math.round(progress) + '%';
            },
            (error) => {
                console.error('Upload error:', error);
                showResult(resultContainer, 'Upload failed: ' + error.message, 'error');
                progressContainer.style.display = 'none';
            },
            () => {
                // Upload completed
                uploadTask.snapshot.ref.getDownloadURL().then((downloadURL) => {
                    console.log('File uploaded successfully:', downloadURL);
                    showResult(resultContainer, 'File uploaded successfully!', 'success', downloadURL, fileName);
                    progressContainer.style.display = 'none';
                    
                    // Clear file input
                    fileInput.value = '';
                    uploadPdfBtn.disabled = true;
                    uploadDocxBtn.disabled = true;
                    
                    // Refresh file list
                    loadFileList();
                });
            }
        );

    } catch (error) {
        console.error('Upload error:', error);
        showResult(resultContainer, 'Upload failed: ' + error.message, 'error');
        progressContainer.style.display = 'none';
    }
}

// Show result message
function showResult(container, message, type, downloadURL = null, fileName = null) {
    container.innerHTML = `
        <div class="result-message ${type}">
            <div class="result-icon">
                ${type === 'success' ? '✓' : '✗'}
            </div>
            <div class="result-content">
                <p>${message}</p>
                ${downloadURL ? `
                    <div class="result-actions">
                        <a href="${downloadURL}" target="_blank" class="download-link">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                <polyline points="7,10 12,15 17,10"></polyline>
                                <line x1="12" y1="15" x2="12" y2="3"></line>
                            </svg>
                            Download File
                        </a>
                        <button onclick="copyToClipboard('${downloadURL}')" class="copy-link">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect>
                                <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"></path>
                            </svg>
                            Copy Link
                        </button>
                    </div>
                ` : ''}
            </div>
        </div>
    `;
    container.style.display = 'block';
}

// Load file list from Firebase Storage
async function loadFileList() {
    try {
        fileList.innerHTML = '<div class="loading">Loading files...</div>';
        
        const storageRef = storage.ref();
        const listRef = storageRef.child(STORAGE_PATH);
        
        const result = await listRef.listAll();
        
        if (result.items.length === 0) {
            fileList.innerHTML = '<div class="no-files">No files found</div>';
            return;
        }

        const fileItems = await Promise.all(result.items.map(async (item) => {
            try {
                const downloadURL = await item.getDownloadURL();
                const metadata = await item.getMetadata();
                
                return {
                    name: item.name,
                    size: formatFileSize(metadata.size),
                    type: metadata.contentType,
                    lastModified: new Date(metadata.timeCreated).toLocaleDateString(),
                    downloadURL: downloadURL
                };
            } catch (error) {
                console.error('Error getting file info:', error);
                return null;
            }
        }));

        const validFiles = fileItems.filter(file => file !== null);
        
        fileList.innerHTML = validFiles.map(file => `
            <div class="file-item">
                <div class="file-info">
                    <div class="file-name">${file.name}</div>
                    <div class="file-details">
                        <span class="file-size">${file.size}</span>
                        <span class="file-type">${file.type}</span>
                        <span class="file-date">${file.lastModified}</span>
                    </div>
                </div>
                <div class="file-actions">
                    <a href="${file.downloadURL}" target="_blank" class="action-btn download">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                            <polyline points="7,10 12,15 17,10"></polyline>
                            <line x1="12" y1="15" x2="12" y2="3"></line>
                        </svg>
                    </a>
                    <button onclick="copyToClipboard('${file.downloadURL}')" class="action-btn copy">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect>
                            <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"></path>
                        </svg>
                    </button>
                </div>
            </div>
        `).join('');

    } catch (error) {
        console.error('Error loading file list:', error);
        fileList.innerHTML = '<div class="error">Failed to load files</div>';
    }
}

// Format file size
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Copy to clipboard
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        showNotification('Link copied to clipboard!', 'success');
    }).catch(err => {
        console.error('Failed to copy: ', err);
        showNotification('Failed to copy link', 'error');
    });
}

// Show notification
function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}
</script>

<style>
/* CSS Variables */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003d7a;
    --success-color: #00B894;
    --error-color: #e74c3c;
    --warning-color: #f39c12;
    --text-primary: #2c3e50;
    --text-secondary: #7f8c8d;
    --bg-primary: #f8f9fa;
    --bg-white: #ffffff;
    --border-color: #e9ecef;
    --shadow: 0 2px 10px rgba(0,0,0,0.1);
    --radius: 8px;
}

/* Main Container */
.document-upload-container {
    min-height: 100vh;
    background: var(--bg-primary);
    padding: 20px;
}

/* Header */
.upload-header {
    background: var(--bg-white);
    border-radius: var(--radius);
    box-shadow: var(--shadow);
    margin-bottom: 30px;
    padding: 30px;
}

.page-title h1 {
    color: var(--text-primary);
    font-size: 2.5rem;
    margin: 0 0 10px 0;
    font-weight: 700;
}

.page-title p {
    color: var(--text-secondary);
    font-size: 1.1rem;
    margin: 0;
}

/* Upload Section */
.upload-section {
    max-width: 1200px;
    margin: 0 auto;
}

.upload-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 30px;
}

/* Upload Cards */
.upload-card {
    background: var(--bg-white);
    border-radius: var(--radius);
    box-shadow: var(--shadow);
    padding: 30px;
    transition: transform 0.2s ease;
}

.upload-card:hover {
    transform: translateY(-2px);
}

.upload-header h3 {
    color: var(--text-primary);
    font-size: 1.5rem;
    margin: 0 0 10px 0;
    font-weight: 600;
}

.upload-header p {
    color: var(--text-secondary);
    margin: 0 0 20px 0;
}

/* Upload Content */
.upload-content {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.file-input {
    padding: 12px;
    border: 2px dashed var(--border-color);
    border-radius: var(--radius);
    background: var(--bg-primary);
    cursor: pointer;
    transition: all 0.2s ease;
}

.file-input:hover {
    border-color: var(--primary-color);
    background: #f0f8ff;
}

.upload-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    padding: 15px 25px;
    border: none;
    border-radius: var(--radius);
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.pdf-btn {
    background: var(--error-color);
    color: white;
}

.pdf-btn:hover:not(:disabled) {
    background: #c0392b;
    transform: translateY(-2px);
}

.docx-btn {
    background: var(--primary-color);
    color: white;
}

.docx-btn:hover:not(:disabled) {
    background: var(--primary-hover);
    transform: translateY(-2px);
}

.upload-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    transform: none;
}

.refresh-btn {
    background: var(--success-color);
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius);
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 500;
    transition: all 0.2s ease;
}

.refresh-btn:hover {
    background: #00a085;
    transform: translateY(-1px);
}

/* Progress Bar */
.progress-container {
    margin-top: 15px;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: var(--border-color);
    border-radius: 4px;
    overflow: hidden;
    margin-bottom: 10px;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, var(--success-color), #00a085);
    width: 0%;
    transition: width 0.3s ease;
}

.progress-text {
    font-size: 0.9rem;
    color: var(--text-secondary);
    font-weight: 500;
}

/* Result Container */
.result-container {
    margin-top: 15px;
}

.result-message {
    display: flex;
    align-items: flex-start;
    gap: 15px;
    padding: 15px;
    border-radius: var(--radius);
    border-left: 4px solid;
}

.result-message.success {
    background: #d4edda;
    border-left-color: var(--success-color);
    color: #155724;
}

.result-message.error {
    background: #f8d7da;
    border-left-color: var(--error-color);
    color: #721c24;
}

.result-icon {
    font-size: 1.2rem;
    font-weight: bold;
}

.result-content p {
    margin: 0 0 10px 0;
    font-weight: 500;
}

.result-actions {
    display: flex;
    gap: 10px;
    margin-top: 10px;
}

.download-link, .copy-link {
    display: flex;
    align-items: center;
    gap: 5px;
    padding: 8px 12px;
    background: var(--primary-color);
    color: white;
    text-decoration: none;
    border-radius: 4px;
    font-size: 0.9rem;
    transition: all 0.2s ease;
}

.download-link:hover, .copy-link:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
}

/* File List */
.file-list {
    margin-top: 20px;
}

.file-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius);
    margin-bottom: 10px;
    background: var(--bg-primary);
    transition: all 0.2s ease;
}

.file-item:hover {
    background: #e8f4fd;
    border-color: var(--primary-color);
}

.file-info {
    flex: 1;
}

.file-name {
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 5px;
}

.file-details {
    display: flex;
    gap: 15px;
    font-size: 0.9rem;
    color: var(--text-secondary);
}

.file-actions {
    display: flex;
    gap: 8px;
}

.action-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 36px;
    height: 36px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-btn.download {
    background: var(--success-color);
    color: white;
}

.action-btn.download:hover {
    background: #00a085;
    transform: translateY(-1px);
}

.action-btn.copy {
    background: var(--primary-color);
    color: white;
}

.action-btn.copy:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
}

/* Loading and Error States */
.loading, .no-files, .error {
    text-align: center;
    padding: 40px;
    color: var(--text-secondary);
    font-style: italic;
}

.error {
    color: var(--error-color);
}

/* Notifications */
.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    padding: 15px 20px;
    border-radius: var(--radius);
    color: white;
    font-weight: 500;
    z-index: 1000;
    animation: slideIn 0.3s ease;
}

.notification.success {
    background: var(--success-color);
}

.notification.error {
    background: var(--error-color);
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

/* Responsive Design */
@media (max-width: 768px) {
    .document-upload-container {
        padding: 10px;
    }
    
    .upload-container {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .upload-card {
        padding: 20px;
    }
    
    .file-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 10px;
    }
    
    .file-actions {
        align-self: flex-end;
    }
}
</style>
@endsection
