@extends('welcome')

@section('title', 'Job Card')

@section('styles')
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="{{ asset('css/Inspection/job_card.blade.css') }}" rel="stylesheet">

@endsection

@section('content')
<div class="inspection2-container">
    <!-- Header Section -->
    <div class="inspection2-header">
        <div class="header-content">
            <div class="back-section">
                <a href="{{ url('/inspection') }}" class="back-link">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M19 12H5M12 19l-7-7 7-7"/>
                    </svg>
                    Back to Job Card Report
                </a>
            </div>
            <div class="page-title">
                <h1>Job Card</h1>
                <p>Complete your job card information</p>
            </div>
        </div>
    </div>


    <!-- Step 1 Form -->
    <div class="form-section" id="step1-form">
        <div class="form-header">
            <h2>1. Job Card Information</h2>
            <p>Fill in the basic information for your job card</p>
        </div>
        
        <form id="job-card-form">
            @csrf
            <div class="form-grid">
                <!-- Left Column: Job Card Info -->
                <div class="form-column">
                    <div class="form-group">
                        <label for="jobCardNumber" class="form-label">Job Card #</label>
                        <b>ZAM-KOB-JC-1002</b>
                    </div>
                    <div class="form-group">
                        <label for="jobCardTitle" class="form-label required">Job Card Title *</label>
                        <input type="text" class="form-control" id="jobCardTitle" name="jobCardTitle" value="Job Card for ZAM-KOB-IR-116" required>
                    </div>
                    <div class="form-group">
                        <label for="country" class="form-label required">Country *</label>
                        <select class="form-control" id="country" name="country" required>
                            <option value="">Select Country</option>
                            <option value="Zambia" selected>Zambia</option>
                            <option value="DRC">DRC</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="project" class="form-label required">Project *</label>
                        <select class="form-control" id="project" name="project" required>
                            <option value="">Select Project</option>
                            <option value="Kobold" selected>Kobold</option>
                            <option value="Lusaka">Lusaka</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="memo" class="form-label">Write Memo</label>
                        <input type="text" class="form-control" id="memo" name="memo" placeholder="Check within a day">
                    </div>
                </div>
                
                <!-- Right Column: Priority and Equipment -->
                <div class="form-column">
                    <div class="form-group">
                        <label class="form-label">Priority</label>
                        <div class="priority-buttons">
                            <button type="button" class="priority-btn" data-priority="low">
                                <i class="fas fa-fire" style="color: #dc2626;"></i>
                                <span>Low</span>
                            </button>
                            <button type="button" class="priority-btn active" data-priority="medium">
                                <i class="fas fa-fire" style="color: #f97316;"></i>
                                <span>Medium</span>
                            </button>
                            <button type="button" class="priority-btn" data-priority="high">
                                <i class="fas fa-fire" style="color: #10b981;"></i>
                                <span>High</span>
                            </button>
                            <button type="button" class="priority-btn" data-priority="emergency">
                                <i class="fas fa-fire" style="color: #dc2626;"></i>
                                <span>Emergency</span>
                            </button>
                        </div>
                        <input type="hidden" id="priority" name="priority" value="medium">
                    </div>
                    <div class="form-group">
                        <label class="form-label required">Equipment *</label>
                        <div class="input-with-button">
                            <input type="text" class="form-control" id="equipmentDisplay" name="equipmentDisplay" placeholder="Select Equipment" readonly required>
                            <button type="button" class="btn-select" onclick="openEquipmentModal()" title="Select Equipment">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                                    <polyline points="3.27,6.96 12,12.01 20.73,6.96"/>
                                    <line x1="12" y1="22.08" x2="12" y2="12"/>
                                </svg>
                            </button>
                        </div>
                        <input type="hidden" id="vehicleNumber" name="vehicleNumber" required>
                        <input type="hidden" id="vehicleName" name="vehicleName" required>
                        <input type="hidden" id="vinNumber" name="vinNumber" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Assign To</label>
                        <div class="assign-to-section">
                            <div class="assign-to-display">
                                <span>No users selected</span>
                                <a href="#" class="select-link" onclick="openUserModal()">Select ></a>
                            </div>
                            <div class="assign-to-placeholder">
                                Tap 'Select' to choose users to assign
                            </div>
                        </div>
                        <input type="hidden" id="assignedUsers" name="assignedUsers">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Start Date</label>
                        <div class="date-input-group">
                            <input type="text" class="form-control date-input" id="startDate" name="startDate" placeholder="Select Date" readonly>
                            <button type="button" class="date-btn" onclick="openDatePicker('startDate')" title="Select Date">
                                <i class="fas fa-calendar"></i>
                            </button>
                        </div>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Due Date</label>
                        <div class="date-input-group">
                            <input type="text" class="form-control date-input" id="dueDate" name="dueDate" placeholder="Select Date" readonly>
                            <button type="button" class="date-btn" onclick="openDatePicker('dueDate')" title="Select Date">
                                <i class="fas fa-calendar"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Collapsible Sections -->
            <div class="collapsible-sections">
                <div class="collapsible-section">
                    <div class="collapsible-header" onclick="toggleCollapsible('additionalDetails')">
                        <span>Additional Details</span>
                        <i class="fas fa-chevron-down"></i>
                    </div>
                    <div class="collapsible-content" id="additionalDetails">
                        <div class="form-group">
                            <label class="form-label">Job Card Type</label>
                            <div class="select-field" onclick="openJobCardTypeModal()">
                                <span id="jobCardTypeDisplay">General</span>
                                <i class="fas fa-chevron-right"></i>
                            </div>
                            <input type="hidden" id="jobCardType" name="jobCardType" value="general">
                        </div>
                        
                        <div class="form-group">
                            <label for="estimatedCost" class="form-label">Estimated Cost</label>
                            <input type="number" class="form-control" id="estimatedCost" name="estimatedCost" placeholder="Enter estimated cost" step="0.01">
                        </div>
                        
                        <div class="form-group">
                            <label for="laborTime" class="form-label">Labor Time</label>
                            <input type="number" class="form-control" id="laborTime" name="laborTime" placeholder="Enter labor time in hours" step="0.1">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Link Inspection</label>
                            <div class="select-field" onclick="openInspectionModal()">
                                <span id="inspectionDisplay">Select Inspection</span>
                                <i class="fas fa-chevron-right"></i>
                            </div>
                            <input type="hidden" id="linkedInspection" name="linkedInspection" value="">
                            <div class="inspection-actions" style="margin-top: 8px; display: flex; gap: 8px;">
                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="refreshTasksFromInspection()" title="Refresh tasks from inspection">
                                    <i class="fas fa-sync-alt"></i> Refresh Tasks
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-info" onclick="createTasksFromLinkedInspection()" title="Create tasks from this inspection">
                                    <i class="fas fa-plus"></i> Create Tasks
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-warning" onclick="debugInspectionTasks()" title="Debug inspection tasks">
                                    <i class="fas fa-bug"></i> Debug Tasks
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-success" onclick="testFaultyItemsToTasks()" title="Test task creation from faulty items">
                                    <i class="fas fa-play"></i> Test Tasks
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="testFirebaseAPI()" title="Test Firebase API directly">
                                    <i class="fas fa-database"></i> Test API
                                </button>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Equipment/Date</label>
                            <div class="equipment-date-display">
                                <span>Equipment: <span id="equipmentDisplayText">N/A</span> | Date: <span id="dateDisplayText">N/A</span></span>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Job Card</label>
                            <div class="select-field" onclick="openJobCardSelectionModal()">
                                <span id="jobCardDisplay">Select</span>
                                <i class="fas fa-chevron-right"></i>
                            </div>
                            <input type="hidden" id="selectedJobCard" name="selectedJobCard" value="">
                        </div>
                        
                        <div class="form-group">
                            <label for="additionalNotes" class="form-label">Additional Notes</label>
                            <textarea class="form-control" id="additionalNotes" name="additionalNotes" rows="3" placeholder="Enter any additional details..."></textarea>
                        </div>
                    </div>
                </div>
                
                <div class="collapsible-section">
                    <div class="collapsible-header" onclick="toggleCollapsible('costDetails')">
                        <span>Cost Details</span>
                        <i class="fas fa-chevron-down"></i>
                    </div>
                    <div class="collapsible-content" id="costDetails">
                        <div class="form-group">
                            <label for="partMaterialCost" class="form-label">Part & Material Cost</label>
                            <input type="number" class="form-control" id="partMaterialCost" name="partMaterialCost" placeholder="0.00" step="0.01" value="0.00">
                        </div>
                        
                        <div class="form-group">
                            <label for="totalLaborCost" class="form-label">Total Labor Cost</label>
                            <input type="number" class="form-control" id="totalLaborCost" name="totalLaborCost" placeholder="0.00" step="0.01" value="0.00">
                        </div>
                        
                        <div class="form-group">
                            <label for="additionalCost" class="form-label">Additional Cost</label>
                            <input type="number" class="form-control" id="additionalCost" name="additionalCost" placeholder="0.00" step="0.01" value="0.00">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="taxAmount" class="form-label">Tax Amount</label>
                                <input type="number" class="form-control" id="taxAmount" name="taxAmount" placeholder="0.00" step="0.01" value="0.00">
                            </div>
                            <div class="form-group">
                                <label for="taxPercentage" class="form-label">Tax(%)</label>
                                <input type="number" class="form-control" id="taxPercentage" name="taxPercentage" placeholder="0.00" step="0.01" value="0.00">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="totalWOCost" class="form-label">Total WO Cost</label>
                            <input type="number" class="form-control" id="totalWOCost" name="totalWOCost" placeholder="0.00" step="0.01" value="0.00" readonly>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Tasks & Items Section -->
            <div class="tasks-section">
                <div class="tasks-header">
                    <h3>Tasks & Items</h3>
                    <div class="task-tabs">
                        <button type="button" class="task-tab active" data-tab="task">Task</button>
                        <button type="button" class="task-tab" data-tab="parts">Parts</button>
                        <button type="button" class="task-tab" data-tab="labor">Labor</button>
                        <button type="button" class="task-tab" data-tab="additional">Additional Co</button>
                    </div>
                </div>
                
                <div class="task-content">
                    <!-- Task Tab Content -->
                    <div id="task-tab-content" class="tab-content active">
                        <div class="task-list">
                            <div class="no-items" style="text-align: center; color: #6b7280; padding: 20px; font-style: italic;">
                                No tasks added yet. Select an inspection or add tasks manually.
                            </div>
                        </div>
                        
                        <button type="button" class="add-task-btn" onclick="addNewTask()">
                            <i class="fas fa-plus"></i>
                            Add Task
                        </button>
                    </div>

                    <!-- Parts Tab Content -->
                    <div id="parts-tab-content" class="tab-content">
                        <div class="parts-list">
                            <p class="no-items">No Parts added yet.</p>
                        </div>

                        <button type="button" class="add-task-btn" onclick="addNewPart()">
                            <i class="fas fa-plus"></i>
                            Add Parts
                        </button>
                    </div>

                    <!-- Labor Tab Content -->
                    <div id="labor-tab-content" class="tab-content">
                        <div class="labor-list">
                            <p class="no-items">No labor items added yet.</p>
                        </div>
                        
                        <button type="button" class="add-task-btn" onclick="addNewLabor()">
                            <i class="fas fa-plus"></i>
                            Add Labor
                        </button>
                    </div>

                    <!-- Additional Co Tab Content -->
                    <div id="additional-tab-content" class="tab-content">
                        <div class="additional-list">
                            <p class="no-items">No additional costs added yet.</p>
                        </div>
                        
                        <button type="button" class="add-task-btn" onclick="addNewAdditional()">
                            <i class="fas fa-plus"></i>
                            Add Additional Co
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- GPS Location Section -->
            <div class="gps-section">
                <div class="form-group">
                    <label class="form-label">GPS Location</label>
                    <div class="gps-controls">
                        <button type="button" class="btn-gps" onclick="handleGetLocation()">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/>
                                <circle cx="12" cy="10" r="3"/>
                            </svg>
                            Get Location
                        </button>
                    </div>
                    <div class="gps-details" id="gpsDetailsBox" style="display: none;">
                        <div class="gps-info-grid">
                            <div class="gps-info-item">
                                <span class="gps-label">Coordinates:</span>
                                <span class="gps-value" id="coordinates">Not available</span>
                            </div>
                            <div class="gps-info-item">
                                <span class="gps-label">GPS Address:</span>
                                <span class="gps-value" id="gpsAddress">Not available</span>
                            </div>
                        </div>
                    </div>
                    <input type="hidden" id="coordinatesInput" name="coordinates">
                    <input type="hidden" id="gpsAddressInput" name="gpsAddress">
                </div>
            </div>
            
            <div class="form-actions">
                <button type="button" class="btn-next" onclick="goToStep2()">
                    <span>Create</span>
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M5 12h14M12 5l7 7-7 7"/>
                    </svg>
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Date Picker Modal -->
<div id="datePickerModal" class="date-picker-modal" style="display: none;">
    <div class="date-picker-content">
        <div class="date-picker-header">
            <h3>Select Date</h3>
            <button type="button" class="close-date-picker" onclick="closeDatePicker()">&times;</button>
        </div>
        <div class="date-picker-body">
            <div id="calendarContainer"></div>
        </div>
        <div class="date-picker-footer">
            <button type="button" class="btn btn-secondary" onclick="closeDatePicker()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="selectToday()">Today</button>
        </div>
    </div>
</div>

<!-- Add Parts Modal -->
<div id="addPartsModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Parts</h3>
            <button type="button" class="close-btn" onclick="closeAddPartsModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="custom-modal-body">
            <form id="addPartsForm">
                <div class="form-group">
                    <label class="form-label required">Item Number *</label>
                    <input type="text" class="form-control" id="itemNumber" name="itemNumber" placeholder="Enter item number" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Part Number *</label>
                    <input type="text" class="form-control" id="partNumber" name="partNumber" placeholder="Enter part number" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Item Name *</label>
                    <input type="text" class="form-control" id="itemName" name="itemName" placeholder="Enter item name" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label required">Quantity *</label>
                        <input type="number" class="form-control" id="quantity" name="quantity" placeholder="0" min="0" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Unit</label>
                        <select class="form-control" id="unit" name="unit">
                            <option value="">Select unit</option>
                            <option value="pcs">Pieces</option>
                            <option value="kg">Kilograms</option>
                            <option value="liters">Liters</option>
                            <option value="meters">Meters</option>
                            <option value="hours">Hours</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Item Cost</label>
                    <input type="number" class="form-control" id="itemCost" name="itemCost" placeholder="0.00" min="0" step="0.01">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Site</label>
                    <input type="text" class="form-control" id="site" name="site" placeholder="Enter site location">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="note" name="note" rows="3" placeholder="Enter additional notes..."></textarea>
                </div>
            </form>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeAddPartsModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="savePart()">Add</button>
        </div>
    </div>
</div>

<!-- Add Labor Modal -->
<div id="addLaborModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Labor Details</h3>
            <button type="button" class="close-btn" onclick="closeAddLaborModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="custom-modal-body">
            <form id="addLaborForm">
                <div class="form-group">
                    <label class="form-label required">Worker Name *</label>
                    <input type="text" class="form-control" id="workerName" name="workerName" placeholder="Worker Name" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Labor Code *</label>
                    <input type="text" class="form-control" id="laborCode" name="laborCode" placeholder="Labor Code" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Rate *</label>
                    <input type="number" class="form-control" id="laborRate" name="laborRate" placeholder="Rate" step="0.01" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Work Hours *</label>
                    <div class="form-row">
                        <div class="form-group">
                            <input type="number" class="form-control" id="workHours" name="workHours" placeholder="Hour" min="0" required>
                        </div>
                        <div class="form-group">
                            <input type="number" class="form-control" id="workMinutes" name="workMinutes" placeholder="Minu" min="0" max="59" required>
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="laborNote" name="laborNote" placeholder="Note" rows="3"></textarea>
                </div>
            </form>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeAddLaborModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="saveLabor()">Save</button>
        </div>
    </div>
</div>

<!-- Add Additional Cost Modal -->
<div id="addAdditionalModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Additional Cost</h3>
            <button type="button" class="close-btn" onclick="closeAddAdditionalModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="custom-modal-body">
            <form id="addAdditionalForm">
                <div class="form-group">
                    <label class="form-label required">Description *</label>
                    <textarea class="form-control" id="additionalDescription" name="additionalDescription" placeholder="Description" rows="4" required></textarea>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Select Labor Code *</label>
                    <select class="form-control" id="laborCodeSelect" name="laborCodeSelect" required>
                        <option value="">Select Labor Code</option>
                        <option value="LAB001">LAB001 - General Labor</option>
                        <option value="LAB002">LAB002 - Skilled Labor</option>
                        <option value="LAB003">LAB003 - Specialized Labor</option>
                        <option value="LAB004">LAB004 - Overtime Labor</option>
                        <option value="LAB005">LAB005 - Emergency Labor</option>
                    </select>
                </div>
            </form>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeAddAdditionalModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="saveAdditional()">Save</button>
        </div>
    </div>
</div>

<!-- Add Task Modal -->
<div id="addTaskModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Task</h3>
            <button type="button" class="close-btn" onclick="closeAddTaskModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="custom-modal-body">
            <form id="addTaskForm">
                <div class="form-group">
                    <label class="form-label required">Task *</label>
                    <input type="text" class="form-control" id="taskName" name="taskName" placeholder="Enter task name" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Type *</label>
                    <div class="type-buttons">
                        <button type="button" class="type-btn active" data-type="repair" onclick="selectTaskType('repair')">
                            Repair
                        </button>
                        <button type="button" class="type-btn" data-type="replace" onclick="selectTaskType('replace')">
                            Replace
                        </button>
                    </div>
                    <input type="hidden" id="taskType" name="taskType" value="repair">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Assign to</label>
                    <div class="assign-to-display">
                        <span>No users selected</span>
                        <a href="#" class="select-link" onclick="openUserModal()">Set ></a>
                    </div>
                    <input type="hidden" id="assignedUsers" name="assignedUsers">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="taskNote" name="taskNote" placeholder="Note" rows="3"></textarea>
                </div>
            </form>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeAddTaskModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="saveTask()">Save</button>
        </div>
    </div>
</div>

<!-- Edit Task Modal -->
<div id="editTaskModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Edit Task</h3>
            <button type="button" class="close-btn" onclick="closeEditTaskModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="custom-modal-body">
            <form id="editTaskForm">
                <div class="form-group">
                    <label class="form-label required">Task *</label>
                    <input type="text" class="form-control" id="editTaskName" name="editTaskName" placeholder="Enter task name" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label required">Type *</label>
                    <div class="type-buttons">
                        <button type="button" class="type-btn active" data-type="repair" onclick="selectEditTaskType('repair')">
                            Repair
                        </button>
                        <button type="button" class="type-btn" data-type="replace" onclick="selectEditTaskType('replace')">
                            Replace
                        </button>
                    </div>
                    <input type="hidden" id="editTaskType" name="editTaskType" value="repair">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Assign to</label>
                    <div class="assign-to-display">
                        <span>No users selected</span>
                        <a href="#" class="select-link" onclick="openUserModal()">Set ></a>
                    </div>
                    <input type="hidden" id="editAssignedUsers" name="editAssignedUsers">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="editTaskNote" name="editTaskNote" placeholder="Note" rows="3"></textarea>
                </div>
            </form>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeEditTaskModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="updateTask()">Save</button>
        </div>
    </div>
</div>

<!-- Job Card Type Selection Modal -->
<div id="jobCardTypeModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Job Card Type</h3>
            <button type="button" class="close-btn" onclick="closeJobCardTypeModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="custom-modal-body">
            <div class="job-card-type-options">
                <div class="job-card-type-option" onclick="selectJobCardType('general')">
                    <span>General</span>
                </div>
                <div class="job-card-type-option" onclick="selectJobCardType('maintenance')">
                    <span>Maintenance</span>
                </div>
                <div class="job-card-type-option" onclick="selectJobCardType('repair')">
                    <span>Repair</span>
                </div>
                <div class="job-card-type-option" onclick="selectJobCardType('installation')">
                    <span>Installation</span>
                </div>
            </div>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-primary" onclick="closeJobCardTypeModal()">Close</button>
        </div>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="equipment-list" id="equipmentList">
                <!-- Equipment items will be loaded here -->
            </div>
        </div>
    </div>
</div>

<!-- Inspection Selection Modal -->
<div id="inspectionModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Inspection Report</h3>
            <div class="modal-header-actions">
                <button type="button" class="btn btn-sm btn-outline-primary" onclick="refreshInspectionModal()" title="Refresh inspection data">
                    <i class="fas fa-sync-alt"></i> Refresh
                </button>
                <button type="button" class="btn btn-sm btn-outline-info" onclick="debugInspectionData()" title="Debug inspection data">
                    <i class="fas fa-bug"></i> Debug
                </button>
                <button type="button" class="btn btn-sm btn-outline-success" onclick="testFaultyItemsToTasks()" title="Test faulty items to tasks">
                    <i class="fas fa-cogs"></i> Test Tasks
                </button>
                <button type="button" class="close-btn" onclick="closeInspectionModal()">&times;</button>
            </div>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="inspectionSearch" class="search-input" placeholder="Search inspections..." onkeyup="filterInspections()">
                <button type="button" class="search-btn" onclick="filterInspections()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="inspection-list-container">
                <div class="inspection-count" id="inspectionCount" style="margin-bottom: 10px; color: #64748b; font-size: 12px;"></div>
                <div class="inspection-list" id="inspectionList" style="max-height: 50vh; overflow-y: auto;">
                    <!-- Inspection items will be loaded here -->
                </div>
            </div>
        </div>
        <div class="custom-modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeInspectionModal()">Close</button>
        </div>
    </div>
</div>

<script>
// JavaScript functions for job card functionality

// Priority selection
document.addEventListener('DOMContentLoaded', function() {
    const priorityButtons = document.querySelectorAll('.priority-btn');
    priorityButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            priorityButtons.forEach(btn => btn.classList.remove('active'));
            // Add active class to clicked button
            this.classList.add('active');
            // Update hidden input
            document.getElementById('priority').value = this.dataset.priority;
        });
    });

    // Task tabs
    const taskTabs = document.querySelectorAll('.task-tab');
    taskTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const tabName = this.getAttribute('data-tab');
            
            // Remove active class from all tabs
            taskTabs.forEach(t => t.classList.remove('active'));
            // Add active class to clicked tab
            this.classList.add('active');
            
            // Hide all tab content
            const allTabContent = document.querySelectorAll('.tab-content');
            allTabContent.forEach(content => content.classList.remove('active'));
            
            // Show selected tab content
            const selectedContent = document.getElementById(tabName + '-tab-content');
            if (selectedContent) {
                selectedContent.classList.add('active');
            }
        });
    });
    
    // Load saved data from localStorage
    console.log('Loading all saved data...');
    loadTasksFromStorage();
    loadPartsFromStorage();
    loadLaborFromStorage();
    loadAdditionalFromStorage();
    console.log('All data loading completed');

    // Check if there's already a linked inspection and create tasks from it
    const linkedInspection = document.getElementById('linkedInspection');
    const inspectionDisplay = document.getElementById('inspectionDisplay');

    if (linkedInspection && linkedInspection.value && linkedInspection.value !== '' && linkedInspection.value !== 'Select Inspection') {
        console.log('Found existing linked inspection:', linkedInspection.value);
        console.log('Inspection display:', inspectionDisplay?.textContent);

        // Update the display if needed
        if (inspectionDisplay && inspectionDisplay.textContent !== linkedInspection.value) {
            inspectionDisplay.textContent = linkedInspection.value;
        }

        // Create tasks from the inspection
        checkAndCreateTasksFromInspection(linkedInspection.value);
    } else {
        console.log('No linked inspection found on page load - using default "Select Inspection"');
        // Ensure default display
        if (inspectionDisplay) inspectionDisplay.textContent = 'Select Inspection';
        if (linkedInspection) linkedInspection.value = '';
    }

    // Test inspection API on page load (for debugging)
    testInspectionAPI();
});

// Global variables for equipment and inspection data
let equipmentData = [];
let inspectionData = [];

// Modal functions
function openEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
    loadEquipmentData();
}

function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
}

// Load equipment data from Firebase
async function loadEquipmentData() {
    try {
        // Show loading state
        const equipmentList = document.getElementById('equipmentList');
        equipmentList.innerHTML = '<div class="text-center p-3"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Loading equipment...</p></div>';

        // Fetch equipment data from Firebase API
        const response = await fetch('/api/fetch-equipment-from-firebase', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.success && result.equipment) {
            equipmentData = result.equipment;
            renderEquipmentList(equipmentData);
            console.log('Equipment data loaded successfully:', result.count + ' items');
        } else {
            throw new Error(result.message || 'Failed to load equipment data');
        }
    } catch (error) {
        console.error('Error loading equipment:', error);
        const equipmentList = document.getElementById('equipmentList');
        equipmentList.innerHTML = '<p class="text-center text-muted p-3">Error loading equipment data. Please try again.</p>';
        showErrorAlert('Equipment Loading Error', 'Failed to load equipment data from Firebase: ' + error.message);
    }
}

// Render equipment list
function renderEquipmentList(equipment) {
    const container = document.getElementById('equipmentList');
    
    if (!equipment || equipment.length === 0) {
        container.innerHTML = '<p class="text-center text-muted p-3">No equipment found</p>';
        return;
    }
    
    const html = equipment.map(item => `
        <div class="equipment-item" onclick="selectEquipment(${JSON.stringify(item).replace(/"/g, '&quot;')})">
            <div class="equipment-name">${item.equipmentName || item.name || 'Unknown Equipment'}</div>
            <div class="equipment-details">
                Model: ${item.model || 'N/A'}<br>
                Category: ${item.mainCategory || item.equipmentCategory || 'N/A'}<br>
                ${item.status ? `Status: ${item.status}` : ''}
                ${item.meterReading ? `<br>Meter: ${item.meterReading} ${item.meterUnit || 'KM'}` : ''}
            </div>
        </div>
    `).join('');
    
    container.innerHTML = html;
}

// Select equipment from Firebase data
function selectEquipment(equipmentData) {
    // Handle both old format (id, name, model, category) and new Firebase format
    if (typeof equipmentData === 'object' && equipmentData !== null) {
        // New Firebase format
        const id = equipmentData.id || equipmentData.firebaseKey || equipmentData.vehicleNumber;
        const name = equipmentData.equipmentName || equipmentData.name || 'Unknown Equipment';
        const model = equipmentData.model || 'N/A';
        const category = equipmentData.mainCategory || equipmentData.equipmentCategory || 'N/A';

        document.getElementById('equipmentDisplay').value = `${name} - ${model} (${category})`;
        document.getElementById('vehicleNumber').value = id;
        document.getElementById('vehicleName').value = name;
        document.getElementById('vinNumber').value = model;

        // Store additional equipment data for later use
        if (equipmentData.meterReading) {
            document.getElementById('equipmentDisplay').setAttribute('data-meter-reading', equipmentData.meterReading);
        }
        if (equipmentData.meterUnit) {
            document.getElementById('equipmentDisplay').setAttribute('data-meter-unit', equipmentData.meterUnit);
        }
    } else {
        // Fallback for old format (backward compatibility)
        const id = arguments[0];
        const name = arguments[1];
        const model = arguments[2];
        const category = arguments[3];

        document.getElementById('equipmentDisplay').value = `${name} - ${model} (${category})`;
        document.getElementById('vehicleNumber').value = id;
        document.getElementById('vehicleName').value = name;
        document.getElementById('vinNumber').value = model;
    }

    closeEquipmentModal();
}

// Filter equipment from Firebase data
function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    const filtered = equipmentData.filter(item => {
        const equipmentName = (item.equipmentName || item.name || '').toLowerCase();
        const model = (item.model || '').toLowerCase();
        const mainCategory = (item.mainCategory || item.equipmentCategory || '').toLowerCase();
        const vehicleNumber = (item.vehicleNumber || item.id || '').toLowerCase();

        return equipmentName.includes(searchTerm) ||
               model.includes(searchTerm) ||
               mainCategory.includes(searchTerm) ||
               vehicleNumber.includes(searchTerm);
    });
    renderEquipmentList(filtered);
}

// Inspection Modal Functions
function openInspectionModal() {
    const modal = document.getElementById('inspectionModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
    loadInspectionData();

    // Clear any cached data to ensure fresh inspection data
    if ('caches' in window) {
        caches.keys().then(names => {
            names.forEach(name => {
                caches.delete(name);
            });
        });
    }

    // Force refresh of the page data
    if (window.localStorage) {
        localStorage.removeItem('inspectionDataCache');
    }
}

function closeInspectionModal() {
    const modal = document.getElementById('inspectionModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
}

// Refresh inspection modal data
function refreshInspectionModal() {
    console.log('Refreshing inspection modal data...');

    // Clear inspection data cache
    inspectionData = [];

    // Clear browser cache for API calls
    if ('caches' in window) {
        caches.keys().then(names => {
            names.forEach(name => {
                if (name.includes('api') || name.includes('fetch-inspection')) {
                    caches.delete(name);
                }
            });
        });
    }

    // Clear local storage cache
    if (window.localStorage) {
        localStorage.removeItem('inspectionDataCache');
        localStorage.removeItem('inspectionData');
    }

    // Force reload inspection data
    loadInspectionData();
    showInfoAlert('Refreshing Data', 'Fetching latest inspection data...');
}

// Load inspection data from Firebase
async function loadInspectionData() {
    try {
        // Show loading state
        const inspectionList = document.getElementById('inspectionList');
        inspectionList.innerHTML = '<div class="text-center p-3"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Loading inspections...</p></div>';

        // Fetch inspection data from Firebase API with enhanced cache busting
        const timestamp = Date.now();
        const cacheBuster = Math.random().toString(36).substring(7);
        const response = await fetch(`/api/fetch-inspection-reports?_t=${timestamp}&_cb=${cacheBuster}`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'Cache-Control': 'no-cache, no-store, must-revalidate, max-age=0',
                'Pragma': 'no-cache',
                'Expires': '0',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.success && result.inspections) {
            inspectionData = result.inspections;
            console.log('Raw inspection data from API:', result.inspections);
            console.log('Sample inspection item:', result.inspections[0]);
            renderInspectionList(inspectionData);
            console.log('Inspection data loaded successfully:', result.count + ' items');
        } else {
            throw new Error(result.message || 'Failed to load inspection data');
        }
    } catch (error) {
        console.error('Error loading inspections:', error);
        const inspectionList = document.getElementById('inspectionList');
        inspectionList.innerHTML = '<p class="text-center text-muted p-3">Error loading inspection data. Please try again.</p>';
        showErrorAlert('Inspection Loading Error', 'Failed to load inspection data from Firebase: ' + error.message);
    }
}

// Render inspection list
function renderInspectionList(inspections) {
    const container = document.getElementById('inspectionList');
    const countElement = document.getElementById('inspectionCount');

    if (!inspections || inspections.length === 0) {
        if (countElement) countElement.textContent = 'No inspections found';
        container.innerHTML = '<p class="text-center text-muted p-3">No inspections found</p>';
        return;
    }

    // Update count with detailed information
    if (countElement) {
        const inspectionsWithFaulty = inspections.filter(inspection => inspection.hasFaultyItems);
        const totalFaultyItems = inspectionsWithFaulty.reduce((total, inspection) => total + (inspection.faultyItemsCount || 0), 0);

        countElement.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <span>${inspections.length} inspection${inspections.length === 1 ? '' : 's'} found</span>
                ${inspectionsWithFaulty.length > 0 ?
                    `<span style="color: #dc2626; font-weight: 600;">
                        ⚠️ ${inspectionsWithFaulty.length} with ${totalFaultyItems} faulty item${totalFaultyItems === 1 ? '' : 's'}
                    </span>` : ''
                }
            </div>
        `;
    }

    const html = inspections.map(inspection => {
        // Enhanced debug logging for each inspection
        console.log('Processing inspection for modal:', {
            requestNumber: inspection.requestNumber,
            firebaseKey: inspection.firebaseKey,
            inspectionNumber: inspection.inspectionNumber,
            hasFaultyItems: inspection.hasFaultyItems,
            faultyItemsCount: inspection.faultyItemsCount,
            equipment: inspection.equipment,
            createdAt: inspection.createdAt,
            country: inspection.country,
            project: inspection.project,
            inspector: inspection.inspector
        });

        // Determine display values
        const displayNumber = inspection.inspectionNumber || inspection.requestNumber || inspection.firebaseKey || 'N/A';
        const faultyCount = inspection.faultyItemsCount || 0;
        const hasFaults = inspection.hasFaultyItems && faultyCount > 0;

        // Create class list for the inspection item
        const itemClasses = ['inspection-item'];
        if (hasFaults) {
            itemClasses.push('has-faulty');
        }

        return `
        <div class="${itemClasses.join(' ')}" onclick="selectInspection(${JSON.stringify(inspection).replace(/"/g, '&quot;')})">
            <div class="inspection-header">
                <div class="inspection-number">${displayNumber}</div>
                <div class="inspection-date">${new Date(inspection.createdAt).toLocaleDateString()}</div>
            </div>
            <div class="inspection-details">
                <div class="inspection-equipment">
                    <strong>Equipment:</strong> ${inspection.equipment?.name || 'N/A'} - ${inspection.equipment?.model || 'N/A'} (${inspection.equipment?.number || 'N/A'})
                </div>
                <div class="inspection-info">
                    <span><strong>Country:</strong> ${inspection.country || 'N/A'}</span>
                    <span><strong>Project:</strong> ${inspection.project || 'N/A'}</span>
                    <span><strong>Inspector:</strong> ${inspection.inspector || 'N/A'}</span>
                </div>
                <div class="inspection-status">
                    ${hasFaults ?
                        `<span class="badge badge-warning" style="background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); color: #dc2626; border: 1px solid #fecaca; font-weight: 600;">
                            ⚠️ ${faultyCount} Faulty Item${faultyCount === 1 ? '' : 's'}
                        </span>` :
                        '<span class="badge badge-success" style="background: #d1fae5; color: #059669; border: 1px solid #a7f3d0;">✅ No Faults</span>'
                    }
                    ${hasFaults ?
                        `                    <div class="faulty-items-count" style="font-size: 11px; color: #dc2626; margin-top: 4px; font-weight: 600;">
                            <i class="fas fa-exclamation-triangle" style="margin-right: 4px;"></i>
                            ${faultyCount} faulty item${faultyCount === 1 ? '' : 's'} found - Click to auto-create tasks
                            <button type="button" class="btn btn-sm btn-danger" onclick="manualCreateTasks(event, ${JSON.stringify(inspection).replace(/"/g, '&quot;')})"
                                style="margin-left: 8px; padding: 2px 6px; font-size: 10px;">
                                Create Tasks
                            </button>
                        <button type="button" class="btn btn-sm btn-info" onclick="debugFirebaseData('${inspection.firebaseKey}')"
                            style="margin-left: 4px; padding: 2px 6px; font-size: 10px;">
                            Debug Data
                        </button>
                        </div>` : ''
                    }
                </div>
            </div>
        </div>
    `}).join('');

    container.innerHTML = html;
}

// Filter inspections
function filterInspections() {
    const searchTerm = document.getElementById('inspectionSearch').value.toLowerCase();
    const filtered = inspectionData.filter(inspection => {
        const inspectionNumber = (inspection.inspectionNumber || inspection.requestNumber || inspection.firebaseKey || '').toLowerCase();
        const equipmentName = (inspection.equipment.name || '').toLowerCase();
        const inspector = (inspection.inspector || '').toLowerCase();
        const country = (inspection.country || '').toLowerCase();
        const project = (inspection.project || '').toLowerCase();

        return inspectionNumber.includes(searchTerm) ||
               equipmentName.includes(searchTerm) ||
               inspector.includes(searchTerm) ||
               country.includes(searchTerm) ||
               project.includes(searchTerm);
    });
    renderInspectionList(filtered);

    // Update count for filtered results
    const countElement = document.getElementById('inspectionCount');
    if (countElement) {
        if (searchTerm.trim() === '') {
            countElement.textContent = `${inspectionData.length} inspection${inspectionData.length === 1 ? '' : 's'} found`;
        } else {
            countElement.textContent = `${filtered.length} of ${inspectionData.length} inspection${inspectionData.length === 1 ? '' : 's'} match search`;
        }
    }
}

// Select inspection and create tasks
async function selectInspection(inspectionData) {
    try {
        console.log('=== INSPECTION SELECTION STARTED ===');
        console.log('Selected inspection data:', inspectionData);

        // Update the display immediately
        const inspectionNumber = inspectionData.inspectionNumber || inspectionData.requestNumber || inspectionData.firebaseKey || 'N/A';
        document.getElementById('inspectionDisplay').textContent = inspectionNumber;
        document.getElementById('linkedInspection').value = inspectionNumber;
        
        console.log('Updated inspection display:', inspectionNumber);

        // Auto-populate equipment, project, and country
        console.log('Auto-populating fields from inspection:', inspectionData);
        if (inspectionData.equipment && inspectionData.equipment.name !== 'N/A') {
            document.getElementById('equipmentDisplay').value = inspectionData.equipmentDisplay || `${inspectionData.equipment.name} - ${inspectionData.equipment.model} (${inspectionData.equipment.number})`;
            document.getElementById('vehicleNumber').value = inspectionData.equipment.number || inspectionData.equipment.vehicleNumber || '';
            document.getElementById('vehicleName').value = inspectionData.equipment.name || '';
            document.getElementById('vinNumber').value = inspectionData.equipment.vinNumber || inspectionData.equipment.model || '';
            console.log('Equipment populated:', {
                display: inspectionData.equipmentDisplay,
                vehicleNumber: inspectionData.equipment.number,
                vehicleName: inspectionData.equipment.name,
                vinNumber: inspectionData.equipment.vinNumber
            });
        }

        // Auto-populate country
        if (inspectionData.country && inspectionData.country !== 'N/A') {
            const countrySelect = document.getElementById('country');
            if (countrySelect) {
                // Select the matching country option
                const options = countrySelect.options;
                let countryFound = false;
                for (let i = 0; i < options.length; i++) {
                    if (options[i].text.toLowerCase() === inspectionData.country.toLowerCase() ||
                        options[i].value.toLowerCase() === inspectionData.country.toLowerCase()) {
                        countrySelect.selectedIndex = i;
                        countryFound = true;
                        break;
                    }
                }
                console.log('Country auto-population:', {
                    inspectionCountry: inspectionData.country,
                    found: countryFound,
                    selectedIndex: countrySelect.selectedIndex,
                    selectedValue: countrySelect.value
                });
            }
        }

        // Auto-populate project
        if (inspectionData.project && inspectionData.project !== 'N/A') {
            const projectSelect = document.getElementById('project');
            if (projectSelect) {
                // Select the matching project option
                const options = projectSelect.options;
                let projectFound = false;
                for (let i = 0; i < options.length; i++) {
                    if (options[i].text.toLowerCase() === inspectionData.project.toLowerCase() ||
                        options[i].value.toLowerCase() === inspectionData.project.toLowerCase()) {
                        projectSelect.selectedIndex = i;
                        projectFound = true;
                        break;
                    }
                }
                console.log('Project auto-population:', {
                    inspectionProject: inspectionData.project,
                    found: projectFound,
                    selectedIndex: projectSelect.selectedIndex,
                    selectedValue: projectSelect.value
                });
            }
        }

        // Close the modal first
        closeInspectionModal();

        // Check if the inspection has faulty items and create tasks
        console.log('=== CHECKING FAULTY ITEMS FOR AUTO TASK CREATION ===');
        console.log('Inspection data for faulty check:', inspectionData);

        const hasFaultyItems = inspectionData.hasFaultyItems || false;
        const faultyItemsCount = inspectionData.faultyItemsCount || 0;

        console.log('Faulty items analysis:', {
            hasFaultyItems: hasFaultyItems,
            faultyItemsCount: faultyItemsCount,
            firebaseKey: inspectionData.firebaseKey,
            willCreateTasks: hasFaultyItems && faultyItemsCount > 0
        });

        // ALWAYS try to create tasks if there are faulty items
        if (hasFaultyItems && faultyItemsCount > 0) {
            console.log('✅ Inspection has faulty items, creating tasks automatically...');
            console.log(`🔧 Auto-creating tasks from ${faultyItemsCount} faulty items in inspection...`);
            
            // Show visual feedback
            showTaskCreationFeedback(`Creating ${faultyItemsCount} tasks from faulty items...`);

            // Try multiple approaches to create tasks
            let tasksCreated = false;

            try {
                // Approach 1: Fetch detailed inspection data
                console.log('Approach 1: Fetching detailed inspection data for:', inspectionData.firebaseKey);
                const inspectionDetails = await fetchInspectionDetails(inspectionData.firebaseKey);

                console.log('Inspection details API response:', inspectionDetails);
                console.log('API Response structure:', {
                    success: inspectionDetails?.success,
                    hasFaultyItems: !!inspectionDetails?.faultyItems,
                    faultyItemsCount: inspectionDetails?.faultyItems?.length || 0,
                    faultyItems: inspectionDetails?.faultyItems
                });

                if (inspectionDetails && inspectionDetails.faultyItems && inspectionDetails.faultyItems.length > 0) {
                    console.log('✅ Found faulty items in API response:', inspectionDetails.faultyItems.length);
                    console.log('Faulty items details:', inspectionDetails.faultyItems);

                    // Verify each faulty item has proper data
                    inspectionDetails.faultyItems.forEach((item, index) => {
                        console.log(`Faulty Item ${index + 1}:`, {
                            id: item.id,
                            name: item.name,
                            status: item.status,
                            note: item.note,
                            section: item.section
                        });
                    });

                    // Create tasks from faulty items
                    console.log('Creating tasks from API faulty items...');
                    await createTasksFromFaultyItems(inspectionDetails.faultyItems, inspectionData);
                    tasksCreated = true;

                    const createdCount = inspectionDetails.faultyItems.length;
                    console.log(`✅ Successfully created ${createdCount} tasks from API data`);

                    // Ensure tasks are visible
                    ensureTasksVisible();

                    console.log(`✅ ${createdCount} task${createdCount === 1 ? '' : 's'} created from faulty items in inspection ${inspectionNumber}`);
                    console.log('✅ Tasks are now visible in the Tasks & Items section below.');
                    showTaskCreationFeedback(`✅ ${createdCount} tasks created from Firebase data!`);
                } else {
                    console.log('❌ No faulty items found in API response');
                    console.log('API Response details:', {
                        success: inspectionDetails?.success,
                        message: inspectionDetails?.message,
                        hasFaultyItems: !!inspectionDetails?.faultyItems,
                        faultyItemsLength: inspectionDetails?.faultyItems?.length,
                        fullResponse: inspectionDetails
                    });
                    console.log('Trying fallback approach...');
                }
            } catch (error) {
                console.error('❌ Error fetching inspection details:', error);
                console.error('Error details:', {
                    message: error.message,
                    stack: error.stack,
                    firebaseKey: inspectionData.firebaseKey
                });
                console.log('Trying fallback approach...');
            }

            // Approach 2: Create tasks using inspection data directly (fallback)
            if (!tasksCreated && inspectionData.faultyItemsCount > 0) {
                try {
                    console.log('Approach 2: Creating tasks using fallback method...');
                    console.log('Inspection data for fallback:', inspectionData);
                    
                    // Try to get better item names from the inspection data if available
                    let fallbackItems = [];
                    
                    // Check if inspection data has any item information we can use
                    if (inspectionData.items && Array.isArray(inspectionData.items)) {
                        console.log('Found items in inspection data:', inspectionData.items);
                        fallbackItems = inspectionData.items.filter(item => 
                            item.status && ['no', 'faulty', 'fail', 'failed', 'not ok', 'defective', 'needs repair'].includes(item.status.toLowerCase())
                        ).map((item, i) => ({
                            id: item.id || `faulty_${Date.now()}_${i + 1}`,
                            name: item.name || item.label || item.itemName || `Faulty Item ${i + 1}`,
                            status: item.status || 'Faulty',
                            note: item.note || item.description || item.comment || `Auto-generated from inspection ${inspectionNumber}`,
                            section: item.section || item.category || 'General'
                        }));
                    }
                    
                    // If we couldn't extract real items, create fallback items
                    if (fallbackItems.length === 0) {
                        console.log('No real items found, creating fallback items...');
                        fallbackItems = Array.from({ length: inspectionData.faultyItemsCount }, (_, i) => ({
                            id: `faulty_${Date.now()}_${i + 1}`,
                            name: `Faulty Item ${i + 1} (from ${inspectionNumber})`,
                            status: 'Faulty',
                            note: `Auto-generated from inspection ${inspectionNumber}. Original faulty item detected during inspection.`,
                            section: 'General'
                        }));
                    }

                    console.log('Created fallback items:', fallbackItems);

                    await createTasksFromFaultyItems(fallbackItems, inspectionData);
                    tasksCreated = true;

                    // Ensure tasks are visible
                    ensureTasksVisible();

                    console.log(`✅ ${inspectionData.faultyItemsCount} task${inspectionData.faultyItemsCount === 1 ? '' : 's'} created using fallback method.`);
                    console.log('✅ Tasks are now visible in the Tasks & Items section below.');
                    showTaskCreationFeedback(`✅ ${inspectionData.faultyItemsCount} tasks created using fallback!`);
                } catch (fallbackError) {
                    console.error('❌ Fallback task creation also failed:', fallbackError);
                    console.error('❌ Task Creation Failed: Failed to create tasks using both API and fallback methods: ' + fallbackError.message);
                }
            }

            if (!tasksCreated) {
                console.log('❌ No tasks were created');
                console.log('❌ Could not create tasks from this inspection. Please try using the manual "Create Tasks" button.');
            }
        } else {
            console.log('❌ Inspection has no faulty items or count is 0:', {
                hasFaultyItems: hasFaultyItems,
                faultyItemsCount: faultyItemsCount
            });

            console.log(`❌ This inspection has no faulty items to create tasks from.`);
            console.log(`Inspection: ${inspectionNumber}, Faulty Items: ${faultyItemsCount}`);
        }
    } catch (error) {
        console.error('Error selecting inspection:', error);
        console.error('❌ Selection Error: Failed to process inspection selection: ' + error.message);
    }
}

// Fetch inspection details
async function fetchInspectionDetails(firebaseKey) {
    try {
        const params = new URLSearchParams({
            firebase_key: firebaseKey
        });

        const response = await fetch(`/api/fetch-inspection-details?${params}`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.success) {
            return result;
        } else {
            throw new Error(result.message || 'Failed to fetch inspection details');
        }
    } catch (error) {
        console.error('Error fetching inspection details:', error);
        console.error('❌ Inspection Details Error: Failed to fetch inspection details: ' + error.message);
        return null;
    }
}

// Create tasks from faulty items
async function createTasksFromFaultyItems(faultyItems, inspectionData) {
    try {
        console.log(`=== CREATING TASKS FROM FAULTY ITEMS ===`);
        console.log(`Creating tasks from ${faultyItems.length} faulty items:`, faultyItems);
        const inspectionNumber = inspectionData.inspectionNumber || inspectionData.requestNumber || inspectionData.firebaseKey || 'N/A';

        let createdTasks = 0;
        let failedTasks = 0;
        let taskResults = [];

        console.log('Starting task creation process...');

        for (let i = 0; i < faultyItems.length; i++) {
            const faultyItem = faultyItems[i];
            try {
                console.log(`Processing faulty item ${i + 1}/${faultyItems.length}:`, {
                    name: faultyItem.name,
                    status: faultyItem.status,
                    section: faultyItem.section,
                    note: faultyItem.note
                });

                // Try to get name from multiple possible fields
                const itemName = faultyItem.name || faultyItem.label || faultyItem.itemName || faultyItem.id;
                
                if (!itemName || itemName.trim() === '') {
                    console.warn('Faulty item has no identifiable name, skipping:', faultyItem);
                    continue;
                }

                // Ensure we have a proper task name from Firebase data
                const taskName = faultyItem.name || faultyItem.label || faultyItem.itemName || `Faulty Item ${createdTasks + 1}`;
                
                console.log(`Creating task for faulty item:`, {
                    originalData: faultyItem,
                    extractedName: taskName,
                    status: faultyItem.status,
                    section: faultyItem.section
                });

                const taskData = {
                    id: 'task_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9) + '_' + i,
                    taskName: taskName,
                    taskType: 'repair',
                    assignedUsers: 'Not assigned',
                    taskNote: `🚨 FAULTY ITEM TASK\n\n📋 Auto-generated from inspection: ${inspectionNumber}\n📍 Section: ${faultyItem.section || 'General'}\n⚠️ Fault Status: ${faultyItem.status || 'Unknown'}\n📝 Original Note: ${faultyItem.note || 'No notes'}\n\nThis task was automatically created from a faulty item identified during the inspection process. Please address this issue as part of the job card work.`,
                    createdAt: new Date().toISOString(),
                    status: 'initiated',
                    jobCardId: getCurrentJobCardId(),
                    source: 'inspection_faulty_item',
                    inspectionReference: inspectionNumber,
                    faultyItemId: faultyItem.id,
                    faultyItemDetails: faultyItem
                };

                console.log('Creating task from faulty item:', taskData);

                // Save task data to localStorage
                saveTaskToStorage(taskData);

                // Add task to the task list
                addTaskToList(taskData);

                taskResults.push({
                    success: true,
                    taskName: taskData.taskName,
                    taskId: taskData.id
                });

                createdTasks++;
                console.log(`✅ Task ${createdTasks}/${faultyItems.length} created successfully: ${taskData.taskName}`);
            } catch (taskError) {
                console.error('❌ Error creating individual task:', taskError, faultyItem);
                taskResults.push({
                    success: false,
                    taskName: faultyItem.name || `Faulty Item ${i + 1}`,
                    error: taskError.message
                });
                failedTasks++;
            }
        }

        console.log(`=== TASK CREATION SUMMARY ===`);
        console.log(`Total tasks processed: ${faultyItems.length}`);
        console.log(`Successfully created: ${createdTasks}`);
        console.log(`Failed: ${failedTasks}`);
        console.log(`Task results:`, taskResults);

        if (createdTasks > 0) {
            console.log('✅ Tasks created successfully, ensuring visibility...');
            
            // Force reload tasks from storage to ensure they appear
            setTimeout(() => {
                loadTasksFromStorage();
            ensureTasksVisible();
            }, 100);

            console.log('Tasks should now be visible in the UI');
        }

        if (failedTasks > 0) {
            console.warn(`⚠️ ${failedTasks} task${failedTasks === 1 ? '' : 's'} could not be created`);
        }

        console.log('Task creation process completed');
        return { createdTasks, failedTasks, taskResults };
    } catch (error) {
        console.error('Error creating tasks from faulty items:', error);
        console.error('❌ Task Creation Error: Failed to create tasks from faulty items: ' + error.message);
        throw error;
    }
}

// Test faulty items to tasks functionality
async function testFaultyItemsToTasks() {
    console.log('=== TESTING FAULTY ITEMS TO TASKS FUNCTIONALITY ===');

    try {
        // Get all inspection data
        const response = await fetch('/api/fetch-inspection-reports', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        const result = await response.json();

        if (result.success && result.inspections) {
            console.log(`Found ${result.inspections.length} inspections to test`);

            // Find inspections with faulty items
            const inspectionsWithFaulty = result.inspections.filter(inspection => inspection.hasFaultyItems);

            if (inspectionsWithFaulty.length === 0) {
                console.log('No inspections with faulty items found');
                showInfoAlert('Test Complete', 'No inspections with faulty items found to test');
                return;
            }

            console.log(`Found ${inspectionsWithFaulty.length} inspections with faulty items`);

            // Test the first inspection with faulty items
            const testInspection = inspectionsWithFaulty[0];
            console.log('Testing with inspection:', testInspection);

            // Fetch inspection details
            const inspectionDetails = await fetchInspectionDetails(testInspection.firebaseKey);

            if (inspectionDetails && inspectionDetails.faultyItems && inspectionDetails.faultyItems.length > 0) {
                console.log(`Found ${inspectionDetails.faultyItems.length} faulty items in this inspection`);

                // Create tasks from faulty items
                await createTasksFromFaultyItems(inspectionDetails.faultyItems, testInspection);

                showSuccessAlert('Test Successful',
                    `Successfully tested faulty items to tasks conversion!\n\n• Inspection: ${testInspection.requestNumber}\n• Faulty items found: ${inspectionDetails.faultyItems.length}\n• Tasks created: ${inspectionDetails.faultyItems.length}\n\nCheck the Tasks & Items section to see the results.`
                );
            } else {
                console.log('No faulty items found in inspection details');
                showInfoAlert('Test Failed', 'No faulty items found in inspection details');
            }
        } else {
            console.error('Failed to fetch inspection data for testing');
            showErrorAlert('Test Failed', 'Could not fetch inspection data for testing');
        }
    } catch (error) {
        console.error('Error testing faulty items to tasks:', error);
        showErrorAlert('Test Error', 'Error testing faulty items to tasks: ' + error.message);
    }
}

// Check and create tasks from existing linked inspection
async function checkAndCreateTasksFromInspection(inspectionNumber) {
    try {
        console.log('Checking for tasks from inspection:', inspectionNumber);

        // Check if tasks already exist for this inspection
        const jobCardId = getCurrentJobCardId();
        const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');

        if (existingTasks[jobCardId]) {
            const inspectionTasks = existingTasks[jobCardId].filter(task =>
                task.inspectionReference === inspectionNumber && task.source === 'inspection_faulty_item'
            );

            if (inspectionTasks.length > 0) {
                console.log(`Found ${inspectionTasks.length} existing tasks from inspection ${inspectionNumber}`);
                // Tasks already exist, just ensure they're displayed
                ensureTasksVisible();
                return;
            }
        }

        // No tasks found, need to create them
        console.log('No existing tasks found, creating tasks from inspection:', inspectionNumber);

        // Fetch inspection data to get faulty items
        const inspectionResponse = await fetch('/api/fetch-inspection-reports', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        const inspectionResult = await inspectionResponse.json();

        if (inspectionResult.success && inspectionResult.inspections) {
            const inspection = inspectionResult.inspections.find(i =>
                (i.inspectionNumber === inspectionNumber) ||
                (i.requestNumber === inspectionNumber) ||
                (i.firebaseKey === inspectionNumber)
            );

            if (inspection) {
                console.log('Found inspection data:', inspection);

                if (inspection.hasFaultyItems) {
                    // Fetch detailed inspection data
                    const inspectionDetails = await fetchInspectionDetails(inspection.firebaseKey);

                    if (inspectionDetails && inspectionDetails.faultyItems && inspectionDetails.faultyItems.length > 0) {
                        console.log(`Creating ${inspectionDetails.faultyItems.length} tasks from inspection ${inspectionNumber}`);

                        // Create tasks from faulty items
                        await createTasksFromFaultyItems(inspectionDetails.faultyItems, inspection);

                        // Update the inspection display
                        document.getElementById('inspectionDisplay').textContent = inspectionNumber;
                        document.getElementById('linkedInspection').value = inspectionNumber;

                        console.log('Tasks created successfully from existing inspection');
                    } else {
                        console.log('No faulty items found in inspection details');
                    }
                } else {
                    console.log('Inspection has no faulty items');
                }
            } else {
                console.log('Inspection not found in API response');
            }
        }
    } catch (error) {
        console.error('Error checking and creating tasks from inspection:', error);
    }
}

// Ensure tasks are visible in the UI
function ensureTasksVisible() {
    console.log('=== ENSURING TASKS ARE VISIBLE ===');

    // Switch to task tab
    const taskTab = document.querySelector('.task-tab[data-tab="task"]');
    const otherTabs = document.querySelectorAll('.task-tab:not([data-tab="task"])');

    if (taskTab) {
        console.log('Found task tab, activating...');

        // Activate task tab
        taskTab.classList.add('active');

        // Deactivate other tabs
        otherTabs.forEach(tab => tab.classList.remove('active'));

        // Show task tab content
        const taskTabContent = document.getElementById('task-tab-content');
        if (taskTabContent) {
            taskTabContent.classList.add('active');
            console.log('Task tab content activated');
        }

        // Hide other tab contents
        const allTabContent = document.querySelectorAll('.tab-content');
        allTabContent.forEach(content => {
            if (content.id !== 'task-tab-content') {
                content.classList.remove('active');
            }
        });

        // Force scroll to task section
        const tasksSection = document.querySelector('.tasks-section');
        if (tasksSection) {
            tasksSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
            console.log('Scrolled to tasks section');
        }

        console.log('Task tab activated and tasks should be visible');
    } else {
        console.warn('Task tab not found!');
    }

    // Also trigger the task tab click event to ensure proper initialization
    if (taskTab) {
        taskTab.click();
    }
}

// Create tasks from currently linked inspection
async function createTasksFromLinkedInspection() {
    try {
        const linkedInspection = document.getElementById('linkedInspection');
        const inspectionDisplay = document.getElementById('inspectionDisplay');

        if (!linkedInspection || !linkedInspection.value) {
            showInfoAlert('No Inspection Linked', 'No inspection is currently linked to create tasks from.');
            return;
        }

        const inspectionNumber = linkedInspection.value;
        console.log('Creating tasks from linked inspection:', inspectionNumber);

        showInfoAlert('Creating Tasks', `Creating tasks from linked inspection ${inspectionNumber}...`);

        // Create new tasks from the inspection
        await checkAndCreateTasksFromInspection(inspectionNumber);

        showSuccessAlert('Tasks Created', 'Tasks have been created from the linked inspection.');
    } catch (error) {
        console.error('Error creating tasks from linked inspection:', error);
        showErrorAlert('Create Tasks Error', 'Failed to create tasks from linked inspection: ' + error.message);
    }
}

// Refresh tasks from linked inspection
async function refreshTasksFromInspection() {
    try {
        const linkedInspection = document.getElementById('linkedInspection');
        const inspectionDisplay = document.getElementById('inspectionDisplay');

        if (!linkedInspection || !linkedInspection.value) {
            showInfoAlert('No Inspection Linked', 'No inspection is currently linked to refresh tasks from.');
            return;
        }

        const inspectionNumber = linkedInspection.value;
        console.log('Refreshing tasks from inspection:', inspectionNumber);

        showInfoAlert('Refreshing Tasks', 'Refreshing tasks from linked inspection...');

        // Clear existing inspection tasks first
        const jobCardId = getCurrentJobCardId();
        const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');

        if (existingTasks[jobCardId]) {
            existingTasks[jobCardId] = existingTasks[jobCardId].filter(task =>
                !(task.inspectionReference === inspectionNumber && task.source === 'inspection_faulty_item')
            );
            localStorage.setItem('jobCardTasks', JSON.stringify(existingTasks));
        }

        // Reload tasks from localStorage to remove old ones
        loadTasksFromStorage();

        // Create new tasks from the inspection
        await checkAndCreateTasksFromInspection(inspectionNumber);

        showSuccessAlert('Tasks Refreshed', 'Tasks have been refreshed from the linked inspection.');
    } catch (error) {
        console.error('Error refreshing tasks from inspection:', error);
        showErrorAlert('Refresh Error', 'Failed to refresh tasks from inspection: ' + error.message);
    }
}

// Manual task creation from inspection modal
async function manualCreateTasks(event, inspectionData) {
    event.preventDefault();
    event.stopPropagation();

    console.log('=== MANUAL TASK CREATION TRIGGERED ===');
    console.log('Manual task creation for inspection:', inspectionData);

    try {
        // Don't close modal yet, let user see the process
        // closeInspectionModal();

        // Set the selected inspection immediately
        const inspectionNumber = inspectionData.inspectionNumber || inspectionData.requestNumber || inspectionData.firebaseKey || 'N/A';
        const inspectionDisplay = document.getElementById('inspectionDisplay');
        const linkedInspection = document.getElementById('linkedInspection');

        if (inspectionDisplay) inspectionDisplay.textContent = inspectionNumber;
        if (linkedInspection) linkedInspection.value = inspectionNumber;

        // Show loading message
        showInfoAlert('Creating Tasks', `Fetching faulty items and creating tasks from inspection ${inspectionNumber}...`);

        // Fetch inspection details
        console.log('Fetching inspection details for manual creation...');
        const inspectionDetails = await fetchInspectionDetails(inspectionData.firebaseKey);

        if (inspectionDetails && inspectionDetails.faultyItems && inspectionDetails.faultyItems.length > 0) {
            console.log('Found faulty items for manual creation:', inspectionDetails.faultyItems);

            // Close modal after starting the process
            closeInspectionModal();

            // Create tasks
            await createTasksFromFaultyItems(inspectionDetails.faultyItems, inspectionData);

            const createdCount = inspectionDetails.faultyItems.length;
            showSuccessAlert('Tasks Created Successfully', `${createdCount} task${createdCount === 1 ? '' : 's'} created from faulty items in inspection ${inspectionNumber}\n\n✅ Tasks are now visible in the Tasks & Items section.`);
        } else {
            console.log('No faulty items found');
            showInfoAlert('No Faulty Items', 'No faulty items found to create tasks from.');
        }
    } catch (error) {
        console.error('Error in manual task creation:', error);
        showErrorAlert('Error', 'Failed to create tasks: ' + error.message);
    }
}

// Debug inspection tasks
function debugInspectionTasks() {
    console.log('=== DEBUGGING INSPECTION TASKS ===');

    const linkedInspection = document.getElementById('linkedInspection');
    const inspectionDisplay = document.getElementById('inspectionDisplay');

    console.log('Linked inspection:', {
        value: linkedInspection?.value || 'None',
        display: inspectionDisplay?.textContent || 'None'
    });

    // Check localStorage for tasks
    const jobCardId = getCurrentJobCardId();
    const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');

    console.log('Tasks in localStorage for job card:', jobCardId);

    if (existingTasks[jobCardId] && existingTasks[jobCardId].length > 0) {
        console.log(`Found ${existingTasks[jobCardId].length} total tasks`);

        const inspectionTasks = existingTasks[jobCardId].filter(task =>
            task.inspectionReference === linkedInspection?.value && task.source === 'inspection_faulty_item'
        );

        console.log(`Found ${inspectionTasks.length} inspection-related tasks`);
        inspectionTasks.forEach((task, index) => {
            console.log(`Task ${index + 1}:`, {
                id: task.id,
                name: task.taskName,
                inspectionReference: task.inspectionReference,
                source: task.source
            });
        });
    } else {
        console.log('No tasks found in localStorage');
    }

    showInfoAlert('Debug Complete', 'Check browser console for detailed inspection tasks analysis.');
}

// Test inspection API function
async function testInspectionAPI() {
    try {
        console.log('Testing inspection API...');
        const timestamp = Date.now();
        const cacheBuster = Math.random().toString(36).substring(7);
        const response = await fetch(`/api/fetch-inspection-reports?_t=${timestamp}&_cb=${cacheBuster}`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        const result = await response.json();
        console.log('Inspection API test result:', result);

        if (result.success && result.inspections) {
            console.log('Sample inspection from API:', result.inspections[0]);

            // Test faulty items counting
            const inspectionsWithFaulty = result.inspections.filter(inspection => inspection.hasFaultyItems);
            console.log('Inspections with faulty items:', inspectionsWithFaulty.length);

            inspectionsWithFaulty.forEach((inspection, index) => {
                console.log(`Inspection ${index + 1} with faulty items:`, {
                    requestNumber: inspection.requestNumber,
                    faultyItemsCount: inspection.faultyItemsCount,
                    hasFaultyItems: inspection.hasFaultyItems
                });
            });

            // Validate faulty items count accuracy
            console.log('=== FAULTY ITEMS VALIDATION ===');
            let validationIssues = 0;

            result.inspections.forEach((inspection, index) => {
                const expectedFaultyCount = inspection.faultyItemsCount || 0;
                const hasFaultyFlag = inspection.hasFaultyItems || false;

                // Check if hasFaultyItems flag matches the count
                const expectedHasFaulty = expectedFaultyCount > 0;
                if (hasFaultyFlag !== expectedHasFaulty) {
                    console.warn(`VALIDATION ISSUE: Inspection ${index + 1} (${inspection.requestNumber})`, {
                        hasFaultyItems: hasFaultyFlag,
                        faultyItemsCount: expectedFaultyCount,
                        expectedHasFaulty: expectedHasFaulty,
                        issue: 'hasFaultyItems flag mismatch'
                    });
                    validationIssues++;
                }

                // Check if count is reasonable (not negative)
                if (expectedFaultyCount < 0) {
                    console.warn(`VALIDATION ISSUE: Inspection ${index + 1} has negative faulty count:`, expectedFaultyCount);
                    validationIssues++;
                }
            });

            console.log(`Validation completed: ${validationIssues} issues found`);
            if (validationIssues === 0) {
                console.log('✅ All faulty items counts are valid!');
            } else {
                console.warn(`⚠️ Found ${validationIssues} validation issues that need attention`);
            }
        }
    } catch (error) {
        console.error('Inspection API test failed:', error);
    }
}

// Enhanced debugging function for inspection data
function debugInspectionData() {
    console.log('=== DEBUGGING INSPECTION DATA ===');

    // Check current inspection data
    console.log('Current inspectionData array:', inspectionData);
    console.log('Number of inspections loaded:', inspectionData.length);

    // Check for faulty items in each inspection
    const inspectionsWithFaulty = inspectionData.filter(inspection => inspection.hasFaultyItems);
    console.log('Inspections with hasFaultyItems = true:', inspectionsWithFaulty.length);

    // Show details of inspections with faulty items
    inspectionsWithFaulty.forEach((inspection, index) => {
        console.log(`Inspection ${index + 1} with faulty items:`, {
            requestNumber: inspection.requestNumber,
            faultyItemsCount: inspection.faultyItemsCount,
            hasFaultyItems: inspection.hasFaultyItems,
            firebaseKey: inspection.firebaseKey
        });
    });

    // Show all inspections with their faulty counts
    console.log('All inspections with faulty counts:');
    inspectionData.forEach((inspection, index) => {
        console.log(`${index + 1}. ${inspection.requestNumber}: ${inspection.faultyItemsCount} faulty items, hasFaultyItems: ${inspection.hasFaultyItems}`);
    });

    showInfoAlert('Debug Complete', `Check browser console for detailed inspection data analysis.\n\nFound ${inspectionsWithFaulty.length} inspections with faulty items.`);
}

// Close modal when clicking outside
window.onclick = function(event) {
    const equipmentModal = document.getElementById('equipmentModal');
    const inspectionModal = document.getElementById('inspectionModal');

    if (event.target === equipmentModal) {
        closeEquipmentModal();
    }

    if (event.target === inspectionModal) {
        closeInspectionModal();
    }
}

function openUserModal() {
    // User selection modal functionality
    showInfoAlert('User Selection', 'User selection modal would open here');
}

// Global variables for calendar
let currentField = null;

function openDatePicker(fieldId) {
    console.log('Date picker button clicked for:', fieldId);

    currentField = fieldId;
    const modal = document.getElementById('datePickerModal');
    const calendarContainer = document.getElementById('calendarContainer');

    // Show modal
    modal.style.display = 'flex';
    document.body.classList.add('modal-open');

    // Generate calendar
    generateCalendar(calendarContainer, fieldId);
}

function closeDatePicker() {
    const modal = document.getElementById('datePickerModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
    currentField = null;
}

function generateCalendar(calendar, fieldId) {
    const today = new Date();
    const currentMonth = today.getMonth();
    const currentYear = today.getFullYear();
    
    let displayMonth = currentMonth;
    let displayYear = currentYear;
    
    function renderCalendar() {
        const monthNames = [
            'January', 'February', 'March', 'April', 'May', 'June',
            'July', 'August', 'September', 'October', 'November', 'December'
        ];
        
        const firstDay = new Date(displayYear, displayMonth, 1);
        const lastDay = new Date(displayYear, displayMonth + 1, 0);
        const daysInMonth = lastDay.getDate();
        const startingDay = firstDay.getDay();
        
        let calendarHTML = `
            <div class="calendar-header">
                <button class="calendar-nav" onclick="changeMonth(-1)">&lt;</button>
                <div class="calendar-month-year">${monthNames[displayMonth]} ${displayYear}</div>
                <button class="calendar-nav" onclick="changeMonth(1)">&gt;</button>
            </div>
            <div class="calendar-grid">
                <div class="calendar-day-header">Sun</div>
                <div class="calendar-day-header">Mon</div>
                <div class="calendar-day-header">Tue</div>
                <div class="calendar-day-header">Wed</div>
                <div class="calendar-day-header">Thu</div>
                <div class="calendar-day-header">Fri</div>
                <div class="calendar-day-header">Sat</div>
        `;
        
        // Add empty cells for days before the first day of the month
        for (let i = 0; i < startingDay; i++) {
            const prevMonthDay = new Date(displayYear, displayMonth, 0).getDate() - startingDay + i + 1;
            calendarHTML += `<div class="calendar-day other-month">${prevMonthDay}</div>`;
        }
        
        // Add days of the current month
        for (let day = 1; day <= daysInMonth; day++) {
            const isToday = day === today.getDate() && displayMonth === currentMonth && displayYear === currentYear;
            const dayClass = isToday ? 'calendar-day today' : 'calendar-day';
            calendarHTML += `<div class="${dayClass}" onclick="selectDate(${day})">${day}</div>`;
        }
        
        // Add empty cells for days after the last day of the month
        const remainingDays = 42 - (startingDay + daysInMonth);
        for (let day = 1; day <= remainingDays; day++) {
            calendarHTML += `<div class="calendar-day other-month">${day}</div>`;
        }
        
        calendarHTML += `
            </div>
            <div class="calendar-actions">
                <button class="calendar-btn" onclick="closeCalendar()">Cancel</button>
                <button class="calendar-btn primary" onclick="selectToday()">Today</button>
            </div>
        `;
        
        calendar.innerHTML = calendarHTML;
    }
    
    window.changeMonth = function(direction) {
        displayMonth += direction;
        if (displayMonth < 0) {
            displayMonth = 11;
            displayYear--;
        } else if (displayMonth > 11) {
            displayMonth = 0;
            displayYear++;
        }
        renderCalendar();
    };
    
    window.selectDate = function(day) {
        const selectedDate = new Date(displayYear, displayMonth, day);
        const formattedDate = selectedDate.toLocaleDateString('en-US', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit'
        });
        
        document.getElementById(fieldId).value = formattedDate;
        closeDatePicker();
    };
    
    window.selectToday = function() {
        const today = new Date();
        const formattedDate = today.toLocaleDateString('en-US', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit'
        });
        
        document.getElementById(fieldId).value = formattedDate;
        closeDatePicker();
    };
    
    renderCalendar();
}

// Close calendar when clicking outside
document.addEventListener('click', function(event) {
    const modal = document.getElementById('datePickerModal');
    if (event.target === modal) {
        closeDatePicker();
    }
});

function toggleCollapsible(sectionId) {
    const section = document.getElementById(sectionId).closest('.collapsible-section');
    section.classList.toggle('active');
}

function addNewTask() {
    // Open Add Task modal
    const modal = document.getElementById('addTaskModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
}

function closeAddTaskModal() {
    // Close Add Task modal
    const modal = document.getElementById('addTaskModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
    // Reset form
    document.getElementById('addTaskForm').reset();
    // Reset type selection
    selectTaskType('repair');
}

function selectTaskType(type) {
    // Remove active class from all type buttons
    const typeButtons = document.querySelectorAll('.type-btn');
    typeButtons.forEach(btn => btn.classList.remove('active'));
    
    // Add active class to selected button
    const selectedButton = document.querySelector(`[data-type="${type}"]`);
    if (selectedButton) {
        selectedButton.classList.add('active');
    }
    
    // Update hidden input
    document.getElementById('taskType').value = type;
}

function saveTask() {
    // Get form data
    const form = document.getElementById('addTaskForm');
    const formData = new FormData(form);
    
    // Validate required fields
    const requiredFields = ['taskName', 'taskType'];
    let isValid = true;
    
    requiredFields.forEach(field => {
        const input = document.getElementById(field);
        if (!input.value.trim()) {
            input.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            input.style.borderColor = '#d1d5db';
        }
    });
    
    if (!isValid) {
        showWarningAlert('Validation Error', 'Please fill in all required fields');
        return;
    }
    
    // Create task data object with additional fields
    const taskData = {
        id: 'task_' + Date.now(),
        taskName: formData.get('taskName'),
        taskType: formData.get('taskType'),
        assignedUsers: formData.get('assignedUsers') || 'Not assigned',
        taskNote: formData.get('taskNote') || 'No description provided',
        createdAt: new Date().toISOString(),
        status: 'initiated',
        jobCardId: getCurrentJobCardId() // Get current job card ID
    };
    
    console.log('Task data:', taskData);
    
    // Save task data to localStorage (for persistence)
    saveTaskToStorage(taskData);
    
    // Add new task to the task list
    addTaskToList(taskData);
    
    // Close modal and reset form
    closeAddTaskModal();
}

// Function to get current job card ID
function getCurrentJobCardId() {
    // Try to get from URL or form
    const urlParams = new URLSearchParams(window.location.search);
    const jobCardId = urlParams.get('job_card_id') || 
                     document.getElementById('jobCardNumber')?.value || 
                     'ZAM-KOB-JC-1002'; // Default fallback
    return jobCardId;
}

// Function to save task to localStorage
function saveTaskToStorage(taskData) {
    try {
        console.log('=== SAVING TASK TO STORAGE ===');
        console.log('Task data to save:', taskData);
        
        // Get existing tasks from localStorage
        const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');
        const jobCardId = taskData.jobCardId || getCurrentJobCardId();
        
        console.log('Using job card ID:', jobCardId);
        console.log('Existing tasks before save:', existingTasks);
        
        // Initialize job card tasks array if it doesn't exist
        if (!existingTasks[jobCardId]) {
            existingTasks[jobCardId] = [];
            console.log('Created new job card array for:', jobCardId);
        }
        
        // Add new task to the array
        existingTasks[jobCardId].push(taskData);
        
        // Save back to localStorage
        localStorage.setItem('jobCardTasks', JSON.stringify(existingTasks));
        
        console.log('✅ Task saved to localStorage:', taskData);
        console.log('Updated tasks in storage:', existingTasks);
    } catch (error) {
        console.error('Error saving task to localStorage:', error);
        console.error('Save Error: Failed to save task data. Please try again.');
    }
}

// Function to load tasks from localStorage
function loadTasksFromStorage() {
    try {
        const jobCardId = getCurrentJobCardId();
        const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');

        console.log('Loading tasks from localStorage for job card:', jobCardId);
        console.log('All stored tasks:', existingTasks);

        if (existingTasks[jobCardId] && existingTasks[jobCardId].length > 0) {
            console.log('Found tasks for this job card:', existingTasks[jobCardId]);

            // Clear existing dynamic tasks only (preserve any static content)
            const taskList = document.querySelector('#task-tab-content .task-list');
            if (taskList) {
                // Remove only dynamic tasks
                const dynamicTasks = taskList.querySelectorAll('.task-item[data-dynamic]');
                console.log('Removing existing dynamic tasks:', dynamicTasks.length);
                dynamicTasks.forEach(task => task.remove());

                // Hide "No items" message if it exists
                const noItemsMsg = taskList.querySelector('.no-items');
                if (noItemsMsg) {
                    noItemsMsg.style.display = 'none';
                }

                // Add saved tasks back
                existingTasks[jobCardId].forEach((taskData, index) => {
                    console.log(`Loading task ${index + 1}:`, taskData);
                    addTaskToList(taskData);
                });

                console.log(`Successfully loaded ${existingTasks[jobCardId].length} tasks from localStorage`);

                // Ensure task tab is visible
                ensureTasksVisible();
            } else {
                console.warn('Task list element not found');
            }
        } else {
            console.log('No tasks found in localStorage for job card:', jobCardId);
            // Show "no items" message if no tasks exist
            const taskList = document.querySelector('#task-tab-content .task-list');
            if (taskList) {
                let noItemsMsg = taskList.querySelector('.no-items');
                if (!noItemsMsg) {
                    noItemsMsg = document.createElement('div');
                    noItemsMsg.className = 'no-items';
                    noItemsMsg.textContent = 'No tasks added yet. Select an inspection or add tasks manually.';
                    noItemsMsg.style.cssText = 'text-align: center; color: #6b7280; padding: 20px; font-style: italic;';
                    taskList.appendChild(noItemsMsg);
                }
                noItemsMsg.style.display = 'block';
            }
        }
    } catch (error) {
        console.error('Error loading tasks from localStorage:', error);
    }
}

function addTaskToList(taskData) {
    try {
        console.log('=== ADDING TASK TO LIST ===');
        console.log('Task data:', taskData);

    // Use provided ID or generate unique task ID
    const taskId = taskData.id || 'task' + Date.now();
        console.log('Using task ID:', taskId);

    // Create task item HTML
    const taskItem = document.createElement('div');
    taskItem.className = 'task-item';
    taskItem.setAttribute('data-dynamic', 'true'); // Mark as dynamic task
    taskItem.setAttribute('data-task-id', taskId); // Add task ID for deletion
        taskItem.setAttribute('id', taskId); // Set ID for easier finding
        
    taskItem.innerHTML = `
        <div class="task-info">
                <div class="task-title">${taskData.taskName || 'Untitled Task'}</div>
                <div class="task-type">${(taskData.taskType || 'repair').charAt(0).toUpperCase() + (taskData.taskType || 'repair').slice(1)}</div>
            <div class="task-description">${taskData.taskNote || 'No description provided'}</div>
        </div>
        <div class="task-status">
            <div class="status-dropdown">
                    <button type="button" class="status-badge ${taskData.status || 'initiated'}" onclick="toggleStatusDropdown('${taskId}')">${(taskData.status || 'initiated').charAt(0).toUpperCase() + (taskData.status || 'initiated').slice(1)}</button>
                <div id="${taskId}-dropdown" class="status-dropdown-menu">
                    <div class="status-option" onclick="changeStatus('${taskId}', 'initiated')">
                        <span class="status-dot initiated"></span>
                        <span>Initiated</span>
                    </div>
                    <div class="status-option" onclick="changeStatus('${taskId}', 'in-progress')">
                        <span class="status-dot in-progress"></span>
                        <span>In Progress</span>
                    </div>
                    <div class="status-option" onclick="changeStatus('${taskId}', 'on-hold')">
                        <span class="status-dot on-hold"></span>
                        <span>On Hold</span>
                    </div>
                    <div class="status-option" onclick="changeStatus('${taskId}', 'complete')">
                        <span class="status-dot complete"></span>
                        <span>Complete</span>
                    </div>
                </div>
            </div>
            <div class="task-options-dropdown">
                <button type="button" class="task-options" onclick="toggleTaskOptions('${taskId}')">
                    <i class="fas fa-ellipsis-v"></i>
                </button>
                <div id="${taskId}-options" class="task-options-menu">
                    <div class="task-option" onclick="editTask('${taskId}')">
                        <i class="fas fa-edit"></i>
                        <span>Edit</span>
                    </div>
                    <div class="task-option" onclick="deleteTask('${taskId}')">
                        <i class="fas fa-trash"></i>
                        <span>Delete</span>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Add task to the task list
    const taskList = document.querySelector('#task-tab-content .task-list');
    if (taskList) {
            console.log('Found task list, adding task...');

            // Hide "No items" message if it exists
        const noItemsMsg = taskList.querySelector('.no-items');
        if (noItemsMsg) {
            noItemsMsg.style.display = 'none';
                console.log('Hidden "no items" message');
        }

            // Add the task to the list
        taskList.appendChild(taskItem);
            console.log('Task added to DOM with ID:', taskId);

            // Ensure the task tab is active
        const taskTab = document.querySelector('.task-tab[data-tab="task"]');
        const otherTabs = document.querySelectorAll('.task-tab:not([data-tab="task"])');

        if (taskTab) {
            // Activate task tab
            taskTab.classList.add('active');
            otherTabs.forEach(tab => tab.classList.remove('active'));

            // Show task tab content
            document.getElementById('task-tab-content').classList.add('active');

            // Hide other tab contents
            const allTabContent = document.querySelectorAll('.tab-content');
            allTabContent.forEach(content => {
                if (content.id !== 'task-tab-content') {
                    content.classList.remove('active');
                }
            });

                console.log('Task tab activated');
        }

            console.log('✅ Task successfully added to UI');
    } else {
            console.error('❌ Task list not found!');
        showErrorAlert('Error', 'Could not find task list to add task.');
        }
    } catch (error) {
        console.error('Error adding task to list:', error);
        showErrorAlert('Error', 'Failed to add task to list: ' + error.message);
    }
}

function addNewPart() {
    // Open Add Parts modal
    const modal = document.getElementById('addPartsModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
}

function closeAddPartsModal() {
    // Close Add Parts modal
    const modal = document.getElementById('addPartsModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
    // Reset form
    document.getElementById('addPartsForm').reset();
}

function savePart() {
    // Get form data
    const form = document.getElementById('addPartsForm');
    const formData = new FormData(form);
    
    // Validate required fields
    const requiredFields = ['itemNumber', 'partNumber', 'itemName', 'quantity'];
    let isValid = true;
    
    requiredFields.forEach(field => {
        const input = document.getElementById(field);
        if (!input.value.trim()) {
            input.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            input.style.borderColor = '#d1d5db';
        }
    });
    
    if (!isValid) {
        showWarningAlert('Validation Error', 'Please fill in all required fields');
        return;
    }
    
    // Create part data object with additional fields
    const partData = {
        id: 'part_' + Date.now(),
        itemNumber: formData.get('itemNumber'),
        partNumber: formData.get('partNumber'),
        itemName: formData.get('itemName'),
        quantity: formData.get('quantity'),
        unit: formData.get('unit'),
        itemCost: formData.get('itemCost'),
        site: formData.get('site'),
        note: formData.get('note'),
        createdAt: new Date().toISOString(),
        jobCardId: getCurrentJobCardId()
    };
    
    console.log('Part data:', partData);
    
    // Save part data to localStorage
    savePartToStorage(partData);
    
    // Add part to the parts list
    addPartToList(partData);
    
    // Close modal and reset form
    closeAddPartsModal();
}

// Function to save part to localStorage
function savePartToStorage(partData) {
    try {
        const existingParts = JSON.parse(localStorage.getItem('jobCardParts') || '{}');
        const jobCardId = partData.jobCardId;
        
        if (!existingParts[jobCardId]) {
            existingParts[jobCardId] = [];
        }
        
        existingParts[jobCardId].push(partData);
        localStorage.setItem('jobCardParts', JSON.stringify(existingParts));
        console.log('Part saved to localStorage:', partData);
    } catch (error) {
        console.error('Error saving part to localStorage:', error);
        showErrorAlert('Save Error', 'Failed to save part data. Please try again.');
    }
}

// Function to add part to the parts list
function addPartToList(partData) {
    const partId = partData.id;
    
    // Create part item HTML
    const partItem = document.createElement('div');
    partItem.className = 'part-item';
    partItem.setAttribute('data-dynamic', 'true');
    partItem.innerHTML = `
        <div class="part-info">
            <div class="part-name">${partData.itemName}</div>
            <div class="part-details">
                <span class="part-number">${partData.partNumber}</span>
                <span class="part-quantity">Qty: ${partData.quantity} ${partData.unit}</span>
                <span class="part-cost">$${partData.itemCost}</span>
            </div>
            <div class="part-note">${partData.note || 'No notes'}</div>
        </div>
        <div class="part-actions">
            <button type="button" class="part-edit-btn" onclick="editPart('${partId}')">
                <i class="fas fa-edit"></i>
            </button>
            <button type="button" class="part-delete-btn" onclick="deletePart('${partId}')">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    
    // Add part to the parts list
    const partsList = document.querySelector('#parts-tab-content .parts-list');
    if (partsList) {
        // Hide "No items" message if it exists
        const noItemsMsg = partsList.querySelector('.no-items');
        if (noItemsMsg) {
            noItemsMsg.style.display = 'none';
        }
        
        partsList.appendChild(partItem);
        console.log('Part added to DOM:', partData);
        showSuccessAlert('Part Added', 'Part has been successfully added and saved!');
    } else {
        console.error('Parts list not found!');
        showErrorAlert('Error', 'Could not find parts list to add part.');
    }
}

// Function to load parts from localStorage
function loadPartsFromStorage() {
    try {
        const jobCardId = getCurrentJobCardId();
        const existingParts = JSON.parse(localStorage.getItem('jobCardParts') || '{}');
        
        if (existingParts[jobCardId] && existingParts[jobCardId].length > 0) {
            console.log('Loading parts from localStorage:', existingParts[jobCardId]);
            
            const partsList = document.querySelector('#parts-tab-content .parts-list');
            if (partsList) {
                // Hide "No items" message
                const noItemsMsg = partsList.querySelector('.no-items');
                if (noItemsMsg) {
                    noItemsMsg.style.display = 'none';
                }
                
                // Clear existing dynamic parts
                const dynamicParts = partsList.querySelectorAll('.part-item[data-dynamic]');
                dynamicParts.forEach(part => part.remove());
                
                // Add saved parts
                existingParts[jobCardId].forEach(partData => {
                    addPartToList(partData);
                });
            }
        }
    } catch (error) {
        console.error('Error loading parts from localStorage:', error);
    }
}

function addNewLabor() {
    // Open Add Labor modal
    const modal = document.getElementById('addLaborModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
}

function closeAddLaborModal() {
    // Close Add Labor modal
    const modal = document.getElementById('addLaborModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
    // Reset form
    document.getElementById('addLaborForm').reset();
}

function saveLabor() {
    // Get form data
    const form = document.getElementById('addLaborForm');
    const formData = new FormData(form);
    
    // Validate required fields
    const requiredFields = ['workerName', 'laborCode', 'laborRate', 'workHours', 'workMinutes'];
    let isValid = true;
    
    requiredFields.forEach(field => {
        const input = document.getElementById(field);
        if (!input.value.trim()) {
            input.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            input.style.borderColor = '#d1d5db';
        }
    });
    
    if (!isValid) {
        showWarningAlert('Validation Error', 'Please fill in all required fields');
        return;
    }
    
    // Create labor data object with additional fields
    const laborData = {
        id: 'labor_' + Date.now(),
        workerName: formData.get('workerName'),
        laborCode: formData.get('laborCode'),
        laborRate: formData.get('laborRate'),
        workHours: formData.get('workHours'),
        workMinutes: formData.get('workMinutes'),
        laborNote: formData.get('laborNote'),
        createdAt: new Date().toISOString(),
        jobCardId: getCurrentJobCardId()
    };
    
    console.log('Labor data:', laborData);
    
    // Save labor data to localStorage
    saveLaborToStorage(laborData);
    
    // Add labor to the labor list
    addLaborToList(laborData);
    
    // Close modal and reset form
    closeAddLaborModal();
}

// Function to save labor to localStorage
function saveLaborToStorage(laborData) {
    try {
        const existingLabor = JSON.parse(localStorage.getItem('jobCardLabor') || '{}');
        const jobCardId = laborData.jobCardId;
        
        if (!existingLabor[jobCardId]) {
            existingLabor[jobCardId] = [];
        }
        
        existingLabor[jobCardId].push(laborData);
        localStorage.setItem('jobCardLabor', JSON.stringify(existingLabor));
        console.log('Labor saved to localStorage:', laborData);
    } catch (error) {
        console.error('Error saving labor to localStorage:', error);
        showErrorAlert('Save Error', 'Failed to save labor data. Please try again.');
    }
}

// Function to add labor to the labor list
function addLaborToList(laborData) {
    const laborId = laborData.id;
    
    // Create labor item HTML
    const laborItem = document.createElement('div');
    laborItem.className = 'labor-item';
    laborItem.setAttribute('data-dynamic', 'true');
    laborItem.innerHTML = `
        <div class="labor-info">
            <div class="labor-worker">${laborData.workerName}</div>
            <div class="labor-details">
                <span class="labor-code">Code: ${laborData.laborCode}</span>
                <span class="labor-rate">Rate: $${laborData.laborRate}/hr</span>
                <span class="labor-hours">Hours: ${laborData.workHours}h ${laborData.workMinutes}m</span>
            </div>
            <div class="labor-note">${laborData.laborNote || 'No notes'}</div>
        </div>
        <div class="labor-actions">
            <button type="button" class="labor-edit-btn" onclick="editLabor('${laborId}')">
                <i class="fas fa-edit"></i>
            </button>
            <button type="button" class="labor-delete-btn" onclick="deleteLabor('${laborId}')">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    
    // Add labor to the labor list
    const laborList = document.querySelector('#labor-tab-content .labor-list');
    if (laborList) {
        // Hide "No items" message if it exists
        const noItemsMsg = laborList.querySelector('.no-items');
        if (noItemsMsg) {
            noItemsMsg.style.display = 'none';
        }
        
        laborList.appendChild(laborItem);
        console.log('Labor added to DOM:', laborData);
        showSuccessAlert('Labor Added', 'Labor has been successfully added and saved!');
    } else {
        console.error('Labor list not found!');
        showErrorAlert('Error', 'Could not find labor list to add labor.');
    }
}

// Function to load labor from localStorage
function loadLaborFromStorage() {
    try {
        const jobCardId = getCurrentJobCardId();
        const existingLabor = JSON.parse(localStorage.getItem('jobCardLabor') || '{}');
        
        if (existingLabor[jobCardId] && existingLabor[jobCardId].length > 0) {
            console.log('Loading labor from localStorage:', existingLabor[jobCardId]);
            
            const laborList = document.querySelector('#labor-tab-content .labor-list');
            if (laborList) {
                // Hide "No items" message
                const noItemsMsg = laborList.querySelector('.no-items');
                if (noItemsMsg) {
                    noItemsMsg.style.display = 'none';
                }
                
                // Clear existing dynamic labor
                const dynamicLabor = laborList.querySelectorAll('.labor-item[data-dynamic]');
                dynamicLabor.forEach(labor => labor.remove());
                
                // Add saved labor
                existingLabor[jobCardId].forEach(laborData => {
                    addLaborToList(laborData);
                });
            }
        }
    } catch (error) {
        console.error('Error loading labor from localStorage:', error);
    }
}

function addNewAdditional() {
    // Open Add Additional Cost modal
    const modal = document.getElementById('addAdditionalModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
}

function closeAddAdditionalModal() {
    // Close Add Additional Cost modal
    const modal = document.getElementById('addAdditionalModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
    // Reset form
    document.getElementById('addAdditionalForm').reset();
}

function saveAdditional() {
    // Get form data
    const form = document.getElementById('addAdditionalForm');
    const formData = new FormData(form);
    
    // Validate required fields
    const requiredFields = ['additionalDescription', 'laborCodeSelect'];
    let isValid = true;
    
    requiredFields.forEach(field => {
        const input = document.getElementById(field);
        if (!input.value.trim()) {
            input.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            input.style.borderColor = '#d1d5db';
        }
    });
    
    if (!isValid) {
        showWarningAlert('Validation Error', 'Please fill in all required fields');
        return;
    }
    
    // Create additional cost data object with additional fields
    const additionalData = {
        id: 'additional_' + Date.now(),
        description: formData.get('additionalDescription'),
        laborCode: formData.get('laborCodeSelect'),
        createdAt: new Date().toISOString(),
        jobCardId: getCurrentJobCardId()
    };
    
    console.log('Additional cost data:', additionalData);
    
    // Save additional cost data to localStorage
    saveAdditionalToStorage(additionalData);
    
    // Add additional cost to the additional cost list
    addAdditionalToList(additionalData);
    
    // Close modal and reset form
    closeAddAdditionalModal();
}

// Function to save additional cost to localStorage
function saveAdditionalToStorage(additionalData) {
    try {
        const existingAdditional = JSON.parse(localStorage.getItem('jobCardAdditional') || '{}');
        const jobCardId = additionalData.jobCardId;
        
        if (!existingAdditional[jobCardId]) {
            existingAdditional[jobCardId] = [];
        }
        
        existingAdditional[jobCardId].push(additionalData);
        localStorage.setItem('jobCardAdditional', JSON.stringify(existingAdditional));
        console.log('Additional cost saved to localStorage:', additionalData);
    } catch (error) {
        console.error('Error saving additional cost to localStorage:', error);
        showErrorAlert('Save Error', 'Failed to save additional cost data. Please try again.');
    }
}

// Function to add additional cost to the additional cost list
function addAdditionalToList(additionalData) {
    const additionalId = additionalData.id;
    
    // Create additional cost item HTML
    const additionalItem = document.createElement('div');
    additionalItem.className = 'additional-item';
    additionalItem.setAttribute('data-dynamic', 'true');
    additionalItem.innerHTML = `
        <div class="additional-info">
            <div class="additional-description">${additionalData.description}</div>
            <div class="additional-details">
                <span class="additional-code">Code: ${additionalData.laborCode}</span>
            </div>
        </div>
        <div class="additional-actions">
            <button type="button" class="additional-edit-btn" onclick="editAdditional('${additionalId}')">
                <i class="fas fa-edit"></i>
            </button>
            <button type="button" class="additional-delete-btn" onclick="deleteAdditional('${additionalId}')">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    
    // Add additional cost to the additional cost list
    const additionalList = document.querySelector('#additional-tab-content .additional-list');
    if (additionalList) {
        // Hide "No items" message if it exists
        const noItemsMsg = additionalList.querySelector('.no-items');
        if (noItemsMsg) {
            noItemsMsg.style.display = 'none';
        }
        
        additionalList.appendChild(additionalItem);
        console.log('Additional cost added to DOM:', additionalData);
        showSuccessAlert('Additional Cost Added', 'Additional cost has been successfully added and saved!');
    } else {
        console.error('Additional cost list not found!');
        showErrorAlert('Error', 'Could not find additional cost list to add item.');
    }
}

// Function to load additional costs from localStorage
function loadAdditionalFromStorage() {
    try {
        const jobCardId = getCurrentJobCardId();
        const existingAdditional = JSON.parse(localStorage.getItem('jobCardAdditional') || '{}');
        
        if (existingAdditional[jobCardId] && existingAdditional[jobCardId].length > 0) {
            console.log('Loading additional costs from localStorage:', existingAdditional[jobCardId]);
            
            const additionalList = document.querySelector('#additional-tab-content .additional-list');
            if (additionalList) {
                // Hide "No items" message
                const noItemsMsg = additionalList.querySelector('.no-items');
                if (noItemsMsg) {
                    noItemsMsg.style.display = 'none';
                }
                
                // Clear existing dynamic additional costs
                const dynamicAdditional = additionalList.querySelectorAll('.additional-item[data-dynamic]');
                dynamicAdditional.forEach(additional => additional.remove());
                
                // Add saved additional costs
                existingAdditional[jobCardId].forEach(additionalData => {
                    addAdditionalToList(additionalData);
                });
            }
        }
    } catch (error) {
        console.error('Error loading additional costs from localStorage:', error);
    }
}

// Status dropdown functions
function toggleStatusDropdown(taskId) {
    // Close all other dropdowns
    const allDropdowns = document.querySelectorAll('.status-dropdown-menu');
    allDropdowns.forEach(dropdown => {
        if (dropdown.id !== taskId + '-dropdown') {
            dropdown.classList.remove('show');
        }
    });
    
    // Toggle current dropdown
    const dropdown = document.getElementById(taskId + '-dropdown');
    if (dropdown) {
        dropdown.classList.toggle('show');
    }
}

function changeStatus(taskId, status) {
    const button = document.querySelector(`#${taskId}-dropdown`).previousElementSibling;
    const dropdown = document.getElementById(taskId + '-dropdown');
    
    // Update button text and class
    button.textContent = status.charAt(0).toUpperCase() + status.slice(1).replace('-', ' ');
    button.className = `status-badge ${status}`;
    
    // Close dropdown
    dropdown.classList.remove('show');
    
    console.log(`Status changed for ${taskId} to ${status}`);
}

// Close dropdowns when clicking outside
document.addEventListener('click', function(event) {
    if (!event.target.closest('.status-dropdown')) {
        const allDropdowns = document.querySelectorAll('.status-dropdown-menu');
        allDropdowns.forEach(dropdown => {
            dropdown.classList.remove('show');
        });
    }
    
    if (!event.target.closest('.task-options-dropdown')) {
        const allTaskOptions = document.querySelectorAll('.task-options-menu');
        allTaskOptions.forEach(menu => {
            menu.classList.remove('show');
        });
    }
});

// Task options functions
function toggleTaskOptions(taskId) {
    // Close all other task option menus
    const allTaskMenus = document.querySelectorAll('.task-options-menu');
    allTaskMenus.forEach(menu => {
        if (menu.id !== taskId + '-options') {
            menu.classList.remove('show');
        }
    });
    
    // Toggle current menu
    const menu = document.getElementById(taskId + '-options');
    if (menu) {
        menu.classList.toggle('show');
    }
}

function editTask(taskId) {
    // Close menu
    const menu = document.getElementById(taskId + '-options');
    if (menu) {
        menu.classList.remove('show');
    }

    // Get task data from the DOM
    const taskItem = document.querySelector(`[onclick*="toggleTaskOptions('${taskId}')"]`).closest('.task-item');
    if (taskItem) {
        const taskTitle = taskItem.querySelector('.task-title').textContent;
        const taskType = taskItem.querySelector('.task-type').textContent.toLowerCase();
        const taskDescription = taskItem.querySelector('.task-description').textContent;

        // Populate edit form
        document.getElementById('editTaskName').value = taskTitle;
        document.getElementById('editTaskNote').value = taskDescription;
        selectEditTaskType(taskType);

        // Store task ID for update
        const modal = document.getElementById('editTaskModal');
        modal.setAttribute('data-task-id', taskId);

        // Open edit modal
        modal.style.display = 'block';
        document.body.classList.add('modal-open');
    }
}

function closeEditTaskModal() {
    // Close Edit Task modal
    const modal = document.getElementById('editTaskModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
    // Reset form
    document.getElementById('editTaskForm').reset();
    // Reset type selection
    selectEditTaskType('repair');
}

function selectEditTaskType(type) {
    // Remove active class from all type buttons
    const typeButtons = document.querySelectorAll('#editTaskModal .type-btn');
    typeButtons.forEach(btn => btn.classList.remove('active'));
    
    // Add active class to selected button
    const selectedButton = document.querySelector(`#editTaskModal [data-type="${type}"]`);
    if (selectedButton) {
        selectedButton.classList.add('active');
    }
    
    // Update hidden input
    document.getElementById('editTaskType').value = type;
}

function updateTask() {
    // Get form data
    const form = document.getElementById('editTaskForm');
    const formData = new FormData(form);
    const taskId = document.getElementById('editTaskModal').getAttribute('data-task-id');
    
    // Validate required fields
    const requiredFields = ['editTaskName', 'editTaskType'];
    let isValid = true;
    
    requiredFields.forEach(field => {
        const input = document.getElementById(field);
        if (!input.value.trim()) {
            input.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            input.style.borderColor = '#d1d5db';
        }
    });
    
    if (!isValid) {
        showWarningAlert('Validation Error', 'Please fill in all required fields');
        return;
    }
    
    // Update task in the DOM
    const taskItem = document.querySelector(`[onclick*="toggleTaskOptions('${taskId}')"]`).closest('.task-item');
    if (taskItem) {
        // Update task title
        taskItem.querySelector('.task-title').textContent = formData.get('editTaskName');
        
        // Update task type
        const taskType = formData.get('editTaskType');
        taskItem.querySelector('.task-type').textContent = taskType.charAt(0).toUpperCase() + taskType.slice(1);
        
        // Update task description
        taskItem.querySelector('.task-description').textContent = formData.get('editTaskNote') || 'No description provided';
        
        // Update task in localStorage
        updateTaskInStorage(taskId, {
            taskName: formData.get('editTaskName'),
            taskType: formData.get('editTaskType'),
            taskNote: formData.get('editTaskNote') || 'No description provided',
            updatedAt: new Date().toISOString()
        });
        
        showSuccessAlert('Success', 'Task updated successfully!');
        
        // Close modal
        closeEditTaskModal();
    }
}

function deleteTask(taskId) {
    console.log('=== DELETING TASK ===');
    console.log('Task ID to delete:', taskId);

    // Debug: Show current localStorage contents
    debugLocalStorage();

    // Close menu
    const menu = document.getElementById(taskId + '-options');
    if (menu) {
        menu.classList.remove('show');
    }

    // Direct deletion (avoiding custom alert/confirm issues)
    console.log('Proceeding with task deletion...');

            try {
                console.log('Confirmed deletion, removing task:', taskId);

        // Find the task item using multiple methods
        let taskItem = null;

        // Method 1: Find by data-task-id attribute (most reliable for dynamic tasks)
        taskItem = document.querySelector(`.task-item[data-task-id="${taskId}"]`);
        console.log('Method 1 - data-task-id search result:', taskItem);

                if (!taskItem) {
            // Method 2: Find by looking for elements with the taskId in onclick attributes
            const allTaskItems = document.querySelectorAll('.task-item[data-dynamic]');
            console.log('Found dynamic task items:', allTaskItems.length);
            
            for (let item of allTaskItems) {
                const optionsButton = item.querySelector('.task-options');
                if (optionsButton && optionsButton.getAttribute('onclick') && 
                    optionsButton.getAttribute('onclick').includes(`'${taskId}'`)) {
                    taskItem = item;
                    console.log('Method 2 - Found task by onclick attribute:', taskItem);
                    break;
                }
            }
                }

                if (!taskItem) {
            // Method 3: Find by element ID
            taskItem = document.getElementById(taskId);
            console.log('Method 3 - Found by ID:', taskItem);
        }

        if (!taskItem) {
            // Method 4: Find by looking for any element with this taskId in any attribute
                    const allElements = document.querySelectorAll('*');
                    for (let element of allElements) {
                if (element.getAttribute('data-task-id') === taskId || 
                    element.id === taskId ||
                    (element.getAttribute('onclick') && element.getAttribute('onclick').includes(`'${taskId}'`))) {
                    taskItem = element.closest('.task-item') || element;
                    console.log('Method 4 - Found task by attribute search:', taskItem);
                            break;
                        }
                    }
                }

                if (taskItem) {
            console.log('✅ Found task item to delete:', taskItem);

            // Remove from localStorage first
                    console.log('Removing from localStorage...');
                    removeTaskFromStorage(taskId);

                    // Remove from DOM with animation
                    console.log('Removing from DOM...');
            taskItem.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
                    taskItem.style.opacity = '0';
            taskItem.style.transform = 'translateX(-20px)';

                    setTimeout(() => {
                        taskItem.remove();
                console.log('✅ Task removed from DOM');

                        // Check if there are no more tasks and show "No items" message
                        const taskList = document.querySelector('#task-tab-content .task-list');
                        if (taskList) {
                            const remainingTasks = taskList.querySelectorAll('.task-item[data-dynamic]');
                    console.log('Remaining dynamic tasks:', remainingTasks.length);

                            if (remainingTasks.length === 0) {
                                let noItemsMsg = taskList.querySelector('.no-items');
                                if (!noItemsMsg) {
                                    noItemsMsg = document.createElement('div');
                                    noItemsMsg.className = 'no-items';
                            noItemsMsg.textContent = 'No tasks added yet. Select an inspection or add tasks manually.';
                                    noItemsMsg.style.cssText = 'text-align: center; color: #6b7280; padding: 20px; font-style: italic;';
                                    taskList.appendChild(noItemsMsg);
                                }
                                noItemsMsg.style.display = 'block';
                        console.log('✅ No items message shown');
                            }
                        }

                // Show success message
                console.log('✅ Task deleted successfully!');
                console.log('✅ Task deletion completed successfully');
                    }, 300);
                } else {
            console.error('❌ Could not find task item with ID:', taskId);
            console.log('Available task items:', document.querySelectorAll('.task-item[data-dynamic]'));
            console.log('All task items:', document.querySelectorAll('.task-item'));
            console.error('Could not find the task to delete. The task may have already been removed.');
                }
            } catch (error) {
                console.error('Error deleting task:', error);
        console.error('Failed to delete task: ' + error.message);
    }
}

// Debug function to show localStorage contents
function debugLocalStorage() {
    console.log('=== LOCALSTORAGE DEBUG ===');
    const allStorage = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');
    console.log('All localStorage contents:', allStorage);
    
    for (const [jobCardId, tasks] of Object.entries(allStorage)) {
        console.log(`Job Card ${jobCardId}:`, tasks.length, 'tasks');
        tasks.forEach((task, index) => {
            console.log(`  Task ${index + 1}:`, {
                id: task.id,
                taskName: task.taskName,
                jobCardId: task.jobCardId
            });
        });
    }
    
    // Also show current job card ID
    console.log('Current job card ID:', getCurrentJobCardId());
}

// Simple visual feedback function
function showTaskCreationFeedback(message) {
    // Create a simple notification
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #3b82f6;
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        font-weight: 600;
        z-index: 10000;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    `;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Remove after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 3000);
}

// Debug Firebase data function
async function debugFirebaseData(firebaseKey) {
    console.log('=== DEBUGGING FIREBASE DATA ===');
    console.log('Firebase Key:', firebaseKey);
    
    try {
        const inspectionDetails = await fetchInspectionDetails(firebaseKey);
        console.log('Raw API Response:', inspectionDetails);
        
        if (inspectionDetails && inspectionDetails.faultyItems) {
            console.log('Faulty Items Found:', inspectionDetails.faultyItems.length);
            inspectionDetails.faultyItems.forEach((item, index) => {
                console.log(`Item ${index + 1}:`, {
                    id: item.id,
                    name: item.name,
                    status: item.status,
                    note: item.note,
                    section: item.section
                });
            });
        } else {
            console.log('No faulty items found in API response');
            console.log('Full response structure:', Object.keys(inspectionDetails || {}));
        }
        
        // Show alert with key information
        const faultyCount = inspectionDetails?.faultyItems?.length || 0;
        alert(`Firebase Debug Results:\n\nFirebase Key: ${firebaseKey}\nFaulty Items Found: ${faultyCount}\n\nCheck console for detailed information.`);
        
    } catch (error) {
        console.error('Error debugging Firebase data:', error);
        alert('Error debugging Firebase data: ' + error.message);
    }
}

// Test Firebase API function
async function testFirebaseAPI() {
    console.log('=== TESTING FIREBASE API ===');
    
    try {
        // Get all inspection reports first
        const response = await fetch('/api/fetch-inspection-reports', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
            }
        });

        const result = await response.json();
        console.log('Inspection reports API response:', result);

        if (result.success && result.inspections && result.inspections.length > 0) {
            // Find an inspection with faulty items
            const inspectionWithFaults = result.inspections.find(inspection => 
                inspection.hasFaultyItems && inspection.faultyItemsCount > 0
            );

            if (inspectionWithFaults) {
                console.log('Found inspection with faulty items:', inspectionWithFaults);
                
                // Test the detailed API
                const detailedResponse = await fetchInspectionDetails(inspectionWithFaults.firebaseKey);
                console.log('Detailed API response:', detailedResponse);
                
                if (detailedResponse && detailedResponse.faultyItems) {
                    console.log('✅ Firebase API Test SUCCESSFUL!');
                    console.log(`Found ${detailedResponse.faultyItems.length} faulty items:`);
                    detailedResponse.faultyItems.forEach((item, index) => {
                        console.log(`  ${index + 1}. ${item.name} (Status: ${item.status})`);
                    });
                    
                    alert(`✅ Firebase API Test SUCCESSFUL!\n\nFound ${detailedResponse.faultyItems.length} faulty items:\n${detailedResponse.faultyItems.map((item, i) => `${i + 1}. ${item.name}`).join('\n')}\n\nCheck console for full details.`);
                } else {
                    console.log('❌ Firebase API Test FAILED - No faulty items in detailed response');
                    alert('❌ Firebase API Test FAILED\n\nNo faulty items found in detailed API response.\nCheck console for details.');
                }
            } else {
                console.log('No inspections with faulty items found');
                alert('No inspections with faulty items found to test.');
            }
        } else {
            console.log('Failed to fetch inspection reports');
            alert('Failed to fetch inspection reports from API.');
        }
        
    } catch (error) {
        console.error('Error testing Firebase API:', error);
        alert('Error testing Firebase API: ' + error.message);
    }
}

// Function to remove task from localStorage
function removeTaskFromStorage(taskId) {
    try {
        console.log('=== REMOVING TASK FROM STORAGE ===');
        console.log('Task ID to remove:', taskId);
        
        const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');
        console.log('All tasks in storage:', existingTasks);
        
        let taskRemoved = false;
        let jobCardIdUsed = null;
        
        // Search through all job cards to find the task
        for (const [jobCardId, tasks] of Object.entries(existingTasks)) {
            console.log(`Checking job card: ${jobCardId} with ${tasks.length} tasks`);
            
            const originalLength = tasks.length;
            const filteredTasks = tasks.filter(task => {
                const match = task.id === taskId;
                console.log(`Checking task: ${task.id} vs ${taskId}, match: ${match}`);
                return !match; // Keep tasks that don't match (remove the matching one)
            });
            
            if (filteredTasks.length < originalLength) {
                console.log(`✅ Found and removing task from job card: ${jobCardId}`);
                existingTasks[jobCardId] = filteredTasks;
                taskRemoved = true;
                jobCardIdUsed = jobCardId;
                break;
            }
        }
        
        if (taskRemoved) {
            // Save back to localStorage
            localStorage.setItem('jobCardTasks', JSON.stringify(existingTasks));
            console.log(`✅ Task removed from localStorage for job card: ${jobCardIdUsed}`);
            console.log('Updated tasks in storage:', existingTasks);
        } else {
            console.log('❌ Task not found in any job card storage');
            console.log('Available tasks:', Object.keys(existingTasks).map(key => `${key}: ${existingTasks[key].length} tasks`));
        }
    } catch (error) {
        console.error('Error removing task from localStorage:', error);
    }
}

// Function to update task in localStorage
function updateTaskInStorage(taskId, updatedData) {
    try {
        const jobCardId = getCurrentJobCardId();
        const existingTasks = JSON.parse(localStorage.getItem('jobCardTasks') || '{}');
        
        if (existingTasks[jobCardId]) {
            // Find and update the task
            const taskIndex = existingTasks[jobCardId].findIndex(task => task.id === taskId);
            if (taskIndex !== -1) {
                // Update the task data
                existingTasks[jobCardId][taskIndex] = {
                    ...existingTasks[jobCardId][taskIndex],
                    ...updatedData
                };
                
                // Save back to localStorage
                localStorage.setItem('jobCardTasks', JSON.stringify(existingTasks));
                console.log('Task updated in localStorage:', taskId, updatedData);
            }
        }
    } catch (error) {
        console.error('Error updating task in localStorage:', error);
    }
}

// Additional Details Functions
function openJobCardTypeModal() {
    // Open Job Card Type modal
    const modal = document.getElementById('jobCardTypeModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
}

function closeJobCardTypeModal() {
    // Close Job Card Type modal
    const modal = document.getElementById('jobCardTypeModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
}

function selectJobCardType(type) {
    // Update the display text
    const displayText = type.charAt(0).toUpperCase() + type.slice(1);
    document.getElementById('jobCardTypeDisplay').textContent = displayText;
    
    // Update the hidden input
    document.getElementById('jobCardType').value = type;
    
    // Close the modal
    closeJobCardTypeModal();
}

function openInspectionModal() {
    document.getElementById('inspectionModal').style.display = 'block';
    loadInspectionData();
}

function openJobCardSelectionModal() {
    showInfoAlert('Job Card Selection', 'Job Card selection modal would open here');
    // This would open a modal with job card options
}

// Cost Details Functions
function calculateTotalWOCost() {
    const partMaterialCost = parseFloat(document.getElementById('partMaterialCost').value) || 0;
    const totalLaborCost = parseFloat(document.getElementById('totalLaborCost').value) || 0;
    const additionalCost = parseFloat(document.getElementById('additionalCost').value) || 0;
    const taxAmount = parseFloat(document.getElementById('taxAmount').value) || 0;
    
    const totalWOCost = partMaterialCost + totalLaborCost + additionalCost + taxAmount;
    document.getElementById('totalWOCost').value = totalWOCost.toFixed(2);
}

// Add event listeners for cost calculation
document.addEventListener('DOMContentLoaded', function() {
    const costFields = ['partMaterialCost', 'totalLaborCost', 'additionalCost', 'taxAmount'];
    costFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            field.addEventListener('input', calculateTotalWOCost);
        }
    });
});

function handleGetLocation() {
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const coordinates = `${position.coords.latitude}, ${position.coords.longitude}`;
                document.getElementById('coordinates').textContent = coordinates;
                document.getElementById('coordinatesInput').value = coordinates;
                document.getElementById('gpsDetailsBox').style.display = 'block';
                
                // You could also reverse geocode to get address
                document.getElementById('gpsAddress').textContent = 'Location retrieved successfully';
                document.getElementById('gpsAddressInput').value = 'Location retrieved successfully';
            },
            function(error) {
                showErrorAlert('Location Error', 'Error getting location: ' + error.message);
            }
        );
    } else {
        showErrorAlert('Location Error', 'Geolocation is not supported by this browser.');
    }
}

function goToStep2() {
    // Validate form
    const requiredFields = document.querySelectorAll('[required]');
    let isValid = true;

    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            field.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            field.style.borderColor = '#d1d5db';
        }
    });

    if (isValid) {
        showSuccessAlert('Form Valid', 'Form is valid! Would proceed to step 2.');
        // Here you would typically show the next step or submit the form
    } else {
        showWarningAlert('Validation Error', 'Please fill in all required fields.');
    }
}

// Manual create tasks from faulty items function
async function manualCreateTasks(event, inspectionData) {
    event.stopPropagation(); // Prevent the inspection item from being selected

    try {
        console.log('Manual task creation triggered for inspection:', inspectionData);

        // Show loading alert
        showInfoAlert('Creating Tasks', 'Manually creating tasks from faulty items...');

        // Fetch inspection details
        const inspectionDetails = await fetchInspectionDetails(inspectionData.firebaseKey);

        if (inspectionDetails && inspectionDetails.faultyItems && inspectionDetails.faultyItems.length > 0) {
            // Create tasks from faulty items
            await createTasksFromFaultyItems(inspectionDetails.faultyItems, inspectionData);

            const inspectionNumber = inspectionData.inspectionNumber || inspectionData.requestNumber || inspectionData.firebaseKey || 'N/A';
            const faultyCount = inspectionDetails.faultyItems.length;

            showSuccessAlert('Manual Task Creation Successful',
                `${faultyCount} task${faultyCount === 1 ? '' : 's'} manually created from inspection ${inspectionNumber}\n\nTasks are now visible in the Tasks & Items section.`
            );
        } else {
            showInfoAlert('No Faulty Items', 'This inspection has no faulty items to create tasks from.');
        }
    } catch (error) {
        console.error('Error in manual task creation:', error);
        showErrorAlert('Manual Task Creation Error', 'Failed to create tasks: ' + error.message);
    }
}
</script>
@endsection