@extends('welcome')

@section('title', 'Job Cards')

@section('content')
<div class="job-card-container" role="main">
    <!-- Header Section -->
    <header class="job-card-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Job Cards</h1>
                <p>Manage all job cards and work orders</p>
            </div>
            <div class="header-actions">
                <div class="search-container">
                    <div class="search-input-wrapper">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search job cards..." id="searchInput" aria-label="Search job cards" autocomplete="off">
                    </div>
                </div>
                <div class="action-buttons">
                    <button class="btn-primary" onclick="window.location.href='/inspection/job_card'" aria-label="Create new job card">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <line x1="12" y1="5" x2="12" y2="19"></line>
                            <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                        Create Job Card
                    </button>
                    <button class="btn-secondary" data-bs-toggle="modal" data-bs-target="#sortingModal" aria-label="Open sorting options">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M3 6h18M7 12h10M10 18h4"/>
                        </svg>
                        Sorting
                    </button>
                    <button class="btn-success" onclick="exportToExcel()" aria-label="Export data to Excel">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                            <line x1="16" y1="13" x2="8" y2="13"></line>
                            <line x1="16" y1="17" x2="8" y2="17"></line>
                            <polyline points="10,9 9,9 8,9"></polyline>
                        </svg>
                        Excel Export
                    </button>
                    <button class="btn-info" onclick="refreshData()" aria-label="Refresh data">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="23 4 23 10 17 10"></polyline>
                            <polyline points="1 20 1 14 7 14"></polyline>
                            <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"></path>
                        </svg>
                        Refresh
                    </button>
                </div>
            </div>
        </div>
    </header>

    <!-- Table Section -->
    <section class="table-section" role="region" aria-label="Job cards table">
        <div class="table-container">
            <table class="job-card-table" id="jobCardTable" role="table" aria-label="Job cards">
                <thead>
                    <tr role="row">
                        <th scope="col">S.No</th>
                        <th scope="col">Job Card #</th>
                        <th scope="col">Title</th>
                        <th scope="col">Priority</th>
                        <th scope="col">Status</th>
                        <th scope="col">Equipment</th>
                        <th scope="col">Project</th>
                        <th scope="col">Start Date</th>
                        <th scope="col">Due Date</th>
                        <th scope="col">Assigned To</th>
                        <th scope="col">Created Date</th>
                    </tr>
                </thead>
                <tbody id="jobCardTableBody" role="rowgroup">
                    <tr class="loading-row" role="row">
                        <td colspan="11" class="loading-cell" role="cell">
                            <div class="loading-content">
                                <div class="spinner" aria-hidden="true"></div>
                                <span>Loading job cards...</span>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>

        <!-- Pagination Section -->
        <div class="pagination-section" id="paginationSection" style="display: none;">
            <div class="pagination-container">
                <div class="pagination-info">
                    <span id="paginationInfo">Showing 1-10 of 0 entries</span>
                </div>
                <div class="pagination-controls">
                    <button class="pagination-btn pagination-first" onclick="goToPage(1)" title="First Page">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M11 17l-5-5 5-5"></path>
                            <path d="M18 17l-5-5 5-5"></path>
                        </svg>
                    </button>
                    <button class="pagination-btn pagination-prev" onclick="goToPage(paginationState.currentPage - 1)" title="Previous Page">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M15 18l-6-6 6-6"></path>
                        </svg>
                    </button>

                    <div class="pagination-pages" id="paginationPages">
                        <!-- Page numbers will be inserted here -->
                    </div>

                    <button class="pagination-btn pagination-next" onclick="goToPage(paginationState.currentPage + 1)" title="Next Page">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M9 18l6-6-6-6"></path>
                        </svg>
                    </button>
                    <button class="pagination-btn pagination-last" onclick="goToPage(paginationState.totalPages)" title="Last Page">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M13 17l5-5-5-5"></path>
                            <path d="M6 17l5-5-5-5"></path>
                        </svg>
                    </button>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Sorting Modal -->
<div class="modal fade" id="sortingModal" tabindex="-1" aria-labelledby="sortingModalLabel" aria-hidden="true" role="dialog">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="sortingModalLabel">Sorting Options</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close modal"></button>
            </div>
            <div class="modal-body">
                <form id="sortingForm" role="form" novalidate>
                    <div class="sort-section">
                        <h6 class="section-title">Sort Order</h6>
                        <div class="radio-group">
                            <div class="radio-item">
                                <input type="radio" id="ascending" name="sortOrder" value="ascending" checked>
                                <label for="ascending">
                                    <span class="radio-custom"></span>
                                    Ascending
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="descending" name="sortOrder" value="descending">
                                <label for="descending">
                                    <span class="radio-custom"></span>
                                    Descending
                                </label>
                            </div>
                        </div>
                    </div>

                    <div class="sort-section">
                        <h6 class="section-title">Sort By</h6>
                        <div class="radio-group">
                            <div class="radio-item">
                                <input type="radio" id="serialNumber" name="sortBy" value="serialNumber" checked>
                                <label for="serialNumber">
                                    <span class="radio-custom"></span>
                                    S.No
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="jobCardNumber" name="sortBy" value="jobCardNumber">
                                <label for="jobCardNumber">
                                    <span class="radio-custom"></span>
                                    Job Card #
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="title" name="sortBy" value="title">
                                <label for="title">
                                    <span class="radio-custom"></span>
                                    Title
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="priority" name="sortBy" value="priority">
                                <label for="priority">
                                    <span class="radio-custom"></span>
                                    Priority
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="status" name="sortBy" value="status">
                                <label for="status">
                                    <span class="radio-custom"></span>
                                    Status
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="startDate" name="sortBy" value="startDate">
                                <label for="startDate">
                                    <span class="radio-custom"></span>
                                    Start Date
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="dueDate" name="sortBy" value="dueDate">
                                <label for="dueDate">
                                    <span class="radio-custom"></span>
                                    Due Date
                                </label>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-cancel" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn-reset" onclick="resetSorting()">Reset</button>
                <button type="button" class="btn-apply" onclick="applySorting()">Apply</button>
            </div>
        </div>
    </div>
</div>

<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003d7a;
    --secondary-color: #3b82f6;
    --secondary-hover: #2563eb;
    --success-color: #10b981;
    --success-hover: #059669;
    --warning-color: #f59e0b;
    --warning-hover: #d97706;
    --info-color: #ffd205;
    --info-hover: #fbbf24;
    --text-primary: #0f172a;
    --text-secondary: #374151;
    --text-muted: #64748b;
    --text-light: #9ca3af;
    --bg-primary: #f8fafc;
    --bg-secondary: #f1f5f9;
    --bg-white: #ffffff;
    --bg-hover: rgba(59, 130, 246, 0.05);
    --border-color: #e2e8f0;
    --border-light: #d1d5db;
    --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    --shadow-md: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    --radius-sm: 0;
    --radius-md: 0;
    --radius-lg: 0;
    --transition-base: all 0.2s ease;
    --transition-slow: all 0.3s ease;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1050;
    --z-index-notification: 10000;
}

/* Main Container */
.job-card-container {
    min-height: 100vh;
}

/* Header Section */
.job-card-header {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-md);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.header-content {
    padding: 32px;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 24px;
}

.page-title h1 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.page-title p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

.header-actions {
    display: flex;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

/* Search Container */
.search-container {
    position: relative;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 16px;
    z-index: 2;
    color: var(--text-muted);
    width: 16px;
    height: 16px;
}

.search-input {
    padding: 12px 16px 12px 44px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-sm);
    height: 44px;
    width: 300px;
    font-size: 14px;
    font-weight: 400;
    color: var(--text-secondary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--secondary-color);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: var(--text-light);
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 12px;
    align-items: center;
}

/* Base Button Styles */
.btn-primary, .btn-secondary, .btn-success, .btn-info {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    transition: var(--transition-base);
    cursor: pointer;
    box-shadow: var(--shadow-sm);
}

.btn-primary {
    background: var(--primary-color);
    color: var(--bg-white);
}

.btn-primary:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--bg-white);
}

.btn-secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: var(--bg-white);
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4b5563 0%, #374151 100%);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--bg-white);
}

.btn-success {
    background: var(--success-color);
    color: var(--bg-white);
}

.btn-success:hover {
    background: var(--success-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--bg-white);
}

.btn-info {
    background: var(--info-color);
    color: var(--bg-white);
}

.btn-info:hover {
    background: var(--info-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: var(--bg-white);
}

/* Table Section */
.table-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-md);
    border-radius: var(--radius-sm);
    overflow: hidden;
}

.table-container {
    overflow-x: auto;
}

.job-card-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.job-card-table thead {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.job-card-table th {
    padding: 16px 12px;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e2e8f0;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    white-space: nowrap;
}

.job-card-table td {
    padding: 16px 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
    white-space: nowrap;
}

.table-row:hover {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

/* Priority Badges */
.priority-badge {
    display: inline-block;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.priority-low {
    background: #f0fdf4;
    color: #16a34a;
    border: 1px solid #bbf7d0;
}

.priority-medium {
    background: #fffbeb;
    color: #d97706;
    border: 1px solid #fed7aa;
}

.priority-high {
    background: #fef2f2;
    color: #dc2626;
    border: 1px solid #fecaca;
}

.priority-emergency {
    background: #7f1d1d;
    color: #ffffff;
    border: 1px solid #991b1b;
}

/* Status Badges */
.status-badge {
    display: inline-block;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-initiated {
    background: #f1f5f9;
    color: #64748b;
}

.status-in-progress {
    background: #fef3c7;
    color: #d97706;
}

.status-on-hold {
    background: #fed7aa;
    color: #ea580c;
}

.status-complete {
    background: #d1fae5;
    color: #059669;
}

/* Equipment Info */
.equipment-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.equipment-number {
    font-weight: 600;
    color: #1e293b;
    font-size: 14px;
}

.equipment-name {
    font-size: 12px;
    color: #64748b;
    font-weight: 400;
}

/* Pagination Styles */
.pagination-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-top: none;
    border-radius: 0 0 var(--radius-sm) var(--radius-sm);
    padding: 16px 24px;
}

.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 16px;
}

.pagination-info {
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 400;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 4px;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 36px;
    height: 36px;
    border: 1px solid var(--border-light);
    background: var(--bg-white);
    color: var(--text-secondary);
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition-base);
    font-size: 14px;
    font-weight: 500;
}

.pagination-btn:hover:not(:disabled) {
    background: var(--bg-hover);
    border-color: var(--secondary-color);
    color: var(--secondary-color);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    background: var(--bg-secondary);
}

.pagination-btn svg {
    width: 16px;
    height: 16px;
}

.pagination-pages {
    display: flex;
    align-items: center;
    gap: 2px;
    margin: 0 8px;
}

.pagination-page {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 36px;
    height: 36px;
    padding: 0 8px;
    border: 1px solid var(--border-light);
    background: var(--bg-white);
    color: var(--text-secondary);
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition-base);
    font-size: 14px;
    font-weight: 500;
}

.pagination-page:hover {
    background: var(--bg-hover);
    border-color: var(--secondary-color);
    color: var(--secondary-color);
}

.pagination-page.active {
    background: var(--secondary-color);
    border-color: var(--secondary-color);
    color: var(--bg-white);
}

.pagination-ellipsis {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 36px;
    height: 36px;
    color: var(--text-muted);
    font-size: 14px;
    user-select: none;
}

/* Loading State */
.loading-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.loading-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.spinner {
    width: 32px;
    height: 32px;
    border: 3px solid #e2e8f0;
    border-top: 3px solid #3b82f6;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-content span {
    color: #64748b;
    font-weight: 400;
    font-size: 14px;
}

/* No Data Styling */
.no-data-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.no-data-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.no-data-content svg {
    color: #9ca3af;
}

.no-data-content h3 {
    color: #374151;
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.no-data-content p {
    color: #6b7280;
    font-weight: 400;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

/* Modal Styling */
.modal-content {
    border: 1px solid #e2e8f0;
    border-radius: 0;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    z-index: 1000222;
}

.modal-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.modal-title {
    color: #0f172a;
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.btn-close {
    background: none;
    border: none;
    font-size: 24px;
    color: #64748b;
    opacity: 1;
    transition: all 0.2s ease;
}

.btn-close:hover {
    color: #374151;
    transform: scale(1.1);
}

.modal-body {
    padding: 24px;
}

.sort-section {
    margin-bottom: 24px;
}

.section-title {
    color: #374151;
    font-weight: 600;
    font-size: 14px;
    margin-bottom: 12px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.radio-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.radio-item {
    display: flex;
    align-items: center;
    gap: 12px;
}

.radio-item input[type="radio"] {
    display: none;
}

.radio-item label {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    transition: all 0.2s ease;
}

.radio-custom {
    width: 18px;
    height: 18px;
    border: 2px solid #d1d5db;
    border-radius: 50%;
    position: relative;
    transition: all 0.2s ease;
}

.radio-item input[type="radio"]:checked + label .radio-custom {
    border-color: #3b82f6;
    background: #3b82f6;
}

.radio-item input[type="radio"]:checked + label .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 6px;
    height: 6px;
    background: white;
    border-radius: 50%;
}

.radio-item label:hover {
    color: #1e293b;
}

.radio-item label:hover .radio-custom {
    border-color: #3b82f6;
}

.modal-footer {
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
    padding: 16px 24px;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
}

.btn-cancel, .btn-reset, .btn-apply {
    padding: 10px 20px;
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
}

.btn-cancel:hover {
    background: #e5e7eb;
    color: #1f2937;
}

.btn-reset {
    background: #fef3c7;
    color: #92400e;
}

.btn-reset:hover {
    background: #fde68a;
    color: #78350f;
}

.btn-apply {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: white;
}

.btn-apply:hover {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .job-card-container {
        padding: 20px;
    }

    .header-content {
        padding: 24px;
    }

    .search-input {
        width: 250px;
    }

    .table-section {
        margin-bottom: 20px;
    }
}

@media (max-width: 992px) {
    .job-card-container {
        padding: 16px;
    }

    .header-content {
        padding: 20px;
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
    }

    .page-title h1 {
        font-size: 28px;
    }

    .page-title p {
        font-size: 15px;
    }

    .header-actions {
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 16px;
    }

    .search-input {
        width: 100%;
        max-width: 300px;
    }

    .action-buttons {
        flex-wrap: wrap;
    }

    .btn-primary, .btn-secondary, .btn-success, .btn-info {
        padding: 11px 18px;
        font-size: 14px;
    }

    .table-section {
        margin-bottom: 18px;
    }

    .job-card-table {
        font-size: 13px;
    }

    .job-card-table th,
    .job-card-table td {
        padding: 14px 10px;
    }

    .modal-dialog {
        margin: 1rem;
        max-width: calc(100% - 2rem);
    }

    .modal-body {
        padding: 20px;
    }

    .modal-footer {
        padding: 16px 20px;
        flex-direction: column;
        gap: 8px;
    }

    .btn-cancel, .btn-reset, .btn-apply {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 768px) {
    .job-card-container {
        padding: 0;
    }

    .job-card-header {
        margin: 0 16px 16px 16px;
    }

    .header-content {
        padding: 16px;
        gap: 16px;
    }

    .page-title h1 {
        font-size: 24px;
        margin-bottom: 6px;
    }

    .page-title p {
        font-size: 14px;
    }

    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }

    .search-container {
        width: 100%;
    }

    .search-input {
        width: 100%;
        padding: 12px 16px 12px 44px;
        font-size: 16px;
        min-height: 48px;
    }

    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
        gap: 8px;
    }

    .btn-primary, .btn-secondary, .btn-success, .btn-info {
        flex: 1;
        justify-content: center;
        min-width: 140px;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }

    .table-section {
        margin: 0 16px 16px 16px;
    }

    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }

    .job-card-table {
        min-width: 1200px;
        font-size: 12px;
    }

    .job-card-table th,
    .job-card-table td {
        padding: 12px 8px;
        white-space: nowrap;
    }

    .job-card-table th {
        font-size: 11px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        background-color: #f8fafc;
    }

    .loading-cell {
        padding: 48px 16px !important;
    }

    .spinner {
        width: 28px;
        height: 28px;
        border-width: 2px;
    }

    .loading-content span {
        font-size: 13px;
    }

    .no-data-cell {
        padding: 48px 16px !important;
    }

    .no-data-content svg {
        width: 40px;
        height: 40px;
    }

    .no-data-content h3 {
        font-size: 16px;
    }

    .no-data-content p {
        font-size: 13px;
        max-width: 300px;
    }

    .modal-dialog {
        margin: 0.5rem;
        max-width: calc(100% - 1rem);
    }

    .modal-header {
        padding: 16px;
    }

    .modal-title {
        font-size: 16px;
    }

    .modal-body {
        padding: 16px;
    }

    .sort-section {
        margin-bottom: 20px;
    }

    .section-title {
        font-size: 13px;
        margin-bottom: 10px;
    }

    .radio-group {
        gap: 6px;
    }

    .radio-item {
        gap: 10px;
    }

    .radio-item label {
        font-size: 13px;
    }

    .radio-custom {
        width: 16px;
        height: 16px;
    }

    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 5px;
        height: 5px;
    }

    .modal-footer {
        padding: 12px 16px;
        flex-direction: column;
        gap: 8px;
    }

    .btn-cancel, .btn-reset, .btn-apply {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }
}

@media (max-width: 576px) {
    .job-card-container {
        padding: 0;
    }

    .job-card-header {
        margin: 0 12px 12px 12px;
    }

    .header-content {
        padding: 12px;
        gap: 12px;
    }

    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }

    .page-title p {
        font-size: 13px;
    }

    .search-input {
        padding: 10px 14px 10px 40px;
        font-size: 16px;
        min-height: 44px;
    }

    .action-buttons {
        gap: 6px;
    }

    .btn-primary, .btn-secondary, .btn-success, .btn-info {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 44px;
        min-width: 120px;
    }

    .table-section {
        margin: 0 12px 12px 12px;
    }

    .job-card-table {
        min-width: 1000px;
        font-size: 11px;
    }

    .job-card-table th,
    .job-card-table td {
        padding: 10px 6px;
    }

    .job-card-table th {
        font-size: 10px;
    }

    .loading-cell {
        padding: 40px 12px !important;
    }

    .spinner {
        width: 24px;
        height: 24px;
    }

    .loading-content span {
        font-size: 12px;
    }

    .no-data-cell {
        padding: 40px 12px !important;
    }

    .no-data-content svg {
        width: 36px;
        height: 36px;
    }

    .no-data-content h3 {
        font-size: 15px;
    }

    .no-data-content p {
        font-size: 12px;
        max-width: 250px;
    }

    .modal-dialog {
        margin: 0.25rem;
        max-width: calc(100% - 0.5rem);
    }

    .modal-header {
        padding: 12px;
    }

    .modal-title {
        font-size: 15px;
    }

    .modal-body {
        padding: 12px;
    }

    .sort-section {
        margin-bottom: 16px;
    }

    .section-title {
        font-size: 12px;
        margin-bottom: 8px;
    }

    .radio-group {
        gap: 4px;
    }

    .radio-item {
        gap: 8px;
    }

    .radio-item label {
        font-size: 12px;
    }

    .radio-custom {
        width: 14px;
        height: 14px;
    }

    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 4px;
        height: 4px;
    }

    .modal-footer {
        padding: 10px 12px;
        flex-direction: column;
        gap: 8px;
    }

    .btn-cancel, .btn-reset, .btn-apply {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }
}

/* Accessibility and Focus Styles */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

.btn-primary:focus,
.btn-secondary:focus,
.btn-success:focus,
.btn-info:focus,
.search-input:focus,
.pagination-btn:focus,
.pagination-page:focus,
.btn-cancel:focus,
.btn-reset:focus,
.btn-apply:focus {
    outline: 2px solid var(--secondary-color);
    outline-offset: 2px;
}

/* Smooth animations */
* {
    transition: var(--transition-base);
}

/* Icon improvements */
.btn-primary svg,
.btn-secondary svg,
.btn-success svg,
.btn-info svg,
.pagination-btn svg {
    flex-shrink: 0;
}

/* Print styles */
@media print {
    .job-card-container {
        background: white !important;
        padding: 0 !important;
    }

    .job-card-header,
    .table-section {
        box-shadow: none !important;
        border: 1px solid #000 !important;
    }

    .btn-primary,
    .btn-secondary,
    .btn-success,
    .btn-info,
    .pagination-btn {
        display: none !important;
    }
}
</style>

<script>
(function() {
    'use strict';

    // Configuration object
    const CONFIG = {
        selectors: {
            searchInput: '#searchInput',
            jobCardTableBody: '#jobCardTableBody',
            sortingModal: '#sortingModal',
            sortingForm: '#sortingForm'
        },
        messages: {
            loading: 'Loading job cards...',
            noData: 'No Data Available',
            noDataDescription: 'No job cards found. Start by creating a new job card.',
            errorLoading: 'Error loading job cards:',
            failedLoading: 'Failed to load job cards',
            tryAgain: 'Please try again later',
            sortingApplied: 'Sorting applied successfully!',
            sortingReset: 'Sorting options reset!',
            dataRefreshed: 'Data refreshed successfully!',
            noDataExport: 'No data to export',
            exportCompleted: 'Excel export completed!'
        },
        timing: {
            notificationTimeout: 3000
        },
        endpoints: {
            getJobCards: '/get-job-cards'
        }
    };

    // State management
    let jobCardData = [];
    let filteredData = [];
    let paginationState = {
        currentPage: 1,
        perPage: 10,
        total: 0,
        totalPages: 0
    };

    // DOM elements cache
    let elements = {};

    // Initialize DOM elements
    function initElements() {
        elements = {
            searchInput: document.querySelector(CONFIG.selectors.searchInput),
            jobCardTableBody: document.querySelector(CONFIG.selectors.jobCardTableBody),
            sortingModal: document.querySelector(CONFIG.selectors.sortingModal),
            sortingForm: document.querySelector(CONFIG.selectors.sortingForm)
        };
    }

    // Initialize search functionality
    function initSearch() {
        if (elements.searchInput) {
            elements.searchInput.addEventListener('input', function() {
                filterJobCards(this.value);
            });
        }
    }

    // Initialize page
    function init() {
        initElements();
        initSearch();
        loadJobCards();
    }

    // Event listeners
    document.addEventListener('DOMContentLoaded', init);

    // Load job cards
    function loadJobCards() {
        if (!elements.jobCardTableBody) return;

        // Show loading state
        showLoadingState();

        console.log('🔄 Loading job cards...');

        // Fetch job cards with pagination
        fetch(`${CONFIG.endpoints.getJobCards}?page=${paginationState.currentPage}&per_page=${paginationState.perPage}`)
            .then(response => {
                console.log('📡 Response status:', response.status);
                return response.json();
            })
            .then(data => {
                console.log('📊 Response data:', data);

                if (data.success) {
                    jobCardData = data.jobCards || [];
                    filteredData = [...jobCardData];

                    // Update pagination state
                    if (data.pagination) {
                        paginationState = {
                            currentPage: data.pagination.current_page,
                            perPage: data.pagination.per_page,
                            total: data.pagination.total,
                            totalPages: data.pagination.total_pages
                        };
                    }

                    displayJobCards(filteredData);
                    updatePaginationInfo();

                    console.log('✅ Job cards loaded successfully:', {
                        count: jobCardData.length,
                        pagination: data.pagination,
                        sample: jobCardData.length > 0 ? jobCardData[0] : 'No data'
                    });
                } else {
                    console.error('❌ API returned error:', data);
                    showErrorState(CONFIG.messages.errorLoading + ' ' + data.message);
                    showNotification(CONFIG.messages.errorLoading + ' ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('❌ Error loading job cards:', error);

                // Show more detailed error information
                let errorMessage = CONFIG.messages.failedLoading;
                if (error.message) {
                    errorMessage += ': ' + error.message;
                }

                showErrorState(errorMessage);
                showNotification(errorMessage, 'error');

                // Also show suggestion to check Firebase debug
                setTimeout(() => {
                    showNotification('💡 Try clicking "Debug Firebase" to check your database connection', 'info');
                }, 2000);
            });
    }

    // Show loading state
    function showLoadingState() {
        if (!elements.jobCardTableBody) return;

        elements.jobCardTableBody.innerHTML = `
            <tr class="loading-row">
                <td colspan="11" class="loading-cell">
                    <div class="loading-content">
                        <div class="spinner"></div>
                        <span>${CONFIG.messages.loading}</span>
                    </div>
                </td>
            </tr>
        `;
    }

    // Show error state
    function showErrorState(message) {
        if (!elements.jobCardTableBody) return;

        elements.jobCardTableBody.innerHTML = `
            <tr>
                <td colspan="11" class="text-center py-4 text-muted">
                    <i class="fas fa-exclamation-triangle fa-2x mb-2 text-warning"></i>
                    <p class="mb-0">Error loading data</p>
                    <small class="text-muted">${message}</small>
                </td>
            </tr>
        `;
    }

    function displayJobCards(jobCards) {
        const tableBody = document.getElementById('jobCardTableBody');

        if (!jobCards || jobCards.length === 0) {
            tableBody.innerHTML = `
                <tr class="no-data-row">
                    <td colspan="11" class="no-data-cell">
                        <div class="no-data-content">
                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                <circle cx="12" cy="12" r="10"></circle>
                                <line x1="12" y1="8" x2="12" y2="12"></line>
                                <line x1="12" y1="16" x2="12.01" y2="16"></line>
                            </svg>
                            <h3>No Data Available</h3>
                            <p>No job cards found. Start by creating a new job card.</p>
                        </div>
                    </td>
                </tr>
            `;
            return;
        }

        // Add index to each job card for S.No
        const rows = jobCards.map((jobCard, index) => {
            const serialNumber = ((paginationState.currentPage - 1) * paginationState.perPage) + index + 1;

            // Format dates
            const startDate = jobCard.startDate ? new Date(jobCard.startDate) : null;
            const dueDate = jobCard.dueDate ? new Date(jobCard.dueDate) : null;
            const createdDate = jobCard.createdAt ? new Date(jobCard.createdAt) : null;

            const formattedStartDate = startDate ? startDate.toLocaleDateString() : 'N/A';
            const formattedDueDate = dueDate ? dueDate.toLocaleDateString() : 'N/A';
            const formattedCreatedDate = createdDate ? createdDate.toLocaleDateString() : 'N/A';

            // Format priority
            const priorityClass = `priority-${jobCard.priority?.toLowerCase() || 'medium'}`;
            const priorityLabel = jobCard.priority || 'Medium';

            // Format status
            const statusClass = `status-${jobCard.status?.toLowerCase().replace(' ', '-') || 'initiated'}`;
            const statusLabel = jobCard.status || 'Initiated';

            return `
                <tr class="table-row">
                    <td class="serial-cell">
                        <span class="serial-number">${serialNumber}</span>
                    </td>
                    <td class="job-card-number">
                        <span class="job-card-id">${jobCard.jobCardNumber || 'N/A'}</span>
                    </td>
                    <td class="title-cell">
                        <span class="job-card-title">${jobCard.title || 'N/A'}</span>
                    </td>
                    <td class="priority-cell">
                        <span class="priority-badge ${priorityClass}">${priorityLabel}</span>
                    </td>
                    <td class="status-cell">
                        <span class="status-badge ${statusClass}">${statusLabel}</span>
                    </td>
                    <td class="equipment-cell">
                        <div class="equipment-info">
                            <span class="equipment-number">${jobCard.equipmentNumber || 'N/A'}</span>
                            <span class="equipment-name">${jobCard.equipmentName || ''}</span>
                        </div>
                    </td>
                    <td class="project-cell">
                        <span class="project-name">${jobCard.project || 'N/A'}</span>
                    </td>
                    <td class="date-cell">${formattedStartDate}</td>
                    <td class="date-cell">${formattedDueDate}</td>
                    <td class="assigned-cell">
                        <span class="assigned-user">${jobCard.assignedTo || 'Not Assigned'}</span>
                    </td>
                    <td class="date-cell">${formattedCreatedDate}</td>
                </tr>
            `;
        }).join('');

        tableBody.innerHTML = rows;
    }

    function filterJobCards(searchTerm) {
        if (!searchTerm.trim()) {
            filteredData = [...jobCardData];
        } else {
            const term = searchTerm.toLowerCase();
            filteredData = jobCardData.filter(jobCard =>
                (jobCard.jobCardNumber && jobCard.jobCardNumber.toLowerCase().includes(term)) ||
                (jobCard.title && jobCard.title.toLowerCase().includes(term)) ||
                (jobCard.priority && jobCard.priority.toLowerCase().includes(term)) ||
                (jobCard.status && jobCard.status.toLowerCase().includes(term)) ||
                (jobCard.equipmentNumber && jobCard.equipmentNumber.toLowerCase().includes(term)) ||
                (jobCard.equipmentName && jobCard.equipmentName.toLowerCase().includes(term)) ||
                (jobCard.project && jobCard.project.toLowerCase().includes(term)) ||
                (jobCard.assignedTo && jobCard.assignedTo.toLowerCase().includes(term))
            );
        }

        displayJobCards(filteredData);
    }

    function applySorting() {
        const sortOrder = document.querySelector('input[name="sortOrder"]:checked').value;
        const sortBy = document.querySelector('input[name="sortBy"]:checked').value;

        filteredData.sort((a, b) => {
            let aValue, bValue;

            // Handle serial number sorting (maintain current order)
            if (sortBy === 'serialNumber') {
                const aIndex = jobCardData.indexOf(a);
                const bIndex = jobCardData.indexOf(b);
                aValue = aIndex;
                bValue = bIndex;
            } else {
                aValue = a[sortBy];
                bValue = b[sortBy];
            }

            // Handle date sorting
            if (sortBy === 'startDate' || sortBy === 'dueDate' || sortBy === 'createdAt') {
                aValue = a[sortBy] ? new Date(a[sortBy]) : new Date(0);
                bValue = b[sortBy] ? new Date(b[sortBy]) : new Date(0);
            }

            // Handle string sorting
            if (typeof aValue === 'string' && typeof bValue === 'string') {
                const comparison = aValue.localeCompare(bValue);
                return sortOrder === 'ascending' ? comparison : -comparison;
            }

            // Handle date sorting
            if (aValue instanceof Date && bValue instanceof Date) {
                return sortOrder === 'ascending' ? aValue - bValue : bValue - aValue;
            }

            return 0;
        });

        displayJobCards(filteredData);

        const modal = bootstrap.Modal.getInstance(document.getElementById('sortingModal'));
        modal.hide();

        showNotification('Sorting applied successfully!', 'success');
    }

    function resetSorting() {
        document.getElementById('sortingForm').reset();
        document.getElementById('ascending').checked = true;
        document.getElementById('serialNumber').checked = true;

        filteredData = [...jobCardData];
        displayJobCards(filteredData);

        showNotification('Sorting options reset!', 'info');
    }

    function refreshData() {
        loadJobCards();
        showNotification('Data refreshed successfully!', 'success');
    }

    // Pagination functions
    function goToPage(page) {
        if (page < 1 || page > paginationState.totalPages) {
            return;
        }

        paginationState.currentPage = page;
        loadJobCards();
    }

    function updatePaginationInfo() {
        const paginationSection = document.getElementById('paginationSection');
        const paginationInfo = document.getElementById('paginationInfo');
        const paginationPages = document.getElementById('paginationPages');

        if (paginationState.total === 0) {
            paginationSection.style.display = 'none';
            return;
        }

        paginationSection.style.display = 'block';

        // Update pagination info text
        const from = ((paginationState.currentPage - 1) * paginationState.perPage) + 1;
        const to = Math.min(paginationState.currentPage * paginationState.perPage, paginationState.total);
        paginationInfo.textContent = `Showing ${from}-${to} of ${paginationState.total} entries`;

        // Update page buttons
        const firstBtn = paginationSection.querySelector('.pagination-first');
        const prevBtn = paginationSection.querySelector('.pagination-prev');
        const nextBtn = paginationSection.querySelector('.pagination-next');
        const lastBtn = paginationSection.querySelector('.pagination-last');

        firstBtn.disabled = paginationState.currentPage === 1;
        prevBtn.disabled = paginationState.currentPage === 1;
        nextBtn.disabled = paginationState.currentPage === paginationState.totalPages;
        lastBtn.disabled = paginationState.currentPage === paginationState.totalPages;

        // Update page numbers display
        paginationPages.innerHTML = generatePageNumbers();
    }

    function generatePageNumbers() {
        const { currentPage, totalPages } = paginationState;
        const pages = [];

        if (totalPages <= 7) {
            // Show all pages if 7 or fewer
            for (let i = 1; i <= totalPages; i++) {
                pages.push(createPageButton(i));
            }
        } else {
            // Always show first page
            pages.push(createPageButton(1));

            if (currentPage > 4) {
                pages.push('<span class="pagination-ellipsis">...</span>');
            }

            // Show current page and surrounding pages
            const start = Math.max(2, currentPage - 1);
            const end = Math.min(totalPages - 1, currentPage + 1);

            for (let i = start; i <= end; i++) {
                pages.push(createPageButton(i));
            }

            if (currentPage < totalPages - 3) {
                pages.push('<span class="pagination-ellipsis">...</span>');
            }

            // Always show last page
            if (totalPages > 1) {
                pages.push(createPageButton(totalPages));
            }
        }

        return pages.join('');
    }

    function createPageButton(page) {
        const isActive = page === paginationState.currentPage;
        return `<button class="pagination-page ${isActive ? 'active' : ''}" onclick="goToPage(${page})">${page}</button>`;
    }

    function exportToExcel() {
        if (filteredData.length === 0) {
            showNotification('No data to export', 'warning');
            return;
        }

        // Create CSV content
        const headers = ['S.No', 'Job Card #', 'Title', 'Priority', 'Status', 'Equipment', 'Project', 'Start Date', 'Due Date', 'Assigned To', 'Created Date'];
        const csvContent = [
            headers.join(','),
            ...filteredData.map((jobCard, index) => {
                const serialNumber = ((paginationState.currentPage - 1) * paginationState.perPage) + index + 1;
                const startDate = jobCard.startDate ? new Date(jobCard.startDate).toLocaleDateString() : '';
                const dueDate = jobCard.dueDate ? new Date(jobCard.dueDate).toLocaleDateString() : '';
                const createdDate = jobCard.createdAt ? new Date(jobCard.createdAt).toLocaleDateString() : '';

                return [
                    serialNumber,
                    jobCard.jobCardNumber || '',
                    jobCard.title || '',
                    jobCard.priority || '',
                    jobCard.status || '',
                    `${jobCard.equipmentNumber || ''} ${jobCard.equipmentName || ''}`.trim(),
                    jobCard.project || '',
                    startDate,
                    dueDate,
                    jobCard.assignedTo || '',
                    createdDate
                ].join(',');
            })
        ].join('\n');

        // Download CSV file
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `job_cards_${new Date().toISOString().split('T')[0]}.csv`;
        a.click();
        window.URL.revokeObjectURL(url);

        showNotification('Excel export completed!', 'success');
    }

    // Notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show position-fixed`;
        notification.style.cssText = `top: 20px; right: 20px; z-index: ${CONFIG.timing.notificationTimeout}; min-width: 300px;`;
        notification.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;

        document.body.appendChild(notification);

        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, CONFIG.timing.notificationTimeout);
    }

    // Make functions globally accessible
    window.loadJobCards = loadJobCards;
    window.filterJobCards = filterJobCards;
    window.applySorting = applySorting;
    window.resetSorting = resetSorting;
    window.refreshData = refreshData;
    window.exportToExcel = exportToExcel;
    window.goToPage = goToPage;
    window.updatePaginationInfo = updatePaginationInfo;

})();
</script>
@endsection
