@extends('welcome')

@section('title', 'Create Service Schedule')

<meta name="csrf-token" content="{{ csrf_token() }}">

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.schedule-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Form Groups */
.form-group {
    margin-bottom: 25px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-group-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 8px;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.form-label-small {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.required {
    color: var(--danger-color);
    font-weight: bold;
}

.form-input, .form-select, .form-textarea {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-input:focus, .form-select:focus, .form-textarea:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-input.error {
    border-color: var(--danger-color);
}

.form-textarea {
    min-height: 100px;
    resize: vertical;
}

.error-message {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 5px;
    display: none;
}

.error-message.show {
    display: block;
}

/* Schedule ID Display */
.schedule-id-display {
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
}

.schedule-id-label {
    font-weight: 500;
    color: var(--text-primary);
}

.schedule-id-value {
    font-weight: 700;
    color: var(--primary-color);
    font-size: 16px;
}

/* Input Groups */
.input-group {
    display: flex;
    gap: 12px;
    align-items: center;
}

.unit-select {
    min-width: 120px;
    flex-shrink: 0;
}

/* Selected Equipment */
.selected-equipment {
    border: 2px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 16px;
    background: var(--bg-secondary);
    min-height: 60px;
    display: flex;
    align-items: center;
}

.no-equipment {
    color: var(--text-secondary);
    font-style: italic;
}

.equipment-details h4 {
    margin: 0 0 8px 0;
    color: var(--primary-color);
    font-size: 16px;
    font-weight: 600;
}

.equipment-details p {
    margin: 0 0 4px 0;
    color: var(--text-secondary);
    font-size: 14px;
}

/* Buttons */
.select-button, .add-task-button {
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    padding: 8px 16px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.select-button:hover, .add-task-button:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
}

/* Tasks List */
.tasks-list {
    border: 2px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 16px;
    background: var(--bg-secondary);
    min-height: 60px;
}

.no-tasks {
    color: var(--text-secondary);
    font-style: italic;
}

.task-item {
    color: var(--text-primary);
    font-size: 14px;
    margin-bottom: 8px;
    padding: 8px;
    background: white;
    border-radius: var(--radius-sm);
    border: 1px solid var(--border-light);
}

/* Repeat Type Buttons */
.repeat-type-buttons {
    display: flex;
    gap: 12px;
    margin-bottom: 16px;
}

.repeat-button {
    padding: 10px 20px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.repeat-button.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.repeat-button:hover:not(.active) {
    border-color: var(--primary-color);
    color: var(--primary-color);
}

/* Intervals */
.intervals-container {
    margin-bottom: 16px;
}

.interval-item {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 12px;
    padding: 12px;
    background: var(--bg-secondary);
    border-radius: var(--radius-md);
    border: 1px solid var(--border-light);
}

.interval-checkbox {
    width: 18px;
    height: 18px;
    accent-color: var(--primary-color);
}

.interval-label {
    font-size: 14px;
    color: var(--text-primary);
    font-weight: 500;
    white-space: nowrap;
}

.interval-input {
    width: 80px;
    padding: 8px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    text-align: center;
}

.interval-unit {
    padding: 8px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    background: white;
    min-width: 100px;
}

/* Time Unit Buttons */
.time-unit-buttons {
    display: flex;
    gap: 8px;
    margin-bottom: 16px;
}

.time-unit-button {
    padding: 8px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: white;
    color: var(--text-secondary);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.time-unit-button.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.time-unit-button:hover:not(.active) {
    border-color: var(--primary-color);
    color: var(--primary-color);
}

/* Info Text */
.info-text {
    color: var(--text-secondary);
    font-size: 13px;
    margin-top: 8px;
    font-style: italic;
}

/* Body scroll lock when modal is open */
body.modal-open {
    overflow: hidden;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    overflow-y: auto;
}

.modal-content {
    background-color: white;
    margin: 0;
    padding: 0;
    border-radius: var(--radius-lg);
    width: 90%;
    max-width: 600px;
    box-shadow: var(--shadow-md);
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    animation: modalSlideIn 0.3s ease;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateX(-50%) scale(0.9);
    }
    to {
        opacity: 1;
        transform: translateX(-50%) scale(1);
    }
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 24px;
    border-bottom: 1px solid var(--border-light);
    background-color: var(--primary-color);
    border-radius: var(--radius-lg) var(--radius-lg) 0 0;
}

.modal-header h3 {
    margin: 0;
    color: white;
    font-size: 20px;
    font-weight: 600;
}

.close-button {
    background: none;
    border: none;
    font-size: 24px;
    color: white;
    cursor: pointer;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: var(--transition-base);
}

.close-button:hover {
    background: rgba(255, 255, 255, 0.1);
}

.modal-body {
    padding: 24px;
}

.modal-footer {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    padding: 20px 24px;
    border-top: 1px solid var(--border-light);
}

/* Equipment List */
.search-container {
    margin-bottom: 16px;
}

.search-input {
    width: 100%;
    padding: 12px 16px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 16px;
}

.equipment-list {
    max-height: 300px;
    overflow-y: auto;
}

.equipment-item {
    padding: 16px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    margin-bottom: 8px;
    cursor: pointer;
    transition: var(--transition-base);
    background: white;
}

.equipment-item:hover {
    background: var(--bg-secondary);
    border-color: var(--primary-color);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.equipment-info h4 {
    margin: 0 0 8px 0;
    color: var(--primary-color);
    font-size: 16px;
    font-weight: 600;
}

.equipment-info p {
    margin: 0 0 4px 0;
    color: var(--text-secondary);
    font-size: 14px;
}

/* Modal Buttons */
.btn-cancel, .btn-confirm {
    padding: 10px 20px;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-cancel {
    background: var(--bg-secondary);
    color: var(--text-primary);
}

.btn-cancel:hover {
    background: var(--border-color);
}

.btn-confirm {
    background: var(--primary-color);
    color: white;
}

.btn-confirm:hover {
    background: var(--primary-hover);
}

/* Loading Modal */
.loading-content {
    text-align: center;
    padding: 40px;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid var(--bg-secondary);
    border-top: 4px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 16px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .input-group {
        flex-direction: column;
        align-items: stretch;
    }
    
    .unit-select {
        min-width: auto;
    }
    
    .repeat-type-buttons {
        flex-direction: column;
    }
    
    .time-unit-buttons {
        flex-wrap: wrap;
    }
    
    .interval-item {
        flex-direction: column;
        align-items: stretch;
        gap: 8px;
    }
    
    .interval-input, .interval-unit {
        width: 100%;
    }
    
    .modal-content {
        width: 95%;
        margin: 10% auto;
    }
}

@media (max-width: 480px) {
    .header-title {
        font-size: 20px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .form-section {
        padding: 16px;
    }
}
</style>
@endsection

@section('content')
<div class="schedule-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('maintenance.service_schedule') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Add Schedule</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="handleCreateSchedule()">
                    <i class="fas fa-save"></i>
                    Create
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="form-container">
            <!-- Schedule ID Section -->
            <div class="form-section">
                <div class="schedule-id-display">
                    <span class="schedule-id-label">Schedule ID</span>
                    <span class="schedule-id-value" id="scheduleIdDisplay">Loading...</span>
                </div>
            </div>

            <!-- Basic Information Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-info-circle"></i>
                    Basic Information
                </h2>

                <form id="scheduleForm" class="schedule-form">
                    @csrf
                    
                    <div class="form-group">
                        <label class="form-label">
                            Country
                            <span class="required">*</span>
                        </label>
                        <select class="form-select" id="countrySelect" name="country" required>
                            <option value="">Select Country</option>
                            <option value="Zambia">Zambia</option>
                            <option value="DRC">DRC</option>
                        </select>
                        <div class="error-message" id="countryError"></div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            Project
                            <span class="required">*</span>
                        </label>
                        <select class="form-select" id="projectSelect" name="project" required disabled>
                            <option value="">Select Project</option>
                        </select>
                        <div class="error-message" id="projectError"></div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            Schedule Title
                            <span class="required">*</span>
                        </label>
                        <input type="text" class="form-input" id="scheduleTitle" name="scheduleTitle" placeholder="Enter schedule title" required>
                        <div class="error-message" id="scheduleTitleError"></div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            Note
                            <span class="required">*</span>
                        </label>
                        <textarea class="form-textarea" id="note" name="note" placeholder="Enter note" required></textarea>
                        <div class="error-message" id="noteError"></div>
                    </div>
                </form>
            </div>

            <!-- Equipment Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-tools"></i>
                    Equipment Selection
                </h2>

                <div class="form-group">
                    <div class="form-group-header">
                        <label class="form-label">
                            Equipment
                            <span class="required">*</span>
                        </label>
                        <button type="button" class="select-button" onclick="openEquipmentModal(event)">Select</button>
                    </div>
                    <div id="selectedEquipment" class="selected-equipment">
                        <div class="no-equipment">No equipment selected</div>
                    </div>
                    <div class="error-message" id="equipmentError"></div>
                </div>
            </div>

            <!-- Service Details Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-cog"></i>
                    Service Details
                </h2>

                <div class="form-group">
                    <label class="form-label">
                        Last Service Meter Reading
                        <span class="required">*</span>
                    </label>
                    <div class="input-group">
                        <input type="number" class="form-input" id="meterReading" name="meterReading" placeholder="0" step="0.01" required>
                        <select class="form-select unit-select" id="meterUnit" name="meterUnit">
                            <option value="Miles">Miles</option>
                            <option value="Kilometers">Kilometers</option>
                            <option value="Hours">Hours</option>
                        </select>
                    </div>
                    <div class="error-message" id="meterReadingError"></div>
                </div>

                <div class="form-group">
                    <label class="form-label">
                        Last Service Date
                        <span class="required">*</span>
                    </label>
                    <div class="input-group">
                        <input type="date" class="form-input" id="lastServiceDate" name="lastServiceDate" required>
                    </div>
                    <div class="error-message" id="lastServiceDateError"></div>
                </div>
            </div>

            <!-- Tasks Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-tasks"></i>
                    Tasks
                </h2>

                <div class="form-group">
                    <div class="form-group-header">
                        <label class="form-label">
                            Tasks
                            <span class="required">*</span>
                        </label>
                        <button type="button" class="add-task-button" onclick="openTaskModal(event)">+ Add Task</button>
                    </div>
                    <div id="tasksList" class="tasks-list">
                        <div class="no-tasks">No tasks added</div>
                    </div>
                    <div class="error-message" id="tasksError"></div>
                </div>
            </div>

            <!-- Schedule Configuration Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-calendar-alt"></i>
                    Schedule Configuration
                </h2>

                <div class="form-group">
                    <label class="form-label">
                        Repeat Type
                        <span class="required">*</span>
                    </label>
                    <div class="repeat-type-buttons">
                        <button type="button" class="repeat-button active" data-type="Repeat">Repeat</button>
                        <button type="button" class="repeat-button" data-type="One time">One time</button>
                    </div>
                    <div class="error-message" id="repeatTypeError"></div>
                </div>

                <div class="form-group" id="intervalsGroup">
                    <label class="form-label-small">Intervals (whichever comes first)</label>
                    <div class="intervals-container">
                        <div class="interval-item">
                            <input type="checkbox" class="interval-checkbox" checked>
                            <span class="interval-label">Every</span>
                            <input type="number" class="interval-input" placeholder="0" min="0">
                            <select class="interval-unit">
                                <option value="Miles">Miles</option>
                                <option value="Kilometers">Kilometers</option>
                                <option value="Hours">Hours</option>
                            </select>
                        </div>
                        <div class="interval-item">
                            <input type="checkbox" class="interval-checkbox">
                            <span class="interval-label">Every</span>
                            <input type="number" class="interval-input" placeholder="0" min="0" disabled>
                            <select class="interval-unit" disabled>
                                <option value="Miles">Miles</option>
                                <option value="Kilometers">Kilometers</option>
                                <option value="Hours">Hours</option>
                            </select>
                        </div>
                    </div>
                    <div class="time-unit-buttons">
                        <button type="button" class="time-unit-button active" data-unit="Day">Day</button>
                        <button type="button" class="time-unit-button" data-unit="Week">Week</button>
                        <button type="button" class="time-unit-button" data-unit="Month">Month</button>
                        <button type="button" class="time-unit-button" data-unit="Year">Year</button>
                    </div>
                    <div class="error-message" id="intervalsError"></div>
                </div>

                <div class="form-group">
                    <label class="form-label">Notify Before</label>
                    <div class="input-group">
                        <input type="number" class="form-input" id="notifyBeforeValue" name="notifyBeforeValue" placeholder="0" min="0">
                        <select class="form-select unit-select" id="notifyBeforeUnit" name="notifyBeforeUnit">
                            <option value="Miles">Miles</option>
                            <option value="Kilometers">Kilometers</option>
                            <option value="Hours">Hours</option>
                        </select>
                    </div>
                    <div class="time-unit-buttons" id="notifyBeforeGroup">
                        <button type="button" class="time-unit-button active" data-unit="Day">Day</button>
                        <button type="button" class="time-unit-button" data-unit="Week">Week</button>
                        <button type="button" class="time-unit-button" data-unit="Month">Month</button>
                    </div>
                    <p class="info-text">The schedule will be marked as 'Due soon' once any of the above threshold is crossed.</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-button" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" placeholder="Search equipment..." class="search-input">
            </div>
            <div id="equipmentList" class="equipment-list">
                <!-- Equipment items will be loaded here -->
            </div>
        </div>
    </div>
</div>

<!-- Task Modal -->
<div id="taskModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Add New Task</h3>
            <button type="button" class="close-button" onclick="closeTaskModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div class="form-group">
                <label class="form-label">Task Name</label>
                <input type="text" id="newTaskInput" class="form-input" placeholder="Enter task name">
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn-cancel" onclick="closeTaskModal()">Cancel</button>
            <button type="button" class="btn-confirm" onclick="addTask()">Done</button>
        </div>
    </div>
</div>

<!-- Loading Modal -->
<div id="loadingModal" class="modal">
    <div class="modal-content loading-content">
        <div class="loading-spinner"></div>
        <p>Creating Schedule...</p>
    </div>
</div>

<script>
// Global variables
let selectedEquipment = null;
let tasks = [];
let repeatType = 'Repeat';
let intervalUnit = 'Day';
let notifyBeforeTime = 'Day';

// Project mapping
const projectMap = {
    'Zambia': [
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' }
    ],
    'DRC': [
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' }
    ]
};

// Initialize the form
document.addEventListener('DOMContentLoaded', function() {
    initializeForm();
    setupEventListeners();
    generateScheduleIdPreview();
});

function initializeForm() {
    // Set default date to today
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('lastServiceDate').value = today;
}

function setupEventListeners() {
    // Country change handler
    document.getElementById('countrySelect').addEventListener('change', function() {
        updateProjectOptions(this.value);
        generateScheduleIdPreview();
    });

    // Project change handler
    document.getElementById('projectSelect').addEventListener('change', function() {
        generateScheduleIdPreview();
    });

    // Equipment search handler
    document.getElementById('equipmentSearch').addEventListener('input', function() {
        filterEquipment(this.value);
    });

    // Repeat type buttons
    document.querySelectorAll('.repeat-button').forEach(button => {
        button.addEventListener('click', function() {
            document.querySelectorAll('.repeat-button').forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            repeatType = this.dataset.type;
            toggleIntervalsGroup();
        });
    });

    // Time unit buttons for intervals
    document.querySelectorAll('#intervalsGroup .time-unit-button').forEach(button => {
        button.addEventListener('click', function() {
            document.querySelectorAll('#intervalsGroup .time-unit-button').forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            intervalUnit = this.dataset.unit;
        });
    });

    // Time unit buttons for notify before
    document.querySelectorAll('#notifyBeforeGroup .time-unit-button').forEach(button => {
        button.addEventListener('click', function() {
            document.querySelectorAll('#notifyBeforeGroup .time-unit-button').forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            notifyBeforeTime = this.dataset.unit;
        });
    });

    // Interval checkboxes
    document.querySelectorAll('.interval-checkbox').forEach((checkbox, index) => {
        checkbox.addEventListener('change', function() {
            const intervalItem = this.closest('.interval-item');
            const inputs = intervalItem.querySelectorAll('input[type="number"], select');
            inputs.forEach(input => {
                input.disabled = !this.checked;
            });
        });
    });
}

function updateProjectOptions(country) {
    const projectSelect = document.getElementById('projectSelect');
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    
    if (country && projectMap[country]) {
        projectMap[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project.value;
            option.textContent = project.label;
            projectSelect.appendChild(option);
        });
        projectSelect.disabled = false;
    } else {
        projectSelect.disabled = true;
    }
}

function generateScheduleIdPreview() {
    const country = document.getElementById('countrySelect').value;
    const project = document.getElementById('projectSelect').value;
    
    if (country && project) {
        const countryCode = getCountryCode(country);
        const projectCode = getProjectCode(project);
        const timestamp = Date.now().toString().slice(-6);
        const scheduleId = `${countryCode}-${projectCode}-SS-${timestamp}`;
        document.getElementById('scheduleIdDisplay').textContent = scheduleId;
    } else {
        document.getElementById('scheduleIdDisplay').textContent = 'Select Country and Project';
    }
}

function getCountryCode(country) {
    const countryCodes = {
        'Zambia': 'ZMB',
        'DRC': 'DRC'
    };
    return countryCodes[country] || country.substring(0, 3).toUpperCase();
}

function getProjectCode(project) {
    if (!project) return 'XXX';
    return project.substring(0, 3).toUpperCase();
}

function toggleIntervalsGroup() {
    const intervalsGroup = document.getElementById('intervalsGroup');
    if (repeatType === 'Repeat') {
        intervalsGroup.style.display = 'block';
    } else {
        intervalsGroup.style.display = 'none';
    }
}

function openEquipmentModal(event) {
    const modal = document.getElementById('equipmentModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
    
    // Position modal content
    setTimeout(() => {
        const modalContent = modal.querySelector('.modal-content');
        if (modalContent && event) {
            const clickY = event.clientY;
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            
            let topPosition = scrollY + clickY - (contentHeight / 2);
            const minTop = scrollY + 20;
            const maxTop = scrollY + viewportHeight - contentHeight - 20;
            
            if (topPosition < minTop) topPosition = minTop;
            if (topPosition > maxTop) topPosition = maxTop;
            
            modalContent.style.top = `${topPosition}px`;
        } else if (modalContent) {
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            const topPosition = scrollY + (viewportHeight - contentHeight) / 2;
            modalContent.style.top = `${topPosition}px`;
        }
    }, 10);
    
    loadEquipmentData();
}

function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
}

function loadEquipmentData() {
    const equipmentList = document.getElementById('equipmentList');
    equipmentList.innerHTML = '<div class="loading-text">Loading equipment...</div>';
    
    // Fetch equipment data from Firebase
    fetch('/api/equipment')
        .then(response => response.json())
        .then(data => {
            console.log('Equipment API response:', data);
            if (data.success && data.equipment) {
                displayEquipment(data.equipment);
            } else if (data.equipment && Array.isArray(data.equipment)) {
                displayEquipment(data.equipment);
            } else {
                // Fallback to mock data if API fails
                const mockData = [
                    { 
                        id: '1', 
                        equipmentName: 'Excavator CAT 320', 
                        entryNumber: 'EQ001', 
                        equipmentNumber: 'CAT-320-001', 
                        meterReading: 1500, 
                        meterUnit: 'Hours',
                        currentMeterReading: 1500
                    },
                    { 
                        id: '2', 
                        equipmentName: 'Bulldozer Komatsu D65', 
                        entryNumber: 'EQ002', 
                        equipmentNumber: 'KOM-D65-002', 
                        meterReading: 2500, 
                        meterUnit: 'Hours',
                        currentMeterReading: 2500
                    },
                    { 
                        id: '3', 
                        equipmentName: 'Dump Truck Volvo A25', 
                        entryNumber: 'EQ003', 
                        equipmentNumber: 'VOL-A25-003', 
                        meterReading: 50000, 
                        meterUnit: 'Miles',
                        currentMeterReading: 50000
                    }
                ];
                displayEquipment(mockData);
            }
        })
        .catch(error => {
            console.error('Error loading equipment:', error);
            // Fallback to mock data
            const mockData = [
                { 
                    id: '1', 
                    equipmentName: 'Excavator CAT 320', 
                    entryNumber: 'EQ001', 
                    equipmentNumber: 'CAT-320-001', 
                    meterReading: 1500, 
                    meterUnit: 'Hours',
                    currentMeterReading: 1500
                },
                { 
                    id: '2', 
                    equipmentName: 'Bulldozer Komatsu D65', 
                    entryNumber: 'EQ002', 
                    equipmentNumber: 'KOM-D65-002', 
                    meterReading: 2500, 
                    meterUnit: 'Hours',
                    currentMeterReading: 2500
                },
                { 
                    id: '3', 
                    equipmentName: 'Dump Truck Volvo A25', 
                    entryNumber: 'EQ003', 
                    equipmentNumber: 'VOL-A25-003', 
                    meterReading: 50000, 
                    meterUnit: 'Miles',
                    currentMeterReading: 50000
                }
            ];
            displayEquipment(mockData);
        });
}

function displayEquipment(equipmentData) {
    const equipmentList = document.getElementById('equipmentList');
    equipmentList.innerHTML = '';
    
    if (!equipmentData || equipmentData.length === 0) {
        equipmentList.innerHTML = '<div class="no-equipment">No equipment available</div>';
        return;
    }
    
    // Store equipment data globally for filtering
    window.allEquipmentData = equipmentData;
    
    equipmentData.forEach(equipment => {
        const equipmentItem = document.createElement('div');
        equipmentItem.className = 'equipment-item';
        
        // Get meter reading from various possible fields
        const meterReading = getMeterReading(equipment);
        const meterUnit = getMeterUnit(equipment);
        
        // Debug: Log the equipment data to see what fields are available
        console.log('Equipment data for display:', equipment);
        
        equipmentItem.innerHTML = `
            <div class="equipment-info">
                <h4>${equipment.equipmentName || equipment.name || 'Unknown Equipment'}</h4>
                <p>Entry Number: ${equipment.entryNumber || 'N/A'}</p>
                <p>Equipment Number: ${equipment.equipmentNumber || equipment.registrationNumber || 'N/A'}</p>
                <p>Meter: ${meterReading} ${meterUnit}</p>
                <p style="font-size: 12px; color: #666;">Debug: ${JSON.stringify(equipment).substring(0, 100)}...</p>
            </div>
        `;
        equipmentItem.addEventListener('click', () => selectEquipment(equipment));
        equipmentList.appendChild(equipmentItem);
    });
}

function getMeterReading(equipment) {
    // Try multiple possible field names for meter reading
    const possibleFields = [
        'meterReading', 'currentMeter', 'odometerReading', 'currentOdometer', 
        'odometer', 'currentMeterReading', 'meter', 'currentReading',
        'totalMeterReading', 'totalOdometer', 'currentMileage', 'mileage',
        'hours', 'totalHours', 'engineHours', 'operatingHours'
    ];
    
    for (const field of possibleFields) {
        if (equipment[field] !== undefined && equipment[field] !== null && equipment[field] !== '') {
            const value = parseFloat(equipment[field]);
            if (!isNaN(value) && value >= 0) {
                console.log(`Found meter reading in field '${field}':`, value);
                return value;
            }
        }
    }
    
    console.log('No valid meter reading found, using 0. Equipment data:', equipment);
    return 0;
}

function getMeterUnit(equipment) {
    // Try multiple possible field names for meter unit
    const possibleFields = [
        'meterUnit', 'unit', 'odometerUnit', 'currentUnit', 
        'meterUnitType', 'unitType', 'measurementUnit', 'unitOfMeasure'
    ];
    
    for (const field of possibleFields) {
        if (equipment[field] !== undefined && equipment[field] !== null && equipment[field] !== '') {
            console.log(`Found meter unit in field '${field}':`, equipment[field]);
            return equipment[field];
        }
    }
    
    // Default to Hours if no unit found
    console.log('No meter unit found, using Hours. Equipment data:', equipment);
    return 'Hours';
}

function filterEquipment(searchTerm) {
    if (!window.allEquipmentData) return;
    
    const filteredData = window.allEquipmentData.filter(equipment => {
        const name = (equipment.equipmentName || equipment.name || '').toLowerCase();
        const entryNumber = (equipment.entryNumber || '').toLowerCase();
        const equipmentNumber = (equipment.equipmentNumber || equipment.registrationNumber || '').toLowerCase();
        const search = searchTerm.toLowerCase();
        
        return name.includes(search) || entryNumber.includes(search) || equipmentNumber.includes(search);
    });
    
    displayEquipment(filteredData);
}

function selectEquipment(equipment) {
    selectedEquipment = equipment;
    
    // Get meter reading and unit using helper functions
    const meterReading = getMeterReading(equipment);
    const meterUnit = getMeterUnit(equipment);
    
    console.log('Equipment selected:', {
        equipment: equipment,
        meterReading: meterReading,
        meterUnit: meterUnit
    });
    
    const selectedEquipmentDiv = document.getElementById('selectedEquipment');
    selectedEquipmentDiv.innerHTML = `
        <div class="equipment-details">
            <h4>${equipment.equipmentName || equipment.name || 'Unknown Equipment'}</h4>
            <p>Entry Number: ${equipment.entryNumber || 'N/A'}</p>
            <p>Equipment Number: ${equipment.equipmentNumber || equipment.registrationNumber || 'N/A'}</p>
            <p>Meter: ${meterReading} ${meterUnit}</p>
        </div>
    `;
    
    // Update meter reading and unit in the form
    document.getElementById('meterReading').value = meterReading;
    document.getElementById('meterUnit').value = meterUnit;
    
    // Update notify before unit to match equipment unit
    document.getElementById('notifyBeforeUnit').value = meterUnit;
    
    // Update all interval units to match equipment unit
    document.querySelectorAll('.interval-unit').forEach(select => {
        select.value = meterUnit;
    });
    
    closeEquipmentModal();
}

function openTaskModal(event) {
    const modal = document.getElementById('taskModal');
    modal.style.display = 'block';
    document.body.classList.add('modal-open');
    
    // Position modal content
    setTimeout(() => {
        const modalContent = modal.querySelector('.modal-content');
        if (modalContent && event) {
            const clickY = event.clientY;
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            
            let topPosition = scrollY + clickY - (contentHeight / 2);
            const minTop = scrollY + 20;
            const maxTop = scrollY + viewportHeight - contentHeight - 20;
            
            if (topPosition < minTop) topPosition = minTop;
            if (topPosition > maxTop) topPosition = maxTop;
            
            modalContent.style.top = `${topPosition}px`;
        } else if (modalContent) {
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            const topPosition = scrollY + (viewportHeight - contentHeight) / 2;
            modalContent.style.top = `${topPosition}px`;
        }
        
        document.getElementById('newTaskInput').value = '';
    }, 10);
}

function closeTaskModal() {
    const modal = document.getElementById('taskModal');
    modal.style.display = 'none';
    document.body.classList.remove('modal-open');
}

function addTask() {
    const taskInput = document.getElementById('newTaskInput');
    const taskText = taskInput.value.trim();
    
    if (taskText) {
        tasks.push(taskText);
        updateTasksList();
        closeTaskModal();
    }
}

function updateTasksList() {
    const tasksList = document.getElementById('tasksList');
    
    if (tasks.length === 0) {
        tasksList.innerHTML = '<div class="no-tasks">No tasks added</div>';
    } else {
        tasksList.innerHTML = tasks.map(task => `<div class="task-item">- ${task}</div>`).join('');
    }
}

function removeTask(index) {
    tasks.splice(index, 1);
    updateTasksList();
}

function validateForm() {
    let isValid = true;
    const errors = {};
    
    // Clear previous errors
    document.querySelectorAll('.error-message').forEach(el => el.textContent = '');
    
    // Validate required fields
    const requiredFields = [
        'countrySelect', 'projectSelect', 'scheduleTitle', 'note', 
        'meterReading', 'lastServiceDate'
    ];
    
    requiredFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (!field.value.trim()) {
            errors[fieldId] = `${fieldId.replace('Select', '').replace(/([A-Z])/g, ' $1').trim()} is required`;
            isValid = false;
        }
    });
    
    // Validate equipment selection
    if (!selectedEquipment) {
        errors.equipment = 'Equipment is required';
        isValid = false;
    }
    
    // Validate tasks
    if (tasks.length === 0) {
        errors.tasks = 'At least one task is required';
        isValid = false;
    }
    
    // Validate intervals for repeat type
    if (repeatType === 'Repeat') {
        const checkedIntervals = document.querySelectorAll('.interval-checkbox:checked');
        const hasValidInterval = Array.from(checkedIntervals).some(checkbox => {
            const intervalItem = checkbox.closest('.interval-item');
            const valueInput = intervalItem.querySelector('.interval-input');
            return valueInput.value && parseFloat(valueInput.value) > 0;
        });
        
        if (!hasValidInterval) {
            errors.intervals = 'At least one valid interval is required for repeating schedules';
            isValid = false;
        }
    }
    
    // Display errors
    Object.keys(errors).forEach(fieldId => {
        const errorElement = document.getElementById(fieldId + 'Error');
        if (errorElement) {
            errorElement.textContent = errors[fieldId];
        }
    });
    
    return isValid;
}

function handleCreateSchedule() {
    if (!validateForm()) {
        showNotification('Please fill all required fields', 'error');
        return;
    }
    
    // Show loading modal
    document.getElementById('loadingModal').style.display = 'block';
    
    // Get meter reading values - prioritize form input over equipment data
    const formMeterReading = parseFloat(document.getElementById('meterReading').value) || 0;
    const formMeterUnit = document.getElementById('meterUnit').value || 'Miles';
    
    // Get equipment meter reading using helper function
    const equipmentMeterReading = selectedEquipment ? getMeterReading(selectedEquipment) : 0;
    const equipmentMeterUnit = selectedEquipment ? getMeterUnit(selectedEquipment) : 'Miles';
    
    // Use form values if available and greater than 0, otherwise use equipment values
    const finalMeterReading = formMeterReading > 0 ? formMeterReading : equipmentMeterReading;
    const finalMeterUnit = formMeterUnit || equipmentMeterUnit;
    
    console.log('Meter reading calculation:', {
        formMeterReading: formMeterReading,
        formMeterUnit: formMeterUnit,
        equipmentMeterReading: equipmentMeterReading,
        equipmentMeterUnit: equipmentMeterUnit,
        finalMeterReading: finalMeterReading,
        finalMeterUnit: finalMeterUnit
    });
    
    // Prepare schedule data (scheduleId will be generated by server)
    const scheduleData = {
        scheduleId: document.getElementById('scheduleIdDisplay').textContent, // This will be overridden by server
        country: document.getElementById('countrySelect').value,
        project: document.getElementById('projectSelect').value,
        title: document.getElementById('scheduleTitle').value,
        note: document.getElementById('note').value,
        equipment: {
            entryNumber: selectedEquipment?.entryNumber || '',
            equipmentName: selectedEquipment?.equipmentName || selectedEquipment?.name || 'TYRE - TYRE',
            equipmentNumber: selectedEquipment?.equipmentNumber || selectedEquipment?.registrationNumber || '',
            meterReading: finalMeterReading,
            meterUnit: finalMeterUnit
        },
        meterReading: finalMeterReading,
        meterUnit: finalMeterUnit,
        lastServiceDate: document.getElementById('lastServiceDate').value,
        tasks: tasks,
        repeatType: repeatType,
        intervals: getIntervalsData(),
        intervalUnit: intervalUnit,
        notifyBefore: {
            value: document.getElementById('notifyBeforeValue').value || '',
            unit: document.getElementById('notifyBeforeUnit').value,
            time: notifyBeforeTime
        },
        status: 'upcoming',
        createdAt: new Date().toISOString()
    };
    
    console.log('Sending schedule data:', scheduleData);
    
    // Submit to server
    fetch('{{ route("maintenance.store_schedule") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({ data: scheduleData })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        document.getElementById('loadingModal').style.display = 'none';
        
        if (data.success) {
            const scheduleId = data.data?.scheduleId || 'Unknown';
            showNotification(`Schedule created successfully! ID: ${scheduleId}`, 'success');
            setTimeout(() => {
                window.location.href = '{{ route("maintenance.service_schedule") }}';
            }, 1500);
        } else {
            let errorMessage = data.message || 'Failed to create schedule';
            if (data.errors) {
                errorMessage += ': ' + Object.values(data.errors).flat().join(', ');
            }
            showNotification(errorMessage, 'error');
        }
    })
    .catch(error => {
        document.getElementById('loadingModal').style.display = 'none';
        console.error('Error:', error);
        showNotification('An error occurred while creating the schedule: ' + error.message, 'error');
    });
}

function getIntervalsData() {
    const intervals = [];
    document.querySelectorAll('.interval-item').forEach((item, index) => {
        const checkbox = item.querySelector('.interval-checkbox');
        const valueInput = item.querySelector('.interval-input');
        const unitSelect = item.querySelector('.interval-unit');
        
        if (checkbox.checked && valueInput.value && parseFloat(valueInput.value) > 0) {
            intervals.push({
                checked: true,
                value: valueInput.value,
                unit: unitSelect.value
            });
        }
    });
    return intervals;
}

function showNotification(message, type) {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    // Style the notification
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 12px 20px;
        border-radius: 4px;
        color: white;
        font-weight: 500;
        z-index: 10000;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
    `;
    
    if (type === 'success') {
        notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
    } else if (type === 'error') {
        notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
    } else {
        notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
    }
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                document.body.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Close modals when clicking outside
window.addEventListener('click', function(event) {
    const equipmentModal = document.getElementById('equipmentModal');
    const taskModal = document.getElementById('taskModal');
    
    if (event.target === equipmentModal) {
        closeEquipmentModal();
    }
    if (event.target === taskModal) {
        closeTaskModal();
    }
});
</script>

@endsection
