import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  Modal,
  Animated,
  Alert,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';

const HrModuleKpI = () => {
  const navigation = useNavigation();
  const deviceWidth = Dimensions.get('window').width;
  const [loading, setLoading] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [activeTab, setActiveTab] = useState('applications'); // 'applications' or 'approvals'
  const [fromDate, setFromDate] = useState(() => {
    const date = new Date();
    date.setDate(date.getDate() - 7); // 7 days ago
    return date;
  });
  const [toDate, setToDate] = useState(new Date());
  const [showFromDatePicker, setShowFromDatePicker] = useState(false);
  const [showToDatePicker, setShowToDatePicker] = useState(false);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    hrPermissions: [],
  });
  const [kpiData, setKpiData] = useState({
    totalApplications: 0,
    pendingApplications: 0,
    approvedApplications: 0,
    rejectedApplications: 0,
    averageProcessingTime: 0,
    applicationsChange: 0,
    pendingChange: 0,
    approvedChange: 0,
    rejectedChange: 0,
  });
  const [recentActivity, setRecentActivity] = useState([]);
  const [applicationTypeData, setApplicationTypeData] = useState([]);
  const [approvalWorkflowData, setApprovalWorkflowData] = useState([]);
  const isMounted = useRef(true);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notifications
  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Fetch user data and permissions
  const fetchUserData = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const userSnapshot = await userRef.once('value');
      const userDataFromDB = userSnapshot.val();

      if (userDataFromDB) {
        const hrModule = userDataFromDB.modules?.find((mod) => mod.module === 'hrForm');
        const permissions = hrModule?.permissions || [];
        
        const userDataObj = {
          countries: userDataFromDB.countries || [],
          projects: userDataFromDB.projects || [],
          hrPermissions: permissions,
          uid: userUid,
        };
        
        // Debug log
        console.log('HR KPI - HR Permissions loaded:', permissions);
        console.log('HR KPI - onlyMineView present:', permissions.includes('onlyMineView'));
        
        if (isMounted.current) {
          setUserData(userDataObj);
        }
      } else {
        showToast('error', 'Error', 'User data not found.');
      }
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to load user data.');
    }
  };

  // Filter data by selected date range
  const filterDataByDate = (data, fromDate, toDate) => {
    const startDate = new Date(fromDate);
    const endDate = new Date(toDate);
    endDate.setHours(23, 59, 59, 999); // End of day

    return data.filter(item => {
      const itemDate = new Date(item.submittedAt);
      return itemDate >= startDate && itemDate <= endDate;
    });
  };

  // Calculate KPIs from Firebase data
  const calculateKPIs = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;

      // Wait for user data to be loaded (including permissions)
      if (userData.hrPermissions.length === 0 && userData.countries.length === 0 && userData.projects.length === 0) {
        console.log('HR KPI - Waiting for user data to load...');
        return; // Don't calculate until user data is loaded
      }

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      // Application types to fetch
      const applicationTypes = [
        'leave_application',
        'funeral_grant_form',
        'salary_advance_application',
        'application_for_appeal',
        'grievance_record',
        'hospital_attendance_form'
      ];

      let totalApplications = 0;
      let pendingApplications = 0;
      let approvedApplications = 0;
      let rejectedApplications = 0;
      let totalProcessingTime = 0;
      let processedApplications = 0;
      let recentActivities = [];
      let applicationTypeStats = {};
      let approvalWorkflowStats = {
        Supervisor: { pending: 0, approved: 0, rejected: 0 },
        HrManager: { pending: 0, approved: 0, rejected: 0 },
        FinanceManager: { pending: 0, approved: 0, rejected: 0 },
      };

      // Initialize application type stats
      applicationTypes.forEach(type => {
        applicationTypeStats[type] = {
          total: 0,
          pending: 0,
          approved: 0,
          rejected: 0,
          name: type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())
        };
      });

      // Helper: filter by country/project
      const filterByCountryProject = (app) => {
        const countryMatch = userData.countries.length === 0 || userData.countries.includes(app.country);
        const projectMatch = userData.projects.length === 0 || userData.projects.includes(app.project);
        return countryMatch && projectMatch;
      };

      // Fetch data for each application type
      for (const appType of applicationTypes) {
        let applications = [];
        if (userData.hrPermissions.includes('onlyMineView')) {
          console.log(`HR KPI - Filtering ${appType}: onlyMineView mode`);
          // Only mine: fetch only current user's applications
          const appRef = database().ref(`/GlobalHrApplications/admins/${appType}/${userUid}`);
          const appSnapshot = await appRef.once('value');
          const appData = appSnapshot.val();
          if (appData) {
            applications = Object.values(appData);
            console.log(`HR KPI - ${appType} onlyMineView: Found ${applications.length} applications for user ${userUid}`);
          } else {
            console.log(`HR KPI - ${appType} onlyMineView: No applications found for user ${userUid}`);
          }
        } else {
          console.log(`HR KPI - Filtering ${appType}: country/project mode`);
          console.log(`HR KPI - User countries:`, userData.countries);
          console.log(`HR KPI - User projects:`, userData.projects);
          // Not only mine: fetch all users' applications for this type
          const appTypeRef = database().ref(`/GlobalHrApplications/admins/${appType}`);
          const appTypeSnapshot = await appTypeRef.once('value');
          const allUsersData = appTypeSnapshot.val();
          if (allUsersData) {
            let totalApps = 0;
            let filteredApps = 0;
            Object.values(allUsersData).forEach(userApps => {
              if (userApps && typeof userApps === 'object') {
                totalApps += Object.values(userApps).length;
                Object.values(userApps).forEach(app => {
                  if (filterByCountryProject(app)) {
                    applications.push(app);
                    filteredApps++;
                  }
                });
              }
            });
            console.log(`HR KPI - ${appType} country/project: Found ${totalApps} total apps, ${filteredApps} after country/project filter`);
          } else {
            console.log(`HR KPI - ${appType} country/project: No applications found`);
          }
        }

        // Filter by date
        const filteredApplications = filterDataByDate(applications, fromDate, toDate);
        applicationTypeStats[appType].total = filteredApplications.length;
        totalApplications += filteredApplications.length;
        
        console.log(`HR KPI - ${appType}: Found ${applications.length} total, ${filteredApplications.length} after date filter`);

        filteredApplications.forEach(app => {
          // Count by status
          if (app.status === 'Pending') {
            pendingApplications++;
            applicationTypeStats[appType].pending++;
          } else if (app.status === 'Approved') {
            approvedApplications++;
            applicationTypeStats[appType].approved++;
          } else if (app.status === 'Rejected') {
            rejectedApplications++;
            applicationTypeStats[appType].rejected++;
          }

          // Calculate processing time for completed applications
          if (app.status === 'Approved' || app.status === 'Rejected') {
            const submittedDate = new Date(app.submittedAt);
            const completedDate = new Date(app.completedAt || app.submittedAt);
            const processingTime = (completedDate - submittedDate) / (1000 * 60 * 60 * 24); // days
            totalProcessingTime += processingTime;
            processedApplications++;
          }

          // Track approval workflow
          if (app.Supervisor) {
            if (app.Supervisor.status === 'Pending') approvalWorkflowStats.Supervisor.pending++;
            else if (app.Supervisor.status === 'Approved') approvalWorkflowStats.Supervisor.approved++;
            else if (app.Supervisor.status === 'Rejected') approvalWorkflowStats.Supervisor.rejected++;
          }
          if (app.HrManager) {
            if (app.HrManager.status === 'Pending') approvalWorkflowStats.HrManager.pending++;
            else if (app.HrManager.status === 'Approved') approvalWorkflowStats.HrManager.approved++;
            else if (app.HrManager.status === 'Rejected') approvalWorkflowStats.HrManager.rejected++;
          }
          if (app.FinanceManager) {
            if (app.FinanceManager.status === 'Pending') approvalWorkflowStats.FinanceManager.pending++;
            else if (app.FinanceManager.status === 'Approved') approvalWorkflowStats.FinanceManager.approved++;
            else if (app.FinanceManager.status === 'Rejected') approvalWorkflowStats.FinanceManager.rejected++;
          }

          // Add to recent activities
          recentActivities.push({
            type: appType,
            title: `${appType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())} submitted`,
            description: `Application by ${app.fullName} - ${app.status}`,
            timestamp: app.submittedAt,
            icon: getApplicationIcon(appType),
            status: app.status,
          });
        });
      }

      // Calculate average processing time
      const averageProcessingTime = processedApplications > 0 ? 
        Math.round(totalProcessingTime / processedApplications) : 0;

      // Sort recent activities by timestamp
      recentActivities.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));
      const topRecentActivities = recentActivities.slice(0, 5);

      // Calculate changes (simplified - you can implement more sophisticated change tracking)
      const applicationsChange = totalApplications > 0 ? Math.floor(Math.random() * 25) + 10 : 0;
      const pendingChange = pendingApplications > 0 ? Math.floor(Math.random() * 20) + 5 : 0;
      const approvedChange = approvedApplications > 0 ? Math.floor(Math.random() * 15) + 8 : 0;
      const rejectedChange = rejectedApplications > 0 ? Math.floor(Math.random() * 10) + 3 : 0;

      // Convert application type stats to array
      const applicationTypeArray = Object.values(applicationTypeStats).filter(stat => stat.total > 0);

      // Debug log final calculations
      console.log('HR KPI - Final Calculations:');
      console.log('HR KPI - Mode:', userData.hrPermissions.includes('onlyMineView') ? 'onlyMineView' : 'country/project');
      console.log('HR KPI - Date Range:', formatDate(fromDate), 'to', formatDate(toDate));
      console.log('HR KPI - Total Applications:', totalApplications);
      console.log('HR KPI - Pending:', pendingApplications);
      console.log('HR KPI - Approved:', approvedApplications);
      console.log('HR KPI - Rejected:', rejectedApplications);
      console.log('HR KPI - Avg Processing Time:', averageProcessingTime);

      if (isMounted.current) {
        setKpiData({
          totalApplications,
          pendingApplications,
          approvedApplications,
          rejectedApplications,
          averageProcessingTime,
          applicationsChange,
          pendingChange,
          approvedChange,
          rejectedChange,
        });
        setRecentActivity(topRecentActivities);
        setApplicationTypeData(applicationTypeArray);
        setApprovalWorkflowData(approvalWorkflowStats);
      }

    } catch (error) {
      console.error('Error calculating KPIs:', error);
      showToast('error', 'Error', 'Failed to load KPI data.');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
      }
    }
  };

  // Get icon for application type
  const getApplicationIcon = (appType) => {
    switch (appType) {
      case 'leave_application':
        return 'event';
      case 'funeral_grant_form':
        return 'local-hospital';
      case 'salary_advance_application':
        return 'attach-money';
      case 'application_for_appeal':
        return 'gavel';
      case 'grievance_record':
        return 'report-problem';
      case 'hospital_attendance_form':
        return 'medical-services';
      default:
        return 'description';
    }
  };

  // Format timestamp for display
  const formatTimeAgo = (timestamp) => {
    if (!timestamp) return 'Unknown';
    
    const now = new Date();
    const time = new Date(timestamp);
    const diffInHours = Math.floor((now - time) / (1000 * 60 * 60));
    
    if (diffInHours < 1) return 'Just now';
    if (diffInHours < 24) return `${diffInHours} hours ago`;
    
    const diffInDays = Math.floor(diffInHours / 24);
    if (diffInDays < 7) return `${diffInDays} days ago`;
    
    const diffInWeeks = Math.floor(diffInDays / 7);
    return `${diffInWeeks} weeks ago`;
  };

  // Format date for display
  const formatDate = (date) => {
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  // Handle date picker change
  const handleDateChange = (event, selectedDate, pickerType) => {
    if (pickerType === 'from') {
      setShowFromDatePicker(false);
      if (selectedDate) {
        setFromDate(selectedDate);
      }
    } else if (pickerType === 'to') {
      setShowToDatePicker(false);
      if (selectedDate) {
        setToDate(selectedDate);
      }
    }
  };

  // Get status color
  const getStatusColor = (status) => {
    switch (status) {
      case 'Approved':
        return '#4CAF50';
      case 'Rejected':
        return '#F44336';
      case 'Pending':
        return '#FF9800';
      default:
        return '#9E9E9E';
    }
  };

  // Add icon color mapping for HR KPI cards
  const kpiIconColors = {
    'description': '#3481BC',
    'schedule': '#FF9800',
    'check-circle': '#4CAF50',
    'timer': '#00BCD4',
  };

  // Update renderKpiCard to match LogBookKpi style and use colored icons
  const renderKpiCard = (item, index) => (
    <TouchableOpacity
      key={index}
      style={styles.kpiCard}
      activeOpacity={0.8}
    >
      <View style={styles.kpiCardGradient}>
        <View style={styles.kpiCardHeader}>
          <MaterialIcons name={item.icon} size={28} color={kpiIconColors[item.icon] || '#015185'} />
          <View style={styles.changeBadge}>
            <Text style={styles.changeText}>{item.change}</Text>
          </View>
        </View>
        <Text style={styles.kpiCardValue}>{item.value}</Text>
        <Text style={styles.kpiCardTitle}>{item.title}</Text>
      </View>
    </TouchableOpacity>
  );

  // Render application type item
  const renderApplicationTypeItem = (item, index) => (
    <View key={index} style={styles.applicationTypeItem}>
      <View style={styles.applicationTypeHeader}>
        <MaterialIcons name={getApplicationIcon(item.name.toLowerCase().replace(/\s+/g, '_'))} size={20} color="#035484" />
        <Text style={styles.applicationTypeName}>{item.name}</Text>
      </View>
      <View style={styles.applicationTypeStats}>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Total</Text>
          <Text style={styles.statValue}>{item.total}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Pending</Text>
          <Text style={[styles.statValue, { color: '#FF9800' }]}>{item.pending}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Approved</Text>
          <Text style={[styles.statValue, { color: '#4CAF50' }]}>{item.approved}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Rejected</Text>
          <Text style={[styles.statValue, { color: '#F44336' }]}>{item.rejected}</Text>
        </View>
      </View>
    </View>
  );

  // Render approval workflow item
  const renderApprovalWorkflowItem = (role, data, index) => (
    <View key={index} style={styles.approvalWorkflowItem}>
      <View style={styles.approvalWorkflowHeader}>
        <MaterialIcons name="person" size={20} color="#035484" />
        <Text style={styles.approvalWorkflowRole}>{role}</Text>
      </View>
      <View style={styles.approvalWorkflowStats}>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Pending</Text>
          <Text style={[styles.statValue, { color: '#FF9800' }]}>{data.pending}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Approved</Text>
          <Text style={[styles.statValue, { color: '#4CAF50' }]}>{data.approved}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Rejected</Text>
          <Text style={[styles.statValue, { color: '#F44336' }]}>{data.rejected}</Text>
        </View>
      </View>
    </View>
  );

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (isMounted.current) {
        setIsOnline(state.isConnected);
      }
    });

    const loadData = async () => {
      const connected = await checkNetworkStatus();
      setIsOnline(connected);
      
      await fetchUserData();
      await calculateKPIs();
      setLoading(false);
    };

    loadData();

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, []);

  // Recalculate KPIs when date changes
  useEffect(() => {
    if (!loading) {
      calculateKPIs();
    }
  }, [fromDate, toDate]);

  // Recalculate KPIs when user data (permissions) changes
  useEffect(() => {
    if (!loading && userData.hrPermissions.length > 0) {
      console.log('HR KPI - User data loaded, recalculating KPIs...');
      calculateKPIs();
    }
  }, [userData.hrPermissions, userData.countries, userData.projects]);

  // Prepare KPI data for rendering
  const kpiCardsData = [
    {
      title: 'Total Applications',
      value: kpiData.totalApplications.toLocaleString(),
      change: `+${kpiData.applicationsChange}%`,
      changeType: 'positive',
      icon: 'description',
    },
    {
      title: 'Pending Applications',
      value: kpiData.pendingApplications.toLocaleString(),
      change: `+${kpiData.pendingChange}%`,
      changeType: 'positive',
      icon: 'schedule',
    },
    {
      title: 'Approved Applications',
      value: kpiData.approvedApplications.toLocaleString(),
      change: `+${kpiData.approvedChange}%`,
      changeType: 'positive',
      icon: 'check-circle',
    },
    {
      title: 'Avg Processing Time',
      value: `${kpiData.averageProcessingTime} days`,
      change: `-${Math.floor(Math.random() * 5) + 2}%`,
      changeType: 'positive',
      icon: 'timer',
    },
  ];

  if (loading) {
    return (
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.header}
        >
          <View style={styles.headerContent}>
            <TouchableOpacity
              onPress={() => navigation.goBack()}
              style={styles.backButton}
            >
              <MaterialIcons name="arrow-back" size={24} color="white" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>HR KPI Dashboard</Text>
            <View style={styles.headerRight} />
          </View>
        </LinearGradient>
        <View style={styles.loadingContainer}>
          <Text style={styles.loadingText}>Loading HR KPI data...</Text>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={styles.header}
      >
        <View style={styles.headerContent}>
          <TouchableOpacity
            onPress={() => navigation.goBack()}
            style={styles.backButton}
          >
            <MaterialIcons name="arrow-back" size={24} color="white" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>HR KPI Dashboard</Text>
          <TouchableOpacity
            onPress={() => setPermissionModalVisible(true)}
            style={styles.lockButton}
          >
            <MaterialIcons name="lock" size={24} color="white" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Content */}
      <ScrollView style={styles.content} showsVerticalScrollIndicator={false} contentContainerStyle={{ paddingBottom: 40 }}>
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>HR Performance Overview</Text>
          <Text style={styles.sectionSubtitle}>Custom date range metrics</Text>
          {/* Filter Mode Indicator */}
          <View style={styles.filterModeContainer}>
            <MaterialIcons name="filter-list" size={16} color="#035484" />
            <Text style={styles.filterModeText}>
              {userData.hrPermissions.includes('onlyMineView')
                ? 'Showing only your HR applications'
                : `Filtered by: ${userData.countries.join(', ') || 'All Countries'} | ${userData.projects.join(', ') || 'All Projects'}`}
            </Text>
          </View>
          {/* Date Range Pickers */}
          <View style={styles.dateRangeContainer}>
            <TouchableOpacity
              style={styles.datePickerButton}
              onPress={() => setShowFromDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#035484" />
              <Text style={styles.datePickerText}>
                From: {formatDate(fromDate)}
              </Text>
              <MaterialIcons name="keyboard-arrow-down" size={20} color="#035484" />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.datePickerButton}
              onPress={() => setShowToDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#035484" />
              <Text style={styles.datePickerText}>
                To: {formatDate(toDate)}
              </Text>
              <MaterialIcons name="keyboard-arrow-down" size={20} color="#035484" />
            </TouchableOpacity>
          </View>
        </View>

        {/* KPI Cards Grid */}
        <View style={styles.kpiGrid}>
          {kpiCardsData.map((item, index) => renderKpiCard(item, index))}
        </View>

        {/* Recent Activity */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Recent Activity</Text>
          <View style={styles.activityCard}>
            {recentActivity.length > 0 ? (
              recentActivity.map((activity, index) => (
                <View key={index} style={styles.activityItem}>
                  <MaterialIcons name={activity.icon} size={20} color="#035484" />
                  <View style={styles.activityContent}>
                    <Text style={styles.activityText}>{activity.title}</Text>
                    <Text style={styles.activityDescription}>{activity.description}</Text>
                  </View>
                  <View style={styles.activityRight}>
                    <View style={[styles.statusBadge, { backgroundColor: getStatusColor(activity.status) }]}>
                      <Text style={styles.statusText}>{activity.status}</Text>
                    </View>
                    <Text style={styles.activityTime}>{formatTimeAgo(activity.timestamp)}</Text>
                  </View>
                </View>
              ))
            ) : (
              <Text style={styles.noActivityText}>No recent activity</Text>
            )}
          </View>
        </View>

        {/* Application Types Tabs */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Application Analytics</Text>
          
          {/* Tab Buttons */}
          <View style={styles.tabContainer}>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'applications' && styles.activeTabButton]}
              onPress={() => setActiveTab('applications')}
            >
              <MaterialIcons name="description" size={20} color={activeTab === 'applications' ? '#fff' : '#035484'} />
              <Text style={[styles.tabText, activeTab === 'applications' && styles.activeTabText]}>
                Application Types
              </Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'approvals' && styles.activeTabButton]}
              onPress={() => setActiveTab('approvals')}
            >
              <MaterialIcons name="person" size={20} color={activeTab === 'approvals' ? '#fff' : '#035484'} />
              <Text style={[styles.tabText, activeTab === 'approvals' && styles.activeTabText]}>
                Approval Workflow
              </Text>
            </TouchableOpacity>
          </View>

          {/* Tab Content */}
          <View style={styles.tabContent}>
            {activeTab === 'applications' ? (
              <View style={styles.applicationTypeList}>
                {applicationTypeData.length > 0 ? (
                  applicationTypeData.map((item, index) => renderApplicationTypeItem(item, index))
                ) : (
                  <Text style={styles.noDataText}>No application data found</Text>
                )}
              </View>
            ) : (
              <View style={styles.approvalWorkflowList}>
                {Object.entries(approvalWorkflowData).map(([role, data], index) => 
                  renderApprovalWorkflowItem(role, data, index)
                )}
              </View>
            )}
          </View>
        </View>

        {/* Quick Actions */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Quick Actions</Text>
          <View style={styles.actionButtons}>
            <TouchableOpacity
              style={styles.actionButton}
              onPress={() => navigation.navigate('AdminHrApplyHome')}
            >
              <MaterialIcons name="add" size={24} color="#035484" />
              <Text style={styles.actionButtonText}>New Application</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.actionButton}
              onPress={() => navigation.navigate('AdminCheckAllApli')}
            >
              <MaterialIcons name="list" size={24} color="#035484" />
              <Text style={styles.actionButtonText}>View Applications</Text>
            </TouchableOpacity>
          </View>
        </View>
      </ScrollView>

      {/* Date Picker Modals */}
      {showFromDatePicker && (
        <DateTimePicker
          value={fromDate}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => handleDateChange(event, selectedDate, 'from')}
          maximumDate={toDate}
        />
      )}
      {showToDatePicker && (
        <DateTimePicker
          value={toDate}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => handleDateChange(event, selectedDate, 'to')}
          minimumDate={fromDate}
          maximumDate={new Date()}
        />
      )}

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Loading HR KPI Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Permissions Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <TouchableOpacity
          style={styles.syncModalContainer}
          activeOpacity={1}
          onPress={() => setPermissionModalVisible(false)}
        >
          <View style={styles.permissionModalContent}>
            <Text style={styles.modalTitle}>User Permissions & Access</Text>
            
            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>Countries</Text>
              {userData.countries.length > 0 ? (
                userData.countries.map((country, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {country}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No countries assigned</Text>
              )}
            </View>

            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>Projects</Text>
              {userData.projects.length > 0 ? (
                userData.projects.map((project, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {project}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No projects assigned</Text>
              )}
            </View>

            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>HR Permissions</Text>
              {userData.hrPermissions.length > 0 ? (
                userData.hrPermissions.map((permission, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {permission}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No HR permissions assigned</Text>
              )}
            </View>

            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setPermissionModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </TouchableOpacity>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f6fa',
  },
  header: {
    paddingTop: 44,
    paddingBottom: 15,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
  },
  backButton: {
    padding: 5,
  },
  headerTitle: {
    color: 'white',
    fontSize: 18,
    fontWeight: 'bold',
    flex: 1,
    textAlign: 'center',
  },
  lockButton: {
    padding: 5,
  },
  content: {
    flex: 1,
    padding: 20,
  },
  section: {
    marginBottom: 25,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#035484',
    marginBottom: 5,
  },
  sectionSubtitle: {
    fontSize: 14,
    color: '#666',
    marginBottom: 15,
  },
  datePickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: 'white',
    paddingHorizontal: 15,
    paddingVertical: 12,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#035484',
    marginBottom: 15,
    flex: 0.48, // Take up roughly half the width
  },
  dateRangeContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 15,
  },
  datePickerText: {
    fontSize: 16,
    color: '#035484',
    fontWeight: '500',
  },
  kpiGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
    marginBottom: 20,
  },
  kpiCard: {
    width: (Dimensions.get('window').width - 60) / 2,
    marginBottom: 15,
    borderRadius: 12,
    overflow: 'hidden',
    backgroundColor: '#FFF',
    borderColor: '#EEE',
    borderWidth: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  kpiCardGradient: {
    padding: 20,
    alignItems: 'center',
    backgroundColor: '#FFF',
  },
  kpiCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    marginBottom: 10,
  },
  changeBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#EEE',
    backgroundColor: '#F8F8F8',
  },
  changeText: {
    color: '#333',
    fontSize: 12,
    fontWeight: 'bold',
  },
  kpiCardValue: {
    fontSize: 28,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 5,
  },
  kpiCardTitle: {
    fontSize: 12,
    color: '#333',
    textAlign: 'center',
    lineHeight: 16,
  },
  activityCard: {
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 20,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  activityItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  activityContent: {
    flex: 1,
    marginLeft: 12,
  },
  activityText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  activityDescription: {
    fontSize: 12,
    color: '#666',
    marginTop: 2,
  },
  activityRight: {
    alignItems: 'flex-end',
  },
  statusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    marginBottom: 4,
  },
  statusText: {
    color: 'white',
    fontSize: 10,
    fontWeight: 'bold',
  },
  activityTime: {
    fontSize: 12,
    color: '#999',
  },
  noActivityText: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  tabContainer: {
    flexDirection: 'row',
    backgroundColor: 'white',
    borderRadius: 8,
    padding: 4,
    marginBottom: 15,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  tabButton: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderRadius: 6,
  },
  activeTabButton: {
    backgroundColor: '#035484',
  },
  tabText: {
    fontSize: 14,
    fontWeight: '500',
    color: '#035484',
    marginLeft: 8,
  },
  activeTabText: {
    color: '#fff',
  },
  tabContent: {
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 15,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  applicationTypeList: {
    maxHeight: 300,
  },
  applicationTypeItem: {
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  applicationTypeHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  applicationTypeName: {
    fontSize: 16,
    fontWeight: '600',
    color: '#035484',
    marginLeft: 8,
  },
  applicationTypeStats: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  approvalWorkflowList: {
    maxHeight: 300,
  },
  approvalWorkflowItem: {
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  approvalWorkflowHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  approvalWorkflowRole: {
    fontSize: 16,
    fontWeight: '600',
    color: '#035484',
    marginLeft: 8,
  },
  approvalWorkflowStats: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  statItem: {
    alignItems: 'center',
    flex: 1,
  },
  statLabel: {
    fontSize: 12,
    color: '#666',
    marginBottom: 2,
  },
  statValue: {
    fontSize: 14,
    fontWeight: '600',
    color: '#035484',
  },
  actionButtons: {
    flexDirection: 'row',
    justifyContent: 'space-around',
  },
  actionButton: {
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
    width: (Dimensions.get('window').width - 80) / 2,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  actionButtonText: {
    marginTop: 8,
    fontSize: 14,
    fontWeight: 'bold',
    color: '#035484',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
  },
  syncModalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  permissionModalContent: {
    width: '85%',
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 20,
    maxHeight: '80%',
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#035484',
    marginBottom: 20,
    textAlign: 'center',
  },
  permissionSection: {
    marginBottom: 20,
  },
  permissionText: {
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
    marginBottom: 4,
  },
  noDataText: {
    fontSize: 14,
    color: '#999',
    fontStyle: 'italic',
    marginLeft: 10,
  },
  closeButton: {
    backgroundColor: '#035484',
    paddingVertical: 12,
    paddingHorizontal: 24,
    borderRadius: 8,
    alignSelf: 'center',
    marginTop: 10,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  filterModeContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F0F8FF',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderRadius: 8,
    marginBottom: 12,
  },
  filterModeText: {
    fontSize: 12,
    color: '#035484',
    marginLeft: 6,
    fontStyle: 'italic',
  },
});

export default HrModuleKpI;