import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  Modal,
  Alert,
  InteractionManager,
  ActivityIndicator,
} from 'react-native';
import { Animated } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';

const { width, height } = Dimensions.get('window');

const HSE_051step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [jobSteps, setJobSteps] = useState([
    {
      id: 1,
      step: '',
      describeJobStep: '',
      hazardsPotentialIncidents: '',
      riskControlMethodsRequired: '',
    }
  ]);
  const [isLoading, setIsLoading] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  const [reportId, setReportId] = useState(null);
  const [pdfUrl, setPdfUrl] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);

  // Loader modal - use native ActivityIndicator to avoid heavy animation on iOS
  // Accepts onDismiss callback so callers can wait for the modal to fully close before showing another modal
  const SyncModal = ({ visible, message = 'Loading Data', progress = 0, onDismiss = () => {} }) => {
    useEffect(() => {
      return () => {};
    }, [visible]);
    return (
      <Modal
        visible={visible}
        transparent
        animationType="none"
        presentationStyle="overFullScreen"
        onDismiss={onDismiss}
        onRequestClose={onDismiss}
        hardwareAccelerated={true}
      >
        <View style={styles.syncModalOverlayCentered}>
          <View style={styles.syncModalBox}>
            <ActivityIndicator size="large" color="#015185" />
            <Text style={[styles.syncModalText, { marginTop: 12 }]}>{message}</Text>
            {typeof progress === 'number' && (
              <View style={styles.progressContainer}>
                <View style={styles.progressBar}>
                  <View style={[styles.progressFill, { width: `${progress}%` }]} />
                </View>
              </View>
            )}
          </View>
        </View>
      </Modal>
    );
  };

  const isMounted = useRef(true);

  useEffect(() => {
    loadSavedData();
    return () => { isMounted.current = false; };
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse051Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        if (parsedData.jobSteps) {
          setJobSteps(parsedData.jobSteps);
        }
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async (data) => {
    try {
      await AsyncStorage.setItem('hse051Step3', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data',
        position: 'top',
      });
    }
  };

  const handleJobStepChange = (id, field, value) => {
    const updatedJobSteps = jobSteps.map(step => 
      step.id === id ? { ...step, [field]: value } : step
    );
    setJobSteps(updatedJobSteps);
    saveData({ jobSteps: updatedJobSteps });
  };

  const addNewEntry = () => {
    const newId = Math.max(...jobSteps.map(step => step.id)) + 1;
    const newEntry = {
      id: newId,
      step: '',
      describeJobStep: '',
      hazardsPotentialIncidents: '',
      riskControlMethodsRequired: '',
    };
    const updatedJobSteps = [...jobSteps, newEntry];
    setJobSteps(updatedJobSteps);
    saveData({ jobSteps: updatedJobSteps });
  };

  const removeEntry = (id) => {
    if (jobSteps.length > 1) {
      const updatedJobSteps = jobSteps.filter(step => step.id !== id);
      setJobSteps(updatedJobSteps);
      saveData({ jobSteps: updatedJobSteps });
    }
  };

  const handleSubmit = async () => {
    // Validate required fields
    const missingFields = [];
    jobSteps.forEach((step, index) => {
      if (!step.step) missingFields.push(`Step ${index + 1} - Step Number`);
      if (!step.describeJobStep) missingFields.push(`Step ${index + 1} - Job Step Description`);
      if (!step.hazardsPotentialIncidents) missingFields.push(`Step ${index + 1} - Hazards`);
      if (!step.riskControlMethodsRequired) missingFields.push(`Step ${index + 1} - Risk Control Methods`);
    });
    
    if (missingFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.slice(0, 3).join(', ')}${missingFields.length > 3 ? '...' : ''}`,
        position: 'top',
      });
      return;
    }

    try {
      // show loader and mark loading state early so UI has chance to render
      setIsLoading(true);
      setSyncModalVisible(true);
      setUploadProgress(0);
      // Let the UI finish rendering the modal before starting heavy work
      await new Promise((resolve) => InteractionManager.runAfterInteractions(resolve));
      // small delay to ensure modal fully visible
      await new Promise((resolve) => setTimeout(resolve, 120));
      // Load step data
      const step1Data = await loadStep1Data();
      const step2Data = await loadStep2Data();
      const step3Data = { jobSteps };

      // Resolve userId
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) throw new Error('User not authenticated');
      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) throw new Error('Admin user not found');
      const userId = Object.keys(userData)[0];

      // Determine reportId (use request number from Step 1 like 43)
      const newReportId = step1Data?.requestNumber ? String(step1Data.requestNumber) : `HSE-051-${Date.now()}`;
      setReportId(newReportId);

      // Prepare base report
      const reportBase = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'processing',
        documentType: 'HSE-051',
        documentName: 'JOB HAZARD ANALYSIS WORKSHEET',
        step1: step1Data || {},
        step2: step2Data || {},
        step3: step3Data || {},
      };

      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${newReportId}`);
      await dbRef.set(reportBase);
      setUploadProgress(30);

      // Generate PDF and get file path
      const pdfPath = await generatePDF();

      // Upload PDF to storage
      const storagePath = `GlobalHSEInspections/admins/${userId}/reports/${newReportId}.pdf`;
      const uploadedUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath);
      setPdfUrl(uploadedUrl);

      // Update report with PDF URL
      await dbRef.update({ pdfDownloadUrl: uploadedUrl, status: 'completed' });
      setUploadProgress(70);

      // Clear temp file
      if (pdfPath) {
        await RNFS.unlink(pdfPath).catch(() => {});
      }

      // Increment global counter
      await incrementReportCounter();

      // Clear local step data (including common step1 used by 43)
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hse051Step2');
      await AsyncStorage.removeItem('hse051Step3');
      setUploadProgress(100);
      // Ensure sync modal is hidden before showing success modal to avoid modal stacking issues
      setSyncModalVisible(false);
      // Wait for dismissal + interactions, then show success modal
      try {
        await new Promise((resolve) => setTimeout(resolve, 300));
        await new Promise((resolve) => InteractionManager.runAfterInteractions(resolve));
      } catch (e) {}
      if (isMounted.current) setShowSuccessModal(true);
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', error.message || 'Failed to submit form. Please try again.');
    } finally {
      // Ensure sync modal hidden and stop loader
      setSyncModalVisible(false);
      setIsLoading(false);
    }
  };

  const handleGoToHSEHome = () => {
    setShowSuccessModal(false);
    navigation.navigate('HSEInspectionHome');
  };

  const handleViewPDF = async () => {
    try {
      // Prefer the stored URL
      if (pdfUrl) {
        setShowSuccessModal(false);
        navigation.navigate('HSEViewPDF', {
          pdfUrl,
          onGoBack: () => navigation.navigate('HSEInspectionHome')
        });
        return;
      }
      if (!reportId) throw new Error('Report ID not available');

      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) throw new Error('User not authenticated');
      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) throw new Error('User not found');
      const userId = Object.keys(userData)[0];
      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const data = snapshot.val();
      if (!data?.pdfDownloadUrl) throw new Error('PDF URL not available');
      setPdfUrl(data.pdfDownloadUrl);
      setShowSuccessModal(false);
      navigation.navigate('HSEViewPDF', {
        pdfUrl: data.pdfDownloadUrl,
        onGoBack: () => navigation.navigate('HSEInspectionHome')
      });
    } catch (e) {
      Toast.show({ type: 'error', text1: 'Error', text2: e.message || 'Could not open PDF', position: 'bottom' });
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      const actualPath = filePath.startsWith('file://') ? filePath.replace('file://', '') : filePath;
      await reference.putFile(actualPath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Upload error:', error);
      throw error;
    }
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      let currentNumber = snapshot.val();
      if (currentNumber === null || currentNumber === undefined) currentNumber = 0;
      currentNumber++;
      await counterRef.set(currentNumber);
      return currentNumber;
    } catch (error) {
      console.error('Error incrementing counter:', error);
      return 0;
    }
  };

  const generatePDF = async () => {
    try {
      setIsLoading(true);
      
      // Load data from all steps
      const step1Data = await loadStep1Data();
      const step2Data = await loadStep2Data();
      const step3Data = { jobSteps };
      
      // Debug: Log the data being used
      console.log('Step 1 Data:', step1Data);
      console.log('Step 2 Data:', step2Data);
      console.log('Step 3 Data:', step3Data);
      
      const html = generatePDFHTML(step1Data, step2Data, step3Data);
      
      const options = {
        html,
        fileName: `HSE_051_JHA_${step2Data.projectSite ? step2Data.projectSite.replace(/\s+/g, '_') : 'Report'}_${new Date().toISOString().split('T')[0]}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      
      if (file.filePath) {
        setGeneratedPdfPath(file.filePath);
        Toast.show({
          type: 'success',
          text1: 'PDF Generated',
          text2: `Saved to: ${file.filePath}`,
          position: 'bottom'
        });
        return file.filePath.startsWith('file://') ? file.filePath : `file://${file.filePath}`;
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      Toast.show({
        type: 'error',
        text1: 'PDF Generation Failed',
        text2: error.message || 'Please try again',
        position: 'bottom'
      });
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const loadStep1Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hseStep1');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 1 data:', error);
      return {};
    }
  };

  const loadStep2Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse051Step2');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 2 data:', error);
      return {};
    }
  };

  const generatePDFHTML = (step1, step2, step3) => {
    const { jobSteps = [] } = step3;
    
    // Generate job steps rows
    let jobStepsRows = '';
    jobSteps.forEach((step, index) => {
      jobStepsRows += `
        <tr>
          <td>${step.step || ''}</td>
          <td>${step.describeJobStep || ''}</td>
          <td>${step.hazardsPotentialIncidents || ''}</td>
          <td>${step.riskControlMethodsRequired || ''}</td>
        </tr>
      `;
    });
    
    // Add empty rows if less than 7
    for (let i = jobSteps.length; i < 7; i++) {
      jobStepsRows += `
        <tr>
          <td></td>
          <td class="empty-row"></td>
          <td class="empty-row"></td>
          <td class="empty-row"></td>
        </tr>
      `;
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Titan Drilling - Job Hazard Analysis Worksheet</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 0;
                  padding: 10px;
                  background-color: #f0f8ff;
                  font-size: 14px;
                  line-height: 1.4;
              }
              .container {
                  max-width: 100%;
                  margin: 0 auto;
                  background-color: white;
                  padding: 20px;
                  box-shadow: 0 0 10px rgba(0,0,0,0.1);
              }
              .header {
                  display: flex;
                  align-items: center;
                  justify-content: space-between;
                  margin-bottom: 20px;
              }
              .logo {
                  font-size: 24px;
                  color: blue;
                  margin-right: 10px;
              }
              .title {
                  font-size: 18px;
                  font-weight: bold;
                  flex-grow: 1;
              }
              .doc-info {
                  text-align: right;
                  font-size: 12px;
              }
              .doc-info table {
                  border-collapse: collapse;
              }
              .doc-info td {
                  padding: 2px 5px;
                  border: 1px solid black;
              }
              .form-title {
                  text-align: center;
                  font-size: 16px;
                  font-weight: bold;
                  margin-bottom: 10px;
              }
              .form-table {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 10px;
              }
              .form-table th, .form-table td {
                  border: 1px solid black;
                  padding: 8px;
                  text-align: left;
              }
              .form-table th {
                  background-color: #d3d3d3;
                  font-weight: normal;
              }
              .empty-row {
                  height: 50px;
              }
              .label-cell {
                  background-color: #d3d3d3;
                  font-weight: normal;
              }
              @media (max-width: 600px) {
                  body {
                      padding: 5px;
                      font-size: 12px;
                  }
                  .header {
                      flex-direction: column;
                      align-items: flex-start;
                  }
                  .doc-info {
                      text-align: left;
                      margin-top: 10px;
                  }
                  .title {
                      font-size: 16px;
                  }
                  .form-table {
                      font-size: 12px;
                  }
                  .form-table th, .form-table td {
                      padding: 5px;
                  }
              }
          </style>
      </head>
      <body>
          <div class="container">
              <div class="header">
                  <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
                  <div class="doc-info">
                      <table>
                          <tr><td>Document Name</td><td>HSEC-FOR-6.1.2 JHA</td></tr>
                          <tr><td>Document ID</td><td>HSE-051</td></tr>
                          <tr><td>Version Date</td><td>8/11/2023</td></tr>
                          <tr><td>Version</td><td>2</td></tr>
                          <tr><td>Approved</td><td>HSM</td></tr>
                          <tr><td>Request #</td><td>${step1.requestNumber || ''}</td></tr>
                      </table>
                  </div>
              </div>
              <div class="form-title">JOB HAZARD ANALYSIS WORKSHEET</div>
              <table class="form-table">
                  <tr><th class="label-cell">TASK BEING ANALYSED:</th><td>${step1.taskBeingAnalysed || ''}</td></tr>
                  <tr><th class="label-cell">PROJECT / SITE:</th><td>${step2.projectSite || ''}</td></tr>
                  <tr><th class="label-cell">AREA:</th><td>${step2.area || ''}</td></tr>
                  <tr><th class="label-cell">DRILLER / PIC / SUPERVISOR:</th><td>${step2.drillerPicSupervisor || ''}</td></tr>
                  <tr><th class="label-cell">DATE:</th><td>${new Date().toLocaleDateString('en-GB')}</td></tr>
                  <tr><th class="label-cell">REQUIRED PPE:</th><td>${step2.requiredPpe || ''}</td></tr>
                  <tr><th class="label-cell">List Associated Procedures/Work Instructions or Temporary Work Instruction:</th><td>${step2.listAssociatedProcedures || ''}</td></tr>
              </table>
              <table class="form-table">
                  <tr>
                      <th>Step</th>
                      <th>Describe Job Step</th>
                      <th>Hazards/Potential Incidents</th>
                      <th>Risk Control Methods Required</th>
                  </tr>
                  ${jobStepsRows}
              </table>
          </div>
      </body>
      </html>
    `;
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.scrollView} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        {isLoading && (
          <View style={styles.loadingContainer}>
            <MaterialIcons name="hourglass-empty" size={40} color="#015185" />
            <Text style={styles.loadingText}>Generating PDF Report...</Text>
            <Text style={styles.loadingSubtext}>Please wait while we compile your data</Text>
          </View>
        )}

        {/* Form Content */}
        <View style={styles.formContainer}>
          {/* Associated Procedures Section */}
          

          {/* JHA Table Section */}
          <View style={styles.section}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>List Associated Procedures/Work Instructions or Temporary Work Instruction:</Text>
            </View>
            
            <View style={styles.formFields}>
              {jobSteps.map((step, index) => (
                <View key={step.id} style={styles.jobStepContainer}>
                  <View style={styles.jobStepHeader}>
                    <Text style={styles.jobStepTitle}>Job Step {index + 1}</Text>
                    {jobSteps.length > 1 && (
                      <TouchableOpacity 
                        style={styles.removeButton} 
                        onPress={() => removeEntry(step.id)}
                      >
                        <MaterialIcons name="delete" size={20} color="#dc3545" />
                      </TouchableOpacity>
                    )}
                  </View>

                  <View style={styles.fieldContainer}>
                    <Text style={styles.fieldLabel}>Step Number</Text>
                    <TextInput
                      style={styles.input}
                      value={step.step}
                      onChangeText={(value) => handleJobStepChange(step.id, 'step', value)}
                      placeholder="Step #"
                      placeholderTextColor="#999"
                      keyboardType="numeric"
                    />
                  </View>

                  <View style={styles.fieldContainer}>
                    <Text style={styles.fieldLabel}>Describe Job Step</Text>
                    <TextInput
                      style={styles.textArea}
                      value={step.describeJobStep}
                      onChangeText={(value) => handleJobStepChange(step.id, 'describeJobStep', value)}
                      placeholder="Describe the job step"
                      placeholderTextColor="#999"
                      multiline
                      numberOfLines={3}
                    />
                  </View>

                  <View style={styles.fieldContainer}>
                    <Text style={styles.fieldLabel}>Hazards/Potential Incidents</Text>
                    <TextInput
                      style={styles.textArea}
                      value={step.hazardsPotentialIncidents}
                      onChangeText={(value) => handleJobStepChange(step.id, 'hazardsPotentialIncidents', value)}
                      placeholder="List hazards and potential incidents"
                      placeholderTextColor="#999"
                      multiline
                      numberOfLines={3}
                    />
                  </View>

                  <View style={styles.fieldContainer}>
                    <Text style={styles.fieldLabel}>Risk Control Methods Required</Text>
                    <TextInput
                      style={styles.textArea}
                      value={step.riskControlMethodsRequired}
                      onChangeText={(value) => handleJobStepChange(step.id, 'riskControlMethodsRequired', value)}
                      placeholder="Specify risk control methods"
                      placeholderTextColor="#999"
                      multiline
                      numberOfLines={3}
                    />
                  </View>
                </View>
              ))}

              {/* Add New Entry Button */}
              <TouchableOpacity style={styles.addButton} onPress={addNewEntry}>
                <MaterialIcons name="add" size={24} color="#fff" />
                <Text style={styles.addButtonText}>Add New Entry</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </ScrollView>

      <SyncModal visible={syncModalVisible} message="Loading Data" progress={uploadProgress} />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContentSimple}>
            <Text style={styles.successTitle}>Success</Text>
            <Text style={styles.successSub}>PDF report has been generated.</Text>
            <View style={styles.successRowButtons}>
              <TouchableOpacity style={styles.successBtn} onPress={handleGoToHSEHome}>
                <Text style={styles.successBtnText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.successBtn, { marginLeft: 10 }]} onPress={handleViewPDF}>
                <Text style={styles.successBtnText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleSubmit} disabled={isLoading}>
          <Text style={styles.nextButtonText}>
            {isLoading ? 'Generating PDF...' : 'Generate Report'}
          </Text>
          <MaterialIcons style={styles.bottomicon1} name={isLoading ? "hourglass-empty" : "check"} size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#f8f9fa',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 20,
  },
  formContainer: {
    padding: 20,
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  formFields: {
    padding: 15,
  },
  jobStepContainer: {
    marginBottom: 20,
    backgroundColor: '#f9f9f9',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    padding: 15,
  },
  jobStepHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
    paddingBottom: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  jobStepTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
  },
  fieldContainer: {
    marginBottom: 15,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 40,
  },
  textArea: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 80,
    textAlignVertical: 'top',
  },
  removeButton: {
    padding: 5,
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#28a745',
    paddingVertical: 12,
    paddingHorizontal: 20,
    margin: 15,
    borderRadius: 8,
    elevation: 2,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  loadingContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    padding: 30,
    backgroundColor: '#f8f9fa',
    margin: 20,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  loadingText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginTop: 15,
    textAlign: 'center',
  },
  loadingSubtext: {
    fontSize: 14,
    color: '#666',
    marginTop: 8,
    textAlign: 'center',
  },
  successModalContentSimple: { backgroundColor: '#fff', borderRadius: 10, padding: 20, alignItems: 'center', width: '80%' },
  successTitle: { fontSize: 20, fontWeight: '700', color: '#015185' },
  successSub: { marginTop: 6, color: '#333' },
  successRowButtons: { flexDirection: 'row', marginTop: 12 },
  successBtn: { backgroundColor: '#015185', paddingVertical: 10, paddingHorizontal: 20, borderRadius: 6 },
  successBtnText: { color: '#fff', fontWeight: '700' },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalOverlayCentered: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalBox: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 30,
    width: width * 0.85,
    alignItems: 'center',
  },
  successIconContainer: {
    marginBottom: 20,
  },
  successModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  successModalSubtitle: {
    fontSize: 16,
    color: '#666',
    marginBottom: 25,
    textAlign: 'center',
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  successModalButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    flex: 1,
    marginHorizontal: 5,
  },
  viewPdfButton: {
    backgroundColor: '#28a745',
  },
  successModalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
});

export default HSE_051step3;