import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  Image,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import { useNavigation } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');

const HSE_158Step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    crewNames: '',
    dateAndShift: '',
    supervisorName: '',
    supervisorSignature: '',
    shiftType: 'DAY', // DAY or NIGHT
    approachObservation: '',
    physicalSymptoms: {
      eyesBloodshot: false,
      poorCoordination: false,
      slowerMovements: false,
      other: '',
    },
    mentalFunctioning: {
      distractedFromTask: false,
      shortTermMemoryLoss: false,
      zombieLikeBehaviour: false,
      poorConcentration: false,
      noddingOffMomentarily: false,
      other: '',
    },
    emotionMotivation: {
      seemsDepressed: false,
      easilyFrustrated: false,
      irritable: false,
      other: '',
    },
    riskAssessment: {
      fatigueIncidentOccurred: '',
      employeeAtRisk: '',
      riskAssociated: '',
      employeeExplanation: '',
      hoursSinceLastSlept: '',
      howPreventFatigue: '',
      lastBreak: '',
      reasonsForPoorSleep: '',
      tasksPerforming: '',
      highRiskTasks: '',
      howLongOnTask: '',
      lastWaterFood: '',
      breakDuration: '',
      lastWaterFood2: '',
      howPreventFatigue2: '',
    },
    fatigueLevel: {
      unsafeToWork: false,
      safeToWork: false,
      restrictedDuties: false,
    },
  });

  const [showDatePicker, setShowDatePicker] = useState(false);
  const [riskSectionExpanded, setRiskSectionExpanded] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse158Step2');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData);
        if (parsedData.dateAndShift) {
          setSelectedDate(new Date(parsedData.dateAndShift));
        }
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async (data) => {
    try {
      await AsyncStorage.setItem('hse158Step2', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data',
        position: 'top',
      });
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData(updatedData);
  };

  const handleNestedChange = (parentField, childField, value) => {
    const updatedData = {
      ...formData,
      [parentField]: {
        ...formData[parentField],
        [childField]: value,
      },
    };
    setFormData(updatedData);
    saveData(updatedData);
  };

  const handleCheckboxChange = (parentField, childField, value) => {
    handleNestedChange(parentField, childField, value);
  };

  const isValidSignatureUri = (uri) => uri && typeof uri === 'string' && uri.startsWith('data:image');

  const handleSignaturePress = () => {
    navigation.navigate('HSE41_42_signature', {
      role: 'Supervisor',
      onSave: (signature) => {
        handleInputChange('supervisorSignature', signature);
      },
    });
  };

  const handleClearSignature = () => {
    handleInputChange('supervisorSignature', '');
  };

  const handleImageError = () => {
    console.error('Supervisor signature image failed to load');
    Toast.show({ 
      type: 'error', 
      text1: 'Error', 
      text2: 'Failed to load supervisor signature. Please try again.', 
      position: 'bottom' 
    });
    handleClearSignature();
  };

  const toggleRiskSection = () => {
    setRiskSectionExpanded(!riskSectionExpanded);
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setSelectedDate(selectedDate);
      const dateString = selectedDate.toLocaleDateString('en-GB');
      handleInputChange('dateAndShift', dateString);
    }
  };

  const formatDate = (date) => {
    if (!date) return 'Select Date';
    if (typeof date === 'string') {
      try {
        date = new Date(date);
      } catch (error) {
        return 'Select Date';
      }
    }
    if (!(date instanceof Date) || isNaN(date.getTime())) {
      return 'Select Date';
    }
    return date.toLocaleDateString('en-GB');
  };

  const handleNext = () => {
    // Validate required fields
    const requiredFields = ['crewNames', 'dateAndShift', 'supervisorName'];
    const missingFields = requiredFields.filter(field => !formData[field]);
    
    if (missingFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.join(', ')}`,
        position: 'top',
      });
      return;
    }
    
    setActiveStep(3);
  };

  const handleBack = () => {
    setActiveStep(1);
  };

  const renderInputField = (label, field, placeholder, required = false, keyboardType = 'default', multiline = false) => (
    <View style={styles.inputContainer}>
      <Text style={styles.label}>
        {label}
        {required && <Text style={styles.required}> *</Text>}
      </Text>
      <TextInput
        style={[styles.input, multiline && styles.textArea]}
        value={formData[field]}
        onChangeText={(value) => handleInputChange(field, value)}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#999"
        multiline={multiline}
        numberOfLines={multiline ? 3 : 1}
      />
    </View>
  );

  const renderCheckbox = (label, checked, onPress) => (
    <TouchableOpacity style={styles.checkboxContainer} onPress={onPress}>
      <View style={[styles.checkbox, checked && styles.checkboxChecked]}>
        {checked && <MaterialIcons name="check" size={16} color="#fff" />}
      </View>
      <Text style={styles.checkboxLabel}>{label}</Text>
    </TouchableOpacity>
  );

  const renderCheckboxWithInput = (label, checked, onPress, otherField, otherValue, onOtherChange) => (
    <View style={styles.checkboxWithInputContainer}>
      <TouchableOpacity style={styles.checkboxContainer} onPress={onPress}>
        <View style={[styles.checkbox, checked && styles.checkboxChecked]}>
          {checked && <MaterialIcons name="check" size={16} color="#fff" />}
        </View>
        <Text style={styles.checkboxLabel}>{label}</Text>
      </TouchableOpacity>
      {checked && label === 'Other' && (
        <TextInput
          style={styles.otherInput}
          value={otherValue}
          onChangeText={onOtherChange}
          placeholder="Specify other"
          placeholderTextColor="#999"
        />
      )}
    </View>
  );

  const renderRadioGroup = (label, field, options) => (
    <View style={styles.radioContainer}>
      <Text style={styles.label}>{label}</Text>
      <View style={styles.radioGroup}>
        {options.map((option) => (
          <TouchableOpacity
            key={option}
            style={styles.radioOption}
            onPress={() => handleInputChange(field, option)}
          >
            <View style={[styles.radioButton, formData[field] === option && styles.radioButtonSelected]}>
              {formData[field] === option && <View style={styles.radioButtonInner} />}
            </View>
            <Text style={styles.radioLabel}>{option}</Text>
          </TouchableOpacity>
        ))}
      </View>
    </View>
  );

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.scrollView} contentContainerStyle={styles.scrollContent}>
        {/* Header */}
        

        {/* Form Content */}
        <View style={styles.formContainer}>
          {/* Crew Details */}
          <View style={styles.section}>
            <Text style={styles.sectionTitle}>Crew Details</Text>
            {renderInputField('Crew Name(s)', 'crewNames', 'Enter crew names', true)}
            
            <View style={styles.rowContainer}>
              <View style={styles.halfWidth}>
                <Text style={styles.label}>Date & Shift *</Text>
                <TouchableOpacity
                  style={styles.dateInput}
                  onPress={showDatePickerModal}
                >
                  <Text style={[
                    styles.dateInputText,
                    !formData.dateAndShift && styles.datePlaceholderText
                  ]}>
                    {formatDate(formData.dateAndShift)}
                  </Text>
                  <MaterialIcons name="calendar-today" size={20} color="#015185" />
                </TouchableOpacity>
              </View>
              <View style={styles.halfWidth}>
                <Text style={styles.label}>Shift Type</Text>
                <View style={styles.shiftToggle}>
                  <TouchableOpacity
                    style={[
                      styles.shiftButton,
                      formData.shiftType === 'DAY' && styles.shiftButtonActive
                    ]}
                    onPress={() => handleInputChange('shiftType', 'DAY')}
                  >
                    <Text style={[
                      styles.shiftButtonText,
                      formData.shiftType === 'DAY' && styles.shiftButtonTextActive
                    ]}>DAY</Text>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={[
                      styles.shiftButton,
                      formData.shiftType === 'NIGHT' && styles.shiftButtonActive
                    ]}
                    onPress={() => handleInputChange('shiftType', 'NIGHT')}
                  >
                    <Text style={[
                      styles.shiftButtonText,
                      formData.shiftType === 'NIGHT' && styles.shiftButtonTextActive
                    ]}>NIGHT</Text>
                  </TouchableOpacity>
                </View>
              </View>
            </View>

            {renderInputField('Supervisor Name', 'supervisorName', 'Enter supervisor name', true)}
            
            {/* Supervisor Signature */}
            <View style={styles.inputContainer}>
              <Text style={styles.label}>Supervisor Signature</Text>
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(formData.supervisorSignature) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleSignaturePress}>
                    <Image
                      source={{ uri: formData.supervisorSignature }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={handleImageError}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleSignaturePress}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
              {isValidSignatureUri(formData.supervisorSignature) && (
                <TouchableOpacity style={styles.clearSignatureButton} onPress={handleClearSignature}>
                  <MaterialIcons name="clear" size={16} color="#dc3545" />
                  <Text style={styles.clearSignatureText}>Clear Signature</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>

          {/* Step 1: Signs and Symptoms */}
          <View style={styles.section}>
            <Text style={styles.sectionTitle}>STEP 1: SIGNS AND SYMPTOMS (please tick)</Text>
            <Text style={styles.questionText}>
              Has there been a change in the crew's behaviour? If so, what have you observed and who?
            </Text>
            
            {renderInputField('Approach Observation', 'approachObservation', 'Describe what was observed on arrival at the site', false, 'default', true)}

            {/* Physical Symptoms */}
            <Text style={styles.categoryTitle}>Physical symptoms:</Text>
            {renderCheckbox(
              'Eyes bloodshot',
              formData.physicalSymptoms.eyesBloodshot,
              () => handleCheckboxChange('physicalSymptoms', 'eyesBloodshot', !formData.physicalSymptoms.eyesBloodshot)
            )}
            {renderCheckbox(
              'Poor co-ordination',
              formData.physicalSymptoms.poorCoordination,
              () => handleCheckboxChange('physicalSymptoms', 'poorCoordination', !formData.physicalSymptoms.poorCoordination)
            )}
            {renderCheckbox(
              'Slower movements',
              formData.physicalSymptoms.slowerMovements,
              () => handleCheckboxChange('physicalSymptoms', 'slowerMovements', !formData.physicalSymptoms.slowerMovements)
            )}
            {renderCheckboxWithInput(
              'Other',
              formData.physicalSymptoms.other !== '',
              () => handleCheckboxChange('physicalSymptoms', 'other', formData.physicalSymptoms.other === '' ? ' ' : ''),
              'other',
              formData.physicalSymptoms.other,
              (value) => handleCheckboxChange('physicalSymptoms', 'other', value)
            )}

            {/* Mental Functioning */}
            <Text style={styles.categoryTitle}>Mental functioning:</Text>
            {renderCheckbox(
              'Distracted from task',
              formData.mentalFunctioning.distractedFromTask,
              () => handleCheckboxChange('mentalFunctioning', 'distractedFromTask', !formData.mentalFunctioning.distractedFromTask)
            )}
            {renderCheckbox(
              'Short term memory loss (forgets instruction)',
              formData.mentalFunctioning.shortTermMemoryLoss,
              () => handleCheckboxChange('mentalFunctioning', 'shortTermMemoryLoss', !formData.mentalFunctioning.shortTermMemoryLoss)
            )}
            {renderCheckbox(
              "'Zombie' like behaviour",
              formData.mentalFunctioning.zombieLikeBehaviour,
              () => handleCheckboxChange('mentalFunctioning', 'zombieLikeBehaviour', !formData.mentalFunctioning.zombieLikeBehaviour)
            )}
            {renderCheckbox(
              'Poor concentration',
              formData.mentalFunctioning.poorConcentration,
              () => handleCheckboxChange('mentalFunctioning', 'poorConcentration', !formData.mentalFunctioning.poorConcentration)
            )}
            {renderCheckbox(
              'Nodding off momentarily',
              formData.mentalFunctioning.noddingOffMomentarily,
              () => handleCheckboxChange('mentalFunctioning', 'noddingOffMomentarily', !formData.mentalFunctioning.noddingOffMomentarily)
            )}
            {renderCheckboxWithInput(
              'Other',
              formData.mentalFunctioning.other !== '',
              () => handleCheckboxChange('mentalFunctioning', 'other', formData.mentalFunctioning.other === '' ? ' ' : ''),
              'other',
              formData.mentalFunctioning.other,
              (value) => handleCheckboxChange('mentalFunctioning', 'other', value)
            )}

            {/* Emotion / Motivation */}
            <Text style={styles.categoryTitle}>Emotion / motivation:</Text>
            {renderCheckbox(
              'Seems depressed',
              formData.emotionMotivation.seemsDepressed,
              () => handleCheckboxChange('emotionMotivation', 'seemsDepressed', !formData.emotionMotivation.seemsDepressed)
            )}
            {renderCheckbox(
              'Easily frustrated with tasks',
              formData.emotionMotivation.easilyFrustrated,
              () => handleCheckboxChange('emotionMotivation', 'easilyFrustrated', !formData.emotionMotivation.easilyFrustrated)
            )}
            {renderCheckbox(
              'Irritable',
              formData.emotionMotivation.irritable,
              () => handleCheckboxChange('emotionMotivation', 'irritable', !formData.emotionMotivation.irritable)
            )}
            {renderCheckboxWithInput(
              'Other',
              formData.emotionMotivation.other !== '',
              () => handleCheckboxChange('emotionMotivation', 'other', formData.emotionMotivation.other === '' ? ' ' : ''),
              'other',
              formData.emotionMotivation.other,
              (value) => handleCheckboxChange('emotionMotivation', 'other', value)
            )}
          </View>

          {/* Step 2: Risk */}
          <View style={styles.section}>
            <TouchableOpacity style={styles.dropdownHeader} onPress={toggleRiskSection}>
              <View style={styles.dropdownTitleContainer}>
                <Text style={styles.dropdownTitle}>STEP 2: RISK</Text>
                <MaterialIcons 
                  name={riskSectionExpanded ? "keyboard-arrow-up" : "keyboard-arrow-down"} 
                  size={24} 
                  color="#015185" 
                />
              </View>
            </TouchableOpacity>
            
            {riskSectionExpanded && (
              <View style={styles.dropdownContent}>
                <View style={styles.riskContainer}>
                  <Text style={styles.riskQuestion}>Has a fatigue related incident occurred?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.fatigueIncidentOccurred}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'fatigueIncidentOccurred', value)}
                    placeholder="Enter details"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>Has an employee (self-report/ by another person) been identified as at risk of fatigue?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.employeeAtRisk}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'employeeAtRisk', value)}
                    placeholder="Enter details"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>Is there a risk associated with an employee's functioning / behaviour?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.riskAssociated}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'riskAssociated', value)}
                    placeholder="Enter details"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>What is the employee's explanation for the behaviour you have observed?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.employeeExplanation}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'employeeExplanation', value)}
                    placeholder="Enter explanation"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>How many hours since the employee last slept?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.hoursSinceLastSlept}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'hoursSinceLastSlept', value)}
                    placeholder="Enter hours"
                    keyboardType="numeric"
                  />

                  <Text style={styles.riskQuestion}>How do they usually prevent fatigue?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.howPreventFatigue}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'howPreventFatigue', value)}
                    placeholder="Enter details"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>When did they last have a break?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.lastBreak}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'lastBreak', value)}
                    placeholder="Enter time"
                  />

                  <Text style={styles.riskQuestion}>Is there a reason/s for not enough sleep or poor sleep (stress, relationship, children, substance use etc)?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.reasonsForPoorSleep}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'reasonsForPoorSleep', value)}
                    placeholder="Enter reasons"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>What tasks have they been performing?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.tasksPerforming}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'tasksPerforming', value)}
                    placeholder="Enter tasks"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>Are those tasks 'high-risk' for fatigue (e.g. repetitious or entail not constant activity)?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.highRiskTasks}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'highRiskTasks', value)}
                    placeholder="Enter details"
                    multiline
                  />

                  <Text style={styles.riskQuestion}>If yes, how long have they been working on that task?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.howLongOnTask}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'howLongOnTask', value)}
                    placeholder="Enter duration"
                  />

                  <Text style={styles.riskQuestion}>When did they last drink water or eat something (dehydrated or hungry)?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.lastWaterFood}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'lastWaterFood', value)}
                    placeholder="Enter time"
                  />

                  <Text style={styles.riskQuestion}>How long was the break?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.breakDuration}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'breakDuration', value)}
                    placeholder="Enter duration"
                  />

                  <Text style={styles.riskQuestion}>When did they last drink water or eat something (dehydrated or hungry)?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.lastWaterFood2}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'lastWaterFood2', value)}
                    placeholder="Enter time"
                  />

                  <Text style={styles.riskQuestion}>How do they usually prevent fatigue?</Text>
                  <TextInput
                    style={styles.riskInput}
                    value={formData.riskAssessment.howPreventFatigue2}
                    onChangeText={(value) => handleNestedChange('riskAssessment', 'howPreventFatigue2', value)}
                    placeholder="Enter details"
                    multiline
                  />
                </View>
              </View>
            )}
    </View>

          {/* Opinion Section */}
          <View style={styles.section}>
            <Text style={styles.sectionTitle}>Opinion regarding level of fatigue:</Text>
            {renderCheckbox(
              'Unsafe to work',
              formData.fatigueLevel.unsafeToWork,
              () => {
                const updatedData = {
                  ...formData,
                  fatigueLevel: {
                    unsafeToWork: !formData.fatigueLevel.unsafeToWork,
                    safeToWork: false,
                    restrictedDuties: false,
                  },
                };
                setFormData(updatedData);
                saveData(updatedData);
              }
            )}
            {renderCheckbox(
              'Safe to work',
              formData.fatigueLevel.safeToWork,
              () => {
                const updatedData = {
                  ...formData,
                  fatigueLevel: {
                    unsafeToWork: false,
                    safeToWork: !formData.fatigueLevel.safeToWork,
                    restrictedDuties: false,
                  },
                };
                setFormData(updatedData);
                saveData(updatedData);
              }
            )}
            {renderCheckbox(
              'Restricted duties',
              formData.fatigueLevel.restrictedDuties,
              () => {
                const updatedData = {
                  ...formData,
                  fatigueLevel: {
                    unsafeToWork: false,
                    safeToWork: false,
                    restrictedDuties: !formData.fatigueLevel.restrictedDuties,
                  },
                };
                setFormData(updatedData);
                saveData(updatedData);
              }
            )}
          </View>
        </View>
      </ScrollView>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display="default"
          onChange={handleDateChange}
        />
      )}

      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#f8f9fa',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 10,
  },
  headerSubtitle: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  formContainer: {
    padding: 20,
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 15,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  questionText: {
    fontSize: 16,
    color: '#333',
    marginBottom: 15,
    fontStyle: 'italic',
  },
  categoryTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginTop: 15,
    marginBottom: 10,
  },
  inputContainer: {
    marginBottom: 20,
  },
  label: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  required: {
    color: '#dc3545',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
  },
  textArea: {
    minHeight: 80,
    textAlignVertical: 'top',
  },
  rowContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  halfWidth: {
    width: '48%',
  },
  shiftToggle: {
    flexDirection: 'row',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    overflow: 'hidden',
  },
  shiftButton: {
    flex: 1,
    paddingVertical: 12,
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  shiftButtonActive: {
    backgroundColor: '#015185',
  },
  shiftButtonText: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  shiftButtonTextActive: {
    color: '#fff',
  },
  checkboxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 10,
  },
  checkbox: {
    width: 20,
    height: 20,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: 10,
    backgroundColor: '#fff',
  },
  checkboxChecked: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  checkboxLabel: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  checkboxWithInputContainer: {
    marginBottom: 15,
  },
  otherInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    fontSize: 14,
    backgroundColor: '#f9f9f9',
    color: '#333',
    marginTop: 5,
    marginLeft: 30,
  },
  radioContainer: {
    marginBottom: 20,
  },
  radioGroup: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
    padding: 5,
  },
  radioOption: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  radioButton: {
    width: 20,
    height: 20,
    borderRadius: 10,
    borderWidth: 2,
    borderColor: '#015185',
    marginRight: 8,
    justifyContent: 'center',
    alignItems: 'center',
  },
  radioButtonSelected: {
    backgroundColor: '#015185',
  },
  radioButtonInner: {
    width: 8,
    height: 8,
    borderRadius: 4,
    backgroundColor: '#fff',
  },
  radioLabel: {
    fontSize: 16,
    color: '#333',
  },
  riskContainer: {
    marginTop: 10,
  },
  riskQuestion: {
    fontSize: 16,
    color: '#333',
    marginBottom: 8,
    fontWeight: '600',
    marginTop: 15,
  },
  riskInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    marginBottom: 15,
    minHeight: 50,
    width: '100%',
  },
  dropdownHeader: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 15,
    paddingVertical: 12,
    marginBottom: 5,
  },
  dropdownTitleContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  dropdownTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
  },
  dropdownContent: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 15,
    borderWidth: 1,
    borderColor: '#e9ecef',
    marginTop: 5,
    borderTopLeftRadius: 0,
    borderTopRightRadius: 0,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  dateInput: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    marginTop: 5,
  },
  dateInputText: {
    flex: 1,
    marginRight: 10,
  },
  datePlaceholderText: {
    color: '#999',
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: width * 0.08,
    height: width * 0.08,
    borderRadius: width * 0.04,
    borderWidth: 2,
    borderColor: '#015185',
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  clearSignatureButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginTop: 10,
    paddingVertical: 8,
    paddingHorizontal: 15,
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  clearSignatureText: {
    color: '#dc3545',
    fontSize: 14,
    marginLeft: 5,
  },
});

export default HSE_158Step2;