import React, { useEffect, useState, useRef } from 'react';
import { View, Text, TouchableOpacity, StyleSheet, Platform, PermissionsAndroid, Modal, Animated, Easing, TextInput } from 'react-native';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';

/*
  SpeechToText component using @appcitor/react-native-voice-to-text

  Installation notes:
  - yarn add @appcitor/react-native-voice-to-text
  - or: npm install @appcitor/react-native-voice-to-text --save
  - iOS: cd ios && pod install
  - Android: Ensure <uses-permission android:name="android.permission.RECORD_AUDIO"/> in AndroidManifest.xml
  - Test on a real device to avoid simulator-related audio issues
*/

const SpeechToText = () => {
  const [isListening, setIsListening] = useState(false);
  const [transcript, setTranscript] = useState('');
  const [error, setError] = useState(null);
  const [micPermissionGranted, setMicPermissionGranted] = useState(false);
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [ignoreVoiceEvents, setIgnoreVoiceEvents] = useState(false);
  const pulse = useRef(new Animated.Value(1)).current;

  useEffect(() => {
    // Check if chosen native module is available
    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      const msg = 'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.';
      console.error(msg);
      setError(msg);
      return;
    }

    setIsVoiceModuleAvailable(true);

    // attach listeners
    const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, (event) => onSpeechResults(event));
    const partialListener = VoiceToText.addEventListener(VoiceToTextEvents.PARTIAL_RESULTS, (event) => setPartialText(event?.value || ''));
    const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, (event) => setVolume(event?.value || 0));
    const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, (event) => onSpeechError(event));
    const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, () => {
      setIsListening(true);
      setModalVisible(true);
      startPulse();
    });
    const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, () => {
      setIsListening(false);
      stopPulse();
    });

    // Log for debugging
    console.log('VoiceToText module initialized:', VoiceToText);

    // On iOS, check speech recognition availability
    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const avail = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(avail === true);
          if (!avail && Platform.OS === 'ios') {
            setError('Speech recognition not available on this device. Test on a real device and ensure permissions are set in Info.plist.');
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    // Check/request microphone permission on Android
    const checkPermissions = async () => {
      if (Platform.OS === 'android') {
        try {
          const granted = await checkMicrophonePermission();
          if (!granted) {
            const requested = await requestMicrophonePermission();
            if (!requested) {
              setError('Microphone permission denied. Please enable it in app settings.');
            }
          }
        } catch (err) {
          console.warn('Permission check error:', err);
          setError('Failed to check microphone permissions.');
        }
      }
    };

    checkPermissions();

    // Cleanup on unmount - remove only individual listeners and avoid calling
    // module-level destroy/removeAllListeners unguarded which can crash native code.
    return () => {
      try { resultsListener && typeof resultsListener.remove === 'function' && resultsListener.remove(); } catch (e) { /* ignore */ }
      try { partialListener && typeof partialListener.remove === 'function' && partialListener.remove(); } catch (e) { /* ignore */ }
      try { volumeListener && typeof volumeListener.remove === 'function' && volumeListener.remove(); } catch (e) { /* ignore */ }
      try { errorListener && typeof errorListener.remove === 'function' && errorListener.remove(); } catch (e) { /* ignore */ }
      try { startListener && typeof startListener.remove === 'function' && startListener.remove(); } catch (e) { /* ignore */ }
      try { endListener && typeof endListener.remove === 'function' && endListener.remove(); } catch (e) { /* ignore */ }
      // Do NOT call VoiceToText.destroy() here unguarded to prevent native crashes
    };
  }, []);

  const onSpeechResults = (e) => {
    // The event payload shape differs between libraries / platforms.
    // Handle arrays, strings and objects defensively to avoid "join is not a function".
    try {
      let payload = e?.value ?? e?.results ?? e?.text ?? e;

      if (Array.isArray(payload)) {
        if (payload.length) setTranscript(payload.join(' '));
        return;
      }

      if (typeof payload === 'string') {
        setTranscript(payload);
        return;
      }

      // If payload is an object with numeric keys or nested results, try to extract strings
      if (payload && typeof payload === 'object') {
        const extracted = Object.values(payload)
          .flatMap((v) => (Array.isArray(v) ? v : [v]))
          .filter((v) => typeof v === 'string');
        if (extracted.length) {
          setTranscript(extracted.join(' '));
          return;
        }
      }

      // Fallback to JSON stringify so user can see the payload
      setTranscript(String(payload ?? ''));
    } catch (err) {
      console.warn('onSpeechResults parse error', err, e);
    }
  };

  const onSpeechError = (e) => {
    const message = e?.error?.message || e?.message || JSON.stringify(e);
    setError(`Speech recognition error: ${message}`);
    setIsListening(false);
    setModalVisible(false);
  };

  const checkMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
      setMicPermissionGranted(granted);
      return granted;
    } catch (err) {
      console.warn('Permission check error:', err);
      return false;
    }
  };

  const requestMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
        {
          title: 'Microphone Permission',
          message: 'This app needs microphone access for speech recognition.',
          buttonPositive: 'OK',
        }
      );
      const isGranted = granted === PermissionsAndroid.RESULTS.GRANTED;
      setMicPermissionGranted(isGranted);
      return isGranted;
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const startRecording = async () => {
    if (!isVoiceModuleAvailable) {
      setError('Voice module is not available. Please check installation.');
      return;
    }

    setError(null);
    setTranscript('');

    if (Platform.OS === 'android' && !micPermissionGranted) {
      const granted = await requestMicrophonePermission();
      if (!granted) {
        setError('Microphone permission denied.');
        return;
      }
    }

    try {
      // on iOS ensure recognition available
      if (Platform.OS === 'ios' && !isRecognitionAvailable) {
        setError('Speech recognition not available on this iOS device or permission denied. Check Info.plist and device settings.');
        return;
      }

      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }

      // show modal immediately while starting
      setModalVisible(true);

      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
    } catch (e) {
      const message = e?.message || JSON.stringify(e);
      if (Platform.OS === 'ios' && message.includes('IsFormatSampleRateAndChannelCountValid')) {
        setError('iOS audio format error. Test on a real device and ensure Info.plist includes NSMicrophoneUsageDescription and NSSpeechRecognitionUsageDescription.');
      } else {
        setError(`Failed to start recording: ${message}`);
      }
      setIsListening(false);
    }
  };

  const stopRecording = async () => {
    // Finalize locally without calling native stop to avoid removeListeners crash.
    try {
      setIgnoreVoiceEvents(true);
      if (partialText && partialText.trim()) {
        setTranscript((prev) => (prev ? prev + ' ' + partialText.trim() : partialText.trim()));
      }
    } catch (e) {
      console.warn('Error finalizing local recognition:', e);
    } finally {
      setIsListening(false);
      setModalVisible(false);
      setPartialText('');
      // don't call native stopListening/destroy here
    }
  };

  const cancelRecording = async () => {
    if (!isVoiceModuleAvailable) return;
    try {
      if (typeof VoiceToText.cancelListening === 'function') {
        await VoiceToText.cancelListening();
      } else if (typeof VoiceToText.cancel === 'function') {
        await VoiceToText.cancel();
      } else {
        throw new Error('cancelListening not available on VoiceToText');
      }
    } catch (e) {
      setError(`Failed to cancel recording: ${e.message || JSON.stringify(e)}`);
    } finally {
      setIsListening(false);
      setModalVisible(false);
    }
  };

  const onDone = async () => {
    try {
      // copy partial text into main input and finalize locally
      if (partialText && partialText.trim()) {
        setTranscript((prev) => (prev ? prev + ' ' + partialText.trim() : partialText.trim()));
      }
      await stopRecording();
    } catch (err) {
      console.warn('onDone stop error', err);
    } finally {
      setModalVisible(false);
    }
  };

  const startPulse = () => {
    pulse.setValue(1);
    Animated.loop(
      Animated.sequence([
        Animated.timing(pulse, { toValue: 1.4, duration: 600, easing: Easing.inOut(Easing.ease), useNativeDriver: true }),
        Animated.timing(pulse, { toValue: 1, duration: 600, easing: Easing.inOut(Easing.ease), useNativeDriver: true }),
      ])
    ).start();
  };

  const stopPulse = () => {
    pulse.stopAnimation();
  };

  return (
    <View style={styles.container}>
      <Text style={styles.title}>Speech to Text</Text>
      <Text style={styles.status}>Status: {isListening ? 'Listening...' : 'Stopped'}</Text>
      <Text style={styles.transcript}>{transcript || 'Press Start and speak'}</Text>
      {error && <Text style={styles.error}>Error: {error}</Text>}

      <View style={styles.inputRow}>
        <TextInput
          style={styles.input}
          value={transcript}
          onChangeText={(t) => setTranscript(t)}
          placeholder="Transcribed text will appear here"
          multiline
        />
        <TouchableOpacity
          style={[styles.button, isListening && styles.disabledButton, { width: 90, marginLeft: 8 }]}
          onPress={startRecording}
          disabled={isListening || !isVoiceModuleAvailable}
        >
          <Text style={styles.buttonText}>Start</Text>
        </TouchableOpacity>
      </View>
      <Modal visible={modalVisible} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Animated.View style={[styles.micCircle, { transform: [{ scale: pulse }] }]}> 
              <Text style={styles.micIcon}>🎤</Text>
            </Animated.View>

            <Text style={styles.partial}>{partialText || 'Listening...'}</Text>

            <View style={styles.modalButtons}>
              <TouchableOpacity style={styles.modalButton} onPress={onDone}>
                <Text style={styles.modalButtonText}>Done</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.cancelBtn]} onPress={cancelRecording}>
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    padding: 12,
  },
  inputRow: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginTop: 12,
  },
  input: {
    flex: 1,
    minHeight: 80,
    padding: 10,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#fff',
  },
  title: {
    fontSize: 18,
    fontWeight: '600',
    marginBottom: 8,
  },
  status: {
    marginBottom: 8,
    color: '#333',
  },
  transcript: {
    minHeight: 60,
    padding: 8,
    backgroundColor: '#f5f5f5',
    borderRadius: 6,
    marginBottom: 8,
  },
  error: {
    color: 'red',
    marginBottom: 8,
  },
  buttonRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  button: {
    flex: 1,
    paddingVertical: 10,
    marginHorizontal: 4,
    backgroundColor: '#007AFF',
    borderRadius: 6,
    alignItems: 'center',
  },
  disabledButton: {
    backgroundColor: '#9CC1FF',
  },
  buttonText: {
    color: '#fff',
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
  },
  micCircle: {
    width: 120,
    height: 120,
    borderRadius: 60,
    backgroundColor: '#ffecec',
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: 12,
  },
  micIcon: {
    fontSize: 34,
  },
  partial: {
    minHeight: 40,
    textAlign: 'center',
    marginBottom: 12,
  },
  modalButtons: {
    flexDirection: 'row',
    width: '100%',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 10,
    marginHorizontal: 6,
    backgroundColor: '#007AFF',
    borderRadius: 8,
    alignItems: 'center',
  },
  cancelBtn: {
    backgroundColor: '#999',
  },
  modalButtonText: {
    color: '#fff',
    fontWeight: '600',
  },
});

export default SpeechToText;