import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  Image,
  Alert,
  Modal,
  ScrollView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';

const Checklist_home = () => {
  const screenWidth = Dimensions.get('window').width * 0.8; // 80% of screen width
  const navigation = useNavigation();
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    checklistPermissions: [],
  });
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    const checkAuthAndChecklists = async () => {
      try {
        // Check user authentication
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (!userEmail) {
          Alert.alert('Error', 'User not authenticated. Please log in again.');
          navigation.replace('AdminLoginScreen');
          return;
        }

        // Verify admin status and fetch user data
        const userSnapshot = await database()
          .ref('/Globalusers/admins')
          .orderByChild('email')
          .equalTo(userEmail)
          .once('value');
        const userDataRaw = userSnapshot.val();
        if (!userDataRaw) {
          Alert.alert('Error', 'User not found in admin database.');
          navigation.replace('AdminLoginScreen');
          return;
        }

        // Extract user data (first user found)
        const userId = Object.keys(userDataRaw)[0];
        const data = userDataRaw[userId];

        // Extract checklist permissions
        const checklistModule = data.modules?.find(mod => mod.module === 'checklist');
        const checklistPermissions = checklistModule?.permissions || [];

        setUserData({
          countries: data.countries || [],
          projects: data.projects || [],
          checklistPermissions: checklistPermissions,
        });

        // Check for local checklists
        const cachedChecklists = await AsyncStorage.getItem('checklists');
        if (!cachedChecklists) {
          // Fetch checklists from Firebase if not cached
          const snapshot = await database()
            .ref('AdminChecklistJson/GlobalChecklistJson')
            .once('value');
          const checklistData = snapshot.val();
          if (checklistData && checklistData.Checklistfile && checklistData.Checklistfile.documents) {
            await AsyncStorage.setItem('checklists', JSON.stringify(checklistData.Checklistfile.documents));
          } else {
            console.warn('No checklists found in Firebase.');
          }
        }
      } catch (error) {
        console.error('Error checking auth or checklists:', error);
        Alert.alert('Error', 'Failed to initialize. Please try again.');
      }
    };

    checkAuthAndChecklists();
  }, [navigation]);

  const scale = (size) => (Dimensions.get('window').width / 375) * size;

  return (
    <View style={styles.container}>
      {/* StatusBar with Gradient */}
      <StatusBar barStyle="light-content" backgroundColor="#015185" />

      {/* Gradient Header */}
      <LinearGradient colors={['#015185', '#3481BC']} style={styles.header}>
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.headerLeft}
            onPress={() => navigation.goBack()}
          >
            <MaterialIcons name="chevron-left" size={scale(28)} color="#fff" />
            <Text style={styles.headerText}>Checklist</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.headerRight}
            onPress={() => setPermissionsModalVisible(true)}
          >
            <MaterialIcons name="lock" size={scale(24)} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Buttons Container */}
      <View style={styles.buttonContainer}>
        {/* Button 1: Create Custom Checklist */}
        {userData.checklistPermissions.includes('create') ? (
          <TouchableOpacity
            onPress={() => navigation.navigate("Create_Custom_c")}
            activeOpacity={0.85}
            style={styles.cardButton}
          >
            <LinearGradient 
              colors={["#015185", "#3481BC"]} 
              style={styles.gradientButton}
              start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
            >
              <View style={styles.buttonContent}>
                <Image
                  source={require('../../../Images/adminhome/checklist/solar_list-check-bold.png')}
                  style={styles.logomenu}
                />
                <Text style={styles.buttonText}>Create Custom Checklist</Text>
              </View>
            </LinearGradient>
          </TouchableOpacity>
        ) : (
          <View style={[styles.cardButton, { position: 'relative' }]}> 
            <LinearGradient 
              colors={["#015185", "#3481BC"]} 
              style={styles.gradientButton}
              start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
            >
              <View style={styles.buttonContent}>
                <Image
                  source={require('../../../Images/adminhome/checklist/solar_list-check-bold.png')}
                  style={styles.logomenu}
                />
                <Text style={styles.buttonText}>Create Custom Checklist</Text>
              </View>
              {/* Overlay */}
              <View style={styles.disabledOverlay}>
                <MaterialIcons name="lock" size={40} color="#fff" style={{ opacity: 0.9 }} />
              </View>
            </LinearGradient>
            <Text style={styles.disabledTooltip}>You don’t have permission to create checklists.</Text>
          </View>
        )}

        {/* Button 2: Checklist Library */}
        <TouchableOpacity
          onPress={() => navigation.navigate("Checklist_library")}
          activeOpacity={0.85}
          style={styles.cardButton}
        >
          <LinearGradient 
            colors={["#015185", "#3481BC"]} 
            style={styles.gradientButton}
            start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
          >
            <View style={styles.buttonContent}>
              <Image
                source={require('../../../Images/adminhome/checklist/tabler_copy-check.png')}
                style={styles.logomenu}
              />
              <Text style={styles.buttonText}>Checklist Library</Text>
            </View>
          </LinearGradient>
        </TouchableOpacity>

        {/* Button 3: Database Upload/Download */}
        {/* <TouchableOpacity
          onPress={() => navigation.navigate("ChecklistUploadNavigation")}
          activeOpacity={0.85}
          style={styles.cardButton}
        >
          <LinearGradient 
            colors={["#015185", "#3481BC"]} 
            style={styles.gradientButton}
            start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
          >
            <View style={styles.buttonContent}>
              <MaterialIcons name="cloud-sync" size={44} color="#fff" />
              <Text style={styles.buttonText}>Database Sync</Text>
            </View>
          </LinearGradient>
        </TouchableOpacity> */}

        {/* Button 4: Downloaded Checklist (always disabled, modern overlay) */}
        <View style={[styles.cardButton, { position: 'relative' }]}> 
          <LinearGradient 
            colors={["#015185", "#3481BC"]} 
            style={styles.gradientButton}
            start={{ x: 0.5, y: 0 }} end={{ x: 0.5, y: 1 }}
          >
            <View style={styles.buttonContent}>
              <Image
                source={require('../../../Images/adminhome/checklist/subway_download-2.png')}
                style={styles.logomenu}
              />
              <Text style={styles.buttonText}>Downloaded Checklist</Text>
            </View>
            {/* Overlay */}
            <View style={styles.disabledOverlay}>
              <View style={styles.comingSoonBadge}>
                <Text style={styles.comingSoonText}>Coming Soon</Text>
              </View>
            </View>
          </LinearGradient>
          <Text style={styles.disabledTooltip}>This feature is coming soon!</Text>
        </View>
      </View>

      {/* Permissions Modal */}
      <Modal
        animationType="slide"
        transparent={true}
        visible={permissionsModalVisible}
        onRequestClose={() => setPermissionsModalVisible(false)}
      >
        <View style={styles.modernModalOverlay}>
          <View style={styles.modernModalContent}>
            <Text style={styles.modernModalTitle}>User Permissions</Text>
            <ScrollView contentContainerStyle={{ paddingBottom: 16 }} showsVerticalScrollIndicator={false}>
              {/* Countries */}
              <View style={styles.section}>
                <Text style={styles.sectionTitle}>Countries</Text>
                <View style={styles.sectionDivider} />
                <View style={styles.chipRow}>
                  {userData.countries.length > 0 ? (
                    userData.countries.map((country, index) => (
                      <View key={index} style={styles.chip}>
                        <MaterialIcons name="flag" size={18} color="#3481BC" style={{ marginRight: 6 }} />
                        <Text style={styles.chipText}>{country}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.chipEmpty}>No countries assigned</Text>
                  )}
                </View>
              </View>

              {/* Projects */}
              <View style={styles.section}>
                <Text style={styles.sectionTitle}>Projects</Text>
                <View style={styles.sectionDivider} />
                <View style={styles.chipRow}>
                  {userData.projects.length > 0 ? (
                    userData.projects.map((project, index) => (
                      <View key={index} style={styles.chip}>
                        <MaterialIcons name="work" size={18} color="#3481BC" style={{ marginRight: 6 }} />
                        <Text style={styles.chipText}>{project}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.chipEmpty}>No projects assigned</Text>
                  )}
                </View>
              </View>

              {/* Checklist Permissions */}
              <View style={styles.section}>
                <Text style={styles.sectionTitle}>Checklist Permissions</Text>
                <View style={styles.sectionDivider} />
                <View style={styles.chipRow}>
                  {userData.checklistPermissions.length > 0 ? (
                    userData.checklistPermissions.map((permission, index) => (
                      <View key={index} style={styles.chip}>
                        <MaterialIcons name="verified-user" size={18} color="#3481BC" style={{ marginRight: 6 }} />
                        <Text style={styles.chipText}>{permission}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.chipEmpty}>No checklist permissions assigned</Text>
                  )}
                </View>
              </View>
            </ScrollView>
            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setPermissionsModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    height: 100,
    justifyContent: 'center',
    alignItems: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
    paddingHorizontal: 10,
    top: 20,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    padding: 10,
  },
  headerText: {
    color: '#fff',
    fontSize: 20,
    fontWeight: '500',
  },
  buttonContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  gradientButton: {
    borderRadius: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    paddingVertical: 0,
  },
  buttonContent: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 28,
  },
  buttonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: 10,
    letterSpacing: 0.2,
  },
  logomenu: {
    width: 44,
    height: 44,
    resizeMode: 'contain',
    marginBottom: 0,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 20,
  },
  modalScroll: {
    maxHeight: Dimensions.get('window').height * 0.5,
    width: '100%',
  },
  modalSection: {
    marginBottom: 20,
  },
  modalSectionTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 6,
    marginRight: 8,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: 12,
    color: '#015185',
    fontWeight: '500',
  },
  noItemsText: {
    fontSize: 14,
    color: '#888',
    marginTop: 8,
  },
  modalCloseButton: {
    marginTop: 20,
    paddingVertical: 10,
    paddingHorizontal: 20,
    backgroundColor: '#ff4444',
    borderRadius: 5,
  },
  modalCloseText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  cardButton: {
    width: Dimensions.get('window').width * 0.8,
    marginBottom: 28,
    borderRadius: 16,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    overflow: 'visible',
    alignSelf: 'center',
  },
  disabledOverlay: {
    ...StyleSheet.absoluteFillObject,
    backgroundColor: 'rgba(0,0,0,0.35)',
    justifyContent: 'center',
    alignItems: 'center',
    borderRadius: 16,
    zIndex: 2,
  },
  disabledTooltip: {
    color: '#888',
    fontSize: 13,
    textAlign: 'center',
    marginTop: 6,
    marginBottom: 8,
    fontWeight: '500',
  },
  comingSoonBadge: {
    backgroundColor: '#fff',
    borderRadius: 12,
    paddingHorizontal: 16,
    paddingVertical: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.10,
    shadowRadius: 2,
    elevation: 2,
  },
  comingSoonText: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 15,
    letterSpacing: 0.5,
  },
  modernModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.25)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modernModalContent: {
    width: '90%',
    maxWidth: 400,
    backgroundColor: '#fff',
    borderRadius: 22,
    padding: 24,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.18,
    shadowRadius: 12,
    elevation: 8,
    alignItems: 'center',
  },
  modernModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#222',
    marginBottom: 18,
    textAlign: 'center',
  },
  section: {
    width: '100%',
    marginBottom: 24,
  },
  sectionTitle: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 17,
    marginBottom: 4,
  },
  sectionDivider: {
    height: 2,
    backgroundColor: '#e0e6ed',
    marginBottom: 12,
    borderRadius: 1,
  },
  chipRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    alignItems: 'center',
    marginBottom: 0,
  },
  chip: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f4f8fb',
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#3481BC',
    paddingHorizontal: 14,
    paddingVertical: 8,
    margin: 6,
  },
  chipText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
  },
  chipEmpty: {
    fontSize: 14,
    color: '#bbb',
    margin: 6,
    fontStyle: 'italic',
  },
  closeButton: {
    backgroundColor: '#ff4444',
    borderRadius: 24,
    paddingHorizontal: 32,
    paddingVertical: 12,
    marginTop: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 4,
  },
  closeButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    textAlign: 'center',
    letterSpacing: 0.5,
  },
});

export default Checklist_home;