import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  TextInput,
  Modal,
  FlatList,
  StyleSheet,
  Dimensions,
  Animated,
  Image,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import Toast from 'react-native-toast-message';
import Add_inspection_item from './Add_inspection_item';

const { width } = Dimensions.get('window');

const Do_inspection_checklist = ({ checklistData, setChecklistData, setActiveStep, setAddItemParams, addItemParams }) => {
  const navigation = useNavigation();
  const [items, setItems] = useState(checklistData.items || []);
  const [modalVisible, setModalVisible] = useState(false);
  const [editModalVisible, setEditModalVisible] = useState(false);
  const [newItem, setNewItem] = useState('');
  const [editItemIndex, setEditItemIndex] = useState(null);
  const [editItemText, setEditItemText] = useState('');
  const [isSaving, setIsSaving] = useState(false);
  const [expandedSections, setExpandedSections] = useState([]);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    setItems(checklistData.items || []);
    const expanded = (checklistData.items || []).map(item => (item.subItems && item.subItems.length > 0));
    setExpandedSections(expanded);
  }, [checklistData.items]);

  const toggleSection = (index) => {
    setExpandedSections((prev) =>
      prev.map((expanded, i) => (i === index ? !expanded : expanded))
    );
  };

  const saveItem = async (itemText) => {
    if (itemText.trim()) {
      const newItems = [...items, { text: itemText, subItems: [] }];
      setItems(newItems);
      setChecklistData({ ...checklistData, items: newItems });
      setExpandedSections([...expandedSections, false]);
      setNewItem('');
      setModalVisible(false);
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Section name cannot be empty.',
      });
    }
  };

  const updateItem = async (index, newText) => {
    if (newText.trim()) {
      const newItems = [...items];
      newItems[index].text = newText;
      setItems(newItems);
      setChecklistData({ ...checklistData, items: newItems });
      setEditItemText('');
      setEditModalVisible(false);
      setEditItemIndex(null);
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Section name cannot be empty.',
      });
    }
  };

  const deleteItem = async (index) => {
    const newItems = items.filter((_, i) => i !== index);
    setItems(newItems);
    setChecklistData({ ...checklistData, items: newItems });
    setExpandedSections(expandedSections.filter((_, i) => i !== index));
    setEditModalVisible(false);
    setEditItemIndex(null);
  };

  const saveSubItem = async (parentIndex, subItem) => {
    const newItems = [...items];
    if (!newItems[parentIndex].subItems) {
      newItems[parentIndex].subItems = [];
    }
    newItems[parentIndex].subItems.push(subItem);
    setItems(newItems);
    setChecklistData({ ...checklistData, items: newItems });
  };

  const updateSubItem = async (parentIndex, subItemIndex, updatedSubItem) => {
    const newItems = [...items];
    newItems[parentIndex].subItems[subItemIndex] = updatedSubItem;
    setItems(newItems);
    setChecklistData({ ...checklistData, items: newItems });
  };

  const deleteSubItem = async (parentIndex, subItemIndex) => {
    const newItems = [...items];
    newItems[parentIndex].subItems = newItems[parentIndex].subItems.filter(
      (_, i) => i !== subItemIndex
    );
    setItems(newItems);
    setChecklistData({ ...checklistData, items: newItems });
  };

  const handleSave = async () => {
    if (items.length === 0) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'At least one section is required.',
      });
      return;
    }

    const hasSubItem = items.some(item => (item.subItems || []).length > 0);
    if (!hasSubItem) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'At least one inspection item is required in any section.',
      });
      return;
    }

    setIsSaving(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      await setChecklistData({ ...checklistData, items });
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Checklist saved successfully.',
      });
      navigation.goBack();
    } catch (error) {
      console.error('Error saving checklist:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save checklist.',
      });
    } finally {
      setIsSaving(false);
      rotateAnim.setValue(0);
    }
  };

  const handleAddSubItem = (index) => {
    setAddItemParams({
      parentIndex: index,
      initialData: {
        itemName: `Inspection Item ${(items[index].subItems || []).length + 1}`,
        description: '',
        inputType: 'Ok/Reject/N/A',
        instruction: '',
        sortOrder: (items[index].subItems || []).length + 1,
        isInspectionRequired: false,
      },
      onSave: (subItem) => saveSubItem(index, subItem),
      onCancel: () => setAddItemParams(null),
    });
  };

  const handleEditSubItem = (index, subItem, subItemIndex) => {
    setAddItemParams({
      parentIndex: index,
      initialData: subItem,
      subItemIndex: subItemIndex,
      onSave: (updatedSubItem) => updateSubItem(index, subItemIndex, updatedSubItem),
      onCancel: () => setAddItemParams(null),
    });
  };

  return (
    <View style={styles.container}>
      <Modal
        transparent={true}
        animationType="fade"
        visible={isSaving}
        onRequestClose={() => {}}
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Saving Checklist Data...</Text>
            </View>
          </View>
        </View>
      </Modal>

      {addItemParams && (
        <Modal
          animationType="slide"
          transparent={true}
          visible={true}
          onRequestClose={() => setAddItemParams(null)}
        >
          <Add_inspection_item
            parentIndex={addItemParams.parentIndex}
            initialData={addItemParams.initialData}
            subItemIndex={addItemParams.subItemIndex}
            onSave={addItemParams.onSave}
            onCancel={addItemParams.onCancel}
          />
        </Modal>
      )}

      <View style={styles.instructionRow}>
        <Text style={styles.previewInstruction}>Tap on list to preview items</Text>
        <LinearGradient
          colors={['#3481BC', '#015185']}
          style={styles.addInspectionItemButtonGradient}
        >
          <TouchableOpacity
            style={styles.addInspectionItemButton}
            onPress={() => setModalVisible(true)}
          >
            <Text style={styles.addInspectionItemButtonText}>+ Add Section</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <FlatList
        data={items}
        keyExtractor={(item, index) => index.toString()}
        renderItem={({ item, index }) => (
          <View style={styles.sectionContainer}>
            <View style={styles.itemRow}>
              <TouchableOpacity style={styles.dropdownContainer} onPress={() => toggleSection(index)}>
                <MaterialIcons
                  name={expandedSections[index] ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={20}
                  color="#0078D4"
                  style={styles.dropdownIcon}
                />
              </TouchableOpacity>
              <Text style={styles.itemText}>{item.text}</Text>
              <View style={styles.itemActions}>
                <TouchableOpacity
                  style={styles.editButton}
                  onPress={() => {
                    setEditItemIndex(index);
                    setEditItemText(item.text);
                    setEditModalVisible(true);
                  }}
                >
                  <MaterialIcons name="edit" size={20} color="#0078D4" />
                </TouchableOpacity>
              </View>
            </View>
            {expandedSections[index] && (item.subItems || []).length > 0 && (
              <FlatList
                data={item.subItems}
                keyExtractor={(subItem, subIndex) => `${index}-${subIndex}`}
                renderItem={({ item: subItem, index: subIndex }) => (
                  <View style={styles.subItemContainer}>
                    <View style={styles.subItemRow}>
                      <Text style={styles.subItemText}>
                        {subItem.itemName || 'Unnamed Inspection Item'}
                      </Text>
                      <View style={styles.subItemActions}>
                        <TouchableOpacity
                          onPress={() => handleEditSubItem(index, subItem, subIndex)}
                          style={styles.editButton}
                        >
                          <MaterialIcons name="edit" size={20} color="#0078D4" />
                        </TouchableOpacity>
                        <TouchableOpacity
                          onPress={() => deleteSubItem(index, subIndex)}
                          style={styles.deleteButton}
                        >
                          <MaterialIcons name="delete" size={20} color="#dc3545" />
                        </TouchableOpacity>
                      </View>
                    </View>
                  </View>
                )}
                style={styles.subList}
              />
            )}
            <LinearGradient
              colors={['#3481BC', '#025383']}
              style={[styles.buttonGradient, styles.addSubItemGradient]}
            >
              <TouchableOpacity
                style={styles.addSubItemButton}
                onPress={() => handleAddSubItem(index)}
              >
                <Text style={styles.addSubItemText}>Add Inspection Item</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        )}
        style={styles.list}
        ListEmptyComponent={
          <Text style={styles.emptyText}>No sections available. Add a new section to start.</Text>
        }
      />

      <View style={styles.buttonContainer}>
        <TouchableOpacity
          style={[styles.footerButton, { justifyContent: 'flex-start' }]}
          onPress={() => setActiveStep(1)}
        >
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.footerButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.footerButton, { justifyContent: 'flex-end' }]}
          onPress={handleSave}
          disabled={isSaving}
        >
          <Text style={styles.footerButtonText}>Save Checklist</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <Modal
        animationType="slide"
        transparent={true}
        visible={modalVisible}
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={[styles.sectionModalTitle, { textAlign: 'center', alignSelf: 'center' }]}>Section Name</Text>
            </View>
            <TextInput
              style={styles.sectionNameInput}
              value={newItem}
              onChangeText={setNewItem}
              underlineColorAndroid="transparent"
            />
            <View style={styles.sectionModalDivider} />
            <View style={styles.sectionModalButtonRow}>
              <LinearGradient
                colors={['#3481BC', '#015185']}
                style={styles.sectionModalButtonGradient}
              >
                <TouchableOpacity
                  style={styles.sectionModalButton}
                  onPress={() => setModalVisible(false)}
                >
                  <Text style={styles.sectionModalButtonText}>Cancel</Text>
                </TouchableOpacity>
              </LinearGradient>
              <LinearGradient
                colors={['#3481BC', '#015185']}
                style={styles.sectionModalButtonGradient}
              >
                <TouchableOpacity
                  style={styles.sectionModalButton}
                  onPress={() => saveItem(newItem)}
                >
                  <Text style={styles.sectionModalButtonText}>Save</Text>
                </TouchableOpacity>
              </LinearGradient>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        animationType="slide"
        transparent={true}
        visible={editModalVisible}
        onRequestClose={() => setEditModalVisible(false)}
      >
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.editSectionTitle}>Edit Section</Text>
            </View>
            <TextInput
              style={styles.editSectionInput}
              value={editItemText}
              onChangeText={setEditItemText}
              underlineColorAndroid="transparent"
            />
            <View style={styles.sectionModalDivider} />
            <View style={styles.editSectionButtonRow}>
              <LinearGradient
                colors={['#3481BC', '#015185']}
                style={styles.editSectionButtonGradient}
              >
                <TouchableOpacity
                  style={styles.editSectionButton}
                  onPress={() => deleteItem(editItemIndex)}
                >
                  <Text style={styles.editSectionButtonText}>Delete</Text>
                </TouchableOpacity>
              </LinearGradient>
              <LinearGradient
                colors={['#3481BC', '#015185']}
                style={styles.editSectionButtonGradient}
              >
                <TouchableOpacity
                  style={styles.editSectionButton}
                  onPress={() => setEditModalVisible(false)}
                >
                  <Text style={styles.editSectionButtonText}>Cancel</Text>
                </TouchableOpacity>
              </LinearGradient>
              <LinearGradient
                colors={['#3481BC', '#015185']}
                style={styles.editSectionButtonGradient}
              >
                <TouchableOpacity
                  style={styles.editSectionButton}
                  onPress={() => updateItem(editItemIndex, editItemText)}
                >
                  <Text style={styles.editSectionButtonText}>Update</Text>
                </TouchableOpacity>
              </LinearGradient>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    padding: width * 0.05,
    backgroundColor: '#f8f9fa',
  },
  instructionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginVertical: width * 0.025,
  },
  previewInstruction: {
    fontSize: width * 0.05,
    color: '#015185',
    fontWeight: 'bold',
  },
  addInspectionItemButtonGradient: {
    borderRadius: 4,
    paddingVertical: 4,
    paddingHorizontal: 10,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    height: 28,
  },
  addInspectionItemButton: {
    width: '100%',
    alignItems: 'center',
    justifyContent: 'center',
  },
  addInspectionItemButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: width * 0.032,
  },
  buttonGradient: {
    borderRadius: 5,
    overflow: 'hidden',
  },
  addSubItemGradient: {
    width: width * 0.4,
    alignSelf: 'center',
    marginVertical: width * 0.025,
  },
  list: {
    flex: 1,
  },
  sectionContainer: {
    marginBottom: width * 0.04,
    backgroundColor: '#fff',
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  itemRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    backgroundColor: '#F5F5F5',
    borderRadius: 6,
    margin: 4,
    minHeight: 44,
  },
  dropdownContainer: {
    justifyContent: 'center',
    alignItems: 'center',
  },
  dropdownIcon: {
    marginRight: width * 0.02,
  },
  itemText: {
    fontSize: width * 0.045,
    color: '#333',
    fontWeight: '600',
    flex: 1,
  },
  itemActions: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  editButton: {
    padding: width * 0.012,
  },
  addSubItemButton: {
    padding: width * 0.025,
    borderRadius: 5,
    alignItems: 'center',
    width: width * 0.4,
  },
  addSubItemText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '600',
    textAlign: 'center',
  },
  subList: {
    marginLeft: width * 0.05,
    marginVertical: width * 0.012,
  },
  subItemContainer: {
    paddingVertical: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  subItemRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  subItemText: {
    fontSize: width * 0.035,
    color: '#555',
    flex: 1,
  },
  subItemActions: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  deleteButton: {
    marginLeft: width * 0.025,
    marginRight: width * 0.05,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.03,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  footerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  footerButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  modalContent: {
    width: '85%',
    backgroundColor: '#fff',
    padding: width * 0.05,
    borderRadius: 12,
    elevation: 5,
  },
  modalHeader: {
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: width * 0.05,
  },
  sectionModalTitle: {
    fontSize: width * 0.06,
    color: '#1565a5',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 8,
  },
  sectionNameInput: {
    fontSize: width * 0.045,
    color: '#222',
    textAlign: 'center',
    borderWidth: 0,
    borderBottomWidth: 2,
    borderColor: '#e0e0e0',
    backgroundColor: 'transparent',
    marginBottom: 0,
    marginTop: 0,
    paddingVertical: 4,
    paddingHorizontal: 0,
    width: '80%',
    alignSelf: 'center',
  },
  sectionModalDivider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginBottom: 18,
    marginTop: 0,
    width: '90%',
    alignSelf: 'center',
  },
  sectionModalButtonRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    marginTop: 18,
    gap: 10,
  },
  sectionModalButtonGradient: {
    borderRadius: 4,
    marginHorizontal: 4,
    minWidth: 70,
  },
  sectionModalButton: {
    paddingVertical: 4,
    paddingHorizontal: 16,
    alignItems: 'center',
    borderRadius: 4,
  },
  sectionModalButtonText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.035,
    textAlign: 'center',
  },
  editSectionTitle: {
    fontSize: width * 0.06,
    color: '#1565a5',
    fontWeight: '700',
    textAlign: 'left',
    marginBottom: 8,
    marginLeft: 4,
  },
  editSectionInput: {
    fontSize: width * 0.045,
    color: '#222',
    textAlign: 'left',
    borderWidth: 0,
    borderBottomWidth: 2,
    borderColor: '#e0e0e0',
    backgroundColor: 'transparent',
    marginBottom: 0,
    marginTop: 0,
    paddingVertical: 4,
    paddingHorizontal: 0,
    marginLeft: 4,
    width: '90%',
  },
  editSectionButtonRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    marginTop: 18,
    gap: 10,
  },
  editSectionButtonGradient: {
    borderRadius: 4,
    marginHorizontal: 4,
    minWidth: 90,
  },
  editSectionButton: {
    paddingVertical: 6,
    paddingHorizontal: 18,
    alignItems: 'center',
    borderRadius: 4,
  },
  editSectionButtonText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.035,
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
});

export default Do_inspection_checklist;