import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  ScrollView,
  FlatList,
  Modal,
  Image,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const TradckDailyreport = () => {
  const navigation = useNavigation();
  const [isLoading, setIsLoading] = useState(true);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [allInspectionHistory, setAllInspectionHistory] = useState([]);
  const [isOnline, setIsOnline] = useState(true);
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [checklistModalVisible, setChecklistModalVisible] = useState(false);
  const [allChecklists, setAllChecklists] = useState([]);
  const [vehicleStats, setVehicleStats] = useState({});
  const [selectedVehicle, setSelectedVehicle] = useState(null);
  const [vehicleModalVisible, setVehicleModalVisible] = useState(false);
  const [allVehicles, setAllVehicles] = useState([]);
  const [showAllVehicles, setShowAllVehicles] = useState(true);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState(null);
  const [isLoadingPermissions, setIsLoadingPermissions] = useState(false);

  // Load all inspection data
  useEffect(() => {
    loadAllInspectionData();
  }, []);

  // Load inspection history when date or checklist changes
  useEffect(() => {
    loadInspectionHistory();
  }, [selectedDate, selectedChecklist, selectedVehicle]);

  const loadAllInspectionData = async () => {
    try {
      setIsLoading(true);
      // Fetch all reports from all users
      let allReports = [];
      try {
        const adminsRef = database().ref('/GlobalDailyreportdata/admins');
        const snapshot = await adminsRef.once('value');
        const adminsData = snapshot.val() || {};
        Object.values(adminsData).forEach((userReports) => {
          if (userReports && typeof userReports === 'object') {
            allReports.push(...Object.values(userReports));
          }
        });
      } catch (error) {
        console.log('Loading reports from local storage');
        const storedReports = await AsyncStorage.getItem('dailyInspections');
        if (storedReports) {
          const localReports = JSON.parse(storedReports);
          allReports = localReports;
        }
      }

      // Extract all unique checklists
      const checklistSet = new Set();
      const vehicleStatsMap = new Map();
      const vehicleSet = new Set();

      allReports.forEach((report) => {
        if (report.step1.checklistTitle) {
          checklistSet.add(report.step1.checklistTitle);
        }

        // Track vehicle statistics
        const equipment = report.step1.selectedEquipment;
        if (equipment && equipment.id) {
          if (!vehicleStatsMap.has(equipment.id)) {
            vehicleStatsMap.set(equipment.id, {
              equipmentName: equipment.equipmentName || 'N/A',
              mainCategory: equipment.mainCategory || 'N/A',
              model: equipment.model || 'N/A',
              status: equipment.status || 'Available',
              inspectionCount: 0,
              completedCount: 0,
              lastInspection: null,
            });
          }
          const stats = vehicleStatsMap.get(equipment.id);
          stats.inspectionCount++;
          if (report.step3?.pdfDownloadUrl) {
            stats.completedCount++;
          }
          
          const inspectionDate = new Date(report.step1.inspectionDate || report.step3?.createdAt);
          if (!stats.lastInspection || inspectionDate > stats.lastInspection) {
            stats.lastInspection = inspectionDate;
          }

          // Add to vehicle set for selection
          vehicleSet.add(JSON.stringify({
            id: equipment.id,
            equipmentName: equipment.equipmentName || 'N/A',
            mainCategory: equipment.mainCategory || 'N/A',
            model: equipment.model || 'N/A',
            status: equipment.status || 'Available',
            meterReading: equipment.meterReading || '0',
            meterUnit: equipment.meterUnit || 'Miles',
          }));
        }
      });

      setAllChecklists(Array.from(checklistSet).map(title => ({ title, id: title })));
      setVehicleStats(Object.fromEntries(vehicleStatsMap));
      // Deduplicate vehicles and aggregate inspection counts
      const vehicleMap = new Map();
      allReports.forEach((report) => {
        const equipment = report.step1.selectedEquipment;
        if (equipment && equipment.id) {
          if (!vehicleMap.has(equipment.id)) {
            vehicleMap.set(equipment.id, {
              id: equipment.id,
              equipmentName: equipment.equipmentName || 'N/A',
              mainCategory: equipment.mainCategory || 'N/A',
              model: equipment.model || 'N/A',
              status: equipment.status || 'Available',
              meterReading: equipment.meterReading || '0',
              meterUnit: equipment.meterUnit || 'Miles',
              inspectionCount: 1,
            });
          } else {
            vehicleMap.get(equipment.id).inspectionCount++;
          }
        }
      });
      setAllVehicles(Array.from(vehicleMap.values()));
    } catch (error) {
      console.error('Error loading all inspection data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load inspection data.',
        position: 'top',
      });
    } finally {
      setIsLoading(false);
    }
  };

  const loadInspectionHistory = async () => {
    try {
      let allReports = [];
      // Load from Firebase (all users)
      try {
        const adminsRef = database().ref('/GlobalDailyreportdata/admins');
        const snapshot = await adminsRef.once('value');
        const adminsData = snapshot.val() || {};
        Object.values(adminsData).forEach((userReports) => {
          if (userReports && typeof userReports === 'object') {
            allReports.push(...Object.values(userReports));
          }
        });
      } catch (error) {
        console.log('Loading reports from local storage');
        const storedReports = await AsyncStorage.getItem('dailyInspections');
        if (storedReports) {
          const localReports = JSON.parse(storedReports);
          allReports = localReports;
        }
      }

      // Filter by selected date (month and year)
      const selectedYear = selectedDate.getFullYear();
      const selectedMonth = selectedDate.getMonth();
      let filteredReports = allReports.filter((report) => {
        const reportDate = new Date(report.step1?.inspectionDate || report.step3?.createdAt);
        return reportDate.getFullYear() === selectedYear && 
               reportDate.getMonth() === selectedMonth;
      });

      // Filter by selected checklist if any
      if (selectedChecklist) {
        filteredReports = filteredReports.filter((report) => {
          return report.step1?.checklistTitle === selectedChecklist.title;
        });
      }

      // Filter by selected vehicle if any
      if (selectedVehicle) {
        filteredReports = filteredReports.filter((report) => {
          return report.step1?.selectedEquipment?.id === selectedVehicle.id;
        });
      }

      // Sort by date (newest first)
      filteredReports.sort((a, b) => {
        const dateA = new Date(a.step1?.inspectionDate || a.step3?.createdAt);
        const dateB = new Date(b.step1?.inspectionDate || b.step3?.createdAt);
        return dateB - dateA;
      });

      setAllInspectionHistory(filteredReports);
    } catch (error) {
      console.error('Error loading inspection history:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load inspection history.',
        position: 'top',
      });
    }
  };

  const onDateChange = (event, date) => {
    setShowDatePicker(Platform.OS === 'ios');
    if (date) {
      setSelectedDate(date);
    }
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity 
      style={styles.checklistItem}
      onPress={() => {
        setSelectedChecklist(selectedChecklist?.title === item.title ? null : item);
        setChecklistModalVisible(false);
      }}
    >
      <View style={styles.checklistInfo}>
        <Text style={styles.checklistTitle}>{item.title}</Text>
      </View>
      <MaterialIcons 
        name={selectedChecklist?.title === item.title ? 'check-box' : 'check-box-outline-blank'} 
        size={24} 
        color={selectedChecklist?.title === item.title ? '#3481BC' : '#666'} 
      />
    </TouchableOpacity>
  );

  const renderVehicleItem = ({ item }) => (
    <TouchableOpacity 
      style={styles.vehicleItem}
      onPress={() => {
        setSelectedVehicle(selectedVehicle?.id === item.id ? null : item);
        setVehicleModalVisible(false);
      }}
    >
      <View style={styles.vehicleInfo}>
        <Text style={styles.vehicleName}>{item.equipmentName}</Text>
        <Text style={styles.vehicleDetails}>
          {item.mainCategory} • {item.model}
        </Text>
        <Text style={styles.vehicleMeter}>
          Meter: {item.meterReading} {item.meterUnit}
        </Text>
        <Text style={styles.inspectionCount}>
          {vehicleStats[item.id]?.inspectionCount || 0} inspection{(vehicleStats[item.id]?.inspectionCount || 0) !== 1 ? 's' : ''}
        </Text>
      </View>
      <View style={styles.vehicleStatus}>
        <Text style={[styles.statusText, { color: getStatusColor(item.status) }]}>
          {item.status}
        </Text>
        <MaterialIcons 
          name={selectedVehicle?.id === item.id ? 'check-box' : 'check-box-outline-blank'} 
          size={24} 
          color={selectedVehicle?.id === item.id ? '#3481BC' : '#666'} 
        />
      </View>
    </TouchableOpacity>
  );

  const renderInspectionItem = ({ item }) => {
    const inspectionDate = new Date(item.step1?.inspectionDate || item.step3?.createdAt);
    const formattedDate = inspectionDate.toLocaleDateString('en-US', {
      weekday: 'short',
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
    const formattedTime = inspectionDate.toLocaleTimeString('en-US', {
      hour: '2-digit',
      minute: '2-digit',
    });

    return (
      <View style={styles.inspectionCard}>
        <View style={styles.inspectionHeader}>
          <View style={styles.dateTimeContainer}>
            <Text style={styles.inspectionDate}>{formattedDate}</Text>
            <Text style={styles.inspectionTime}>{formattedTime}</Text>
          </View>
          <View style={styles.statusContainer}>
            <View style={[styles.statusBadge, { backgroundColor: item.step3?.pdfDownloadUrl ? '#2ecc71' : '#f39c12' }]}>
              <Text style={styles.statusBadgeText}>
                {item.step3?.pdfDownloadUrl ? 'Completed' : 'In Progress'}
              </Text>
            </View>
          </View>
        </View>
        
        <View style={styles.inspectionDetails}>
          <Text style={styles.detailLabel}>Vehicle:</Text>
          <Text style={styles.detailValue}>{item.step1?.selectedEquipment?.equipmentName || 'N/A'}</Text>
        </View>
        
        <View style={styles.inspectionDetails}>
          <Text style={styles.detailLabel}>Request #:</Text>
          <Text style={styles.detailValue}>{item.step1?.requestNumber || 'N/A'}</Text>
        </View>
        
        <View style={styles.inspectionDetails}>
          <Text style={styles.detailLabel}>Inspector:</Text>
          <Text style={styles.detailValue}>{item.step1?.inspector || 'N/A'}</Text>
        </View>
        
        <View style={styles.inspectionDetails}>
          <Text style={styles.detailLabel}>Checklist:</Text>
          <Text style={styles.detailValue}>{item.step1?.checklistTitle || 'N/A'}</Text>
        </View>

        <View style={styles.inspectionDetails}>
          <Text style={styles.detailLabel}>Country/Project:</Text>
          <Text style={styles.detailValue}>{item.step1?.country || 'N/A'} - {item.step1?.project || 'N/A'}</Text>
        </View>
        
        {item.step3?.pdfDownloadUrl && (
          <TouchableOpacity 
            style={styles.viewPdfButton}
            onPress={() => {
              navigation.navigate('PdfviewPage', {
                pdfUrl: item.step3.pdfDownloadUrl,
                onGoBack: () => navigation.navigate('TradckDailyreport'),
              });
            }}
          >
            <MaterialIcons name="picture-as-pdf" size={20} color="#fff" />
            <Text style={styles.viewPdfText}>View PDF</Text>
          </TouchableOpacity>
        )}
      </View>
    );
  };

  const renderVehicleStats = () => {
    const vehicleEntries = Object.entries(vehicleStats);
    if (vehicleEntries.length === 0) return null;

    return (
      <View style={styles.statsSection}>
        <Text style={styles.statsTitle}>All Vehicles with Inspections</Text>
        <Text style={styles.statsSubtitle}>
          {vehicleEntries.length} vehicle{vehicleEntries.length !== 1 ? 's' : ''} with inspection reports
        </Text>
        <ScrollView horizontal showsHorizontalScrollIndicator={false} style={styles.vehiclesScroll} contentContainerStyle={styles.vehiclesScrollContent}>
          {vehicleEntries.map(([vehicleId, stats]) => (
            <View key={vehicleId} style={styles.vehicleCardHorizontal}>
              <View style={styles.vehicleCardHeader}>
                <Text style={styles.vehicleCardName}>{stats.equipmentName}</Text>
                <View style={[styles.statusBadge, { backgroundColor: getStatusColor(stats.status) }]}> 
                  <Text style={styles.statusBadgeText}>{stats.status}</Text>
                </View>
              </View>
              <View style={styles.vehicleCardDetails}>
                <Text style={styles.vehicleCardCategory}>{stats.mainCategory} • {stats.model}</Text>
                <View style={styles.vehicleCardStats}>
                  <View style={styles.statItem}><MaterialIcons name="assignment" size={16} color="#3481BC" /><Text style={styles.statLabel}>Total:</Text><Text style={styles.statValue}>{stats.inspectionCount}</Text></View>
                  <View style={styles.statItem}><MaterialIcons name="check-circle" size={16} color="#2ecc71" /><Text style={styles.statLabel}>Completed:</Text><Text style={[styles.statValue, { color: '#2ecc71' }]}>{stats.completedCount}</Text></View>
                  <View style={styles.statItem}><MaterialIcons name="pending" size={16} color="#f39c12" /><Text style={styles.statLabel}>Pending:</Text><Text style={[styles.statValue, { color: '#f39c12' }]}>{stats.inspectionCount - stats.completedCount}</Text></View>
                </View>
                {stats.lastInspection && (<View style={styles.lastInspectionRow}><MaterialIcons name="schedule" size={14} color="#666" /><Text style={styles.lastInspectionText}>Last: {stats.lastInspection.toLocaleDateString()}</Text></View>)}
              </View>
              <TouchableOpacity style={styles.viewVehicleButton} onPress={() => { setSelectedVehicle(allVehicles.find(v => v.id === vehicleId) || null); setVehicleModalVisible(false); }}>
                <Text style={styles.viewVehicleText}>View Inspections</Text>
                <MaterialIcons name="chevron-right" size={16} color="#3481BC" />
              </TouchableOpacity>
            </View>
          ))}
        </ScrollView>
      </View>
    );
  };

  // Fetch user permissions (similar to DailyIns_report_home)
  const fetchUserPermissions = async () => {
    try {
      setIsLoadingPermissions(true);
      const user = auth().currentUser;
      const userUid = await AsyncStorage.getItem('userUid');
      if (!user || !userUid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not authenticated or UID not found.',
          position: 'top',
        });
        return;
      }
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');
      const userData = snapshot.val() || {};
      // Transform modules array into a flat object
      const modules = {};
      (userData.modules || []).forEach((module) => {
        modules[module.module] = module.permissions && module.permissions.length > 0;
      });
      setUserPermissions({
        profile: {
          country: Array.isArray(userData.countries) && userData.countries.length > 0
            ? userData.countries.join(', ')
            : (userData.selectedCountry || userData.country || (userData.profile && userData.profile.country) || 'Not Set'),
          project: Array.isArray(userData.projects) && userData.projects.length > 0
            ? userData.projects.join(', ')
            : (userData.selectedProject || userData.project || (userData.profile && userData.profile.project) || 'Not Set'),
          email: user.email || 'Not Available',
          uid: userUid,
        },
        modules: modules,
      });
      setPermissionModalVisible(true);
    } catch (error) {
      console.error('Error fetching permissions:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load permissions.',
        position: 'top',
      });
    } finally {
      setIsLoadingPermissions(false);
    }
  };

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={styles.header}
      >
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.backButton}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Daily Report Track</Text>
         <TouchableOpacity onPress={fetchUserPermissions} style={styles.refreshButton}>
           <MaterialIcons name="settings" size={24} color="#fff" />
         </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Scrollable Content */}
      <ScrollView 
        style={styles.scrollContainer}
        contentContainerStyle={styles.scrollContent}
        showsVerticalScrollIndicator={false}
      >
        {/* Checklist Selection */}
        <View style={styles.checklistSection}>
          <TouchableOpacity 
            style={styles.checklistSelector}
            onPress={() => setChecklistModalVisible(true)}
          >
            <View style={styles.selectorContent}>
              <MaterialIcons name="assignment" size={24} color="#3481BC" />
              <View style={styles.selectorText}>
                <Text style={styles.selectorLabel}>Filter by Checklist</Text>
                <Text style={styles.selectorValue}>
                  {selectedChecklist ? selectedChecklist.title : 'All checklists'}
                </Text>
              </View>
            </View>
            <MaterialIcons name="keyboard-arrow-down" size={24} color="#666" />
          </TouchableOpacity>
        </View>

        {/* Vehicle Selection */}
        <View style={styles.vehicleSection}>
          <TouchableOpacity 
            style={styles.vehicleSelector}
            onPress={() => setVehicleModalVisible(true)}
          >
            <View style={styles.selectorContent}>
              <MaterialIcons name="directions-car" size={24} color="#3481BC" />
              <View style={styles.selectorText}>
                <Text style={styles.selectorLabel}>Filter by Vehicle</Text>
                <Text style={styles.selectorValue}>
                  {selectedVehicle ? selectedVehicle.equipmentName : 'All vehicles'}
                </Text>
              </View>
            </View>
            <MaterialIcons name="keyboard-arrow-down" size={24} color="#666" />
          </TouchableOpacity>
        </View>

        {/* Date Selector */}
        <View style={styles.dateSection}>
          <TouchableOpacity 
            style={styles.dateSelector}
            onPress={() => setShowDatePicker(true)}
          >
            <View style={styles.dateContent}>
              <MaterialIcons name="calendar-today" size={20} color="#3481BC" />
              <Text style={styles.dateText}>
                {selectedDate.toLocaleDateString('en-US', {
                  weekday: 'long',
                  year: 'numeric',
                  month: 'long',
                  day: 'numeric',
                })}
              </Text>
            </View>
            <MaterialIcons name="keyboard-arrow-down" size={20} color="#666" />
          </TouchableOpacity>
        </View>

        {/* Vehicle Statistics */}
        {renderVehicleStats()}

        {/* All Inspection History */}
        <View style={styles.historySection}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>All Inspection History</Text>
            <Text style={styles.sectionSubtitle}>
              {selectedDate.toLocaleDateString('en-US', { month: 'long', year: 'numeric' })}
              {selectedChecklist ? ` • ${selectedChecklist.title}` : ''}
              {selectedVehicle ? ` • ${selectedVehicle.equipmentName}` : ''}
            </Text>
          </View>

          {isLoading ? (
            <View style={styles.loadingContainer}>
              <Text style={styles.loadingText}>Loading...</Text>
            </View>
          ) : (
            <View style={styles.inspectionListContainer}>
              {allInspectionHistory.length > 0 ? (
                allInspectionHistory.map((item, index) => (
                  <View key={item.id || item.step1?.requestNumber || index}>
                    {renderInspectionItem({ item })}
                  </View>
                ))
              ) : (
                <View style={styles.emptyContainer}>
                  <MaterialIcons name="assignment" size={48} color="#ccc" />
                  <Text style={styles.emptyText}>
                    No inspections found for {selectedDate.toLocaleDateString('en-US', { month: 'long', year: 'numeric' })}
                    {selectedChecklist ? ` with checklist "${selectedChecklist.title}"` : ''}
                    {selectedVehicle ? ` for vehicle "${selectedVehicle.equipmentName}"` : ''}
                  </Text>
                </View>
              )}
            </View>
          )}
        </View>
      </ScrollView>

      {/* Checklist Selection Modal */}
      <Modal
        visible={checklistModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setChecklistModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>Filter by Checklist</Text>
              <TouchableOpacity onPress={() => setChecklistModalVisible(false)}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <FlatList
              data={allChecklists}
              renderItem={renderChecklistItem}
              keyExtractor={(item) => item.id}
              contentContainerStyle={styles.checklistList}
              ListEmptyComponent={
                <View style={styles.emptyContainer}>
                  <Text style={styles.emptyText}>No checklists found</Text>
                </View>
              }
            />
          </View>
        </View>
      </Modal>

      {/* Vehicle Selection Modal */}
      <Modal
        visible={vehicleModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setVehicleModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.vehicleModalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>Filter by Vehicle</Text>
              <TouchableOpacity onPress={() => setVehicleModalVisible(false)}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <FlatList
              data={allVehicles}
              renderItem={renderVehicleItem}
              keyExtractor={(item) => item.id}
              contentContainerStyle={styles.checklistList}
              ListEmptyComponent={
                <View style={styles.emptyContainer}>
                  <Text style={styles.emptyText}>No vehicles found</Text>
                </View>
              }
              style={styles.vehicleListScrollable}
            />
          </View>
        </View>
      </Modal>

      {/* Date Picker */}
      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display="default"
          onChange={onDateChange}
          maximumDate={new Date()}
        />
      )}

      {/* Permission Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.5)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{ width: '85%', maxHeight: '80%', backgroundColor: '#fff', borderRadius: 10, padding: 20 }}>
            <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 15 }}>
              <Text style={{ fontSize: 18, fontWeight: 'bold', color: '#333' }}>User Permissions</Text>
              <TouchableOpacity onPress={() => setPermissionModalVisible(false)} style={{ padding: 5 }}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            {isLoadingPermissions ? (
              <View style={{ flexDirection: 'column', alignItems: 'center', marginTop: 20 }}>
                <MaterialIcons name="hourglass-empty" size={30} color="#3481BC" />
                <Text style={{ fontSize: 15, color: '#333', marginTop: 10 }}>Loading permissions...</Text>
              </View>
            ) : userPermissions ? (
              <>
                <View style={{ marginBottom: 15 }}>
                  <Text style={{ fontSize: 16, fontWeight: 'bold', color: '#015185', marginBottom: 10 }}>User Profile</Text>
                  <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 8 }}>
                    <Text style={{ fontSize: 15, fontWeight: '600', color: '#015185' }}>Country:</Text>
                    <Text style={{ fontSize: 15, color: '#666', fontWeight: '500' }}>{userPermissions.profile.country}</Text>
                  </View>
                  <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 8 }}>
                    <Text style={{ fontSize: 15, fontWeight: '600', color: '#015185' }}>Project:</Text>
                    <Text style={{ fontSize: 15, color: '#666', fontWeight: '500' }}>{userPermissions.profile.project}</Text>
                  </View>
                  <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 8 }}>
                    <Text style={{ fontSize: 15, fontWeight: '600', color: '#015185' }}>Email:</Text>
                    <Text style={{ fontSize: 15, color: '#666', fontWeight: '500' }}>{userPermissions.profile.email}</Text>
                  </View>
                  <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 8 }}>
                    <Text style={{ fontSize: 15, fontWeight: '600', color: '#015185' }}>User ID:</Text>
                    <Text style={{ fontSize: 15, color: '#666', fontWeight: '500' }}>{userPermissions.profile.uid}</Text>
                  </View>
                </View>
                <View style={{ height: 1, backgroundColor: '#e0e0e0', marginVertical: 15 }} />
                <View style={{ marginTop: 15 }}>
                  <Text style={{ fontSize: 16, fontWeight: 'bold', color: '#015185', marginBottom: 10 }}>Module Permissions</Text>
                  {Object.keys(userPermissions.modules).length > 0 ? (
                    Object.entries(userPermissions.modules).map(([moduleName, isEnabled]) => (
                      <View key={moduleName} style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 10 }}>
                        <Text style={{ fontSize: 15, fontWeight: '600', color: '#015185' }}>{moduleName}</Text>
                        <View style={{ paddingVertical: 4, paddingHorizontal: 10, borderRadius: 5, backgroundColor: isEnabled ? '#4CAF50' : '#f44336' }}>
                          <Text style={{ color: '#fff', fontSize: 13, fontWeight: 'bold' }}>{isEnabled ? 'Enabled' : 'Disabled'}</Text>
                        </View>
                      </View>
                    ))
                  ) : (
                    <Text style={{ fontSize: 14, color: '#888', textAlign: 'center', marginTop: 10 }}>No module permissions found</Text>
                  )}
                </View>
              </>
            ) : (
              <Text style={{ fontSize: 16, color: '#e74c3c', textAlign: 'center', marginTop: 10 }}>Failed to load permissions</Text>
            )}
            <TouchableOpacity
              style={{ backgroundColor: '#3481BC', borderRadius: 8, paddingVertical: 12, paddingHorizontal: 24, marginTop: 20, alignSelf: 'center' }}
              onPress={() => setPermissionModalVisible(false)}
            >
              <Text style={{ color: '#fff', fontSize: 16, fontWeight: 'bold' }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  header: {
    height: TOTAL_HEADER_HEIGHT,
    justifyContent: 'center',
    paddingTop: StatusBar.currentHeight || 40,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#fff',
  },
  refreshButton: {
    padding: 8,
  },
  checklistSection: {
    padding: 20,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  checklistSelector: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  selectorContent: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  selectorText: {
    marginLeft: 12,
    flex: 1,
  },
  selectorLabel: {
    fontSize: 12,
    color: '#666',
    marginBottom: 2,
  },
  selectorValue: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  vehicleSection: {
    padding: 20,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  vehicleSelector: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  dateSection: {
    padding: 20,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  dateSelector: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  dateContent: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  dateText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 12,
  },
  statsSection: {
    padding: 20,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  statsTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 15,
  },
  statsSubtitle: {
    fontSize: 14,
    color: '#666',
    marginBottom: 15,
  },
  vehiclesGrid: {
    // replaced by vehiclesScroll for horizontal scrolling
    display: 'none',
  },
  vehiclesScroll: {
    marginTop: 10,
    marginBottom: 10,
  },
  vehiclesScrollContent: {
    paddingLeft: 10,
    paddingRight: 10,
    alignItems: 'flex-start',
  },
  vehicleCardHorizontal: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 15,
    marginRight: 14,
    width: 240,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  vehicleCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  vehicleCardName: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    flex: 1,
    marginRight: 10,
  },
  statusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
  },
  statusBadgeText: {
    fontSize: 12,
    fontWeight: '600',
    color: '#fff',
  },
  vehicleCardDetails: {
    marginBottom: 12,
  },
  vehicleCardCategory: {
    fontSize: 14,
    color: '#666',
    marginBottom: 4,
  },
  vehicleCardStats: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 8,
  },
  statItem: {
    alignItems: 'center',
  },
  statLabel: {
    fontSize: 12,
    color: '#666',
    marginTop: 4,
  },
  statValue: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
  },
  lastInspectionRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  lastInspectionText: {
    fontSize: 12,
    color: '#666',
    marginLeft: 4,
  },
  viewVehicleButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#e0e0e0',
    paddingVertical: 8,
    paddingHorizontal: 12,
    borderRadius: 6,
  },
  viewVehicleText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#3481BC',
  },
  historySection: {
    flex: 1,
    padding: 20,
  },
  sectionHeader: {
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 5,
  },
  sectionSubtitle: {
    fontSize: 14,
    color: '#666',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 40,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
  },
  inspectionListContainer: {
    // Container for the inspection items
  },
  inspectionCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 16,
    marginBottom: 12,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  inspectionHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 12,
  },
  dateTimeContainer: {
    flex: 1,
  },
  inspectionDate: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  inspectionTime: {
    fontSize: 14,
    color: '#666',
    marginTop: 2,
  },
  statusContainer: {
    alignItems: 'flex-end',
  },
  statusBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
  },
  statusBadgeText: {
    fontSize: 12,
    fontWeight: '600',
    color: '#fff',
  },
  inspectionDetails: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  detailValue: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
    flex: 1,
    textAlign: 'right',
  },
  viewPdfButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#3481BC',
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 6,
    marginTop: 12,
  },
  viewPdfText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginLeft: 6,
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 40,
  },
  emptyText: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    marginTop: 12,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'flex-end',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    maxHeight: '70%',
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
  },
  checklistList: {
    padding: 20,
  },
  checklistItem: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    padding: 16,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginBottom: 8,
  },
  checklistInfo: {
    flex: 1,
  },
  checklistTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  vehicleItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 16,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  vehicleInfo: {
    flex: 1,
  },
  vehicleName: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 2,
  },
  vehicleDetails: {
    fontSize: 12,
    color: '#666',
    marginBottom: 4,
  },
  vehicleMeter: {
    fontSize: 12,
    color: '#666',
    marginBottom: 4,
  },
  inspectionCount: {
    fontSize: 12,
    color: '#666',
  },
  vehicleStatus: {
    alignItems: 'flex-end',
  },
  statusText: {
    fontSize: 12,
    fontWeight: '600',
  },
  scrollContainer: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 20, // Add some padding at the bottom for the date picker
  },
  vehicleModalContent: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    maxHeight: '70%',
    width: '100%',
  },
  vehicleListScrollable: {
    maxHeight: 350,
  },
});

export default TradckDailyreport;