import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  Dimensions,
  Image,
  Animated,
  Modal,
  ScrollView,
  Linking,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import LinearGradient from 'react-native-linear-gradient';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { PanGestureHandler, State } from 'react-native-gesture-handler';

import WebView from 'react-native-webview';
import Pdf from 'react-native-pdf';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

// Delete option removed per requirements
const DROPDOWN_HEIGHT = 44;

const Documenthome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [imageModalVisible, setImageModalVisible] = useState(false);
  const [viewerVisible, setViewerVisible] = useState(false);
  const [viewerUrl, setViewerUrl] = useState('');
  const [viewerType, setViewerType] = useState(''); // image | pdf | docx
  const [viewerDoc, setViewerDoc] = useState(null);
  const [viewerLoadError, setViewerLoadError] = useState(false);
  const [createFolderModalVisible, setCreateFolderModalVisible] = useState(false);
  const [deleteConfirmVisible, setDeleteConfirmVisible] = useState(false);
  const [documentToDelete, setDocumentToDelete] = useState(null);
  const [newFolderName, setNewFolderName] = useState('');
  const [newSubfolderName, setNewSubfolderName] = useState('');
  const [selectedImage, setSelectedImage] = useState(null);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [documentFolders, setDocumentFolders] = useState({});
  const [filteredDocumentFolders, setFilteredDocumentFolders] = useState({});
  const [currentView, setCurrentView] = useState({ type: 'main', folder: null, subfolder: null });
  const [isOnline, setIsOnline] = useState(true);
  const [imageLoadError, setImageLoadError] = useState(false);
  // Dropdown removed per requirements
  // const [dropdownVisible, setDropdownVisible] = useState(false);
  // const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [selectedFolder, setSelectedFolder] = useState(null); // Retained for potential future features
  const [selectedSubfolder, setSelectedSubfolder] = useState(null);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const initialLoadDoneRef = useRef(false);
  
  // Swipe gesture animations
  const swipeAnim = useRef(new Animated.Value(0)).current;
  const swipeOpacity = useRef(new Animated.Value(1)).current;

  // Display helper for folder names without changing underlying keys/logic
  const getDisplayFolderName = (folderKey) => {
    if (folderKey === 'AdminProPreDocuments') return 'Admin Documents';
    return folderKey;
  };

  // Display helper: Capitalize words and replace separators
  const formatDisplayName = (name) => {
    if (!name || typeof name !== 'string') return '';
    const cleaned = name.replace(/[._-]+/g, ' ').trim();
    return cleaned
      .split(' ')
      .filter(Boolean)
      .map((w) => w.charAt(0).toUpperCase() + w.slice(1))
      .join(' ');
  };

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable);
    });

    return () => unsubscribe();
  }, []);

  // iOS-specific navigation handling for direct MainApp access and navigation state
  useEffect(() => {
    if (Platform.OS === 'ios') {
      const unsubscribe = navigation.addListener('focus', () => {
        // Handle direct MainApp access - ensure we're in a valid state
        try {
          // Validate current state first to prevent crashes
          validateCurrentState();
          
          // Check if we're coming from a deep navigation or direct access
          const currentRoute = navigation.getCurrentRoute();
          if (currentRoute?.name === 'Documenthome') {
            // If we're in a deep folder view, stay there (don't reset)
            // This allows proper nested navigation
            if (currentView.type === 'main') {
              // Only reset if we're already at main level
              setCurrentView({ type: 'main', folder: null, subfolder: null });
            }
          }
        } catch (error) {
          console.log('iOS navigation focus error:', error);
          // Fallback to safe state
          setCurrentView({ type: 'main', folder: null, subfolder: null });
        }
      });

      return unsubscribe;
    }
  }, [navigation, currentView.type]);

  // iOS Navigation State Management - Prevents crashes on direct access and handles MainApp navigation
  useEffect(() => {
    if (Platform.OS === 'ios') {
      const unsubscribe = navigation.addListener('state', (e) => {
        // Handle navigation state changes safely
        try {
          const currentRoute = e.data.state?.routes?.[e.data.state.index];
          if (currentRoute?.name === 'Documenthome') {
            // Check if this is a direct MainApp access or navigation from another screen
            const previousRoute = e.data.state?.routes?.[e.data.state.index - 1];
            
            if (!previousRoute || previousRoute.name === 'MainApp') {
              // Direct MainApp access - ensure we start at main level
              if (currentView.type !== 'main') {
                setCurrentView({ type: 'main', folder: null, subfolder: null });
              }
            }
            // If navigating from another screen, preserve current view state
            
            // Reset any pending operations
            setSyncModalVisible(false);
            setImageModalVisible(false);
            setViewerVisible(false);
            setCreateFolderModalVisible(false);
            setDeleteConfirmVisible(false);
            setImageLoadError(false);
            setViewerLoadError(false);
          }
        } catch (error) {
          console.log('Navigation state handling error:', error);
          // Fallback to safe state
          setCurrentView({ type: 'main', folder: null, subfolder: null });
        }
      });

      return unsubscribe;
    }
  }, [navigation, currentView.type]);



  // Load documents from Firebase and AsyncStorage
  const loadDocuments = async () => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Admin not authenticated. Please log in again.',
        });
        navigation.navigate('AdminLoginScreen');
        return;
      }

      // Load offline documents from AsyncStorage
      let offlineDocs = [];
      try {
        const offlineData = await AsyncStorage.getItem('offlineDocuments');
        offlineDocs = offlineData ? JSON.parse(offlineData) : [];
      } catch (error) {
        console.error('Error loading offline documents:', error);
      }

      // Helper: Merge two folder structures: { [docType]: { [subfolder]: doc[] } }
      const mergeFolderStructures = (base, addition) => {
        const merged = { ...base };
        Object.keys(addition || {}).forEach((docType) => {
          if (!merged[docType]) merged[docType] = {};
          Object.keys(addition[docType] || {}).forEach((sub) => {
            if (!merged[docType][sub]) merged[docType][sub] = [];
            merged[docType][sub] = [...merged[docType][sub], ...(addition[docType][sub] || [])];
            merged[docType][sub].sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
          });
        });
        return merged;
      };

      // Helper: Load AdminProPreDocuments from Firebase Storage
      const loadStorageFolders = async () => {
        const result = {};
        try {
          const basePath = 'AdminProPreDocuments';
          const baseRef = storage().ref(basePath);
          const topLevel = await baseRef.listAll();
          if (!result[basePath]) result[basePath] = {};
          await Promise.all(
            (topLevel.prefixes || []).map(async (prefixRef) => {
              const subfolderName = prefixRef.name;
              const sub = await prefixRef.listAll();
              const files = await Promise.all(
                (sub.items || []).map(async (itemRef) => {
                  let url = '';
                  try {
                    url = await itemRef.getDownloadURL();
                  } catch (e) {
                    // ignore URL errors, still show item by name
                  }
                  const name = itemRef.name;
                  const extMatch = name.match(/\.([0-9a-z]+)$/i);
                  const ext = extMatch ? (extMatch[1] || '').toLowerCase() : '';
                  return {
                    id: itemRef.fullPath,
                    type: ext || 'file',
                    description: name,
                    equipment: { entryNumber: '-', equipmentName: '-' },
                    documentType: basePath,
                    subfolderName,
                    file: url,
                    isOffline: false,
                    createdAt: new Date().toISOString(),
                  };
                })
              );
              files.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
              result[basePath][subfolderName] = files;
            })
          );
        } catch (error) {
          console.error('Error loading storage folders:', error);
        }
        return result;
      };

      // Load online documents from Firebase Realtime Database
      const documentsRef = database().ref(`/Documents/${adminUid}`);
      documentsRef.on('value', async (snapshot) => {
        const data = snapshot.val() || {};
        const folders = {};

        // Process Firebase documents
        Object.keys(data).forEach((docType) => {
          folders[docType] = {};
          Object.keys(data[docType]).forEach((subfolder) => {
            folders[docType][subfolder] = Object.keys(data[docType][subfolder]).map((key) => ({
              ...data[docType][subfolder][key],
              id: key,
              isOffline: false,
            }));
            folders[docType][subfolder].sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
          });
        });

        // Merge offline documents
        offlineDocs.forEach((doc) => {
          const docType = doc.documentType;
          const subfolder = doc.subfolderName;
          if (!folders[docType]) {
            folders[docType] = {};
          }
          if (!folders[docType][subfolder]) {
            folders[docType][subfolder] = [];
          }
          folders[docType][subfolder].push({
            ...doc,
            file: doc.fileUri,
            isOffline: true,
          });
          folders[docType][subfolder].sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        });

        // Also load Storage-based folders and merge
        const storageFolders = await loadStorageFolders();
        const mergedFolders = mergeFolderStructures(folders, storageFolders);

        // Sort folders alphabetically
        const sortedFolders = {};
        const folderOrder = Object.keys(mergedFolders).sort();
        
        // Apply alphabetical order to main folders
        folderOrder.forEach(folderKey => {
          sortedFolders[folderKey] = mergedFolders[folderKey];
          
          // Also sort subfolders alphabetically within each main folder
          if (mergedFolders[folderKey] && typeof mergedFolders[folderKey] === 'object') {
            const subfolderKeys = Object.keys(mergedFolders[folderKey]).sort();
            const sortedSubfolders = {};
            subfolderKeys.forEach(subKey => {
              sortedSubfolders[subKey] = mergedFolders[folderKey][subKey];
            });
            sortedFolders[folderKey] = sortedSubfolders;
          }
        });

        setDocumentFolders(sortedFolders);
        setFilteredDocumentFolders(sortedFolders);
        if (!initialLoadDoneRef.current) {
          initialLoadDoneRef.current = true;
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
        }
      });

      return () => documentsRef.off();
    } catch (error) {
      console.error('Error loading documents:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load documents.',
      });
    }
  };

  useEffect(() => {
    let isMounted = true;
    
    try {
      // iOS: Safe initialization with error handling
      if (Platform.OS === 'ios') {
        // Ensure we start in a safe state
        if (isMounted) {
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          setSyncModalVisible(false);
          setImageModalVisible(false);
          setViewerVisible(false);
          setCreateFolderModalVisible(false);
          setDeleteConfirmVisible(false);
          setImageLoadError(false);
          setViewerLoadError(false);
        }
        
        // Small delay for iOS to ensure proper initialization
        setTimeout(() => {
          if (isMounted) {
            setSyncModalVisible(true);
            Animated.loop(
              Animated.timing(rotateAnim, {
                toValue: 1,
                duration: 1000,
                useNativeDriver: true,
              })
            ).start();
            loadDocuments();
          }
        }, 100);
      } else {
        // Android: Direct initialization
        if (isMounted) {
          setSyncModalVisible(true);
          Animated.loop(
            Animated.timing(rotateAnim, {
              toValue: 1,
              duration: 1000,
              useNativeDriver: true,
            })
          ).start();
          loadDocuments();
        }
      }
    } catch (error) {
      console.log('Component initialization error:', error);
      // Fallback to safe state
      if (isMounted) {
        setCurrentView({ type: 'main', folder: null, subfolder: null });
        setSyncModalVisible(false);
      }
    }

    return () => {
      isMounted = false;
    };
  }, []);



  // Single consolidated navigation back handler - prevents crashes
  useEffect(() => {
    const unsubscribe = navigation.addListener('beforeRemove', (e) => {
      try {
        // iOS: Add extra safety check to prevent crashes
        if (Platform.OS === 'ios') {
          // Validate current state before proceeding
          if (!currentView || typeof currentView !== 'object') {
            console.log('iOS: Invalid currentView state, resetting to main');
            e.preventDefault();
            setCurrentView({ type: 'main', folder: null, subfolder: null });
            return;
          }
          
          // Validate documentFolders exists
          if (!documentFolders || typeof documentFolders !== 'object') {
            console.log('iOS: Invalid documentFolders state, resetting to main');
            e.preventDefault();
            setCurrentView({ type: 'main', folder: null, subfolder: null });
            return;
          }
        }
        
        // If we are at top-level, allow default back behavior
        if (currentView.type === 'main') return;
        
        // Prevent leaving the screen - handle nested navigation
        e.preventDefault();
        
        // Safe navigation logic for both platforms
        if (currentView.type === 'subfolder' && currentView.folder) {
          // Validate folder exists before navigating
          if (documentFolders[currentView.folder]) {
            setCurrentView({ type: 'folder', folder: currentView.folder, subfolder: null });
          } else {
            // Folder doesn't exist, reset to main to prevent crash
            console.log('Invalid folder detected, resetting to main');
            setCurrentView({ type: 'main', folder: null, subfolder: null });
          }
        } else if (currentView.type === 'folder') {
          setCurrentView({ type: 'main', folder: null, subfolder: null });
        } else {
          // Fallback to main if something goes wrong
          setCurrentView({ type: 'main', folder: null, subfolder: null });
        }
      } catch (error) {
        console.log('Navigation error, resetting to main:', error);
        e.preventDefault();
        setCurrentView({ type: 'main', folder: null, subfolder: null });
      }
    });
    
    return unsubscribe;
  }, [navigation, currentView, documentFolders]);

  // Handle search
  const handleSearch = (query) => {
    try {
      // iOS: Safe search with validation
      if (Platform.OS === 'ios') {
        // Validate query and document folders
        if (!query || typeof query !== 'string') {
          setSearchQuery('');
          setFilteredDocumentFolders(documentFolders || {});
          return;
        }
        
        if (!documentFolders || typeof documentFolders !== 'object') {
          setSearchQuery(query);
          setFilteredDocumentFolders({});
          return;
        }
        
        setSearchQuery(query);
        
        if (query.trim()) {
          const filtered = {};
          // Maintain alphabetical folder order
          Object.keys(documentFolders).sort().forEach((docType) => {
            try {
              filtered[docType] = {};
              // Also maintain alphabetical order for subfolders
              if (documentFolders[docType] && typeof documentFolders[docType] === 'object') {
                Object.keys(documentFolders[docType]).sort().forEach((subfolder) => {
                  try {
                    const subfolderDocs = documentFolders[docType][subfolder];
                    if (Array.isArray(subfolderDocs)) {
                      const filteredDocs = subfolderDocs.filter((doc) => {
                        try {
                          return (
                            (doc.documentType && doc.documentType.toLowerCase().includes(query.toLowerCase())) ||
                            (doc.type && doc.type.toLowerCase().includes(query.toLowerCase())) ||
                            (doc.description && doc.description.toLowerCase().includes(query.toLowerCase())) ||
                            (doc.subfolderName && doc.subfolderName.toLowerCase().includes(query.toLowerCase())) ||
                            (doc.equipment && doc.equipment.equipmentName && doc.equipment.equipmentName.toLowerCase().includes(query.toLowerCase())) ||
                            (doc.equipment && doc.equipment.entryNumber && doc.equipment.entryNumber.toLowerCase().includes(query.toLowerCase()))
                          );
                        } catch (filterError) {
                          console.log('Filter error for doc:', filterError);
                          return false;
                        }
                      });
                      if (filteredDocs.length > 0) {
                        filtered[docType][subfolder] = filteredDocs;
                      }
                    }
                  } catch (subfolderError) {
                    console.log('Subfolder search error:', subfolderError);
                  }
                });
              }
              if (Object.keys(filtered[docType]).length === 0) {
                delete filtered[docType];
              }
            } catch (docTypeError) {
              console.log('DocType search error:', docTypeError);
            }
          });
          setFilteredDocumentFolders(filtered);
        } else {
          setFilteredDocumentFolders(documentFolders || {});
        }
      } else {
        // Android: Direct search
        setSearchQuery(query);
        if (query) {
          const filtered = {};
          // Maintain alphabetical folder order
          Object.keys(documentFolders).sort().forEach((docType) => {
            filtered[docType] = {};
            // Also maintain alphabetical order for subfolders
            Object.keys(documentFolders[docType]).sort().forEach((subfolder) => {
              const filteredDocs = documentFolders[docType][subfolder].filter(
                (doc) =>
                  doc.documentType.toLowerCase().includes(query.toLowerCase()) ||
                  doc.type.toLowerCase().includes(query.toLowerCase()) ||
                  doc.description.toLowerCase().includes(query.toLowerCase()) ||
                  doc.subfolderName.toLowerCase().includes(query.toLowerCase()) ||
                  doc.equipment.equipmentName.toLowerCase().includes(query.toLowerCase()) ||
                  doc.equipment.entryNumber.toLowerCase().includes(query.toLowerCase())
              );
              if (filteredDocs.length > 0) {
                filtered[docType][subfolder] = filteredDocs;
              }
            });
            if (Object.keys(filtered[docType]).length === 0) {
              delete filtered[docType];
            }
          });
          setFilteredDocumentFolders(filtered);
        } else {
          setFilteredDocumentFolders(documentFolders);
        }
      }
    } catch (error) {
      console.log('Search error, resetting to safe state:', error);
      // Fallback to safe state
      setSearchQuery('');
      setFilteredDocumentFolders(documentFolders || {});
    }
  };

  // Handle refresh
  const handleRefresh = () => {
    try {
      // iOS: Safe refresh with state validation
      if (Platform.OS === 'ios') {
        // Ensure we're in a safe state before refreshing
        if (currentView.type !== 'main') {
          safeSetCurrentView({ type: 'main', folder: null, subfolder: null });
        }
        
        setIsRefreshing(true);
        setSyncModalVisible(true);

        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();

        // Safe document loading
        try {
          loadDocuments();
        } catch (error) {
          console.log('Document loading error during refresh:', error);
          // Continue with refresh even if loading fails
        }
        
        setSearchQuery('');

        setTimeout(() => {
          setIsRefreshing(false);
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
        }, 2000);
      } else {
        // Android: Direct refresh
        setIsRefreshing(true);
        setSyncModalVisible(true);

        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();

        loadDocuments();
        setSearchQuery('');

        setTimeout(() => {
          setIsRefreshing(false);
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
        }, 2000);
      }
    } catch (error) {
      console.log('Refresh error, resetting to safe state:', error);
      // Fallback to safe state
      safeSetCurrentView({ type: 'main', folder: null, subfolder: null });
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  // Handle delete document
  const handleDelete = async (docType, subfolder, docId, isOffline) => {
    // Show confirmation dialog first
    setDocumentToDelete({ docType, subfolder, docId, isOffline });
    setDeleteConfirmVisible(true);
  };

  // Confirm delete document
  const confirmDeleteDocument = async () => {
    if (!documentToDelete) return;
    
    const { docType, subfolder, docId, isOffline } = documentToDelete;
    
    try {
      if (isOffline) {
        const offlineDocs = await AsyncStorage.getItem('offlineDocuments');
        let docs = offlineDocs ? JSON.parse(offlineDocs) : [];
        docs = docs.filter((doc) => doc.id !== docId);
        await AsyncStorage.setItem('offlineDocuments', JSON.stringify(docs));
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Offline document deleted.',
        });
      } else {
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Admin not authenticated. Please log in again.',
          });
          return;
        }

        await database().ref(`/Documents/${adminUid}/${docType}/${subfolder}/${docId}`).remove();
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Document deleted from database.',
        });
      }

      const updatedFolders = { ...documentFolders };
      updatedFolders[docType][subfolder] = updatedFolders[docType][subfolder].filter(
        (doc) => doc.id !== docId
      );
      if (updatedFolders[docType][subfolder].length === 0) {
        delete updatedFolders[docType][subfolder];
        if (Object.keys(updatedFolders[docType]).length === 0) {
          delete updatedFolders[docType];
        }
      }
      setDocumentFolders(updatedFolders);
      setFilteredDocumentFolders(updatedFolders);
    } catch (error) {
      console.error('Error deleting document:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to delete document.',
      });
    } finally {
      setDeleteConfirmVisible(false);
      setDocumentToDelete(null);
    }
  };

  // Cancel delete document
  const cancelDeleteDocument = () => {
    setDeleteConfirmVisible(false);
    setDocumentToDelete(null);
  };



  // Handle delete folder/subfolder (triggered from dropdown)
  // Delete flow removed per requirements

  const confirmDeleteFolder = async () => {
    if (!selectedFolder) return;

    setSyncModalVisible(true); // Show loading modal during deletion

    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Admin not authenticated. Please log in again.',
        });
        return;
      }

      const updatedFolders = { ...documentFolders };
      if (currentView.type === 'main') {
        // Delete folder
        delete updatedFolders[selectedFolder];
        if (isOnline) {
          await database().ref(`/Documents/${adminUid}/${selectedFolder}`).remove();
        }
      } else if (currentView.type === 'folder') {
        // Delete subfolder
        delete updatedFolders[selectedFolder][selectedSubfolder];
        if (Object.keys(updatedFolders[selectedFolder]).length === 0) {
          delete updatedFolders[selectedFolder];
        }
        if (isOnline) {
          await database()
            .ref(`/Documents/${adminUid}/${selectedFolder}/${selectedSubfolder}`)
            .remove();
        }
      }

      setDocumentFolders(updatedFolders);
      setFilteredDocumentFolders(updatedFolders);

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: currentView.type === 'main' ? 'Folder deleted.' : 'Subfolder deleted.',
      });

      // Navigate back after deletion
      if (currentView.type === 'main') {
        setCurrentView({ type: 'main', folder: null, subfolder: null });
      } else if (currentView.type === 'folder') {
        setCurrentView({ type: 'folder', folder: selectedFolder, subfolder: null });
      }
    } catch (error) {
      console.error('Error deleting folder/subfolder:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to delete folder/subfolder.',
      });
      await loadDocuments(); // Revert state if deletion fails
    } finally {
      setSyncModalVisible(false);
      setSelectedFolder(null);
      setSelectedSubfolder(null);
    }
  };

  // const cancelDeleteFolder = () => {};

  // Dropdown actions removed (delete hidden per requirements)
  const dropdownActions = {};

  // Handle view document
  const handleView = async (doc) => {
    const fileUrl = doc.file;
    const isOffline = doc.isOffline;

    if (!fileUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'File URL is missing.',
      });
      return;
    }

    const extensionMatch = fileUrl.match(/\.([0-9a-z]+)(?:[\?#]|$)/i);
    const extension = extensionMatch ? extensionMatch[1].toLowerCase() : '';
    const isImage = ['jpg', 'jpeg', 'png', 'gif'].includes(extension);
    const isPdf = extension === 'pdf';
    const isDocx = extension === 'docx';

    if (isImage) {
      setViewerType('image');
      setViewerUrl(fileUrl);
      setViewerDoc(doc);
      setViewerLoadError(false);
      setViewerVisible(true);
    } else if (isPdf) {
      try {
        let pdfUrl = fileUrl;
        if (isOffline) {
          // For offline files, try to open directly
          pdfUrl = fileUrl.startsWith('file://') ? fileUrl : `file://${fileUrl}`;
        }
        setViewerType('pdf');
        setViewerUrl(pdfUrl);
        setViewerDoc(doc);
        setViewerLoadError(false);
        setViewerVisible(true);
      } catch (error) {
        console.error('Error opening PDF:', error);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to open PDF.' });
      }
    } else if (isDocx) {
      try {
        const gviewUrl = `https://docs.google.com/gview?embedded=1&url=${encodeURIComponent(fileUrl)}`;
        setViewerType('docx');
        setViewerUrl(gviewUrl);
        setViewerDoc(doc);
        setViewerLoadError(false);
        setViewerVisible(true);
      } catch (error) {
        console.error('Error opening DOCX:', error);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to open document.' });
      }
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Unsupported file type.',
      });
    }
  };



  // Safe navigation functions for iOS
  const safeSetCurrentView = (newView) => {
    try {
      if (Platform.OS === 'ios') {
        // iOS: Add validation and error handling
        if (!newView || typeof newView !== 'object') {
          console.log('Invalid view state, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return;
        }
        
        // Validate the new view state
        if (newView.type === 'folder' && !newView.folder) {
          console.log('Invalid folder view, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return;
        }
        
        if (newView.type === 'subfolder' && (!newView.folder || !newView.subfolder)) {
          console.log('Invalid subfolder view, resetting to folder');
          setCurrentView({ type: 'folder', folder: newView.folder || 'Equipments', subfolder: null });
          return;
        }
      }
      
      setCurrentView(newView);
    } catch (error) {
      console.log('Navigation error, resetting to main:', error);
      setCurrentView({ type: 'main', folder: null, subfolder: null });
    }
  };

  // iOS state validation function to prevent crashes
  const validateCurrentState = () => {
    try {
      if (Platform.OS === 'ios') {
        // Check if currentView is valid
        if (!currentView || typeof currentView !== 'object') {
          console.log('iOS: Invalid currentView detected, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return false;
        }
        
        // Check if documentFolders is valid
        if (!documentFolders || typeof documentFolders !== 'object') {
          console.log('iOS: Invalid documentFolders detected, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return false;
        }
        
        // Validate current view state
        if (currentView.type === 'subfolder' && (!currentView.folder || !documentFolders[currentView.folder])) {
          console.log('iOS: Invalid subfolder state detected, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return false;
        }
        
        if (currentView.type === 'folder' && (!currentView.folder || !documentFolders[currentView.folder])) {
          console.log('iOS: Invalid folder state detected, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return false;
        }
      }
      return true;
    } catch (error) {
      console.log('iOS state validation error:', error);
      setCurrentView({ type: 'main', folder: null, subfolder: null });
      return false;
    }
  };

  // Swipe gesture handlers for navigation
  const onSwipeGestureEvent = Animated.event(
    [{ nativeEvent: { translationX: swipeAnim } }],
    { useNativeDriver: false }
  );

  const onSwipeHandlerStateChange = (event) => {
    if (event.nativeEvent.state === State.END) {
      const { translationX } = event.nativeEvent;
      
      // Swipe right threshold for going back
      if (translationX > 100) {
        handleSwipeBack();
      }
      
      // Reset animation
      Animated.parallel([
        Animated.timing(swipeAnim, {
          toValue: 0,
          duration: 200,
          useNativeDriver: false,
        }),
        Animated.timing(swipeOpacity, {
          toValue: 1,
          duration: 200,
          useNativeDriver: false,
        }),
      ]).start();
    }
  };

  const handleSwipeBack = () => {
    try {
      // iOS: Add extra safety check to prevent crashes
      if (Platform.OS === 'ios') {
        // Validate current state before proceeding
        if (!currentView || typeof currentView !== 'object') {
          console.log('iOS: Invalid currentView state, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return;
        }
        
        // Validate documentFolders exists
        if (!documentFolders || typeof documentFolders !== 'object') {
          console.log('iOS: Invalid documentFolders state, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
          return;
        }
      }
      
      // Handle swipe back navigation
      if (currentView.type === 'subfolder' && currentView.folder) {
        // Validate folder exists before navigating
        if (documentFolders[currentView.folder]) {
          setCurrentView({ type: 'folder', folder: currentView.folder, subfolder: null });
        } else {
          // Folder doesn't exist, reset to main to prevent crash
          console.log('Invalid folder detected, resetting to main');
          setCurrentView({ type: 'main', folder: null, subfolder: null });
        }
      } else if (currentView.type === 'folder') {
        setCurrentView({ type: 'main', folder: null, subfolder: null });
      } else if (currentView.type === 'main') {
        // At main level, go back to previous screen
        navigation.goBack();
      }
    } catch (error) {
      console.log('Swipe back navigation error, resetting to main:', error);
      setCurrentView({ type: 'main', folder: null, subfolder: null });
    }
  };

  // Navigate to folder view (show subfolders) - iOS crash prevention
  const viewFolder = (docType) => {
    try {
      if (Platform.OS === 'ios') {
        // iOS: Validate folder exists before navigating
        if (!docType || !documentFolders[docType]) {
          console.log('iOS: Invalid folder navigation attempt, staying at main');
          Toast.show({
            type: 'error',
            text1: 'Navigation Error',
            text2: 'Folder not found. Please refresh and try again.',
          });
          return;
        }
      }
      safeSetCurrentView({ type: 'folder', folder: docType, subfolder: null });
    } catch (error) {
      console.log('Folder navigation error:', error);
      // Stay at current view to prevent crash
    }
  };

  // Navigate to subfolder view - iOS crash prevention
  const viewSubfolder = (docType, subfolder) => {
    try {
      if (Platform.OS === 'ios') {
        // iOS: Validate folder and subfolder exist before navigating
        if (!docType || !subfolder || !documentFolders[docType] || !documentFolders[docType][subfolder]) {
          console.log('iOS: Invalid subfolder navigation attempt, staying at folder');
          Toast.show({
            type: 'error',
            text1: 'Navigation Error',
            text2: 'Subfolder not found. Please refresh and try again.',
          });
          return;
        }
      }
      safeSetCurrentView({ type: 'subfolder', folder: docType, subfolder });
    } catch (error) {
      console.log('Subfolder navigation error:', error);
      // Stay at current view to prevent crash
    }
  };

  // Navigate back to parent folder or main view - proper nested navigation for both platforms
  const handleBack = () => {
    try {
      // Both platforms: Step up one level for proper nested navigation
      if (currentView.type === 'subfolder') {
        safeSetCurrentView({ type: 'folder', folder: currentView.folder, subfolder: null });
      } else if (currentView.type === 'folder') {
        safeSetCurrentView({ type: 'main', folder: null, subfolder: null });
      } else {
        // At main level, go back to previous screen
        navigation.goBack();
      }
    } catch (error) {
      console.log('Back navigation error, resetting to main:', error);
      safeSetCurrentView({ type: 'main', folder: null, subfolder: null });
    }
  };

  // Handle creating a new folder or subfolder
  const handleCreateFolder = async () => {
    if (!newFolderName.trim() && currentView.type === 'main') {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Folder name is required.',
      });
      return;
    }
    if (!newSubfolderName.trim() && currentView.type === 'folder') {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Subfolder name is required.',
      });
      return;
    }

    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Admin not authenticated. Please log in again.',
        });
        return;
      }

      if (currentView.type === 'main') {
        // Create a new folder
        const folderName = newFolderName.trim();
        if (documentFolders[folderName]) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Folder already exists.',
          });
          return;
        }
        const updatedFolders = { ...documentFolders, [folderName]: {} };
        setDocumentFolders(updatedFolders);
        setFilteredDocumentFolders(updatedFolders);

        if (isOnline) {
          await database().ref(`/Documents/${adminUid}/${folderName}`).set({});
        }
      } else if (currentView.type === 'folder') {
        // Create a new subfolder
        const subfolderName = newSubfolderName.trim();
        if (documentFolders[currentView.folder][subfolderName]) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Subfolder already exists.',
          });
          return;
        }
        const updatedFolders = { ...documentFolders };
        updatedFolders[currentView.folder][subfolderName] = [];
        setDocumentFolders(updatedFolders);
        setFilteredDocumentFolders(updatedFolders);

        if (isOnline) {
          await database().ref(`/Documents/${adminUid}/${currentView.folder}/${subfolderName}`).set({});
        }
      }

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: currentView.type === 'main' ? 'Folder created.' : 'Subfolder created.',
      });
      setCreateFolderModalVisible(false);
      setNewFolderName('');
      setNewSubfolderName('');
    } catch (error) {
      console.error('Error creating folder/subfolder:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to create folder/subfolder.',
      });
    }
  };

  // Handle creating a new document
  const handleCreateDocument = () => {
    try {
      // iOS: Validate navigation parameters before navigating
      if (Platform.OS === 'ios') {
        const navigationParams = {
          folder: currentView.folder || 'Equipments',
          subfolder: currentView.subfolder || '',
          onDocumentCreated: () => {
            try {
              loadDocuments(); // Refresh documents after creation
            } catch (error) {
              console.log('Document refresh error:', error);
              // Continue without refresh if there's an error
            }
          }
        };
        
        navigation.navigate('DocumentuploadS', navigationParams);
      } else {
        // Android: Direct navigation
        navigation.navigate('DocumentuploadS', {
          folder: currentView.folder,
          subfolder: currentView.subfolder,
          onDocumentCreated: () => loadDocuments(),
        });
      }
    } catch (error) {
      console.log('Document creation navigation error:', error);
      // Fallback navigation with safe parameters
      navigation.navigate('DocumentuploadS', {
        folder: 'Equipments',
        subfolder: '',
        onDocumentCreated: () => {
          try {
            loadDocuments();
          } catch (refreshError) {
            console.log('Fallback refresh error:', refreshError);
          }
        }
      });
    }
  };

  // Handle document download
  const handleDownload = async (doc) => {
    try {
      if (!doc || !doc.file) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Download URL not available',
        });
        return;
      }

      // For iOS, we can use the default browser or share functionality
      if (Platform.OS === 'ios') {
        // Use Linking to open in browser or share
        const supported = await Linking.canOpenURL(doc.file);
        if (supported) {
          await Linking.openURL(doc.file);
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Cannot open this file type',
          });
        }
      } else {
        // For Android, you can implement download functionality
        Toast.show({
          type: 'info',
          text1: 'Info',
          text2: 'Download functionality available on Android',
        });
      }
    } catch (error) {
      console.error('Error handling download:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to process download',
      });
    }
  };

  // Render back button text
  const renderBackButtonText = () => {
    if (currentView.type === 'subfolder') {
      return `Back to ${getDisplayFolderName(currentView.folder)}`;
    } else if (currentView.type === 'folder') {
      return 'Back to Documents';
    }
    return '';
  };

  // Render header title based on current view
  const renderHeaderTitle = () => {
    if (currentView.type === 'subfolder') {
      return formatDisplayName(currentView.subfolder);
    } else if (currentView.type === 'folder') {
      return getDisplayFolderName(currentView.folder);
    }
    return 'Documents';
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={handleBack}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>{renderHeaderTitle()}</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
            {(currentView.type === 'main' || currentView.type === 'folder') && (
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => setCreateFolderModalVisible(true)}
              >
                <MaterialIcons name="folder" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
            )}
          </View>
        </View>

        {/* Back Button Text and Search Bar */}
        <View style={styles.headerBottom}>
          {currentView.type !== 'main' && (
            <View style={styles.backButtonContainer}>
              <Text style={styles.backButtonText}>{renderBackButtonText()}</Text>
              {Platform.OS === 'ios' && currentView.type !== 'main' && (
                <View style={styles.swipeHintContainer}>
                  <MaterialIcons name="swipe-right" size={16} color="rgba(255, 255, 255, 0.8)" />
                  <Text style={styles.iosSwipeHint}>Swipe right to go back</Text>
                </View>
              )}
            </View>
          )}
          <View style={styles.searchContainer}>
            <View style={styles.searchWrapper}>
              <TextInput
                style={styles.searchInput}
                placeholder="Search..."
                placeholderTextColor="#888"
                value={searchQuery}
                onChangeText={handleSearch}
              />
              <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
            </View>
          </View>
        </View>
      </LinearGradient>

      {/* Content with Swipe Gestures */}
      <PanGestureHandler
        onGestureEvent={onSwipeGestureEvent}
        onHandlerStateChange={onSwipeHandlerStateChange}
        activeOffsetX={[-20, 20]}
        failOffsetY={[-20, 20]}
      >
        <Animated.View
          style={[
            styles.contentContainer,
            {
              transform: [{ translateX: swipeAnim }],
              opacity: swipeOpacity,
            },
          ]}
        >
          <ScrollView style={styles.content}>
        {currentView.type === 'main' ? (
          <>
            {Object.keys(filteredDocumentFolders).length > 0 ? (
              Object.keys(filteredDocumentFolders).map((docType) => {
                // Calculate total documents in this folder
                const totalDocs = Object.values(filteredDocumentFolders[docType]).reduce((total, subfolder) => {
                  return total + (subfolder.length || 0);
                }, 0);
                
                return (
                  <View key={docType} style={styles.folderRow}>
                    <TouchableOpacity
                      style={styles.folderContent}
                      onPress={() => viewFolder(docType)}
                    >
                      <View style={styles.folderHeader}>
                        <MaterialIcons name="folder" size={45} color="#015185" />
                        <View style={styles.folderInfo}>
                          <Text style={styles.folderTitle}>{getDisplayFolderName(docType)}</Text>
                          <Text style={styles.documentCount}>{totalDocs} document{totalDocs !== 1 ? 's' : ''}</Text>
                        </View>
                      </View>
                      <View style={styles.swipeIndicator}>
                        <MaterialIcons name="chevron-right" size={20} color="#015185" />
                      </View>
                    </TouchableOpacity>
                  </View>
                );
              })
            ) : (
              <Text style={styles.noDataText}>No Folders Found</Text>
            )}
          </>
        ) : currentView.type === 'folder' ? (
          <>
            {Object.keys(filteredDocumentFolders[currentView.folder] || {}).length > 0 ? (
              Object.keys(filteredDocumentFolders[currentView.folder]).map((subfolder) => {
                // Get document count for this subfolder
                const docCount = filteredDocumentFolders[currentView.folder][subfolder]?.length || 0;
                
                return (
                  <View key={subfolder} style={styles.subfolderRow}>
                    <TouchableOpacity
                      style={styles.subfolderContent}
                      onPress={() => viewSubfolder(currentView.folder, subfolder)}
                    >
                      <View style={styles.folderHeader}>
                        <MaterialIcons name="folder" size={45} color="#3481BC" />
                        <View style={styles.folderInfo}>
                          <Text style={styles.subfolderTitle}>{formatDisplayName(subfolder)}</Text>
                          <Text style={styles.documentCount}>{docCount} document{docCount !== 1 ? 's' : ''}</Text>
                        </View>
                      </View>
                      <View style={styles.swipeIndicator}>
                        <MaterialIcons name="chevron-right" size={20} color="#3481BC" />
                      </View>
                    </TouchableOpacity>
                  </View>
                );
              })
            ) : (
              <Text style={styles.noDataText}>No Subfolders Found</Text>
            )}
          </>
        ) : (
          <>
            {filteredDocumentFolders[currentView.folder]?.[currentView.subfolder]?.length > 0 ? (
              filteredDocumentFolders[currentView.folder][currentView.subfolder].map((doc) => (
                <View key={doc.id} style={styles.documentRow}>
                  <View style={styles.documentInfo}>
                    <View style={styles.documentDetailItem}>
                      <Text style={styles.detailLabel}>Type:</Text>
                      <View style={styles.typeContainer}>
                        <Text style={styles.detailValue}>{doc.type}</Text>
                        {doc.documentType === 'AdminProPreDocuments' && (
                          <View style={styles.protectedBadge}>
                            <MaterialIcons name="lock" size={12} color="#6c757d" />
                            <Text style={styles.protectedText}>Protected</Text>
                          </View>
                        )}
                      </View>
                    </View>
                    <View style={styles.documentDetailItem}>
                      <Text style={styles.detailLabel}>Description:</Text>
                      <Text style={styles.detailValue}>{doc.description}</Text>
                    </View>
                    <View style={styles.documentDetailItem}>
                      <Text style={styles.detailLabel}>Equipment:</Text>
                      <Text style={styles.detailValue}>{doc.equipment.entryNumber}</Text>
                    </View>
                    <View style={styles.documentDetailItem}>
                      <Text style={styles.detailLabel}>Status:</Text>
                      <Text
                        style={[
                          styles.detailValue,
                          doc.isOffline ? styles.offlineStatus : 
                          doc.documentType === 'AdminProPreDocuments' ? styles.readonlyStatus : styles.onlineStatus,
                        ]}
                      >
                        {doc.isOffline ? 'Pending Sync' : 
                         doc.documentType === 'AdminProPreDocuments' ? 'Read Only' : 'Synced'}
                      </Text>
                    </View>
                    <View style={styles.documentDetailItemNoBorder}>
                      <View style={styles.buttonContainer}>
                        <TouchableOpacity
                          style={styles.viewButton}
                          onPress={() => {
                            try {
                              // iOS: Safe navigation with validation
                              if (Platform.OS === 'ios') {
                                if (!doc.file) {
                                  Toast.show({
                                    type: 'error',
                                    text1: 'Error',
                                    text2: 'Document file not available',
                                  });
                                  return;
                                }
                                
                                const navigationParams = {
                                  url: doc.file,
                                  name: doc.description || 'Document',
                                  title: doc.type || 'Document',
                                };
                                
                                navigation.navigate('viewdocuments', navigationParams);
                              } else {
                                // Android: Direct navigation
                                navigation.navigate('viewdocuments', {
                                  url: doc.file,
                                  name: doc.description,
                                  title: doc.type || 'Document',
                                });
                              }
                            } catch (error) {
                              console.log('Document view navigation error:', error);
                              Toast.show({
                                type: 'error',
                                text1: 'Error',
                                text2: 'Failed to open document',
                              });
                            }
                          }}
                        >
                          <Text style={styles.viewButtonText}>View</Text>
                        </TouchableOpacity>
                        
                        {/* Delete button - only show for non-AdminProPreDocuments */}
                        {doc.documentType !== 'AdminProPreDocuments' && (
                          <TouchableOpacity
                            style={styles.deleteButton}
                            onPress={() => handleDelete(doc.documentType, doc.subfolderName, doc.id, doc.isOffline)}
                          >
                            <Text style={styles.deleteButtonText}>Delete</Text>
                          </TouchableOpacity>
                        )}
                      </View>
                    </View>
                  </View>
                </View>
              ))
            ) : (
              <Text style={styles.noDataText}>No Documents Found</Text>
            )}
          </>
        )}
          </ScrollView>
        </Animated.View>
      </PanGestureHandler>

      {/* Add Document Button */}
      <View style={styles.addButtonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.addButton}
            onPress={handleCreateDocument}
          >
            <MaterialIcons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>Add Document</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Create Folder/Subfolder Modal */}
      <Modal visible={createFolderModalVisible} transparent animationType="fade">
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            {/* Folder Icon at Top Right */}
            <View style={styles.modalHeader}>
              <View style={styles.modalHeaderLeft} />
              <MaterialIcons name="folder" size={40} color="#3481BC" />
              <TouchableOpacity
                style={styles.modalCloseButton}
                onPress={() => setCreateFolderModalVisible(false)}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            
            <Text style={styles.modalTitle}>
              {currentView.type === 'main' ? 'Create New Folder' : 'Create New Subfolder'}
            </Text>
            <TextInput
              style={styles.modalInput}
              placeholder={currentView.type === 'main' ? 'Folder Name' : 'Subfolder Name'}
              placeholderTextColor="#888"
              value={currentView.type === 'main' ? newFolderName : newSubfolderName}
              onChangeText={(text) =>
                currentView.type === 'main' ? setNewFolderName(text) : setNewSubfolderName(text)
              }
            />
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setCreateFolderModalVisible(false)}
              >
                <Text style={styles.cancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={[styles.modalButton, styles.createButton]} 
                onPress={handleCreateFolder}
              >
                <MaterialIcons name="add" size={20} color="#fff" />
                <Text style={styles.createButtonText}>Create</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Delete Confirmation Modal */}
      <Modal visible={deleteConfirmVisible} transparent animationType="fade">
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Delete Document</Text>
            <Text style={styles.modalSubText}>
              Are you sure you want to delete this document? This action cannot be undone.
            </Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={cancelDeleteDocument}
              >
                <Text style={styles.cancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.deleteButton]}
                onPress={confirmDeleteDocument}
              >
                <Text style={styles.deleteButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Dropdown removed per requirements */}

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Image View Modal */}
      <Modal visible={imageModalVisible} transparent animationType="fade">
        <View style={styles.imageModalContainer}>
          <View style={styles.imageModalContent}>
            {selectedImage && !imageLoadError ? (
              <Image
                source={{ uri: selectedImage }}
                style={styles.modalImage}
                onError={() => {
                  setImageLoadError(true);
                  Toast.show({
                    type: 'error',
                    text1: 'Error',
                    text2: 'Failed to load image.',
                  });
                }}
                onLoad={() => setImageLoadError(false)}
              />
            ) : (
              <Text style={styles.noImageText}>
                {imageLoadError ? 'Failed to load image.' : 'Image not available.'}
              </Text>
            )}
            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => {
                setImageModalVisible(false);
                setImageLoadError(false);
              }}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Unified Viewer Modal for images, PDFs, and DOCX */}
      <Modal visible={viewerVisible} transparent animationType="fade" onRequestClose={() => setViewerVisible(false)}>
        <View style={styles.imageModalContainer}>
          <View style={[styles.imageModalContent, { width: '95%', height: '85%' }]}>
            {viewerType === 'image' && !viewerLoadError && (
              <Image source={{ uri: viewerUrl }} style={{ width: '100%', height: '80%', resizeMode: 'contain' }} onError={() => setViewerLoadError(true)} />
            )}
            {viewerType === 'pdf' && !viewerLoadError && (
              <Pdf source={{ uri: viewerUrl }} style={{ width: '100%', height: '80%' }} onError={() => setViewerLoadError(true)} />
            )}
            {viewerType === 'docx' && !viewerLoadError && (
              <WebView source={{ uri: viewerUrl }} style={{ width: '100%', height: '80%' }} onError={() => setViewerLoadError(true)} />
            )}
            {viewerLoadError && (
              <Text style={styles.noImageText}>Failed to load document.</Text>
            )}
            <View style={{ flexDirection: 'row', justifyContent: 'space-between', width: '100%' }}>
              <TouchableOpacity style={styles.closeButton} onPress={() => setViewerVisible(false)}>
                <Text style={styles.closeButtonText}>Close</Text>
              </TouchableOpacity>
              {!!viewerDoc && (
                <TouchableOpacity style={styles.closeButton} onPress={() => handleDownload(viewerDoc)}>
                  <Text style={styles.closeButtonText}>Download</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
    flexShrink: 1,
  },
  headerBottom: {
    paddingHorizontal: width * 0.04,
    paddingBottom: width * 0.025,
  },
  backButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
    marginBottom: 5,
  },
  backButtonContainer: {
    marginBottom: 10,
  },
  iosSwipeHint: {
    color: 'rgba(255, 255, 255, 0.8)',
    fontSize: width * 0.03,
    fontStyle: 'italic',
    textAlign: 'center',
  },
  swipeHintContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginTop: 5,
  },
  swipeIndicator: {
    marginLeft: 10,
    opacity: 0.7,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  content: {
    flex: 1,
  },
  contentContainer: {
    flex: 1,
  },
  noDataText: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
    marginTop: 20,
  },
  folderRow: {
    marginHorizontal: 15,
    marginVertical: '2%',
    backgroundColor: '#E8F0FE',
    borderRadius: 5,
    padding: 10,
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  subfolderRow: {
    marginHorizontal: 15,
    marginVertical: '2%',
    backgroundColor: '#F0F4F8',
    borderRadius: 5,
    padding: 10,
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  folderContent: {
    flex: 1,
  },
  subfolderContent: {
    flex: 1,
  },
  folderHeader: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  folderInfo: {
    marginLeft: 10,
    flex: 1,
  },
  folderTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 2,
  },
  subfolderTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#3481BC',
    marginBottom: 2,
  },
  documentCount: {
    fontSize: 12,
    color: '#666',
    fontWeight: '400',
  },
  menuButton: {
    padding: 4,
  },
  documentRow: {
    marginHorizontal: 15,
    marginVertical: '3%',
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
  },
  documentInfo: {
    flex: 1,
  },
  documentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: '5%',
  },
  documentDetailItemNoBorder: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingHorizontal: '5%',
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailValue: {
    fontSize: 16,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  offlineStatus: {
    color: '#FF6200',
  },
  onlineStatus: {
    color: '#00A100',
  },
  readonlyStatus: {
    color: '#6c757d',
    fontStyle: 'italic',
  },
  typeContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  protectedBadge: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    paddingHorizontal: 6,
    paddingVertical: 2,
    borderRadius: 10,
    marginLeft: 8,
    borderWidth: 1,
    borderColor: '#dee2e6',
  },
  protectedText: {
    fontSize: 10,
    color: '#6c757d',
    marginLeft: 2,
    fontWeight: '500',
  },

  viewButton: {
    backgroundColor: '#3481BC',
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    minWidth: 80,
    alignItems: 'center',
    alignSelf: 'flex-start',
  },
  viewButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  deleteButton: {
    backgroundColor: '#FF4444',
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    minWidth: 80,
    alignItems: 'center',
  },
  deleteButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  addButtonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  documentButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    gap: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
    marginBottom: 15,
  },
  modalHeaderLeft: {
    width: 40,
  },
  modalCloseButton: {
    padding: 5,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 15,
  },
  modalSubText: {
    fontSize: 16,
    color: '#666',
    marginBottom: 20,
    textAlign: 'center',
    lineHeight: 22,
  },
  modalInput: {
    width: '100%',
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    padding: 10,
    fontSize: 16,
    marginBottom: 15,
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
    minWidth: 100,
    alignItems: 'center',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  createButton: {
    backgroundColor: '#3481BC',
    flexDirection: 'row',
    alignItems: 'center',
    gap: 8,
  },
  createButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  imageModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.8)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  imageModalContent: {
    width: '90%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalImage: {
    width: '100%',
    height: 300,
    resizeMode: 'contain',
    marginBottom: 20,
  },
  noImageText: {
    fontSize: 16,
    color: '#333',
    marginBottom: 20,
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  confirmModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    alignItems: 'center',
  },
  confirmModalText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  confirmModalSubText: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  deleteButton: {
    backgroundColor: '#FF4444',
  },
  deleteButtonText: {
    color: '#FFFFFF',
    fontSize: 16,
    fontWeight: '600',
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.2)',
    top: '-3%',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#eee',
    width: 220,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    paddingVertical: 4,
    zIndex: 999,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: 16,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
});

export default Documenthome;