import React, { useState, useEffect, useRef } from 'react';
import { View, Text, TouchableOpacity, StyleSheet, ScrollView, TextInput, Alert, Modal, Animated, Dimensions } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import database from '@react-native-firebase/database';

const { width } = Dimensions.get('window');

const DvirPChecklist = {
  artifact_id: "2aa56fee-59ce-4356-8429-fac734f38ede",
  formTitle: "Driver's Pre-Trip Release Form",
  fields: {
    date: { type: "date", label: "Date", value: "2025-06-03" },
    vehicleRegistration: { type: "text", label: "Vehicle Registration", value: "" },
  },
  requirements: {
    type: "checklist",
    label: "Requirements",
    items: [
      { id: "req1", label: "I hold a valid driver's licence", checked: false },
      { id: "req2", label: "I will obey speed limits and road signs", checked: false },
      { id: "req3", label: "I will not overload the vehicle and ensure the load is secure", checked: false },
      { id: "req4", label: "I have completed the vehicle pre check list and satisfied all is order for my trip", checked: false },
      { id: "req5", label: "I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)", checked: false },
      { id: "req6", label: "I have read and understand the Titan driving procedure, National driving rules and cliental rules", checked: false },
      { id: "req7", label: "I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)", checked: false },
    ],
  },
  hazards: {
    type: "checklist",
    label: "Hazards – Are you aware of:",
    items: [
      { id: "haz1", label: "Pedestrians crossing the road, school children and drunkards", checked: false },
      { id: "haz2", label: "Motor bikes and taxis stopping suddenly or pulling in to the road suddenly", checked: false },
      { id: "haz3", label: "Oncoming vehicles overtaking on blind rises or corners", checked: false },
      { id: "haz4", label: "Heavy trucks or plant taking up width of the road", checked: false },
      { id: "haz5", label: "Riots or unrest", checked: false },
      { id: "haz6", label: "Gravel/Sandy roads can be slippery – Loss of control", checked: false },
      { id: "haz7", label: "Tyre blowouts or punctures – Loss of control", checked: false },
    ],
  },
  authorization: {
    type: "section",
    label: "Authorization and Instructions",
    fields: {
      authorizedBy: { type: "text", label: "Who has authorized this trip and the vehicle to be used?", value: "" },
      instructions: { type: "textarea", label: "What are the instructions given to you?", value: "" },
      departureTime: { type: "time", label: "Time instructed to depart", value: "18:28" },
      returnTime: { type: "time", label: "Time instructed to return", value: "" },
    },
  },
  travelNote: {
    type: "note",
    value: "NB. Unless otherwise instructed you are required to leave no earlier than sunrise and arrive at your destination no later than sunset. Failing to arrive then you should park in a safe area for the night. You are required to inform your supervisor on arrival at your destination and the same when you depart.",
  },
  passengers: {
    type: "list",
    label: "Passengers' names authorized to travel with you",
    items: [
      { id: "pass1", label: "Passenger 1", type: "text", value: "" },
      { id: "pass2", label: "Passenger 2", type: "text", value: "" },
    ],
  },
  additionalNote: {
    type: "note",
    value: "No personal business/issues to be conducted unless you have authorization to do so.",
  },
  signatures: {
    type: "section",
    label: "Signatures",
    fields: {
      driverName: { type: "text", label: "Driver's Name", value: "" },
      driverSignature: { type: "text", label: "Driver's Signature", value: "" },
      supervisorName: { type: "text", label: "Supervisor's Name", value: "" },
      supervisorSignature: { type: "text", label: "Supervisor's Signature", value: "" },
    },
  },
};

const inspectionSections = [
  {
    key: 'requirements',
    label: DvirPChecklist.requirements.label,
    items: DvirPChecklist.requirements.items.map((item) => ({
      id: item.id,
      text: item.label,
      subItems: [{ itemName: 'Status', inputType: 'Status' }],
    })),
  },
  {
    key: 'hazards',
    label: DvirPChecklist.hazards.label,
    items: DvirPChecklist.hazards.items.map((item) => ({
      id: item.id,
      text: item.label,
      subItems: [{ itemName: 'Status', inputType: 'Status' }],
    })),
  },
];

const DvirDoinspectionStep2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [expanded, setExpanded] = useState(null);
  const [itemStates, setItemStates] = useState({});
  const [checklist, setChecklist] = useState(null);
  const [sections, setSections] = useState(inspectionSections);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    loadChecklist();
  }, []);

  const loadChecklist = async () => {
    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const selectedChecklistIds = await AsyncStorage.getItem('selectedChecklistIds');
      let updatedSections = [...inspectionSections];
      if (selectedChecklistIds) {
        const state = await NetInfo.fetch();
        let allChecklists = [];
        if (state.isConnected) {
          const snapshot = await database().ref('checklists').once('value');
          allChecklists = snapshot.val() ? Object.keys(snapshot.val()).map((id) => ({
            id,
            ...snapshot.val()[id],
          })) : [];
          await AsyncStorage.setItem('checklists', JSON.stringify(allChecklists));
        } else {
          const storedChecklists = await AsyncStorage.getItem('checklists');
          allChecklists = storedChecklists ? JSON.parse(storedChecklists) : [];
        }
        const ids = JSON.parse(selectedChecklistIds);
        const selectedChecklist = allChecklists.find((c) => ids.includes(c.id));
        if (selectedChecklist && selectedChecklist.items) {
          setChecklist(selectedChecklist);
          updatedSections = [
            ...updatedSections,
            ...selectedChecklist.items.map((section) => ({
              ...section,
              key: section.id || section.text,
              label: section.text,
            })),
          ];
        }
      }
      setSections(updatedSections);
      console.log('Updated sections:', updatedSections);
    } catch (error) {
      console.error('Error loading checklist:', error);
      Alert.alert('Error', 'Failed to load checklist data.');
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const handleExpand = (key) => {
    setExpanded(expanded === key ? null : key);
  };

  const handleStatus = (sectionKey, itemId, subItemName, status) => {
    setItemStates((prev) => {
      const key = `${sectionKey}_${itemId}_${subItemName}`;
      return {
        ...prev,
        [key]: { ...prev[key], status },
      };
    });
  };

  const handleInput = (sectionKey, itemId, subItemName, text) => {
    setItemStates((prev) => {
      const key = `${sectionKey}_${itemId}_${subItemName}`;
      return {
        ...prev,
        [key]: { ...prev[key], note: text },
      };
    });
  };

  const saveStep2Data = async () => {
    try {
      const step2Data = {
        itemStates,
        checklistId: checklist?.id || null,
        timestamp: new Date().getTime(),
      };
      const key = `step2-${step2Data.timestamp}`;
      await AsyncStorage.setItem(key, JSON.stringify(step2Data));
      console.log('Step 2 Data Saved:', step2Data);
      return step2Data;
    } catch (error) {
      console.error('Error saving Step 2 data:', error);
      throw error;
    }
  };

  const handleNext = async () => {
    if (Object.keys(itemStates).length === 0) {
      Alert.alert('Error', 'Please complete at least one checklist item.');
      return;
    }
    try {
      await saveStep2Data();
      setActiveStep(3);
    } catch (error) {
      Alert.alert('Error', error.message || 'Failed to proceed to Step 3.');
    }
  };

  const handleBack = () => {
    navigation.goBack();
  };

  const getStatusOptions = (sectionKey) => {
    if (sectionKey === 'requirements' || sectionKey === 'hazards') {
      return ['Yes', 'No'];
    }
    return ['Pass', 'Fail', 'NA'];
  };

  return (
    <View style={styles.container}>
      <Text style={styles.headerText}>Tap on list to expand</Text>
      <ScrollView contentContainerStyle={styles.scrollContent}>
        {sections.map((section) => (
          <View key={section.key || section.id} style={styles.card}>
            <TouchableOpacity
              style={[
                styles.cardHeader,
                expanded === (section.key || section.id) && styles.cardHeaderActive,
              ]}
              onPress={() => handleExpand(section.key || section.id)}
            >
              <MaterialIcons
                name={expanded === (section.key || section.id) ? 'arrow-drop-up' : 'arrow-drop-down'}
                size={22}
                color={expanded === (section.key || section.id) ? '#015185' : '#4CAF50'}
              />
              <Text
                style={[
                  styles.cardHeaderText,
                  expanded === (section.key || section.id) && styles.cardHeaderTextActive,
                ]}
              >
                {section.label || section.text}
              </Text>
            </TouchableOpacity>
            {expanded === (section.key || section.id) && (
              <View style={styles.cardContent}>
                {(section.items || []).map((item) => (
                  <View key={item.id} style={styles.itemRow}>
                    <Text style={styles.itemLabel}>{item.text}</Text>
                    {item.subItems.map((subItem) => (
                      <View key={subItem.itemName} style={styles.subItemContainer}>
                        <Text style={styles.subItemLabel}>{subItem.itemName}</Text>
                        {subItem.inputType === 'Status' ? (
                          <View style={styles.statusRow}>
                            {getStatusOptions(section.key || section.id).map((status) => (
                              <TouchableOpacity
                                key={status}
                                style={[
                                  styles.statusButton,
                                  itemStates[`${section.key || section.id}_${item.id}_${subItem.itemName}`]?.status === status &&
                                    styles.statusButtonActive,
                                ]}
                                onPress={() => handleStatus(section.key || section.id, item.id, subItem.itemName, status)}
                              >
                                <Text
                                  style={[
                                    styles.statusButtonText,
                                    itemStates[`${section.key || section.id}_${item.id}_${subItem.itemName}`]?.status === status &&
                                      styles.statusButtonTextActive,
                                  ]}
                                >
                                  {status}
                                </Text>
                              </TouchableOpacity>
                            ))}
                          </View>
                        ) : (
                          <TextInput
                            style={styles.input}
                            placeholder="Enter value"
                            value={itemStates[`${section.key || section.id}_${item.id}_${subItem.itemName}`]?.note || ''}
                            onChangeText={(text) => handleInput(section.key || section.id, item.id, subItem.itemName, text)}
                            keyboardType={subItem.inputType === 'Numeric' ? 'numeric' : 'default'}
                          />
                        )}
                      </View>
                    ))}
                    <View style={styles.iconRow}>
                      <MaterialIcons name="content-copy" size={22} color="#015185" style={{ marginRight: 10 }} />
                      <MaterialIcons name="camera-alt" size={22} color="#015185" />
                    </View>
                  </View>
                ))}
              </View>
            )}
          </View>
        ))}
      </ScrollView>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Loading Checklist Data</Text>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  headerText: {
    fontSize: 16,
    color: '#015185',
    margin: 16,
    fontWeight: '500',
  },
  scrollContent: {
    padding: 10,
  },
  card: {
    marginBottom: 12,
    borderRadius: 8,
    backgroundColor: '#f5f5f5',
  },
  cardHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: 10,
    backgroundColor: '#f0f0f0',
    borderRadius: 8,
  },
  cardHeaderActive: {
    backgroundColor: '#e9e9e9',
  },
  cardHeaderText: {
    fontWeight: '500',
    color: '#015185',
    marginLeft: 6,
  },
  cardHeaderTextActive: {
    color: '#015185',
    fontWeight: 'bold',
  },
  cardContent: {
    padding: 10,
    backgroundColor: '#fff',
    borderBottomLeftRadius: 8,
    borderBottomRightRadius: 8,
  },
  itemRow: {
    marginBottom: 16,
  },
  itemLabel: {
    fontSize: 14,
    color: '#222',
    marginBottom: 6,
    fontWeight: '500',
  },
  subItemContainer: {
    marginBottom: 8,
  },
  subItemLabel: {
    fontSize: 13,
    color: '#333',
    marginBottom: 4,
  },
  statusRow: {
    flexDirection: 'row',
    marginBottom: 6,
  },
  statusButton: {
    backgroundColor: '#eee',
    borderRadius: 4,
    paddingVertical: 4,
    paddingHorizontal: 12,
    marginRight: 8,
  },
  statusButtonActive: {
    backgroundColor: '#015185',
  },
  statusButtonText: {
    color: '#015185',
    fontWeight: '500',
  },
  statusButtonTextActive: {
    color: '#fff',
  },
  input: {
    borderBottomWidth: 1,
    borderBottomColor: '#bbb',
    marginBottom: 6,
    fontSize: 13,
    paddingVertical: 2,
    color: '#222',
  },
  iconRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
});

export default DvirDoinspectionStep2;