import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, Dimensions, SafeAreaView, StatusBar, Platform, ScrollView } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import ApplicationList from './ApplicationList';

const { width } = Dimensions.get('window');

const AdminAllAppealApli = () => {
  const navigation = useNavigation();
  const [applications, setApplications] = useState([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isOnline, setIsOnline] = useState(true);

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected || false);
    });

    return () => unsubscribe();
  }, []);

  useEffect(() => {
    const fetchApplications = async () => {
      setIsLoading(true);
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        const currentUser = auth().currentUser;
        if (!currentUser || currentUser.uid !== userUid) {
          showToast('error', 'Authentication Error', 'Invalid session. Please log in again.');
          await AsyncStorage.multiRemove(['userUid', 'userData']);
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        if (!isOnline) {
          const cachedApplications = await AsyncStorage.getItem('offlineApplications');
          if (cachedApplications) {
            const parsedApplications = JSON.parse(cachedApplications);
            const filteredApplications = parsedApplications
              .filter((app) => app.applicationType === 'application_for_appeal' && app.userUid === userUid)
              .map((app) => ({ ...app, applicationId: app.applicationId || `offline_${Date.now()}` }));
            setApplications(filteredApplications);
            showToast('info', 'Offline Mode', 'Loaded cached appeal applications.');
          } else {
            showToast('warning', 'Offline Mode', 'No cached appeal applications available.');
          }
          setIsLoading(false);
          return;
        }

        // Fetch all users' appeal applications
        const applicationRef = database().ref(`/GlobalHrApplications/admins/application_for_appeal`);
        applicationRef.on('value', (snapshot) => {
          const data = snapshot.val();
          const applicationList = [];
          if (data) {
            Object.entries(data).forEach(([userKey, userApps]) => {
              Object.entries(userApps).forEach(([appKey, appValue]) => {
                applicationList.push({ ...appValue, applicationId: appKey, userUid: userKey });
              });
            });
          }
          // Sort: mine first, then others
          const mine = applicationList.filter(app => app.userUid === userUid);
          const others = applicationList.filter(app => app.userUid !== userUid);
          setApplications([...mine, ...others]);
          setIsLoading(false);
        });

        return () => applicationRef.off();
      } catch (error) {
        console.error('Error fetching appeal applications:', error);
        showToast('error', 'Error', 'Failed to load appeal applications.');
        const cachedApplications = await AsyncStorage.getItem('offlineApplications');
        if (cachedApplications) {
          const parsedApplications = JSON.parse(cachedApplications);
          const filteredApplications = parsedApplications
            .filter((app) => app.applicationType === 'application_for_appeal' && app.userUid === userUid)
            .map((app) => ({ ...app, applicationId: app.applicationId || `offline_${Date.now()}` }));
          setApplications(filteredApplications);
          showToast('warning', 'Offline Mode', 'Loaded cached appeal applications due to error.');
        }
        setIsLoading(false);
      }
    };

    fetchApplications();
  }, [isOnline, navigation]);

  return (
    <View style={styles.flex1}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.headerGradient}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <SafeAreaView style={styles.safeAreaHeader}>
          <View style={styles.headerContentFixed}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitleFixed}>Appeal Applications</Text>
            <View style={{ width: width * 0.06 }} />
          </View>
        </SafeAreaView>
      </LinearGradient>
      <ScrollView contentContainerStyle={styles.scrollContent} bounces={false} showsVerticalScrollIndicator={false}>
        <ApplicationList
          applications={applications}
          isLoading={isLoading}
          applicationType="application_for_appeal"
          navigation={navigation}
        />
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  flex1: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  headerGradient: {
    width: '100%',
    paddingBottom: Platform.OS === 'android' ? StatusBar.currentHeight || 24 : 0,
  },
  safeAreaHeader: {
    backgroundColor: 'transparent',
  },
  headerContentFixed: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'android' ? ((StatusBar.currentHeight || width * 0.4) - 30) : 0,
    paddingBottom: width * 0.03,
    minHeight: Platform.OS === 'android' ? width * 0.01 - 10 : width * 0.01,
  },
  headerTitleFixed: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  backButton: {
    padding: width * 0.02,
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  scrollContent: {
    flexGrow: 1,
    paddingBottom: width * 0.04,
  },
});

export default AdminAllAppealApli;